note
	legal: "See notice at end of class."
	status: "See notice at end of class."
class
	CONVERTER

create
	make

feature -- Initialization

	make (a_class_name: READABLE_STRING_GENERAL; output_file, input_file: READABLE_STRING_GENERAL)
			-- Create a class `a_class_name' in the file `output_file'.
		require
			a_class_name_not_void: a_class_name /= Void
			a_class_name_not_empty: not a_class_name.is_empty
			output_file_not_void: output_file /= Void
			output_file_not_empty: not output_file.is_empty
			input_file_not_void: input_file /= Void
		do
			class_name := a_class_name.as_upper
			create class_file.make_with_name (output_file)
			class_file.create_read_write
			insert_header_in_file (class_file, input_file)
		end

feature -- Access

	class_file: PLAIN_TEXT_FILE
			-- Output file to store the eiffel class.

	class_name: READABLE_STRING_GENERAL
			-- Name of the eiffel class to genarate.

feature -- Basic operations

	extract_definition (input_file: STRING_32)
			-- Scans input_file for "#define id integer" and puts them in output_file
		require
			input_file_not_void: input_file /= Void
			input_file_not_empty: not input_file.is_empty
			input_file_exists: file_exists (input_file)
		local
			a_file: PLAIN_TEXT_FILE
		do
			create a_file.make_open_read (input_file)
			from
				a_file.start
			until
				a_file.after
			loop
				a_file.read_character
				if a_file.last_character.is_equal ('#') then
					scan_definition (a_file)
				else
					if not white_space (a_file.last_character)
					and then not a_file.end_of_file then
						a_file.next_line
					end
				end
			end
			check
				end_of_input_reached: a_file.end_of_file
			end
			a_file.close
		end

	close_file
			-- Puts class-statement end in ouput_file and closes it.
		require
			class_file_is_open: class_file.is_open_write
		do
			class_file.putstring ("%Nend")
			class_file.new_line
			class_file.close
		ensure
			class_file_is_closed: class_file.is_closed
		end

feature -- Status report

	file_exists (filename: STRING_32): BOOLEAN
			-- Check if a file with filename exists
		require
			filename_not_void: filename /= Void
		do
			Result := (create {PLAIN_TEXT_FILE}.make_with_name (filename)).exists
		end

feature {NONE} -- Implementation

	scan_definition (a_file: PLAIN_TEXT_FILE)
			-- Scan a "define id value" in `a_file' if any to come
		require
			a_file_not_void: a_file /= Void
			a_file_exists: a_file.exists
			a_file_is_open: a_file.is_open_read
			a_file_valid_pos: a_file.last_character.is_equal ('#')
		local
			l_string, id: STRING_8
		do
			a_file.read_word
			l_string := a_file.last_string
			if l_string.same_string ("define") then
				a_file.read_word
				id := a_file.last_string.twin
				a_file.read_word
				l_string := a_file.last_string
				if l_string.is_integer or l_string.substring_index ("0x", 1) = 1 then
					insert_id_in_file (id, l_string, class_file)
				else
					scan_to_previous_white_space (a_file)
				end
			end
		end

	insert_header_in_file (a_file: PLAIN_TEXT_FILE; input_file: READABLE_STRING_GENERAL)
			-- Fill the file with the basic stuff
		require
			a_file_not_void: a_file /= Void
			a_file_exists: a_file.exists
			a_file_is_open: a_file.is_open_write
			input_file_not_void: input_file /= Void
		local
			u: UTF_CONVERTER
			l_utf8: BOOLEAN
		do
			l_utf8 := not input_file.is_valid_as_string_8 or not class_name.is_valid_as_string_8
			if l_utf8 then
					-- The content of the file has to be UTF-8 encoded.
				a_file.put_string ({UTF_CONVERTER}.utf_8_bom_to_string_8)
			end
			a_file.put_string ("indexing%N")
			a_file.put_string ("%Tdescription: %"Generated by h2e from the file %%%N")
			a_file.put_string ("%T%T%%")
			a_file.put_string
				(if l_utf8 then
					u.utf_32_string_to_utf_8_string_8 (input_file)
				else
						-- Per earlier check, it is safe to truncate.
					input_file.to_string_8
				end)
			a_file.put_string (".%"%N%N")
			a_file.put_string ("class%N%T")
			a_file.put_string
				(if l_utf8 then
					u.utf_32_string_to_utf_8_string_8 (class_name)
				else
					class_name.to_string_8
				end)
			a_file.putstring ("%N%Nfeature -- Access%N%N")
		end

	scan_to_previous_white_space (a_file: PLAIN_TEXT_FILE)
			-- Go to the previous white space in `a_file'
		require
			a_file_not_void: a_file /= Void
			a_file_exists: a_file.exists
		do
			from
				a_file.back
			until
				a_file.item = '%N' or
				a_file.item = '%T' or
				a_file.item = ' '
			loop
				a_file.back
			end
		end

	insert_id_in_file (id, value: STRING_8; a_file: PLAIN_TEXT_FILE)
			-- Add a `id' to the class file.
		require
			id_not_void: id /= Void
			id_not_empty: not id.is_empty
			value_not_void: value /= Void
			a_file_not_void: a_file /= Void
			a_file_exists: a_file.exists
			a_file_is_open: a_file.is_open_write
		do
			if id.item (1) /= '_' then
				-- Do not write ids starting by '_'
				a_file.putchar ('%T')
				id.to_lower
				id.put (id.item (1).upper, 1)
				a_file.putstring (id)
				a_file.putstring (": INTEGER is ")
				a_file.putstring (value)
				a_file.new_line
			end
		end

	white_space (a_c: CHARACTER): BOOLEAN
			-- Is `a_c' white space?
		do
			Result := a_c = '%T' or else
				a_c = '%N' or else
				a_c = ' '
		end

note
	copyright:	"Copyright (c) 1984-2020, Eiffel Software and others"
	license:	"Eiffel Forum License v2 (see http://www.eiffel.com/licensing/forum.txt)"
	source: "[
			Eiffel Software
			5949 Hollister Ave., Goleta, CA 93117 USA
			Telephone 805-685-1006, Fax 805-685-6869
			Website http://www.eiffel.com
			Customer support http://support.eiffel.com
		]"

end
