﻿note

	description:

		"C code generators"

	library: "Gobo Eiffel Tools Library"
	copyright: "Copyright (c) 2004-2025, Eric Bezault and others"
	license: "MIT License"

class ET_C_GENERATOR

inherit

	ET_AST_NULL_PROCESSOR
		rename
			make as make_null
		redefine
			process_across_expression,
			process_across_instruction,
			process_assigner_instruction,
			process_assignment,
			process_assignment_attempt,
			process_attribute,
			process_bang_instruction,
			process_binary_integer_constant,
			process_bracket_expression,
			process_c1_character_constant,
			process_c2_character_constant,
			process_c3_character_constant,
			process_call_agent,
			process_check_instruction,
			process_constant_attribute,
			process_convert_builtin_expression,
			process_convert_from_expression,
			process_convert_to_expression,
			process_create_expression,
			process_create_instruction,
			process_current,
			process_current_address,
			process_debug_instruction,
			process_deferred_function,
			process_deferred_procedure,
			process_do_function,
			process_do_function_inline_agent,
			process_do_procedure,
			process_do_procedure_inline_agent,
			process_equality_expression,
			process_explicit_convert_from_expression,
			process_explicit_convert_to_expression,
			process_expression_address,
			process_extended_attribute,
			process_external_function,
			process_external_function_inline_agent,
			process_external_procedure,
			process_external_procedure_inline_agent,
			process_false_constant,
			process_feature_address,
			process_hexadecimal_integer_constant,
			process_identifier,
			process_if_expression,
			process_if_instruction,
			process_infix_cast_expression,
			process_infix_expression,
			process_inline_separate_instruction,
			process_inlined_expression,
			process_inspect_expression,
			process_inspect_instruction,
			process_invariants,
			process_iteration_cursor,
			process_loop_instruction,
			process_manifest_array,
			process_manifest_tuple,
			process_manifest_type,
			process_named_object_test,
			process_object_equality_expression,
			process_object_test,
			process_octal_integer_constant,
			process_old_expression,
			process_old_object_test,
			process_once_function,
			process_once_function_inline_agent,
			process_once_manifest_string,
			process_once_procedure,
			process_once_procedure_inline_agent,
			process_parenthesis_expression,
			process_parenthesis_instruction,
			process_parenthesized_expression,
			process_precursor_expression,
			process_precursor_instruction,
			process_prefix_expression,
			process_qualified_call_expression,
			process_qualified_call_instruction,
			process_quantifier_expression,
			process_regular_integer_constant,
			process_regular_manifest_string,
			process_regular_real_constant,
			process_repeat_instruction,
			process_result,
			process_result_address,
			process_retry_instruction,
			process_semicolon_symbol,
			process_special_manifest_string,
			process_static_call_expression,
			process_static_call_instruction,
			process_strip_expression,
			process_true_constant,
			process_underscored_integer_constant,
			process_underscored_real_constant,
			process_unique_attribute,
			process_unqualified_call_expression,
			process_unqualified_call_instruction,
			process_verbatim_string,
			process_void
		end

	ET_SHARED_TOKEN_CONSTANTS
		export {NONE} all end

	ET_SHARED_IDENTIFIER_TESTER
		export {NONE} all end

	ET_SHARED_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID
		export {NONE} all end

	KL_SHARED_STREAMS
		export {NONE} all end

	KL_SHARED_FILE_SYSTEM
		export {NONE} all end

	KL_SHARED_EXECUTION_ENVIRONMENT
		export {NONE} all end

	KL_SHARED_STRING_EQUALITY_TESTER
		export {NONE} all end

	UT_SHARED_ISE_VARIABLES
		export {NONE} all end

	UT_SHARED_GOBO_VARIABLES
		export {NONE} all end

	KL_IMPORTED_CHARACTER_ROUTINES
		export {NONE} all end

	KL_IMPORTED_INTEGER_ROUTINES
		export {NONE} all end

	KL_IMPORTED_STRING_ROUTINES
		export {NONE} all end

	UT_IMPORTED_FORMATTERS
		export {NONE} all end

create

	make

feature {NONE} -- Initialization

	make (a_system: like current_dynamic_system; a_system_processor: like system_processor)
			-- Create a new C code generator.
		require
			a_system_not_void: a_system /= Void
			a_system_processor_not_void: a_system_processor /= Void
		local
			l_buffer: STRING
			l_inline_separate_arguments: ET_INLINE_SEPARATE_ARGUMENTS
			l_inline_separate_argument_name: ET_IDENTIFIER
			l_compound: ET_COMPOUND
			l_result: ET_RESULT
		do
			system_processor := a_system_processor
			current_dynamic_system := a_system
			dynamic_types := a_system.dynamic_types
			short_names := True
			split_mode := True
			split_threshold := default_split_threshold
			use_boehm_gc := current_system.use_boehm_gc
			system_name := "unknown"
			current_file := null_output_stream
			header_file := null_output_stream
			current_type := a_system.unknown_type
			current_feature := dummy_feature
			current_non_inlined_feature := current_feature
			create extra_dynamic_type_sets.make_with_capacity (50)
			extra_dynamic_type_sets.put_last (a_system.any_type)
			current_dynamic_type_sets := extra_dynamic_type_sets
			create l_buffer.make (1024)
			create current_function_header_buffer.make (l_buffer)
			create l_buffer.make (100000)
			create current_function_body_buffer.make (l_buffer)
			create l_buffer.make (1024)
			create current_separate_function_header_buffer.make (l_buffer)
			create l_buffer.make (10000)
			create current_separate_function_body_buffer.make (l_buffer)
			create old_expressions.make (100)
			create old_expression_temp_variables.make_map (100)
			create old_expression_exception_temp_variables.make_map (100)
			create temp_variables.make (40)
			create used_temp_variables.make (40)
			create free_temp_variables.make (40)
			create frozen_temp_variables.make (40)
			create volatile_temp_variables.make (40)
			create separate_temp_variables.make (40)
			create used_separate_temp_variables.make (40)
			create free_separate_temp_variables.make (40)
			create frozen_separate_temp_variables.make (40)
			create volatile_separate_temp_variables.make (40)
			create unused_equality_expressions.make (10)
			create unused_equality_types.make (100)
			create unused_object_equality_types.make (100)
			create unused_unqualified_call_expressions.make (10)
			create unused_qualified_call_expressions.make (10)
			create unused_actual_argument_lists.make (10)
			create unused_result_expressions.make (10)
			create unused_identifiers.make (10)
			create conforming_types.make_with_capacity (100)
			create non_conforming_types.make_with_capacity (100)
			create equality_type_set.make_empty (current_dynamic_system.unknown_type)
			create equality_common_types.make_with_capacity (100)
			create equality_common_reference_types.make_with_capacity (100)
			create unused_standalone_type_sets.make (50)
			create operand_stack.make (5000)
			create call_operands.make (5000)
			create attachment_dynamic_type_ids.make (100)
			create target_dynamic_type_ids.make (100)
			create target_dynamic_types.make_map (100)
			create standalone_type_sets.make_with_capacity (50)
			create deep_twin_types.make (100)
			create deep_equal_types.make (100)
			create deep_feature_target_type_sets.make_map (100)
			create current_object_tests.make (20)
			create current_object_test_local_types.make (20)
			create current_object_test_locals.make_map (20)
			create object_test_local_name.make ("m1")
			object_test_local_name.set_object_test_local (True)
			object_test_local_name.set_seed (1)
			create current_iteration_cursor_types.make (20)
			create current_iteration_cursors.make_map (20)
			create iteration_cursor_name.make ("u1")
			iteration_cursor_name.set_iteration_item (True)
			iteration_cursor_name.set_seed (1)
			create current_inline_separate_arguments.make (20)
			create current_equalities.make (20)
			create current_agents.make (20)
			create agent_open_operands.make (20)
			create agent_closed_operands.make (20)
			agent_target := tokens.current_keyword
			create agent_arguments.make_with_capacity (100)
			create agent_instruction.make (agent_target, current_feature.static_feature.name, Void)
			create agent_expression.make (agent_target, current_feature.static_feature.name, Void)
			create l_inline_separate_argument_name.make ("sa")
			l_inline_separate_argument_name.set_inline_separate_argument (True)
			l_inline_separate_argument_name.set_seed (1)
			create agent_inline_separate_argument.make (agent_target, l_inline_separate_argument_name)
			create l_inline_separate_arguments.make_with_capacity (1)
			l_inline_separate_arguments.put_first (agent_inline_separate_argument)
			create l_compound.make_with_capacity (1)
			l_compound.put_first (agent_instruction)
			create agent_controlled_target_instruction.make (l_inline_separate_arguments, l_compound)
			create l_result.make
			create agent_expression_assignment.make (l_result, agent_expression)
			create l_compound.make_with_capacity (1)
			l_compound.put_first (agent_instruction)
			create agent_controlled_target_expression.make (l_inline_separate_arguments, l_compound)
			agent_closed_operands_type := current_dynamic_system.unknown_type
			create agent_manifest_tuple.make_with_capacity (20)
			create separate_call_arguments.make_with_capacity (100)
			create separate_call_instruction.make (tokens.current_keyword, current_feature.static_feature.name, Void)
			create separate_call_expression.make (tokens.current_keyword, current_feature.static_feature.name, Void)
			create separate_arguments.make (10)
			create formal_arguments.make (20)
			create manifest_array_types.make (100)
			create big_manifest_array_types.make (100)
			create manifest_tuple_types.make (100)
			create once_features.make_map (10000)
			create once_per_process_counts.make_filled (0, 1, once_kind_count)
			create once_per_thread_counts.make_filled (0, 1, once_kind_count)
			create constant_features.make_map (10000)
			create inline_constants.make_map (10000)
			create dispose_procedures.make_map (dynamic_types.count)
			create dynamic_type_id_set_names.make_map (10000)
			dynamic_type_id_set_names.set_key_equality_tester (string_equality_tester)
			create used_inlined_operands.make (1000)
			create unused_inlined_operands.make (1000)
			create used_call_contexts.make (100)
			create unused_call_contexts.make (100)
			if a_system.current_system.inlining_mode then
				max_nested_inlining_count := a_system.current_system.inlining_size.max (0)
			end
			create inlined_void.make
			create inlined_integer.make ("0", 0, False)
			create inlined_real.make ("0.0")
			create inlined_character.make ('U')
			create inlined_boolean.make
			create called_features.make (1000)
			create inlined_features.make (10000)
			create included_header_filenames.make (100)
			included_header_filenames.set_equality_tester (string_equality_tester)
			create included_cpp_header_filenames.make (50)
			included_cpp_header_filenames.set_equality_tester (string_equality_tester)
			create included_runtime_header_files.make (100)
			included_runtime_header_files.set_key_equality_tester (string_equality_tester)
			create included_runtime_c_files.make (100)
			included_runtime_c_files.set_equality_tester (string_equality_tester)
			create c_filenames.make_map (100)
			c_filenames.set_key_equality_tester (string_equality_tester)
			make_rescue_data
			make_volatile_data
			make_external_regexps
		ensure
			current_dynamic_system_set: current_dynamic_system = a_system
			system_processor_set: system_processor = a_system_processor
		end

feature -- Access

	current_system: ET_SYSTEM
			-- Surrounding Eiffel system
			-- (Note: there is a frozen feature called `system' in
			-- class GENERAL of SmartEiffel 1.0)
		do
			Result := current_dynamic_system.current_system
		ensure
			current_system_not_void: Result /= Void
		end

	current_dynamic_system: ET_DYNAMIC_SYSTEM
			-- Surrounding Eiffel dynamic system

	system_processor: ET_SYSTEM_PROCESSOR
			-- System processor currently used

feature -- Error handling

	error_handler: ET_ERROR_HANDLER
			-- Error handler
		do
			Result := system_processor.error_handler
		ensure
			error_handler_not_void: Result /= Void
		end

feature -- Status report

	short_names: BOOLEAN
			-- Should short names be generated for type and feature names?

	has_fatal_error: BOOLEAN
			-- Has a fatal error occurred when generating `current_dynamic_system'?

feature -- Measurement

	never_void_target_count: INTEGER
			-- Number of calls whose target is guaranteed to be non-Void at compilation time

	can_be_void_target_count: INTEGER
			-- Number of calls for which it was not possible to guarantee at compilation
			-- time whether its target would never be Void

feature -- Compilation options

	console_application_mode: BOOLEAN
			-- Should the generated application be a console application
			-- (or a Windows GUI application)?
		do
			Result := current_system.console_application_mode
		end

	multithreaded_mode: BOOLEAN
			-- Should the generated application be thread-capable?
		do
			Result := current_system.multithreaded_mode
		end

	scoop_mode: BOOLEAN
			-- Should the generated application be SCOOP-capable?
		do
			Result := current_system.scoop_mode
		end

	use_threads: BOOLEAN
			-- Does the generated application use threads?
			-- True in multithreaded or SCOOP modes.
		do
			Result := multithreaded_mode or use_scoop
		end

	use_scoop: BOOLEAN
			-- Should the generated application use SCOOP?
			-- It needs to be SCOOP-capable and to have some separate creations.
		do
			Result := scoop_mode and current_dynamic_system.has_separate_creation
		end

	scoop_exceptions_stop_when_dirty: BOOLEAN
			-- Should the application stop when a SCOOP region becomes dirty after an unhandled exception?

	check_for_void_target_mode: BOOLEAN
			-- Should the attachment status of the target of qualified calls
			-- be checked at runtime?
		do
			Result := current_system.check_for_void_target_mode
		end

	exception_trace_mode: BOOLEAN
			-- Should the generated application be able to provide an exception trace?
			-- An exception trace is the execution path from the root creation procedure
			-- to the feature where an exception occurred.
		do
			Result := current_system.exception_trace_mode
		end

	total_order_on_reals_mode: BOOLEAN
			-- Should NaN values will be lower than any other real values,
			-- and comparing NaN with another NaN will yield True and not False
			-- as usually done in IEEE arithmetic?
		do
			Result := current_system.total_order_on_reals_mode
		end

	line_generation_mode: BOOLEAN
			-- Should information about positions of constructs in the Eiffel code
			-- be included in the generated C code (e.g. using #line clauses)?
		do
			Result := current_system.line_generation_mode
		end

	current_in_exception_trace: BOOLEAN = False
			-- Should address of current object be displayed when generating an exception trace?

	finalize_mode: BOOLEAN
			-- Compilation with optimizations turned on?

	split_mode: BOOLEAN
			-- Should generated C code be split over several C files
			-- instead of being held in a single possibly large C file?
			-- When in split mode, the next C construct will be generated
			-- in a new C file if the current C file is already larger
			-- than `split_threshold' bytes.

	split_threshold: INTEGER
			-- When in split mode, the next C construct will be generated
			-- in a new C file if the current C file is already larger
			-- than `split_threshold' bytes.

	trace_mode: BOOLEAN
			-- Should the generated application be compiled with trace turned on?
			-- The trace is displayed each time the execution enters or exits
			-- from a feature.
		do
			Result := current_system.trace_mode
		end

	use_boehm_gc: BOOLEAN
			-- Should the application be compiled with the Boehm GC?

feature -- Compilation options setting

	set_finalize_mode (b: BOOLEAN)
			-- Set `finalize_mode' to `b'.
		do
			finalize_mode := b
		ensure
			finalize_mode_set: finalize_mode = b
		end

	set_split_mode (b: BOOLEAN)
			-- Set `split_mode' to `b'.
		do
			split_mode := b
		ensure
			split_mode_set: split_mode = b
		end

	set_split_threshold (v: INTEGER)
			-- Set `split_threshold' to `v'.
		require
			v_positive: v > 0
		do
			split_threshold := v
		ensure
			split_threshold_set: split_threshold = v
		end

	set_use_boehm_gc (b: BOOLEAN)
			-- Set `use_boehm_gc' to `b'.
		do
			use_boehm_gc := b
		ensure
			use_boehm_gc_set: use_boehm_gc = b
		end

	set_scoop_exceptions_stop_when_dirty (b: BOOLEAN)
			-- Set `scoop_exceptions_stop_when_dirty' to `b'.
		do
			scoop_exceptions_stop_when_dirty := b
		ensure
			scoop_exceptions_stop_when_dirty_set: scoop_exceptions_stop_when_dirty = b
		end

feature -- Generation

	generate (a_system_name: STRING)
			-- Generate C code and C compilation script file for `current_dynamic_system'.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_system_name_not_void: a_system_name /= Void
		local
			l_variables: like c_config
		do
			has_fatal_error := False
			l_variables := c_config
			if use_boehm_gc then
				generate_boehm_gc_c_files (a_system_name)
			end
			generate_external_configuration (a_system_name, l_variables)
			generate_c_code (a_system_name)
			generate_compilation_script (a_system_name, l_variables)
			c_filenames.wipe_out
		end

feature {NONE} -- Compilation script generation

	generate_compilation_script (a_system_name: STRING; a_variables: like c_config)
			-- Generate script to compile the C code.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_system_name_not_void: a_system_name /= Void
			a_variables_not_void: a_variables /= Void
			no_void_variable: not a_variables.has_void_item
			cc_defined: a_variables.has ("cc")
			link_defined: a_variables.has ("link")
			exe_defined: a_variables.has ("exe")
			obj_defined: a_variables.has ("obj")
		local
			l_base_name: STRING
			l_env_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_script_filename: STRING
			l_file: KL_TEXT_OUTPUT_FILE
			l_link_template: STRING
			l_command_name: STRING
			l_rc_template: STRING
			l_rc_filename: STRING
			l_res_filename: detachable STRING
			l_make_template: STRING
			i, nb: INTEGER
			l_cursor: DS_HASH_TABLE_CURSOR [STRING, STRING]
			l_external_make_pathnames: DS_ARRAYED_LIST [STRING]
			l_pathname: STRING
			l_replacement: STRING
			l_external_resource_pathnames: DS_ARRAYED_LIST [STRING]
			l_cc_template: STRING
			l_filename: STRING
			l_obj_filenames: STRING
			l_obj: STRING
			l_windows_variables: DS_STRING_HASH_TABLE
		do
			l_base_name := a_system_name
			if operating_system.is_windows then
				l_script_filename := l_base_name + bat_file_extension
			else
				l_script_filename := l_base_name + sh_file_extension
			end
			create l_env_regexp.make
			l_env_regexp.compile ("\$\(([^)]+)\)")
			l_script_filename := file_system.pathname (c_folder, l_script_filename)
			create l_file.make (l_script_filename)
			l_file.recursive_open_write
			if l_file.is_open_write then
				if operating_system.is_windows then
					l_file.put_line ("@echo off")
						-- Set some environment variables.
					create l_windows_variables.make (10)
					if attached gobo_variables.gobo_value as l_gobo_value and then not l_gobo_value.is_empty then
						l_file.put_line ("set " + gobo_variables.gobo_variable + "=" + l_gobo_value)
						l_windows_variables.force ("%%GOBO%%", "GOBO")
					end
					if attached gobo_variables.zig_value as l_zig_value and then not l_zig_value.is_empty then
						l_file.put_line ("set " + gobo_variables.zig_variable + "=" + l_zig_value)
						l_windows_variables.force ("%%ZIG%%", "ZIG")
					end
					if use_boehm_gc and then attached gobo_variables.boehm_gc_value as l_boehm_gc_value and then not l_boehm_gc_value.is_empty then
						l_file.put_line ("set " + gobo_variables.boehm_gc_variable + "=" + l_boehm_gc_value)
						l_windows_variables.force ("%%BOEHM_GC%%", "BOEHM_GC")
					end
				else
					l_file.put_line ("#!/bin/sh")
						-- Set some environment variables only if they are not defined yet.
						-- (See https://tldp.org/LDP/Bash-Beginners-Guide/html/sect_10_03.html#sect_10_03_03)
					if attached gobo_variables.gobo_value as l_gobo_value and then not l_gobo_value.is_empty then
						l_file.put_line ("export " + gobo_variables.gobo_variable + "=${" + gobo_variables.gobo_variable + ":-" + l_gobo_value + "}")
					end
					if attached gobo_variables.zig_value as l_zig_value and then not l_zig_value.is_empty then
						l_file.put_line ("export " + gobo_variables.zig_variable + "=${" + gobo_variables.zig_variable + ":-" + l_zig_value + "}")
					end
					if use_boehm_gc and then attached gobo_variables.boehm_gc_value as l_boehm_gc_value and then not l_boehm_gc_value.is_empty then
						l_file.put_line ("export " + gobo_variables.boehm_gc_variable + "=${" + gobo_variables.boehm_gc_variable + ":-" + l_boehm_gc_value + "}")
					end
					if attached ise_variables.ise_platform_value as l_ise_platform_value and then not l_ise_platform_value.is_empty then
						l_file.put_line ("export " + ise_variables.ise_platform_variable + "=${" + ise_variables.ise_platform_variable + ":-" + l_ise_platform_value + "}")
					end
				end
					-- Make files.
				a_variables.search ("make")
				if a_variables.found then
					l_make_template := a_variables.found_item
					l_external_make_pathnames := current_system.external_make_pathnames
					nb := l_external_make_pathnames.count
					from i := 1 until i > nb loop
						l_pathname := l_external_make_pathnames.item (i)
						l_env_regexp.match (l_pathname)
						l_replacement := STRING_.new_empty_string (l_pathname, 6)
						l_replacement.append_string ("${\1\}")
						l_pathname := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
						if not l_pathname.starts_with ("%"") then
							l_pathname := "%"" + l_pathname + "%""
						end
						a_variables.force (l_pathname, "makefile")
						l_command_name := a_variables.expanded_string (l_make_template, False)
						l_file.put_line (l_command_name)
						i := i + 1
					end
				end
				l_obj_filenames := a_variables.value ("objs")
				if l_obj_filenames = Void then
					create l_obj_filenames.make (100)
					a_variables.force (l_obj_filenames, "objs")
				end
					-- Compile files in reverse order so that it looks like
					-- a countdown since the filenames are numbered.
				l_cc_template := a_variables.item ("cc")
				l_obj := a_variables.item ("obj")
				l_cursor := c_filenames.new_cursor
				from l_cursor.finish until l_cursor.before loop
					if not l_obj_filenames.is_empty then
						l_obj_filenames.append_character (' ')
					end
					l_filename := l_cursor.key
					l_obj_filenames.append_string (l_filename)
					l_obj_filenames.append_string (l_obj)
					l_filename := l_filename + l_cursor.item
					a_variables.force (l_filename, "c")
					l_command_name := a_variables.expanded_string (l_cc_template, False)
					if l_windows_variables /= Void then
						l_command_name := l_windows_variables.expanded_string (l_command_name, False)
					end
					l_file.put_line (l_command_name)
					l_cursor.back
				end
					-- Resource files.
				a_variables.search ("rc")
				if a_variables.found then
					l_rc_template := a_variables.found_item
					l_external_resource_pathnames := current_system.external_resource_pathnames
					nb := l_external_resource_pathnames.count
					from i := 1 until i > nb loop
						l_pathname := l_external_resource_pathnames.item (i)
						l_env_regexp.match (l_pathname)
						l_replacement := STRING_.new_empty_string (l_pathname, 6)
						l_replacement.append_string ("${\1\}")
						l_pathname := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
						if file_system.has_extension (l_pathname, resx_file_extension) then
								-- Skip .Net resource files.
							l_res_filename := Void
						elseif file_system.has_extension (l_pathname, res_file_extension) then
							l_res_filename := l_pathname
						else
							l_rc_filename := l_pathname
							if file_system.has_extension (l_pathname, rc_file_extension) then
								l_res_filename := l_pathname.twin
								l_res_filename.remove_tail (rc_file_extension.count)
								l_res_filename.append_string (res_file_extension)
							else
								l_res_filename := l_pathname.twin
								l_res_filename.append_string (res_file_extension)
							end
							a_variables.force (l_rc_filename, "rc_file")
							a_variables.force (l_res_filename, "res_file")
							l_command_name := a_variables.expanded_string (l_rc_template, False)
							if l_windows_variables /= Void then
								l_command_name := l_windows_variables.expanded_string (l_command_name, False)
							end
							l_file.put_line (l_command_name)
						end
						if l_res_filename /= Void then
							if not l_obj_filenames.is_empty then
								l_obj_filenames.append_character (' ')
							end
							if l_res_filename.starts_with ("%"") then
								l_obj_filenames.append_string (l_res_filename)
							else
								l_obj_filenames.append_character ('%"')
								l_obj_filenames.append_string (l_res_filename)
								l_obj_filenames.append_character ('%"')
							end
						end
						i := i + 1
					end
					l_rc_filename :=  l_base_name + rc_file_extension
					if file_system.file_exists (l_rc_filename) then
						l_rc_filename := file_system.pathname (file_system.relative_parent_directory, l_rc_filename)
						l_res_filename := l_base_name + res_file_extension
						a_variables.force (l_rc_filename, "rc_file")
						a_variables.force (l_res_filename, "res_file")
						l_command_name := a_variables.expanded_string (l_rc_template, False)
						if l_windows_variables /= Void then
							l_command_name := l_windows_variables.expanded_string (l_command_name, False)
						end
						l_file.put_line (l_command_name)
						if not l_obj_filenames.is_empty then
							l_obj_filenames.append_character (' ')
						end
						l_obj_filenames.append_string (l_res_filename)
					end
				end
				l_link_template := a_variables.item ("link")
				l_command_name := a_variables.expanded_string (l_link_template, False)
				if l_windows_variables /= Void then
					l_command_name := l_windows_variables.expanded_string (l_command_name, False)
				end
				l_file.put_line (l_command_name)
				l_file.close
					-- Set executable mode.
				l_file.change_mode (0c777)
			else
				set_fatal_error
				report_cannot_write_error (l_script_filename)
			end
		end

	c_compiler_used: STRING
			-- C compiler used
		local
			l_filename: STRING
		do
			if attached Execution_environment.variable_value ("GOBO_CC") as l_gobo_cc and then not l_gobo_cc.is_empty then
				Result := l_gobo_cc
			else
				l_filename := file_system.nested_pathname ("${GOBO}", <<"tool", "gec", "backend", "c", "config", "default.cfg">>)
				l_filename := Execution_environment.interpreted_string (l_filename)
				Result := file_system.file_first_line (l_filename)
				if Result.is_empty then
					Result := "zig"
				end
			end
		ensure
			c_compiler_used_not_void: Result /= Void
		end

	c_config: DS_STRING_HASH_TABLE
			-- C compiler configuration
		local
			l_filename: STRING
			l_file: KL_TEXT_INPUT_FILE
			l_c_config_parser: UT_CONFIG_PARSER
			l_cursor: DS_HASH_TABLE_CURSOR [STRING, STRING]
		do
			create Result.make_map (10)
			Result.set_key_equality_tester (string_equality_tester)
			Result.put_new ("$ZIG cc -c $c", "cc")
			Result.put_new ("$ZIG cc -o $exe $objs $libs", "link")
				-- Put some platform-dependent default values.
			if operating_system.is_windows then
				Result.put_new (".obj", "obj")
				Result.put_new (".exe", "exe")
			else
				Result.put_new (".o", "obj")
				Result.put_new ("", "exe")
			end
			l_filename := file_system.nested_pathname ("${GOBO}", <<"tool", "gec", "backend", "c", "config", c_compiler_used + cfg_file_extension>>)
			l_filename := Execution_environment.interpreted_string (l_filename)
			create l_file.make (l_filename)
			l_file.open_read
			if l_file.is_open_read then
				create l_c_config_parser.make (error_handler)
				if operating_system.is_windows then
					l_c_config_parser.define_value ("True", c_eif_windows)
				elseif {PLATFORM}.is_unix then
					l_c_config_parser.define_value ("True", c_eif_linux)
				end
				if not finalize_mode then
					l_c_config_parser.define_value ("True", "EIF_WORKBENCH")
				end
				if use_boehm_gc then
					l_c_config_parser.define_value ("True", c_ge_use_boehm_gc)
				end
				if console_application_mode then
					l_c_config_parser.define_value ("True", "EIF_CONSOLE")
				end
				if use_threads then
					l_c_config_parser.define_value ("True", c_ge_use_threads)
				end
				if use_scoop then
					l_c_config_parser.define_value ("True", c_ge_use_scoop)
				end
				l_c_config_parser.parse_file (l_file)
				l_file.close
				if l_c_config_parser.has_error then
					set_fatal_error
				else
					l_cursor := l_c_config_parser.config_values.new_cursor
					from l_cursor.start until l_cursor.after loop
						Result.force (l_cursor.item, l_cursor.key)
						l_cursor.forth
					end
				end
			else
				set_fatal_error
				report_cannot_read_error (l_filename)
			end
		ensure
			c_config_not_void: Result /= Void
			no_void_variable: not Result.has_void_item
			cc_defined: Result.has ("cc")
			link_defined: Result.has ("link")
			exe_defined: Result.has ("exe")
			obj_defined: Result.has ("obj")
		end

feature {NONE} -- Generate external C files

	generate_external_configuration (a_system_name: STRING; a_variables: like c_config)
			-- Generate C compilation script file for `current_dynamic_system'.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_system_name_not_void: a_system_name /= Void
			a_variables_not_void: a_variables /= Void
			no_void_variable: not a_variables.has_void_item
			cc_defined: a_variables.has ("cc")
			link_defined: a_variables.has ("link")
			exe_defined: a_variables.has ("exe")
			obj_defined: a_variables.has ("obj")
		local
			l_obj_filenames: STRING
			l_base_name: STRING
			i, nb: INTEGER
			l_env_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_com_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_com_runtime_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_curl_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_libpng_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_net_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_vision2_windows_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_vision2_gtk3_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_wel_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_zlib_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_ise_compiler_bench_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_ise_compiler_cli_writer_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_ise_compiler_platform_regexp: RX_PCRE_REGULAR_EXPRESSION
			l_replacement: STRING
			l_external_include_pathnames: DS_ARRAYED_LIST [STRING]
			l_external_library_pathnames: DS_ARRAYED_LIST [STRING]
			l_external_object_pathnames: DS_ARRAYED_LIST [STRING]
			l_external_cflags: DS_ARRAYED_LIST [STRING]
			l_external_linker_flags: DS_ARRAYED_LIST [STRING]
			l_external_obj_filenames: STRING
			l_includes: STRING
			l_libs: STRING
			l_cflags: STRING
			l_lflags: STRING
			l_cflag: STRING
			l_lflag: STRING
			l_pathname: STRING
			l_runtime_included: BOOLEAN
			l_external_c_files_generated: BOOLEAN
			l_windows_libs_added_to_library_paths: BOOLEAN
			old_system_name: STRING
		do
			old_system_name := system_name
			system_name := a_system_name
			l_base_name := a_system_name
			create l_env_regexp.make
			l_env_regexp.compile ("\$\(([^)]+)\)")
			create l_com_regexp.make
			l_com_regexp.set_case_insensitive (True)
			l_com_regexp.compile ("(.*[\\/]library[\\/]com[\\/]).*[\\/](mt)?com\.(lib|a)")
			create l_com_runtime_regexp.make
			l_com_runtime_regexp.set_case_insensitive (True)
			l_com_runtime_regexp.compile ("(.*[\\/]library[\\/]com[\\/]).*[\\/](mt)?com_runtime\.(lib|a)")
			create l_curl_regexp.make
			l_curl_regexp.set_case_insensitive (True)
			l_curl_regexp.compile ("(.*[\\/]library[\\/]curl[\\/]).*[\\/](mt)?eiffel_curl(_static)?\.(lib|o)")
			create l_libpng_regexp.make
			l_libpng_regexp.set_case_insensitive (True)
			l_libpng_regexp.compile ("(.*[\\/]C_library[\\/]libpng)[\\/].*[\\/]libpng\.lib")
			create l_net_regexp.make
			l_net_regexp.set_case_insensitive (True)
			l_net_regexp.compile ("(.*[\\/]library[\\/]net[\\/]).*[\\/]((mt)?net\.lib|lib(mt)?net\.a)")
			create l_vision2_gtk3_regexp.make
			l_vision2_gtk3_regexp.set_case_insensitive (True)
			l_vision2_gtk3_regexp.compile ("`(.*[\\/]library[\\/]vision2[\\/]implementation[\\/]gtk3[\\/]clib)[\\/]vision2-gtk-config +(--threads +)?--object`")
			create l_vision2_windows_regexp.make
			l_vision2_windows_regexp.set_case_insensitive (True)
			l_vision2_windows_regexp.compile ("(.*[\\/]library[\\/]vision2[\\/]).*[\\/](mt)?vision2\.lib")
			create l_wel_regexp.make
			l_wel_regexp.set_case_insensitive (True)
			l_wel_regexp.compile ("(.*[\\/]library[\\/]wel[\\/]).*[\\/](mt)?wel\.lib")
			create l_zlib_regexp.make
			l_zlib_regexp.set_case_insensitive (True)
			l_zlib_regexp.compile ("(.*[\\/]C_library[\\/]zlib)[\\/].*[\\/]zlib\.lib")
			create l_ise_compiler_bench_regexp.make
			l_ise_compiler_bench_regexp.set_case_insensitive (True)
			l_ise_compiler_bench_regexp.compile (".*[\\/]Src[\\/]C[\\/]bench([\\/]((mt)?w?compiler\.lib|lib(mt)?w?compiler\.a))?")
			create l_ise_compiler_cli_writer_regexp.make
			l_ise_compiler_cli_writer_regexp.set_case_insensitive (True)
			l_ise_compiler_cli_writer_regexp.compile ("(.*[\\/]Src[\\/]framework[\\/]cli_writer[\\/]).*[\\/]cli_writer\.lib")
			create l_ise_compiler_platform_regexp.make
			l_ise_compiler_platform_regexp.set_case_insensitive (True)
			l_ise_compiler_platform_regexp.compile (".*[\\/]Src[\\/]C[\\/]platform([\\/](platform\.lib|libplatform\.a))?")
				-- Include files.
			create l_includes.make (256)
			l_external_include_pathnames := current_system.external_include_pathnames
			nb := l_external_include_pathnames.count
			if nb >= 1 then
				add_runtime_to_include_paths (l_includes)
				l_runtime_included := True
			end
			from i := 1 until i > nb loop
				l_pathname := l_external_include_pathnames.item (i)
				l_env_regexp.match (l_pathname)
				l_replacement := STRING_.new_empty_string (l_pathname, 6)
				l_replacement.append_string ("${\1\}")
				l_pathname := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
				if l_ise_compiler_bench_regexp.recognizes (l_pathname) then
					-- Ignore. Use Gobo Eiffel runtime instead of ISE Eiffel runtime.
				elseif l_ise_compiler_platform_regexp.recognizes (l_pathname) then
					-- Ignore. Use Gobo Eiffel runtime instead of ISE Eiffel runtime.
				else
					add_to_include_paths (l_pathname, l_includes)
				end
				i := i + 1
			end
			a_variables.force (l_includes, "includes")
				-- C flags.
			create l_cflags.make (256)
			a_variables.search ("cflags")
			if a_variables.found then
				l_cflags.append_string (a_variables.found_item)
			end
			l_external_cflags := current_system.external_cflags
			nb := l_external_cflags.count
			from i := 1 until i > nb loop
				l_cflag := l_external_cflags.item (i)
				l_env_regexp.match (l_cflag)
				l_replacement := STRING_.new_empty_string (l_cflag, 6)
				l_replacement.append_string ("${\1\}")
				l_cflag := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
				if not l_cflags.is_empty then
					l_cflags.append_character (' ')
				end
				l_cflags.append_string (l_cflag)
				i := i + 1
			end
			a_variables.force (l_cflags, "cflags")
				-- Linker flags.
			create l_lflags.make (256)
			create l_libs.make (256)
			a_variables.search ("lflags")
			if a_variables.found then
				l_lflags.append_string (a_variables.found_item)
			end
			if c_compiler_used.same_string ("zig") and operating_system.is_windows then
				if
					not current_system.external_linker_flags.is_empty or
					not current_system.external_library_pathnames.is_empty or
					not current_system.external_object_pathnames.is_empty
				then
					add_windows_lib_to_linker_flags (l_lflags)
				end
			end
			l_external_linker_flags := current_system.external_linker_flags
			nb := l_external_linker_flags.count
			from i := 1 until i > nb loop
				l_lflag := l_external_linker_flags.item (i)
				l_env_regexp.match (l_lflag)
				l_replacement := STRING_.new_empty_string (l_lflag, 6)
				l_replacement.append_string ("${\1\}")
				l_lflag := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
				if l_com_regexp.recognizes (l_lflag) then
					generate_external_c_files ("com", l_com_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_com_runtime_regexp.recognizes (l_lflag) then
					generate_external_c_files ("com_runtime", l_com_runtime_regexp.captured_substring (1) + "Clib_runtime")
					l_external_c_files_generated := True
				elseif l_curl_regexp.recognizes (l_lflag) then
					generate_external_c_files ("curl", l_curl_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_libpng_regexp.recognizes (l_lflag) then
					add_to_include_paths (l_libpng_regexp.captured_substring (1), l_includes)
					generate_external_c_files ("libpng", l_libpng_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_net_regexp.recognizes (l_lflag) then
					generate_external_c_files ("net", l_net_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") and operating_system.is_windows then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_vision2_gtk3_regexp.recognizes (l_lflag) then
					generate_external_c_files ("vision2_gtk3", l_vision2_gtk3_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_vision2_windows_regexp.recognizes (l_lflag) then
					generate_external_c_files ("vision2_windows", l_vision2_windows_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_wel_regexp.recognizes (l_lflag) then
					generate_external_c_files ("wel", l_wel_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_zlib_regexp.recognizes (l_lflag) then
					generate_external_c_files ("zlib", l_zlib_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_lflag.ends_with (".lib") then
					add_to_library_paths (l_lflag, l_libs)
				else
					add_to_linker_flags (l_lflag, l_lflags)
				end
				i := i + 1
			end
			a_variables.force (l_lflags, "lflags")
				-- C libraries.
			l_external_library_pathnames := current_system.external_library_pathnames
			nb := l_external_library_pathnames.count
			from i := 1 until i > nb loop
				l_pathname := l_external_library_pathnames.item (i)
				l_env_regexp.match (l_pathname)
				l_replacement := STRING_.new_empty_string (l_pathname, 6)
				l_replacement.append_string ("${\1\}")
				l_pathname := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
				if l_com_regexp.recognizes (l_pathname) then
					generate_external_c_files ("com", l_com_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_com_runtime_regexp.recognizes (l_pathname) then
					generate_external_c_files ("com_runtime", l_com_runtime_regexp.captured_substring (1) + "Clib_runtime")
					l_external_c_files_generated := True
				elseif l_curl_regexp.recognizes (l_pathname) then
					generate_external_c_files ("curl", l_curl_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_libpng_regexp.recognizes (l_pathname) then
					add_to_include_paths (l_libpng_regexp.captured_substring (1), l_includes)
					generate_external_c_files ("libpng", l_libpng_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_net_regexp.recognizes (l_pathname) then
					generate_external_c_files ("net", l_net_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") and operating_system.is_windows then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_vision2_gtk3_regexp.recognizes (l_pathname) then
					generate_external_c_files ("vision2_gtk3", l_vision2_gtk3_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_vision2_windows_regexp.recognizes (l_pathname) then
					generate_external_c_files ("vision2_windows", l_vision2_windows_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_wel_regexp.recognizes (l_pathname) then
					generate_external_c_files ("wel", l_wel_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_zlib_regexp.recognizes (l_pathname) then
					generate_external_c_files ("zlib", l_zlib_regexp.captured_substring (1))
					l_external_c_files_generated := True
				else
					add_to_library_paths (l_pathname, l_libs)
				end
				i := i + 1
			end
			a_variables.force (l_libs, "libs")
				-- Executable name.
			a_variables.force (l_base_name + a_variables.item ("exe"), "exe")
				-- Object files.
			create l_obj_filenames.make (100)
			create l_external_obj_filenames.make (100)
			l_external_object_pathnames := current_system.external_object_pathnames
			nb := l_external_object_pathnames.count
			from i := 1 until i > nb loop
				l_pathname := l_external_object_pathnames.item (i)
				l_env_regexp.match (l_pathname)
				l_replacement := STRING_.new_empty_string (l_pathname, 6)
				l_replacement.append_string ("${\1\}")
				l_pathname := Execution_environment.interpreted_string (l_env_regexp.replace_all (l_replacement))
				if l_com_regexp.recognizes (l_pathname) then
					generate_external_c_files ("com", l_com_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_com_runtime_regexp.recognizes (l_pathname) then
					generate_external_c_files ("com_runtime", l_com_runtime_regexp.captured_substring (1) + "Clib_runtime")
					l_external_c_files_generated := True
				elseif l_curl_regexp.recognizes (l_pathname) then
					generate_external_c_files ("curl", l_curl_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_libpng_regexp.recognizes (l_pathname) then
					add_to_include_paths (l_libpng_regexp.captured_substring (1), l_includes)
					generate_external_c_files ("libpng", l_libpng_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_net_regexp.recognizes (l_pathname) then
					generate_external_c_files ("net", l_net_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") and operating_system.is_windows then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_vision2_gtk3_regexp.recognizes (l_pathname) then
					generate_external_c_files ("vision2_gtk3", l_vision2_gtk3_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_vision2_windows_regexp.recognizes (l_pathname) then
					generate_external_c_files ("vision2_windows", l_vision2_windows_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_wel_regexp.recognizes (l_pathname) then
					generate_external_c_files ("wel", l_wel_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
					if not l_windows_libs_added_to_library_paths and c_compiler_used.same_string ("zig") then
						add_windows_libs_to_library_paths (l_libs)
						l_windows_libs_added_to_library_paths := True
					end
				elseif l_zlib_regexp.recognizes (l_pathname) then
					generate_external_c_files ("zlib", l_zlib_regexp.captured_substring (1))
					l_external_c_files_generated := True
				elseif l_ise_compiler_bench_regexp.recognizes (l_pathname) then
					-- Ignore. Use Gobo Eiffel runtime instead of ISE Eiffel runtime.
				elseif l_ise_compiler_cli_writer_regexp.recognizes (l_pathname) then
					generate_external_c_files ("cli_writer", l_ise_compiler_cli_writer_regexp.captured_substring (1) + "Clib")
					l_external_c_files_generated := True
				elseif l_ise_compiler_platform_regexp.recognizes (l_pathname) then
					-- Ignore. Use Gobo Eiffel runtime instead of ISE Eiffel runtime.
				else
					if not l_external_obj_filenames.is_empty then
						l_external_obj_filenames.append_character (' ')
					end
					if l_pathname.starts_with ("%"") then
						l_external_obj_filenames.append_string (l_pathname)
					else
						l_external_obj_filenames.append_character ('%"')
						l_external_obj_filenames.append_string (l_pathname)
						l_external_obj_filenames.append_character ('%"')
					end
				end
				i := i + 1
			end
			if l_external_c_files_generated then
				if not l_runtime_included then
						-- Add '$GOBO/backend/c/runtime' to the list of include paths.
						-- That way if one such include file tries to include one of the runtime
						-- 'eif_*.h' file, it can be found by the C compiler.
					add_runtime_to_include_paths (l_includes)
				end
				if c_compiler_used.same_string ("zig") or c_compiler_used.same_string ("clang") then
						-- Avoid warnings of the form:
						-- ~~~~~~~~~~~~
						-- warning: passing 'EIF_POINTER' (aka 'volatile void *') to parameter of type 'const void *' discards qualifiers
						-- ~~~~~~~~~~~~~
						-- when compiling external C files.
					if not l_cflags.is_empty then
						l_cflags.append_character (' ')
					end
					l_cflags.append_string ("-Wno-incompatible-pointer-types-discards-qualifiers")
				elseif c_compiler_used.same_string ("msc") then
						-- Avoid warnings of the form:
						-- ~~~~~~~~~~~~
						-- warning C4090: 'function': different 'volatile' qualifiers
						-- ~~~~~~~~~~~~~
						-- when compiling external C files.
					if not l_cflags.is_empty then
						l_cflags.append_character (' ')
					end
					l_cflags.append_string ("-wd4090")
				end
			end
			if not l_external_obj_filenames.is_empty then
				l_obj_filenames.append_character (' ')
				l_obj_filenames.append_string (l_external_obj_filenames)
			end
			a_variables.force (l_obj_filenames, "objs")
				-- Header file.
			a_variables.force (l_base_name + h_file_extension, "header")
			system_name := old_system_name
		end

	add_to_include_paths (a_pathname: STRING; a_include_paths: STRING)
			-- Add `a_pathname' to the list of include paths `a_include_paths'.
		require
			a_pathname_not_void: a_pathname /= Void
			a_include_paths_not_void: a_include_paths /= Void
		do
			if not a_include_paths.is_empty then
				a_include_paths.append_character (' ')
			end
			a_include_paths.append_string ("-I")
			if a_pathname.starts_with ("%"") then
				a_include_paths.append_string (a_pathname)
			else
				a_include_paths.append_character ('%"')
				a_include_paths.append_string (a_pathname)
				a_include_paths.append_character ('%"')
			end
		end

	add_runtime_to_include_paths (a_include_paths: STRING)
			-- Add '$GOBO/backend/c/runtime' to the list of include paths `a_include_paths'.
			-- Needed when at least one include path has been specified in the ECF file.
			-- That way if one such include file tries to include one of the runtime
			-- 'eif_*.h' file, it can be found by the C compiler.
		require
			a_include_paths_not_void: a_include_paths /= Void
		local
			l_pathname: STRING
		do
			l_pathname := file_system.nested_pathname ("${GOBO}", <<"tool", "gec", "backend", "c", "runtime">>)
			l_pathname := Execution_environment.interpreted_string (l_pathname)
			add_to_include_paths (l_pathname, a_include_paths)
		end

	add_to_library_paths (a_pathname: STRING; a_library_paths: STRING)
			-- Add `a_pathname' to the list of library paths `a_library_paths'.
		require
			a_pathname_not_void: a_pathname /= Void
			a_library_paths_not_void: a_library_paths /= Void
		local
			l_pathname: STRING
		do
			l_pathname := a_pathname
			if c_compiler_used.same_string ("zig") and operating_system.is_windows then
					-- There is a bug in zig 0.14 where the -L linker flag is
					-- not taken into account to search for lib files.
				l_pathname := windows_lib_full_pathname (l_pathname)
			end
			if not a_library_paths.has_substring (l_pathname) then
				if not a_library_paths.is_empty then
					a_library_paths.append_character (' ')
				end
				if l_pathname.starts_with ("%"") then
					a_library_paths.append_string (l_pathname)
				else
					a_library_paths.append_character ('%"')
					a_library_paths.append_string (l_pathname)
					a_library_paths.append_character ('%"')
				end
			end
		end

	add_windows_libs_to_library_paths (a_library_paths: STRING)
			-- Add Windows standard libraries to the list of library paths `a_library_paths'.
		require
			a_library_paths_not_void: a_library_paths /= Void
		do
			add_to_library_paths ("USER32.lib", a_library_paths)
			add_to_library_paths ("WS2_32.lib", a_library_paths)
			add_to_library_paths ("ADVAPI32.lib", a_library_paths)
			add_to_library_paths ("GDI32.lib", a_library_paths)
			add_to_library_paths ("SHELL32.lib", a_library_paths)
			add_to_library_paths ("MSIMG32.lib", a_library_paths)
			add_to_library_paths ("COMDLG32.lib", a_library_paths)
			add_to_library_paths ("UUID.lib", a_library_paths)
			add_to_library_paths ("OLE32.lib", a_library_paths)
			add_to_library_paths ("OLEAUT32.lib", a_library_paths)
			add_to_library_paths ("COMCTL32.lib", a_library_paths)
			add_to_library_paths ("MPR.lib", a_library_paths)
			add_to_library_paths ("SHLWAPI.lib", a_library_paths)
			add_to_library_paths ("WINSPOOL.lib", a_library_paths)
		end

	add_to_linker_flags (a_lflag: STRING; a_lflags: STRING)
			-- Add `a_lflag' to the list of linker flags `a_lflags'.
		require
			a_lflag_not_void: a_lflag /= Void
			a_lflags_not_void: a_lflags /= Void
		do
			if not a_lflags.is_empty then
				a_lflags.append_character (' ')
			end
			a_lflags.append_string (a_lflag)
		end

	add_windows_lib_to_linker_flags (a_lflags: STRING)
			-- Add Windows standard library search paths to the list of linker flags `a_lflags'.
		require
			a_lflags_not_void: a_lflags /= Void
		local
			l_splitter: ST_SPLITTER
		do
			if attached Execution_environment.variable_value ("LIB") as l_lib and then not l_lib.is_empty then
				create l_splitter.make_with_separators (";")
				across l_splitter.split (l_lib) as l_pathname loop
					add_to_linker_flags ("-L%"" + l_pathname + "%"", a_lflags)
				end
			end
		end

	windows_lib_full_pathname (a_pathname: STRING): STRING
			-- Add Windows library search path to `a_pathname' when possible.
			-- This is a workaround for a bug in zig 0.14 where the -L linker
			-- flag is not taken into account to search for lib files.
		require
			a_pathname_not_void: a_pathname /= Void
		local
			l_splitter: ST_SPLITTER
			l_pathnames: DS_LIST [STRING]
			l_full_pathname: STRING
		do
			Result := a_pathname
			if not a_pathname.as_lower.ends_with (".lib") then
				-- This is not a lib file.
			elseif a_pathname.has ('$') or a_pathname.has ('/') or a_pathname.has ('\') then
				-- Do not add library search path.
			elseif not attached Execution_environment.variable_value ("LIB") as l_lib or else l_lib.is_empty then
				-- No library search path.
			else
				create l_splitter.make_with_separators (";")
				l_pathnames := l_splitter.split (l_lib)
				from l_pathnames.start until l_pathnames.after loop
					l_full_pathname := file_system.pathname (l_pathnames.item_for_iteration, a_pathname)
					if file_system.file_exists (l_full_pathname) then
						Result := l_full_pathname
						l_pathnames.go_after
					else
						l_pathnames.forth
					end
				end
			end
		ensure
			windows_lib_full_pathname_not_void: Result /= Void
		end

	generate_boehm_gc_c_files (a_system_name: STRING)
			-- Generate C files for Boehm GC.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_system_name_not_void: a_system_name /= Void
			use_boehm_gc: use_boehm_gc
		local
			l_c_file: KL_TEXT_OUTPUT_FILE
			l_c_filename: STRING
			l_basename: STRING
			l_filename: STRING
		do
			if not attached gobo_variables.boehm_gc_value as l_boehm_gc_folder or else l_boehm_gc_folder.is_empty then
				set_fatal_error
				report_undefined_environment_variable_error (gobo_variables.boehm_gc_variable)
			else
				l_basename := a_system_name + "_" + (c_filenames.count + 1).out
				l_c_filename := l_basename + c_file_extension
				create l_c_file.make (file_system.pathname (c_folder, l_c_filename))
				l_c_file.recursive_open_write
				if not l_c_file.is_open_write then
					set_fatal_error
					report_cannot_write_error (l_c_filename)
				else
					c_filenames.force_last (c_file_extension, l_basename)
					l_c_file.put_line ("[
/* C files for Boehm GC. */

#if defined(WIN32) || defined(WINVER) || defined(_WIN32_WINNT) || defined(_WIN32) || defined(__WIN32__) || defined(__TOS_WIN__) || defined(_WIN_MSC_VER32)
#	define GE_WINDOWS
#elif defined(macintosh) || defined(Macintosh) || defined(__APPLE__) || defined(__MACH__)
#	define GE_MACOS
#elif defined(__OpenBSD__)
#	define GE_OPENBSD
#endif

/*
 * Workaround for crashes (illegal instruction signal) when calling
 * `memset` in Azure Devops pipelines under Windows.
 */
#if defined(GE_WINDOWS) && defined(__clang__)
#include <string.h>
extern void* GE_memset(void* str, int c, size_t n);
#define memset(x, y, z) GE_memset((x), (y), (z))
#endif

#define GC_IGNORE_WARN
#define GC_NOT_DLL
#define GC_THREADS
#define PARALLEL_MARK
#define THREAD_LOCAL_ALLOC
#define GC_ENABLE_SUSPEND_THREAD
#define LARGE_CONFIG
#define ALL_INTERIOR_POINTERS
#define ENABLE_DISCLAIM
#define GC_ATOMIC_UNCOLLECTABLE
#define GC_GCJ_SUPPORT
#define JAVA_FINALIZATION
#define NO_EXECUTE_PERMISSION
#define USE_MMAP
#define USE_MUNMAP

#if defined(GE_WINDOWS)
#	undef GC_NO_THREAD_DECLS
#	undef GC_NO_THREAD_REDIRECTS
#	define EMPTY_GETENV_RESULTS
#	define DONT_USE_USER32_DLL
#else
#	if !defined(GE_MACOS)
#		define GC_PTHREAD_START_STANDALONE
#	endif
#	ifndef _REENTRANT
#		define _REENTRANT
#	endif
#	define HANDLE_FORK
#endif

#if defined(__clang__) || defined(__GNUC__) || defined(__MINGW32__) || defined(__MINGW64__)
#	define GC_BUILTIN_ATOMIC
#endif

#if defined(__clang__)
#	define HAVE_DL_ITERATE_PHDR
#	define GC_REQUIRE_WCSDUP
#	define HAVE_DLADDR
#	define HAVE_SYS_TYPES_H
#	define HAVE_UNISTD_H
#	if defined(GE_MACOS)
#		define HAVE_PTHREAD_SETNAME_NP_WITHOUT_TID
#	elif !defined(GE_WINDOWS)
#		if defined(GE_OPENBSD)
#			define HAVE_PTHREAD_SET_NAME_NP
#		else
#			define HAVE_PTHREAD_SETNAME_NP_WITH_TID
#		endif
#		define HAVE_PTHREAD_SIGMASK
#		define NO_GETCONTEXT
#	endif
#endif
]")
					l_c_file.put_new_line
					l_filename := file_system.nested_pathname (l_boehm_gc_folder, <<"extra", "gc.c">>)
					l_filename := STRING_.replaced_all_substrings (l_filename, "\", "\\")
					l_c_file.put_string (c_include)
					l_c_file.put_character (' ')
					l_c_file.put_character ('"')
					l_c_file.put_string (l_filename)
					l_c_file.put_character ('"')
					l_c_file.put_new_line
					l_c_file.put_string (c_ifdef)
					l_c_file.put_character (' ')
					l_c_file.put_line ("GC_PTHREAD_START_STANDALONE")
					l_filename := file_system.pathname (l_boehm_gc_folder, "pthread_start.c")
					l_filename := STRING_.replaced_all_substrings (l_filename, "\", "\\")
					l_c_file.put_string (c_include)
					l_c_file.put_character (' ')
					l_c_file.put_character ('"')
					l_c_file.put_string (l_filename)
					l_c_file.put_character ('"')
					l_c_file.put_new_line
					l_c_file.put_line (c_endif)
					l_c_file.put_new_line
					l_c_file.close
				end
			end
		end

	generate_external_c_files (a_library_name, a_clib_dirname: STRING)
			-- Generate C files for `a_library_name' with the C code found in `a_clib_dirname'.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_library_name_not_void: a_library_name /= Void
			a_clib_dirname_not_void: a_clib_dirname /= Void
		local
			l_external_file: KL_TEXT_OUTPUT_FILE
			l_external_filename: detachable STRING
			l_dir: KL_DIRECTORY
			l_filename: STRING
			l_extension: STRING
			l_basename: STRING
			l_c_filenames: detachable DS_ARRAYED_LIST [STRING]
			l_cpp_filenames: detachable DS_ARRAYED_LIST [STRING]
			l_all_filenames: DS_ARRAYED_LIST [DS_ARRAYED_LIST [STRING]]
			l_filenames: detachable DS_ARRAYED_LIST [STRING]
			l_sorter: DS_QUICK_SORTER [STRING]
			l_comparator: UC_STRING_COMPARATOR
			i, nb: INTEGER
			j, nb2: INTEGER
			l_common_defines: DS_ARRAYED_LIST [STRING]
			l_common_undefines: DS_ARRAYED_LIST [STRING]
			l_common_includes: DS_ARRAYED_LIST [STRING]
			l_filenames_only_windows: detachable DS_HASH_SET [STRING]
			l_filenames_except_windows: detachable DS_HASH_SET [STRING]
			l_filenames_need_define: detachable DS_HASH_TABLE [TUPLE [first: STRING; second: detachable STRING], STRING]
			l_need_define: detachable TUPLE [first: STRING; second: detachable STRING]
			l_endif: BOOLEAN
			l_single_files: BOOLEAN
		do
			if a_library_name.same_string ("zlib") then
					-- Compile one C file at a time.
					-- Otherwise we get conflicts.
				l_single_files := True
			end
			create l_all_filenames.make (2)
			create l_dir.make (a_clib_dirname)
			l_dir.open_read
			if not l_dir.is_open_read then
				set_fatal_error
				report_cannot_read_error (a_clib_dirname)
			else
				from
					l_dir.read_entry
				until
					l_dir.end_of_input
				loop
					l_filename := l_dir.last_entry
					if l_filename.ends_with (c_file_extension) then
						if l_c_filenames = Void or l_single_files then
							create l_c_filenames.make (50)
							l_all_filenames.force_last (l_c_filenames)
						end
						l_c_filenames.force_last (l_filename)
					elseif l_filename.ends_with (cpp_file_extension) then
						if l_cpp_filenames = Void or l_single_files then
							create l_cpp_filenames.make (50)
							l_all_filenames.force_last (l_cpp_filenames)
						end
						l_cpp_filenames.force_last (l_filename)
					end
					l_dir.read_entry
				end
				l_dir.close
			end
			create l_comparator
			create l_sorter.make (l_comparator)
			create l_common_defines.make (15)
			create l_common_undefines.make (15)
			create l_common_includes.make (5)
			if use_threads then
				l_common_defines.force_last (c_ge_use_threads)
			end
			if use_scoop then
				l_common_defines.force_last (c_ge_use_scoop)
			end
			if a_library_name.same_string ("net") then
				l_common_defines.force_last ("FD_SETSIZE 256")
				l_common_undefines.force_last ("_UNICODE")
				l_common_undefines.force_last ("UNICODE")
				l_common_undefines.force_last ("DEBUG")
				include_runtime_header_file ("eif_retrieve.h", False, header_file)
				include_runtime_header_file ("eif_store.h", False, header_file)
				include_runtime_header_file ("eif_threads.h", False, header_file)
			elseif a_library_name.same_string ("zlib") then
				l_common_undefines.force_last ("_UNICODE")
				l_common_undefines.force_last ("UNICODE")
			end
			if not a_library_name.same_string ("zlib") then
				l_common_includes.force_last ("ge_eiffel.h")
			end
			if a_library_name.same_string ("com") or a_library_name.same_string ("com_runtime") then
					-- Two header files needed to compile EiffelCOM.
				l_common_includes.force_last ("eif_cecil.h")
				l_common_includes.force_last ("eif_plug.h")
			end
			nb2 := l_all_filenames.count
			from j := 1 until j > nb2 loop
				l_filenames := l_all_filenames.item (j)
				l_filenames.sort (l_sorter)
				if l_filenames.first.ends_with (c_file_extension) then
					l_extension := c_file_extension
				else
					l_extension := cpp_file_extension
				end
				l_basename := system_name + "_" + (c_filenames.count + 1).out
				l_external_filename := file_system.pathname (c_folder, l_basename + l_extension)
				create l_external_file.make (l_external_filename)
				l_external_file.recursive_open_write
				if not l_external_file.is_open_write then
					set_fatal_error
					report_cannot_write_error (l_external_filename)
				else
					c_filenames.force_last (l_extension, l_basename)
					l_external_file.put_string ("/* C files for ")
					l_external_file.put_string (a_library_name)
					l_external_file.put_string (" */")
					l_external_file.put_new_line
					l_external_file.put_new_line
					nb := l_common_defines.count
					from i := 1 until i > nb loop
						l_external_file.put_string (c_define)
						l_external_file.put_character (' ')
						l_external_file.put_line (l_common_defines.item (i))
						i := i + 1
					end
					nb := l_common_undefines.count
					from i := 1 until i > nb loop
						l_external_file.put_string (c_undef)
						l_external_file.put_character (' ')
						l_external_file.put_line (l_common_undefines.item (i))
						i := i + 1
					end
					if not l_common_defines.is_empty or not l_common_undefines.is_empty then
						l_external_file.put_new_line
					end
					nb := l_common_includes.count
					from i := 1 until i > nb loop
						l_external_file.put_string (c_include)
						l_external_file.put_character (' ')
						l_external_file.put_character ('"')
						l_external_file.put_string (l_common_includes.item (i))
						l_external_file.put_character ('"')
						l_external_file.put_new_line
						i := i + 1
					end
					if nb > 0 then
						l_external_file.put_new_line
					end
					if a_library_name.same_string ("net") then
						create l_filenames_only_windows.make_equal (3)
						l_filenames_only_windows.force_last ("ipv6win.c")
						create l_filenames_except_windows.make_equal (2)
						l_filenames_except_windows.force_last ("local.c")
						l_filenames_except_windows.force_last ("ipv6ux.c")
						create l_filenames_need_define.make (1)
						l_filenames_need_define.force_last (["SOCKETADDRESS", "SOCKETADDRESS__2"], "network.c")
					elseif a_library_name.same_string ("libpng") then
						create l_filenames_except_windows.make_equal (1)
						l_filenames_except_windows.force_last ("example.c")
					end
					nb := l_filenames.count
					from i := 1 until i > nb loop
						l_filename := l_filenames.item (i)
						if l_filenames_only_windows /= Void and then l_filenames_only_windows.has (l_filename) then
							l_endif := True
							l_external_file.put_string (c_ifdef)
							l_external_file.put_character (' ')
							l_external_file.put_line (c_eif_windows)
						elseif l_filenames_except_windows /= Void and then l_filenames_except_windows.has (l_filename) then
							l_endif := True
							l_external_file.put_string (c_ifndef)
							l_external_file.put_character (' ')
							l_external_file.put_line (c_eif_windows)
						end
						if l_filenames_need_define /= Void and then l_filenames_need_define.has (l_filename) then
							l_need_define := l_filenames_need_define.item (l_filename)
							l_external_file.put_string (c_define)
							l_external_file.put_character (' ')
							l_external_file.put_string (l_need_define.first)
							if attached l_need_define.second as l_second then
								l_external_file.put_character (' ')
								l_external_file.put_string (l_second)
							end
							l_external_file.put_new_line
						end
						l_filename := file_system.pathname (a_clib_dirname, l_filename)
						l_filename := STRING_.replaced_all_substrings (l_filename, "\", "\\")
						l_external_file.put_string (c_include)
						l_external_file.put_character (' ')
						l_external_file.put_character ('"')
						l_external_file.put_string (l_filename)
						l_external_file.put_character ('"')
						l_external_file.put_new_line
						if l_endif then
							l_endif := False
							l_external_file.put_line (c_endif)
						elseif l_need_define /= Void then
							l_external_file.put_string (c_undef)
							l_external_file.put_character (' ')
							l_external_file.put_line (l_need_define.first)
							l_need_define := Void
						end
						i := i + 1
					end
					l_external_file.close
				end
				j := j + 1
			end
		end

feature {NONE} -- C code Generation

	generate_c_code (a_system_name: STRING)
			-- Generate C code for `current_dynamic_system'.
			-- Set `has_fatal_error' if a fatal error occurred.
		require
			a_system_name_not_void: a_system_name /= Void
		local
			old_system_name: STRING
			old_file: KI_TEXT_OUTPUT_STREAM
			old_header_file: KI_TEXT_OUTPUT_STREAM
			l_header_filename: STRING
			l_header_file: KL_TEXT_OUTPUT_FILE
			l_dynamic_feature: ET_DYNAMIC_FEATURE
			l_newline_needed_in_header_file: BOOLEAN
		do
			old_system_name := system_name
			system_name := a_system_name
			l_header_filename := file_system.pathname (c_folder, a_system_name + h_file_extension)
			create l_header_file.make (l_header_filename)
			l_header_file.recursive_open_write
			if not l_header_file.is_open_write then
				set_fatal_error
				report_cannot_write_error (l_header_filename)
			else
				open_c_file
				open_cpp_file
				old_header_file := header_file
				header_file := l_header_file
				old_file := current_file
				current_file := current_function_body_buffer
				generate_ids
				if trace_mode then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_eif_trace)
					l_newline_needed_in_header_file := True
				end
				if exception_trace_mode then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_eif_exception_trace)
					l_newline_needed_in_header_file := True
				end
				if use_boehm_gc then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_ge_use_boehm_gc)
					l_newline_needed_in_header_file := True
				end
				if use_threads then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_ge_use_threads)
					l_newline_needed_in_header_file := True
				end
				if use_scoop then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_ge_use_scoop)
					l_newline_needed_in_header_file := True
				end
				if scoop_exceptions_stop_when_dirty then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_ge_scoop_exceptions_stop_when_dirty)
					l_newline_needed_in_header_file := True
				end
				if l_newline_needed_in_header_file then
					header_file.put_new_line
				end
				if current_system.any_type.base_class.universe.character_type.same_named_type_with_type_marks (current_system.any_type.base_class.universe.character_32_type, tokens.implicit_detachable_type_mark, current_system.any_type, tokens.implicit_detachable_type_mark, current_system.any_type) then
					header_file.put_string (c_define)
					header_file.put_character (' ')
					header_file.put_line (c_eif_character)
					header_file.put_character (' ')
					header_file.put_line (c_eif_character_32)
					header_file.put_new_line
				end
				include_runtime_header_file ("ge_eiffel.h", True, header_file)
				header_file.put_new_line
				if use_threads then
					include_runtime_header_file ("ge_thread_types.h", True, header_file)
					header_file.put_new_line
				end
				include_runtime_header_file ("ge_once.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_exception.h", True, header_file)
				header_file.put_new_line
				if use_threads then
					include_runtime_header_file ("ge_thread.h", True, header_file)
					header_file.put_new_line
				end
				if use_scoop then
					include_runtime_header_file ("ge_scoop.h", True, header_file)
					header_file.put_new_line
				end
				include_runtime_header_file ("ge_native_string.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_arguments.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_types.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_string.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_console.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_main.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_gc.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_signal.h", True, header_file)
				header_file.put_new_line
				include_runtime_header_file ("ge_identified.h", True, header_file)
				header_file.put_new_line
					-- Two header files needed to compile EiffelCOM.
				include_runtime_header_file ("eif_cecil.h", False, header_file)
				include_runtime_header_file ("eif_plug.h", False, header_file)
				print_start_extern_c (header_file)
				print_types (header_file)
				flush_to_c_file
				header_file.put_new_line
				print_default_declarations
				current_file.put_new_line
				flush_to_c_file
				header_file.put_new_line
				print_new_string_8_function
				print_new_string_32_function
				print_new_immutable_string_8_function
				print_new_immutable_string_32_function
				print_special_character_8_base_address_function
				print_special_character_32_base_address_function
					-- Print object allocation functions.
				print_object_allocation_functions
				print_once_per_object_data_allocation_functions
				print_once_per_object_data_dispose_functions
					-- Print Eiffel feature functions.
				if attached current_dynamic_system.root_creation_procedure as l_root_procedure then
					if attached current_dynamic_system.ise_exception_manager_init_exception_manager_feature as l_ise_exception_manager_init_exception_manager_feature then
						register_called_feature (l_ise_exception_manager_init_exception_manager_feature)
					end
					if attached current_dynamic_system.ise_exception_manager_last_exception_feature as l_ise_exception_manager_last_exception_feature then
						register_called_feature (l_ise_exception_manager_last_exception_feature)
					end
					if attached current_dynamic_system.ise_exception_manager_once_raise_feature as l_ise_exception_manager_once_raise_feature then
						register_called_feature (l_ise_exception_manager_once_raise_feature)
					end
					if attached current_dynamic_system.ise_exception_manager_set_exception_data_feature as l_ise_exception_manager_set_exception_data_feature then
						register_called_feature (l_ise_exception_manager_set_exception_data_feature)
					end
					if attached current_dynamic_system.arguments_argument_array_feature as l_arguments_argument_array_feature then
						register_called_feature (l_arguments_argument_array_feature)
					end
					register_called_feature (l_root_procedure)
					from until called_features.is_empty loop
						l_dynamic_feature := called_features.last
						called_features.remove_last
						print_feature (l_dynamic_feature)
					end
				end
					-- Print inline feature names.
				from inlined_features.start until inlined_features.after loop
					l_dynamic_feature := inlined_features.item_for_iteration
					header_file.put_string ("/* Inlined ")
					print_routine_name (l_dynamic_feature, l_dynamic_feature.target_type, header_file)
					header_file.put_character (' ')
					header_file.put_character ('-')
					header_file.put_character (' ')
					print_feature_name_in_comment (l_dynamic_feature, l_dynamic_feature.target_type, header_file)
					header_file.put_string (" */")
					header_file.put_new_line
					inlined_features.forth
				end
				inlined_features.wipe_out
					-- Print polymorphic call functions.
				print_polymorphic_query_call_functions
				print_polymorphic_tuple_label_call_functions
				print_polymorphic_procedure_call_functions
					-- Print features which build manifest arrays.
				from manifest_array_types.start until manifest_array_types.after loop
					print_manifest_array_function (manifest_array_types.item_for_iteration)
					current_file.put_new_line
					flush_to_c_file
					manifest_array_types.forth
				end
				manifest_array_types.wipe_out
					-- Print features which build big manifest arrays.
				from big_manifest_array_types.start until big_manifest_array_types.after loop
					print_big_manifest_array_function (big_manifest_array_types.item_for_iteration)
					current_file.put_new_line
					flush_to_c_file
					big_manifest_array_types.forth
				end
				big_manifest_array_types.wipe_out
					-- Print features which build manifest tuples.
				from manifest_tuple_types.start until manifest_tuple_types.after loop
					print_manifest_tuple_function (manifest_tuple_types.item_for_iteration)
					current_file.put_new_line
					flush_to_c_file
					manifest_tuple_types.forth
				end
				manifest_tuple_types.wipe_out
					-- Print all functions necessary to implement 'deep_twin'.
				print_deep_twin_functions
					-- Print constants declarations.
				print_constants_declaration
				current_file.put_new_line
				flush_to_c_file
				print_init_const_function
				current_file.put_new_line
				flush_to_c_file
					-- Print 'GE_types' array.
				print_types_array
				current_file.put_new_line
				flush_to_c_file
					-- Print 'GE_dts' dynamic type id set constants.
				print_dynamic_type_id_set_constants
				flush_to_c_file
					-- Print 'main' function.
				print_main_function
				current_file.put_new_line
				flush_to_c_file
				print_end_extern_c (header_file)
				header_file.put_new_line
					-- Include runtime C files.
				from included_runtime_c_files.start until included_runtime_c_files.after loop
					print_end_extern_c (current_file)
					include_runtime_file (included_runtime_c_files.item_for_iteration, current_file)
					print_start_extern_c (current_file)
					flush_to_c_file
					included_runtime_c_files.forth
				end
					-- Include runtime header files.
				header_file.put_new_line
				from included_runtime_header_files.start until included_runtime_header_files.after loop
					if not included_runtime_header_files.item_for_iteration then
						include_runtime_file (included_runtime_header_files.key_for_iteration, header_file)
					end
					included_runtime_header_files.forth
				end
					-- Include C++ header files.
				if not included_cpp_header_filenames.is_empty then
					header_file.put_string (c_ifdef)
					header_file.put_character (' ')
					header_file.put_line (c_cplusplus)
					from included_cpp_header_filenames.start until included_cpp_header_filenames.after loop
						header_file.put_string (c_include)
						header_file.put_character (' ')
						header_file.put_string (included_cpp_header_filenames.item_for_iteration)
						header_file.put_new_line
						included_cpp_header_filenames.forth
					end
					header_file.put_line (c_endif)
				end
					-- Include header files.
				from included_header_filenames.start until included_header_filenames.after loop
					header_file.put_string (c_include)
					header_file.put_character (' ')
					header_file.put_string (included_header_filenames.item_for_iteration)
					header_file.put_new_line
					included_header_filenames.forth
				end
				included_header_filenames.wipe_out
				included_runtime_header_files.wipe_out
				included_runtime_c_files.wipe_out
				once_features.wipe_out
				once_per_process_counts.fill_with (0)
				once_per_thread_counts.fill_with (0)
				constant_features.wipe_out
				inline_constants.wipe_out
				dispose_procedures.wipe_out
				dynamic_types := current_dynamic_system.dynamic_types
				dynamic_type_id_set_names.wipe_out
				l_header_file.close
				if
					current_dynamic_system.string_8_type.has_once_per_object_routines or
					current_dynamic_system.string_32_type.has_once_per_object_routines or
					current_dynamic_system.immutable_string_8_type.has_once_per_object_routines or
					current_dynamic_system.immutable_string_32_type.has_once_per_object_routines or
					current_dynamic_system.has_special_once_per_object_routines or
					type_info_ancestors_used or
					type_info_attributes_used or
					type_info_attribute_name_used or
					type_info_attribute_type_id_used or
					type_info_attribute_storable_type_id_used or
					type_info_attribute_dynamic_type_set_used or
					type_info_attribute_offset_used or
					type_info_attribute_size_used or
					type_info_generator_used or
					type_info_name_used or
					type_info_generic_parameters_used or
					type_info_object_size_used
				then
					file_system.rename_file (l_header_filename, l_header_filename + "ge")
					l_header_file.open_write
					if not l_header_file.is_open_write then
						set_fatal_error
						report_cannot_write_error (l_header_filename)
					else
						if current_dynamic_system.string_8_type.has_once_per_object_routines then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_has_string_8_once_per_object)
						end
						if current_dynamic_system.string_32_type.has_once_per_object_routines then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_has_string_32_once_per_object)
						end
						if current_dynamic_system.immutable_string_8_type.has_once_per_object_routines then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_has_immutable_string_8_once_per_object)
						end
						if current_dynamic_system.immutable_string_32_type.has_once_per_object_routines then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_has_immutable_string_32_once_per_object)
						end
						if current_dynamic_system.has_special_once_per_object_routines then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_has_special_once_per_object)
						end
						if type_info_ancestors_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_ancestors)
						end
						if type_info_attributes_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attributes)
						end
						if type_info_attribute_name_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_name)
						end
						if type_info_attribute_type_id_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_type_id)
						end
						if type_info_attribute_storable_type_id_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_storable_type_id)
						end
						if type_info_attribute_dynamic_type_set_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_dynamic_type_set)
						end
						if type_info_attribute_offset_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_offset)
						end
						if type_info_attribute_size_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_attribute_size)
						end
						if type_info_generator_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_type_generator)
						end
						if type_info_name_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_type_name)
						end
						if type_info_generic_parameters_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_type_generic_parameters)
						end
						if type_info_object_size_used then
							header_file.put_string (c_define)
							header_file.put_character (' ')
							header_file.put_line (c_ge_use_type_object_size)
						end
						if not l_newline_needed_in_header_file then
								-- A newline has not been written to the header file yet
								-- for the "#define USE_...". Write a newline now.
							header_file.put_new_line
						end
						include_file (l_header_filename + "ge", header_file)
						l_header_file.close
						file_system.delete_file (l_header_filename + "ge")
					end
					type_info_ancestors_used := False
					type_info_attributes_used := False
					type_info_attribute_name_used := False
					type_info_attribute_type_id_used := False
					type_info_attribute_storable_type_id_used := False
					type_info_attribute_dynamic_type_set_used := False
					type_info_attribute_offset_used := False
					type_info_attribute_size_used := False
					type_info_generator_used := False
					type_info_name_used := False
					type_info_generic_parameters_used := False
					type_info_object_size_used := False
				end
				close_c_file
				close_cpp_file
				header_file := old_header_file
				current_file := old_file
				current_type := current_dynamic_system.unknown_type
				current_feature := dummy_feature
				current_non_inlined_feature := current_feature
				current_dynamic_type_sets := extra_dynamic_type_sets
				current_index := 0
				index_offset := 0
			end
			system_name := old_system_name
		end

	generate_ids
			-- Generate types and feature ids.
		local
			l_dynamic_types: DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE]
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			i, nb: INTEGER
			l_features: ET_DYNAMIC_FEATURE_LIST
			l_feature: ET_DYNAMIC_FEATURE
			j, nb2: INTEGER
			k, nb3: INTEGER
			l_count: INTEGER
			l_id: INTEGER
		do
			if current_dynamic_system.ise_runtime_type_conforms_to_feature /= Void then
				sort_types
			end
			l_dynamic_types := dynamic_types
			nb := l_dynamic_types.count
			from i := 1 until i > nb loop
				l_type := l_dynamic_types.item (i)
				l_count := l_count + 1
				l_type.set_id (l_count)
				if l_type.is_alive or l_type.has_static then
						-- Process dynamic queries.
					l_features := l_type.queries
					nb2 := l_features.count
					from j := 1 until j > nb2 loop
						l_feature := l_features.item (j)
						l_feature.set_id (j)
						if attached l_feature.first_precursor as l_first_precursor then
							l_first_precursor.set_id (1)
							if attached l_feature.other_precursors as l_other_precursors then
								nb3 := l_other_precursors.count
								from k := 2 until k > nb3 loop
									l_other_precursors.item (k).set_id (k)
									k := k + 1
								end
							end
						end
						j := j + 1
					end
						-- Process dynamic procedures.
					l_id := nb2 + 1
					l_features := l_type.procedures
					nb2 := l_features.count
					from j := 1 until j > nb2 loop
						l_feature := l_features.item (j)
						l_feature.set_id (l_id)
						if attached l_feature.first_precursor as l_first_precursor then
							l_first_precursor.set_id (1)
							if attached l_feature.other_precursors as l_other_precursors then
								nb3 := l_other_precursors.count
								from k := 2 until k > nb3 loop
									l_other_precursors.item (k).set_id (k)
									k := k + 1
								end
							end
						end
						l_id := l_id + 1
						j := j + 1
					end
				end
				i := i + 1
			end
		end

feature {NONE} -- Feature generation

	print_features (a_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print features of `a_type' to `current_file' and its signature to `header_file'.
		require
			a_type_not_void: a_type /= Void
		local
			old_type: like current_type
			l_features: ET_DYNAMIC_FEATURE_LIST
			i, nb: INTEGER
		do
			old_type := current_type
			current_type := a_type
			l_features := a_type.queries
			nb := l_features.count
			from i := 1 until i > nb loop
				print_feature (l_features.item (i))
				i := i + 1
			end
			l_features := a_type.procedures
			nb := l_features.count
			from i := 1 until i > nb loop
				print_feature (l_features.item (i))
				i := i + 1
			end
			if attached a_type.invariants as l_invariants then
				print_feature (l_invariants)
			end
			current_type := old_type
		end

	print_feature (a_feature: ET_DYNAMIC_FEATURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
			-- Also print the precursors used in its body (calls to 'precursor').
		require
			a_feature_not_void: a_feature /= Void
		local
			old_type: like current_type
			old_feature: like current_feature
			old_dynamic_type_sets: like current_dynamic_type_sets
			old_current_index: INTEGER
			old_index_offset: INTEGER
			i, nb: INTEGER
			l_equality_types: ET_DYNAMIC_EQUALITY_TYPES
		do
			if not a_feature.is_semistrict (current_dynamic_system) then
				old_type := current_type
				current_type := a_feature.target_type
				old_feature := current_feature
				current_feature := a_feature
				current_non_inlined_feature := a_feature
				old_dynamic_type_sets := current_dynamic_type_sets
				current_dynamic_type_sets := current_feature.dynamic_type_sets
				old_current_index := current_index
				current_index := current_feature.current_index
				old_index_offset := index_offset
				index_offset := 0
				a_feature.static_feature.process (Current)
					-- Print declaration of agents used in `a_feature'.
					-- Note that we compute `current_agents.count' at each iteration
					-- because inline agents may contain other inline agents
					-- which will be added to `current_agents' on the fly.
				from i := 1 until i > current_agents.count loop
					index_offset := current_agents.item_2 (i)
					print_agent_declaration (i, current_agents.item_1 (i))
					i := i + 1
				end
				current_agents.wipe_out
					-- Print object-test functions.
				nb := current_object_tests.count
				from i := 1 until i > nb loop
					index_offset := current_object_tests.item_2 (i)
					print_object_test_function (i, current_object_tests.item_1 (i))
					i := i + 1
				end
				current_object_tests.wipe_out
					-- Print equality functions.
				nb := current_equalities.count
				from i := 1 until i > nb loop
					l_equality_types := current_equalities.item (i)
					print_equality_function (i, l_equality_types)
					if attached {ET_DYNAMIC_OBJECT_EQUALITY_TYPES} l_equality_types as l_object_equality_types then
						free_object_equality_types (l_object_equality_types)
					else
						free_equality_types (l_equality_types)
					end
					i := i + 1
				end
				current_equalities.wipe_out
				free_inlined_operands
				free_call_contexts
				if use_scoop then
						-- Reset the number of separate calls appearing in `a_feature'.
					current_separate_call_count := 0
				end
				current_dynamic_type_sets := old_dynamic_type_sets
				current_index := old_current_index
				index_offset := old_index_offset
				current_feature := old_feature
				current_non_inlined_feature := old_feature
				current_type := old_type
			end
		end

	print_address_routine (a_feature: ET_ROUTINE)
			-- Print wrapper of `a_feature' used to pass its address, to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
			is_address: current_feature.is_address
		local
			l_result_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			i, nb_args: INTEGER
			old_file: KI_TEXT_OUTPUT_STREAM
			l_name: ET_IDENTIFIER
		do
			old_file := current_file
			current_file := current_function_header_buffer
			print_feature_name_comment (current_feature, current_type, header_file)
			print_feature_name_comment (current_feature, current_type, current_file)
			l_result_type_set := current_feature.result_type_set
			if l_result_type_set /= Void then
				l_result_type := l_result_type_set.static_type.primary_type
			end
				--
				-- Print signature to `header_file' and `current_file'.
				--
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			if l_result_type /= Void then
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
			else
				header_file.put_string (c_void)
				current_file.put_string (c_void)
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_address_routine_name (current_feature, current_type, header_file)
			print_address_routine_name (current_feature, current_type, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_type_declaration (current_type, header_file)
			print_type_declaration (current_type, current_file)
			if current_type.is_expanded then
				if current_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					header_file.put_character (' ')
					header_file.put_string (c_volatile)
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				header_file.put_character ('*')
				current_file.put_character ('*')
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_current_name (header_file)
			print_current_name (current_file)
			l_arguments := a_feature.arguments
			if l_arguments /= Void then
				nb_args := l_arguments.count
				if nb_args > 0 then
					from i := 1 until i > nb_args loop
						header_file.put_character (',')
						current_file.put_character (',')
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						print_type_declaration (l_argument_type, header_file)
						print_type_declaration (l_argument_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_name := l_arguments.formal_argument (i).name
						print_argument_name (l_name, header_file)
						print_argument_name (l_name, current_file)
						i := i + 1
					end
				end
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			if l_result_type /= Void then
				current_file.put_string (c_return)
				current_file.put_character (' ')
			end
			if current_type.base_class.invariants_enabled then
				current_file.put_string (c_ge_unqualified)
				current_file.put_character ('(')
				current_file.put_string (c_ge_current_context)
				current_file.put_character ('(')
				current_file.put_character (')')
				print_comma
			end
			print_routine_name (current_feature, current_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ge_current_context)
			current_file.put_character ('(')
			current_file.put_character (')')
			print_comma
			print_current_name (current_file)
			if l_arguments /= Void and then nb_args > 0 then
				from i := 1 until i > nb_args loop
					print_comma
					l_name := l_arguments.formal_argument (i).name
					print_argument_name (l_name, current_file)
					i := i + 1
				end
			end
			current_file.put_character (')')
			print_after_call (current_type)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				--
				-- Clean up.
				--
			current_file := old_file
				-- Flush to file.
			flush_to_c_file
		end

	print_constants_declaration
			-- Print declarations of constant attributes
			-- to `header_file' and `current_file'.
		local
			l_feature: ET_FEATURE
			l_context: ET_TYPE_CONTEXT
			l_constant_type: ET_DYNAMIC_PRIMARY_TYPE
			l_constant: ET_INLINE_CONSTANT
		do
			from constant_features.start until constant_features.after loop
				l_feature := constant_features.key_for_iteration
				if not once_features.has (l_feature) then
					if not attached l_feature.type as l_type then
							-- Internal error: a constant attribute has a type.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_constants_declaration", 1, "constant attribute with no type.")
					else
							-- The deanchored form of the type of the constant feature
							-- should not contain any formal generic parameter (in fact
							-- it should be one of BOOLEAN, CHARACTER possibly sized,
							-- INTEGER possibly sized, NATURAL possible sized, REAL possibly
							-- sized, STRING possibly sized, or 'TYPE [X]' where X is a
							-- stand-alone type), therefore it is OK to use the class where
							-- this feature has been written as context.
						l_context := l_feature.implementation_class
						l_constant_type := current_dynamic_system.dynamic_primary_type (l_type, l_context)
						header_file.put_string (c_extern)
						header_file.put_character (' ')
						print_type_declaration (l_constant_type, header_file)
						print_type_declaration (l_constant_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						print_once_value_name (l_feature, header_file)
						print_once_value_name (l_feature, current_file)
						header_file.put_character (';')
						current_file.put_character (';')
						header_file.put_new_line
						current_file.put_new_line
					end
				end
				constant_features.forth
			end
			from inline_constants.start until inline_constants.after loop
				l_constant := inline_constants.key_for_iteration
				l_constant_type := inline_constants.item_for_iteration
				header_file.put_string (c_extern)
				header_file.put_character (' ')
				print_type_declaration (l_constant_type, header_file)
				print_type_declaration (l_constant_type, current_file)
				header_file.put_character (' ')
				current_file.put_character (' ')
				print_inline_constant_name (l_constant, header_file)
				print_inline_constant_name (l_constant, current_file)
				header_file.put_character (';')
				current_file.put_character (';')
				header_file.put_new_line
				current_file.put_new_line
				inline_constants.forth
			end
		end

	print_deferred_function (a_feature: ET_DEFERRED_FUNCTION)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
				-- Internal error: deferred features cannot be executed at run-time.
			set_fatal_error
			error_handler.report_giaac_error (generator, "print_deferred_function", 1, "deferred functions cannot be executed at run-time.")
		end

	print_deferred_procedure (a_feature: ET_DEFERRED_PROCEDURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
				-- Internal error: deferred features cannot be executed at run-time.
			set_fatal_error
			error_handler.report_giaac_error (generator, "print_deferred_procedure", 1, "deferred procedures cannot be executed at run-time.")
		end

	print_do_function (a_feature: ET_DO_FUNCTION)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			print_internal_function (a_feature)
		end

	print_do_procedure (a_feature: ET_DO_PROCEDURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			print_internal_procedure (a_feature)
		end

	print_external_function (a_feature: ET_EXTERNAL_FUNCTION)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_function", 1, "inconsistent `current_feature'.")
			else
				if current_feature.is_regular then
					print_external_routine (a_feature, False, False)
				end
				if current_feature.is_address then
					print_address_routine (a_feature)
				end
			end
		end

	print_external_procedure (a_feature: ET_EXTERNAL_PROCEDURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_procedure", 1, "inconsistent `current_feature'.")
			else
				if current_feature.is_regular then
					print_external_routine (a_feature, False, False)
				end
				if current_feature.is_creation then
					print_external_routine (a_feature, True, False)
				end
				if current_feature.is_address then
					print_address_routine (a_feature)
				end
			end
		end

	print_external_routine (a_feature: ET_EXTERNAL_ROUTINE; a_creation: BOOLEAN; a_inline: BOOLEAN)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
			-- `a_inline' means that `a_feature' is a C inline or C++ inline feature and
			-- we want to generate its code in a separate function.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
			is_creation: a_creation implies current_feature.is_creation
			one_kind: (not a_creation and not a_inline) or (a_creation xor a_inline)
		local
			l_result_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			l_name: ET_IDENTIFIER
			i, nb_args: INTEGER
			l_language: ET_EXTERNAL_LANGUAGE
			l_language_value: ET_MANIFEST_STRING
			l_language_string: STRING
			l_signature_arguments: detachable STRING
			l_signature_result: detachable STRING
			l_alias_value: ET_MANIFEST_STRING
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_struct_field_name: STRING
			l_struct_field_type: detachable STRING
			l_struct_type: STRING
			old_file: KI_TEXT_OUTPUT_STREAM
			l_is_inline: BOOLEAN
			l_is_cpp: BOOLEAN
			l_cpp_class_type: STRING
			l_dll_file: STRING
			l_has_include_files: BOOLEAN
		do
			old_file := current_file
			current_file := current_function_header_buffer
			if
				a_feature.implementation_class.name.same_class_name (tokens.file_class_name) and then
				a_feature.implementation_feature.name.same_feature_name (tokens.c_retrieved_feature_name)
			then
				type_info_name_used := True
				type_info_attributes_used:= True
				type_info_attribute_name_used:= True
				type_info_attribute_offset_used:= True
				type_info_attribute_size_used:= True
				type_info_attribute_type_id_used:= True
				type_info_attribute_dynamic_type_set_used := True
			elseif
				a_feature.implementation_class.name.same_class_name (tokens.file_class_name) and then
				(a_feature.implementation_feature.name.same_feature_name (tokens.c_basic_store_feature_name) or
				a_feature.implementation_feature.name.same_feature_name (tokens.c_general_store_feature_name) or
				a_feature.implementation_feature.name.same_feature_name (tokens.c_independent_store_feature_name))
			then
				type_info_generator_used := True
				type_info_generic_parameters_used := True
				type_info_attributes_used := True
				type_info_attribute_name_used := True
				type_info_attribute_offset_used := True
				type_info_attribute_size_used := True
				type_info_attribute_type_id_used := True
				type_info_attribute_storable_type_id_used := True
			end
				--
				-- Print signature to `header_file' and `current_file'.
				--
			print_feature_name_comment (current_feature, current_type, header_file)
			print_feature_name_comment (current_feature, current_type, current_file)
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			l_result_type_set := current_feature.result_type_set
			if l_result_type_set /= Void then
				l_result_type := l_result_type_set.static_type.primary_type
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
			elseif a_creation then
				print_type_declaration (current_type, header_file)
				print_type_declaration (current_type, current_file)
			else
				header_file.put_string (c_void)
				current_file.put_string (c_void)
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			if a_inline then
				print_inline_routine_name (current_feature, current_type, header_file)
				print_inline_routine_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
			elseif a_creation then
				print_creation_procedure_name (current_feature, current_type, header_file)
				print_creation_procedure_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
			else
				print_routine_name (current_feature, current_type, header_file)
				print_routine_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				if not current_feature.is_static then
					header_file.put_character (',')
					header_file.put_character (' ')
					current_file.put_character (',')
					current_file.put_character (' ')
					print_type_declaration (current_type, header_file)
					print_type_declaration (current_type, current_file)
					if current_type.is_expanded then
						if current_type.is_basic then
								-- Note that non-basic basic expanded types are already declared as volatile.
							header_file.put_character (' ')
							header_file.put_string (c_volatile)
							current_file.put_character (' ')
							current_file.put_string (c_volatile)
						end
						header_file.put_character ('*')
						current_file.put_character ('*')
					end
					header_file.put_character (' ')
					current_file.put_character (' ')
					print_current_name (header_file)
					print_current_name (current_file)
				end
			end
			l_arguments := a_feature.arguments
			if l_arguments /= Void then
				nb_args := l_arguments.count
				if nb_args > 0 then
					from i := 1 until i > nb_args loop
						header_file.put_character (',')
						current_file.put_character (',')
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						print_type_declaration (l_argument_type, header_file)
						print_type_declaration (l_argument_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_name := l_arguments.formal_argument (i).name
						print_argument_name (l_name, header_file)
						print_argument_name (l_name, current_file)
						i := i + 1
					end
				end
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				--
				-- Print body to `current_file'.
				--
			current_file.put_character ('{')
			current_file.put_new_line
			indent
				-- Determine the kind of external.
			l_language := a_feature.language
			l_language_value := l_language.manifest_string
			l_language_string := l_language_value.value
			if external_c_inline_regexp.recognizes (l_language_string) then
				l_is_inline := True
			elseif external_cpp_inline_regexp.recognizes (l_language_string) then
				l_is_inline := True
				l_is_cpp := True
			end
				--
				-- Declaration of variables.
				--
			if current_type.base_class.invariants_enabled and not current_feature.is_static then
			print_indentation
				current_file.put_string (c_uint32_t)
				current_file.put_character (' ')
				current_file.put_string (c_in_qualified_call)
				print_assign_to
				if a_creation then
					current_file.put_character ('2')
				else
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_in_qualified_call)
				end
				print_semicolon_newline
			end
				-- Variable for exception trace.
			if exception_trace_mode and not a_inline then
				print_indentation
				current_file.put_string (c_ge_call)
				current_file.put_character (' ')
				current_file.put_string (c_tc)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('{')
				if current_in_exception_trace then
					if current_feature.is_static or a_creation then
						current_file.put_character ('0')
					else
						print_current_name (current_file)
					end
					current_file.put_character (',')
				end
				print_escaped_string (current_type.base_class.upper_name)
				current_file.put_character (',')
				print_escaped_string (a_feature.lower_name)
				current_file.put_character (',')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character ('}')
				current_file.put_character (';')
				current_file.put_new_line
			end
				-- Variable for 'Result' entity.
			if l_result_type /= Void and (not l_is_inline or else ((exception_trace_mode or trace_mode) and not a_inline)) then
				print_indentation
				print_type_declaration (l_result_type, current_file)
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_default_entity_value (l_result_type, current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
				--
				-- Instructions.
				--
			current_file := current_function_body_buffer
			if not a_inline then
					-- Call stack.
				if exception_trace_mode then
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_tc_address)
					current_file.put_character (';')
					current_file.put_new_line
				end
				print_feature_trace_message_call (True)
			end
			if current_type.base_class.invariants_enabled and not current_feature.is_static and not a_creation then
				print_all_invariants (True)
			end
			if current_type.base_class.preconditions_enabled then
				print_all_preconditions (a_feature)
			end
			if current_type.base_class.postconditions_enabled then
				print_all_old_expression_declarations (a_feature)
			end
			if a_creation then
				print_malloc_current (a_feature)
				if exception_trace_mode and then current_in_exception_trace then
					print_indentation
					current_file.put_string (c_tc)
					current_file.put_character ('.')
					current_file.put_string (c_object)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_current_name (current_file)
					current_file.put_character (';')
					current_file.put_new_line
				end
			end
				-- Put the body of the feature in a block so that there
				-- is no C compilation error if some variables are
				-- declared after the instructions above.
			print_indentation
			current_file.put_character ('{')
			current_file.put_new_line
			if l_is_inline then
				if l_is_cpp then
						-- Regexp: C++ [blocking] inline [use {<include> "," ...}+]
						-- \3: include files
					if external_cpp_inline_regexp.match_count > 3 and then external_cpp_inline_regexp.captured_substring_count (3) > 0 then
						print_external_cpp_includes (external_cpp_inline_regexp.captured_substring (3))
					end
				else
						-- Regexp: C [blocking] inline [use {<include> "," ...}+]
						-- \3: include files
					if external_c_inline_regexp.match_count > 3 and then external_c_inline_regexp.captured_substring_count (3) > 0 then
						print_external_c_includes (external_c_inline_regexp.captured_substring (3))
					end
				end
				if (exception_trace_mode or trace_mode) and not a_inline then
						-- The inline C code may contain 'return' statement.
						-- So we need to enclose its code in a separate C function.
					if l_result_type_set /= Void then
						print_indentation
						print_result_name (current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
					end
					print_inline_routine_name (current_feature, current_type, current_file)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					if l_arguments /= Void then
						from i := 1 until i > nb_args loop
							current_file.put_character (',')
							current_file.put_character (' ')
							l_name := l_arguments.formal_argument (i).name
							print_argument_name (l_name, current_file)
							i := i + 1
						end
					end
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				else
					print_external_c_inline_body (a_feature)
				end
			elseif a_feature.is_builtin then
				print_external_builtin_body (a_feature)
			elseif external_c_regexp.recognizes (l_language_string) then
					-- Regexp: C [blocking] [signature ["(" {<type> "," ...}* ")"] [":" <type>]] [use {<include> "," ...}+]
					-- \5: has signature arguments
					-- \6: signature arguments
					-- \11: signature result
					-- \18: include files
				if external_c_regexp.match_count > 18 and then external_c_regexp.captured_substring_count (18) > 0 then
					l_has_include_files := True
					print_external_c_includes (external_c_regexp.captured_substring (18))
				end
				if external_c_regexp.match_count > 5 and then external_c_regexp.captured_substring_count (5) > 0 then
					l_signature_arguments := external_c_regexp.captured_substring (6)
				end
				if external_c_regexp.match_count > 11 and then external_c_regexp.captured_substring_count (11) > 0 then
					l_signature_result := external_c_regexp.captured_substring (11)
				end
				if not l_has_include_files then
						-- We need to generate a prototype for this external routine.
					print_external_c_prototype (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause)
				end
				print_external_c_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause, False)
			elseif external_c_macro_regexp.recognizes (l_language_string) then
					-- Regexp: C [blocking] macro [signature ["(" {<type> "," ...}* ")"] [":" <type>]] use {<include> "," ...}+
					-- \4: has signature arguments
					-- \5: signature arguments
					-- \10: signature result
					-- \17: include files
				print_external_c_includes (external_c_macro_regexp.captured_substring (17))
				if external_c_macro_regexp.match_count > 4 and then external_c_macro_regexp.captured_substring_count (4) > 0 then
					l_signature_arguments := external_c_macro_regexp.captured_substring (5)
				end
				if external_c_macro_regexp.match_count > 10 and then external_c_macro_regexp.captured_substring_count (10) > 0 then
					l_signature_result := external_c_macro_regexp.captured_substring (10)
				end
				print_external_c_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause, True)
			elseif external_c_struct_regexp.recognizes (l_language_string) then
					-- Regexp: C struct <struct-type> (access|get) <field-name> [type <field-type>] use {<include> "," ...}+
					-- \1: struct type
					-- \6: field name
					-- \9: field type
					-- \16: include files
				print_external_c_includes (external_c_struct_regexp.captured_substring (16))
				l_struct_type := external_c_struct_regexp.captured_substring (1)
				l_struct_field_name := external_c_struct_regexp.captured_substring (6)
				if external_c_struct_regexp.match_count > 9 and then external_c_struct_regexp.captured_substring_count (9) > 0 then
					l_struct_field_type := external_c_struct_regexp.captured_substring (9)
				end
				print_external_c_struct_body (l_arguments, l_result_type_set, l_struct_type, l_struct_field_name, l_struct_field_type)
			elseif old_external_c_regexp.recognizes (l_language_string) then
					-- Regexp: C [["(" {<type> "," ...}* ")"] [":" <type>]] ["|" {<include> "," ...}+]
					-- \1: has signature
					-- \2: has signature arguments
					-- \3: signature arguments
					-- \5: signature result
					-- \7: include files
				if old_external_c_regexp.match_count > 7 and then old_external_c_regexp.captured_substring_count (7) > 0 then
					l_has_include_files := True
					print_external_c_includes (old_external_c_regexp.captured_substring (7))
				end
				if old_external_c_regexp.match_count > 1 and then old_external_c_regexp.captured_substring_count (1) > 0 then
					if old_external_c_regexp.match_count > 2 and then old_external_c_regexp.captured_substring_count (2) > 0 then
						l_signature_arguments := old_external_c_regexp.captured_substring (3)
					end
					if old_external_c_regexp.match_count > 5 and then old_external_c_regexp.captured_substring_count (5) > 0 then
						l_signature_result := old_external_c_regexp.captured_substring (5)
					end
				end
				if not l_has_include_files then
						-- We need to generate a prototype for this external routine.
					print_external_c_prototype (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause)
				end
				print_external_c_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause, False)
			elseif old_external_c_macro_regexp.recognizes (l_language_string) then
					-- Regexp: C "[" macro <include> "]" ["(" {<type> "," ...}* ")"] [":" <type>] ["|" {<include> "," ...}+]
					-- \1: include file
					-- \2: has signature arguments
					-- \3: signature arguments
					-- \5: signature result
					-- \7: include files
				print_external_c_includes (old_external_c_macro_regexp.captured_substring (1))
				if old_external_c_macro_regexp.match_count > 7 and then old_external_c_macro_regexp.captured_substring_count (7) > 0 then
					print_external_c_includes (old_external_c_macro_regexp.captured_substring (7))
				end
				if old_external_c_macro_regexp.match_count > 2 and then old_external_c_macro_regexp.captured_substring_count (2) > 0 then
					l_signature_arguments := old_external_c_macro_regexp.captured_substring (3)
					if old_external_c_macro_regexp.match_count > 5 and then old_external_c_macro_regexp.captured_substring_count (5) > 0 then
						l_signature_result := old_external_c_macro_regexp.captured_substring (5)
					end
				end
				print_external_c_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause, True)
			elseif old_external_c_struct_regexp.recognizes (l_language_string) then
					-- Regexp: C "[" struct <include> "]" "(" {<type> "," ...}+ ")" [":" <type>]
					-- \1: include file
					-- \2: signature arguments
					-- \4: signature result
				print_external_c_includes (old_external_c_struct_regexp.captured_substring (1))
				l_signature_arguments := old_external_c_struct_regexp.captured_substring (2)
				if old_external_c_struct_regexp.match_count > 4 and then old_external_c_struct_regexp.captured_substring_count (4) > 0 then
					l_signature_result := old_external_c_struct_regexp.captured_substring (4)
				end
				if attached a_feature.alias_clause as l_alias then
					l_alias_value := l_alias.manifest_string
					l_struct_field_name := l_alias_value.value
				else
					l_struct_field_name := a_feature.implementation_feature.lower_name
				end
				create l_splitter.make_with_separators (",")
				l_list := l_splitter.split (l_signature_arguments)
				inspect l_list.count
				when 1 then
					l_struct_type := l_list.item (1)
					l_struct_field_type := l_signature_result
					print_external_c_struct_body (l_arguments, l_result_type_set, l_struct_type, l_struct_field_name, l_struct_field_type)
				when 2 then
					l_struct_type := l_list.item (1)
					l_struct_field_type := l_list.item (2)
					print_external_c_struct_body (l_arguments, l_result_type_set, l_struct_type, l_struct_field_name, l_struct_field_type)
				else
-- TODO: syntax error
				end
			elseif external_cpp_macro_regexp.recognizes (l_language_string) then
					-- Regexp: C++ [blocking] macro [signature ["(" {<type> "," ...}* ")"] [":" <type>]] use {<include> "," ...}+
					-- \4: has signature arguments
					-- \5: signature arguments
					-- \10: signature result
					-- \17: include files
				l_is_cpp := True
				print_external_cpp_includes (external_cpp_macro_regexp.captured_substring (17))
				if external_cpp_macro_regexp.match_count > 4 and then external_cpp_macro_regexp.captured_substring_count (4) > 0 then
					l_signature_arguments := external_cpp_macro_regexp.captured_substring (5)
				end
				if external_cpp_macro_regexp.match_count > 10 and then external_cpp_macro_regexp.captured_substring_count (10) > 0 then
					l_signature_result := external_cpp_macro_regexp.captured_substring (10)
				end
				print_external_c_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause, True)
			elseif external_cpp_regexp.recognizes (l_language_string) then
					-- Regexp: C++ [blocking] <class_type> [signature ["(" {<type> "," ...}* ")"] [":" <type>]] [use {<include> "," ...}+]
					-- \2: class type
					-- \5: has signature arguments
					-- \6: signature arguments
					-- \11: signature result
					-- \18: include files
				l_is_cpp := True
				if external_cpp_regexp.match_count > 18 and then external_cpp_regexp.captured_substring_count (18) > 0 then
					l_has_include_files := True
					print_external_cpp_includes (external_cpp_regexp.captured_substring (18))
				end
				l_cpp_class_type := external_cpp_regexp.captured_substring (2)
				if external_cpp_regexp.match_count > 5 and then external_cpp_regexp.captured_substring_count (5) > 0 then
					l_signature_arguments := external_cpp_regexp.captured_substring (6)
				end
				if external_cpp_regexp.match_count > 11 and then external_cpp_regexp.captured_substring_count (11) > 0 then
					l_signature_result := external_cpp_regexp.captured_substring (11)
				end
				if not l_has_include_files then
						-- We need to generate a prototype for this external routine.
					print_external_c_prototype (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_signature_arguments, l_signature_result, a_feature.alias_clause)
				end
				print_external_cpp_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_cpp_class_type, l_signature_arguments, l_signature_result, a_feature.alias_clause)
			elseif external_cpp_struct_regexp.recognizes (l_language_string) then
					-- Regexp: C++ struct <struct-type> (access|get) <field-name> [type <field-type>] use {<include> "," ...}+
					-- \1: struct type
					-- \6: field name
					-- \9: field type
					-- \16: include files
				l_is_cpp := True
				print_external_cpp_includes (external_cpp_struct_regexp.captured_substring (16))
				l_struct_type := external_cpp_struct_regexp.captured_substring (1)
				l_struct_field_name := external_cpp_struct_regexp.captured_substring (6)
				if external_cpp_struct_regexp.match_count > 9 and then external_cpp_struct_regexp.captured_substring_count (9) > 0 then
					l_struct_field_type := external_cpp_struct_regexp.captured_substring (9)
				end
				print_external_c_struct_body (l_arguments, l_result_type_set, l_struct_type, l_struct_field_name, l_struct_field_type)
			elseif external_dllwin_regexp.recognizes (l_language_string) then
					-- Regexp: [blocking] dllwin <dll_file> [signature ["(" {<type> "," ...}* ")"] [":" <type>]] [use {<include> "," ...}+]
					-- \2: dll file
					-- \5: has signature arguments
					-- \6: signature arguments
					-- \11: signature result
					-- \18: include files
				if external_dllwin_regexp.match_count > 18 and then external_dllwin_regexp.captured_substring_count (18) > 0 then
					print_external_c_includes (external_dllwin_regexp.captured_substring (18))
				end
				l_dll_file := external_dllwin_regexp.captured_substring (2)
				if external_dllwin_regexp.match_count > 5 and then external_dllwin_regexp.captured_substring_count (5) > 0 then
					l_signature_arguments := external_dllwin_regexp.captured_substring (6)
				end
				if external_dllwin_regexp.match_count > 11 and then external_dllwin_regexp.captured_substring_count (11) > 0 then
					l_signature_result := external_dllwin_regexp.captured_substring (11)
				end
				print_external_dllwin_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_dll_file, l_signature_arguments, l_signature_result, a_feature.alias_clause)
			elseif old_external_dllwin32_regexp.recognizes (l_language_string) then
					-- Regexp: C "[" dllwin32 <dll_file> "]" ["(" {<type> "," ...}* ")"] [":" <type>]
					-- \1: dll file
					-- \2: has signature arguments
					-- \3: signature arguments
					-- \5: signature result
				l_dll_file := old_external_dllwin32_regexp.captured_substring (1)
				if old_external_dllwin32_regexp.match_count > 2 and then old_external_dllwin32_regexp.captured_substring_count (2) > 0 then
					l_signature_arguments := old_external_dllwin32_regexp.captured_substring (3)
					if old_external_dllwin32_regexp.match_count > 5 and then old_external_dllwin32_regexp.captured_substring_count (5) > 0 then
						l_signature_result := old_external_dllwin32_regexp.captured_substring (5)
					end
				end
				print_external_dllwin_body (a_feature.implementation_feature.name, l_arguments, l_result_type_set, l_dll_file, l_signature_arguments, l_signature_result, a_feature.alias_clause)
			else
error_handler.report_warning_message ("**** language not recognized: " + l_language_string)
			end
				-- Close the block containing the body of the feature.
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if current_type.base_class.invariants_enabled and not current_feature.is_static then
				print_all_invariants (False)
			end
			if current_type.base_class.postconditions_enabled then
				print_all_postconditions (a_feature)
			end
			if not a_inline then
				print_feature_trace_message_call (False)
					-- Call stack.
				if exception_trace_mode then
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_tc)
					current_file.put_character ('.')
					current_file.put_string (c_caller)
					current_file.put_character (';')
					current_file.put_new_line
				end
			end
			if l_result_type /= Void and (not l_is_inline or else ((exception_trace_mode or trace_mode) and not a_inline)) then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			elseif a_creation then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_current_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			if l_is_cpp then
				flush_to_cpp_file
			else
				flush_to_c_file
			end
				--
				-- Clean up.
				--
			current_file := old_file
			if (exception_trace_mode or trace_mode) and l_is_inline and not a_inline then
				print_external_routine (a_feature, False, True)
			end
		end

	print_external_builtin_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			valid_feature: current_feature.static_feature = a_feature
		local
			i, nb: INTEGER
			l_name: ET_IDENTIFIER
			l_argument_type: ET_DYNAMIC_TYPE
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			l_argument: ET_FORMAL_ARGUMENT
			old_file: KI_TEXT_OUTPUT_STREAM
			l_has_separate_arguments: BOOLEAN
			l_separate_arguments_count: INTEGER
		do
			old_file := current_file
			current_file := current_function_header_buffer
				--
				-- Declaration of variables.
				--
			if use_scoop then
					-- Declaration of SCOOP sessions to register separate calls.
				l_arguments := a_feature.arguments
				if l_arguments /= Void then
					nb := l_arguments.count
					from i := 1 until i > nb loop
						l_argument := l_arguments.formal_argument (i)
						l_name := l_argument.name
						l_argument_type := dynamic_type_set (l_name).static_type
						if l_argument_type.is_separate then
							l_has_separate_arguments := True
							l_separate_arguments_count := l_separate_arguments_count + 1
							print_indentation
							current_file.put_string (c_ge_scoop_session)
							current_file.put_character ('*')
							current_file.put_character (' ')
							print_separate_argument_session_name (l_name, current_file)
							current_file.put_character (' ')
							current_file.put_character ('=')
							current_file.put_character (' ')
							current_file.put_character ('0')
							print_semicolon_newline
						end
						i := i + 1
					end
				end
				if l_has_separate_arguments then
					print_indentation
					current_file.put_string (c_ge_scoop_region)
					current_file.put_character ('*')
					current_file.put_character (' ')
					current_file.put_string (c_sr)
					print_assign_to
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_region)
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_scoop_condition)
					current_file.put_character ('*')
					current_file.put_character (' ')
					current_file.put_string (c_scond)
					print_assign_to
					current_file.put_character ('0')
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_rescue)
					current_file.put_character (' ')
					current_file.put_character ('r')
					current_file.put_character (';')
					current_file.put_new_line
				end
			end
				--
				-- Instructions.
				--
			current_file := current_function_body_buffer
			if l_has_separate_arguments then
					-- Start the current SCOOP sessions.
				if l_separate_arguments_count > 1 then
					print_indentation
					current_file.put_string (c_ge_scoop_multisessions_open_start)
					current_file.put_character ('(')
					current_file.put_character (')')
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_scond)
					print_assign_to
					current_file.put_string (c_ge_new_scoop_condition)
					current_file.put_character ('(')
					current_file.put_integer (l_separate_arguments_count)
					current_file.put_character (')')
					print_semicolon_newline
				end
				check l_has_separate_arguments: l_arguments /= Void then end
				nb := l_arguments.count
				from i := 1 until i > nb loop
					l_argument := l_arguments.formal_argument (i)
					l_name := l_argument.name
					l_argument_type := dynamic_type_set (l_name).static_type
					if l_argument_type.is_separate then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_formal_argument (l_name)
						current_file.put_character (')')
						current_file.put_character (' ')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_open)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_attribute_region_access (l_name, l_argument_type.primary_type, False)
						print_comma
						current_file.put_string (c_scond)
						current_file.put_character (')')
						if l_separate_arguments_count > 1 then
							current_file.put_character (';')
							current_file.put_character (' ')
							current_file.put_string (c_else)
							current_file.put_character (' ')
							current_file.put_string (c_ge_scoop_condition_decrement)
							current_file.put_character ('(')
							current_file.put_string (c_scond)
							current_file.put_character (')')
						end
						print_semicolon_newline
					end
					i := i + 1
				end
				if l_separate_arguments_count > 1 then
					print_indentation
					current_file.put_string (c_ge_scoop_multisessions_open_stop)
					current_file.put_character ('(')
					current_file.put_character (')')
					print_semicolon_newline
				end
				print_indentation
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				current_file.put_string (c_ge_setjmp)
				current_file.put_character ('(')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_character ('j')
				current_file.put_character ('b')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('!')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
					-- Make sure to exit from the current SCOOP sessions
					-- before propagating the exception.
				check l_has_separate_arguments: l_arguments /= Void then end
				nb := l_arguments.count
				from i := 1 until i > nb loop
					l_argument := l_arguments.formal_argument (i)
					l_name := l_argument.name
					l_argument_type := dynamic_type_set (l_name).static_type
					if l_argument_type.is_separate then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_close)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						print_semicolon_newline
					end
					i := i + 1
				end
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ge_jump_to_last_rescue)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			end
			if a_feature.is_function then
				print_external_builtin_function_body (a_feature)
			else
				print_external_builtin_procedure_body (a_feature)
			end
			if l_has_separate_arguments then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
					-- Exit from the current SCOOP sessions.
				check l_has_separate_arguments: l_arguments /= Void then end
				nb := l_arguments.count
				from i := 1 until i > nb loop
					l_argument := l_arguments.formal_argument (i)
					l_name := l_argument.name
					l_argument_type := dynamic_type_set (l_name).static_type
					if l_argument_type.is_separate then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_close)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						print_semicolon_newline
					end
					i := i + 1
				end
			end
				-- Clean up.
			current_file := old_file
		end

	print_external_builtin_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_class_code
			when {ET_TOKEN_CODES}.builtin_any_class then
				print_external_builtin_any_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_arguments_32_class then
				print_external_builtin_arguments_32_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_boolean_class then
				print_external_builtin_boolean_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_boolean_ref_class then
				print_external_builtin_boolean_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_8_class then
				print_external_builtin_character_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_8_ref_class then
				print_external_builtin_character_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_32_class then
				print_external_builtin_character_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_32_ref_class then
				print_external_builtin_character_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_com_failure_class then
				print_external_builtin_com_failure_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_class then
				print_external_builtin_exception_manager_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_factory_class then
				print_external_builtin_exception_manager_factory_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_function_class then
				print_external_builtin_function_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_identified_routines_class then
				print_external_builtin_identified_routines_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_8_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_8_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_16_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_16_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_32_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_32_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_64_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_64_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_class then
				print_external_builtin_ise_runtime_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_memory_class then
				print_external_builtin_memory_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_8_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_8_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_16_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_16_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_32_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_32_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_64_class then
				print_external_builtin_integer_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_64_ref_class then
				print_external_builtin_integer_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_platform_class then
				print_external_builtin_platform_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_pointer_class then
				print_external_builtin_pointer_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_pointer_ref_class then
				print_external_builtin_pointer_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_32_class then
				print_external_builtin_real_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_32_ref_class then
				print_external_builtin_real_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_64_class then
				print_external_builtin_real_n_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_64_ref_class then
				print_external_builtin_real_n_ref_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_special_class then
				print_external_builtin_special_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_class then
				print_external_builtin_tuple_function_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_type_class then
				print_external_builtin_type_function_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_any_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "ANY".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_any: current_feature.is_builtin_any_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_any_conforms_to then
				print_builtin_any_conforms_to_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_deep_twin then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_any_deep_twin_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_any_generating_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_any_generating_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_any_generator then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_any_generator_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_any_is_deep_equal then
				print_builtin_any_is_deep_equal_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_is_equal then
				print_builtin_any_is_equal_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_same_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_any_same_type_call (current_feature, current_type, False)
				print_semicolon_newline
			when {ET_TOKEN_CODES}.builtin_any_standard_is_equal then
				print_builtin_any_standard_is_equal_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_standard_twin then
				print_builtin_any_standard_twin_body (a_feature)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_any_tagged_out then
				print_builtin_any_tagged_out_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_twin then
				print_builtin_any_twin_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_any_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_arguments_32_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "ARGUMENTS_32".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_arguments_32: current_feature.is_builtin_arguments_32_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_arguments_32_argument_count then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_arguments_32_argument_count_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_arguments_32_i_th_argument_pointer then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_arguments_32_i_th_argument_pointer_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_arguments_32_i_th_argument_string then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_arguments_32_i_th_argument_string_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_arguments_32_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_boolean_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "BOOLEAN".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean: current_feature.is_builtin_boolean_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_and then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_boolean_and_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_boolean_not then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_boolean_not_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_boolean_or then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_boolean_or_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_boolean_xor then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_boolean_xor_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_boolean_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_boolean_ref_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "BOOLEAN_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean_ref: current_feature.is_builtin_boolean_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_ref_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_boolean_ref_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_boolean_ref_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_character_n_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classed "CHARACTER_N".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n: current_feature.is_builtin_character_n_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_code then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_character_n_code_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_character_n_natural_32_code then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_character_n_natural_32_code_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_character_n_to_character_8 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_character_n_to_character_8_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_character_n_to_character_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_character_n_to_character_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_character_n_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_character_n_ref_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classed "CHARACTER_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n_ref: current_feature.is_builtin_character_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_ref_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_character_n_ref_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_character_n_ref_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_com_failure_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "COM_FAILURE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_com_failure: current_feature.is_builtin_com_failure_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_com_failure_c_strlen then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_c_strlen_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_ccom_hresult_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult_code then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_ccom_hresult_code_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult_facility then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_ccom_hresult_facility_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_com_failure_character_size then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_character_size_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_com_failure_cwin_error_text then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_com_failure_cwin_error_text_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_com_failure_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_exception_manager_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "EXCEPTION_MANAGER".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_exception_manager: current_feature.is_builtin_exception_manager_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_exception_manager_exception_from_code then
				print_builtin_exception_manager_exception_from_code_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_is_caught then
				print_builtin_exception_manager_is_caught_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_is_ignorable then
				print_builtin_exception_manager_is_ignorable_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_is_ignored then
				print_builtin_exception_manager_is_ignored_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_is_raisable then
				print_builtin_exception_manager_is_raisable_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_last_exception then
				print_builtin_exception_manager_last_exception_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_type_of_code then
				print_builtin_exception_manager_type_of_code_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_exception_manager_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_exception_manager_factory_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "EXCEPTION_MANAGER_FACTORY".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_exception_manager_factory: current_feature.is_builtin_exception_manager_factory_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_exception_manager_factory_exception_manager then
				print_builtin_exception_manager_factory_exception_manager_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_exception_manager_factory_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_function_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "FUNCTION".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_function: current_feature.is_builtin_function_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_function_fast_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_function_fast_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_function_item then
				print_builtin_function_item_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_function_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_identified_routines_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "IDENTIFIED_ROUTINES".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_identified_routines: current_feature.is_builtin_identified_routines_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_current_object_id then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_identified_routines_eif_current_object_id_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_id_object then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_identified_routines_eif_id_object_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_is_object_id_of_current then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_identified_routines_eif_is_object_id_of_current_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_object_id then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_identified_routines_eif_object_id_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_identified_routines_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_integer_n_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classes "INTEGER_N" and "NATURAL_N".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n: current_feature.is_builtin_integer_n_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_8 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_integer_8_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_16 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_integer_16_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_integer_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_integer_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_8 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_natural_8_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_16 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_natural_16_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_natural_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_as_natural_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_and then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_and_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_not then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_not_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_or then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_or_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_shift_left then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_shift_left_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_shift_right then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_shift_right_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_xor then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_bit_xor_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_identity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_identity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_integer_quotient then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_integer_quotient_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_integer_remainder then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_integer_remainder_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_is_less then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_is_less_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_minus then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_minus_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_opposite then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_opposite_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_plus then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_plus_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_power then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_power_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_product then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_product_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_quotient then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_quotient_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_character_8 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_character_8_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_character_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_character_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_double then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_double_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_real_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_real_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_to_real_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_integer_n_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_integer_n_ref_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classes "INTEGER_N_REF" and "NATURAL_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n_ref: current_feature.is_builtin_integer_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_ref_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_integer_n_ref_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_integer_n_ref_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_ise_runtime_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "ISE_RUNTIME".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_runtime: current_feature.is_builtin_ise_runtime_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_runtime_attached_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_attached_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_boolean_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_boolean_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_boolean_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_boolean_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_8_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_character_8_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_character_8_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_32_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_character_32_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_character_32_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_check_assert then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_check_assert_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_compiler_version then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_compiler_version_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_detachable_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_detachable_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_dynamic_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_dynamic_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_dynamic_type_at_offset then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_dynamic_type_at_offset_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_eif_gen_param_id then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_eif_gen_param_id_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_count_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_field_count_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_name_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_field_name_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_offset_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_field_offset_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_static_type_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_field_static_type_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_type_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_field_type_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generating_type_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_generating_type_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generating_type_8_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_generating_type_8_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generator_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_generator_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generator_8_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_generator_8_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generic_parameter_count then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_generic_parameter_count_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_in_assertion then
				print_builtin_ise_runtime_in_assertion_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_8_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_8_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_8_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_16_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_16_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_16_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_16_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_32_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_32_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_32_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_64_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_64_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_integer_64_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_attached_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_attached_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_copy_semantics_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_copy_semantics_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_expanded then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_expanded_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_field_expanded_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_field_expanded_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_field_transient_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_field_transient_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_object_marked then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_object_marked_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_copy_semantics_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_copy_semantics_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_expanded then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_of_expanded_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_of_reference_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference_or_basic_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_of_reference_or_basic_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_special_of_reference_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_tuple then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_tuple_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_tuple_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_is_tuple_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_8_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_8_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_8_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_16_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_16_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_16_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_16_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_32_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_32_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_32_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_64_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_64_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_natural_64_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_instance_of then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_new_instance_of_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_special_of_reference_instance_of then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_new_special_of_reference_instance_of_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_tuple_instance_of then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_new_tuple_instance_of_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_type_instance_of then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_new_type_instance_of_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_object_size then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_object_size_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_once_objects then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_once_objects_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_persistent_field_count_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_persistent_field_count_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_pointer_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_pointer_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_pointer_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_pointer_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_pre_ecma_mapping_status then
				print_builtin_ise_runtime_pre_ecma_mapping_status_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_raw_reference_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_raw_reference_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_raw_reference_field_at_offset then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_raw_reference_field_at_offset_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_32_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_real_32_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_real_32_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_64_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_real_64_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_real_64_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_reference_field_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field_at then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_reference_field_at_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field_at_offset then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_reference_field_at_offset_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_storable_version_of_type then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_storable_version_of_type_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_type_id_from_name then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_type_id_from_name_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_type_conforms_to then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_ise_runtime_type_conforms_to_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_ise_runtime_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_memory_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "MEMORY".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_memory: current_feature.is_builtin_memory_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_memory_find_referers then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_memory_find_referers_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_memory_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_platform_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "PLATFORM".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_platform: current_feature.is_builtin_platform_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_platform_boolean_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_boolean_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_character_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_character_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_double_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_double_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_integer_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_integer_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_64_bits then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_64_bits_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_dotnet then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_dotnet_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_mac then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_mac_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_scoop_capable then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_scoop_capable_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_thread_capable then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_thread_capable_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_unix then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_unix_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_vms then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_vms_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_vxworks then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_vxworks_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_is_windows then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_is_windows_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_pointer_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_pointer_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_real_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_real_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_platform_wide_character_bytes then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_platform_wide_character_bytes_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_platform_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_pointer_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "POINTER".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer: current_feature.is_builtin_pointer_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_hash_code then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_pointer_hash_code_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_pointer_is_default_pointer then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_pointer_is_default_pointer_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_pointer_out then
				print_builtin_pointer_out_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_pointer_plus then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_pointer_plus_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_pointer_to_integer_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_pointer_to_integer_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_pointer_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_pointer_ref_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "POINTER_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer_ref: current_feature.is_builtin_pointer_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_ref_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_pointer_ref_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_pointer_ref_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_real_n_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classes "REAL_N".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n: current_feature.is_builtin_real_n_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ceiling_real_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ceiling_real_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ceiling_real_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ceiling_real_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_floor_real_32 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_floor_real_32_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_floor_real_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_floor_real_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_identity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_identity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_equal then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_is_equal_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_greater then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_is_greater_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_greater_equal then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_is_greater_equal_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_less then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_is_less_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_less_equal then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_is_less_equal_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_maximum_number then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_maximum_number_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_minimum_number then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ieee_minimum_number_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_is_less then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_is_less_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_is_nan then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_is_nan_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_is_negative_infinity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_is_negative_infinity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_is_positive_infinity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_is_positive_infinity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_minus then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_minus_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_opposite then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_opposite_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_out then
				print_builtin_real_n_out_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_n_plus then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_plus_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_power then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_power_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_product then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_product_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_quotient then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_quotient_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_to_double then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_to_double_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_integer then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_truncated_to_integer_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_integer_64 then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_truncated_to_integer_64_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_real then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_truncated_to_real_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_real_n_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_real_n_ref_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in classes "REAL_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n_ref: current_feature.is_builtin_real_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ref_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ref_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ref_nan then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ref_nan_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ref_negative_infinity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ref_negative_infinity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_real_n_ref_positive_infinity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_real_n_ref_positive_infinity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_real_n_ref_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_special_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "SPECIAL".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_special: current_feature.is_builtin_special_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_special_aliased_resized_area then
				print_builtin_special_aliased_resized_area_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_special_base_address then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_special_base_address_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_special_capacity then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_special_capacity_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_special_count then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_special_count_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_special_element_size then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_special_element_size_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_special_item then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_special_item_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_special_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_tuple_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "TUPLE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_tuple: current_feature.is_builtin_tuple_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_tuple_boolean_item then
				print_builtin_tuple_boolean_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_character_8_item then
				print_builtin_tuple_character_8_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_character_32_item then
				print_builtin_tuple_character_32_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_count then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_tuple_count_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_tuple_integer_8_item then
				print_builtin_tuple_integer_8_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_integer_16_item then
				print_builtin_tuple_integer_16_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_integer_32_item then
				print_builtin_tuple_integer_32_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_integer_64_item then
				print_builtin_tuple_integer_64_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_item_code then
				print_builtin_tuple_item_code_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_natural_8_item then
				print_builtin_tuple_natural_8_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_natural_16_item then
				print_builtin_tuple_natural_16_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_natural_32_item then
				print_builtin_tuple_natural_32_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_natural_64_item then
				print_builtin_tuple_natural_64_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_object_comparison then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_tuple_object_comparison_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_tuple_pointer_item then
				print_builtin_tuple_pointer_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_real_32_item then
				print_builtin_tuple_real_32_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_real_64_item then
				print_builtin_tuple_real_64_item_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_reference_item then
				print_builtin_tuple_reference_item_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_tuple_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_type_function_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in function introduced in class "TYPE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_function: a_feature.is_function
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_type: current_feature.is_builtin_type_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_type_default then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_default_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_generic_parameter_type then
				print_builtin_type_generic_parameter_type_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_type_generic_parameter_count then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_generic_parameter_count_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_has_default then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_has_default_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_is_attached then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_is_attached_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_is_deferred then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_is_deferred_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_is_expanded then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_is_expanded_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_runtime_name then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_runtime_name_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_type_type_id then
				fill_call_formal_arguments (a_feature)
				print_indentation_assign_to_result
				print_builtin_type_type_id_call (current_feature, current_type, False)
				print_semicolon_newline
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_type_function_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_class_code
			when {ET_TOKEN_CODES}.builtin_any_class then
				print_external_builtin_any_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_boolean_ref_class then
				print_external_builtin_boolean_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_8_ref_class then
				print_external_builtin_character_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_character_32_ref_class then
				print_external_builtin_character_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_com_failure_class then
				print_external_builtin_com_failure_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_class then
				print_external_builtin_exception_manager_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_identified_routines_class then
				print_external_builtin_identified_routines_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_8_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_16_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_32_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_integer_64_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_exception_manager_class then
				print_external_builtin_ise_exception_manager_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_class then
				print_external_builtin_ise_runtime_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_memory_class then
				print_external_builtin_memory_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_8_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_16_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_32_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_natural_64_ref_class then
				print_external_builtin_integer_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_pointer_ref_class then
				print_external_builtin_pointer_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_procedure_class then
				print_external_builtin_procedure_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_32_ref_class then
				print_external_builtin_real_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_real_64_ref_class then
				print_external_builtin_real_n_ref_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_special_class then
				print_external_builtin_special_procedure_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_class then
				print_external_builtin_tuple_procedure_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_any_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "ANY".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_any: current_feature.is_builtin_any_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_any_copy then
				print_builtin_any_copy_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_any_standard_copy then
				print_builtin_any_standard_copy_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_any_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_boolean_ref_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "BOOLEAN_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean_ref: current_feature.is_builtin_boolean_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_ref_set_item then
				fill_call_formal_arguments (a_feature)
				print_builtin_boolean_ref_set_item_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_boolean_ref_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_character_n_ref_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in classes "CHARACTER_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n_ref: current_feature.is_builtin_character_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_ref_set_item then
				fill_call_formal_arguments (a_feature)
				print_builtin_character_n_ref_set_item_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_character_n_ref_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_com_failure_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "COM_FAILURE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_com_failure: current_feature.is_builtin_com_failure_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_com_failure_cwin_local_free then
				fill_call_formal_arguments (a_feature)
				print_builtin_com_failure_cwin_local_free_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_com_failure_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_exception_manager_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "EXCEPTION_MANAGER".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_exception_manager: current_feature.is_builtin_exception_manager_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_exception_manager_catch then
				print_builtin_exception_manager_catch_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_ignore then
				print_builtin_exception_manager_ignore_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_raise then
				print_builtin_exception_manager_raise_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_exception_manager_set_is_ignored then
				print_builtin_exception_manager_set_is_ignored_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_exception_manager_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_identified_routines_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "IDENTIFIED_ROUTINES".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_identified_routines: current_feature.is_builtin_identified_routines_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_object_id_free then
				fill_call_formal_arguments (a_feature)
				print_builtin_identified_routines_eif_object_id_free_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_identified_routines_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_integer_n_ref_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in classes "INTEGER_N_REF" and "NATURAL_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n_ref: current_feature.is_builtin_integer_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_ref_set_item then
				fill_call_formal_arguments (a_feature)
				print_builtin_integer_n_ref_set_item_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_integer_n_ref_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_ise_exception_manager_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "ISE_EXCEPTION_MANAGER".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_exception_manager: current_feature.is_builtin_ise_exception_manager_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_exception_manager_developer_raise then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_exception_manager_developer_raise_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_ise_exception_manager_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_ise_runtime_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "ISE_RUNTIME".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_runtime: current_feature.is_builtin_ise_runtime_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_runtime_lock_marking then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_lock_marking_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_mark_object then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_mark_object_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_boolean_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_boolean_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_boolean_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_boolean_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_8_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_character_8_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_character_8_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_32_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_character_32_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_character_32_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_8_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_8_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_8_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_16_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_16_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_16_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_16_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_32_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_32_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_32_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_64_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_64_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_integer_64_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_8_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_8_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_8_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_8_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_16_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_16_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_16_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_16_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_32_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_32_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_32_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_64_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_64_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_natural_64_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_pointer_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_pointer_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_pointer_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_pointer_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_pre_ecma_mapping then
				print_builtin_ise_runtime_set_pre_ecma_mapping_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_32_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_real_32_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_32_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_real_32_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_64_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_real_64_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_64_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_real_64_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_reference_field then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_reference_field_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_reference_field_at then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_set_reference_field_at_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_unlock_marking then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_unlock_marking_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_ise_runtime_unmark_object then
				fill_call_formal_arguments (a_feature)
				print_builtin_ise_runtime_unmark_object_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_ise_runtime_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_memory_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "MEMORY".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_memory: current_feature.is_builtin_memory_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_memory_free then
				fill_call_formal_arguments (a_feature)
				print_builtin_memory_free_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_memory_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_pointer_ref_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "POINTER_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer_ref: current_feature.is_builtin_pointer_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_ref_set_item then
				fill_call_formal_arguments (a_feature)
				print_builtin_pointer_ref_set_item_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_pointer_ref_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_procedure_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "PROCEDURE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_procedure: current_feature.is_builtin_procedure_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_procedure_call then
				print_builtin_procedure_call_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_procedure_fast_call then
				fill_call_formal_arguments (a_feature)
				print_builtin_procedure_fast_call_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_procedure_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_real_n_ref_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in classes "REAL_N_REF".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n_ref: current_feature.is_builtin_real_n_ref_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ref_set_item then
				fill_call_formal_arguments (a_feature)
				print_builtin_real_n_ref_set_item_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_real_n_ref_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_special_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "SPECIAL".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_special: current_feature.is_builtin_special_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_special_make_empty then
				-- Do nothing: already done in `print_malloc_current'.
			when {ET_TOKEN_CODES}.builtin_special_extend then
				print_builtin_special_extend_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_special_put then
				fill_call_formal_arguments (a_feature)
				print_builtin_special_put_call (current_feature, current_type, False)
				call_operands.wipe_out
			when {ET_TOKEN_CODES}.builtin_special_set_count then
				print_builtin_special_set_count_body (a_feature)
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_special_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_builtin_tuple_procedure_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of built-in feature `a_feature'.
			-- `a_feature' is a built-in procedure introduced in class "TUPLE".
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_procedure: a_feature.is_procedure
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_tuple: current_feature.is_builtin_tuple_class
			valid_feature: current_feature.static_feature = a_feature
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_tuple_put_boolean then
				print_builtin_tuple_put_boolean_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_character_8 then
				print_builtin_tuple_put_character_8_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_character_32 then
				print_builtin_tuple_put_character_32_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_integer_8 then
				print_builtin_tuple_put_integer_8_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_integer_16 then
				print_builtin_tuple_put_integer_16_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_integer_32 then
				print_builtin_tuple_put_integer_32_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_integer_64 then
				print_builtin_tuple_put_integer_64_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_natural_8 then
				print_builtin_tuple_put_natural_8_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_natural_16 then
				print_builtin_tuple_put_natural_16_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_natural_32 then
				print_builtin_tuple_put_natural_32_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_natural_64 then
				print_builtin_tuple_put_natural_64_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_pointer then
				print_builtin_tuple_put_pointer_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_real_32 then
				print_builtin_tuple_put_real_32_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_real_64 then
				print_builtin_tuple_put_real_64_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_put_reference then
				print_builtin_tuple_put_reference_body (a_feature)
			when {ET_TOKEN_CODES}.builtin_tuple_set_object_comparison then
				fill_call_formal_arguments (a_feature)
				print_builtin_tuple_set_object_comparison_call (current_feature, current_type, False)
				call_operands.wipe_out
			else
					-- Internal error: unknown built-in feature.
					-- This error should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_external_builtin_tuple_procedure_body", 1, "unknown built-in feature.")
			end
		end

	print_external_c_prototype (a_feature_name: detachable ET_FEATURE_NAME;
		a_arguments: detachable ET_FORMAL_ARGUMENT_LIST; a_result_type_set: detachable ET_DYNAMIC_TYPE_SET;
		a_signature_arguments, a_signature_result: detachable STRING;
		a_alias: detachable ET_EXTERNAL_ALIAS)
			-- Print to `header_file' a prototype for the external C function.
			-- If `a_feature_name' is Void then the name of the C function will be found in the alias.
			-- `a_signature_arguments' and `a_signature_result', if not Void,
			-- are the signature types declared in the Language part.
			-- `a_result_type_set' is not Void if the external feature is a query.
		require
			name_not_void: a_feature_name /= Void or else a_alias /= Void
		local
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			l_alias_value: ET_MANIFEST_STRING
			i, nb_args: INTEGER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
		do
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			if a_signature_result /= Void then
				header_file.put_string (a_signature_result)
			elseif a_result_type_set /= Void then
				l_result_type := a_result_type_set.static_type.primary_type
				print_type_declaration (l_result_type, header_file)
			else
				header_file.put_string (c_void)
			end
			header_file.put_character (' ')
			if a_alias /= Void then
				l_alias_value := a_alias.manifest_string
				header_file.put_string (l_alias_value.value)
			elseif a_feature_name /= Void then
				header_file.put_string (a_feature_name.lower_name)
			else
				check precondition_name_not_void: False then end
			end
			header_file.put_character ('(')
			if a_signature_arguments /= Void then
				header_file.put_string (a_signature_arguments)
			elseif a_arguments /= Void and then not a_arguments.is_empty then
				nb_args := a_arguments.count
				from i := 1 until i > nb_args loop
					if i /= 1 then
						header_file.put_character (',')
					end
					l_argument_type_set := argument_type_set (i)
					l_argument_type := l_argument_type_set.static_type.primary_type
					if l_argument_type.base_class.is_typed_pointer_class then
							-- The argument is declared of type 'TYPED_POINTER [XX]'.
							-- In that case we use the corresponding pointer (i.e.
							-- the first attribute of the object).
						l_actual_parameters := l_argument_type.base_type.actual_parameters
						if l_actual_parameters = Void or else l_actual_parameters.is_empty then
								-- Internal error: TYPED_POINTER [XX] has one generic parameter.
								-- This should have been checked already.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_external_c_prototype", 1, "missing generic parameter for type 'TYPED_POINTER'.")
						else
							l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
							print_type_declaration (l_actual_parameter, header_file)
							if l_actual_parameter.is_expanded then
								header_file.put_character ('*')
							end
						end
					else
						print_type_declaration (l_argument_type, header_file)
					end
					i := i + 1
				end
			end
			header_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
		end

	print_external_c_body (a_feature_name: detachable ET_FEATURE_NAME;
		a_arguments: detachable ET_FORMAL_ARGUMENT_LIST; a_result_type_set: detachable ET_DYNAMIC_TYPE_SET;
		a_signature_arguments, a_signature_result: detachable STRING;
		a_alias: detachable ET_EXTERNAL_ALIAS; is_macro: BOOLEAN)
			-- Print body of external C function to `current_file'.
			-- If `a_feature_name' is Void then the name of the C function will be found in the alias.
			-- `a_signature_arguments' and `a_signature_result', if not Void,
			-- are the signature types declared in the Language part.
			-- `a_result_type_set' is not Void if the external feature is a query.
		require
			name_not_void: a_feature_name /= Void or else a_alias /= Void
		local
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			l_alias_value: ET_MANIFEST_STRING
			i, nb_args: INTEGER
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_cursor: DS_LIST_CURSOR [STRING]
			l_name: ET_IDENTIFIER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
		do
			print_indentation
			if a_result_type_set /= Void then
				l_result_type := a_result_type_set.static_type.primary_type
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_declaration_type_cast_from_c (l_result_type, current_file)
				current_file.put_character ('(')
			end
			if a_alias /= Void then
				l_alias_value := a_alias.manifest_string
				current_file.put_string (l_alias_value.value)
			elseif a_feature_name /= Void then
				current_file.put_string (a_feature_name.lower_name)
			else
				check precondition_name_not_void: False then end
			end
			if a_arguments /= Void and then not a_arguments.is_empty then
				nb_args := a_arguments.count
				current_file.put_character ('(')
				if a_signature_arguments /= Void then
					create l_splitter.make_with_separators (",")
					l_list := l_splitter.split (a_signature_arguments)
					if l_list.count /= nb_args then
-- TODO: error
					end
					l_cursor := l_list.new_cursor
					l_cursor.start
					from i := 1 until i > nb_args loop
						if i /= 1 then
							current_file.put_character (',')
						end
						if not l_cursor.after then
							current_file.put_character ('(')
							current_file.put_string (l_cursor.item)
							current_file.put_character (')')
							l_cursor.forth
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_c_body", 1, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							if l_argument_type = current_dynamic_system.pointer_type then
									-- Get rid of the volatile type marker.
								current_file.put_character ('(')
								current_file.put_string (c_void)
								current_file.put_character ('*')
								current_file.put_character (')')
							end
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				else
					from i := 1 until i > nb_args loop
						if i /= 1 then
							current_file.put_character (',')
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type = current_dynamic_system.pointer_type then
								-- Discard the 'volatile' qualifier.
							current_file.put_character ('(')
							current_file.put_string (c_void)
							current_file.put_character ('*')
							current_file.put_character (')')
							print_argument_name (l_name, current_file)
						elseif l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_c_body", 2, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				end
				current_file.put_character (')')
			elseif not is_macro then
				current_file.put_character ('(')
				current_file.put_character (')')
			end
			if a_result_type_set /= Void then
				current_file.put_character (')')
			end
			current_file.put_character (';')
			current_file.put_new_line
		end

	print_external_c_struct_body (a_arguments: detachable ET_FORMAL_ARGUMENT_LIST; a_result_type_set: detachable ET_DYNAMIC_TYPE_SET;
		a_struct_type, a_field_name: STRING; a_field_type: detachable STRING)
			-- Print body of external C struct to `current_file'.
			-- `a_result_type_set' is not Void if the external feature is a query.
		require
			a_struct_type_not_void: a_struct_type /= Void
			a_field_name_not_void: a_field_name /= Void
		local
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_field_name: STRING
			l_name: ET_IDENTIFIER
			nb_args: INTEGER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
		do
			if a_result_type_set /= Void then
				l_result_type := a_result_type_set.static_type.primary_type
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_declaration_type_cast_from_c (l_result_type, current_file)
				current_file.put_character ('(')
			end
			l_field_name := a_field_name
			if not l_field_name.is_empty then
				inspect l_field_name.item (1)
				when '@' then
					l_field_name := STRING_.cloned_string (l_field_name)
					l_field_name.remove_head (1)
				when '&' then
					l_field_name := STRING_.cloned_string (l_field_name)
					l_field_name.remove_head (1)
					current_file.put_character ('&')
				else
				end
			end
			current_file.put_character ('(')
			current_file.put_character ('(')
			current_file.put_character ('(')
			current_file.put_string (a_struct_type)
			current_file.put_character ('*')
			current_file.put_character (')')
			if a_arguments /= Void then
				nb_args := a_arguments.count
				if nb_args >= 1 then
					l_name := a_arguments.formal_argument (1).name
					l_argument_type_set := dynamic_type_set (l_name)
					l_argument_type := l_argument_type_set.static_type.primary_type
					if l_argument_type.base_class.is_typed_pointer_class then
							-- The argument is declared of type 'TYPED_POINTER [XX]'.
							-- In that case we use the corresponding pointer (i.e.
							-- the first attribute of the object).
						l_actual_parameters := l_argument_type.base_type.actual_parameters
						if l_actual_parameters = Void or else l_actual_parameters.is_empty then
								-- Internal error: TYPED_POINTER [XX] has one generic parameter.
								-- This should have been checked already.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_external_c_struct_body", 1, "missing generic parameter for type 'TYPED_POINTER'.")
						else
							l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
							current_file.put_character ('(')
							print_type_declaration (l_actual_parameter, current_file)
							if l_actual_parameter.is_expanded then
								current_file.put_character ('*')
							end
							current_file.put_character (')')
							current_file.put_character ('(')
							print_argument_name (l_name, current_file)
							current_file.put_character ('.')
							current_file.put_character ('a')
							current_file.put_character ('1')
							current_file.put_character (')')
						end
					else
						if l_argument_type = current_dynamic_system.pointer_type then
								-- Get rid of the volatile type marker.
							current_file.put_character ('(')
							current_file.put_string (c_void)
							current_file.put_character ('*')
							current_file.put_character (')')
						end
						print_argument_name (l_name, current_file)
					end
				end
			else
-- TODO: error
			end
			current_file.put_character (')')
			current_file.put_character ('-')
			current_file.put_character ('>')
			current_file.put_string (l_field_name)
			current_file.put_character (')')
			if l_result_type = Void then
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				if a_field_type /= Void then
					current_file.put_character ('(')
					current_file.put_string (a_field_type)
					current_file.put_character (')')
				end
				if a_arguments /= Void and then nb_args >= 2 then
					l_name := a_arguments.formal_argument (2).name
					l_argument_type_set := dynamic_type_set (l_name)
					l_argument_type := l_argument_type_set.static_type.primary_type
					if l_argument_type.base_class.is_typed_pointer_class then
							-- The argument is declared of type 'TYPED_POINTER [XX]'.
							-- In that case we use the corresponding pointer (i.e.
							-- the first attribute of the object).
						l_actual_parameters := l_argument_type.base_type.actual_parameters
						if l_actual_parameters = Void or else l_actual_parameters.is_empty then
								-- Internal error: TYPED_POINTER [XX] has one generic parameter.
								-- This should have been checked already.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_external_c_struct_body", 2, "missing generic parameter for type 'TYPED_POINTER'.")
						else
							l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
							current_file.put_character ('(')
							print_type_declaration (l_actual_parameter, current_file)
							if l_actual_parameter.is_expanded then
								current_file.put_character ('*')
							end
							current_file.put_character (')')
							current_file.put_character ('(')
							print_argument_name (l_name, current_file)
							current_file.put_character ('.')
							current_file.put_character ('a')
							current_file.put_character ('1')
							current_file.put_character (')')
						end
					else
						if l_argument_type = current_dynamic_system.pointer_type then
								-- Get rid of the volatile type marker.
							current_file.put_character ('(')
							current_file.put_string (c_void)
							current_file.put_character ('*')
							current_file.put_character (')')
						end
						print_argument_name (l_name, current_file)
					end
				else
-- TODO: error
				end
			end
			if l_result_type /= Void then
				current_file.put_character (')')
			end
			current_file.put_character (';')
			current_file.put_new_line
		end

	print_external_c_inline_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print body of external "C inline" `a_feature' to `current_file'.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
		local
			l_alias_value: ET_MANIFEST_STRING
			l_c_code: STRING
			l_argument_name: ET_IDENTIFIER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
			l_name: STRING
			i, j, nb: INTEGER
			i2, nb2: INTEGER
			i3, nb3: INTEGER
			l_max: INTEGER
			l_max_index: INTEGER
			l_semicolon_needed: BOOLEAN
			l_newline_needed_before_semicolon: BOOLEAN
			l_has_result_type_cast: BOOLEAN
			c, c3: CHARACTER
		do
			if attached a_feature.alias_clause as l_alias then
				l_alias_value := l_alias.manifest_string
				l_c_code := l_alias_value.value
					-- Check if it looks like legacy code: a C statement
					-- with no terminating semicolon.
				from
					i := l_c_code.count
				until
					i < 1
				loop
					inspect l_c_code.item (i)
					when ' ', '%N', '%R', '%T' then
						i := i - 1
					when ';' then
							-- This is a terminating semicolon.
						i := 0
					when '}' then
							-- This is a terminating '}'.
						i := 0
					when '/' then
						if i > 1 and then l_c_code.item (i - 1) = '*' then
								-- This is a terminating comment.
						else
								-- There is no terminating semicolon.
							l_semicolon_needed := True

						end
						i := 0
					when 'f' then
						if i > 5 and then c_endif.same_characters (l_c_code, i - 5, i, 1) then
								-- This is a terminating #endif.
							l_newline_needed_before_semicolon := True
							i := i - 6
						else
								-- There is no terminating semicolon.
							l_semicolon_needed := True
							i := 0

						end
					else
							-- There is no terminating semicolon.
						l_semicolon_needed := True
						i := 0
					end
				end
				if not l_semicolon_needed then
					l_newline_needed_before_semicolon := False
				end
				if a_feature.is_function and then not l_c_code.has_substring (c_return) then
						-- This looks like legacy code.
						-- With ECMA we need to write the 'return' keyword in the alias clause.
					current_file.put_string (c_return)
					current_file.put_character (' ')
					if attached current_feature.result_type_set as l_result_type_set then
						print_declaration_type_cast_from_c (l_result_type_set.static_type.primary_type, current_file)
						current_file.put_character ('(')
						l_has_result_type_cast := True
					end
					l_semicolon_needed := True
				end
				if attached a_feature.arguments as l_formal_arguments then
					nb := l_c_code.count
					from i := 1 until i > nb loop
						c := l_c_code.item (i)
						if c = '$' then
							i := i + 1
							if i <= nb then
								c := l_c_code.item (i)
								inspect c
								when '$' then
									current_file.put_character ('$')
									i := i + 1
								when 'a'..'z', 'A'..'Z' then
									l_max := 0
									l_max_index := 0
									nb2 := l_formal_arguments.count
									from i2 := 1 until i2 > nb2 loop
										l_name := l_formal_arguments.formal_argument (i2).name.name
										nb3 := l_name.count
										if nb3 > l_max then
											from
												i3 := 1
												j := i
											until
												j > nb or
												i3 > nb3
											loop
												c := l_c_code.item (j)
												c3 := l_name.item (i3)
												if CHARACTER_.as_lower (c3) = CHARACTER_.as_lower (c) then
													i3 := i3 + 1
													j := j + 1
												else
													j := nb + 1
												end
											end
											if i3 > nb3 then
												l_max_index := i2
												l_max := nb3
											end
										end
										i2 := i2 + 1
									end
									if l_max_index /= 0 then
										l_argument_name := l_formal_arguments.formal_argument (l_max_index).name
										l_argument_type_set := argument_type_set (l_max_index)
										l_argument_type := l_argument_type_set.static_type.primary_type
										if l_argument_type.base_class.is_typed_pointer_class then
												-- The argument is declared of type 'TYPED_POINTER [XX]'.
												-- In that case we use the corresponding pointer (i.e.
												-- the first attribute of the object).
											l_actual_parameters := l_argument_type.primary_type.base_type.actual_parameters
											if l_actual_parameters = Void or else l_actual_parameters.is_empty then
													-- Internal error: TYPED_POINTER [XX] has one generic parameter.
													-- This should have been checked already.
												set_fatal_error
												error_handler.report_giaac_error (generator, "print_external_c_inline_body", 1, "missing generic parameter for type 'TYPED_POINTER'.")
											else
												l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
												current_file.put_character ('(')
												print_type_declaration (l_actual_parameter, current_file)
												if l_actual_parameter.is_expanded then
													current_file.put_character ('*')
												end
												current_file.put_character (')')
												current_file.put_character ('(')
												print_argument_name (l_argument_name, current_file)
												current_file.put_character ('.')
												current_file.put_character ('a')
												current_file.put_character ('1')
												current_file.put_character (')')
											end
										else
											if l_argument_type = current_dynamic_system.pointer_type then
													-- Get rid of the volatile type marker.
												current_file.put_character ('(')
												current_file.put_string (c_void)
												current_file.put_character ('*')
												current_file.put_character (')')
											end
											print_argument_name (l_argument_name, current_file)
										end
										i := i + l_max
									else
										current_file.put_character ('$')
										current_file.put_character (l_c_code.item (i))
										i := i + 1
									end
								else
									current_file.put_character ('$')
									current_file.put_character (c)
									i := i + 1
								end
							else
								current_file.put_character ('$')
							end
						else
							current_file.put_character (c)
							i := i + 1
						end
					end
				else
					current_file.put_string (l_c_code)
				end
				if l_has_result_type_cast then
					current_file.put_character (')')
				end
				if l_semicolon_needed then
					if l_newline_needed_before_semicolon then
						current_file.put_new_line
					end
					current_file.put_character (';')
				end
				current_file.put_new_line
			end
		end

	print_external_c_includes (a_include_filenames: STRING)
			-- Print C includes declarations to `header_file'.
			-- `a_include_filenames' are the filenames (with the
			-- < > or " " characters included) separated by
			-- commas.
		require
			a_include_filenames_not_void: a_include_filenames /= Void
		local
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_cursor: DS_LIST_CURSOR [STRING]
			l_include_filename: STRING
		do
			create l_splitter.make_with_separators (",")
			l_list := l_splitter.split (a_include_filenames)
			l_cursor := l_list.new_cursor
			from l_cursor.start until l_cursor.after loop
				l_include_filename := l_cursor.item
				STRING_.left_adjust (l_include_filename)
				STRING_.right_adjust (l_include_filename)
				include_header_filename (l_include_filename, header_file)
				l_cursor.forth
			end
		end

	print_external_cpp_body (a_feature_name: detachable ET_FEATURE_NAME;
		a_arguments: detachable ET_FORMAL_ARGUMENT_LIST; a_result_type_set: detachable ET_DYNAMIC_TYPE_SET;
		a_cpp_class_type: STRING; a_signature_arguments, a_signature_result: detachable STRING; a_alias: detachable ET_EXTERNAL_ALIAS)
			-- Print body of external C++ function to `current_file'.
			-- If `a_feature_name' is Void then the name of the C++ function will be found in the alias.
			-- `a_cpp_class_type' is the name of the C++ class type.
			-- `a_signature_arguments' and `a_signature_result', if not Void,
			-- are the signature types declared in the Language part.
			-- `a_result_type_set' is not Void if the external feature is a query.
		require
			name_not_void: a_feature_name /= Void or else a_alias /= Void
			a_cpp_class_type_not_void: a_cpp_class_type /= Void
		local
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			l_alias_value: ET_MANIFEST_STRING
			i, nb_args: INTEGER
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_cursor: DS_LIST_CURSOR [STRING]
			l_name: ET_IDENTIFIER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
		do
			print_indentation
			if a_result_type_set /= Void then
				l_result_type := a_result_type_set.static_type.primary_type
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_declaration_type_cast_from_c (l_result_type, current_file)
				current_file.put_character ('(')
			end
			if a_arguments = Void or else a_arguments.is_empty then
-- TODO: error
			else
				nb_args := a_arguments.count
				current_file.put_character ('(')
				current_file.put_character ('(')
				current_file.put_string (a_cpp_class_type)
				current_file.put_character ('*')
				current_file.put_character (')')
				l_name := a_arguments.formal_argument (1).name
				l_argument_type_set := dynamic_type_set (l_name)
				l_argument_type := l_argument_type_set.static_type.primary_type
				if l_argument_type.base_class.is_typed_pointer_class then
						-- The argument is declared of type 'TYPED_POINTER [XX]'.
						-- In that case we use the corresponding pointer (i.e.
						-- the first attribute of the object).
					l_actual_parameters := l_argument_type.base_type.actual_parameters
					if l_actual_parameters = Void or else l_actual_parameters.is_empty then
							-- Internal error: TYPED_POINTER [XX] has one generic parameter.
							-- This should have been checked already.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_external_cpp_body", 1, "missing generic parameter for type 'TYPED_POINTER'.")
					else
						l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
						current_file.put_character ('(')
						print_type_declaration (l_actual_parameter, current_file)
						if l_actual_parameter.is_expanded then
							current_file.put_character ('*')
						end
						current_file.put_character (')')
						current_file.put_character ('(')
						print_argument_name (l_name, current_file)
						current_file.put_character ('.')
						current_file.put_character ('a')
						current_file.put_character ('1')
						current_file.put_character (')')
					end
				else
					if l_argument_type = current_dynamic_system.pointer_type then
							-- Get rid of the volatile type marker.
						current_file.put_character ('(')
						current_file.put_string (c_void)
						current_file.put_character ('*')
						current_file.put_character (')')
					end
					print_argument_name (l_name, current_file)
				end
				current_file.put_character (')')
				current_file.put_string (c_arrow)
				if a_alias /= Void then
					l_alias_value := a_alias.manifest_string
					current_file.put_string (l_alias_value.value)
				elseif a_feature_name /= Void then
					current_file.put_string (a_feature_name.lower_name)
				else
					check precondition_name_not_void: False then end
				end
				current_file.put_character ('(')
				if a_signature_arguments /= Void then
					create l_splitter.make_with_separators (",")
					l_list := l_splitter.split (a_signature_arguments)
					if l_list.count /= nb_args - 1 then
-- TODO: error
					end
					l_cursor := l_list.new_cursor
					l_cursor.start
					from i := 2 until i > nb_args loop
						if i /= 2 then
							current_file.put_character (',')
						end
						if not l_cursor.after then
							current_file.put_character ('(')
							current_file.put_string (l_cursor.item)
							current_file.put_character (')')
							l_cursor.forth
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := dynamic_type_set (l_name)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_cpp_body", 2, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							if l_argument_type = current_dynamic_system.pointer_type then
									-- Get rid of the volatile type marker.
								current_file.put_character ('(')
								current_file.put_string (c_void)
								current_file.put_character ('*')
								current_file.put_character (')')
							end
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				else
					from i := 2 until i > nb_args loop
						if i /= 2 then
							current_file.put_character (',')
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := dynamic_type_set (l_name)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type = current_dynamic_system.pointer_type then
								-- When compiling with C++, MSVC++ does not want to convert
								-- 'void*' to non-'void*' implicitly.
							current_file.put_string ("(char*)")
							print_argument_name (l_name, current_file)
						elseif l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_cpp_body", 3, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							if l_argument_type = current_dynamic_system.pointer_type then
									-- Get rid of the volatile type marker.
								current_file.put_character ('(')
								current_file.put_string (c_void)
								current_file.put_character ('*')
								current_file.put_character (')')
							end
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				end
				current_file.put_character (')')
			end
			if a_result_type_set /= Void then
				current_file.put_character (')')
			end
			current_file.put_character (';')
			current_file.put_new_line
		end

	print_external_cpp_includes (a_include_filenames: STRING)
			-- Print C++ includes declarations to `header_file'.
			-- `a_include_filenames' are the filenames (with the
			-- < > or " " characters included) separated by
			-- commas.
		require
			a_include_filenames_not_void: a_include_filenames /= Void
		local
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_cursor: DS_LIST_CURSOR [STRING]
			l_include_filename: STRING
		do
			create l_splitter.make_with_separators (",")
			l_list := l_splitter.split (a_include_filenames)
			l_cursor := l_list.new_cursor
			from l_cursor.start until l_cursor.after loop
				l_include_filename := l_cursor.item
				STRING_.left_adjust (l_include_filename)
				STRING_.right_adjust (l_include_filename)
				include_cpp_header_filename (l_include_filename, header_file)
				l_cursor.forth
			end
		end

	print_external_dllwin_body (a_feature_name: detachable ET_FEATURE_NAME;
		a_arguments: detachable ET_FORMAL_ARGUMENT_LIST; a_result_type_set: detachable ET_DYNAMIC_TYPE_SET;
		a_dll_file: STRING; a_signature_arguments, a_signature_result: detachable STRING;
		a_alias: detachable ET_EXTERNAL_ALIAS)
			-- Print body of external dllwin function to `current_file'.
			-- If `a_feature_name' is Void then the name of the C function in the DLL will be found in the alias.
			-- `a_dll_file' is the name of the DLL file.
			-- `a_signature_arguments' and `a_signature_result', if not Void,
			-- are the signature types declared in the Language part.
			-- `a_result_type_set' is not Void if the external feature is a query.
		require
			name_not_void: a_feature_name /= Void or else a_alias /= Void
			a_dll_file_not_void: a_dll_file /= Void
		local
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_alias_value: ET_MANIFEST_STRING
			i, nb_args: INTEGER
			l_splitter: ST_SPLITTER
			l_list: DS_LIST [STRING]
			l_cursor: DS_LIST_CURSOR [STRING]
			l_name: ET_IDENTIFIER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_parameters: detachable ET_ACTUAL_PARAMETERS
			l_actual_parameter: ET_DYNAMIC_PRIMARY_TYPE
			l_function_name: STRING
		do
			include_runtime_header_file ("ge_exception.h", False, header_file)
			include_runtime_header_file ("ge_dll.h", False, header_file)
			print_indentation
			current_file.put_line ("static char done = 0;")
			print_indentation
			current_file.put_line ("static EIF_POINTER fp = NULL;")
			print_indentation
			current_file.put_line ("if (!done) {")
			indent
			print_indentation
			current_file.put_line ("HMODULE a_result;")
			print_indentation
			current_file.put_string ("a_result = GE_load_dll(")
			if a_dll_file.is_empty or else a_dll_file.item (1) /= '%"' then
				current_file.put_character ('%"')
				current_file.put_string (a_dll_file)
				current_file.put_character ('%"')
			else
				current_file.put_string (a_dll_file)
			end
			current_file.put_character (')')
			current_file.put_character (';')
			current_file.put_new_line
			print_indentation
			current_file.put_line ("if (a_result == NULL) {")
			indent
			indent
			print_indentation
			current_file.put_string ("/* Cannot load library ")
			current_file.put_string (a_dll_file)
			current_file.put_line (" */")
			dedent
			print_indentation
			current_file.put_string (c_ge_raise)
			current_file.put_character ('(')
			current_file.put_string (c_ge_ex_prog)
			current_file.put_character (')')
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			print_indentation
			current_file.put_string ("fp = (EIF_POINTER)GetProcAddress(a_result,%"")
			if a_alias /= Void then
				l_alias_value := a_alias.manifest_string
				l_function_name := l_alias_value.value
			elseif a_feature_name /= Void then
				l_function_name := a_feature_name.lower_name
			else
				check precondition_name_not_void: False then end
			end
			current_file.put_string (l_function_name)
			current_file.put_line ("%");")
			print_indentation
			current_file.put_line ("if (fp == NULL) {")
			indent
			indent
			print_indentation
			current_file.put_string ("/* Cannot find entry point of ")
			current_file.put_string (l_function_name)
			current_file.put_line (" */")
			dedent
			print_indentation
			current_file.put_string (c_ge_raise)
			current_file.put_character ('(')
			current_file.put_string (c_ge_ex_prog)
			current_file.put_character (')')
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			print_indentation
			current_file.put_line ("done = (char) 1;")
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			print_indentation
			if a_result_type_set /= Void then
				l_result_type := a_result_type_set.static_type.primary_type
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_declaration_type_cast_from_c (l_result_type, current_file)
				current_file.put_character ('(')
			end
			current_file.put_character ('(')
			current_file.put_character ('(')
			if l_result_type /= Void then
				if a_signature_result /= Void then
					current_file.put_string (a_signature_result)
				else
					print_type_declaration (l_result_type, current_file)
				end
			else
				current_file.put_string (c_void)
			end
			current_file.put_string (" (__stdcall *) ")
			current_file.put_character ('(')
			if a_arguments /= Void and then not a_arguments.is_empty then
				nb_args := a_arguments.count
				if a_signature_arguments /= Void then
					current_file.put_string (a_signature_arguments)
				else
					from i := 1 until i > nb_args loop
						if i /= 1 then
							current_file.put_character (',')
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type = current_dynamic_system.pointer_type then
								-- When compiling with C++, MSVC++ does not want to convert
								-- 'void*' to non-'void*' implicitly.
							current_file.put_string ("char*")
						elseif l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_dllwin_body", 1, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
							end
						else
							print_type_declaration (l_argument_type, current_file)
						end
						i := i + 1
					end
				end
			end
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string ("fp")
			current_file.put_character (')')
			if a_arguments /= Void and then nb_args > 0 then
				current_file.put_character ('(')
				if a_signature_arguments /= Void then
					create l_splitter.make_with_separators (",")
					l_list := l_splitter.split (a_signature_arguments)
					if l_list.count /= nb_args then
-- TODO: error
					end
					l_cursor := l_list.new_cursor
					l_cursor.start
					from i := 1 until i > nb_args loop
						if i /= 1 then
							current_file.put_character (',')
						end
						if not l_cursor.after then
							current_file.put_character ('(')
							current_file.put_string (l_cursor.item)
							current_file.put_character (')')
							l_cursor.forth
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_dllwin_body", 2, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							if l_argument_type = current_dynamic_system.pointer_type then
									-- Get rid of the volatile type marker.
								current_file.put_character ('(')
								current_file.put_string (c_void)
								current_file.put_character ('*')
								current_file.put_character (')')
							end
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				else
					from i := 1 until i > nb_args loop
						if i /= 1 then
							current_file.put_character (',')
						end
						l_name := a_arguments.formal_argument (i).name
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						if l_argument_type = current_dynamic_system.pointer_type then
								-- When compiling with C++, MSVC++ does not want to convert
								-- 'void*' to non-'void*' implicitly.
							current_file.put_string ("(char*)")
							print_argument_name (l_name, current_file)
						elseif l_argument_type.base_class.is_typed_pointer_class then
								-- The argument is declared of type 'TYPED_POINTER [XX]'.
								-- In that case we use the corresponding pointer (i.e.
								-- the first attribute of the object).
							l_actual_parameters := l_argument_type.base_type.actual_parameters
							if l_actual_parameters = Void or else l_actual_parameters.is_empty then
									-- Internal error: TYPED_POINTER [XX] has one generic parameter.
									-- This should have been checked already.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_external_dllwin_body", 3, "missing generic parameter for type 'TYPED_POINTER'.")
							else
								l_actual_parameter := current_dynamic_system.dynamic_primary_type (l_actual_parameters.type (1), current_system.any_type)
								current_file.put_character ('(')
								print_type_declaration (l_actual_parameter, current_file)
								if l_actual_parameter.is_expanded then
									current_file.put_character ('*')
								end
								current_file.put_character (')')
								current_file.put_character ('(')
								print_argument_name (l_name, current_file)
								current_file.put_character ('.')
								current_file.put_character ('a')
								current_file.put_character ('1')
								current_file.put_character (')')
							end
						else
							if l_argument_type = current_dynamic_system.pointer_type then
									-- Get rid of the volatile type marker.
								current_file.put_character ('(')
								current_file.put_string (c_void)
								current_file.put_character ('*')
								current_file.put_character (')')
							end
							print_argument_name (l_name, current_file)
						end
						i := i + 1
					end
				end
				current_file.put_character (')')
			else
				current_file.put_character ('(')
				current_file.put_character (')')
			end
			if a_result_type_set /= Void then
				current_file.put_character (')')
			end
			current_file.put_character (';')
			current_file.put_new_line
		end

	print_internal_function (a_feature: ET_INTERNAL_FUNCTION)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_internal_function", 1, "inconsistent `current_feature'.")
			else
				if current_feature.is_regular then
					print_internal_feature (a_feature, False)
				end
				if current_feature.is_address then
					print_address_routine (a_feature)
				end
			end
		end

	print_internal_procedure (a_feature: ET_INTERNAL_PROCEDURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_internal_procedure", 1, "inconsistent `current_feature'.")
			else
				if current_feature.is_regular then
					print_internal_feature (a_feature, False)
				end
				if current_feature.is_creation then
					print_internal_feature (a_feature, True)
				end
				if current_feature.is_address then
					print_address_routine (a_feature)
				end
			end
		end

	print_internal_feature (a_feature: ET_INTERNAL_FEATURE; a_creation: BOOLEAN)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
			is_creation: a_creation implies current_feature.is_creation
		local
			l_result_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			i, nb_args: INTEGER
			old_file: KI_TEXT_OUTPUT_STREAM
			l_name: ET_IDENTIFIER
			l_is_empty: BOOLEAN
			l_once_kind: INTEGER
			l_once_index: INTEGER
			l_is_once: BOOLEAN
			l_is_once_per_process: BOOLEAN
			l_is_once_per_thread: BOOLEAN
			l_is_once_per_object: BOOLEAN
			l_is_self_initialized_attribute: BOOLEAN
		do
			old_file := current_file
			current_file := current_function_header_buffer
			print_feature_name_comment (current_feature, current_type, header_file)
			print_feature_name_comment (current_feature, current_type, current_file)
			l_result_type_set := current_feature.result_type_set
			if l_result_type_set /= Void then
				l_result_type := l_result_type_set.static_type.primary_type
			end
				--
				-- Print signature to `header_file' and `current_file'.
				--
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			if l_result_type /= Void then
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
			elseif a_creation then
				print_type_declaration (current_type, header_file)
				print_type_declaration (current_type, current_file)
			else
				header_file.put_string (c_void)
				current_file.put_string (c_void)
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			if a_creation then
				print_creation_procedure_name (current_feature, current_type, header_file)
				print_creation_procedure_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
			else
				print_routine_name (current_feature, current_type, header_file)
				print_routine_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				if not current_feature.is_static then
					header_file.put_character (',')
					header_file.put_character (' ')
					current_file.put_character (',')
					current_file.put_character (' ')
					print_type_declaration (current_type, header_file)
					print_type_declaration (current_type, current_file)
					if current_type.is_expanded then
						if current_type.is_basic then
								-- Note that non-basic expanded types are already declared as volatile.
							header_file.put_character (' ')
							header_file.put_string (c_volatile)
							current_file.put_character (' ')
							current_file.put_string (c_volatile)
						end
						header_file.put_character ('*')
						current_file.put_character ('*')
					end
					header_file.put_character (' ')
					current_file.put_character (' ')
					print_current_name (header_file)
					print_current_name (current_file)
				end
			end
			l_arguments := a_feature.arguments
			if l_arguments /= Void then
				nb_args := l_arguments.count
				if nb_args > 0 then
					from i := 1 until i > nb_args loop
						header_file.put_character (',')
						current_file.put_character (',')
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_argument_type_set := argument_type_set (i)
						l_argument_type := l_argument_type_set.static_type.primary_type
						print_type_declaration (l_argument_type, header_file)
						print_type_declaration (l_argument_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_name := l_arguments.formal_argument (i).name
						print_argument_name (l_name, header_file)
						print_argument_name (l_name, current_file)
						i := i + 1
					end
				end
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			if current_type.base_class.invariants_enabled and not current_feature.is_static then
				print_indentation
				current_file.put_string (c_uint32_t)
				current_file.put_character (' ')
				current_file.put_string (c_in_qualified_call)
				print_assign_to
				if a_creation then
					current_file.put_character ('2')
				else
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_in_qualified_call)
				end
				print_semicolon_newline
			end
				-- Call stack.
			if exception_trace_mode then
				l_is_empty := not a_creation and then (not attached a_feature.compound as l_compound or else l_compound.is_empty) and then a_feature.rescue_clause = Void
				if not l_is_empty then
					print_indentation
					current_file.put_string (c_ge_call)
					current_file.put_character (' ')
					current_file.put_string (c_tc)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('{')
					if current_in_exception_trace then
						if current_feature.is_static or a_creation then
							current_file.put_character ('0')
						else
							print_current_name (current_file)
						end
						current_file.put_character (',')
					end
					print_escaped_string (current_type.base_class.upper_name)
					current_file.put_character (',')
					print_escaped_string (a_feature.lower_name)
					current_file.put_character (',')
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character ('}')
					current_file.put_character (';')
					current_file.put_new_line
				end
			end
				--
				-- Print body to `current_file'.
				--
			current_file := current_function_body_buffer
			if a_feature.is_once then
				l_is_once := True
				l_is_once_per_process := a_feature.is_once_per_process
				l_is_once_per_thread := a_feature.is_once_per_thread
				l_is_once_per_object := a_feature.is_once_per_object
				if l_is_once_per_process or l_is_once_per_thread then
					register_once_feature (current_feature)
					l_once_index := once_features.value (a_feature.implementation_feature)
					l_once_kind := once_kind (current_feature)
				end
				if l_is_once_per_object then
						-- Create once-per-object data for 'Current' object if not created yet.
					if use_threads then
						print_once_per_object_global_mutex_lock
					end
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('!')
					print_attribute_onces_access (tokens.current_keyword, current_type, False)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					print_attribute_onces_access (tokens.current_keyword, current_type, False)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_ge_new_once_per_object_data)
					current_file.put_integer (current_type.id)
					current_file.put_character ('(')
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					if use_threads then
						print_once_per_object_global_mutex_unlock
					end
				end
				if not use_threads or else l_is_once_per_thread then
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_once_status (current_feature, l_once_kind, l_once_index)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_raise_once_exception_if_any (current_feature, l_once_kind, l_once_index)
					print_return_once_value (current_feature, l_once_kind, l_once_index)
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				else
					if use_threads and then l_is_once_per_process then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_thread_safe_once_per_process_status (current_feature, l_once_kind, l_once_index)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
						print_raise_thread_safe_once_exception_if_any (current_feature, l_once_kind, l_once_index)
						print_return_thread_safe_once_value (current_feature, l_once_kind, l_once_index)
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_character (' ')
						current_file.put_string (c_else)
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
					end
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('!')
					print_once_mutex_try_lock (current_feature, l_once_kind, l_once_index)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
						-- Wait for other thread to complete (lock/unlock).
					print_once_mutex_lock (current_feature, l_once_kind, l_once_index)
					print_once_mutex_unlock (current_feature, l_once_kind, l_once_index)
					if l_result_type /= Void then
						print_assign_once_value_to_result (current_feature, l_once_kind, l_once_index)
						if l_is_once_per_process then
							print_assign_result_to_thread_safe_once_value (current_feature, l_once_kind, l_once_index)
						end
					end
					if l_is_once_per_process then
						print_assign_once_exception_to_thread_safe_once_exception (current_feature, l_once_kind, l_once_index)
						print_assign_called_to_thread_safe_once_status (current_feature, l_once_kind, l_once_index)
					end
					print_raise_once_exception_if_any (current_feature, l_once_kind, l_once_index)
					print_return_statement (a_feature)
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_once_status_is_completed (current_feature, l_once_kind, l_once_index)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					if l_result_type /= Void then
						print_assign_once_value_to_result (current_feature, l_once_kind, l_once_index)
						if l_is_once_per_process then
							print_assign_result_to_thread_safe_once_value (current_feature, l_once_kind, l_once_index)
						end
					end
					if l_is_once_per_process then
						print_assign_once_exception_to_thread_safe_once_exception (current_feature, l_once_kind, l_once_index)
						print_assign_called_to_thread_safe_once_status (current_feature, l_once_kind, l_once_index)
					end
					print_once_mutex_unlock (current_feature, l_once_kind, l_once_index)
					print_raise_once_exception_if_any (current_feature, l_once_kind, l_once_index)
					print_return_statement (a_feature)
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_once_status (current_feature, l_once_kind, l_once_index)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_return_once_value (current_feature, l_once_kind, l_once_index)
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				end
				print_assign_called_to_once_status (current_feature, l_once_kind, l_once_index)
				if l_result_type /= Void then
						-- The following is to make sure that if a once-function is called
						-- recursively, the semantics specified by ECMA will be satisfied.
						-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
						-- to the once-function should return the value of 'Result' as it
						-- was when the recursive calls occurred.
					print_assign_result_to_once_value (current_feature, l_once_kind, l_once_index)
				end
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
				if use_threads and then l_is_once_per_process then
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				end
			elseif a_feature.is_attribute then
					-- Self-initialized attribute.
				l_is_self_initialized_attribute := True
					-- Return the value of the attribute if already initialized.
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_attribute_access (current_feature, tokens.current_keyword, current_type, False)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_attribute_access (current_feature, tokens.current_keyword, current_type, False)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			end
				-- Call stack.
			if exception_trace_mode and then not l_is_empty then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_tc_address)
				current_file.put_character (';')
				current_file.put_new_line
			end
			print_feature_trace_message_call (True)
			if a_creation then
				print_malloc_current (a_feature)
				if exception_trace_mode and then current_in_exception_trace then
					print_indentation
					current_file.put_string (c_tc)
					current_file.put_character ('.')
					current_file.put_string (c_object)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_current_name (current_file)
					current_file.put_character (';')
					current_file.put_new_line
				end
			end
			print_internal_feature_body_declaration (a_feature, l_result_type, a_creation)
			print_feature_trace_message_call (False)
				-- Call stack.
			if exception_trace_mode and then not l_is_empty then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_tc)
				current_file.put_character ('.')
				current_file.put_string (c_caller)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if l_is_once then
				if l_result_type /= Void and then (l_result_type.is_expanded and then not l_result_type.is_basic) then
						-- The following is to make sure that if a once-function is called
						-- recursively, the semantics specified by ECMA will be satisfied.
						-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
						-- to the once-function should return the value of 'Result' as it
						-- was when the recursive calls occurred.
						-- There is a special treatment for once-per-process and once-per-thread
						-- functions of non-basic expanded types because a boxed version of the
						-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
						-- So we usede the content of the boxed value in place of Result
						-- in case its fields get modified. Likewise for once-per-object functions
						-- even though the Result is not boxed in that case. (See `print_result'.)
						-- Set it back to Result here.
					print_assign_once_value_to_result (current_feature, l_once_kind, l_once_index)
				end
				if use_threads then
					if l_is_once_per_process then
						if l_result_type /= Void then
							print_assign_result_to_thread_safe_once_value (current_feature, l_once_kind, l_once_index)
						end
						print_assign_called_to_thread_safe_once_status (current_feature, l_once_kind, l_once_index)
					end
					if not l_is_once_per_thread then
						print_assign_completed_to_once_status (current_feature, l_once_kind, l_once_index)
						print_once_mutex_unlock (current_feature, l_once_kind, l_once_index)
					end
				end
			elseif l_is_self_initialized_attribute then
					-- Record value of self-initialized attribute.
				print_indentation
				print_attribute_access (current_feature, tokens.current_keyword, current_type, False)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if l_result_type /= Void then
				print_return_statement (a_feature)
			elseif a_creation then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				if current_type.is_expanded then
					current_file.put_character ('*')
				end
				print_current_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
		end

	print_internal_feature_body_declaration (a_feature: ET_INTERNAL_FEATURE_CLOSURE; a_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE; a_creation: BOOLEAN)
			-- Print the local variable declarations, the compound and the
			-- rescue clause of `a_feature' to `current_file'.
		require
			a_feature_not_void: a_feature /= Void
		local
			i, nb: INTEGER
			l_name: ET_IDENTIFIER
			l_argument_type: ET_DYNAMIC_TYPE
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			l_argument: ET_FORMAL_ARGUMENT
			l_local_type_set: ET_DYNAMIC_TYPE_SET
			l_local_type: ET_DYNAMIC_PRIMARY_TYPE
			l_rescue: detachable ET_COMPOUND
			old_file: KI_TEXT_OUTPUT_STREAM
			l_result_written_in_body: BOOLEAN
			l_result_read_in_body: BOOLEAN
			l_result_written_in_rescue: BOOLEAN
			l_result_read_in_rescue: BOOLEAN
			l_is_once: BOOLEAN
			l_once_kind: INTEGER
			l_once_index: INTEGER
			l_has_separate_arguments: BOOLEAN
			l_has_separate_formal_arguments: BOOLEAN
			l_separate_formal_arguments_count: INTEGER
		do
			old_file := current_file
			current_file := current_function_header_buffer
				--
				-- Declaration of variables.
				--
				-- Variable for rescue chain.
			if current_agent = Void then
				if current_feature.is_once then
					l_is_once := True
					once_features.search (current_feature.static_feature.implementation_feature)
					if once_features.found then
						l_once_index := once_features.found_item
						l_once_kind := once_kind (current_feature)
					end
				end
			end
			if use_scoop then
					-- Declaration of SCOOP sessions to register separate calls,
					-- even for those of inline separate instructions contained
					-- in `a_feature'.
				if attached a_feature.inline_separate_arguments as l_inline_separate_arguments then
					nb := l_inline_separate_arguments.count
					l_has_separate_arguments := nb > 0
					from i := 1 until i > nb loop
						print_indentation
						current_file.put_string (c_ge_scoop_session)
						current_file.put_character ('*')
						current_file.put_character (' ')
						print_separate_argument_session_name (l_inline_separate_arguments.argument (i).name, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_character ('0')
						print_semicolon_newline
						i := i + 1
					end
				end
				l_arguments := a_feature.arguments
				if l_arguments /= Void then
					nb := l_arguments.count
					from i := 1 until i > nb loop
						l_argument := l_arguments.formal_argument (i)
						l_name := l_argument.name
						l_argument_type := dynamic_type_set (l_name).static_type
						if l_argument_type.is_separate then
							l_has_separate_formal_arguments := True
							l_separate_formal_arguments_count := l_separate_formal_arguments_count + 1
							l_has_separate_arguments := True
							print_indentation
							current_file.put_string (c_ge_scoop_session)
							current_file.put_character ('*')
							current_file.put_character (' ')
							print_separate_argument_session_name (l_name, current_file)
							current_file.put_character (' ')
							current_file.put_character ('=')
							current_file.put_character (' ')
							current_file.put_character ('0')
							print_semicolon_newline
						end
						i := i + 1
					end
				end
				if l_has_separate_arguments then
					print_indentation
					current_file.put_string (c_ge_scoop_region)
					current_file.put_character ('*')
					current_file.put_character (' ')
					current_file.put_string (c_sr)
					print_assign_to
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_region)
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_scoop_condition)
					current_file.put_character ('*')
					current_file.put_character (' ')
					current_file.put_string (c_scond)
					print_assign_to
					current_file.put_character ('0')
					print_semicolon_newline
				end
			end
			l_rescue := a_feature.rescue_clause
			if l_rescue /= Void or l_is_once then
				print_indentation
				current_file.put_string (c_ge_rescue)
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_volatile)
				current_file.put_character (' ')
				current_file.put_string (c_uint32_t)
				current_file.put_character (' ')
				current_file.put_string (c_tr)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_in_rescue)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if l_has_separate_arguments then
				print_indentation
				current_file.put_string (c_ge_rescue)
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character (';')
				current_file.put_new_line
			end
				--
				-- Instructions.
				--
			current_file := current_function_body_buffer
			if current_type.base_class.invariants_enabled and not current_closure.is_static and not a_creation then
				print_all_invariants (True)
			end
			if current_type.base_class.preconditions_enabled then
				print_all_preconditions (a_feature)
			end
			if current_type.base_class.postconditions_enabled then
				print_all_old_expression_declarations (a_feature)
			end
			if l_has_separate_formal_arguments then
					-- Start the current SCOOP sessions.
				if l_separate_formal_arguments_count > 1 then
					print_indentation
					current_file.put_string (c_ge_scoop_multisessions_open_start)
					current_file.put_character ('(')
					current_file.put_character (')')
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_scond)
					print_assign_to
					current_file.put_string (c_ge_new_scoop_condition)
					current_file.put_character ('(')
					current_file.put_integer (l_separate_formal_arguments_count)
					current_file.put_character (')')
					print_semicolon_newline
				end
				check l_has_separate_formal_arguments: l_arguments /= Void then end
				nb := l_arguments.count
				from i := 1 until i > nb loop
					l_argument := l_arguments.formal_argument (i)
					l_name := l_argument.name
					l_argument_type := dynamic_type_set (l_name).static_type
					if l_argument_type.is_separate then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_formal_argument (l_name)
						current_file.put_character (')')
						current_file.put_character (' ')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_open)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_attribute_region_access (l_name, l_argument_type.primary_type, False)
						print_comma
						current_file.put_string (c_scond)
						current_file.put_character (')')
						if l_separate_formal_arguments_count > 1 then
							current_file.put_character (';')
							current_file.put_character (' ')
							current_file.put_string (c_else)
							current_file.put_character (' ')
							current_file.put_string (c_ge_scoop_condition_decrement)
							current_file.put_character ('(')
							current_file.put_string (c_scond)
							current_file.put_character (')')
						end
						print_semicolon_newline
					end
					i := i + 1
				end
				if l_separate_formal_arguments_count > 1 then
					print_indentation
					current_file.put_string (c_ge_scoop_multisessions_open_stop)
					current_file.put_character ('(')
					current_file.put_character (')')
					print_semicolon_newline
				end
			end
			reset_volatile_data
			reset_rescue_data
			locals_written := locals_written_in_rescue
			locals_read := locals_read_in_rescue
			if l_has_separate_arguments then
				print_indentation
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				current_file.put_string (c_ge_setjmp)
				current_file.put_character ('(')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character ('.')
				current_file.put_character ('j')
				current_file.put_character ('b')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('!')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character (';')
				current_file.put_new_line
					-- Make sure to exit from the current SCOOP sessions
					-- before propagating the exception.
				if l_has_separate_formal_arguments then
					check l_has_separate_formal_arguments: l_arguments /= Void then end
					nb := l_arguments.count
					from i := 1 until i > nb loop
						l_argument := l_arguments.formal_argument (i)
						l_name := l_argument.name
						l_argument_type := dynamic_type_set (l_name).static_type
						if l_argument_type.is_separate then
							print_indentation
							current_file.put_string (c_if)
							current_file.put_character (' ')
							current_file.put_character ('(')
							print_separate_argument_session_name (l_name, current_file)
							current_file.put_character (')')
							current_file.put_character (' ')
							current_file.put_string (c_ge_scoop_session_close)
							current_file.put_character ('(')
							current_file.put_string (c_sr)
							print_comma
							print_separate_argument_session_name (l_name, current_file)
							current_file.put_character (')')
							print_semicolon_newline
						end
						i := i + 1
					end
				end
					-- Even the SCOOP sessions from the inline separate instructions.
				if attached a_feature.inline_separate_arguments as l_inline_separate_arguments then
					nb := l_inline_separate_arguments.count
					from i := 1 until i > nb loop
						l_name := l_inline_separate_arguments.argument (i).name
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_close)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						print_semicolon_newline
						i := i + 1
					end
				end
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ge_jump_to_last_rescue)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			end
			if l_rescue /= Void or l_is_once then
				print_indentation
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				current_file.put_string (c_ge_setjmp)
				current_file.put_character ('(')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_character ('j')
				current_file.put_character ('b')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('!')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				if exception_trace_mode and then l_rescue /= Void then
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_tc_address)
					current_file.put_character (';')
					current_file.put_new_line
				end
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_in_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_tr)
				current_file.put_character (' ')
				current_file.put_character ('+')
				current_file.put_character (' ')
				current_file.put_character ('1')
				current_file.put_character (';')
				current_file.put_new_line
				if l_rescue /= Void then
					has_rescue := True
					print_compound (l_rescue)
				end
				if l_is_once then
					print_assign_last_exception_to_once_exception (current_feature, l_once_kind, l_once_index)
					if use_threads then
						if a_feature.is_once_per_process then
							print_assign_once_exception_to_thread_safe_once_exception (current_feature, l_once_kind, l_once_index)
							print_assign_called_to_thread_safe_once_status (current_feature, l_once_kind, l_once_index)
						end
						if not current_feature.is_once_per_thread then
							print_assign_completed_to_once_status (current_feature, l_once_kind, l_once_index)
							print_once_mutex_unlock (current_feature, l_once_kind, l_once_index)
						end
					end
				end
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
				if l_rescue /= Void then
					print_indentation
					current_file.put_string (c_ge_raise)
					current_file.put_character ('(')
					current_file.put_string (c_ge_ex_fail)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				else
					print_indentation
					current_file.put_string (c_ge_jump_to_last_rescue)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				end
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
				if l_rescue /= Void then
					current_file.put_string (c_ge_retry)
					current_file.put_character (':')
					current_file.put_new_line
				end
			end
			l_result_written_in_rescue := result_written
			l_result_read_in_rescue := result_read
			locals_written := locals_written_in_body
			locals_read := locals_read_in_body
			if attached a_feature.compound as l_compound then
				print_compound (l_compound)
			end
			if current_type.base_class.invariants_enabled and not current_closure.is_static then
				print_all_invariants (False)
			end
			if current_type.base_class.postconditions_enabled then
				print_all_postconditions (a_feature)
			end
			l_result_written_in_body := result_written
			l_result_read_in_body := result_read
			if l_rescue /= Void or l_is_once then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if a_result_type /= Void then
					-- The 'Result' entity is always implicitly read in the body to return its value.
				l_result_read_in_body := True
			end
			if l_has_separate_arguments then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('2')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if l_has_separate_formal_arguments then
					-- Exit from the current SCOOP sessions.
				check l_has_separate_formal_arguments: l_arguments /= Void then end
				nb := l_arguments.count
				from i := 1 until i > nb loop
					l_argument := l_arguments.formal_argument (i)
					l_name := l_argument.name
					l_argument_type := dynamic_type_set (l_name).static_type
					if l_argument_type.is_separate then
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_close)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						print_semicolon_newline
					end
					i := i + 1
				end
			end
				--
				-- Local variables and result declaration.
				--
			current_file := current_function_header_buffer
				-- Variable for 'Result' entity.
			if a_result_type /= Void then
				print_indentation
				print_type_declaration (a_result_type, current_file)
				if
					(not a_result_type.is_expanded or else a_result_type.is_basic) and
					(volatile_result or
					has_rescue and then
						(l_result_written_in_body or l_result_written_in_rescue) and then
						(has_retry or l_result_read_in_rescue))
				then
						-- The implementation of the rescue mechanism in C uses 'setjmp'
						-- and 'longjmp'. The use of these two C functions requires that
						-- any local variable modified between the call to 'setjmp' and
						-- the call to 'longjmp' to be declared as 'volatile', otherwise its
						-- value may be lost after calling 'longjmp' if the C optimizer
						-- decided to implement it with 'register'.
						--
						-- Likewise, if the address of 'Result' is used, we have to make
						-- sure that he C optimizer will not implement it with 'register'.
						--
						-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
						--
						-- Note that non-basic expanded types are already declared as volatile.
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_default_entity_value (a_result_type, current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
				-- Local variables.
			if attached a_feature.locals as l_locals then
				nb := l_locals.count
				from i := 1 until i > nb loop
					l_name := l_locals.local_variable (i).name
					l_local_type_set := dynamic_type_set (l_name)
					l_local_type := l_local_type_set.static_type.primary_type
					print_indentation
					print_type_declaration (l_local_type, current_file)
					if
						(not l_local_type.is_expanded or else l_local_type.is_basic) and
						(volatile_locals.has (l_name.seed) or
						has_rescue and then
							(locals_written_in_body.has (l_name.seed) or locals_written_in_rescue.has (l_name.seed)) and then
							(has_retry or locals_read_in_rescue.has (l_name.seed)))
					then
							-- The implementation of the rescue mechanism in C uses 'setjmp'
							-- and 'longjmp'. The use of these two C functions requires that
							-- any local variable modified between the call to 'setjmp' and
							-- the call to 'longjmp' to be declared as 'volatile', otherwise its
							-- value may be lost after calling 'longjmp' if the C optimizer
							-- decided to implement it with 'register'.
							--
							-- Likewise, if the address of the local variable is used, we
							-- have to make sure that the C optimizer will not implement it
							-- with 'register'.
							--
							-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
							--
							-- Note that non-basic expanded types are already declared as volatile.
						current_file.put_character (' ')
						current_file.put_string (c_volatile)
					end
					current_file.put_character (' ')
					print_local_name (l_name, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_default_entity_value (l_local_type, current_file)
					current_file.put_character (';')
					current_file.put_new_line
					i := i + 1
				end
			end
			print_object_test_local_declarations
			print_iteration_cursor_declarations
			print_inline_separate_argument_declarations (a_feature)
				-- Clean up.
			reset_volatile_data
			reset_rescue_data
			current_file := old_file
		end

	print_once_function (a_feature: ET_ONCE_FUNCTION)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			print_internal_function (a_feature)
		end

	print_once_procedure (a_feature: ET_ONCE_PROCEDURE)
			-- Print `a_feature' to `current_file' and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			print_internal_procedure (a_feature)
		end

	print_attribute (a_feature: ET_ATTRIBUTE)
			-- Print function wrapper for `a_feature' to `current_file'
			-- and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_attribute", 1, "inconsistent `current_feature'.")
			else
				print_attribute_wrapper (a_feature)
			end
		end

	print_extended_attribute (a_feature: ET_EXTENDED_ATTRIBUTE)
			-- Print function wrapper for `a_feature' to `current_file'
			-- and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if
				a_feature.has_self_initializing_code or
				(current_type.base_class.preconditions_enabled and then not current_feature.preconditions.is_empty) or
				(current_type.base_class.postconditions_enabled and then not current_feature.postconditions.is_empty) or
				current_type.base_class.invariants_enabled
			then
				print_internal_feature (a_feature, False)
			else
				print_attribute (a_feature)
			end
		end

	print_constant_attribute (a_feature: ET_CONSTANT_ATTRIBUTE)
			-- Print function wrapper for `a_feature' to `current_file'
			-- and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_constant_attribute", 1, "inconsistent `current_feature'.")
			elseif current_feature.is_regular then
				print_attribute_wrapper (a_feature)
			end
		end

	print_unique_attribute (a_feature: ET_UNIQUE_ATTRIBUTE)
			-- Print function wrapper for `a_feature' to `current_file'
			-- and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
		do
			if current_feature.static_feature /= a_feature then
					-- Internal error: inconsistent `current_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_unique_attribute", 1, "inconsistent `current_feature'.")
			elseif current_feature.is_regular then
				print_attribute_wrapper (a_feature)
			end
		end

	print_attribute_wrapper (a_feature: ET_QUERY)
			-- Print function wrapper for `a_feature' to `current_file'
			-- and its signature to `header_file'.
		require
			a_feature_not_void: a_feature /= Void
			no_arguments: a_feature.arguments = Void
			valid_feature: current_feature.static_feature = a_feature
		local
			old_file: KI_TEXT_OUTPUT_STREAM
			l_result_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_wrapper_expression: ET_UNQUALIFIED_CALL_EXPRESSION
			l_name: ET_FEATURE_NAME
			l_result: ET_RESULT
			l_index: INTEGER
		do
			old_file := current_file
			current_file := current_function_header_buffer
				--
				-- Print signature to `header_file' and `current_file'.
				--
			print_feature_name_comment (current_feature, current_type, header_file)
			print_feature_name_comment (current_feature, current_type, current_file)
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			l_result_type_set := current_feature.result_type_set
			if l_result_type_set = Void then
					-- Internal error: a query has a result type set.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_attribute_wrapper", 1, "attribute with no type.")
			else
				l_result_type := l_result_type_set.static_type.primary_type
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
				header_file.put_character (' ')
				current_file.put_character (' ')
				print_routine_name (current_feature, current_type, header_file)
				print_routine_name (current_feature, current_type, current_file)
				header_file.put_character ('(')
				current_file.put_character ('(')
				print_context_type_declaration (header_file)
				header_file.put_character (' ')
				header_file.put_string (c_ac)
				print_context_type_declaration (current_file)
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				if not current_feature.is_static then
					header_file.put_character (',')
					header_file.put_character (' ')
					current_file.put_character (',')
					current_file.put_character (' ')
					print_type_declaration (current_type, header_file)
					print_type_declaration (current_type, current_file)
					if current_type.is_expanded then
						if current_type.is_basic then
								-- Note that non-basic expanded types are already declared as volatile.
							header_file.put_character (' ')
							header_file.put_string (c_volatile)
							current_file.put_character (' ')
							current_file.put_string (c_volatile)
						end
						header_file.put_character ('*')
						current_file.put_character ('*')
					end
					header_file.put_character (' ')
					current_file.put_character (' ')
					print_current_name (header_file)
					print_current_name (current_file)
				end
				header_file.put_character (')')
				current_file.put_character (')')
				header_file.put_character (';')
				header_file.put_new_line
				current_file.put_new_line
					--
					-- Print body to `current_file'.
					--
				current_file.put_character ('{')
				current_file.put_new_line
				indent
					--
					-- Declaration of variables.
					--
				if current_type.base_class.invariants_enabled then
					print_indentation
					current_file.put_string (c_uint32_t)
					current_file.put_character (' ')
					current_file.put_string (c_in_qualified_call)
					print_assign_to
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_in_qualified_call)
					print_semicolon_newline
				end
					-- Variable for exception trace.
				if exception_trace_mode then
					print_indentation
					current_file.put_string (c_ge_call)
					current_file.put_character (' ')
					current_file.put_string (c_tc)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('{')
					if current_in_exception_trace then
						print_current_name (current_file)
						current_file.put_character (',')
					end
					print_escaped_string (current_type.base_class.upper_name)
					current_file.put_character (',')
					print_escaped_string (a_feature.lower_name)
					current_file.put_character (',')
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character ('}')
					current_file.put_character (';')
					current_file.put_new_line
				end
					-- Variable for 'Result' entity.
				print_indentation
				print_type_declaration (l_result_type, current_file)
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_default_entity_value (l_result_type, current_file)
				current_file.put_character (';')
				current_file.put_new_line
					--
					-- Instructions.
					--
				current_file := current_function_body_buffer
					-- Call stack.
				if exception_trace_mode then
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_tc_address)
					current_file.put_character (';')
					current_file.put_new_line
				end
					--
					-- Note that the precondition is True in attributes
					-- (implicit 'require True' on immediate declarations
					-- and 'require else True' on redeclarations).
					--
				if current_type.base_class.postconditions_enabled then
					print_all_old_expression_declarations (a_feature)
				end
					-- Prepare dynamic type sets of wrapper feature.
				extra_dynamic_type_sets.force_last (l_result_type_set)
				l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
					-- Prepare call expression to attribute feature.
				l_name := a_feature.name
				l_name.set_seed (a_feature.first_seed)
				l_wrapper_expression := new_unqualified_call_expression (l_name, Void)
				l_wrapper_expression.set_index (l_index)
					-- Print assignment of call expression to result entity.
				l_result := tokens.result_keyword
				print_assignment_operand (l_wrapper_expression, l_result_type_set, l_result, l_result_type)
				fill_call_operands (1)
				if call_operands.first /= l_result then
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_expression (call_operands.first)
					current_file.put_character (';')
					current_file.put_new_line
				end
				if current_type.base_class.invariants_enabled and not current_feature.is_static then
					print_all_invariants (False)
				end
				if current_type.base_class.postconditions_enabled then
					print_all_postconditions (a_feature)
				end
					-- Clean up.
				call_operands.wipe_out
				extra_dynamic_type_sets.remove_last
				free_unqualified_call_expression (l_wrapper_expression)
					-- Call stack.
				if exception_trace_mode then
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_call)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_tc)
					current_file.put_character ('.')
					current_file.put_string (c_caller)
					current_file.put_character (';')
					current_file.put_new_line
				end
					-- Return attribute value.
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				current_file.put_character ('}')
				current_file.put_new_line
				current_file.put_new_line
			end
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
		end

	print_invariants_wrapper
			-- Print procedure which checks all invariants (even those inherited
			-- from ancestors) of `current_type' to `current_file'.
		local
			old_file: KI_TEXT_OUTPUT_STREAM
			l_old_index_offset: INTEGER
			l_invariants: ET_INVARIANTS
			i, nb: INTEGER
		do
			old_file := current_file
			current_file := current_function_header_buffer
				--
				-- Print signature to `header_file' and `current_file'.
				--
			print_feature_name_comment (current_feature, current_type, header_file)
			print_feature_name_comment (current_feature, current_type, current_file)
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			header_file.put_string (c_void)
			header_file.put_character (' ')
			current_file.put_string (c_void)
			current_file.put_character (' ')
			print_invariants_name (current_type, header_file)
			print_invariants_name (current_type, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			current_file.put_character (',')
			current_file.put_character (' ')
			print_type_declaration (current_type, header_file)
			print_type_declaration (current_type, current_file)
			if current_type.is_expanded then
				if current_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					header_file.put_character (' ')
					header_file.put_string (c_volatile)
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				header_file.put_character ('*')
				current_file.put_character ('*')
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_current_name (header_file)
			print_current_name (current_file)
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				--
				-- Print body to `current_file'.
				--
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			current_file := current_function_body_buffer
			l_old_index_offset := index_offset
			if attached current_feature.invariants as l_all_invariants then
				nb := l_all_invariants.count
				if nb > 0 then
					print_before_assertions
					from i := 1 until i > nb loop
						index_offset := l_all_invariants.item_2 (i)
						l_invariants := l_all_invariants.item_1 (i)
						print_assertions (l_invariants, c_ge_ex_cinv)
						i := i + 1
					end
					print_after_assertions
				end
				index_offset := l_old_index_offset
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
			print_object_test_local_declarations
			print_iteration_cursor_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			reset_volatile_data
			reset_rescue_data
			current_file := old_file
		end

	print_object_test_local_declarations
			-- Print the object-test local declarations  used in the
			-- generated C function to `current_file'.
		local
			i, nb: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			nb := current_object_test_local_types.count
			from i := 1 until i > nb loop
				l_type := current_object_test_local_types.item (i)
				print_indentation
				print_type_declaration (l_type, current_file)
				if
					(not l_type.is_expanded or else l_type.is_basic) and
					volatile_object_test_locals.has (i)
				then
						-- If the address of the object-test local is used, we have
						-- to make sure that the C optimizer will not implement it
						-- with 'register'.
						--
						-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
						--
						-- Note that non-basic expanded types are already declared as volatile.
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				current_file.put_character (' ')
				current_object_test_locals.force_last (i, object_test_local_name.seed)
				print_object_test_local_name (object_test_local_name, current_file)
				print_assign_to
				print_default_entity_value (l_type, current_file)
				print_semicolon_newline
				i := i + 1
			end
			current_object_test_locals.wipe_out
			current_object_test_local_types.wipe_out
		end

	print_iteration_cursor_declarations
			-- Print the iteration cursor declarations used in the
			-- generated C function to `current_file'.
		local
			i, nb: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			nb := current_iteration_cursor_types.count
			from i := 1 until i > nb loop
				l_type := current_iteration_cursor_types.item (i)
				print_indentation
				print_type_declaration (l_type, current_file)
				if
					(not l_type.is_expanded or else l_type.is_basic) and
					volatile_iteration_cursors.has (i)
				then
						-- If the address of the iteration cursor is used, we have
						-- to make sure that the C optimizer will not implement it
						-- with 'register'.
						--
						-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
						--
						-- Note that non-basic expanded types are already declared as volatile.
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				current_file.put_character (' ')
				current_iteration_cursors.force_last (i, iteration_cursor_name.seed)
				print_iteration_cursor_name (iteration_cursor_name, current_file)
				print_semicolon_newline
				i := i + 1
			end
			current_iteration_cursors.wipe_out
			current_iteration_cursor_types.wipe_out
		end

	print_inline_separate_argument_declarations (a_feature: ET_CLOSURE)
			-- Print the inline separate argument declarations of
			-- `a_feature' to `current_file'.
		require
			a_feature_not_void: a_feature /= Void
		local
			i, nb: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if attached a_feature.inline_separate_arguments as l_inline_separate_arguments then
				nb := l_inline_separate_arguments.count
				from i := 1 until i > nb loop
					if attached l_inline_separate_arguments.argument (i).name as l_name then
						if current_inline_separate_arguments.has (l_name.seed) then
							l_type := dynamic_type_set (l_name).static_type.primary_type
							print_indentation
							print_type_declaration (l_type, current_file)
							if
								(not l_type.is_expanded or else l_type.is_basic) and
								volatile_inline_separate_arguments.has (l_name.seed)
							then
									-- If the address of the inline separate argument is used, we have
									-- to make sure that the C optimizer will not implement it
									-- with 'register'.
									--
									-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
									--
									-- Note that non-basic expanded types are already declared as volatile.
								current_file.put_character (' ')
								current_file.put_string (c_volatile)
							end
							current_file.put_character (' ')
							print_inline_separate_argument_name (l_name, current_file)
							print_semicolon_newline
						end
					end
					i := i + 1
				end
			end
			current_inline_separate_arguments.wipe_out
		end

	print_temporary_variable_declarations
			-- Print the temporary variable declarations used in the
			-- generated C function to `current_file'.
		local
			i, nb: INTEGER
			l_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_name: ET_IDENTIFIER
		do
			indent
			nb := used_temp_variables.count
			from i := 1 until i > nb loop
				l_type := used_temp_variables.item (i)
				if l_type = Void then
					l_type := free_temp_variables.item (i)
				end
				if l_type /= Void then
					l_name := temp_variables.item (i)
					print_indentation
					print_type_declaration (l_type, current_file)
					if
						(not l_type.is_expanded or else l_type.is_basic) and
						volatile_temp_variables.item (i)
					then
							-- If the address of the temporary variable is used, we have
							-- to make sure that the C optimizer will not implement it
							-- with 'register'.
							--
							-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
							--
							-- Note that non-basic expanded types are already declared as volatile.
						current_file.put_character (' ')
						current_file.put_string (c_volatile)
					end
					current_file.put_character (' ')
					print_temp_name (l_name, current_file)
					print_semicolon_newline
				end
				i := i + 1
			end
			dedent
			reset_temp_variables
		end

feature {NONE} -- Instruction generation

	print_across_instruction (an_instruction: ET_ACROSS_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			print_iteration_instruction (an_instruction)
		end

	print_assigner_instruction (an_instruction: ET_ASSIGNER_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			print_qualified_call_instruction (an_instruction)
		end

	print_assignment (an_instruction: ET_ASSIGNMENT)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_source: ET_EXPRESSION
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_source_type: ET_DYNAMIC_TYPE
			l_target: ET_WRITABLE
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_once_kind: INTEGER
			l_once_index: INTEGER
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_source := an_instruction.source
			l_source_type_set := dynamic_type_set (l_source)
			l_target := an_instruction.target
			l_target_type_set := dynamic_type_set (l_target)
			l_source_type := l_source_type_set.static_type
			l_target_type := l_target_type_set.static_type
			print_assignment_operand (l_source, l_source_type_set, l_target, l_target_type)
			fill_call_operands (1)
			if call_operands.first /= l_target then
				print_indentation
				print_writable (l_target)
				print_assign_to
				if an_instruction.no_source_twin then
					print_expression (call_operands.first)
				else
					print_attachment_expression (call_operands.first, l_source_type_set, l_target_type)
				end
				print_semicolon_newline
			end
			call_operands.wipe_out
			if current_agent = Void and then current_feature.is_once then
					-- Make sure that if a once-function is called
					-- recursively, the semantics specified by ECMA will be satisfied.
					-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
					-- to the once-function should return the value of 'Result' as it
					-- was when the recursive calls occurred.
					-- There is a special treatment for once-per-process and once-per-thread
					-- functions of non-basic expanded types because a boxed version of the
					-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
					-- So we need to use the content of the boxed value in place of Result
					-- in case its fields get modified. Likewise for once-per-object functions
					-- even though the Result is not boxed in that case. (See `print_result'.)
				if l_target.is_result and then not (l_target_type.is_expanded and then not l_target_type.is_basic) then
					once_features.search (current_feature.static_feature.implementation_feature)
					if once_features.found then
						l_once_index := once_features.found_item
						l_once_kind := once_kind (current_feature)
					end
					print_assign_result_to_once_value (current_feature, l_once_kind, l_once_index)
				end
			end
		end

	print_assignment_attempt (an_instruction: ET_ASSIGNMENT_ATTEMPT)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			i, nb: INTEGER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_source: ET_EXPRESSION
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_source_type: ET_DYNAMIC_TYPE
			l_target: ET_WRITABLE
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_non_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_conforming_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
			l_old_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_can_be_void: BOOLEAN
			l_once_index: INTEGER
			l_once_kind: INTEGER
			l_same_scoop_region: BOOLEAN
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_source := an_instruction.source
			l_source_type_set := dynamic_type_set (l_source)
			l_source_type := l_source_type_set.static_type
			l_target := an_instruction.target
			l_target_type_set := dynamic_type_set (l_target)
			l_target_type := l_target_type_set.static_type
			l_target_primary_type := l_target_type.primary_type
			nb := l_source_type_set.count
			l_conforming_types := conforming_types
			l_conforming_types.resize (nb)
			l_non_conforming_types := non_conforming_types
			l_non_conforming_types.resize (nb)
			from i := 1 until i > nb loop
				l_dynamic_type := l_source_type_set.dynamic_type (i)
				if l_dynamic_type.conforms_to_primary_type (l_target_primary_type) then
					l_conforming_types.put_last (l_dynamic_type)
				else
					l_non_conforming_types.put_last (l_dynamic_type)
				end
				i := i + 1
			end
			print_attachment_operand (l_source, l_source_type_set)
			fill_call_operands (1)
			if l_non_conforming_types.is_empty then
					-- Direct assignment.
				print_indentation
				print_writable (l_target)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attachment_expression (call_operands.first, l_source_type_set, l_target_type)
				current_file.put_character (';')
				current_file.put_new_line
			elseif l_conforming_types.is_empty then
				if l_target_type.is_expanded then
					-- ISE 6.3 does not change the value of the target
					-- when the type of the target is expanded and the
					-- assignment attempt fails. Note that it would have
					-- made more sense to set the target to its default
					-- value in that case.
				else
					print_indentation
					print_writable (l_target)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_void)
					current_file.put_character (';')
					current_file.put_new_line
				end
			else
				l_can_be_void := l_source_type_set.can_be_void
				if l_can_be_void then
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('(')
					print_expression (call_operands.first)
					current_file.put_character (')')
					current_file.put_character ('=')
					current_file.put_character ('=')
					current_file.put_string (c_eif_void)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					if l_target_type.is_expanded then
						-- ISE 6.3 does not change the value of the target
						-- when the type of the target is expanded and the
						-- assignment attempt fails. Note that it would have
						-- made more sense to set the target to its default
						-- value in that case.
					else
						print_indentation
						print_writable (l_target)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_eif_void)
						current_file.put_character (';')
						current_file.put_new_line
					end
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
				end
				l_same_scoop_region := use_scoop and then (not l_target_type.is_separate and l_source_type.is_separate)
				if l_same_scoop_region then
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_attribute_region_access (call_operands.first, l_target_primary_type, False)
					current_file.put_character (' ')
					current_file.put_character ('!')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_region)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					if l_target_type.is_expanded then
						-- ISE 6.3 does not change the value of the target
						-- when the type of the target is expanded and the
						-- assignment attempt fails. Note that it would have
						-- made more sense to set the target to its default
						-- value in that case.
					else
						print_indentation
						print_writable (l_target)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_eif_void)
						current_file.put_character (';')
						current_file.put_new_line
					end
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
				end
				if l_can_be_void or l_same_scoop_region then
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				end
				check
						-- None of `l_non_conforming_types' and `l_conforming_types' are empty.
						-- Therefore the source is polymorphic (at least two dynamic types, one
						-- conforming and one not conforming). As a consequence `l_source_type'
						-- is not expanded because for expanded types there is no type other
						-- than itself that conforms to it.
						-- So we know that a type-id is always available.
					not_expanded: not l_source_type.is_expanded
				end
				print_indentation
				current_file.put_string (c_switch)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_attribute_type_id_access (call_operands.first, l_source_type.primary_type, False)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				l_conforming_type_set := new_standalone_type_set (l_source_type)
				l_old_conforming_types := l_conforming_type_set.dynamic_types
				l_conforming_type_set.reset_with_types (l_source_type, l_conforming_types)
				if l_source_type_set.is_never_void then
					l_conforming_type_set.set_never_void
				end
				if l_non_conforming_types.count < l_conforming_types.count then
					nb := l_non_conforming_types.count
					from i := 1 until i > nb loop
						l_dynamic_type := l_non_conforming_types.dynamic_type (i)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						i := i + 1
					end
					indent
					if l_target_type.is_expanded then
						-- ISE 6.3 does not change the value of the target
						-- when the type of the target is expanded and the
						-- assignment attempt fails. Note that it would have
						-- made more sense to set the target to its default
						-- value in that case.
					else
						print_indentation
						print_writable (l_target)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_eif_void)
						current_file.put_character (';')
						current_file.put_new_line
					end
					print_indentation
					current_file.put_string (c_break)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_string (c_default)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					print_indentation
					print_writable (l_target)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_attachment_expression (call_operands.first, l_conforming_type_set, l_target_type)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
				else
					nb := l_conforming_types.count
					from i := 1 until i > nb loop
						l_dynamic_type := l_conforming_types.dynamic_type (i)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						i := i + 1
					end
					indent
					print_indentation
					print_writable (l_target)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_attachment_expression (call_operands.first, l_conforming_type_set, l_target_type)
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_break)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_string (c_default)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					if l_target_type.is_expanded then
						-- ISE 6.3 does not change the value of the target
						-- when the type of the target is expanded and the
						-- assignment attempt fails.
					else
						print_indentation
						print_writable (l_target)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_eif_void)
						current_file.put_character (';')
						current_file.put_new_line
					end
					dedent
				end
				l_conforming_type_set.reset_with_types (current_dynamic_system.unknown_type, l_old_conforming_types)
				free_standalone_type_set (l_conforming_type_set)
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
				if l_can_be_void or l_same_scoop_region then
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				end
			end
			l_conforming_types.wipe_out
			l_non_conforming_types.wipe_out
			call_operands.wipe_out
			if current_agent = Void and then current_feature.is_once then
					-- Make sure that if a once-function is called
					-- recursively, the semantics specified by ECMA will be satisfied.
					-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
					-- to the once-function should return the value of 'Result' as it
					-- was when the recursive calls occurred.
					-- There is a special treatment for once-per-process and once-per-thread
					-- functions of non-basic expanded types because a boxed version of the
					-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
					-- So we need to use the content of the boxed value in place of Result
					-- in case its fields get modified. Likewise for once-per-object functions.
					-- even though the Result is not boxed in that case. (See `print_result'.)
				if l_target.is_result and then not (l_target_type.is_expanded and then not l_target_type.is_basic) then
					once_features.search (current_feature.static_feature.implementation_feature)
					if once_features.found then
						l_once_index := once_features.found_item
						l_once_kind := once_kind (current_feature)
					end
					print_assign_result_to_once_value (current_feature, l_once_kind, l_once_index)
				end
			end
		end

	print_bang_instruction (an_instruction: ET_BANG_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			print_creation_instruction (an_instruction)
		end

	print_check_instruction (an_instruction: ET_CHECK_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			if attached an_instruction.then_compound as l_then_compound then
				if line_generation_mode then
					print_position (an_instruction.position, current_feature.static_feature.implementation_class)
				end
				print_assertions (an_instruction, c_ge_ex_check)
				print_compound (l_then_compound)
			elseif current_type.base_class.check_assertions_enabled and then not an_instruction.are_all_true then
				if line_generation_mode then
					print_position (an_instruction.position, current_feature.static_feature.implementation_class)
				end
				print_before_assertions
				print_assertions (an_instruction, c_ge_ex_check)
				print_after_assertions
			else
				-- Do not check assertions at run-time, or nothing to check.
			end
		end

	print_compound (a_compound: ET_COMPOUND)
			-- Print `a_compound'.
		require
			a_compound_not_void: a_compound /= Void
		local
			i, nb: INTEGER
		do
			nb := a_compound.count
			from i := 1 until i > nb loop
				print_instruction (a_compound.item (i))
				i := i + 1
			end
		end

	print_create_instruction (an_instruction: ET_CREATE_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			print_creation_instruction (an_instruction)
		end

	print_creation_instruction (an_instruction: ET_CREATION_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_target: ET_WRITABLE
			l_dynamic_type: ET_DYNAMIC_TYPE
			l_dynamic_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
			l_static_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_seed: INTEGER
			l_actuals: detachable ET_ACTUAL_ARGUMENT_LIST
			l_actual: ET_EXPRESSION
			l_dynamic_procedure: detachable ET_DYNAMIC_FEATURE
			i, nb: INTEGER
			l_once_index: INTEGER
			l_once_kind: INTEGER
			l_is_separate_call: BOOLEAN
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
				-- Look for the static and dynamic types of the creation.
			l_target := an_instruction.target
			l_dynamic_type_set := dynamic_type_set (l_target)
			l_static_type := l_dynamic_type_set.static_type
			l_static_primary_type := l_static_type.primary_type
			if attached an_instruction.type as l_type then
				l_dynamic_type := current_dynamic_system.dynamic_type (l_type, current_type.base_type)
			else
					-- Look for the type of the target.
				l_dynamic_type := l_static_type
			end
			l_dynamic_primary_type := l_dynamic_type.primary_type
			if attached an_instruction.creation_call as l_creation_call then
				l_seed := l_creation_call.name.seed
				l_actuals := l_creation_call.arguments
			else
				l_seed := current_system.default_create_seed
				l_actuals := Void
			end
			l_dynamic_procedure := l_dynamic_primary_type.seeded_dynamic_procedure (l_seed, current_dynamic_system)
			if l_dynamic_procedure = Void then
					-- Internal error: there should be a procedure with `l_seed'.
					-- It has been computed in ET_FEATURE_CHECKER or else an
					-- error should have already been reported.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_creation_instruction", 1, "creation procedure not found.")
			else
				if l_actuals /= Void then
					nb := l_actuals.count
					from i := 1 until i > nb loop
						l_actual := l_actuals.actual_argument (i)
						print_attachment_operand (l_actual, dynamic_type_set (l_actual))
						i := i + 1
					end
				end
				fill_call_operands (nb)
				print_indentation
				print_writable (l_target)
				print_assign_to
				if use_scoop and l_dynamic_type.is_separate then
					print_adapted_separate_creation_procedure_call (an_instruction, l_dynamic_procedure, l_dynamic_primary_type, l_static_primary_type)
					l_is_separate_call := True
				else
					print_adapted_creation_procedure_call (l_dynamic_procedure, l_dynamic_primary_type, l_static_primary_type)
				end
				print_semicolon_newline
				call_operands.wipe_out
			end
			if current_agent = Void and then current_feature.is_once then
					-- Make sure that if a once-function is called
					-- recursively, the semantics specified by ECMA will be satisfied.
					-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
					-- to the once-function should return the value of 'Result' as it
					-- was when the recursive calls occurred.
					-- There is a special treatment for once-per-process and once-per-thread
					-- functions of non-basic expanded types because a boxed version of the
					-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
					-- So we need to use the content of the boxed value in place of Result
					-- in case its fields get modified. Likewise for once-per-object functions.
					-- even though the Result is not boxed in that case. (See `print_result'.)
				if l_target.is_result and then not (l_static_primary_type.is_expanded and then not l_static_primary_type.is_basic) then
					once_features.search (current_feature.static_feature.implementation_feature)
					if once_features.found then
						l_once_index := once_features.found_item
						l_once_kind := once_kind (current_feature)
					end
					print_assign_result_to_once_value (current_feature, l_once_kind, l_once_index)
				end
			end
			if l_is_separate_call then
					-- Print the separate call declaration now, otherwise some
					-- temporary data (temporary dynamic type sets, reused calls,
					-- etc.) may not be available anymore.
				print_separate_call_declaration (current_separate_call_count, an_instruction, current_closure)
			end
		end

	print_debug_instruction (an_instruction: ET_DEBUG_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			-- Do nothing.
		end

	print_if_instruction (an_instruction: ET_IF_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_elseif: ET_ELSEIF_PART
			i, nb: INTEGER
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			print_operand (an_instruction.expression)
			fill_call_operands (1)
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			if attached an_instruction.then_compound as l_then_compound then
				indent
				print_compound (l_then_compound)
				dedent
			end
			print_indentation
			current_file.put_character ('}')
			if attached an_instruction.elseif_parts as l_elseif_parts then
				nb := l_elseif_parts.count
				from i := 1 until i > nb loop
					l_elseif := l_elseif_parts.item (i)
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					if line_generation_mode then
						print_position (l_elseif.position, current_feature.static_feature.implementation_class)
					end
					indent
					print_operand (l_elseif.expression)
					fill_call_operands (1)
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_expression (call_operands.first)
					call_operands.wipe_out
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					if attached l_elseif.then_compound as l_then_compound then
						indent
						print_compound (l_then_compound)
						dedent
					end
					print_indentation
					current_file.put_character ('}')
					i := i + 1
				end
			end
			if attached an_instruction.else_compound as l_else_compound then
				current_file.put_character (' ')
				current_file.put_string (c_else)
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				print_compound (l_else_compound)
				dedent
				print_indentation
				current_file.put_character ('}')
			end
			from i := 1 until i > nb loop
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				i := i + 1
			end
			current_file.put_new_line
		end

	print_inline_separate_instruction (a_instruction: ET_INLINE_SEPARATE_INSTRUCTION)
			-- Print `a_instruction'.
		require
			a_instruction_not_void: a_instruction /= Void
		local
			l_arguments: ET_INLINE_SEPARATE_ARGUMENTS
			l_argument: ET_INLINE_SEPARATE_ARGUMENT
			l_name: ET_IDENTIFIER
			i, nb: INTEGER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_TYPE
			l_dynamic_primary_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if line_generation_mode then
				print_position (a_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_arguments := a_instruction.arguments
			nb := l_arguments.count
			from i := 1 until i > nb loop
				l_argument := l_arguments.argument (i)
				l_name := l_argument.name
				current_inline_separate_arguments.force_last (l_name.seed)
				l_dynamic_type_set := dynamic_type_set (l_name)
				l_dynamic_type := l_dynamic_type_set.static_type
					-- Call to inline separate argument expression.
				print_assignment_operand (l_argument.expression, l_dynamic_type_set, l_name, l_dynamic_type)
				fill_call_operands (1)
				if call_operands.first /= l_name then
					print_indentation
					print_inline_separate_argument_name (l_name, current_file)
					print_assign_to
					print_attachment_expression (call_operands.first, l_dynamic_type_set, l_dynamic_type)
					print_semicolon_newline
				end
				call_operands.wipe_out
				i := i + 1
			end
			if attached a_instruction.compound as l_compound then
				if use_scoop then
						-- Start the current SCOOP sessions.
					if nb > 1 then
						print_indentation
						current_file.put_string (c_ge_scoop_multisessions_open_start)
						current_file.put_character ('(')
						current_file.put_character (')')
						print_semicolon_newline
						print_indentation
						current_file.put_string (c_scond)
						print_assign_to
						current_file.put_string (c_ge_new_scoop_condition)
						current_file.put_character ('(')
						current_file.put_integer (nb)
						current_file.put_character (')')
						print_semicolon_newline
					else
						print_indentation
						current_file.put_string (c_scond)
						print_assign_to
						current_file.put_character ('0')
						print_semicolon_newline
					end
					from i := 1 until i > nb loop
						l_argument := l_arguments.argument (i)
						l_name := l_argument.name
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_inline_separate_argument_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
						print_indentation
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session_open)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						l_dynamic_type_set := dynamic_type_set (l_name)
						l_dynamic_primary_type := l_dynamic_type_set.static_type.primary_type
						print_attribute_region_access (l_name, l_dynamic_primary_type, False)
						print_comma
						current_file.put_string (c_scond)
						current_file.put_character (')')
						print_semicolon_newline
						dedent
						print_indentation
						current_file.put_character ('}')
						if nb > 1 then
							current_file.put_character (' ')
							current_file.put_string (c_else)
							current_file.put_character (' ')
							current_file.put_character ('{')
							current_file.put_new_line
							indent
							print_indentation
							current_file.put_string (c_ge_scoop_condition_decrement)
							current_file.put_character ('(')
							current_file.put_string (c_scond)
							current_file.put_character (')')
							print_semicolon_newline
							dedent
							print_indentation
							current_file.put_character ('}')
						end
						current_file.put_new_line
						i := i + 1
					end
					if nb > 1 then
						print_indentation
						current_file.put_string (c_ge_scoop_multisessions_open_stop)
						current_file.put_character ('(')
						current_file.put_character (')')
						print_semicolon_newline
					end
				end
				print_compound (l_compound)
				if use_scoop then
						-- Exit from the current SCOOP sessions.
					from i := 1 until i > nb loop
						l_argument := l_arguments.argument (i)
						l_name := l_argument.name
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
						print_indentation
						current_file.put_string (c_ge_scoop_session_close)
						current_file.put_character ('(')
						current_file.put_string (c_sr)
						print_comma
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (')')
						print_semicolon_newline
							-- Set the session to NULL so that we don't try to
							-- exit again from it in case of an exception.
						print_indentation
						print_separate_argument_session_name (l_name, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_character ('0')
						print_semicolon_newline
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_new_line
						i := i + 1
					end
				end
			end
		end

	print_inspect_instruction (an_instruction: ET_INSPECT_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_expression: ET_EXPRESSION
			l_when_part: ET_WHEN_PART
			l_choices: ET_CHOICE_LIST
			l_choice: ET_CHOICE
			i, nb: INTEGER
			j, nb2: INTEGER
			l_has_case: BOOLEAN
			l_lower: detachable ET_CONSTANT
			l_upper: detachable ET_CONSTANT
			l_lower_integer: detachable ET_INTEGER_CONSTANT
			l_upper_integer: detachable ET_INTEGER_CONSTANT
			l_lower_character: detachable ET_CHARACTER_CONSTANT
			l_upper_character: detachable ET_CHARACTER_CONSTANT
			k, nb3: INTEGER
			l_i_nat32, l_nb_nat32: NATURAL_32
			l_value_type_set: ET_DYNAMIC_TYPE_SET
			l_value_type: ET_DYNAMIC_PRIMARY_TYPE
			l_stop: BOOLEAN
		do
-- TODO.
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_expression := an_instruction.conditional.expression
			l_value_type_set := dynamic_type_set (l_expression)
			l_value_type := l_value_type_set.static_type.primary_type
			print_operand (l_expression)
			fill_call_operands (1)
			print_indentation
			current_file.put_string (c_switch)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			if attached an_instruction.when_parts as l_when_parts then
				nb := l_when_parts.count
				from i := 1 until i > nb loop
					l_when_part := l_when_parts.item (i)
					if line_generation_mode then
						print_position (l_when_part.position, current_feature.static_feature.implementation_class)
					end
					l_choices := l_when_part.choices
					nb2 := l_choices.count
					if nb2 = 0 then
						-- Do nothing.
					else
						l_has_case := False
						from j := 1 until j > nb2 loop
							l_choice := l_choices.choice (j)
							if l_choice.is_range then
-- TODO
								l_lower := choice_constant (l_choice.lower)
								l_upper := choice_constant (l_choice.upper)
								if attached {ET_INTEGER_CONSTANT} l_lower as l_integer_constant then
									l_lower_integer := l_integer_constant
									l_lower_character := Void
								elseif attached {ET_CHARACTER_CONSTANT} l_lower as l_character_constant then
									l_lower_character := l_character_constant
									l_lower_integer := Void
								else
									l_lower_integer := Void
									l_lower_character := Void
								end
								if attached {ET_INTEGER_CONSTANT} l_upper as l_integer_constant then
									l_upper_integer := l_integer_constant
									l_upper_character := Void
								elseif attached {ET_CHARACTER_CONSTANT} l_upper as l_character_constant then
									l_upper_character := l_character_constant
									l_upper_integer := Void
								else
									l_upper_integer := Void
									l_upper_character := Void
								end
								if l_lower_integer /= Void and l_upper_integer /= Void then
									from
-- TODO: check type of inspect value.
										k := l_lower_integer.to_integer_32
										nb3 := l_upper_integer.to_integer_32
									until
										k > nb3
									loop
										l_has_case := True
										print_indentation
										current_file.put_string (c_case)
										current_file.put_character (' ')
										print_integer_value (k.abs.to_natural_64, k < 0, l_value_type, True)
										current_file.put_character (':')
										current_file.put_new_line
										k := k + 1
									end
								elseif l_lower_character /= Void and l_upper_character /= Void then
									from
										l_i_nat32 := l_lower_character.value.natural_32_code
										l_nb_nat32 := l_upper_character.value.natural_32_code
										l_stop := l_i_nat32 > l_nb_nat32
									until
										l_stop
									loop
										l_has_case := True
										print_indentation
										current_file.put_string (c_case)
										current_file.put_character (' ')
										if current_system.character_32_type.same_named_type (l_value_type.base_type, current_type.base_type, current_type.base_type) then
											current_file.put_string (c_ge_nat32)
											current_file.put_character ('(')
											current_file.put_natural_32 (l_i_nat32)
											current_file.put_character (')')
										else
												-- Type cast needed when compiling with mingw when the character
												-- is represented by '\xyz' where x is not 0.
											print_type_cast (l_value_type, current_file)
											print_escaped_character_8 (l_i_nat32.to_character_8)
										end
										current_file.put_character (':')
										current_file.put_new_line
										if l_i_nat32 = l_nb_nat32 then
											l_stop := True
										else
											l_i_nat32 := l_i_nat32 + 1
										end
									end
								else
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_inspect_instruction - range")
								end
							else
								l_has_case := True
								print_indentation
								current_file.put_string (c_case)
								current_file.put_character (' ')
								l_lower := choice_constant (l_choice.lower)
								if attached {ET_INTEGER_CONSTANT} l_lower as l_integer_constant then
									print_integer_value (l_integer_constant.value, l_integer_constant.is_negative, l_value_type, True)
								elseif attached {ET_CHARACTER_CONSTANT} l_lower as l_character_constant then
									if current_system.character_32_type.same_named_type (l_value_type.base_type, current_type.base_type, current_type.base_type) then
										current_file.put_string (c_ge_nat32)
										current_file.put_character ('(')
										current_file.put_natural_32 (l_character_constant.value.natural_32_code)
										current_file.put_character (')')
									else
											-- Type cast needed when compiling with mingw when the character
											-- is represented by '\xyz' where x is not 0.
										print_type_cast (l_value_type, current_file)
										print_escaped_character_8 (l_character_constant.value.to_character_8)
									end
								else
									if l_value_type /= dynamic_type_set (l_choice.lower).static_type.primary_type then
										print_type_cast (l_value_type, current_file)
									end
									print_expression (l_choice.lower)
								end
								current_file.put_character (':')
								current_file.put_new_line
							end
							j := j + 1
						end
						if l_has_case then
							indent
							if attached l_when_part.then_compound as l_then_compound then
								print_compound (l_then_compound)
							end
							print_indentation
							current_file.put_string (c_break)
							current_file.put_character (';')
							current_file.put_new_line
							dedent
						end
					end
					i := i + 1
				end
			end
			print_indentation
			current_file.put_string (c_default)
			current_file.put_character (':')
			current_file.put_new_line
			if attached an_instruction.else_compound as l_else_compound then
				indent
				print_compound (l_else_compound)
				print_indentation
				current_file.put_string (c_break)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
			else
				indent
				print_indentation
				current_file.put_string (c_ge_raise)
				current_file.put_character ('(')
				current_file.put_string (c_ge_ex_when)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_break)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
			end
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
		end

	choice_constant (a_choice_constant: ET_CHOICE_CONSTANT): detachable ET_CONSTANT
			-- Constant value of the inspect choice `a_choice_constant',
			-- or Void if it appears not to be a constant after all
			--
			-- Note that validity `a_choice_constant' as an expression
			-- is assumed to have already been checked.
			-- It will just return Void if something unexpected is encountered.
		require
			a_choice_constant_not_void: a_choice_constant /= Void
		local
			l_query: detachable ET_QUERY
			l_seed: INTEGER
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if attached {ET_CONSTANT} a_choice_constant as l_constant then
				Result := l_constant
			else
				if attached {ET_IDENTIFIER} a_choice_constant as l_identifier then
					if l_identifier.is_argument then
							-- This is not a constant.
					elseif l_identifier.is_local then
							-- This is not a constant.
					elseif l_identifier.is_object_test_local then
							-- This is not a constant.
					elseif l_identifier.is_iteration_item then
							-- This is not a constant.
					else
						l_seed := l_identifier.seed
						l_query := current_type.base_class.seeded_query (l_seed)
					end
				elseif attached {ET_UNQUALIFIED_FEATURE_CALL} a_choice_constant as l_unqualified_call and then l_unqualified_call.arguments_count = 0 then
					l_seed := l_unqualified_call.name.seed
					l_query := current_type.base_class.seeded_query (l_seed)
				elseif attached {ET_STATIC_CALL_EXPRESSION} a_choice_constant as l_static_call then
					l_target_type := current_dynamic_system.dynamic_primary_type (l_static_call.type, current_type.base_type)
					l_query := l_target_type.base_class.seeded_query (l_static_call.name.seed)
				end
				if attached {ET_CONSTANT_QUERY} l_query as l_constant_query then
					Result := l_constant_query.constant
				end
			end
		end

	print_instruction (an_instruction: ET_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			an_instruction.process (Current)
		end

	print_iteration_instruction (an_instruction: ET_ITERATION_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_expression: ET_EXPRESSION
			l_cursor_name: ET_IDENTIFIER
			l_cursor_type: ET_DYNAMIC_TYPE
			l_cursor_type_set: ET_DYNAMIC_TYPE_SET
			l_temp_var: detachable ET_IDENTIFIER
			l_temp_old_var: detachable ET_IDENTIFIER
			l_var_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_cursor_name := an_instruction.unfolded_cursor_name
			l_cursor_type_set := dynamic_type_set (l_cursor_name)
			l_cursor_type := l_cursor_type_set.static_type
			current_iteration_cursor_types.force_last (l_cursor_type.primary_type)
			current_iteration_cursors.force_last (current_iteration_cursor_types.count, l_cursor_name.seed)
				-- Call to `new_cursor'.
			print_assignment_operand (an_instruction.new_cursor_expression, l_cursor_type_set, l_cursor_name, l_cursor_type)
			fill_call_operands (1)
			if call_operands.first /= l_cursor_name then
				print_indentation
				print_iteration_cursor_name (l_cursor_name, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attachment_expression (call_operands.first, l_cursor_type_set, l_cursor_type)
				current_file.put_character (';')
				current_file.put_new_line
			end
			call_operands.wipe_out
			if attached an_instruction.from_compound as l_from_compound then
				print_compound (l_from_compound)
			end
			if current_type.base_class.loop_assertions_enabled and attached an_instruction.variant_part as l_variant_part then
				l_var_type := dynamic_type_set (l_variant_part.expression).static_type.primary_type
				l_temp_var := new_temp_variable (l_var_type)
				l_temp_old_var := new_temp_variable (l_var_type)
				print_indentation
				print_temp_name (l_temp_old_var, current_file)
				print_assign_to
				current_file.put_integer (-1)
				print_semicolon_newline
			end
			print_indentation
			current_file.put_string (c_while)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('1')
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			if current_type.base_class.loop_assertions_enabled then
				print_loop_assertions_and_while_header (an_instruction, l_temp_var, l_temp_old_var)
			end
			print_operand (an_instruction.cursor_after_expression)
			if attached an_instruction.until_conditional as l_until_conditional then
				l_expression := l_until_conditional.expression
				print_operand (l_expression)
				fill_call_operands (2)
			else
				fill_call_operands (1)
			end
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			current_file.put_character (')')
			if call_operands.count = 2 then
				print_or_else
				current_file.put_character ('(')
				print_expression (call_operands.item (2))
				current_file.put_character (')')
			end
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			current_file.put_string (c_break)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if attached an_instruction.loop_compound as l_loop_compound then
				print_compound (l_loop_compound)
			end
			print_instruction (an_instruction.cursor_forth_instruction)
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if l_temp_var /= Void then
				mark_temp_variable_free (l_temp_var)
			end
			if l_temp_old_var /= Void then
				mark_temp_variable_free (l_temp_old_var)
			end
		end

	print_loop_instruction (an_instruction: ET_LOOP_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_expression: ET_EXPRESSION
			l_temp_var: detachable ET_IDENTIFIER
			l_temp_old_var: detachable ET_IDENTIFIER
			l_var_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			if attached an_instruction.from_compound as l_from_compound and then not l_from_compound.is_empty then
				print_compound (l_from_compound)
			end
			if current_type.base_class.loop_assertions_enabled and attached an_instruction.variant_part as l_variant_part then
				l_var_type := dynamic_type_set (l_variant_part.expression).static_type.primary_type
				l_temp_var := new_temp_variable (l_var_type)
				l_temp_old_var := new_temp_variable (l_var_type)
				print_indentation
				print_temp_name (l_temp_old_var, current_file)
				print_assign_to
				current_file.put_integer (-1)
				print_semicolon_newline
			end
			print_indentation
			current_file.put_string (c_while)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('1')
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			if current_type.base_class.loop_assertions_enabled then
				print_loop_assertions_and_while_header (an_instruction, l_temp_var, l_temp_old_var)
			end
			l_expression := an_instruction.until_expression
			if line_generation_mode then
				print_position (l_expression.position, current_feature.static_feature.implementation_class)
			end
			print_operand (l_expression)
			fill_call_operands (1)
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			current_file.put_string (c_break)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if attached an_instruction.loop_compound as l_loop_compound then
				print_compound (l_loop_compound)
			end
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if l_temp_var /= Void then
				mark_temp_variable_free (l_temp_var)
			end
			if l_temp_old_var /= Void then
				mark_temp_variable_free (l_temp_old_var)
			end
		end

	print_precursor_instruction (an_instruction: ET_PRECURSOR_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		local
			l_precursor_keyword: ET_PRECURSOR_KEYWORD
			l_procedure: detachable ET_PROCEDURE
			l_parent_type, l_ancestor: detachable ET_BASE_TYPE
			l_class: ET_CLASS
			l_current_class: ET_CLASS
			l_class_impl: ET_CLASS
			l_dynamic_precursor: ET_DYNAMIC_PRECURSOR
			l_parent_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			i, nb: INTEGER
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_actual: ET_EXPRESSION
			had_error: BOOLEAN
			l_old_in_qualified_call: BOOLEAN
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := False
			l_parent_type := an_instruction.parent_type
			if l_parent_type = Void then
					-- Internal error: the Precursor construct should already
					-- have been resolved when flattening the features of the
					-- implementation class of current feature.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_precursor_instruction", 1, "precursor with no parent type.")
			else
				had_error := has_fatal_error
				has_fatal_error := False
				if l_parent_type.is_generic then
					l_current_class := current_type.base_class
					l_class_impl := current_feature.static_feature.implementation_class
					if l_current_class /= l_class_impl then
							-- Resolve generic parameters in the context of `current_type'.
						if not l_current_class.ancestors_built or else l_current_class.has_ancestors_error then
								-- 'ancestor_builder' should already have been executed
								-- on `l_current_class' at this stage, and any error
								-- should have already been reported.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_precursor_instruction", 2, "ancestors not built.")
						else
							l_ancestor := l_current_class.ancestor (l_parent_type)
							if l_ancestor = Void then
									-- Internal error: `l_parent_type' is an ancestor
									-- of `l_class_impl', and hence of `l_current_class'.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_precursor_instruction", 3, "parent type not found.")
							else
								l_parent_type := l_ancestor
							end
						end
					end
				end
				if not has_fatal_error then
					has_fatal_error := had_error
					l_precursor_keyword := an_instruction.precursor_keyword
					l_class := l_parent_type.base_class
					l_procedure := l_class.seeded_procedure (l_precursor_keyword.seed)
					if l_procedure = Void then
							-- Internal error: the Precursor construct should
							-- already have been resolved when flattening the
							-- features of `l_class_impl'.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_precursor_instruction", 4, "precursor procedure not found.")
					else
						l_parent_dynamic_type := current_dynamic_system.dynamic_primary_type (l_parent_type, current_type.base_type)
						l_dynamic_precursor := current_feature.dynamic_precursor (l_procedure, l_parent_dynamic_type, current_dynamic_system)
						operand_stack.force (tokens.current_keyword)
						if attached an_instruction.arguments as l_actuals then
							nb := l_actuals.count
							from i := 1 until i > nb loop
								l_actual := l_actuals.actual_argument (i)
								l_actual_type_set := dynamic_type_set (l_actual)
								print_attachment_operand (l_actual, l_actual_type_set)
								i := i + 1
							end
						end
						nb := nb + 1
						fill_call_operands (nb)
						print_procedure_call (l_dynamic_precursor, current_type, False)
						call_operands.wipe_out
					end
				end
			end
			in_qualified_call := l_old_in_qualified_call
		end

	print_qualified_call_instruction (a_call: ET_QUALIFIED_FEATURE_CALL_INSTRUCTION)
			-- Print qualified call instruction.
		require
			a_call_not_void: a_call /= Void
		local
			l_name: ET_CALL_NAME
			l_target: ET_EXPRESSION
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_manifest_tuple_operand: detachable ET_MANIFEST_TUPLE
			l_actuals: detachable ET_ACTUAL_ARGUMENTS
			l_actual: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_target_operand: ET_EXPRESSION
			l_seed: INTEGER
			i, nb: INTEGER
			l_target_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			nb2: INTEGER
			l_switch: BOOLEAN
			l_printed: BOOLEAN
			l_is_separate_call: BOOLEAN
			l_old_in_qualified_call: BOOLEAN
		do
			if line_generation_mode then
				print_position (a_call.position, current_feature.static_feature.implementation_class)
			end
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := True
			l_target := a_call.target
			l_target_type_set := dynamic_type_set (l_target)
			l_target_type := l_target_type_set.static_type
			l_target_static_type := l_target_type.primary_type
			l_name := a_call.name
			l_actuals := a_call.arguments
			l_seed := l_name.seed
				-- Check whether this a call to ROUTINE.call with a manifest
				-- tuple as argument. We have a special treatment in that case
				-- to avoid having to create the manifest tuple when possible.
			if
				not (use_scoop and l_target_type.is_separate) and then
				not a_call.is_tuple_label and then
				l_seed = current_system.routine_call_seed and then
				not a_call.is_call_agent
			then
				if l_actuals /= Void and then l_actuals.count = 1 then
					if attached {ET_MANIFEST_TUPLE} l_actuals.actual_argument (1) as l_manifest_tuple then
						print_procedure_target_operand (l_target, l_target_static_type)
						nb := l_manifest_tuple.count
						from i := 1 until i > nb loop
							l_actual := l_manifest_tuple.expression (i)
							l_actual_type_set := dynamic_type_set (l_actual)
							print_attachment_operand (l_actual, l_actual_type_set)
							i := i + 1
						end
						nb := nb + 1
						fill_call_operands (nb)
						l_manifest_tuple_operand := agent_manifest_tuple
						l_manifest_tuple_operand.resize (nb - 1)
						from i := nb until i < 2 loop
							l_manifest_tuple_operand.put_first (call_operands.item (i))
							i := i - 1
						end
						l_manifest_tuple_operand.set_index (l_manifest_tuple.index)
						mark_expressions_frozen (l_manifest_tuple_operand)
						l_target_operand := call_operands.first
						call_operands.wipe_out
						call_operands.put_last (l_target_operand)
						call_operands.put_last (l_manifest_tuple_operand)
						nb := 2
					end
				end
			end
			if l_manifest_tuple_operand = Void then
				print_procedure_target_operand (l_target, l_target_static_type)
				if l_actuals /= Void then
					nb := l_actuals.count
					from i := 1 until i > nb loop
						l_actual := l_actuals.actual_argument (i)
						l_actual_type_set := dynamic_type_set (l_actual)
						print_attachment_operand (l_actual, l_actual_type_set)
						i := i + 1
					end
				end
				nb := nb + 1
				fill_call_operands (nb)
			end
			if use_scoop and a_call /= separate_call_instruction and l_target_type.is_separate then
				print_separate_qualified_call_instruction (a_call)
				l_is_separate_call := True
			else
				nb2 := l_target_type_set.count
				if nb2 = 0 then
						-- Call on Void target.
					print_indentation
					current_file.put_character ('(')
					print_procedure_target_expression (call_operands.first, l_target_static_type, True)
					from i := 2 until i > nb loop
						current_file.put_character (',')
						current_file.put_character (' ')
						print_expression (call_operands.item (i))
						i := i + 1
					end
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				elseif nb2 = 1 then
						-- Static binding.
					l_target_dynamic_type := l_target_type_set.dynamic_type (1)
					print_named_procedure_call (a_call.name, l_target_dynamic_type, True)
				else
						-- Dynamic binding.
					if l_name.is_tuple_label then
-- TODO: fields in Tuples are not necessarily aligned, and some need (un)boxing,
-- for example:
--    t1: TUPLE [a: ANY; b: INTEGER]
--    t2: TUPLE [a: INTEGER_64; b: INTEGER]
--    t1 := t2
--    t1.a := {INTEGER_64} 3
--    t1.b := 1
-- But when there are aligned with no (un)boxing, we should avoid having polymorphic calls.
					end
					if l_printed then
						-- Do nothing.
					elseif nb2 > 2 then
						if not attached current_feature.procedure_call (a_call) as l_dynamic_call then
								-- Internal error: the dynamic call should have been created in ET_DYNAMIC_TYPE_BUILDER.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_qualified_call_instruction", 1, "dynamic call not found.")
						else
							register_polymorphic_called_features (l_dynamic_call)
							print_indentation
							print_call_name (l_dynamic_call, l_target_static_type, False, current_file)
							current_file.put_character ('(')
							current_file.put_string (c_ac)
							current_file.put_character (',')
							current_file.put_character (' ')
							print_procedure_target_expression (call_operands.first, l_target_static_type, True)
							if l_manifest_tuple_operand /= Void then
								nb := l_manifest_tuple_operand.count
								from i := 1 until i > nb loop
									current_file.put_character (',')
									current_file.put_character (' ')
									print_expression (l_manifest_tuple_operand.expression (i))
									i := i + 1
								end
							else
								from i := 2 until i > nb loop
									current_file.put_character (',')
									current_file.put_character (' ')
									print_expression (call_operands.item (i))
									i := i + 1
								end
							end
							current_file.put_character (')')
							current_file.put_character (';')
							current_file.put_new_line
						end
					elseif l_switch then
						print_indentation
						current_file.put_string (c_switch)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_attribute_type_id_access (call_operands.first, l_target_static_type, True)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
							-- First type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (1)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_target_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						indent
						print_named_procedure_call (a_call.name, l_target_dynamic_type, False)
						print_indentation
						current_file.put_string (c_break)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
							-- Second type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (2)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_target_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						indent
						print_named_procedure_call (a_call.name, l_target_dynamic_type, False)
						print_indentation
						current_file.put_string (c_break)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_new_line
					else
							-- First type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (1)
						print_indentation
						current_file.put_string (c_if)
						current_file.put_character (' ')
						current_file.put_character ('(')
						print_attribute_type_id_access (call_operands.first, l_target_static_type, True)
						current_file.put_character ('=')
						current_file.put_character ('=')
						current_file.put_integer (l_target_dynamic_type.id)
						current_file.put_character (')')
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
						print_named_procedure_call (a_call.name, l_target_dynamic_type, False)
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_character (' ')
							-- Second type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (2)
						current_file.put_string (c_else)
						current_file.put_character (' ')
						current_file.put_character ('{')
						current_file.put_new_line
						indent
						print_named_procedure_call (a_call.name, l_target_dynamic_type, False)
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_new_line
					end
				end
			end
			if l_manifest_tuple_operand /= Void then
				mark_expressions_unfrozen (l_manifest_tuple_operand)
				l_manifest_tuple_operand.wipe_out
			end
			call_operands.wipe_out
			in_qualified_call := l_old_in_qualified_call
			if l_is_separate_call then
					-- Print the separate call declaration now, otherwise some
					-- temporary data (temporary dynamic type sets, reused calls,
					-- etc.) may not be available anymore.
				print_separate_call_declaration (current_separate_call_count, a_call, current_closure)
			end
		end

	print_separate_qualified_call_instruction (a_call: ET_QUALIFIED_FEATURE_CALL_INSTRUCTION)
			-- Print qualified call instruction whose target type is separate.
			-- Operands can be found in `call_operands'.
		require
			a_call_not_void: a_call /= Void
			call_operands_not_empty: not call_operands.is_empty
			use_scoop: use_scoop
		local
			i, nb: INTEGER
			l_separate_arguments: like separate_arguments
			l_separate_argument: ET_IDENTIFIER
		do
			current_separate_call_count := current_separate_call_count + 1
			print_indentation
			print_separate_call_name (current_separate_call_count, current_feature, current_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			l_separate_arguments := separate_arguments
			l_separate_arguments.wipe_out
			a_call.target.add_separate_arguments (l_separate_arguments, current_closure)
			nb := l_separate_arguments.count
			from i := 1 until i > nb loop
				l_separate_argument := l_separate_arguments.item (i)
				if
					l_separate_argument.is_inline_separate_argument or
					l_separate_argument.is_argument and then dynamic_type_set (l_separate_argument).static_type.is_separate
				then
					print_comma
					print_separate_argument_session_name (l_separate_argument, current_file)
				end
				i := i + 1
			end
			l_separate_arguments.wipe_out
			nb := call_operands.count
			from i := 1 until i > nb loop
				print_comma
				print_expression (call_operands.item (i))
				i := i + 1
			end
			current_file.put_character (')')
			print_semicolon_newline
		end

	print_repeat_instruction (a_instruction: ET_REPEAT_INSTRUCTION)
			-- Print `a_instruction'.
		require
			a_instruction_not_void: a_instruction /= Void
		do
			print_iteration_instruction (a_instruction)
		end

	print_retry_instruction (an_instruction: ET_RETRY_INSTRUCTION)
			-- Print `an_instruction'.
		require
			an_instruction_not_void: an_instruction /= Void
		do
			if line_generation_mode then
				print_position (an_instruction.position, current_feature.static_feature.implementation_class)
			end
			print_indentation
			current_file.put_string (c_ac)
			current_file.put_string (c_arrow)
			current_file.put_string (c_in_rescue)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_string (c_tr)
			current_file.put_character (';')
			current_file.put_new_line
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_string (c_tr)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_character ('0')
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			current_file.put_string (c_ac)
			current_file.put_string (c_arrow)
			current_file.put_string (c_exception_code)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_character ('0')
			current_file.put_character (';')
			current_file.put_new_line
			print_indentation
			current_file.put_string (c_ac)
			current_file.put_string (c_arrow)
			current_file.put_string (c_exception_tag)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_string (c_char)
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_character ('0')
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			print_indentation
			current_file.put_string (c_goto)
			current_file.put_character (' ')
			current_file.put_string (c_ge_retry)
			current_file.put_character (';')
			current_file.put_new_line
			has_retry := True
		end

	print_static_call_instruction (a_instruction: ET_STATIC_CALL_INSTRUCTION)
			-- Print `a_instruction'.
		require
			a_instruction_not_void: a_instruction /= Void
		local
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			i, nb: INTEGER
			l_actuals: detachable ET_ACTUAL_ARGUMENTS
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_actual: ET_EXPRESSION
			l_old_in_qualified_call: BOOLEAN
		do
			if line_generation_mode then
				print_position (a_instruction.position, current_feature.static_feature.implementation_class)
			end
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := True
				-- Add dummy target (which will not be used).
			operand_stack.force (tokens.current_keyword)
			l_actuals := a_instruction.arguments
			if l_actuals /= Void then
				nb := l_actuals.count
				from i := 1 until i > nb loop
					l_actual := l_actuals.actual_argument (i)
					l_actual_type_set := dynamic_type_set (l_actual)
					print_attachment_operand (l_actual, l_actual_type_set)
					i := i + 1
				end
			end
			fill_call_operands (nb + 1)
			l_target_type := current_dynamic_system.dynamic_primary_type (a_instruction.type, current_type.base_type)
			print_named_procedure_call (a_instruction.name, l_target_type, False)
			call_operands.wipe_out
			in_qualified_call := l_old_in_qualified_call
		end

	print_unqualified_call_instruction (a_call: ET_UNQUALIFIED_FEATURE_CALL_INSTRUCTION)
			-- Print unqualified call instruction.
		require
			a_call_not_void: a_call /= Void
		local
			l_actuals: detachable ET_ACTUAL_ARGUMENTS
			l_actual: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_manifest_tuple_operand: detachable ET_MANIFEST_TUPLE
			l_target_operand: ET_EXPRESSION
			l_seed: INTEGER
			i, nb: INTEGER
			l_old_in_qualified_call: BOOLEAN
		do
			if line_generation_mode then
				print_position (a_call.position, current_feature.static_feature.implementation_class)
			end
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := False
			if not attached current_type.seeded_dynamic_procedure (a_call.name.seed, current_dynamic_system) as l_dynamic_feature then
					-- Internal error: there should be a procedure for that seed.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_unqualified_call_instruction", 1, "procedure not found.")
			else
				l_actuals := a_call.arguments
				l_seed := a_call.name.seed
					-- Check whether this a call to ROUTINE.call with a manifest
					-- tuple as argument. We have a special treatment in that case
					-- to avoid having to create the manifest tuple when possible.
				if l_seed = current_system.routine_call_seed and then not a_call.is_call_agent then
					if l_actuals /= Void and then l_actuals.count = 1 then
						if attached {ET_MANIFEST_TUPLE} l_actuals.actual_argument (1) as l_manifest_tuple then
							operand_stack.force (tokens.current_keyword)
							nb := l_manifest_tuple.count
							from i := 1 until i > nb loop
								l_actual := l_manifest_tuple.expression (i)
								l_actual_type_set := dynamic_type_set (l_actual)
								print_attachment_operand (l_actual, l_actual_type_set)
								i := i + 1
							end
							nb := nb + 1
							fill_call_operands (nb)
							l_manifest_tuple_operand := agent_manifest_tuple
							l_manifest_tuple_operand.resize (nb - 1)
							from i := nb until i < 2 loop
								l_manifest_tuple_operand.put_first (call_operands.item (i))
								i := i - 1
							end
							l_manifest_tuple_operand.set_index (l_manifest_tuple.index)
							mark_expressions_frozen (l_manifest_tuple_operand)
							l_target_operand := call_operands.first
							call_operands.wipe_out
							call_operands.put_last (l_target_operand)
							call_operands.put_last (l_manifest_tuple_operand)
							nb := 2
						end
					end
				end
				if l_manifest_tuple_operand = Void then
					operand_stack.force (tokens.current_keyword)
					if l_actuals /= Void then
						nb := l_actuals.count
						from i := 1 until i > nb loop
							l_actual := l_actuals.actual_argument (i)
							l_actual_type_set := dynamic_type_set (l_actual)
							print_attachment_operand (l_actual, l_actual_type_set)
							i := i + 1
						end
					end
					nb := nb + 1
					fill_call_operands (nb)
				end
				print_named_procedure_call (a_call.name, current_type, False)
				call_operands.wipe_out
				if l_manifest_tuple_operand /= Void then
					mark_expressions_unfrozen (l_manifest_tuple_operand)
					l_manifest_tuple_operand.wipe_out
				end
			end
			in_qualified_call := l_old_in_qualified_call
		end

feature {NONE} -- Procedure call generation

	print_adapted_creation_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type, a_static_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to creation procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_static_type' is the static type of the declared type of the target,
			-- used to adapt the result of the creation if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_creation: a_feature.is_creation
			a_target_type_not_void: a_target_type /= Void
			a_static_type_not_void: a_static_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			print_adapted_expression_with_agent (agent print_creation_procedure_call (a_feature, a_target_type), a_target_type, a_static_type, False)
		end

	print_adapted_separate_creation_procedure_call (a_separate_call: ET_SEPARATE_CALL; a_feature: ET_DYNAMIC_FEATURE; a_target_type, a_static_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to separate creation procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_static_type' is the static type of the declared type of the target,
			-- used to adapt the result of the creation if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_separate_creation: a_feature.is_separate_creation
			a_target_type_not_void: a_target_type /= Void
			a_static_type_not_void: a_static_type /= Void
			call_operands_not_empty: not call_operands.is_empty
			use_scoop: use_scoop
		do
			print_adapted_expression_with_agent (agent print_separate_creation_procedure_call (a_separate_call, a_feature, a_target_type), a_target_type, a_static_type, False)
		end

	print_creation_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to creation procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_creation: a_feature.is_creation
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			i, nb: INTEGER
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type_set: ET_DYNAMIC_TYPE_SET
		do
			register_called_feature (a_feature)
			print_creation_procedure_name (a_feature, a_target_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			nb := call_operands.count
			from i := 1 until i > nb loop
				print_comma
				l_actual_type_set := dynamic_type_set (call_operands.item (i))
				l_formal_type_set := argument_type_set_in_feature (i, a_feature)
				print_attachment_expression (call_operands.item (i), l_actual_type_set, l_formal_type_set.static_type)
				i := i + 1
			end
			current_file.put_character (')')
		end

	print_separate_creation_procedure_call (a_separate_call: ET_SEPARATE_CALL; a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to separate creation procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- Operands can be found in `call_operands'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_separate_creation: a_feature.is_separate_creation
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
			use_scoop: use_scoop
		local
			i, nb: INTEGER
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type_set: ET_DYNAMIC_TYPE_SET
		do
			current_separate_call_count := current_separate_call_count + 1
			register_called_feature (a_feature)
			print_separate_call_name (current_separate_call_count, current_feature, current_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			nb := call_operands.count
			from i := 1 until i > nb loop
				print_comma
				l_actual_type_set := dynamic_type_set (call_operands.item (i))
				l_formal_type_set := argument_type_set_in_feature (i, a_feature)
				print_attachment_expression (call_operands.item (i), l_actual_type_set, l_formal_type_set.static_type)
				i := i + 1
			end
			current_file.put_character (')')
		end

	print_named_procedure_call (a_name: ET_CALL_NAME; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_name' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_name_not_void: a_name /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_dynamic_feature: detachable ET_DYNAMIC_FEATURE
			l_seed: INTEGER
		do
			l_seed := a_name.seed
			if a_name.is_tuple_label then
				print_tuple_label_setter_call (a_name, a_target_type, a_check_void_target)
			else
				l_dynamic_feature := a_target_type.seeded_dynamic_procedure (l_seed, current_dynamic_system)
				if l_dynamic_feature = Void then
						-- Internal error: there should be a procedure with `l_seed'.
						-- It has been computed in ET_FEATURE_CHECKER.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_named_procedure_call", 1, "procedure not found.")
				else
					print_procedure_call (l_dynamic_feature, a_target_type, a_check_void_target)
				end
			end
		end

	print_tuple_label_setter_call (a_name: ET_CALL_NAME; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to a Tuple label setter (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_name_not_void: a_name /= Void
			tuple_label: a_name.is_tuple_label
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_tuple_expression: ET_EXPRESSION
			l_item_type_sets: ET_DYNAMIC_TYPE_SET_LIST
			l_item_type_set: ET_DYNAMIC_TYPE_SET
			l_source: ET_EXPRESSION
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_seed: INTEGER
		do
			if call_operands.count /= 2 then
					-- Internal error: the Tuple label setter should have one argument.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_tuple_label_setter_call", 1, "wrong number of arguments.")
			else
				if not attached {ET_DYNAMIC_TUPLE_TYPE} a_target_type as l_tuple_type then
						-- Internal error: the type of the target should be a Tuple type.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_tuple_label_setter_call", 2, "target type is not 'TUPLE'.")
				else
					l_seed := a_name.seed
					l_item_type_sets := l_tuple_type.item_type_sets
					if l_seed < 1 or l_seed > l_item_type_sets.count then
							-- Internal error: invalid Tuple label.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_tuple_label_setter_call", 3, "invalid Tuple label.")
					else
						print_indentation
						l_tuple_expression := call_operands.first
						print_attribute_tuple_item_access (l_seed, l_tuple_expression, a_target_type, a_check_void_target)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_character ('(')
						l_source := call_operands.item (2)
						l_source_type_set := dynamic_type_set (l_source)
						l_item_type_set := l_item_type_sets.item (l_seed)
						print_attachment_expression (l_source, l_source_type_set, l_item_type_set.static_type)
						current_file.put_character (')')
						current_file.put_character (';')
						current_file.put_new_line
					end
				end
			end
		end

	print_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			if a_feature.is_builtin then
				print_builtin_procedure_call (a_feature, a_target_type, a_check_void_target)
			elseif is_inlinable_procedure (a_feature, a_target_type) then
				print_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_non_inlined_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a non-inlined version of a call to procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type_set: ET_DYNAMIC_TYPE_SET
			i, nb: INTEGER
		do
			register_called_feature (a_feature)
			print_indentation
			print_before_call (a_target_type, in_qualified_call)
			print_routine_name (a_feature, a_target_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			nb := call_operands.count
			if not a_feature.is_static then
				print_comma
				print_procedure_target_expression (call_operands.first, a_target_type, a_check_void_target)
			end
			from i := 2 until i > nb loop
				print_comma
				l_actual_type_set := dynamic_type_set (call_operands.item (i))
				l_formal_type_set := argument_type_set_in_feature (i - 1, a_feature)
				print_attachment_expression (call_operands.item (i), l_actual_type_set, l_formal_type_set.static_type)
				i := i + 1
			end
			current_file.put_character (')')
			print_after_call (a_target_type)
			print_semicolon_newline
		end

	print_inlined_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' an inlined version of a call to procedure `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
			a_feature_is_inlinable: is_inlinable_procedure (a_feature, a_target_type)
		local
			l_old_call_target_type: like call_target_type
			l_operands_count: INTEGER
			l_old_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_caller_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_old_current_index: INTEGER
			l_old_index_offset: INTEGER
		do
			l_operands_count := call_operands.count
			if l_operands_count /= a_feature.static_feature.arguments_count + 1 then
					-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_procedure_call", 1, "wrong number of arguments.")
			elseif not attached {ET_INTERNAL_PROCEDURE} a_feature.static_feature as l_internal_procedure then
					-- Internal error: only internal routines are inlinable
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_procedure_call", 2, "only internal routines are inlinable.")
			elseif attached l_internal_procedure.compound as l_compound and then not l_compound.is_empty then
				register_inlined_feature (a_feature)
				nested_inlining_count := nested_inlining_count + 1
				l_old_inlining_context := inlining_context
				l_caller_inlining_context := l_old_inlining_context
				if l_caller_inlining_context = Void then
					l_caller_inlining_context := new_call_context (current_feature)
				end
				l_target_type_set := dynamic_type_set (call_operands.first)
				l_inlining_context := new_call_context (a_feature)
				fill_inlining_operands (a_target_type, l_inlining_context, l_caller_inlining_context)
				inlining_context := l_inlining_context
				l_old_call_target_type := call_target_type
				call_target_type := Void
				current_feature := a_feature
				current_type := a_feature.target_type
				current_dynamic_type_sets := a_feature.dynamic_type_sets
				l_old_current_index := current_index
				current_index := a_feature.current_index
				l_old_index_offset := index_offset
				index_offset := 0
				print_indentation
				current_file.put_string ("/* -> ")
				print_routine_name (current_feature, current_type, current_file)
				current_file.put_string (" */")
				current_file.put_new_line
				print_compound (l_compound)
				print_indentation
				current_file.put_string ("/* <- */")
				current_file.put_new_line
				free_inlining_operands (l_target_type_set, l_inlining_context, l_caller_inlining_context)
				current_feature := l_caller_inlining_context.current_feature
				current_type := current_feature.target_type
				current_dynamic_type_sets := current_feature.dynamic_type_sets
				current_index := l_old_current_index
				index_offset := l_old_index_offset
				call_target_type := l_old_call_target_type
				inlining_context := l_old_inlining_context
				nested_inlining_count := nested_inlining_count - 1
			else
				register_inlined_feature (a_feature)
				print_indentation
				current_file.put_string ("/* -> ")
				print_routine_name (a_feature, a_feature.target_type, current_file)
				current_file.put_string (" */")
				current_file.put_new_line
				print_indentation
				current_file.put_string ("/* <- */")
				current_file.put_new_line
			end
		end

	print_builtin_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_class_code
			when {ET_TOKEN_CODES}.builtin_any_class then
				print_builtin_any_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_ref_class then
				print_builtin_boolean_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_8_ref_class then
				print_builtin_character_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_32_ref_class then
				print_builtin_character_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_class then
				print_builtin_com_failure_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_identified_routines_class then
				print_builtin_identified_routines_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_8_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_16_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_32_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_64_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_exception_manager_class then
				print_builtin_ise_exception_manager_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_class then
				print_builtin_ise_runtime_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_memory_class then
				print_builtin_memory_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_8_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_16_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_32_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_64_ref_class then
				print_builtin_integer_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_ref_class then
				print_builtin_pointer_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_procedure_class then
				print_builtin_procedure_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_32_ref_class then
				print_builtin_real_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_64_ref_class then
				print_builtin_real_n_ref_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_class then
				print_builtin_special_procedure_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_tuple_class then
				print_builtin_tuple_procedure_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_any_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ANY".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_any: a_feature.is_builtin_any_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_any_copy then
				print_builtin_any_copy_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_standard_copy then
				print_builtin_any_standard_copy_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_boolean_ref_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "BOOLEAN_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean_ref: a_feature.is_builtin_boolean_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_ref_set_item then
				print_builtin_boolean_ref_set_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_character_n_ref_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "CHARACTER_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n_ref: a_feature.is_builtin_character_n_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_ref_set_item then
				print_builtin_character_n_ref_set_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_com_failure_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "COM_FAILURE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_com_failure: a_feature.is_builtin_com_failure_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_com_failure_cwin_local_free then
				print_builtin_com_failure_cwin_local_free_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_identified_routines_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "IDENTIFIED_ROUTINES".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_identified_routines: a_feature.is_builtin_identified_routines_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_object_id_free then
				print_builtin_identified_routines_eif_object_id_free_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_integer_n_ref_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "INTEGER_N_REF" and "NATURAL_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n_ref: a_feature.is_builtin_integer_n_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_ref_set_item then
				print_builtin_integer_n_ref_set_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_ise_exception_manager_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ISE_EXCEPTION_MANAGER".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_exception_manager: a_feature.is_builtin_ise_exception_manager_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_exception_manager_developer_raise then
				print_builtin_ise_exception_manager_developer_raise_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_ise_runtime_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ISE_RUNTIME".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_runtime: a_feature.is_builtin_ise_runtime_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_runtime_lock_marking then
				print_builtin_ise_runtime_lock_marking_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_mark_object then
				print_builtin_ise_runtime_mark_object_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_boolean_field then
				print_builtin_ise_runtime_set_boolean_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_boolean_field_at then
				print_builtin_ise_runtime_set_boolean_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_8_field then
				print_builtin_ise_runtime_set_character_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_8_field_at then
				print_builtin_ise_runtime_set_character_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_32_field then
				print_builtin_ise_runtime_set_character_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_character_32_field_at then
				print_builtin_ise_runtime_set_character_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_8_field then
				print_builtin_ise_runtime_set_integer_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_8_field_at then
				print_builtin_ise_runtime_set_integer_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_16_field then
				print_builtin_ise_runtime_set_integer_16_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_16_field_at then
				print_builtin_ise_runtime_set_integer_16_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_32_field then
				print_builtin_ise_runtime_set_integer_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_32_field_at then
				print_builtin_ise_runtime_set_integer_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_64_field then
				print_builtin_ise_runtime_set_integer_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_integer_64_field_at then
				print_builtin_ise_runtime_set_integer_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_8_field then
				print_builtin_ise_runtime_set_natural_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_8_field_at then
				print_builtin_ise_runtime_set_natural_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_16_field then
				print_builtin_ise_runtime_set_natural_16_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_16_field_at then
				print_builtin_ise_runtime_set_natural_16_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_32_field then
				print_builtin_ise_runtime_set_natural_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_32_field_at then
				print_builtin_ise_runtime_set_natural_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_64_field then
				print_builtin_ise_runtime_set_natural_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_natural_64_field_at then
				print_builtin_ise_runtime_set_natural_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_pointer_field then
				print_builtin_ise_runtime_set_pointer_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_pointer_field_at then
				print_builtin_ise_runtime_set_pointer_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_32_field then
				print_builtin_ise_runtime_set_real_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_32_field_at then
				print_builtin_ise_runtime_set_real_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_64_field then
				print_builtin_ise_runtime_set_real_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_real_64_field_at then
				print_builtin_ise_runtime_set_real_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_reference_field then
				print_builtin_ise_runtime_set_reference_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_set_reference_field_at then
				print_builtin_ise_runtime_set_reference_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_unlock_marking then
				print_builtin_ise_runtime_unlock_marking_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_unmark_object then
				print_builtin_ise_runtime_unmark_object_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_memory_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "MEMORY".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_memory: a_feature.is_builtin_memory_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_memory_free then
				print_builtin_memory_free_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_pointer_ref_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "POINTER_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer_ref: a_feature.is_builtin_pointer_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_ref_set_item then
				print_builtin_pointer_ref_set_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_procedure_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "PROCEDURE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_procedure: a_feature.is_builtin_procedure_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_procedure_call then
				print_builtin_procedure_call_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_procedure_fast_call then
				print_builtin_procedure_fast_call_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_real_n_ref_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "REAL_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n_ref: a_feature.is_builtin_real_n_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ref_set_item then
				print_builtin_real_n_ref_set_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_special_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "SPECIAL".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_special: a_feature.is_builtin_special_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_special_put then
				print_builtin_special_put_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_tuple_procedure_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to procedure `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "TUPLE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_tuple: a_feature.is_builtin_tuple_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_tuple_set_object_comparison then
				print_builtin_tuple_set_object_comparison_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_procedure_call (a_feature, a_target_type, a_check_void_target)
			end
		end

feature {NONE} -- Expression generation

	print_across_expression (an_expression: ET_ACROSS_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_iteration_expression (an_expression)
		end

	print_adapted_attribute_access (an_attribute: ET_DYNAMIC_FEATURE; a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to `an_attribute' applied to `a_target' of dynamic type `a_target_type'.
			-- Useful for example when the access to `a_target' needs to be adapted depending on
			-- the expandedness of the dynamic and static types of `a_target'. For example if the
			-- static type is a reference type and the dynamic type is expanded, then `a_target' will
			-- need to be unboxed before accessing its attribute.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_attribute_not_void: an_attribute /= Void
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if dynamic_type_set (a_target).is_expanded then
				print_attribute_access (an_attribute, a_target, a_target_type, a_check_void_target)
			else
				print_boxed_attribute_access (an_attribute, a_target, a_target_type, a_check_void_target)
			end
		end

	print_adapted_expression_with_agent (a_print_expression: PROCEDURE;
		a_source_type_set: ET_DYNAMIC_TYPE_SET; a_target_type: ET_DYNAMIC_PRIMARY_TYPE;
		a_use_boxed_pointer: BOOLEAN)
			-- Print to `current_file' an expression (using `a_print_expression') with
			-- dynamic type set `a_source_type_set' when the static type expected
			-- by the caller is `a_target_type'. This is useful for example when the
			-- expression is a call to a query and this call is one of the alternatives
			-- of a polymorphic call whose static type expected by the caller is
			-- `a_target_type'. In that case the result of the query needs to be
			-- adapted to match the kind of result type expected by the caller.
			-- For example, if we have:
			--
			--    l: LIST [ANY]
			--    li: LIST [INTEGER]
			--    ls: LIST [STRING]
			--    ...
			--    a: ANY
			--    if b then
			--       l := li
			--    else
			--       l := ls
			--    end
			--    a := l.item
			--
			-- depending on the value of 'b', the call to 'item' will yield an integer
			-- or a string. If it's an integer (this corresponds to the static type
			-- of `a_source_type_set'), then its value will need to be boxed because
			-- the signature of 'item' in 'LIST [ANY]' is of reference type (this
			-- corresponds to `a_target_type').
			--
			-- `a_use_boxed_pointer' indicates whether we should keep the address of the
			-- expression, and not just its value, if boxing is needed.
			-- Useful when the expression is an expanded attribute, tuple label or
			-- "SPECIAL" item to be used as the target of another call.
			--
			-- Note that contrary to `print_attachment_expression', the result of the
			-- query should not be copied or cloned. It will occur later on when the call
			-- is the source of an assignment or passed as actual argument (in the example
			-- above it will occur when the value of the call to 'l.item' is assigned
			-- to 'a').
			--
			-- And also, contrary to `print_attachment_expression', the static type of
			-- the result of the query is expected to conform to the type of the underlying
			-- call. Unless the query has a result type of the form 'like argument'.
			-- For example 'clone (1)' is expected to be of type "INTEGER" and
			-- 'clone ("gobo")' is expected to be of type "STRING". But the corresponding
			-- generated function for 'clone' will be declared to return an "ANY".
		require
			a_print_expression_not_void: a_print_expression /= Void
			a_source_type_set_not_void: a_source_type_set /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			l_source_type: ET_DYNAMIC_PRIMARY_TYPE
			l_do_check_void: BOOLEAN
		do
			l_source_type := a_source_type_set.static_type.primary_type
			if a_target_type.is_expanded then
					-- Only expanded types conform to expanded types.
					-- So no unboxing is needed, and copy or clone will
					-- not be called at this stage.
					--
					-- However we may need to unbox when the query has
					-- a result type of the form 'like argument' (see
					-- header comment).
				if not l_source_type.is_expanded then
						-- The source object has been boxed.
					current_file.put_character ('(')
					print_boxed_type_cast (a_target_type, current_file)
					current_file.put_character ('(')
					if a_source_type_set.can_be_void  then
						can_be_void_target_count := can_be_void_target_count + 1
						if check_for_void_target_mode then
							if finalize_mode then
								current_file.put_string (c_ge_void)
							else
								current_file.put_string (c_ge_void2)
							end
							current_file.put_character ('(')
							l_do_check_void := True
						end
					else
						never_void_target_count := never_void_target_count + 1
					end
					a_print_expression.call ([])
					if l_do_check_void then
						if not finalize_mode then
							print_comma
							current_file.put_integer (can_be_void_target_count)
						end
						current_file.put_character (')')
					end
					current_file.put_character (')')
					current_file.put_character (')')
						-- The source object has been boxed.
					current_file.put_string (c_arrow)
					print_boxed_attribute_item_name (a_target_type, current_file)
				else
					a_print_expression.call ([])
				end
			else
				if l_source_type.is_expanded then
						-- We need to box the object, but without triggering a call to
						-- 'copy' (it will be called during subsequent attachment).
					if a_use_boxed_pointer then
						current_file.put_string (c_ge_boxed_pointer)
						current_file.put_integer (l_source_type.id)
						current_file.put_character ('(')
						current_file.put_string (c_ac)
						print_comma
						current_file.put_character ('&')
						current_file.put_character ('(')
						a_print_expression.call ([])
						current_file.put_character (')')
						current_file.put_character (')')
					else
						current_file.put_string (c_ge_boxed)
						current_file.put_integer (l_source_type.id)
						current_file.put_character ('(')
						current_file.put_string (c_ac)
						print_comma
						a_print_expression.call ([])
						current_file.put_character (')')
					end
				else
						-- Even if the dynamic type set of the expression contains
						-- expanded types, there will be no copy or clone
						-- at this stage.
					a_print_expression.call ([])
				end
			end
		end

	print_assignment_operand (a_operand: ET_EXPRESSION; a_source_type_set: ET_DYNAMIC_TYPE_SET; a_target: ET_WRITABLE; a_target_type: ET_DYNAMIC_TYPE)
			-- Print `a_operand' of dynamic type set `a_source_type_set' when it
			-- is the source of an assignment to `a_target' of type `a_target_type'.
			-- Make sure that the object is cloned when its type is expanded.
			-- Note that the object will be boxed or unboxed when needed
			-- later, in the call to `print_attachment_expression'.
		require
			a_operand_not_void: a_operand /= Void
			a_source_type_set_not_void: a_source_type_set /= Void
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if (a_source_type_set.is_expanded = a_target_type.is_expanded) and then not is_twin_needed_in_attachment (a_source_type_set) then
				assignment_target := a_target
			else
				assignment_target := Void
			end
			print_attachment_operand (a_operand, a_source_type_set)
			assignment_target := Void
		end

	print_attachment_expression (an_expression: ET_EXPRESSION; a_source_type_set: ET_DYNAMIC_TYPE_SET; a_target_type: ET_DYNAMIC_TYPE)
			-- Print `an_expression' of dynamic type set `a_source_type_set'
			-- when it is to be attached to an entity of type `a_target_type'.
			-- Make sure that the expression is boxed or unboxed when needed.
			-- Useful when printing the source of an assignment or actual
			-- arguments of a routine.
			-- Note that the cloning of then object when its type is expanded
			-- is done earlier, in the call to `print_attachment_operand'.
			--
			-- Note that the static type of the source does not necessarily
			-- conform to the target type. This can be the case for example
			-- with assignment attempts or because of covariance of arguments
			-- as in the example below:
			--
			--    c1: COMPARABLE
			--    c2: COMPARABLE
			--    ...
			--    c1 := 1
			--    c2 := 2
			--    c1 < c2
			--
			-- At run-time, 'INTEGER.is_less alias "<"' will be called. 'c2' is of static
			-- type COMPARABLE, which does not conform to the expected argument
			-- type INTEGER. But according to the dynamic type set mechanism there
			-- is no CAT-call in that case. So we will just need to unbox the
			-- integer 2 (which had been previously boxed in the assignment to
			-- 'c2') before passing it as argument of 'INTEGER.is_less alias "<"'.
		require
			an_expression_not_void: an_expression /= Void
			a_source_type_set_not_void: a_source_type_set /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			l_source_type: ET_DYNAMIC_TYPE
			l_target_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_source_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_non_conforming_types: DS_ARRAYED_LIST [INTEGER]
			l_has_non_conforming_types: BOOLEAN
			l_dts_ids: STRING
			l_dts_name: detachable STRING
			i, nb: INTEGER
			l_need_unboxing: BOOLEAN
			l_target_type_is_attached: BOOLEAN
		do
			l_source_type := a_source_type_set.static_type
			l_source_primary_type := l_source_type.primary_type
			l_target_primary_type := a_target_type.primary_type
			if not l_source_primary_type.conforms_to_primary_type (l_target_primary_type) then
					-- Make sure that CAT-call errors will be reported at run-time.
				nb := a_source_type_set.count
				l_non_conforming_types := attachment_dynamic_type_ids
				if nb > l_non_conforming_types.capacity then
					l_non_conforming_types.resize (nb)
				end
				from i := 1 until i > nb loop
					l_dynamic_type := a_source_type_set.dynamic_type (i)
					if not l_dynamic_type.conforms_to_primary_type (l_target_primary_type) then
						l_non_conforming_types.put_last (l_dynamic_type.id)
					end
					i := i + 1
				end
				if not l_non_conforming_types.is_empty then
					l_has_non_conforming_types := True
					l_non_conforming_types.sort (dynamic_type_id_sorter)
					nb := l_non_conforming_types.count
					create l_dts_ids.make (5 * nb)
					from i := 1 until i > nb loop
						if i /= 1 then
							l_dts_ids.append_character (',')
						end
						l_dts_ids.append_string (l_non_conforming_types.item (i).out)
						i := i + 1
					end
					dynamic_type_id_set_names.search (l_dts_ids)
					if dynamic_type_id_set_names.found then
						l_dts_name := dynamic_type_id_set_names.found_item
					else
						l_dts_name := c_ge_dts + (dynamic_type_id_set_names.count + 1).out
						dynamic_type_id_set_names.force_last_new (l_dts_name, l_dts_ids)
					end
				end
				l_non_conforming_types.wipe_out
			end
			l_need_unboxing := l_target_primary_type.is_expanded and then (not l_source_primary_type.is_expanded or l_has_non_conforming_types)
			if l_need_unboxing then
				current_file.put_character ('(')
				print_boxed_type_cast (l_target_primary_type, current_file)
				current_file.put_character ('(')
			end
			if l_has_non_conforming_types then
				current_file.put_string (c_ge_catcall)
				current_file.put_character ('(')
				current_file.put_character ('(')
				print_eif_any_type_name (current_file)
				current_file.put_character ('*')
				current_file.put_character (')')
				current_file.put_character ('(')
			end
			if l_target_primary_type.is_expanded then
				if l_source_primary_type.is_expanded then
					if l_has_non_conforming_types then
							-- We need to box the source object in order
							-- to pass it to 'GE_catcall'.
						print_boxed_expression (an_expression, l_source_primary_type)
					else
						print_expression (an_expression)
					end
				else
						-- The source object has been boxed.
					print_expression (an_expression)
				end
			else
				if l_source_primary_type.is_expanded then
						-- We need to box the source object.
					print_boxed_expression (an_expression, l_source_primary_type)
				else
						-- Deal with void-safety: when `a_target_type' is attached,
						-- then we need to check that we don't pass Void.
					l_target_type_is_attached := a_target_type.is_attached
-- TODO: Improve dynamic type set 'is_never_void' in void-safe mode before
-- using this code. Otherwise we end up with too many unnecessary checks for void.
--					print_non_void_expression (an_expression, l_target_type_is_attached)
					print_expression (an_expression)
				end
			end
			if l_has_non_conforming_types and l_dts_name /= Void then
				current_file.put_character (')')
				current_file.put_character (',')
				current_file.put_string (l_dts_name)
				current_file.put_character (',')
				current_file.put_integer (nb)
				current_file.put_character (')')
			end
			if l_need_unboxing then
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
				print_boxed_attribute_item_name (l_source_primary_type, current_file)
			end
		end

	print_attachment_operand (a_operand: ET_EXPRESSION; a_source_type_set: ET_DYNAMIC_TYPE_SET)
			-- Print `a_operand' of dynamic type set `a_source_type_set'
			-- when it is to be attached to an entity.
			-- Make sure that the object is cloned when its type is expanded.
			-- Useful when printing the source of an assignment or actual arguments
			-- of a routine.
			-- Note that the object will be boxed or unboxed when needed
			-- later, in the call to `print_attachment_expression'.
		require
			a_operand_not_void: a_operand /= Void
			a_source_type_set_not_void: a_source_type_set /= Void
		local
			l_source_type: ET_DYNAMIC_TYPE
			l_source_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_twin_feature_name: ET_IDENTIFIER
			l_twin_expression: ET_QUALIFIED_CALL_EXPRESSION
			l_twin_operand: ET_EXPRESSION
			l_inlined_twin_operand: ET_DYNAMIC_INLINED_EXPRESSION
			l_inlined_call_operand: ET_DYNAMIC_INLINED_EXPRESSION
			l_call_context: ET_DYNAMIC_CALL_CONTEXT
			l_twin_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
			l_twin_static_type: ET_DYNAMIC_TYPE
			l_query_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_checked_void: BOOLEAN
			l_call_operand: ET_EXPRESSION
			l_temp: detachable ET_IDENTIFIER
			l_assignment_target: like assignment_target
		do
			l_source_type := a_source_type_set.static_type
			l_source_primary_type := l_source_type.primary_type
			if l_source_primary_type.is_expanded then
				if l_source_primary_type.is_basic and then not l_source_primary_type.has_redefined_copy_routine then
						-- Optimization: no need to explicitly call 'twin' in that case.
					print_operand (a_operand)
				else
					l_twin_feature_name := new_twin_feature_name
					l_twin_expression := new_qualified_call_expression (a_operand, l_twin_feature_name, Void)
					if attached {ET_DYNAMIC_INLINED_EXPRESSION} a_operand as l_inlined_operand then
						l_call_context := inlining_context
						if l_call_context = Void then
							l_call_context := new_call_context (current_feature)
						end
						l_inlined_twin_operand := new_inlined_operand (a_operand, l_call_context)
						l_inlined_twin_operand.set_expression (l_twin_expression)
						l_inlined_twin_operand.set_context (l_call_context)
						l_twin_operand := l_inlined_twin_operand
					else
						l_twin_expression.set_index (a_operand.index)
						l_twin_operand := l_twin_expression
					end
					print_operand (l_twin_operand)
					free_qualified_call_expression (l_twin_expression)
					free_twin_feature_name (l_twin_feature_name)
				end
			else
					-- Check to see whether some of the types in the source type set are expanded.
					-- If so, 'twin' needs to be called on them.
					--
					-- Note `l_twin_type_set' will only be made up of expanded types.
					-- So make sure that its static type is not marked as separate.
					-- This is needed when processing separate calls because we don't
					-- want this call to 'twin' to be considered as separate.
				if attached l_source_primary_type.attached_type as l_source_primary_attached_type then
					l_twin_static_type := l_source_primary_attached_type
				else
					l_twin_static_type := l_source_primary_type
				end
				l_twin_type_set := new_standalone_type_set (l_twin_static_type)
				l_twin_type_set.put_expanded_types (a_source_type_set)
				l_twin_type_set.set_never_void
				if l_twin_type_set.is_empty then
					print_operand (a_operand)
					free_standalone_type_set (l_twin_type_set)
				else
					l_assignment_target := assignment_target
					assignment_target := Void
					print_operand (a_operand)
					fill_call_operands (1)
					l_call_operand := call_operands.first
					call_operands.wipe_out
					l_call_context := inlining_context
					if l_call_context = Void then
						l_call_context := new_call_context (current_feature)
					end
					print_indentation
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_source_primary_type)
						mark_temp_variable_frozen (l_temp)
						l_inlined_call_operand := new_inlined_operand (l_call_operand, l_call_context)
						l_inlined_call_operand.set_expression (l_temp)
						l_inlined_call_operand.set_context (l_call_context)
						operand_stack.force (l_inlined_call_operand)
						print_temp_name (l_temp, current_file)
					end
					print_assign_to
					current_file.put_character ('(')
					if a_source_type_set.can_be_void and then not l_call_operand.is_never_void then
						l_checked_void := True
						current_file.put_character ('(')
						print_expression (l_call_operand)
						print_and_then
					end
					current_file.put_character ('(')
					current_file.put_string (c_ge_type_infos)
					current_file.put_character ('[')
					print_boxed_attribute_type_id_access (l_call_operand, l_source_primary_type, False)
					current_file.put_character (']')
					current_file.put_character ('.')
					current_file.put_string (c_flags)
					current_file.put_character ('&')
					current_file.put_string (c_ge_type_flag_expanded)
					current_file.put_character (')')
					if l_checked_void then
						current_file.put_character (')')
					end
					current_file.put_character ('?')
					l_inlined_call_operand := new_inlined_operand (l_call_operand, l_call_context)
					l_inlined_call_operand.set_dynamic_type_set (l_twin_type_set)
					l_twin_feature_name := new_twin_feature_name
					l_twin_expression := new_qualified_call_expression (l_inlined_call_operand, l_twin_feature_name, Void)
					l_inlined_twin_operand := new_inlined_operand (l_inlined_call_operand, l_call_context)
					l_inlined_twin_operand.set_expression (l_twin_expression)
					l_inlined_twin_operand.set_context (l_call_context)
						-- Register the call to 'twin' so that it is handled correctly
						-- in case of polymorphism.
						-- The special treatment for polymorphism only occurs
						-- when the target has more than 2 possible dynamic types.
					if l_twin_type_set.count > 2 then
						create l_query_call.make (l_twin_expression, l_twin_type_set, l_twin_type_set, current_feature, current_feature.static_feature, current_type, current_index, index_offset)
						l_source_primary_type.put_query_call (l_query_call)
					end
					print_expression (l_inlined_twin_operand)
					current_file.put_character (':')
					print_expression (l_call_operand)
					current_file.put_character (')')
					print_semicolon_newline
					if l_temp /= Void then
						mark_temp_variable_unfrozen (l_temp)
					end
						-- Clean up.
					if l_query_call = Void then
						free_qualified_call_expression (l_twin_expression)
						free_twin_feature_name (l_twin_feature_name)
						free_standalone_type_set (l_twin_type_set)
					end
				end
			end
		end

	print_attribute_access (an_attribute: ET_DYNAMIC_FEATURE; a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to `an_attribute' applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_attribute_not_void: an_attribute /= Void
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_name (an_attribute, a_target_type, current_file)
		end

	print_attribute_special_capacity_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'capacity' pseudo attribute of class SPECIAL applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_special_capacity_name (a_target_type, current_file)
		end

	print_attribute_special_count_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'count' pseudo attribute of class SPECIAL applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_special_count_name (a_target_type, current_file)
		end

	print_attribute_special_item_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'item' pseudo attribute of class SPECIAL applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_special_item_name (a_target_type, current_file)
		end

	print_attribute_special_indexed_item_access (an_index: ET_EXPRESSION; a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' access to item at indexed `an_index'
			-- in the 'item' pseudo attribute of class SPECIAL applied to
			-- object `a_target' of type `a_target_type'. `an_index' must
			-- be declared of (i.e. its static type must be) one of the
			-- possibly sized integer types.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_index_not_void: an_index /= Void
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			print_attribute_special_item_access (a_target, a_target_type, a_check_void_target)
			current_file.put_character ('[')
			print_expression (an_index)
			current_file.put_character (']')
		end

	print_attribute_tuple_item_access (i: INTEGER; a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to `i'-th 'item' pseudo attribute of class TUPLE
			-- applied to Tuple object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_tuple_item_name (i, a_target_type, current_file)
		end

	print_attribute_type_id_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'type_id' pseudo attribute applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				current_file.put_character ('(')
				print_type_declaration (a_target_type, current_file)
				current_file.put_character (')')
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_type_id_name (a_target_type, current_file)
		end

	print_attribute_flags_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'flags' pseudo attribute applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				current_file.put_character ('(')
				print_type_declaration (a_target_type, current_file)
				current_file.put_character (')')
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_flags_name (a_target_type, current_file)
		end

	print_attribute_region_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to SCOOP 'region' pseudo attribute applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
			use_scoop: use_scoop
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				current_file.put_character ('(')
				print_type_declaration (a_target_type, current_file)
				current_file.put_character (')')
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_region_name (a_target_type, current_file)
		end

	print_attribute_onces_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'onces' pseudo attribute applied to object `a_target' of type `a_target_type'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_expression (a_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_onces_name (a_target_type, current_file)
		end

	print_binary_integer_constant (a_constant: ET_BINARY_INTEGER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		do
			print_integer_constant (a_constant)
		end

	print_boxed_attribute_access (an_attribute: ET_DYNAMIC_FEATURE; a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to `an_attribute' applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- There is no assumption about `a_target_type': it may be reference or expanded.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_attribute_not_void: an_attribute /= Void
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			if a_target_type.is_expanded then
				current_file.put_character ('(')
				print_boxed_attribute_item_access (a_target, a_target_type, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character ('.')
			else
				current_file.put_character ('(')
				print_type_cast (a_target_type, current_file)
				current_file.put_character ('(')
				print_non_void_expression (a_target, a_check_void_target)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_string (c_arrow)
			end
			print_attribute_name (an_attribute, a_target_type, current_file)
		end

	print_boxed_attribute_flags_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'flags' pseudo attribute applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- There is no assumption about `a_target_type': it may be reference or expanded.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			current_file.put_character ('(')
			current_file.put_character ('(')
			print_boxed_type_declaration (a_target_type, current_file)
			current_file.put_character (')')
			current_file.put_character ('(')
			print_non_void_expression (a_target, a_check_void_target)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string (c_arrow)
			print_attribute_flags_name (a_target_type, current_file)
		end

	print_boxed_attribute_region_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to SCOOP 'region' pseudo attribute applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- There is no assumption about `a_target_type': it may be reference or expanded.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
			use_scoop: use_scoop
		do
			current_file.put_character ('(')
			current_file.put_character ('(')
			print_boxed_type_declaration (a_target_type, current_file)
			current_file.put_character (')')
			current_file.put_character ('(')
			print_non_void_expression (a_target, a_check_void_target)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string (c_arrow)
			print_attribute_region_name (a_target_type, current_file)
		end

	print_boxed_attribute_item_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'item' pseudo attribute applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
			a_target_type_expanded: a_target_type.is_expanded
		do
			current_file.put_character ('(')
			print_boxed_type_cast (a_target_type, current_file)
			current_file.put_character ('(')
			print_non_void_expression (a_target, a_check_void_target)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string (c_arrow)
			print_boxed_attribute_item_name (a_target_type, current_file)
		end

	print_boxed_attribute_pointer_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'pointer' pseudo attribute applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
			a_target_type_expanded: a_target_type.is_expanded
		do
			current_file.put_character ('(')
			print_boxed_type_cast (a_target_type, current_file)
			current_file.put_character ('(')
			print_non_void_expression (a_target, a_check_void_target)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string (c_arrow)
			print_boxed_attribute_pointer_name (a_target_type, current_file)
		end

	print_boxed_attribute_type_id_access (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print access to 'type_id' pseudo attribute applied to `a_target' of type the boxed version of `a_target_type'.
			-- The static type (i.e. declared type) of `a_target' is assumed to be of reference type.
			-- There is no assumption about `a_target_type': it may be reference or expanded.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			current_file.put_character ('(')
			current_file.put_character ('(')
			print_boxed_type_declaration (a_target_type, current_file)
			current_file.put_character (')')
			current_file.put_character ('(')
			print_non_void_expression (a_target, a_check_void_target)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_string (c_arrow)
			print_attribute_type_id_name (a_target_type, current_file)
		end

	print_boxed_expression (an_expression: ET_EXPRESSION; a_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print boxed version of `an_expression' of type `a_type'.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
		require
			an_expression_not_void: an_expression /= Void
			a_type_not_void: a_type /= Void
			a_type_expanded: a_type.is_expanded
		do
			current_file.put_string (c_ge_boxed)
			current_file.put_integer (a_type.id)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			print_comma
			print_expression (an_expression)
			current_file.put_character (')')
		end

	print_bracket_expression (an_expression: ET_BRACKET_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_qualified_call_expression (an_expression)
		end

	print_character_constant (a_constant: ET_CHARACTER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			l_temp: ET_IDENTIFIER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				if in_call_target then
					in_operand := False
					l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
					l_temp := new_temp_variable (l_dynamic_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_character_constant (a_constant)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
					in_operand := True
				else
					operand_stack.force (a_constant)
				end
			else
				l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
				print_type_cast (l_dynamic_type, current_file)
				current_file.put_character ('(')
				if current_system.character_32_type.same_named_type (l_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
					current_file.put_string (c_ge_nat32)
					current_file.put_character ('(')
					current_file.put_natural_32 (a_constant.value.natural_32_code)
					current_file.put_character (')')
				else
					print_escaped_character_8 (a_constant.value.to_character_8)
				end
				current_file.put_character (')')
			end
		end

	print_check_void_expression (a_expression: ET_EXPRESSION; a_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void: BOOLEAN)
			-- Print `a_expression'.
			-- `a_dynamic_type' is one of the possible dynamic types of `a_expression'.
			-- `a_check_void' means that we need to check whether the expression is Void or not.
		require
			a_expression_not_void: a_expression /= Void
			a_dynamic_type_not_void: a_dynamic_type /= Void
		local
			l_do_check_void: BOOLEAN
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
		do
			if a_check_void then
				l_dynamic_type_set := dynamic_type_set (a_expression)
				if not a_dynamic_type.is_expanded and then l_dynamic_type_set.can_be_void and not a_expression.is_never_void then
					can_be_void_target_count := can_be_void_target_count + 1
					if check_for_void_target_mode then
						if finalize_mode then
							current_file.put_string (c_ge_void)
						else
							current_file.put_string (c_ge_void2)
						end
						current_file.put_character ('(')
						l_do_check_void := True
					end
				else
					never_void_target_count := never_void_target_count + 1
				end
			end
			a_expression.process (Current)
			if l_do_check_void then
				if not finalize_mode then
					print_comma
					current_file.put_integer (can_be_void_target_count)
				end
				current_file.put_character (')')
			end
		end

	print_convert_builtin_expression (an_expression: ET_CONVERT_BUILTIN_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_convert_feature: ET_CONVERT_FEATURE
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_source_type: ET_DYNAMIC_PRIMARY_TYPE
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
		do
			l_convert_feature := an_expression.convert_feature
			l_target_type_set := dynamic_type_set (an_expression)
			l_target_type := l_target_type_set.static_type.primary_type
			l_source_type_set := dynamic_type_set (an_expression.expression)
			l_source_type := l_source_type_set.static_type.primary_type
			if l_source_type.conforms_to_primary_type (l_target_type) then
-- TODO: built-in feature with formal generic parameter? Should not be needed with ECMA Eiffel.
				an_expression.expression.process (Current)
			else
-- TODO: built-in feature between basic types? Should not be needed with ECMA Eiffel.
				l_assignment_target := assignment_target
				assignment_target := Void
				print_operand (an_expression.expression)
				fill_call_operands (1)
				if in_operand then
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_indentation
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_target_type)
						mark_temp_variable_frozen (l_temp)
							-- We will set the index of `l_temp' later because
							-- it could still be used in `call_operands'.
						l_temp_index := an_expression.index
						operand_stack.force (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
					end
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('(')
				end
				print_type_cast (l_target_type, current_file)
				current_file.put_character ('(')
				print_expression (call_operands.item (1))
				current_file.put_character (')')
				if in_operand then
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				end
				call_operands.wipe_out
				if l_temp /= Void then
					mark_temp_variable_unfrozen (l_temp)
					if l_temp_index /= 0 then
							-- We had to wait until this stage to set the index of `l_temp'
							-- because it could have still been used in `call_operands'.
						l_temp.set_index (l_temp_index)
					end
				end
			end
		end

	print_convert_from_expression (an_expression: ET_CONVERT_FROM_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_creation_expression (an_expression)
		end

	print_convert_to_expression (an_expression: ET_CONVERT_TO_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_qualified_call_expression (an_expression)
		end

	print_create_expression (an_expression: ET_CREATE_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_creation_expression (an_expression)
		end

	print_creation_expression (an_expression: ET_CREATION_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_seed: INTEGER
			i, nb: INTEGER
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
			l_actual: ET_EXPRESSION
			l_is_separate_call: BOOLEAN
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_target_type_set := dynamic_type_set (an_expression)
			l_target_type := l_target_type_set.static_type
			l_target_primary_type := l_target_type.primary_type
			if attached an_expression.arguments as l_actuals then
				nb := l_actuals.count
				from i := 1 until i > nb loop
					l_actual := l_actuals.actual_argument (i)
					print_attachment_operand (l_actual, dynamic_type_set (l_actual))
					i := i + 1
				end
			end
			fill_call_operands (nb)
			if in_operand then
				print_indentation
				if l_assignment_target /= Void then
					operand_stack.force (l_assignment_target)
					print_writable (l_assignment_target)
				else
					l_temp := new_temp_variable (l_target_primary_type)
					mark_temp_variable_frozen (l_temp)
						-- We will set the index of `l_temp' later because
						-- it could still be used in `call_operands'.
					l_temp_index := an_expression.index
					operand_stack.force (l_temp)
					print_temp_name (l_temp, current_file)
				end
				print_assign_to
				current_file.put_character ('(')
			end
			if attached an_expression.creation_call as l_creation_call then
				l_seed := l_creation_call.name.seed
			else
				l_seed := current_system.default_create_seed
			end
			if not attached l_target_primary_type.seeded_dynamic_procedure (l_seed, current_dynamic_system) as l_procedure then
					-- Internal error: there should be a procedure with `l_seed'.
					-- It has been computed in ET_FEATURE_CHECKER or else an
					-- error should have already been reported.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_creation_expression", 1, "creation procedure not found.")
			elseif use_scoop and l_target_type.is_separate then
				print_separate_creation_procedure_call (an_expression, l_procedure, l_target_primary_type)
				l_is_separate_call := True
			else
				print_creation_procedure_call (l_procedure, l_target_primary_type)
			end
			if in_operand then
				current_file.put_character (')')
				print_semicolon_newline
			end
			call_operands.wipe_out
			if l_temp /= Void then
				mark_temp_variable_unfrozen (l_temp)
				if l_temp_index /= 0 then
						-- We had to wait until this stage to set the index of `l_temp'
						-- because it could have still been used in `call_operands'.
					l_temp.set_index (l_temp_index)
				end
			end
			if l_is_separate_call then
					-- Print the separate call declaration now, otherwise some
					-- temporary data (temporary dynamic type sets, reused calls,
					-- etc.) may not be available anymore.
				print_separate_call_declaration (current_separate_call_count, an_expression, current_closure)
			end
		end

	print_current (an_expression: ET_CURRENT)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			if
				attached inlining_context as l_inlining_context and then
				attached {ET_DYNAMIC_INLINED_EXPRESSION} l_inlining_context.target as l_inlined_target
			then
				process_inlined_expression (l_inlined_target)
			elseif current_agent /= Void then
				agent_target.process (Current)
			elseif in_operand then
				operand_stack.force (an_expression)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				if current_type.is_expanded then
					print_current_name (current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (an_expression, l_call_target_type, False)
				else
					print_current_name (current_file)
				end
			elseif not current_type.is_expanded then
				print_current_name (current_file)
			else
				current_file.put_character ('*')
				print_current_name (current_file)
			end
		end

	print_current_address (an_expression: ET_CURRENT_ADDRESS)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_queries: ET_DYNAMIC_FEATURE_LIST
			l_temp: ET_IDENTIFIER
			l_pointer_type: ET_DYNAMIC_PRIMARY_TYPE
			l_pointer: BOOLEAN
		do
			l_pointer_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.pointer_type, current_type.base_type)
			l_dynamic_type_set := dynamic_type_set (an_expression)
			l_dynamic_type := l_dynamic_type_set.static_type.primary_type
			l_pointer := (l_dynamic_type = l_pointer_type)
			if not l_pointer then
					-- $Current is of type TYPED_POINTER.
				l_temp := new_temp_variable (l_dynamic_type)
				l_temp.set_index (an_expression.index)
				operand_stack.force (l_temp)
				l_queries := l_dynamic_type.queries
				if l_queries.is_empty then
						-- Internal error: TYPED_POINTER should have an attribute
						-- `pointer_item' at first position.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_current_address", 1, "'TYPED_POINTER' has no attribute `pointer_item'.")
				else
					print_indentation
					print_attribute_access (l_queries.first, l_temp, l_dynamic_type, False)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
				end
			end
			if l_pointer and in_operand then
					-- $Current is of type POINTER.
				operand_stack.force (an_expression)
			else
				print_type_cast (l_pointer_type, current_file)
				current_file.put_character ('(')
				if current_type.is_expanded then
					current_file.put_character ('&')
					print_expression (an_expression.current_keyword)
				elseif attached {ET_DYNAMIC_SPECIAL_TYPE} current_type as l_special_type then
					print_attribute_special_item_access (an_expression.current_keyword, l_special_type, False)
				else
					print_expression (an_expression.current_keyword)
				end
				current_file.put_character (')')
			end
			if not l_pointer then
				current_file.put_character (';')
				current_file.put_new_line
			end
		end

	print_equality_expression (a_expression: ET_EQUALITY_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_equality_types: ET_DYNAMIC_EQUALITY_TYPES
			l_assignment_target: like assignment_target
		do
			l_left_operand := a_expression.left
			l_right_operand := a_expression.right
			l_left_type_set := dynamic_type_set (l_left_operand)
			l_right_type_set := dynamic_type_set (l_right_operand)
			l_assignment_target := assignment_target
			assignment_target := Void
				-- Process operands.
			print_operand (l_left_operand)
				-- Call `print_attachment_operand' and not `print_operand'
				-- because the right operand might be the argument of a call
				-- to 'is_equal' and it will require some cloning if its
				-- type is expanded.
			print_attachment_operand (l_right_operand, l_right_type_set)
			fill_call_operands (2)
			assignment_target := l_assignment_target
			l_equality_types := new_equality_types (l_left_type_set, l_right_type_set)
			print_equality_call (a_expression, a_expression.operator.is_not_equal, l_equality_types)
			if current_equalities.is_empty or else current_equalities.last /= l_equality_types then
				free_equality_types (l_equality_types)
			end
			call_operands.wipe_out
		end

	print_expression (an_expression: ET_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := False
			in_procedure_call_target := False
			call_target_type := Void
			an_expression.process (Current)
			call_target_type := old_target_type
			in_procedure_call_target := l_old_in_procedure_call_target
			in_operand := old_in_operand
		end

	print_expression_address (an_expression: ET_EXPRESSION_ADDRESS)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: ET_IDENTIFIER
			l_feature_address: ET_FEATURE_ADDRESS
		do
			l_dynamic_type_set := dynamic_type_set (an_expression.expression)
			l_dynamic_type := l_dynamic_type_set.static_type.primary_type
			l_temp := new_temp_variable (l_dynamic_type)
			mark_temp_variable_frozen (l_temp)
			l_temp.set_index (an_expression.expression.index)
			print_indentation
			print_temp_name (l_temp, current_file)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			print_expression (an_expression.expression)
			current_file.put_character (';')
			current_file.put_new_line
			create l_feature_address.make (l_temp)
			l_feature_address.set_index (an_expression.index)
			print_feature_address (l_feature_address)
		end

	print_false_constant (a_constant: ET_FALSE_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			l_temp: ET_IDENTIFIER
		do
			if in_operand then
				if in_call_target then
					l_temp := new_temp_variable (dynamic_type_set (a_constant).static_type.primary_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_false)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
				else
					operand_stack.force (a_constant)
				end
			else
				current_file.put_string (c_eif_false)
			end
		end

	print_feature_address (an_expression: ET_FEATURE_ADDRESS)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_queries: ET_DYNAMIC_FEATURE_LIST
			l_value_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_special_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_temp: ET_IDENTIFIER
			l_pointer: BOOLEAN
			l_name: ET_FEATURE_NAME
			l_name_expression: detachable ET_EXPRESSION
			l_query: detachable ET_DYNAMIC_FEATURE
			l_procedure: detachable ET_DYNAMIC_FEATURE
			l_pointer_type: ET_DYNAMIC_PRIMARY_TYPE
			l_implementation_class_name: STRING
		do
			l_pointer_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.pointer_type, current_type.base_type)
			l_dynamic_type_set := dynamic_type_set (an_expression)
			l_dynamic_type := l_dynamic_type_set.static_type.primary_type
			l_pointer := (l_dynamic_type = l_pointer_type)
			if not l_pointer then
					-- $feature_name is of type TYPED_POINTER.
				l_temp := new_temp_variable (l_dynamic_type)
				l_temp.set_index (an_expression.index)
				operand_stack.force (l_temp)
				l_queries := l_dynamic_type.queries
				if l_queries.is_empty then
						-- Internal error: TYPED_POINTER should have an attribute
						-- `pointer_item' at first position.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_feature_address", 1, "'TYPED_POINTER' has no attribute `pointer_item'.")
				else
					print_indentation
					print_attribute_access (l_queries.first, l_temp, l_dynamic_type, False)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
				end
			end
			if l_pointer and in_operand then
					-- $feature_name is of type POINTER.
				operand_stack.force (an_expression)
			else
				current_file.put_character ('(')
				l_name := an_expression.name
				if not attached {ET_IDENTIFIER} l_name as l_identifier then
					-- Do nothing here.
				elseif l_identifier.is_argument then
					l_name_expression := l_identifier
					l_value_type_set := dynamic_type_set (l_identifier)
				elseif l_identifier.is_local then
					l_name_expression := l_identifier
					if has_rescue then
							-- Keep track of the fact that the value of this local variable can
							-- possibly be modified. Useful to determine the 'volatile' status
							-- of the local variable when current feature has a rescue clause.
						locals_written.force_last (l_identifier.seed)
					end
					l_value_type_set := dynamic_type_set (l_identifier)
					if l_value_type_set.is_expanded then
							-- The address of the local variable is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_locals.force_last (l_identifier.seed)
					end
				elseif l_identifier.is_object_test_local then
					l_name_expression := l_identifier
					l_value_type_set := dynamic_type_set (l_identifier)
					if l_value_type_set.is_expanded then
							-- The address of the object-test local is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_object_test_locals.force_last (current_object_test_locals.value (l_identifier.seed))
					end
				elseif l_identifier.is_iteration_item then
					l_name_expression := l_identifier
					l_value_type_set := dynamic_type_set (l_identifier)
					if l_value_type_set.is_expanded then
							-- The address of the iteration cursor is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_iteration_cursors.force_last (current_iteration_cursors.value (l_identifier.seed))
					end
				elseif l_identifier.is_inline_separate_argument then
					l_name_expression := l_identifier
					l_value_type_set := dynamic_type_set (l_identifier)
					if l_value_type_set.is_expanded then
							-- The address of the inline separate argument is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_inline_separate_arguments.force_last (l_identifier.seed)
					end
				elseif l_identifier.is_temporary then
					l_name_expression := l_identifier
					l_value_type_set := dynamic_type_set (l_identifier)
					if l_value_type_set.is_expanded then
							-- The address of the temporary variable is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						mark_temp_variable_volatile (l_identifier)
					end
				end
				if l_name_expression /= Void and l_value_type_set /= Void then
					if l_value_type_set.is_expanded then
						print_type_cast (l_pointer_type, current_file)
						current_file.put_character ('&')
						print_expression (l_name_expression)
					else
						l_special_type := l_value_type_set.special_type
						l_implementation_class_name := current_feature.static_feature.implementation_class.upper_name
						if
							l_special_type /= Void and
							not l_implementation_class_name.same_string ("REFLECTED_OBJECT") and
							not l_implementation_class_name.same_string ("REFLECTED_REFERENCE_OBJECT") and
							not l_implementation_class_name.same_string ("OBJECT_GRAPH_MARKER") and
							not l_implementation_class_name.same_string ("SED_OBJECTS_TABLE") and
							not l_implementation_class_name.same_string ("INTERNAL")
						then
								-- Address of the first item in the SPECIAL object.
								-- However in reflection classes we expect to get the
								-- address of the SPECIAL object itself for introspection.
							current_file.put_character ('(')
							print_expression (l_name_expression)
							current_file.put_character ('?')
							print_type_cast (l_pointer_type, current_file)
							current_file.put_character ('(')
							current_file.put_string (c_ge_types)
							current_file.put_character ('[')
							print_attribute_type_id_access (l_name_expression, l_value_type_set.static_type.primary_type, False)
							current_file.put_character (']')
							current_file.put_character ('[')
							current_file.put_character ('0')
							current_file.put_character (']')
							current_file.put_character ('.')
							current_file.put_string (c_is_special)
							current_file.put_character ('?')
							print_type_cast (l_pointer_type, current_file)
							print_attribute_special_item_access (l_name_expression, l_special_type, False)
							current_file.put_character (':')
							print_type_cast (l_pointer_type, current_file)
							print_expression (l_name_expression)
							current_file.put_character (')')
							current_file.put_character (':')
							print_type_cast (l_pointer_type, current_file)
							current_file.put_character ('0')
							current_file.put_character (')')
						else
							print_type_cast (l_pointer_type, current_file)
							print_expression (l_name_expression)
						end
					end
				else
					l_query := current_type.seeded_dynamic_query (l_name.seed, current_dynamic_system)
					if l_query /= Void then
						if l_query.is_attribute then
							l_value_type_set := l_query.result_type_set
							if l_value_type_set = Void then
									-- Internal error: we know that `l_query' is an attribute.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_feature_address", 2, "attribute with no type.")
							elseif l_value_type_set.is_expanded then
								print_type_cast (l_pointer_type, current_file)
								current_file.put_character ('&')
								current_file.put_character ('(')
								print_attribute_access (l_query, tokens.current_keyword, current_type, False)
								current_file.put_character (')')
							else
								l_special_type := l_value_type_set.special_type
								l_implementation_class_name := current_feature.static_feature.implementation_class.upper_name
								if
									l_special_type /= Void and
									not l_implementation_class_name.same_string ("REFLECTED_OBJECT") and
									not l_implementation_class_name.same_string ("REFLECTED_REFERENCE_OBJECT") and
									not l_implementation_class_name.same_string ("OBJECT_GRAPH_MARKER") and
									not l_implementation_class_name.same_string ("SED_OBJECTS_TABLE") and
									not l_implementation_class_name.same_string ("INTERNAL")
								then
										-- Address of the first item in the SPECIAL object.
										-- However in reflection classes we expect to get the
										-- address of the SPECIAL object itself for introspection.
									l_temp := new_temp_variable (l_value_type_set.static_type.primary_type)
									current_file.put_character ('(')
									current_file.put_character ('(')
									print_temp_name (l_temp, current_file)
									current_file.put_character (' ')
									current_file.put_character ('=')
									current_file.put_character (' ')
									print_attribute_access (l_query, tokens.current_keyword, current_type, False)
									current_file.put_character (')')
									current_file.put_character ('?')
									print_type_cast (l_pointer_type, current_file)
									current_file.put_character ('(')
									current_file.put_string (c_ge_types)
									current_file.put_character ('[')
									print_attribute_type_id_access (l_temp, l_value_type_set.static_type.primary_type, False)
									current_file.put_character (']')
									current_file.put_character ('[')
									current_file.put_character ('0')
									current_file.put_character (']')
									current_file.put_character ('.')
									current_file.put_string (c_is_special)
									current_file.put_character ('?')
									print_type_cast (l_pointer_type, current_file)
									print_attribute_special_item_access (l_temp, l_special_type, False)
									current_file.put_character (':')
									print_type_cast (l_pointer_type, current_file)
									print_attribute_access (l_query, tokens.current_keyword, current_type, False)
									current_file.put_character (')')
									current_file.put_character (':')
									print_type_cast (l_pointer_type, current_file)
									current_file.put_character ('0')
									current_file.put_character (')')
									mark_temp_variable_free (l_temp)
								else
									print_type_cast (l_pointer_type, current_file)
									print_attribute_access (l_query, tokens.current_keyword, current_type, False)
								end
							end
						else
							register_called_feature (l_query)
							print_type_cast (l_pointer_type, current_file)
							current_file.put_character ('(')
							current_file.put_character ('&')
							print_address_routine_name (l_query, current_type, current_file)
							current_file.put_character (')')
						end
					else
						l_procedure := current_type.seeded_dynamic_procedure (l_name.seed, current_dynamic_system)
						if l_procedure /= Void then
							register_called_feature (l_procedure)
							print_type_cast (l_pointer_type, current_file)
							current_file.put_character ('(')
							current_file.put_character ('&')
							print_address_routine_name (l_procedure, current_type, current_file)
							current_file.put_character (')')
						else
								-- Internal error: It has been checked in ET_FEATURE_CHECKER that
								-- we should have either $argument, $local or $feature_name.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_feature_address", 3, "unknown `name' in '$name'.")
						end
					end
				end
				current_file.put_character (')')
			end
			if not l_pointer then
				print_semicolon_newline
			end
		end

	print_formal_argument (a_name: ET_IDENTIFIER)
			-- Print formal argument `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_argument: a_name.is_argument
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
			l_seed: INTEGER
		do
			l_seed := a_name.seed
			if
				attached inlining_context as l_inlining_context and then
				(l_seed >= 1 and l_seed <= l_inlining_context.arguments.count) and then
				attached {ET_DYNAMIC_INLINED_EXPRESSION} l_inlining_context.arguments.item (l_seed) as l_inlined_argument
			then
				process_inlined_expression (l_inlined_argument)
			elseif current_agent /= Void then
				l_seed := a_name.seed
				if l_seed < 1 or l_seed > agent_arguments.count then
						-- Internal error: we know at this stage that the
						-- number of formal and actual aguments is the same.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_formal_argument", 1, "invalid formal argument.")
				else
					agent_arguments.actual_argument (l_seed).process (Current)
				end
			elseif in_operand then
				operand_stack.force (a_name)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_name)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
					current_file.put_character ('&')
					print_argument_name (a_name, current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
				else
					print_argument_name (a_name, current_file)
				end
			else
				print_argument_name (a_name, current_file)
			end
		end

	print_hexadecimal_integer_constant (a_constant: ET_HEXADECIMAL_INTEGER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		do
			print_integer_constant (a_constant)
		end

	print_if_expression (a_expression: ET_IF_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_temp: ET_IDENTIFIER
			l_temp_index: INTEGER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_TYPE
			l_expression: ET_EXPRESSION
			l_expression_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_elseif: ET_ELSEIF_EXPRESSION
			i, nb: INTEGER
		do
			assignment_target := Void
				-- Declaration of temporary result.
			l_dynamic_type_set := dynamic_type_set (a_expression)
			l_dynamic_type := l_dynamic_type_set.static_type
			l_temp := new_temp_variable (l_dynamic_type.primary_type)
			mark_temp_variable_frozen (l_temp)
				-- We will set the index of `l_temp' later because
				-- it could still be used in `call_operands'.
			l_temp_index := a_expression.index
			operand_stack.force (l_temp)
			print_operand (a_expression.conditional_expression)
			fill_call_operands (1)
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			l_expression := a_expression.then_expression
			l_expression_dynamic_type_set := dynamic_type_set (l_expression)
			print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_dynamic_type)
			fill_call_operands (1)
			if call_operands.first /= l_temp then
				print_indentation
				print_temp_name (l_temp, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_dynamic_type)
				current_file.put_character (';')
				current_file.put_new_line
			end
			call_operands.wipe_out
			dedent
			print_indentation
			current_file.put_character ('}')
			if attached a_expression.elseif_parts as l_elseif_parts then
				nb := l_elseif_parts.count
				from i := 1 until i > nb loop
					l_elseif := l_elseif_parts.item (i)
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_operand (l_elseif.conditional_expression)
					fill_call_operands (1)
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_expression (call_operands.first)
					call_operands.wipe_out
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					l_expression := l_elseif.then_expression
					l_expression_dynamic_type_set := dynamic_type_set (l_expression)
					print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_dynamic_type)
					fill_call_operands (1)
					if call_operands.first /= l_temp then
						print_indentation
						print_temp_name (l_temp, current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_dynamic_type)
						current_file.put_character (';')
						current_file.put_new_line
					end
					call_operands.wipe_out
					dedent
					print_indentation
					current_file.put_character ('}')
					i := i + 1
				end
			end
			current_file.put_character (' ')
			current_file.put_string (c_else)
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			l_expression := a_expression.else_expression
			l_expression_dynamic_type_set := dynamic_type_set (l_expression)
			print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_dynamic_type)
			fill_call_operands (1)
			if call_operands.first /= l_temp then
				print_indentation
				print_temp_name (l_temp, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_dynamic_type)
				current_file.put_character (';')
				current_file.put_new_line
			end
			call_operands.wipe_out
			dedent
			print_indentation
			current_file.put_character ('}')
			from i := 1 until i > nb loop
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				i := i + 1
			end
			current_file.put_new_line
			if l_temp_index /= 0 then
					-- We had to wait until this stage to set the index of `l_temp'
					-- because it could have still been used in `call_operands'.
				l_temp.set_index (l_temp_index)
			end
			mark_temp_variable_unfrozen (l_temp)
		end

	print_infix_cast_expression (an_expression: ET_INFIX_CAST_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			an_expression.expression.process (Current)
		end

	print_infix_expression (an_expression: ET_INFIX_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_name: ET_CALL_NAME
			l_target: ET_EXPRESSION
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_static_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			l_target := an_expression.left
			l_name := an_expression.name
			l_target_type_set := dynamic_type_set (l_target)
			l_target_type := l_target_type_set.static_type
			l_target_static_type := l_target_type.primary_type
			if
				l_target_static_type = current_dynamic_system.boolean_type and in_operand and then
				(l_name.is_infix_and_then or l_name.is_infix_or_else or l_name.is_infix_implies or
				l_name.is_infix_and or l_name.is_infix_or or
				l_name.is_infix_and_then_symbol or l_name.is_infix_or_else_symbol or
				l_name.is_infix_implies_symbol or l_name.is_infix_and_symbol or l_name.is_infix_or_symbol)
			then
				print_semistrict_infix_expression (an_expression)
			else
				print_qualified_call_expression (an_expression)
			end
		end

	print_semistrict_infix_expression (a_call: ET_INFIX_EXPRESSION)
			-- Print semistrict infix expression.
		require
			a_call_not_void: a_call /= Void
			boolean_operator: dynamic_type_set (a_call.left).static_type.primary_type = current_dynamic_system.boolean_type
			a_call_is_semistrict: a_call.name.is_infix_and_then or a_call.name.is_infix_or_else or a_call.name.is_infix_implies or
				a_call.name.is_infix_and or a_call.name.is_infix_or or
				a_call.name.is_infix_and_then_symbol or a_call.name.is_infix_or_else_symbol or
				a_call.name.is_infix_implies_symbol or a_call.name.is_infix_and_symbol or a_call.name.is_infix_or_symbol
		local
			l_name: ET_CALL_NAME
			l_target: ET_EXPRESSION
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
			l_semistrict_target: ET_WRITABLE
			l_implies: BOOLEAN
			l_or: BOOLEAN
			l_actual: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_semistrict_target_not_set: BOOLEAN
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_target := a_call.left
			l_name := a_call.name
			l_target_type_set := dynamic_type_set (l_target)
			l_target_type := l_target_type_set.static_type
			l_target_static_type := l_target_type.primary_type
			if in_operand then
				if l_assignment_target /= Void then
					l_semistrict_target := l_assignment_target
					print_operand (l_target)
				else
					l_temp := new_temp_variable (l_target_static_type)
						-- We will set the index of `l_temp' later because
						-- it could still be used in `call_operands'.
					l_temp_index := l_target.index
					mark_temp_variable_frozen (l_temp)
					l_semistrict_target := l_temp
					print_assignment_operand (l_target, l_target_type_set, l_semistrict_target, l_target_static_type)
				end
				fill_call_operands (1)
				l_semistrict_target_not_set := l_semistrict_target /= call_operands.first
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				l_or := l_name.is_infix_or_else or l_name.is_infix_or or l_name.is_infix_or_else_symbol or l_name.is_infix_or_symbol
				if l_or then
					current_file.put_character ('!')
					current_file.put_character ('(')
				end
				print_expression (call_operands.first)
				if l_or then
					current_file.put_character (')')
				end
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				call_operands.wipe_out
				indent
				l_actual := a_call.right
				l_actual_type_set := dynamic_type_set (l_actual)
				print_assignment_operand (l_actual, l_actual_type_set, l_semistrict_target, l_target_static_type)
				fill_call_operands (1)
				if l_semistrict_target /= call_operands.first then
					print_indentation
					print_writable (l_semistrict_target)
					print_assign_to
					print_attachment_expression (call_operands.first, l_actual_type_set, l_target_static_type)
					print_semicolon_newline
				end
				dedent
				print_indentation
				current_file.put_character ('}')
				l_implies := l_name.is_infix_implies or l_name.is_infix_implies_symbol
				if l_implies or l_semistrict_target_not_set then
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					print_writable (l_semistrict_target)
					print_assign_to
					if l_or or l_implies then
						current_file.put_string (c_eif_true)
					else
						current_file.put_string (c_eif_false)
					end
					print_semicolon_newline
					dedent
					print_indentation
					current_file.put_character ('}')
				end
				current_file.put_new_line
				operand_stack.force (l_semistrict_target)
				call_operands.wipe_out
				if l_temp /= Void then
					mark_temp_variable_unfrozen (l_temp)
					if l_temp_index /= 0 then
							-- We had to wait until this stage to set the index of `l_temp'
							-- because it could have still been used in `call_operands'.
						l_temp.set_index (l_temp_index)
					end
				end
			end
		end

	print_inline_separate_argument (a_name: ET_IDENTIFIER)
			-- Print inline separate argument `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_inline_separate_argument: a_name.is_inline_separate_argument
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
		do
			if in_operand then
				operand_stack.force (a_name)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_name)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
						--
						-- The address of the inline separate argument is used, we have to make
						-- sure that the C optimizer will not implement it with 'register'.
					volatile_inline_separate_arguments.force_last (a_name.seed)
					current_file.put_character ('&')
					print_inline_separate_argument_name (a_name, current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
				else
					print_inline_separate_argument_name (a_name, current_file)
				end
			else
				print_inline_separate_argument_name (a_name, current_file)
			end
		end

	print_inlined_expression (a_expression: ET_INLINED_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_old_feature: ET_DYNAMIC_FEATURE
			l_new_inlined_operand: ET_DYNAMIC_INLINED_EXPRESSION
			l_old_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_expression: ET_EXPRESSION
			l_index: INTEGER
			l_old_index: INTEGER
			l_old_current_index: INTEGER
			l_old_index_offset: INTEGER
		do
			if not attached {ET_DYNAMIC_INLINED_EXPRESSION} a_expression as l_inlined_expression then
					-- Internal error: not supported.
				set_fatal_error
				error_handler.report_giaab_error (generator, "print_inlined_expression", 1)
			else
				l_old_inlining_context := inlining_context
				l_inlining_context := l_inlined_expression.context
				inlining_context := l_inlining_context
				l_old_feature := current_feature
				current_feature := l_inlining_context.current_feature
				current_type := current_feature.target_type
				current_dynamic_type_sets := current_feature.dynamic_type_sets
				l_old_current_index := current_index
				current_index := current_feature.current_index
				l_old_index_offset := index_offset
				index_offset := 0
				l_expression := l_inlined_expression.expression
				l_old_index := l_expression.index
				extra_dynamic_type_sets.force_last (l_inlined_expression.dynamic_type_set)
				l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
				l_expression.set_index (l_index)
				l_expression.process (Current)
				if in_operand then
					if operand_stack.is_empty then
							-- Internal error: an operand should have been added.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_inlined_expression", 2, "operand has not been added")
					elseif operand_stack.item = l_expression then
						operand_stack.replace (l_inlined_expression)
					else
						l_new_inlined_operand := new_inlined_operand (operand_stack.item, l_inlining_context)
						operand_stack.replace (l_new_inlined_operand)
					end
				end
				l_expression.set_index (l_old_index)
				extra_dynamic_type_sets.remove_last
				current_feature := l_old_feature
				current_type := current_feature.target_type
				current_dynamic_type_sets := current_feature.dynamic_type_sets
				current_index := l_old_current_index
				index_offset := l_old_index_offset
				inlining_context := l_old_inlining_context
			end
		end

	print_integer_constant (a_constant: ET_INTEGER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			l_temp: ET_IDENTIFIER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				if in_call_target then
					in_operand := False
					l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
					l_temp := new_temp_variable (l_dynamic_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_integer_constant (a_constant)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
					in_operand := True
				else
					operand_stack.force (a_constant)
				end
			else
				l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
				print_type_cast (l_dynamic_type, current_file)
				current_file.put_character ('(')
				print_integer_value (a_constant.value, a_constant.is_negative, l_dynamic_type, False)
				current_file.put_character (')')
			end
		end

	print_integer_value (a_abs_value: NATURAL_64; a_is_negative: BOOLEAN; a_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE; a_in_case_statement: BOOLEAN)
			-- Print integer of type `a_dynamic_type' with absolute value `a_abs_value'
			-- with a minus sign if `a_is_negative'.
			-- `a_in_case_statement' is used because lcc-win32 does not consider 64 bit constants as constants in case statement.
			-- We have to cast them down to 32 bit constants, with the risk that they do not fit in 32 bits.
		require
			a_dynamic_type_not_void: a_dynamic_type /= Void
		do
			if current_universe_impl.integer_8_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				current_file.put_string (c_ge_int8)
				current_file.put_character ('(')
				if a_is_negative then
					current_file.put_character ('-')
				end
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
			elseif current_universe_impl.integer_16_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				current_file.put_string (c_ge_int16)
				current_file.put_character ('(')
				if a_is_negative then
					current_file.put_character ('-')
				end
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
			elseif current_universe_impl.integer_32_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				if a_is_negative and then a_abs_value = {INTEGER_32}.Max_value.to_natural_64 + 1 then
					current_file.put_string (c_ge_min_int32)
				else
					current_file.put_string (c_ge_int32)
					current_file.put_character ('(')
					if a_is_negative then
						current_file.put_character ('-')
					end
					current_file.put_natural_64 (a_abs_value)
					current_file.put_character (')')
				end
			elseif current_universe_impl.integer_64_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				if a_in_case_statement then
					current_file.put_string (c_ge_case_int64)
					current_file.put_character ('(')
				end
				if a_is_negative and then a_abs_value = {INTEGER_64}.Max_value.to_natural_64 + 1 then
					current_file.put_string (c_ge_min_int64)
				else
					current_file.put_string (c_ge_int64)
					current_file.put_character ('(')
					if a_is_negative then
						current_file.put_character ('-')
					end
					current_file.put_natural_64 (a_abs_value)
					current_file.put_character (')')
				end
				if a_in_case_statement then
					current_file.put_character (')')
				end
			elseif current_universe_impl.natural_8_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				current_file.put_string (c_ge_nat8)
				current_file.put_character ('(')
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
			elseif current_universe_impl.natural_16_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				current_file.put_string (c_ge_nat16)
				current_file.put_character ('(')
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
			elseif current_universe_impl.natural_32_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				current_file.put_string (c_ge_nat32)
				current_file.put_character ('(')
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
			elseif current_universe_impl.natural_64_type.same_named_type (a_dynamic_type.base_type, current_type.base_type, current_type.base_type) then
				if a_in_case_statement then
					current_file.put_string (c_ge_case_nat64)
					current_file.put_character ('(')
				end
				current_file.put_string (c_ge_nat64)
				current_file.put_character ('(')
				current_file.put_natural_64 (a_abs_value)
				current_file.put_character (')')
				if a_in_case_statement then
					current_file.put_character (')')
				end
			else
				if a_is_negative then
					current_file.put_character ('-')
				end
				current_file.put_natural_64 (a_abs_value)
			end
		end

	print_inspect_expression (a_expression: ET_INSPECT_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_temp: ET_IDENTIFIER
			l_temp_index: INTEGER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_expression_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_TYPE
			l_expression: ET_EXPRESSION
			l_when_part: ET_WHEN_EXPRESSION
			l_choices: ET_CHOICE_LIST
			l_choice: ET_CHOICE
			i, nb: INTEGER
			j, nb2: INTEGER
			l_has_case: BOOLEAN
			l_lower: detachable ET_CONSTANT
			l_upper: detachable ET_CONSTANT
			l_lower_integer: detachable ET_INTEGER_CONSTANT
			l_upper_integer: detachable ET_INTEGER_CONSTANT
			l_lower_character: detachable ET_CHARACTER_CONSTANT
			l_upper_character: detachable ET_CHARACTER_CONSTANT
			k, nb3: INTEGER
			l_i_nat32, l_nb_nat32: NATURAL_32
			l_value_type_set: ET_DYNAMIC_TYPE_SET
			l_value_type: ET_DYNAMIC_PRIMARY_TYPE
			l_stop: BOOLEAN
		do
-- TODO
			assignment_target := Void
				-- Declaration of temporary result.
			l_dynamic_type_set := dynamic_type_set (a_expression)
			l_dynamic_type := l_dynamic_type_set.static_type
			l_temp := new_temp_variable (l_dynamic_type.primary_type)
			mark_temp_variable_frozen (l_temp)
				-- We will set the index of `l_temp' later because
				-- it could still be used in `call_operands'.
			l_temp_index := a_expression.index
			operand_stack.force (l_temp)
			l_expression := a_expression.conditional.expression
			l_value_type_set := dynamic_type_set (l_expression)
			l_value_type := l_value_type_set.static_type.primary_type
			print_operand (l_expression)
			fill_call_operands (1)
			print_indentation
			current_file.put_string (c_switch)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			call_operands.wipe_out
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			if attached a_expression.when_parts as l_when_parts then
				nb := l_when_parts.count
				from i := 1 until i > nb loop
					l_when_part := l_when_parts.item (i)
					l_choices := l_when_part.choices
					nb2 := l_choices.count
					if nb2 = 0 then
						-- Do nothing.
					else
						l_has_case := False
						from j := 1 until j > nb2 loop
							l_choice := l_choices.choice (j)
							if l_choice.is_range then
-- TODO
								l_lower := choice_constant (l_choice.lower)
								l_upper := choice_constant (l_choice.upper)
								if attached {ET_INTEGER_CONSTANT} l_lower as l_integer_constant then
									l_lower_integer := l_integer_constant
									l_lower_character := Void
								elseif attached {ET_CHARACTER_CONSTANT} l_lower as l_character_constant then
									l_lower_character := l_character_constant
									l_lower_integer := Void
								else
									l_lower_integer := Void
									l_lower_character := Void
								end
								if attached {ET_INTEGER_CONSTANT} l_upper as l_integer_constant then
									l_upper_integer := l_integer_constant
									l_upper_character := Void
								elseif attached {ET_CHARACTER_CONSTANT} l_upper as l_character_constant then
									l_upper_character := l_character_constant
									l_upper_integer := Void
								else
									l_upper_integer := Void
									l_upper_character := Void
								end
								if l_lower_integer /= Void and l_upper_integer /= Void then
									from
-- TODO: check type of inspect value.
										k := l_lower_integer.to_integer_32
										nb3 := l_upper_integer.to_integer_32
									until
										k > nb3
									loop
										l_has_case := True
										print_indentation
										current_file.put_string (c_case)
										current_file.put_character (' ')
										print_integer_value (k.abs.to_natural_64, k < 0, l_value_type, True)
										current_file.put_character (':')
										current_file.put_new_line
										k := k + 1
									end
								elseif l_lower_character /= Void and l_upper_character /= Void then
									from
										l_i_nat32 := l_lower_character.value.natural_32_code
										l_nb_nat32 := l_upper_character.value.natural_32_code
										l_stop := l_i_nat32 > l_nb_nat32
									until
										l_stop
									loop
										l_has_case := True
										print_indentation
										current_file.put_string (c_case)
										current_file.put_character (' ')
										if current_system.character_32_type.same_named_type (l_value_type.base_type, current_type.base_type, current_type.base_type) then
											current_file.put_string (c_ge_nat32)
											current_file.put_character ('(')
											current_file.put_natural_32 (l_i_nat32)
											current_file.put_character (')')
										else
												-- Type cast needed when compiling with mingw when the character
												-- is represented by '\xyz' where x is not 0.
											print_type_cast (l_value_type, current_file)
											print_escaped_character_8 (l_i_nat32.to_character_8)
										end
										current_file.put_character (':')
										current_file.put_new_line
										if l_i_nat32 = l_nb_nat32 then
											l_stop := True
										else
											l_i_nat32 := l_i_nat32 + 1
										end
									end
								else
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_inspect_expression - range")
								end
							else
								l_has_case := True
								print_indentation
								current_file.put_string (c_case)
								current_file.put_character (' ')
								l_lower := choice_constant (l_choice.lower)
								if attached {ET_INTEGER_CONSTANT} l_lower as l_integer_constant then
									print_integer_value (l_integer_constant.value, l_integer_constant.is_negative, l_value_type, True)
								elseif attached {ET_CHARACTER_CONSTANT} l_lower as l_character_constant then
									if current_system.character_32_type.same_named_type (l_value_type.base_type, current_type.base_type, current_type.base_type) then
										current_file.put_string (c_ge_nat32)
										current_file.put_character ('(')
										current_file.put_natural_32 (l_character_constant.value.natural_32_code)
										current_file.put_character (')')
									else
											-- Type cast needed when compiling with mingw when the character
											-- is represented by '\xyz' where x is not 0.
										print_type_cast (l_value_type, current_file)
										print_escaped_character_8 (l_character_constant.value.to_character_8)
									end
								else
									if l_value_type /= dynamic_type_set (l_choice.lower).static_type.primary_type then
										print_type_cast (l_value_type, current_file)
									end
									print_expression (l_choice.lower)
								end
								current_file.put_character (':')
								current_file.put_new_line
							end
							j := j + 1
						end
						if l_has_case then
							indent
							l_expression := l_when_part.then_expression
							l_expression_dynamic_type_set := dynamic_type_set (l_expression)
							print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_dynamic_type)
							fill_call_operands (1)
							if call_operands.first /= l_temp then
								print_indentation
								print_temp_name (l_temp, current_file)
								current_file.put_character (' ')
								current_file.put_character ('=')
								current_file.put_character (' ')
								print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_dynamic_type)
								current_file.put_character (';')
								current_file.put_new_line
							end
							call_operands.wipe_out
							print_indentation
							current_file.put_string (c_break)
							current_file.put_character (';')
							current_file.put_new_line
							dedent
						end
					end
					i := i + 1
				end
			end
			print_indentation
			current_file.put_string (c_default)
			current_file.put_character (':')
			current_file.put_new_line
			if attached a_expression.else_part as l_else_part then
				indent
				l_expression := l_else_part.expression
				l_expression_dynamic_type_set := dynamic_type_set (l_expression)
				print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_dynamic_type)
				fill_call_operands (1)
				if call_operands.first /= l_temp then
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_dynamic_type)
					current_file.put_character (';')
					current_file.put_new_line
				end
				call_operands.wipe_out
				print_indentation
				current_file.put_string (c_break)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
			else
				indent
				print_indentation
				current_file.put_string (c_ge_raise)
				current_file.put_character ('(')
				current_file.put_string (c_ge_ex_when)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_break)
				current_file.put_character (';')
				current_file.put_new_line
				dedent
			end
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if l_temp_index /= 0 then
					-- We had to wait until this stage to set the index of `l_temp'
					-- because it could have still been used in `call_operands'.
				l_temp.set_index (l_temp_index)
			end
			mark_temp_variable_unfrozen (l_temp)
		end

	print_iteration_cursor (a_iteration_cursor: ET_ITERATION_CURSOR)
			-- Print `a_iteration_cursor'.
		require
			a_iteration_cursor_not_void: a_iteration_cursor /= Void
		local
			l_name: ET_IDENTIFIER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
		do
			l_name := a_iteration_cursor.item_name
			if in_operand then
				operand_stack.force (a_iteration_cursor)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_iteration_cursor)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
						--
						-- The address of the iteration cursor is used, we have to make
						-- sure that the C optimizer will not implement it with 'register'.
					volatile_iteration_cursors.force_last (current_iteration_cursors.value (l_name.seed))
					current_file.put_character ('&')
					print_iteration_cursor_name (l_name, current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (l_name, l_call_target_type, call_target_check_void)
				else
					print_iteration_cursor_name (l_name, current_file)
				end
			else
				print_iteration_cursor_name (l_name, current_file)
			end
		end

	print_iteration_item (a_name: ET_IDENTIFIER)
			-- Print iteration item `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_iteration_item: a_name.is_iteration_item
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
			l_current_closure: ET_CLOSURE
			l_iteration_component: ET_ITERATION_COMPONENT
			l_seed: INTEGER
			l_done: BOOLEAN
		do
			if attached {ET_INLINE_AGENT} current_agent as l_inline_agent then
				l_current_closure := l_inline_agent
			else
				l_current_closure := current_feature.static_feature
			end
			if attached l_current_closure.iteration_components as l_iteration_components then
				l_seed := a_name.seed
				if l_seed >= 1 and l_seed <= l_iteration_components.count then
					l_iteration_component := l_iteration_components.iteration_component (l_seed)
					if a_name /= l_iteration_component.unfolded_cursor_name and then not l_iteration_component.has_cursor_name then
							-- We are not in the case of 'across ... as ...' with 'obsolete_iteration_mode' set to True.
							-- Print the unfolded form of iteration item: 'unfolded_cursor_name.item'
						print_qualified_call_expression (l_iteration_component.cursor_item_expression)
						l_done := True
					end
				end
			end
			if l_done then
				-- Already printed.
			elseif in_operand then
				operand_stack.force (a_name)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_name)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
						--
						-- The address of the iteration cursor is used, we have to make
						-- sure that the C optimizer will not implement it with 'register'.
					volatile_iteration_cursors.force_last (current_iteration_cursors.value (a_name.seed))
					current_file.put_character ('&')
					print_iteration_cursor_name (a_name, current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
				else
					print_iteration_cursor_name (a_name, current_file)
				end
			else
				print_iteration_cursor_name (a_name, current_file)
			end
		end

	print_iteration_expression (an_expression: ET_ITERATION_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_temp: ET_IDENTIFIER
			l_temp_index: INTEGER
			l_expression: ET_EXPRESSION
			l_cursor_name: ET_IDENTIFIER
			l_cursor_type: ET_DYNAMIC_TYPE
			l_cursor_type_set: ET_DYNAMIC_TYPE_SET
			l_boolean_type: ET_DYNAMIC_PRIMARY_TYPE
			l_expression_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_temp_var: detachable ET_IDENTIFIER
			l_temp_old_var: detachable ET_IDENTIFIER
			l_var_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			assignment_target := Void
			l_cursor_name := an_expression.unfolded_cursor_name
			l_cursor_type_set := dynamic_type_set (l_cursor_name)
			l_cursor_type := l_cursor_type_set.static_type
			current_iteration_cursor_types.force_last (l_cursor_type.primary_type)
			current_iteration_cursors.force_last (current_iteration_cursor_types.count, l_cursor_name.seed)
				-- Call to `new_cursor'.
			print_assignment_operand (an_expression.new_cursor_expression, l_cursor_type_set, l_cursor_name, l_cursor_type)
			fill_call_operands (1)
			if call_operands.first /= l_cursor_name then
				print_indentation
				print_iteration_cursor_name (l_cursor_name, current_file)
				print_assign_to
				print_attachment_expression (call_operands.first, l_cursor_type_set, l_cursor_type)
				print_semicolon_newline
			end
			call_operands.wipe_out
				-- Declaration of temporary result.
			l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
			l_temp := new_temp_variable (l_boolean_type)
			mark_temp_variable_frozen (l_temp)
				-- We will set the index of `l_temp' later because
				-- it could still be used in `call_operands'.
			l_temp_index := an_expression.index
			operand_stack.force (l_temp)
			print_indentation
			print_temp_name (l_temp, current_file)
			print_assign_to
			if an_expression.is_all then
				current_file.put_string (c_eif_true)
			else
				current_file.put_string (c_eif_false)
			end
			print_semicolon_newline
			if current_type.base_class.loop_assertions_enabled and attached an_expression.variant_part as l_variant_part then
				l_var_type := dynamic_type_set (l_variant_part.expression).static_type.primary_type
				l_temp_var := new_temp_variable (l_var_type)
				l_temp_old_var := new_temp_variable (l_var_type)
				print_indentation
				print_temp_name (l_temp_old_var, current_file)
				print_assign_to
				current_file.put_integer (-1)
				print_semicolon_newline
			end
			print_indentation
			current_file.put_string (c_while)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('1')
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			if current_type.base_class.loop_assertions_enabled then
				print_loop_assertions_and_while_header (an_expression, l_temp_var, l_temp_old_var)
			end
			print_operand (an_expression.cursor_after_expression)
			if attached an_expression.until_conditional as l_until_conditional then
				l_expression := l_until_conditional.expression
				print_operand (l_expression)
				fill_call_operands (2)
			else
				fill_call_operands (1)
			end
			print_indentation
			current_file.put_string (c_if)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('(')
			if an_expression.is_all then
				current_file.put_character ('!')
			end
			print_temp_name (l_temp, current_file)
			current_file.put_character (')')
			current_file.put_character ('|')
			current_file.put_character ('|')
			current_file.put_character ('(')
			print_expression (call_operands.first)
			current_file.put_character (')')
			if call_operands.count = 2 then
				current_file.put_character ('|')
				current_file.put_character ('|')
				current_file.put_character ('(')
				print_expression (call_operands.item (2))
				current_file.put_character (')')
			end
			current_file.put_character (')')
			current_file.put_character (' ')
			current_file.put_character ('{')
			current_file.put_new_line
			call_operands.wipe_out
			indent
			print_indentation
			current_file.put_string (c_break)
			print_semicolon_newline
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
				-- Iteration expression.
			l_expression := an_expression.iteration_expression
			l_expression_dynamic_type_set := dynamic_type_set (l_expression)
			print_assignment_operand (l_expression, l_expression_dynamic_type_set, l_temp, l_boolean_type)
			fill_call_operands (1)
			if call_operands.first /= l_temp then
				print_indentation
				print_temp_name (l_temp, current_file)
				print_assign_to
				print_attachment_expression (call_operands.first, l_expression_dynamic_type_set, l_boolean_type)
				print_semicolon_newline
			end
			call_operands.wipe_out
				-- Call to `forth'.
			print_instruction (an_expression.cursor_forth_instruction)
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			if l_temp_var /= Void then
				mark_temp_variable_free (l_temp_var)
			end
			if l_temp_old_var /= Void then
				mark_temp_variable_free (l_temp_old_var)
			end
			mark_temp_variable_unfrozen (l_temp)
			if l_temp_index /= 0 then
					-- We had to wait until this stage to set the index of `l_temp'
					-- because it could have still been used in `call_operands'.
				l_temp.set_index (l_temp_index)
			end
		end

	print_local_variable (a_name: ET_IDENTIFIER)
			-- Print local variable `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_local: a_name.is_local
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				operand_stack.force (a_name)
			else
				if has_rescue then
						-- Keep track of the fact that this local variable has been
						-- read. Useful to determine the 'volatile' status of the local
						-- variable when current feature has a rescue clause.
					locals_read.force_last (a_name.seed)
				end
				if attached call_target_type as l_call_target_type then
					check in_call_target: in_call_target end
					l_dynamic_type_set := dynamic_type_set (a_name)
					l_static_type := l_dynamic_type_set.static_type.primary_type
					if l_static_type.is_expanded then
							-- Pass the address of the expanded object.
							--
							-- The address of the local variable is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_locals.force_last (a_name.seed)
						current_file.put_character ('(')
						current_file.put_character ('&')
						print_local_name (a_name, current_file)
						current_file.put_character (')')
					elseif l_call_target_type.is_expanded then
							-- We need to unbox the object and then pass its address.
						print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
					else
						print_local_name (a_name, current_file)
					end
				else
					print_local_name (a_name, current_file)
				end
			end
		end

	print_manifest_array (an_expression: ET_MANIFEST_ARRAY)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			i, nb, nb2: INTEGER
			l_assignment_target: like assignment_target
			l_int_promoted: BOOLEAN
			l_double_promoted: BOOLEAN
			l_queries: ET_DYNAMIC_FEATURE_LIST
			l_integer_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_static_item_type: detachable ET_DYNAMIC_TYPE
			l_static_item_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_item_type_set: ET_DYNAMIC_TYPE_SET
			l_item: ET_EXPRESSION
			l_argument_limit: INTEGER
		do
			--l_argument_limit := 253 -- tcc limitation
			l_argument_limit := 1024 -- lcc-win32 limitation
			l_assignment_target := assignment_target
			assignment_target := Void
			l_dynamic_type := dynamic_type_set (an_expression).static_type.primary_type
			if l_dynamic_type.attribute_count < 3 then
					-- Internal error: class "ARRAY" should have at least the
					-- features 'area', 'lower' and 'upper' as first features.
					-- Already reported in ET_DYNAMIC_SYSTEM.compile_kernel.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_manifest_array", 1, "'ARRAY' has less than 3 attributes.")
			else
				l_queries := l_dynamic_type.queries
				l_dynamic_type_set := l_queries.first.result_type_set
				if l_dynamic_type_set = Void then
						-- Error in feature 'area', already reported in ET_DYNAMIC_SYSTEM.compile_kernel.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_manifest_array", 2, "attribute `area' has no type.")
				elseif not attached {ET_DYNAMIC_SPECIAL_TYPE} l_dynamic_type_set.static_type.primary_type as l_special_type then
						-- Internal error: it has already been checked in ET_DYNAMIC_SYSTEM.compile_kernel
						-- that the attribute `area' is of SPECIAL type.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_manifest_array", 3, "attribute `area' is not of type 'SPECIAL'.")
				else
					l_static_item_type := l_special_type.item_type_set.static_type
				end
				l_dynamic_type_set := l_queries.item (3).result_type_set
				if l_dynamic_type_set = Void then
						-- Error in feature 'upper', already reported in ET_DYNAMIC_SYSTEM.compile_kernel.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_manifest_array", 4, "attribute `upper' has no type.")
				else
					l_integer_type := l_dynamic_type_set.static_type.primary_type
				end
			end
			if l_static_item_type /= Void and l_integer_type /= Void then
				l_static_item_primary_type := l_static_item_type.primary_type
				nb := an_expression.count
				from i := 1 until i > nb loop
					l_item := an_expression.expression (i)
					l_dynamic_item_type_set := dynamic_type_set (l_item)
					print_attachment_operand (l_item, l_dynamic_item_type_set)
					i := i + 1
				end
				fill_call_operands (nb)
				if
					l_static_item_primary_type = current_dynamic_system.boolean_type or
					l_static_item_primary_type = current_dynamic_system.character_8_type or
					l_static_item_primary_type = current_dynamic_system.integer_8_type or
					l_static_item_primary_type = current_dynamic_system.natural_8_type or
					l_static_item_primary_type = current_dynamic_system.integer_16_type or
					l_static_item_primary_type = current_dynamic_system.natural_16_type
				then
						-- ISO C 99 says that through "..." the types are promoted to
						-- 'int', and that promotion to 'int' leaves the type unchanged
						-- if all values cannot be represented with an 'int' or
						-- 'unsigned int'.
					l_int_promoted := True
				elseif
					l_static_item_primary_type = current_dynamic_system.real_32_type
				then
						-- ISO C 99 says that 'float' is promoted to 'double' when
						-- passed as argument of a function.
					l_double_promoted := True
				end
				manifest_array_types.force_last (l_dynamic_type)
					-- `nb2' is the maximum number of array elements that can be passed to the
					-- function 'GE_ma<type-id>' based on C compiler limitation.
				nb2 := nb.min (l_argument_limit - 2)
				if in_operand then
					if l_assignment_target /= Void and nb <= nb2 then
						operand_stack.force (l_assignment_target)
						print_indentation
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_dynamic_type)
						mark_temp_variable_frozen (l_temp)
							-- We will set the index of `l_temp' later because
							-- it could still be used in `call_operands'.
						l_temp_index := an_expression.index
						operand_stack.force (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
					end
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
				end
				current_file.put_string (c_ge_ma)
				current_file.put_integer (l_dynamic_type.id)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				print_type_cast (l_integer_type, current_file)
				current_file.put_integer (nb)
				print_comma
				print_type_cast (l_integer_type, current_file)
				current_file.put_integer (nb2)
				from i := 1 until i > nb loop
					current_file.put_character (',')
						-- Print one item per line for better readability,
						-- avoiding too long lines for big arrays.
					current_file.put_new_line
					l_item := call_operands.item (i)
					l_dynamic_item_type_set := dynamic_type_set (l_item)
					if l_int_promoted then
						current_file.put_character ('(')
						current_file.put_string (c_int)
						current_file.put_character (')')
						current_file.put_character ('(')
						print_attachment_expression (l_item, l_dynamic_item_type_set, l_static_item_type)
						current_file.put_character (')')
					elseif l_double_promoted then
						current_file.put_character ('(')
						current_file.put_string (c_double)
						current_file.put_character (')')
						current_file.put_character ('(')
						print_attachment_expression (l_item, l_dynamic_item_type_set, l_static_item_type)
						current_file.put_character (')')
					else
						print_attachment_expression (l_item, l_dynamic_item_type_set, l_static_item_type)
					end
					i := i + 1
					if i > nb2 and i <= nb then
							-- `nb2' is the maximum number of array elements that can be passed to the
							-- function 'GE_ma<type-id>' based on C compiler limitation.
						check l_temp_not_void: l_temp /= Void then
							nb2 := (nb - i + 1).min (l_argument_limit - 3)
							current_file.put_character (')')
							current_file.put_character (';')
							current_file.put_new_line
							print_indentation
							current_file.put_string (c_ge_bma)
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character ('(')
							print_expression (l_temp)
							print_comma
							print_type_cast (l_integer_type, current_file)
							current_file.put_integer (i - 1)
							print_comma
							print_type_cast (l_integer_type, current_file)
							current_file.put_integer (nb2)
							nb2 := nb2 + i - 1
							big_manifest_array_types.force_last (l_dynamic_type)
						end
					end
				end
				current_file.put_character (')')
				if in_operand then
					current_file.put_character (';')
					current_file.put_new_line
				end
				call_operands.wipe_out
				if l_temp /= Void then
					mark_temp_variable_unfrozen (l_temp)
					if l_temp_index /= 0 then
							-- We had to wait until this stage to set the index of `l_temp'
							-- because it could have still been used in `call_operands'.
						l_temp.set_index (l_temp_index)
					end
				end
			end
		end

	print_manifest_string (a_string: ET_MANIFEST_STRING)
			-- Print `a_string'.
		require
			a_string_not_void: a_string /= Void
		local
			l_string: STRING
			l_atomic: BOOLEAN
			l_temp: ET_IDENTIFIER
			l_assignment_target: like assignment_target
			l_string_type: ET_DYNAMIC_PRIMARY_TYPE
		do
				-- We consider manifest strings as being atomic expressions.
			l_atomic := True
			l_assignment_target := assignment_target
			assignment_target := Void
			if in_operand then
				if l_atomic then
					operand_stack.force (a_string)
				else
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_indentation
						print_writable (l_assignment_target)
					else
						l_string_type := dynamic_type_set (a_string).static_type.primary_type
						l_temp := new_temp_variable (l_string_type)
						l_temp.set_index (a_string.index)
						operand_stack.force (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
					end
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('(')
				end
			end
			if not l_atomic or not in_operand then
				l_string := a_string.value
				l_string_type := dynamic_type_set (a_string).static_type.primary_type
				if current_system.string_32_type.same_named_type_with_type_marks (l_string_type.base_type, tokens.implicit_detachable_type_mark, current_system.any_type, tokens.implicit_detachable_type_mark, current_system.any_type) then
					current_file.put_string (c_ge_ms32_from_utf32le)
					current_file.put_character ('(')
					print_utf8_as_escaped_string_32 (l_string)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_integer ({UC_UTF8_ROUTINES}.unicode_character_count (l_string))
					current_file.put_character (')')
				elseif current_system.immutable_string_32_type.same_named_type_with_type_marks (l_string_type.base_type, tokens.implicit_detachable_type_mark, current_system.any_type, tokens.implicit_detachable_type_mark, current_system.any_type) then
					current_file.put_string (c_ge_ims32_from_utf32le)
					current_file.put_character ('(')
					print_utf8_as_escaped_string_32 (l_string)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_integer ({UC_UTF8_ROUTINES}.unicode_character_count (l_string))
					current_file.put_character (')')
				elseif current_system.immutable_string_8_type.same_named_type_with_type_marks (l_string_type.base_type, tokens.implicit_detachable_type_mark, current_system.any_type, tokens.implicit_detachable_type_mark, current_system.any_type) then
					current_file.put_string (c_ge_ims8)
					current_file.put_character ('(')
					print_utf8_as_escaped_string_8 (l_string)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_integer ({UC_UTF8_ROUTINES}.unicode_character_count (l_string))
					current_file.put_character (')')
				else
					current_file.put_string (c_ge_ms8)
					current_file.put_character ('(')
					print_utf8_as_escaped_string_8 (l_string)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_integer ({UC_UTF8_ROUTINES}.unicode_character_count (l_string))
					current_file.put_character (')')
				end
			end
			if not l_atomic and in_operand then
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
			end
		end

	print_manifest_tuple (an_expression: ET_MANIFEST_TUPLE)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			i, nb: INTEGER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_assignment_target: like assignment_target
			l_item_type_set: ET_DYNAMIC_TYPE_SET
			l_item: ET_EXPRESSION
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_dynamic_type_set := dynamic_type_set (an_expression)
			if not attached {ET_DYNAMIC_TUPLE_TYPE} l_dynamic_type_set.static_type.primary_type as l_tuple_type then
					-- Internal error: the dynamic type of `an_expression'
					-- should be a Tuple_type.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_manifest_tuple", 1, "manifest tuple is not of type 'TUPLE'.")
			else
				nb := an_expression.count
				manifest_tuple_types.force_last (l_tuple_type)
				if in_operand then
					from i := 1 until i > nb loop
						l_item := an_expression.expression (i)
						l_item_type_set := dynamic_type_set (l_item)
						print_attachment_operand (l_item, l_item_type_set)
						i := i + 1
					end
					fill_call_operands (nb)
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_indentation
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_tuple_type)
						mark_temp_variable_frozen (l_temp)
							-- We will set the index of `l_temp' later because
							-- it could still be used in `call_operands'.
						l_temp_index := an_expression.index
						operand_stack.force (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
					end
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
				end
				current_file.put_string (c_ge_mt)
				current_file.put_integer (l_tuple_type.id)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				from i := 1 until i > nb loop
					print_comma
					if in_operand then
						l_item := call_operands.item (i)
					else
						l_item := an_expression.expression (i)
					end
					l_item_type_set := dynamic_type_set (l_item)
					print_attachment_expression (l_item, l_item_type_set, l_item_type_set.static_type)
					i := i + 1
				end
				current_file.put_character (')')
				if in_operand then
					current_file.put_character (';')
					current_file.put_new_line
					call_operands.wipe_out
					if l_temp /= Void then
						mark_temp_variable_unfrozen (l_temp)
						if l_temp_index /= 0 then
								-- We had to wait until this stage to set the index of `l_temp'
								-- because it could have still been used in `call_operands'.
							l_temp.set_index (l_temp_index)
						end
					end
				end
			end
		end

	print_manifest_type (an_expression: ET_MANIFEST_TYPE)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_type: ET_DYNAMIC_TYPE
			l_meta_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_index: INTEGER
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
				l_type := current_dynamic_system.dynamic_type (an_expression.type, current_type.base_type)
				l_meta_type := l_type.meta_type
				if l_meta_type = Void then
						-- Internal error: the meta type of this type should have been
						-- computed. It is nothing else than the type of this manifest
						-- type expression.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_manifest_type", 1, "no meta type found for manifest type.")
				else
					current_file.put_character ('(')
					current_file.put_character ('(')
					print_type_declaration (l_meta_type, current_file)
					current_file.put_character (')')
					current_file.put_character ('&')
					current_file.put_character ('(')
					current_file.put_string (c_ge_types)
					current_file.put_character ('[')
					current_file.put_integer (l_type.primary_type.id)
					current_file.put_character (']')
					current_file.put_character ('[')
					if l_type = l_type.primary_type then
						current_file.put_integer (0)
					else
						l_index := 0
						if l_type.is_attached then
							l_index := 1
						end
						if use_scoop and then l_type.is_separate then
							l_index := l_index + 2
						end
						current_file.put_integer (l_index)
					end
					current_file.put_character (']')
					current_file.put_character (')')
					current_file.put_character (')')
				end
			end
		end

	print_non_void_expression (an_expression: ET_EXPRESSION; a_check_void: BOOLEAN)
			-- Print `an_expression' which needs to be non-void.
			-- The static type (i.e. declared type) of `an_expression' is assumed to be of reference type.
			-- Useful for example when we want to get access to an attribute of this object.
			-- `a_check_void' means that we do need to check whether it is actually Void or not.
		require
			an_expression_not_void: an_expression /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := False
			in_procedure_call_target := False
			call_target_type := Void
			print_check_void_expression (an_expression, current_dynamic_system.none_type, a_check_void)
			call_target_type := old_target_type
			in_procedure_call_target := l_old_in_procedure_call_target
			in_operand := old_in_operand
		end

	print_object_equality_expression (a_expression: ET_OBJECT_EQUALITY_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_equality_types: ET_DYNAMIC_OBJECT_EQUALITY_TYPES
			l_assignment_target: like assignment_target
		do
			l_left_operand := a_expression.left
			l_right_operand := a_expression.right
			l_left_type_set := dynamic_type_set (l_left_operand)
			l_right_type_set := dynamic_type_set (l_right_operand)
			l_assignment_target := assignment_target
			assignment_target := Void
				-- Process operands.
			print_operand (l_left_operand)
				-- Call `print_attachment_operand' and not `print_operand'
				-- because the right operand might be the argument of a call
				-- to 'is_equal' and it will require some cloning if its
				-- type is expanded.
			print_attachment_operand (l_right_operand, l_right_type_set)
			fill_call_operands (2)
			assignment_target := l_assignment_target
			l_equality_types := new_object_equality_types (l_left_type_set, l_right_type_set)
			print_equality_call (a_expression, a_expression.operator.is_not_tilde, l_equality_types)
			if current_equalities.is_empty or else current_equalities.last /= l_equality_types then
				free_object_equality_types (l_equality_types)
			end
			call_operands.wipe_out
		end

	print_object_test (an_expression: ET_OBJECT_TEST)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_assignment_target: like assignment_target
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			i, nb: INTEGER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_source: ET_EXPRESSION
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_source_type: ET_DYNAMIC_TYPE
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			nb_conforming_types: INTEGER
			nb_non_conforming_types: INTEGER
			l_can_be_void: BOOLEAN
			l_name: detachable ET_IDENTIFIER
			l_type: detachable ET_TYPE
			l_same_scoop_region: BOOLEAN
			l_conforming_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_source := an_expression.expression
			l_source_type_set := dynamic_type_set (l_source)
			l_source_type := l_source_type_set.static_type
			l_name := an_expression.name
			l_type := an_expression.type
			if l_name /= Void then
				l_target_type_set := dynamic_type_set (l_name)
				l_target_type := l_target_type_set.static_type
				current_object_test_local_types.force_last (l_target_type.primary_type)
				current_object_test_locals.force_last (current_object_test_local_types.count, l_name.seed)
			elseif l_type /= Void then
				l_target_type := current_dynamic_system.dynamic_type (l_type, current_type.base_type)
			else
				l_target_type := l_source_type
			end
			l_target_primary_type := l_target_type.primary_type
			print_attachment_operand (l_source, l_source_type_set)
			fill_call_operands (1)
			if in_operand then
				if l_assignment_target /= Void then
					operand_stack.force (l_assignment_target)
					print_indentation
					print_writable (l_assignment_target)
				else
					l_temp := new_temp_variable (dynamic_type_set (an_expression).static_type.primary_type)
					mark_temp_variable_frozen (l_temp)
						-- We will set the index of `l_temp' later because
						-- it could still be used in `call_operands'.
					l_temp_index := an_expression.index
					operand_stack.force (l_temp)
					print_indentation
					print_temp_name (l_temp, current_file)
				end
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
			end
				-- Check type conformance.
			nb := l_source_type_set.count
			if l_type = Void then
				nb_conforming_types := nb
				nb_non_conforming_types := 0
			else
				from i := 1 until i > nb loop
					l_dynamic_type := l_source_type_set.dynamic_type (i)
					if l_dynamic_type.conforms_to_primary_type (l_target_primary_type) then
						nb_conforming_types := nb_conforming_types + 1
					else
						nb_non_conforming_types := nb_non_conforming_types + 1
					end
					i := i + 1
				end
			end
			if nb_conforming_types = 0 then
					-- No type in the source type set conform to the target type.
				current_file.put_string (c_eif_false)
			else
				l_can_be_void := l_source_type_set.can_be_void
				if l_can_be_void then
					current_file.put_character ('(')
					print_expression (call_operands.first)
					current_file.put_character ('?')
					current_file.put_character ('(')
				end
				l_same_scoop_region := use_scoop and then l_type /= Void and then (not l_target_type.is_separate and l_source_type.is_separate)
				if l_same_scoop_region then
					current_file.put_character ('(')
					print_attribute_region_access (call_operands.first, l_target_primary_type, False)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_region)
					current_file.put_character ('?')
					current_file.put_character ('(')
				end
				if l_name /= Void then
					current_file.put_character ('(')
					print_object_test_local_name (l_name, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					l_conforming_type_set := new_standalone_type_set (l_source_type)
					if l_source_type_set.is_never_void then
						l_conforming_type_set.set_never_void
					end
					print_attachment_expression (call_operands.first, l_conforming_type_set, l_target_type)
					free_standalone_type_set (l_conforming_type_set)
					current_file.put_character (',')
					current_file.put_character (' ')
				end
				if nb_non_conforming_types = 0 then
					-- All types in the source type set conform to the target type.
					current_file.put_string (c_eif_true)
				else
					current_object_tests.force_last (an_expression, index_offset)
					print_object_test_function_name (current_object_tests.count, current_feature, current_type, current_file)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					print_comma
					print_expression (call_operands.first)
					current_file.put_character (')')
				end
				if l_name /= Void then
					current_file.put_character (')')
				end
				if l_same_scoop_region then
					current_file.put_character (')')
					current_file.put_character (':')
					current_file.put_string (c_eif_false)
					current_file.put_character (')')
				end
				if l_can_be_void then
					current_file.put_character (')')
					current_file.put_character (':')
					current_file.put_string (c_eif_false)
					current_file.put_character (')')
				end
			end
			call_operands.wipe_out
			if in_operand then
				current_file.put_character (';')
				current_file.put_new_line
				if l_temp /= Void then
					mark_temp_variable_unfrozen (l_temp)
					if l_temp_index /= 0 then
							-- We had to wait until this stage to set the index of `l_temp'
							-- because it could have still been used in `call_operands'.
						l_temp.set_index (l_temp_index)
					end
				end
			end
		end

	print_object_test_local (a_name: ET_IDENTIFIER)
			-- Print object-test local `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_object_test_local: a_name.is_object_test_local
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
		do
			if in_operand then
				operand_stack.force (a_name)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_name)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
						--
						-- The address of the object-test local is used, we have to make
						-- sure that the C optimizer will not implement it with 'register'.
					volatile_object_test_locals.force_last (current_object_test_locals.value (a_name.seed))
					current_file.put_character ('&')
					print_object_test_local_name (a_name, current_file)
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
				else
					print_object_test_local_name (a_name, current_file)
				end
			else
				print_object_test_local_name (a_name, current_file)
			end
		end

	print_octal_integer_constant (a_constant: ET_OCTAL_INTEGER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		do
			print_integer_constant (a_constant)
		end

	print_old_expression (an_expression: ET_OLD_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_temp: ET_IDENTIFIER
			l_exception_temp: ET_IDENTIFIER
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
				old_expression_temp_variables.search (an_expression)
				old_expression_exception_temp_variables.search (an_expression)
				if not old_expression_temp_variables.found then
						-- Internal error: the value of the old expression should have been
						-- computed at this stage.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_old_expression", 1, "no value computed for old expression.")
				elseif not old_expression_exception_temp_variables.found then
						-- Internal error: the exception marker of the old expression should have been
						-- set at this stage.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_old_expression", 2, "no exception marker for old expression.")
				else
					l_temp := old_expression_temp_variables.found_item
					l_exception_temp := old_expression_exception_temp_variables.found_item
					current_file.put_character ('(')
					print_temp_name (l_exception_temp, current_file)
					current_file.put_character ('?')
					current_file.put_character ('(')
					current_file.put_string (c_ge_raise_old_exception)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					print_comma
					print_temp_name (l_exception_temp, current_file)
					current_file.put_character (')')
					current_file.put_character (',')
					print_temp_name (l_temp, current_file)
					current_file.put_character (')')
					current_file.put_character (':')
					print_temp_name (l_temp, current_file)
					current_file.put_character (')')
				end
			end
		end

	print_once_manifest_string (an_expression: ET_ONCE_MANIFEST_STRING)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
				l_dynamic_type := dynamic_type_set (an_expression).static_type.primary_type
				inline_constants.force_last (l_dynamic_type, an_expression)
				print_inline_constant_name (an_expression, current_file)
			end
		end

	print_operand (an_operand: ET_EXPRESSION)
			-- Print `an_operand'.
		require
			an_operand_not_void: an_operand /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := True
			in_procedure_call_target := False
			call_target_type := Void
			an_operand.process (Current)
			call_target_type := old_target_type
			in_procedure_call_target := l_old_in_procedure_call_target
			in_operand := old_in_operand
		end

	print_parenthesized_expression (an_expression: ET_PARENTHESIZED_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			an_expression.expression.process (Current)
		end

	print_precursor_expression (an_expression: ET_PRECURSOR_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_precursor_keyword: ET_PRECURSOR_KEYWORD
			l_query: detachable ET_QUERY
			l_parent_type, l_ancestor: detachable ET_BASE_TYPE
			l_class: ET_CLASS
			l_current_class: ET_CLASS
			l_class_impl: ET_CLASS
			l_dynamic_precursor: ET_DYNAMIC_PRECURSOR
			l_parent_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			had_error: BOOLEAN
			l_old_in_qualified_call: BOOLEAN
		do
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := False
			l_parent_type := an_expression.parent_type
			if l_parent_type = Void then
					-- Internal error: the Precursor construct should already
					-- have been resolved when flattening the features of the
					-- implementation class of current feature.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_precursor_expression", 1, "precursor with no parent type.")
			else
				had_error := has_fatal_error
				has_fatal_error := False
				if l_parent_type.is_generic then
					l_current_class := current_type.base_class
					l_class_impl := current_feature.static_feature.implementation_class
					if l_current_class /= l_class_impl then
							-- Resolve generic parameters in the context of `current_type'.
						l_current_class.process (system_processor.ancestor_builder)
						if not l_current_class.ancestors_built or else l_current_class.has_ancestors_error then
								-- 'ancestor_builder' should already have been executed
								-- on `l_current_class' at this stage, and any error
								-- should have already been reported.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_precursor_expression", 2, "ancestors not built.")
						else
							l_ancestor := l_current_class.ancestor (l_parent_type)
							if l_ancestor = Void then
									-- Internal error: `l_parent_type' is an ancestor
									-- of `l_class_impl', and hence of `l_current_class'.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_precursor_expression", 3, "parent type not found.")
							else
								l_parent_type := l_ancestor
							end
						end
					end
				end
				if not has_fatal_error then
					has_fatal_error := had_error
					l_precursor_keyword := an_expression.precursor_keyword
					l_class := l_parent_type.base_class
					l_query := l_class.seeded_query (l_precursor_keyword.seed)
					if l_query = Void then
							-- Internal error: the Precursor construct should
							-- already have been resolved when flattening the
							-- features of `l_class_impl'.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_precursor_expression", 4, "precursor query not found.")
					else
						l_parent_dynamic_type := current_dynamic_system.dynamic_primary_type (l_parent_type, current_type.base_type)
						l_dynamic_precursor := current_feature.dynamic_precursor (l_query, l_parent_dynamic_type, current_dynamic_system)
						if l_dynamic_precursor.is_constant_attribute or l_dynamic_precursor.is_unique_attribute then
							print_unqualified_constant_attribute_call_expression (an_expression, l_dynamic_precursor)
						else
							print_unqualified_function_call_expression (an_expression, l_dynamic_precursor)
						end
					end
				end
			end
			in_qualified_call := l_old_in_qualified_call
		end

	print_prefix_expression (an_expression: ET_PREFIX_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			print_qualified_call_expression (an_expression)
		end

	print_procedure_target_expression (an_expression: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print `an_expression' when appearing as the target of a procedure call.
			-- `a_target_type' is one of the possible dynamic types of `an_expression'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_expression_not_void: an_expression /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			old_target_check_void: BOOLEAN
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			old_target_check_void := call_target_check_void
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := False
			call_target_type := a_target_type
			in_procedure_call_target := True
			call_target_check_void := a_check_void_target
			print_check_void_expression (an_expression, a_target_type, a_check_void_target)
			call_target_check_void := old_target_check_void
			in_procedure_call_target := l_old_in_procedure_call_target
			call_target_type := old_target_type
			in_operand := old_in_operand
		end

	print_procedure_target_operand (an_operand: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print `an_operand' when appearing as the target of a procedure call.
			-- `a_target_type' is one of the possible dynamic types of `an_operand'.
		require
			an_operand_not_void: an_operand /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := True
			call_target_type := a_target_type
			in_procedure_call_target := True
			an_operand.process (Current)
			in_procedure_call_target := l_old_in_procedure_call_target
			call_target_type := old_target_type
			in_operand := old_in_operand
		end

	print_qualified_call_expression (a_call: ET_QUALIFIED_FEATURE_CALL_EXPRESSION)
			-- Print qualified call expression.
		require
			a_call_not_void: a_call /= Void
		local
			l_name: ET_CALL_NAME
			l_target: ET_EXPRESSION
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_actuals: detachable ET_ACTUAL_ARGUMENTS
			l_seed: INTEGER
			i, nb: INTEGER
			l_target_type: ET_DYNAMIC_TYPE
			l_target_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_static_type: ET_DYNAMIC_PRIMARY_TYPE
			nb2: INTEGER
			l_call_type_set: ET_DYNAMIC_TYPE_SET
			l_call_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
			l_printed: BOOLEAN
			l_manifest_tuple_operand: detachable ET_MANIFEST_TUPLE
			l_target_operand: ET_EXPRESSION
			l_force_result_boxing: BOOLEAN
			l_dynamic_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_actual: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_is_separate_call: BOOLEAN
			l_old_max_nested_inlining_count: INTEGER
			l_inlined_call: BOOLEAN
			l_is_basic_attribute: BOOLEAN
			l_old_in_qualified_call: BOOLEAN
		do
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := True
			l_assignment_target := assignment_target
			assignment_target := Void
			l_target := a_call.target
			l_name := a_call.name
			l_actuals := a_call.arguments
			l_target_type_set := dynamic_type_set (l_target)
			l_target_type := l_target_type_set.static_type
			l_target_static_type := l_target_type.primary_type
				-- Note that `l_dynamic_call' may be Void at this stage because in the
				-- implementation of the built-in feature 'ANY.standard_copy' we create
				-- non-polymorphic calls which have no associated dynamic calls.
			l_dynamic_call := current_feature.query_call (a_call)
			l_call_type_set := dynamic_type_set (a_call)
			l_call_type := l_call_type_set.static_type.primary_type
			if
				in_call_target and then
				l_call_type.is_expanded and then
				(not l_call_type.is_basic or else in_procedure_call_target) and then
				(l_dynamic_call = Void or else l_dynamic_call.has_field_access (current_dynamic_system))
			then
					-- We want the result to be boxed even though the result static type is expanded.
					-- This is needed when the call is an attribute, tuple label, "SPECIAL" item or
					-- a once function, the call result type is expanded, and it is the target of
					-- another call. In that case we force the result type to be of reference type in
					-- order to force the boxing of the result so that the other call is applied on
					-- the result object itself and not on a clone of this object.
					-- Note that for optimization we don't force boxing when the type is a basic
					-- type unless the result object is the target of a procedure call (e.g.
					-- 'copy', 'standard_copy' or 'set_item' which will overwrite the value of
					-- the target).
				l_call_type := current_dynamic_system.any_type
				l_force_result_boxing := True
			end
			l_seed := l_name.seed
				-- Check whether this a call to FUNCTION.item with a manifest
				-- tuple as argument. We have a special treatment in that case
				-- to avoid having to create the manifest tuple when possible.
			if
				not (use_scoop and l_target_type.is_separate) and then
				not a_call.is_tuple_label and then
				l_seed = current_system.function_item_seed and then
				not a_call.is_call_agent
			then
				if l_actuals /= Void and then l_actuals.count = 1 then
					if attached {ET_MANIFEST_TUPLE} l_actuals.actual_argument (1) as l_manifest_tuple then
						print_target_operand (l_target, l_target_static_type)
						nb := l_manifest_tuple.count
						from i := 1 until i > nb loop
							l_actual := l_manifest_tuple.expression (i)
							l_actual_type_set := dynamic_type_set (l_actual)
							print_attachment_operand (l_actual, l_actual_type_set)
							i := i + 1
						end
						nb := nb + 1
						fill_call_operands (nb)
						l_manifest_tuple_operand := agent_manifest_tuple
						l_manifest_tuple_operand.resize (nb - 1)
						from i := nb until i < 2 loop
							l_manifest_tuple_operand.put_first (call_operands.item (i))
							i := i - 1
						end
						l_manifest_tuple_operand.set_index (l_manifest_tuple.index)
						mark_expressions_frozen (l_manifest_tuple_operand)
						l_target_operand := call_operands.first
						call_operands.wipe_out
						call_operands.put_last (l_target_operand)
						call_operands.put_last (l_manifest_tuple_operand)
						nb := 2
					end
				end
			end
			if l_manifest_tuple_operand = Void then
				print_target_operand (l_target, l_target_static_type)
				if l_actuals /= Void then
					nb := l_actuals.count
					from i := 1 until i > nb loop
						l_actual := l_actuals.actual_argument (i)
						l_actual_type_set := dynamic_type_set (l_actual)
						print_attachment_operand (l_actual, l_actual_type_set)
						i := i + 1
					end
				end
				nb := nb + 1
				fill_call_operands (nb)
			end
			l_is_separate_call := use_scoop and a_call /= separate_call_expression and l_target_type.is_separate
			nb2 := l_target_type_set.count
			if nb2 = 1 and not l_is_separate_call and not l_name.is_tuple_label then
					-- We can inline the call only if it's not a separate call,
					-- it's not a tuple access, and the call is not polymorphic.
				l_target_dynamic_type := l_target_type_set.dynamic_type (1)
				if attached l_target_dynamic_type.seeded_dynamic_query (l_name.seed, current_dynamic_system) as l_dynamic_feature then
					if l_target_static_type.is_basic and l_dynamic_feature.is_attribute then
							-- Built-in feature 'item' in basic type such as INTEGER.
							-- The result is the object itself (i.e. the target of the
							-- call) which is already in `call_operands'.
						l_inlined_call := True
						l_is_basic_attribute := True
					elseif is_inlinable_function (l_dynamic_feature, l_target_dynamic_type) then
						print_adapted_query_call (l_dynamic_feature, l_target_dynamic_type, l_call_type, True)
						fill_call_operands (1)
						l_inlined_call := True
					end
					if l_inlined_call then
						if in_operand then
							if l_force_result_boxing then
								l_printed := False
							elseif attached {ET_IDENTIFIER} call_operands.first as l_identifier then
								if l_identifier.is_temporary and then is_temp_variable_free (l_identifier) then
									l_temp := l_identifier
									mark_temp_variable_used (l_temp)
									mark_temp_variable_frozen (l_temp)
									l_temp_index := a_call.index
								end
								operand_stack.force (l_identifier)
								l_printed := True
							elseif l_is_basic_attribute then
								operand_stack.force (call_operands.first)
								l_printed := True
							end
						else
							print_attachment_expression (call_operands.first, l_call_type_set, l_call_type)
							l_printed := True
						end
					end
				end
			end
			if not l_printed then
				if in_operand then
					print_indentation
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_call_type)
						mark_temp_variable_frozen (l_temp)
							-- We will set the index of `l_temp' later because
							-- it could still be used in `call_operands'.
						l_temp_index := a_call.index
						if l_force_result_boxing then
								-- `current_dynamic_system.any_type' is the first type set
								-- in `extra_dynamic_type_sets'.
							l_temp_index := current_dynamic_type_sets.count + 1
						end
						operand_stack.force (l_temp)
						print_temp_name (l_temp, current_file)
					end
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('(')
				end
				if l_is_separate_call then
					print_separate_qualified_call_expression (a_call)
				elseif l_inlined_call then
					print_attachment_expression (call_operands.first, l_call_type_set, l_call_type)
				elseif nb2 = 0 then
						-- Call on Void target.
					current_file.put_character ('(')
					print_target_expression (call_operands.first, l_target_static_type, True)
					from i := 2 until i > nb loop
						current_file.put_character (',')
						current_file.put_character (' ')
						print_expression (call_operands.item (i))
						i := i + 1
					end
					current_file.put_character (',')
					current_file.put_character (' ')
					print_typed_default_entity_value (l_call_type, current_file)
					current_file.put_character (')')
				elseif nb2 = 1 then
						-- Static binding.
					l_target_dynamic_type := l_target_type_set.dynamic_type (1)
					print_adapted_named_query_call (l_name, l_target_dynamic_type, l_call_type, True)
				else
						-- Dynamic binding.
					if l_name.is_tuple_label then
						-- Fields in Tuples are not necessarily aligned, and some need (un)boxing,
						-- for example:
						--    t1: TUPLE [a: ANY; b: INTEGER]
						--    t2: TUPLE [a: CHARACTER; b: INTEGER]
						--    t1 := t2
						--    any := t1.a
						--    int := t1.b
						-- TODO (optimization): But when there are aligned with no (un)boxing,
						-- we could avoid having polymorphic calls.
					elseif not attached l_target_static_type.base_class.seeded_query (l_seed) as l_query then
							-- Internal error: there should be a query with `a_seed'.
							-- It has been computed in ET_FEATURE_CHECKER.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_qualified_call_expression", 1, "query not found.")
						l_printed := True
					elseif l_query.is_constant_attribute then
						l_target_dynamic_type := l_target_type_set.dynamic_type (1)
						print_adapted_named_query_call (l_name, l_target_dynamic_type, l_call_type, True)
						l_printed := True
					elseif l_query.is_unique_attribute then
						l_target_dynamic_type := l_target_type_set.dynamic_type (1)
						print_adapted_named_query_call (l_name, l_target_dynamic_type, l_call_type, True)
						l_printed := True
					end
					if l_printed then
						-- Do nothing.
					elseif nb2 = 2 then
							-- No inlining here.
						l_old_max_nested_inlining_count := max_nested_inlining_count
						max_nested_inlining_count := 0
							-- First type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (1)
						current_file.put_character ('(')
						current_file.put_character ('(')
						print_attribute_type_id_access (call_operands.first, l_target_static_type, True)
						current_file.put_character ('=')
						current_file.put_character ('=')
						current_file.put_integer (l_target_dynamic_type.id)
						current_file.put_character (')')
						current_file.put_character ('?')
						print_adapted_named_query_call (l_name, l_target_dynamic_type, l_call_type, False)
						current_file.put_character (':')
							-- Second type.
						l_target_dynamic_type := l_target_type_set.dynamic_type (2)
						print_adapted_named_query_call (l_name, l_target_dynamic_type, l_call_type, False)
						current_file.put_character (')')
						max_nested_inlining_count := l_old_max_nested_inlining_count
					elseif l_dynamic_call = Void then
							-- Internal error: the dynamic call should have been created in ET_DYNAMIC_TYPE_BUILDER.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_qualified_call_expression", 2, "dynamic call not found.")
					else
						l_dynamic_call.set_force_result_boxing (l_force_result_boxing)
						register_polymorphic_called_features (l_dynamic_call)
						print_call_name (l_dynamic_call, l_target_static_type, l_force_result_boxing, current_file)
						current_file.put_character ('(')
						current_file.put_string (c_ac)
						current_file.put_character (',')
						current_file.put_character (' ')
						print_target_expression (call_operands.first, l_target_static_type, True)
						if l_manifest_tuple_operand /= Void then
							nb := l_manifest_tuple_operand.count
							from i := 1 until i > nb loop
								current_file.put_character (',')
								current_file.put_character (' ')
								print_expression (l_manifest_tuple_operand.expression (i))
								i := i + 1
							end
						else
							from i := 2 until i > nb loop
								current_file.put_character (',')
								current_file.put_character (' ')
								print_expression (call_operands.item (i))
								i := i + 1
							end
						end
						current_file.put_character (')')
					end
				end
				if in_operand then
					current_file.put_character (')')
					print_semicolon_newline
				end
			end
			call_operands.wipe_out
			if l_manifest_tuple_operand /= Void then
				mark_expressions_unfrozen (l_manifest_tuple_operand)
				l_manifest_tuple_operand.wipe_out
			end
			if l_temp /= Void then
				mark_temp_variable_unfrozen (l_temp)
				if l_temp_index /= 0 then
						-- We had to wait until this stage to set the index of `l_temp'
						-- because it could have still been used in `call_operands'.
					l_temp.set_index (l_temp_index)
				end
			end
			in_qualified_call := l_old_in_qualified_call
			if l_is_separate_call then
					-- Print the separate call declaration now, otherwise some
					-- temporary data (temporary dynamic type sets, reused calls,
					-- etc.) may not be available anymore.
				print_separate_call_declaration (current_separate_call_count, a_call, current_closure)
			end
		end

	print_separate_qualified_call_expression (a_call: ET_QUALIFIED_FEATURE_CALL_EXPRESSION)
			-- Print qualified call expression whose target type is separate.
			-- Operands can be found in `call_operands'.
		require
			a_call_not_void: a_call /= Void
			call_operands_not_empty: not call_operands.is_empty
			use_scoop: use_scoop
		local
			i, nb: INTEGER
			l_separate_arguments: like separate_arguments
			l_separate_argument: ET_IDENTIFIER
		do
			current_separate_call_count := current_separate_call_count + 1
			print_separate_call_name (current_separate_call_count, current_feature, current_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			l_separate_arguments := separate_arguments
			l_separate_arguments.wipe_out
			a_call.target.add_separate_arguments (l_separate_arguments, current_closure)
			nb := l_separate_arguments.count
			from i := 1 until i > nb loop
				l_separate_argument := l_separate_arguments.item (i)
				if
					l_separate_argument.is_inline_separate_argument or
					l_separate_argument.is_argument and then dynamic_type_set (l_separate_argument).static_type.is_separate
				then
					print_comma
					print_separate_argument_session_name (l_separate_argument, current_file)
				end
				i := i + 1
			end
			l_separate_arguments.wipe_out
			nb := call_operands.count
			from i := 1 until i > nb loop
				print_comma
				print_expression (call_operands.item (i))
				i := i + 1
			end
			current_file.put_character (')')
		end

	print_quantifier_expression (a_expression: ET_QUANTIFIER_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		do
			print_iteration_expression (a_expression)
		end

	print_regular_integer_constant (a_constant: ET_REGULAR_INTEGER_CONSTANT)
			-- Check validity of `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		do
			print_integer_constant (a_constant)
		end

	print_regular_manifest_string (a_string: ET_REGULAR_MANIFEST_STRING)
			-- Print `a_string'.
		require
			a_string_not_void: a_string /= Void
		do
			print_manifest_string (a_string)
		end

	print_regular_real_constant (a_constant: ET_REGULAR_REAL_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			l_temp: ET_IDENTIFIER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				if in_call_target then
					in_operand := False
					l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
					l_temp := new_temp_variable (l_dynamic_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_regular_real_constant (a_constant)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
					in_operand := True
				else
					operand_stack.force (a_constant)
				end
			else
				l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
				print_type_cast (l_dynamic_type, current_file)
				current_file.put_character ('(')
				if a_constant.is_negative then
					current_file.put_character ('-')
				end
				current_file.put_string (a_constant.literal)
				current_file.put_character (')')
			end
		end

	print_result (an_expression: ET_RESULT)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_once_kind: INTEGER
			l_once_index: INTEGER
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
				if attached call_target_type as l_call_target_type then
					check in_call_target: in_call_target end
					if has_rescue then
							-- Keep track of the fact that the value of the 'Result' entity has
							-- been read. Useful to determine the 'volatile' status of the 'Result'
							-- entity when current feature has a rescue clause.
						result_read := True
					end
					l_dynamic_type_set := dynamic_type_set (an_expression)
					l_static_type := l_dynamic_type_set.static_type.primary_type
					if l_static_type.is_expanded then
							-- Pass the address of the expanded object.
							--
							-- The address of the 'Result' entity is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						volatile_result := True
						current_file.put_character ('(')
						if current_agent = Void and then current_feature.is_once then
								-- Make sure that if a once-function is called
								-- recursively, the semantics specified by ECMA will be satisfied.
								-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
								-- to the once-function should return the value of 'Result' as it
								-- was when the recursive calls occurred.
								-- There is a special treatment for once-per-process and once-per-thread
								-- functions of non-basic expanded types because a boxed version of the
								-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
								-- So we need to use the content of the boxed value in place of Result
								-- in case its fields get modified. Likewise for once-per-object functions
								-- even though the Result is not boxed in that case.
							if not l_static_type.is_basic then
								if current_feature.is_once_per_object then
									current_file.put_character ('&')
									print_once_value (current_feature, 0, 0)
								else
									once_features.search (current_feature.static_feature.implementation_feature)
									if once_features.found then
										l_once_index := once_features.found_item
										l_once_kind := once_kind (current_feature)
									end
										-- The once value has been boxed.
										-- It needs to be unboxed.
									current_file.put_character ('(')
									current_file.put_character ('(')
									print_boxed_type_cast (l_static_type, current_file)
									current_file.put_character ('(')
									print_once_value (current_feature, l_once_kind, l_once_index)
									current_file.put_character (')')
									current_file.put_character (')')
									current_file.put_string (c_arrow)
									print_boxed_attribute_pointer_name (l_static_type, current_file)
									current_file.put_character (')')
								end
							else
								current_file.put_character ('&')
								print_result_name (current_file)
							end
						else
							current_file.put_character ('&')
							print_result_name (current_file)
						end
						current_file.put_character (')')
					elseif l_call_target_type.is_expanded then
							-- We need to unbox the object and then pass its address.
						print_boxed_attribute_pointer_access (an_expression, l_call_target_type, call_target_check_void)
					else
						print_result_name (current_file)
					end
				elseif current_agent = Void and then current_feature.is_once then
						-- Make sure that if a once-function is called
						-- recursively, the semantics specified by ECMA will be satisfied.
						-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
						-- to the once-function should return the value of 'Result' as it
						-- was when the recursive calls occurred.
						-- There is a special treatment for once-per-process and once-per-thread
						-- functions of non-basic expanded types because a boxed version of the
						-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
						-- So we need to use the content of the boxed value in place of Result
						-- in case its fields get modified. Likewise for once-per-object functions
						-- even though the Result is not boxed in that case.
					l_dynamic_type_set := dynamic_type_set (an_expression)
					l_static_type := l_dynamic_type_set.static_type.primary_type
					if l_static_type.is_expanded and then not l_static_type.is_basic then
						if current_feature.is_once_per_object then
							print_once_value (current_feature, 0, 0)
						else
							once_features.search (current_feature.static_feature.implementation_feature)
							if once_features.found then
								l_once_index := once_features.found_item
								l_once_kind := once_kind (current_feature)
							end
								-- The once value has been boxed.
								-- It needs to be unboxed.
							current_file.put_character ('(')
							print_boxed_type_cast (l_static_type, current_file)
							current_file.put_character ('(')
							print_once_value (current_feature, l_once_kind, l_once_index)
							current_file.put_character (')')
							current_file.put_character (')')
							current_file.put_string (c_arrow)
							print_boxed_attribute_item_name (l_static_type, current_file)
						end
					else
						print_result_name (current_file)
					end
				else
					print_result_name (current_file)
				end
			end
		end

	print_result_address (an_expression: ET_RESULT_ADDRESS)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_queries: ET_DYNAMIC_FEATURE_LIST
			l_result_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_temp: ET_IDENTIFIER
			l_pointer: BOOLEAN
			l_pointer_type: ET_DYNAMIC_PRIMARY_TYPE
			l_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_once_kind: INTEGER
			l_once_index: INTEGER
		do
			l_pointer_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.pointer_type, current_type.base_type)
			if has_rescue then
					-- Keep track of the fact that the value of the 'Result' entity can
					-- possibly be modified and read. Useful to determine the 'volatile'
					-- status of the 'Result' entity when current feature has a rescue clause.
				result_read := True
				result_written := True
			end
				-- The address of the 'Result' entity is used, we have to make
				-- sure that the C optimizer will not implement it with 'register'.
			volatile_result := True
			l_dynamic_type_set := dynamic_type_set (an_expression)
			l_dynamic_type := l_dynamic_type_set.static_type.primary_type
			l_pointer := (l_dynamic_type = l_pointer_type)
			if not l_pointer then
					-- $Result is of type TYPED_POINTER.
				l_temp := new_temp_variable (l_dynamic_type)
				l_temp.set_index (an_expression.index)
				operand_stack.force (l_temp)
				l_queries := l_dynamic_type.queries
				if l_queries.is_empty then
						-- Internal error: TYPED_POINTER should have an attribute
						-- `pointer_item' at first position.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_result_address", 1, "'TYPED_POINTER' has no attribute `pointer_item'.")
				else
					print_indentation
					print_attribute_access (l_queries.first, l_temp, l_dynamic_type, False)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
				end
			end
			if l_pointer and in_operand then
					-- $Result is of type POINTER.
				operand_stack.force (an_expression)
			else
				current_file.put_character ('(')
				l_result_type_set := current_feature.result_type_set
				if l_result_type_set = Void then
						-- Internal error: it should have been checked elsewhere that
						-- the current feature is a function.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_result_address", 2, "enclosing feature is not a function.")
				elseif l_result_type_set.is_expanded then
					print_type_cast (l_pointer_type, current_file)
					if current_agent = Void and then current_feature.is_once then
							-- Make sure that if a once-function is called
							-- recursively, the semantics specified by ECMA will be satisfied.
							-- ECMA 367-2 section 8.23.22 page 124 says that the recursive calls
							-- to the once-function should return the value of 'Result' as it
							-- was when the recursive calls occurred.
							-- There is a special treatment for once-per-process and once-per-thread
							-- functions of non-basic expanded types because a boxed version of the
							-- Result (and not the Result itself) is kept in the 'GE_onces' C struct.
							-- So we need to use the content of the boxed value in place of Result
							-- in case its fields get modified. Likewise for once-per-object functions
							-- even though the Result is not boxed in that case.
						l_static_type := l_result_type_set.static_type.primary_type
						if not l_static_type.is_basic then
							if current_feature.is_once_per_object then
								current_file.put_character ('&')
								print_once_value (current_feature, 0, 0)
							else
								once_features.search (current_feature.static_feature.implementation_feature)
								if once_features.found then
									l_once_index := once_features.found_item
									l_once_kind := once_kind (current_feature)
								end									-- The once value has been boxed.
									-- It needs to be unboxed.
								current_file.put_character ('(')
								print_boxed_type_cast (l_static_type, current_file)
								current_file.put_character ('(')
								print_once_value (current_feature, l_once_kind, l_once_index)
								current_file.put_character (')')
								current_file.put_character (')')
								current_file.put_string (c_arrow)
								print_boxed_attribute_pointer_name (l_static_type, current_file)
							end
						else
							current_file.put_character ('&')
							print_result_name (current_file)
						end
					else
						current_file.put_character ('&')
						print_result_name (current_file)
					end
				elseif attached l_result_type_set.special_type as l_special_type then
					current_file.put_character ('(')
					print_result_name (current_file)
					current_file.put_character ('?')
					print_type_cast (l_pointer_type, current_file)
					current_file.put_character ('(')
					current_file.put_string (c_ge_types)
					current_file.put_character ('[')
					print_attribute_type_id_access (tokens.result_keyword, l_result_type_set.static_type.primary_type, False)
					current_file.put_character (']')
					current_file.put_character ('[')
					current_file.put_character ('0')
					current_file.put_character (']')
					current_file.put_character ('.')
					current_file.put_string (c_is_special)
					current_file.put_character ('?')
					print_type_cast (l_pointer_type, current_file)
					print_attribute_special_item_access (an_expression.result_keyword, l_special_type, False)
					current_file.put_character (':')
					print_type_cast (l_pointer_type, current_file)
					print_result_name (current_file)
					current_file.put_character (')')
					current_file.put_character (':')
					print_type_cast (l_pointer_type, current_file)
					current_file.put_character ('0')
					current_file.put_character (')')
				else
					print_type_cast (l_pointer_type, current_file)
					print_result_name (current_file)
				end
				current_file.put_character (')')
			end
			if not l_pointer then
				current_file.put_character (';')
				current_file.put_new_line
			end
		end

	print_special_manifest_string (a_string: ET_SPECIAL_MANIFEST_STRING)
			-- Print `a_string'.
		require
			a_string_not_void: a_string /= Void
		do
			print_manifest_string (a_string)
		end

	print_static_call_expression (a_expression: ET_STATIC_CALL_EXPRESSION)
			-- Print `a_expression'.
		require
			a_expression_not_void: a_expression /= Void
		local
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_feature: ET_DYNAMIC_FEATURE
			l_old_in_qualified_call: BOOLEAN
		do
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := True
			l_target_type := current_dynamic_system.dynamic_primary_type (a_expression.type, current_type.base_type)
			if not attached l_target_type.base_class.seeded_query (a_expression.name.seed) as l_query then
					-- Internal error: there should be a query with `l_seed'.
					-- It has been computed in ET_FEATURE_CHECKER or else an
					-- error should have already been reported.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_static_call_expression", 1, "query not found.")
			elseif l_query.is_attribute then
					-- Internal error: no object available.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_static_call_expression", 2, "query is an attribute.")
			elseif l_query.is_constant_attribute or l_query.is_unique_attribute then
				l_dynamic_feature := l_target_type.dynamic_query (l_query, current_dynamic_system)
				check l_dynamic_feature_valid: l_dynamic_feature.is_constant_attribute or l_dynamic_feature.is_unique_attribute end
				print_unqualified_constant_attribute_call_expression (a_expression, l_dynamic_feature)
			else
				l_dynamic_feature := l_target_type.dynamic_query (l_query, current_dynamic_system)
				print_unqualified_function_call_expression (a_expression, l_dynamic_feature)
			end
			in_qualified_call := l_old_in_qualified_call
		end

	print_strip_expression (an_expression: ET_STRIP_EXPRESSION)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
-- TODO.
error_handler.report_warning_message ("ET_C_GENERATOR.print_strip_expression")
				current_file.put_string (c_eif_void)
			end
		end

	print_target_expression (an_expression: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print `an_expression' when appearing as the target of a call.
			-- `a_target_type' is one of the possible dynamic types of `an_expression'.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
		require
			an_expression_not_void: an_expression /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			old_target_check_void: BOOLEAN
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			old_target_check_void := call_target_check_void
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := False
			call_target_type := a_target_type
			in_procedure_call_target := False
			call_target_check_void := a_check_void_target
			print_check_void_expression (an_expression, a_target_type, a_check_void_target)
			call_target_check_void := old_target_check_void
			in_procedure_call_target := l_old_in_procedure_call_target
			call_target_type := old_target_type
			in_operand := old_in_operand
		end

	print_target_operand (an_operand: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print `an_operand' when appearing as the target of a call.
			-- `a_target_type' is one of the possible dynamic types of `an_operand'.
		require
			an_operand_not_void: an_operand /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := True
			call_target_type := a_target_type
			in_procedure_call_target := False
			an_operand.process (Current)
			in_procedure_call_target := l_old_in_procedure_call_target
			call_target_type := old_target_type
			in_operand := old_in_operand
		end

	print_temporary_variable (a_name: ET_IDENTIFIER)
			-- Print temporary variable `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_temporary: a_name.is_temporary
		do
			if in_operand then
				operand_stack.force (a_name)
			else
				if attached call_target_type as l_call_target_type then
					check in_call_target: in_call_target end
					if not is_temp_variable_known (a_name) then
							-- Internal error: the temporary
							-- variable should be known at this stage.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_temporary_variable", 1, "unknown temporary variable.")
					elseif not attached temp_variable_type (a_name) as l_static_type then
							-- Internal error: the type of the temporary
							-- variable should be known at this stage.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_temporary_variable", 2, "missing type for temporary variable.")
					elseif l_static_type.is_expanded then
							-- Pass the address of the expanded object.
							--
							-- The address of the temporary variable is used, we have to make
							-- sure that the C optimizer will not implement it with 'register'.
						mark_temp_variable_volatile (a_name)
						current_file.put_character ('&')
						print_temp_name (a_name, current_file)
					elseif l_call_target_type.is_expanded then
							-- We need to unbox the object and then pass its address.
						print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
					else
						print_temp_name (a_name, current_file)
					end
				else
					print_temp_name (a_name, current_file)
				end
			end
		end

	print_true_constant (a_constant: ET_TRUE_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			l_temp: ET_IDENTIFIER
		do
			if in_operand then
				if in_call_target then
					l_temp := new_temp_variable (dynamic_type_set (a_constant).static_type.primary_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
				else
					operand_stack.force (a_constant)
				end
			else
				current_file.put_string (c_eif_true)
			end
		end

	print_unboxed_expression (an_expression: ET_EXPRESSION; a_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void: BOOLEAN)
			-- Print to `current_file' unboxed version of `an_expression' of type `a_type'.
			-- There is no assumption about static type (i.e. declared type) of
			-- `an_expression' or about `a_type': they may be reference or expanded.
			-- (The boxed version of a type makes sure that each object
			-- of that type contains its type-id. It can be the type itself
			-- if it already contains its type-id, or a wrapper otherwise.)
			-- `a_check_void' means that we need to check whether the expression is Void or not.
		require
			an_expression_not_void: an_expression /= Void
			a_type_not_void: a_type /= Void
		local
			l_static_type: ET_DYNAMIC_TYPE
		do
			if a_type.is_expanded then
				l_static_type := dynamic_type_set (an_expression).static_type
				if l_static_type.is_expanded then
					print_expression (an_expression)
				else
						-- We need to unbox the object.
					current_file.put_character ('*')
					print_boxed_attribute_pointer_access (an_expression, a_type, a_check_void)
				end
			else
				print_expression (an_expression)
			end
		end

	print_underscored_integer_constant (a_constant: ET_UNDERSCORED_INTEGER_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		do
			print_integer_constant (a_constant)
		end

	print_underscored_real_constant (a_constant: ET_UNDERSCORED_REAL_CONSTANT)
			-- Print `a_constant'.
		require
			a_constant_not_void: a_constant /= Void
		local
			i, nb: INTEGER
			l_literal: STRING
			l_temp: ET_IDENTIFIER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if in_operand then
				if in_call_target then
					in_operand := False
					l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
					l_temp := new_temp_variable (l_dynamic_type)
					print_indentation
					print_temp_name (l_temp, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					print_underscored_real_constant (a_constant)
					current_file.put_character (';')
					current_file.put_new_line
					l_temp.set_index (a_constant.index)
					operand_stack.force (l_temp)
					in_operand := True
				else
					operand_stack.force (a_constant)
				end
			else
				l_dynamic_type := dynamic_type_set (a_constant).static_type.primary_type
				print_type_cast (l_dynamic_type, current_file)
				current_file.put_character ('(')
				if a_constant.is_negative then
					current_file.put_character ('-')
				end
				l_literal := a_constant.literal
				nb := l_literal.count
				from i := 1 until i > nb loop
					if l_literal.item (i) /= '_' then
						current_file.put_character (l_literal.item (i))
					end
					i := i + 1
				end
				current_file.put_character (')')
			end
		end

	print_unqualified_call_expression (a_call: ET_UNQUALIFIED_FEATURE_CALL_EXPRESSION)
			-- Print unqualified call expression.
		require
			a_call_not_void: a_call /= Void
		local
			l_old_in_qualified_call: BOOLEAN
		do
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := False
			if not attached current_type.seeded_dynamic_query (a_call.name.seed, current_dynamic_system) as l_dynamic_feature then
					-- Internal error: there should be a query for that seed.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_unqualified_call_expression", 1, "query not found.")
			elseif l_dynamic_feature.is_attribute_with_no_self_initializing_code then
				print_unqualified_attribute_call_expression (a_call, l_dynamic_feature)
			elseif l_dynamic_feature.is_constant_attribute or l_dynamic_feature.is_unique_attribute then
				print_unqualified_constant_attribute_call_expression (a_call, l_dynamic_feature)
			else
				print_unqualified_function_call_expression (a_call, l_dynamic_feature)
			end
			in_qualified_call := l_old_in_qualified_call
		end

	print_unqualified_attribute_call_expression (a_call: ET_UNQUALIFIED_FEATURE_CALL_EXPRESSION; a_feature: ET_DYNAMIC_FEATURE)
			-- Print expression `a_call' which is an unqualified call to the
			-- attribute `a_feature' (static binding).
		require
			a_call_not_void: a_call /= Void
			a_feature_not_void: a_feature /= void
			a_feature_is_attribute: a_feature.is_attribute_with_no_self_initializing_code
		local
			l_call_type: ET_DYNAMIC_PRIMARY_TYPE
			l_old_call_target: detachable ET_EXPRESSION
		do
			assignment_target := Void
			l_call_type := dynamic_type_set (a_call).static_type.primary_type
			if in_operand then
				operand_stack.force (a_call)
			elseif a_feature.is_builtin then
					-- This is a built-in attribute (such as feature 'item' in 'INTEGER_REF').
					-- The call might be the operand (target or argument) of another call.
					-- Therefore we need to take some care when dealing with `call_operands'.
					-- We need to keep track of the operands of this possible other call
					-- while processing the current call to the built-in attribute, for
					-- which only the target ('Current') needs to be on `call_operands'.
				if call_operands.is_empty then
					call_operands.force_last (tokens.current_keyword)
				else
					l_old_call_target := call_operands.first
					call_operands.replace (tokens.current_keyword, 1)
				end
				if in_call_target and l_call_type.is_expanded then
						-- Pass the address of the built-in attribute expanded object.
					current_file.put_character ('&')
					current_file.put_character ('(')
						-- No need to call `print_adapted_query_call' here because an
						-- unqualified call is not polymorphic, and therefore the type
						-- of the query is exactly the type expected by the caller.
					print_query_call (a_feature, current_type, False)
					current_file.put_character (')')
				else
						-- No need to call `print_adapted_query_call' here because an
						-- unqualified call is not polymorphic, and therefore the type
						-- of the query is exactly the type expected by the caller.
					print_query_call (a_feature, current_type, False)
				end
				if l_old_call_target /= Void then
					call_operands.replace (l_old_call_target, 1)
				else
					call_operands.wipe_out
				end
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				if l_call_type.is_expanded then
						-- Pass the address of the expanded object.
					current_file.put_character ('&')
					current_file.put_character ('(')
					print_attribute_access (a_feature, tokens.current_keyword, current_type, False)
					current_file.put_character (')')
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					current_file.put_character ('&')
					current_file.put_character ('(')
					print_boxed_attribute_item_access (a_call, l_call_target_type, call_target_check_void)
					current_file.put_character (')')
				else
					print_attribute_access (a_feature, tokens.current_keyword, current_type, False)
				end
			else
				print_attribute_access (a_feature, tokens.current_keyword, current_type, False)
			end
		end

	print_unqualified_constant_attribute_call_expression (a_call: ET_CALL_EXPRESSION_WITH_ACTUAL_ARGUMENTS; a_feature: ET_DYNAMIC_FEATURE)
			-- Print expression `a_call' which is an unqualified or static (non-object) call
			-- to the constant or unique attribute `a_feature' (static binding).
		require
			a_call_not_void: a_call /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_constant_attribute: a_feature.is_constant_attribute or a_feature.is_unique_attribute
		local
			l_call_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: detachable ET_IDENTIFIER
			l_old_call_target: detachable ET_EXPRESSION
		do
			assignment_target := Void
			if in_operand then
				if in_call_target then
					l_call_type := dynamic_type_set (a_call).static_type.primary_type
					if l_call_type.is_expanded then
						l_temp := new_temp_variable (l_call_type)
						mark_temp_variable_frozen (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
						print_assign_to
						in_operand := False
						print_unqualified_constant_attribute_call_expression (a_call, a_feature)
						in_operand := True
						print_semicolon_newline
						l_temp.set_index (a_call.index)
						operand_stack.force (l_temp)
						mark_temp_variable_unfrozen (l_temp)
					else
						operand_stack.force (a_call)
					end
				else
					operand_stack.force (a_call)
				end
			else
					-- The call might be the operand (target or argument) of another call.
					-- Therefore we need to take some care when dealing with `call_operands'.
					-- We need to keep track of the operands of this possible other call
					-- while processing the current call to the constantd or unique attribute,
					-- for which only the target ('Current') needs to be on `call_operands'.
				if call_operands.is_empty then
					call_operands.force_last (tokens.current_keyword)
				else
					l_old_call_target := call_operands.first
					call_operands.replace (tokens.current_keyword, 1)
				end
				print_query_call (a_feature, a_feature.target_type, False)
				if l_old_call_target /= Void then
					call_operands.replace (l_old_call_target, 1)
				else
					call_operands.wipe_out
				end
			end
		end

	print_unqualified_function_call_expression (a_call: ET_CALL_EXPRESSION_WITH_ACTUAL_ARGUMENTS; a_feature: ET_DYNAMIC_FEATURE)
			-- Print expression `a_call' which is an unqualified or static (non-object) call
			-- to the function `a_feature' (static binding).
		require
			a_call_not_void: a_call /= Void
			a_feature_not_void: a_feature /= Void
		local
			l_actuals: ET_ACTUAL_ARGUMENTS
			l_actual: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			i, nb: INTEGER
			l_call_type_set: ET_DYNAMIC_TYPE_SET
			l_call_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
			l_manifest_tuple_operand: detachable ET_MANIFEST_TUPLE
			l_target_operand: ET_EXPRESSION
			l_seed: INTEGER
			l_force_result_boxing: BOOLEAN
			l_inlined_call: BOOLEAN
			l_printed: BOOLEAN
			l_standard_is_equal: BOOLEAN
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_call_type_set := dynamic_type_set (a_call)
			l_call_type := l_call_type_set.static_type.primary_type
			if
				in_call_target and then
				(a_feature.is_builtin_special_item or a_feature.is_once) and then
				l_call_type.is_expanded and then
				(not l_call_type.is_basic or else in_procedure_call_target)
			then
					-- We want the result be boxed even though the result static type is expanded.
					-- This is needed when the call is a "SPECIAL" item or a once function, the call
					-- result type is expanded, and it is the target of another call. In that case we
					-- force the result type to be of reference type in order to force the boxing of
					-- the result so that the other call is applied on the result object itself and
					-- not on a copy of this object.
					-- Note that for optimization we don't force boxing when the type is a basic
					-- type unless the result object is the target of a procedure call (e.g.
					-- 'copy', 'standard_copy' or 'set_item' which will overwrite the value of
					-- the target).
				l_call_type := current_dynamic_system.any_type
				l_force_result_boxing := True
			end
			l_seed := a_call.name.seed
			l_actuals := a_call.arguments
				-- Check whether this a call to FUNCTION.item with a manifest
				-- tuple as argument. We have a special treatment in that case
				-- to avoid having to create the manifest tuple when possible.
			if l_seed = current_system.function_item_seed and then not a_call.is_call_agent then
				if l_actuals /= Void and then l_actuals.count = 1 and then attached {ET_MANIFEST_TUPLE} l_actuals.actual_argument (1) as l_manifest_tuple then
					operand_stack.force (tokens.current_keyword)
					nb := l_manifest_tuple.count
					from i := 1 until i > nb loop
						l_actual := l_manifest_tuple.expression (i)
						l_actual_type_set := dynamic_type_set (l_actual)
						print_attachment_operand (l_actual, l_actual_type_set)
						i := i + 1
					end
					nb := nb + 1
					fill_call_operands (nb)
					l_manifest_tuple_operand := agent_manifest_tuple
					l_manifest_tuple_operand.resize (nb - 1)
					from i := nb until i < 2 loop
						l_manifest_tuple_operand.put_first (call_operands.item (i))
						i := i - 1
					end
					l_manifest_tuple_operand.set_index (l_manifest_tuple.index)
					mark_expressions_frozen (l_manifest_tuple_operand)
					l_target_operand := call_operands.first
					call_operands.wipe_out
					call_operands.put_last (l_target_operand)
					call_operands.put_last (l_manifest_tuple_operand)
					nb := 2
				end
			end
			if l_manifest_tuple_operand = Void then
				l_standard_is_equal := a_feature.is_builtin_any_class and a_feature.builtin_feature_code = {ET_TOKEN_CODES}.builtin_any_standard_is_equal
				operand_stack.force (tokens.current_keyword)
				nb := 0
				if l_actuals /= Void then
					nb := l_actuals.count
					from i := 1 until i > nb loop
						l_actual := l_actuals.actual_argument (i)
						l_actual_type_set := dynamic_type_set (l_actual)
						if l_standard_is_equal then
								-- Do not clone expanded arguments of 'ANY.standard_is_equal'
								-- otherwise this might violate the invariant
								-- 'reflexive_equality: standard_is_equal (Current)'.
							print_operand (l_actual)
						else
							print_attachment_operand (l_actual, l_actual_type_set)
						end
						i := i + 1
					end
				end
				nb := nb + 1
				fill_call_operands (nb)
			end
			if is_inlinable_function (a_feature, a_feature.target_type) then
					-- When there is no result type of the form 'like argument',
					-- then there is no need to call `print_adapted_query_call'
					-- because an unqualified or static call is not polymorphic,
					-- and therefore the type of the query is exactly the type
					-- expected by the caller. This is another story with 'like argument'.
					-- For example 'clone (1)' is expected to be of type INTEGER
					-- and 'clone ("gobo")' is expected to be of type STRING.
					-- But the corresponding generated function for 'clone' will
					-- be declared to return an ANY. Hence the need for a call
					-- to `print_adapted_query_call' in this case.
					-- There is also the case when `l_force_result_boxing' is True.
				print_adapted_query_call (a_feature, a_feature.target_type, l_call_type, False)
				l_inlined_call := True
				fill_call_operands (1)
				if in_operand then
					if l_force_result_boxing then
						l_printed := False
					elseif attached {ET_IDENTIFIER} call_operands.first as l_identifier then
						if l_identifier.is_temporary and then is_temp_variable_free (l_identifier) then
							l_temp := l_identifier
							mark_temp_variable_used (l_temp)
							mark_temp_variable_frozen (l_temp)
							l_temp_index := a_call.index
						end
						operand_stack.force (l_identifier)
						l_printed := True
					end
				else
					print_attachment_expression (call_operands.first, l_call_type_set, l_call_type)
					l_printed := True
				end
			end
			if not l_printed then
				if in_operand then
					if l_assignment_target /= Void then
						operand_stack.force (l_assignment_target)
						print_indentation
						print_writable (l_assignment_target)
					else
						l_temp := new_temp_variable (l_call_type)
						mark_temp_variable_frozen (l_temp)
							-- We will set the index of `l_temp' later because
							-- it could still be used in `call_operands'.
						l_temp_index := a_call.index
						if l_force_result_boxing then
								-- `current_dynamic_system.any_type' is the first type set
								-- in `extra_dynamic_type_sets'.
							l_temp_index := current_dynamic_type_sets.count + 1
						end
						operand_stack.force (l_temp)
						print_indentation
						print_temp_name (l_temp, current_file)
					end
					print_assign_to
					current_file.put_character ('(')
				end
				if l_inlined_call then
					print_attachment_expression (call_operands.first, l_call_type_set, l_call_type)
				else
						-- When there is no result type of the form 'like argument',
						-- then there is no need to call `print_adapted_query_call'
						-- because an unqualified or static call is not polymorphic,
						-- and therefore the type of the query is exactly the type
						-- expected by the caller. This is another story with 'like argument'.
						-- For example 'clone (1)' is expected to be of type INTEGER
						-- and 'clone ("gobo")' is expected to be of type STRING.
						-- But the corresponding generated function for 'clone' will
						-- be declared to return an ANY. Hence the need for a call
						-- to `print_adapted_query_call' in this case.
						-- There is also the case when `l_force_result_boxing' is True.
					print_adapted_query_call (a_feature, a_feature.target_type, l_call_type, False)
				end
				if in_operand then
					current_file.put_character (')')
					print_semicolon_newline
				end
			end
			call_operands.wipe_out
			if l_manifest_tuple_operand /= Void then
				mark_expressions_unfrozen (l_manifest_tuple_operand)
				l_manifest_tuple_operand.wipe_out
			end
			if l_temp /= Void then
				mark_temp_variable_unfrozen (l_temp)
				if l_temp_index /= 0 then
						-- We had to wait until this stage to set the index of `l_temp'
						-- because it could have still been used in `call_operands'.
					l_temp.set_index (l_temp_index)
				end
			end
		end

	print_verbatim_string (a_string: ET_VERBATIM_STRING)
			-- Print `a_string'.
		require
			a_string_not_void: a_string /= Void
		do
			print_manifest_string (a_string)
		end

	print_void (an_expression: ET_VOID)
			-- Print `an_expression'.
		require
			an_expression_not_void: an_expression /= Void
		do
			if in_operand then
				operand_stack.force (an_expression)
			else
				current_file.put_string (c_eif_void)
			end
		end

	print_writable (a_writable: ET_WRITABLE)
			-- Print `a_writable'.
		require
			a_writable_not_void: a_writable /= Void
		local
			l_query: detachable ET_DYNAMIC_FEATURE
			l_seed: INTEGER
			old_in_operand: BOOLEAN
			old_target_type: like call_target_type
			l_old_in_procedure_call_target: BOOLEAN
			l_was_read: BOOLEAN
		do
			old_in_operand := in_operand
			old_target_type := call_target_type
			l_old_in_procedure_call_target := in_procedure_call_target
			in_operand := False
			in_procedure_call_target := False
			call_target_type := Void
			if attached {ET_IDENTIFIER} a_writable as l_identifier then
				if l_identifier.is_argument then
					print_formal_argument (l_identifier)
				elseif l_identifier.is_temporary then
					print_temporary_variable (l_identifier)
				elseif l_identifier.is_iteration_item then
						-- An iteration item is not writable, but we
						-- need this code here when initializing it.
					print_iteration_item (l_identifier)
				elseif l_identifier.is_inline_separate_argument then
						-- An inline separate argument is not writable, but we
						-- need this code here when initializing it.
					print_inline_separate_argument (l_identifier)
				elseif l_identifier.is_local then
					if has_rescue then
							-- Keep track of the fact that the value of this local variable
							-- has been modified. Useful to determine the 'volatile' status
							-- of the local variable when current feature or inline agent
							-- has a rescue clause.
							--
							-- Set `has_rescue' to False before calling `print_local_variable'
							-- because the fact that we are in a rescue clause has already
							-- been taken into account.
						l_seed := l_identifier.seed
						locals_written.force_last (l_seed)
						l_was_read := locals_read.has (l_seed)
						has_rescue := False
						print_local_variable (l_identifier)
						has_rescue := True
						if not l_was_read and locals_read.has (l_seed) then
							locals_read.remove (l_seed)
						end
					else
						print_local_variable (l_identifier)
					end
				else
					l_seed := l_identifier.seed
					l_query := current_type.seeded_dynamic_query (l_seed, current_dynamic_system)
					if l_query = Void then
							-- Internal error: there should be a query with `l_seed'.
							-- It has been computed in ET_FEATURE_CHECKER.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_writable", 1, "attribute not found.")
					else
						print_attribute_access (l_query, tokens.current_keyword, current_type, False)
					end
				end
			else
				check
					is_result: a_writable.is_result
				end
				if has_rescue then
						-- Keep track of the fact that the value of the 'Result' entity
						-- has been modified. Useful to determine the 'volatile' status
						-- of the 'Result' entity when current feature or inline agent
						-- has a rescue clause.
						--
						-- Set `has_rescue' to False before calling `print_local_variable'
						-- because the fact that we are in a rescue clause has already
						-- been taken into account.
					result_written := True
					l_was_read := result_read
					has_rescue := False
					a_writable.process (Current)
					has_rescue := True
					result_read := l_was_read
				else
					a_writable.process (Current)
				end
			end
			call_target_type := old_target_type
			in_procedure_call_target := l_old_in_procedure_call_target
			in_operand := old_in_operand
		end

feature {NONE} -- Equality generation

	print_equality_call (a_expression: ET_EXPRESSION; a_negated: BOOLEAN; a_equality_types: ET_DYNAMIC_EQUALITY_TYPES)
			-- Print `a_expression' which is supposed to be equality expression
			-- ('=' or '/=' as well as '~' or '/~') with operand types `a_equality_types'.
			-- `a_negated' means that the operator is either "/=" or "/~".
			-- Operands can be found in `call_operands'.
		require
			a_expression_not_void: a_expression /= Void
			a_equality_types_not_void: a_equality_types /= Void
		local
			l_common_object_equality_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_common_reference_equality_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			l_assignment_target: like assignment_target
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_boolean_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_left_type_set := a_equality_types.left
			l_right_type_set := a_equality_types.right
			l_boolean_type := dynamic_type_set (a_expression).static_type.primary_type
			if in_operand then
				if l_assignment_target /= Void then
					operand_stack.force (l_assignment_target)
					print_indentation
					print_writable (l_assignment_target)
				else
					l_temp := new_temp_variable (l_boolean_type)
					mark_temp_variable_frozen (l_temp)
						-- We will set the index of `l_temp' later because
						-- it could still be used in `call_operands'.
					l_temp_index := a_expression.index
					operand_stack.force (l_temp)
					print_indentation
					print_temp_name (l_temp, current_file)
				end
				print_assign_to
			end
				-- Find out which types are in the dynamic type set of both operands.
			l_common_object_equality_types := equality_common_types
			l_common_object_equality_types.resize (l_left_type_set.count.min (l_right_type_set.count))
			l_common_reference_equality_types := equality_common_reference_types
			l_common_reference_equality_types.resize (l_left_type_set.count.min (l_right_type_set.count))
			a_equality_types.add_common_types_to_list (l_common_object_equality_types, l_common_reference_equality_types)
			if l_common_object_equality_types.is_empty then
				print_equality_call_with_zero_object_equality (a_negated, l_common_object_equality_types, l_common_reference_equality_types)
			elseif l_common_object_equality_types.count = 1 then
				print_equality_call_with_one_object_equality (a_negated, l_common_object_equality_types, l_common_reference_equality_types)
			else
				l_left_operand := call_operands.first
				l_right_operand := call_operands.item (2)
				current_equalities.force_last (a_equality_types)
				if a_negated then
					print_declaration_type_cast (l_boolean_type, current_file)
					current_file.put_character ('(')
					current_file.put_string (c_not)
				end
				print_equality_function_name (current_equalities.count, current_non_inlined_feature, current_non_inlined_feature.target_type, current_file)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				print_expression (l_left_operand)
				print_comma
				print_expression (l_right_operand)
				current_file.put_character (')')
				if a_negated then
					current_file.put_character (')')
				end
			end
			if in_operand then
				print_semicolon_newline
			end
			l_common_object_equality_types.wipe_out
			l_common_reference_equality_types.wipe_out
			if l_temp /= Void then
				mark_temp_variable_unfrozen (l_temp)
				if l_temp_index /= 0 then
						-- We had to wait until this stage to set the index of `l_temp'
						-- because it could have still been used in `call_operands'.
					l_temp.set_index (l_temp_index)
				end
			end
		end

	print_equality_call_with_zero_object_equality (a_negated: BOOLEAN; a_common_object_equality_types, a_common_reference_equality_types: ET_DYNAMIC_PRIMARY_TYPES)
			-- Print equality call when there is no type in common between
			-- the dynamic type sets of the left and right operands for which
			-- object equality is needed.
			-- `a_negated' means that the operator is either "/=" or "/~".
			-- Operands can be found in `call_operands'.
		require
			a_common_object_equality_types_not_void: a_common_object_equality_types /= Void
			a_common_object_equality_types_is_empty: a_common_object_equality_types.is_empty
			a_common_reference_equality_types_not_void: a_common_reference_equality_types /= Void
			two_operands: call_operands.count = 2
		local
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_eif_true: STRING
			l_eif_false: STRING
			l_equal: STRING
			l_boolean_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			if a_negated then
				l_eif_true := c_eif_false
				l_eif_false := c_eif_true
				l_equal := c_not_equal
			else
				l_eif_true := c_eif_true
				l_eif_false := c_eif_false
				l_equal := c_equal
			end
			l_left_operand := call_operands.first
			l_right_operand := call_operands.item (2)
			l_left_type_set := dynamic_type_set (l_left_operand)
			l_right_type_set := dynamic_type_set (l_right_operand)
			if l_left_type_set.is_expanded or l_right_type_set.is_expanded then
				current_file.put_string (l_eif_false)
			elseif l_left_type_set.is_empty and l_right_type_set.is_empty then
					-- We know for sure that both operands are Void.
				current_file.put_string (l_eif_true)
			elseif a_common_reference_equality_types.is_empty and (not l_left_type_set.can_be_void or not l_right_type_set.can_be_void) then
					-- There is no type in common between the dynamic type sets
					-- of the left and right operands. The only way for the equality
					-- to succeed is that both operands be Void.
				current_file.put_string (l_eif_false)
			else
				l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
				print_declaration_type_cast (l_boolean_type, current_file)
				current_file.put_character ('(')
				print_expression (l_left_operand)
				current_file.put_string (l_equal)
				print_expression (l_right_operand)
				current_file.put_character (')')
			end
		end

	print_equality_call_with_one_object_equality (a_negated: BOOLEAN; a_common_object_equality_types, a_common_reference_equality_types: ET_DYNAMIC_PRIMARY_TYPES)
			-- Print equality call when there is exactly one type in common between
			-- the dynamic type sets of the left and right operands for which
			-- object equality is needed.
			-- `a_negated' means that the operator is either "/=" or "/~".
			-- Operands can be found in `call_operands'.
		require
			a_common_object_equality_types_not_void: a_common_object_equality_types /= Void
			a_common_object_equality_types_count_one: a_common_object_equality_types.count = 1
			a_common_reference_equality_types_not_void: a_common_reference_equality_types /= Void
			two_operands: call_operands.count = 2
		local
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_parenthese_count: INTEGER
			l_formal_type: ET_DYNAMIC_TYPE
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_left_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_right_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_eif_true: STRING
			l_eif_false: STRING
			l_equal: STRING
			l_index: INTEGER
			l_old_index: INTEGER
			l_both_multi_types: BOOLEAN
			l_boolean_type: ET_DYNAMIC_PRIMARY_TYPE
			l_old_max_nested_inlining_count: INTEGER
			l_old_in_qualified_call: BOOLEAN
		do
			if a_negated then
				l_eif_true := c_eif_false
				l_eif_false := c_eif_true
				l_equal := c_not_equal
			else
				l_eif_true := c_eif_true
				l_eif_false := c_eif_false
				l_equal := c_equal
			end
			l_left_operand := call_operands.first
			l_right_operand := call_operands.item (2)
			l_left_type_set := dynamic_type_set (l_left_operand)
			l_right_type_set := dynamic_type_set (l_right_operand)
			l_left_static_type := l_left_type_set.static_type.primary_type
			l_right_static_type := l_right_type_set.static_type.primary_type
			l_dynamic_type := a_common_object_equality_types.dynamic_type (1)
			if not attached l_dynamic_type.seeded_dynamic_query (current_system.is_equal_seed, current_dynamic_system) as l_is_equal_feature then
					-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_equality_call_with_one_object_equality", 1, "feature `is_equal' not found.")
			else
				if l_left_type_set.can_be_void then
					l_parenthese_count := l_parenthese_count + 1
					current_file.put_character ('(')
					current_file.put_character ('(')
					print_expression (l_left_operand)
					print_equal_to
					current_file.put_string (c_eif_void)
					current_file.put_character (')')
					current_file.put_character ('?')
					if l_right_type_set.can_be_void then
						l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
						print_declaration_type_cast (l_boolean_type, current_file)
						current_file.put_character ('(')
						print_expression (l_right_operand)
						current_file.put_string (l_equal)
						current_file.put_string (c_eif_void)
						current_file.put_character (')')
					else
						current_file.put_string (l_eif_false)
					end
					current_file.put_character (':')
				end
				if l_right_type_set.can_be_void then
					l_parenthese_count := l_parenthese_count + 1
					current_file.put_character ('(')
					current_file.put_character ('(')
					print_expression (l_right_operand)
					print_equal_to
					current_file.put_string (c_eif_void)
					current_file.put_character (')')
					current_file.put_character ('?')
					current_file.put_string (l_eif_false)
					current_file.put_character (':')
				end
					-- Check that both operands are of the same type.	
				if l_left_type_set.count > 1 or l_right_type_set.count > 1 then
						-- At least one of the operands can have several types in their type set.
						-- Check that they are of the expected type.
					l_parenthese_count := l_parenthese_count + 1
					current_file.put_character ('(')
					if a_common_reference_equality_types.is_empty then
						current_file.put_character ('(')
						print_attribute_type_id_access (l_left_operand, l_left_static_type, False)
						print_equal_to
						print_attribute_type_id_access (l_right_operand, l_right_static_type, False)
						current_file.put_character (')')
					else
						l_both_multi_types := l_left_type_set.count > 1 and l_right_type_set.count > 1
						if l_both_multi_types then
							current_file.put_character ('(')
						end
						if l_left_type_set.count > 1 then
							current_file.put_character ('(')
							print_attribute_type_id_access (l_left_operand, l_left_static_type, False)
							print_equal_to
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character (')')
						end
						if l_both_multi_types then
							print_and_then
						end
						if l_right_type_set.count > 1 then
							current_file.put_character ('(')
							print_attribute_type_id_access (l_right_operand, l_right_static_type, False)
							print_equal_to
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character (')')
						end
						if l_both_multi_types then
							current_file.put_character (')')
						end
					end
					current_file.put_character ('?')
				else
						-- The type sets of both operands contain only one type,
						-- which is `l_dynamic_type'. So we know for sure that
						-- both operands are of the same type.
				end
					-- Call 'is_equal'.
				l_formal_type := argument_type_set_in_feature (1, l_is_equal_feature).static_type
				if not l_dynamic_type.conforms_to_primary_type (l_formal_type.primary_type) then
						-- We won't be able to call 'is_equal' because the type of the
						-- actual argument does not conform to formal type. The only
						-- way of the equality to be True as to have both operands Void.
					current_file.put_string (l_eif_false)
				else
						-- The main purpose of object equality is to be CAT-call free.
						-- Therefore we have to make sure that `l_dynamic_type' is the
						-- only type in the dynamic type set of the right operand when
						-- calling `print_attachment_expression'.
					if l_right_static_type = l_dynamic_type then
						l_actual_type_set := l_dynamic_type
					elseif l_right_type_set.count = 1 then
						l_actual_type_set := l_right_type_set
					else
						equality_type_set.reset (l_right_static_type)
						equality_type_set.put_type (l_dynamic_type)
						if l_right_type_set.is_never_void then
							equality_type_set.set_never_void
						end
						l_actual_type_set := equality_type_set
					end
					extra_dynamic_type_sets.force_last (l_actual_type_set)
					l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
					l_old_index := l_right_operand.index
					l_right_operand.set_index (l_index)
					if a_negated then
						l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
						print_declaration_type_cast (l_boolean_type, current_file)
						current_file.put_character ('(')
						current_file.put_string (c_not)
					end
					if l_dynamic_type.has_redefined_is_equal_routine then
							-- No inlining.
						l_old_max_nested_inlining_count := max_nested_inlining_count
						max_nested_inlining_count := 0
						l_old_in_qualified_call := in_qualified_call
						in_qualified_call := True
						print_query_call (l_is_equal_feature, l_dynamic_type, False)
						in_qualified_call := l_old_in_qualified_call
						max_nested_inlining_count := l_old_max_nested_inlining_count
					else
							-- Make sure to call the built-in version, even for types
							-- such as INTEGER_32, etc. where 'is_equal' is redefined
							-- as 'Result := other.item = item' instead of using the
							-- built-in version from "ANY" (which introduces an infinite
							-- recursive call.).
						print_builtin_any_is_equal_call (l_is_equal_feature, l_dynamic_type, False)
					end
					if a_negated then
						current_file.put_character (')')
					end
					l_right_operand.set_index (l_old_index)
					extra_dynamic_type_sets.remove_last
						-- Clean up `equality_type_set'.
					if l_actual_type_set = equality_type_set then
						equality_type_set.reset (current_dynamic_system.unknown_type)
					end
				end
				if l_left_type_set.count > 1 or l_right_type_set.count > 1 then
					current_file.put_character (':')
						-- Reference equality.
					if a_common_reference_equality_types.is_empty then
						current_file.put_string (l_eif_false)
					else
						print_equality_call_with_zero_object_equality (a_negated, current_dynamic_system.unknown_type, a_common_reference_equality_types)
					end
				end
				from until l_parenthese_count = 0 loop
					current_file.put_character (')')
					l_parenthese_count := l_parenthese_count - 1
				end
			end
		end

	print_equality_function (i: INTEGER; a_equality_types: ET_DYNAMIC_EQUALITY_TYPES)
			-- Print function corresponding to `i'-th equality ('=' or '/=', as well as
			-- '~' or '/~') with operand types `a_equality_types' to `current_file' and
			-- its signature to `header_file'.
		require
			a_equality_types_not_void: a_equality_types /= Void
		local
			old_file: KI_TEXT_OUTPUT_STREAM
			l_common_object_equality_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_common_reference_equality_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_left_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_right_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_left_operand: ET_IDENTIFIER
			l_right_operand: ET_IDENTIFIER
			l_index: INTEGER
		do
			l_left_type_set := a_equality_types.left
			l_right_type_set := a_equality_types.right
			l_left_static_type := l_left_type_set.static_type.primary_type
			l_right_static_type := l_right_type_set.static_type.primary_type
				-- Print signature to `header_file' and `current_file'.
			old_file := current_file
			current_file := current_function_header_buffer
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			l_result_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
			print_type_declaration (l_result_type, header_file)
			print_type_declaration (l_result_type, current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_equality_function_name (i, current_feature, current_type, header_file)
			print_equality_function_name (i, current_feature, current_type, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			l_left_operand := formal_argument (1)
			extra_dynamic_type_sets.force_last (l_left_type_set)
			l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
			l_left_operand.set_index (l_index)
			print_type_declaration (l_left_static_type, header_file)
			header_file.put_character (' ')
			print_argument_name (l_left_operand, header_file)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_type_declaration (l_left_static_type, current_file)
			current_file.put_character (' ')
			print_argument_name (l_left_operand, current_file)
			current_file.put_character (',')
			current_file.put_character (' ')
			l_right_operand := formal_argument (2)
			extra_dynamic_type_sets.force_last (l_right_type_set)
			l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
			l_right_operand.set_index (l_index)
			print_type_declaration (l_right_static_type, header_file)
			header_file.put_character (' ')
			print_argument_name (l_right_operand, header_file)
			print_type_declaration (l_right_static_type, current_file)
			current_file.put_character (' ')
			print_argument_name (l_right_operand, current_file)
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print body to `current_file'.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			current_file := current_function_body_buffer
				-- Find out which types are in the dynamic type set of both operands.
			l_common_object_equality_types := equality_common_types
			l_common_object_equality_types.resize (l_left_type_set.count.min (l_right_type_set.count))
			l_common_reference_equality_types := equality_common_reference_types
			l_common_reference_equality_types.resize (l_left_type_set.count.min (l_right_type_set.count))
			a_equality_types.add_common_types_to_list (l_common_object_equality_types, l_common_reference_equality_types)
			operand_stack.force (l_left_operand)
			operand_stack.force (l_right_operand)
			fill_call_operands (2)
			print_equality_function_body (l_common_object_equality_types, l_common_reference_equality_types)
			call_operands.wipe_out
			l_common_object_equality_types.wipe_out
			l_common_reference_equality_types.wipe_out
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			l_left_operand.set_index (1)
			l_right_operand.set_index (2)
			extra_dynamic_type_sets.remove_last
			extra_dynamic_type_sets.remove_last
			current_file := old_file
		end

	print_equality_function_body (a_common_object_equality_types, a_common_reference_equality_types: ET_DYNAMIC_PRIMARY_TYPES)
			-- Print equality function body.
			-- Operands can be found in `call_operands'.
		require
			a_common_object_equality_types_not_void: a_common_object_equality_types /= Void
			a_common_reference_equality_types_not_void: a_common_reference_equality_types /= Void
			two_operands: call_operands.count = 2
		local
			l_left_operand: ET_EXPRESSION
			l_right_operand: ET_EXPRESSION
			l_left_type_set: ET_DYNAMIC_TYPE_SET
			l_right_type_set: ET_DYNAMIC_TYPE_SET
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_formal_type: ET_DYNAMIC_TYPE
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_left_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_right_static_type: ET_DYNAMIC_PRIMARY_TYPE
			i, nb: INTEGER
			l_index: INTEGER
			l_old_index: INTEGER
			l_boolean_type: ET_DYNAMIC_PRIMARY_TYPE
			l_old_max_nested_inlining_count: INTEGER
			l_old_in_qualified_call: BOOLEAN
		do
			if a_common_object_equality_types.is_empty then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_equality_call_with_zero_object_equality (False, a_common_object_equality_types, a_common_reference_equality_types)
				print_semicolon_newline
			elseif a_common_object_equality_types.count = 1 then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_equality_call_with_one_object_equality (False, a_common_object_equality_types, a_common_reference_equality_types)
				print_semicolon_newline
			else
				l_left_operand := call_operands.first
				l_right_operand := call_operands.item (2)
				l_left_type_set := dynamic_type_set (l_left_operand)
				l_right_type_set := dynamic_type_set (l_right_operand)
				l_left_static_type := l_left_type_set.static_type.primary_type
				l_right_static_type := l_right_type_set.static_type.primary_type
				if l_left_type_set.can_be_void then
					print_indentation_if
					print_expression (l_left_operand)
					print_equal_to
					current_file.put_string (c_eif_void)
					print_then_newline
					indent
					print_indentation
					current_file.put_string (c_return)
					current_file.put_character (' ')
					if l_right_type_set.can_be_void then
						l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
						print_declaration_type_cast (l_boolean_type, current_file)
						current_file.put_character ('(')
						print_expression (l_right_operand)
						print_equal_to
						current_file.put_string (c_eif_void)
						current_file.put_character (')')
					else
						current_file.put_string (c_eif_false)
					end
					print_semicolon_newline
					dedent
					print_indentation_end_newline
				end
				if l_right_type_set.can_be_void then
					print_indentation_if
					print_expression (l_right_operand)
					print_equal_to
					current_file.put_string (c_eif_void)
					print_then_newline
					indent
					print_indentation
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_string (c_eif_false)
					print_semicolon_newline
					dedent
					print_indentation_end_newline
				end
				print_indentation
				current_file.put_string (c_switch)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_attribute_type_id_access (l_left_operand, l_left_static_type, False)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				nb := a_common_object_equality_types.count
				from i := 1 until i > nb loop
					l_dynamic_type := a_common_object_equality_types.dynamic_type (i)
					print_indentation
					current_file.put_string (c_case)
					current_file.put_character (' ')
					current_file.put_integer (l_dynamic_type.id)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					if not attached l_dynamic_type.seeded_dynamic_query (current_system.is_equal_seed, current_dynamic_system) as l_is_equal_feature then
							-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_equality_function_body", 1, "feature `is_equal' not found.")
					else
						l_formal_type := argument_type_set_in_feature (1, l_is_equal_feature).static_type
						if not l_dynamic_type.conforms_to_primary_type (l_formal_type.primary_type) then
								-- We won't be able to call 'is_equal' because the type of the
								-- actual argument does not conform to formal type.
							print_indentation
							current_file.put_string (c_return)
							current_file.put_character (' ')
							current_file.put_string (c_eif_false)
							print_semicolon_newline
						else
							print_indentation
							current_file.put_string (c_return)
							current_file.put_character (' ')
							l_boolean_type := current_dynamic_system.dynamic_primary_type (current_universe_impl.boolean_type, current_type.base_type)
							print_declaration_type_cast (l_boolean_type, current_file)
							current_file.put_character ('(')
							current_file.put_character ('(')
							print_attribute_type_id_access (l_right_operand, l_right_static_type, False)
							print_equal_to
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character (')')
							print_and_then
								-- Call 'is_equal'.
								--
								-- The main purpose of object equality is to be CAT-call free.
								-- Therefore we have to make sure that `l_dynamic_type' is the
								-- only type in the dynamic type set of the right operand when
								-- calling `print_attachment_expression'.
							if l_right_static_type = l_dynamic_type then
								l_actual_type_set := l_dynamic_type
							else
								equality_type_set.reset (l_right_static_type)
								equality_type_set.put_type (l_dynamic_type)
								if l_right_type_set.is_never_void then
									equality_type_set.set_never_void
								end
								l_actual_type_set := equality_type_set
							end
							extra_dynamic_type_sets.force_last (l_actual_type_set)
							l_index := current_dynamic_type_sets.count + extra_dynamic_type_sets.count
							l_old_index := l_right_operand.index
							l_right_operand.set_index (l_index)
							if l_dynamic_type.has_redefined_is_equal_routine then
									-- No inlining.
								l_old_max_nested_inlining_count := max_nested_inlining_count
								max_nested_inlining_count := 0
								l_old_in_qualified_call := in_qualified_call
								in_qualified_call := True
								print_query_call (l_is_equal_feature, l_dynamic_type, False)
								in_qualified_call := l_old_in_qualified_call
								max_nested_inlining_count := l_old_max_nested_inlining_count
							else
									-- Make sure to call the built-in version, even for types
									-- such as INTEGER_32, etc. where 'is_equal' is redefined
									-- as 'Result := other.item = item' instead of using the
									-- built-in version from "ANY" (which introduces an infinite
									-- recursive call.).
								print_builtin_any_is_equal_call (l_is_equal_feature, l_dynamic_type, False)
							end
							l_right_operand.set_index (l_old_index)
							extra_dynamic_type_sets.remove_last
								-- Clean up `equality_type_set'.
							if l_actual_type_set = equality_type_set then
								equality_type_set.reset (current_dynamic_system.unknown_type)
							end
							current_file.put_character (')')
							print_semicolon_newline
						end
					end
					dedent
					i := i + 1
				end
					-- Reference equality.
				print_indentation
				current_file.put_string (c_default)
				current_file.put_character (':')
				current_file.put_new_line
				indent
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				if a_common_reference_equality_types.is_empty then
					current_file.put_string (c_eif_false)
				else
					print_equality_call_with_zero_object_equality (False, current_dynamic_system.unknown_type, a_common_reference_equality_types)
				end
				print_semicolon_newline
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			end
		end

feature {NONE} -- Object-test generation

	print_object_test_function (i: INTEGER; a_object_test: ET_OBJECT_TEST)
			-- Print function corresponding to `i'-th object-test `a_object_test'
			-- to `current_file' and its signature to `header_file'.
			-- This function will return True or False depending if it's
			-- successful or not.
			-- The object passed as argument is guaranteed (by the caller) to
			-- be non-Void and in the SCOOP current region if the object-test
			-- checks for that (i.e. the type is not separate but the expression is).
		require
			a_object_test_not_void: a_object_test /= Void
		local
			old_file: KI_TEXT_OUTPUT_STREAM
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			j, nb: INTEGER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_source_type_set: ET_DYNAMIC_TYPE_SET
			l_source_type: ET_DYNAMIC_TYPE
			l_source_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_target_type: ET_DYNAMIC_TYPE
			l_target_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_non_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_conforming_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
			l_old_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_source_argument: ET_IDENTIFIER
			l_name: detachable ET_IDENTIFIER
			l_type: detachable ET_TYPE
		do
			l_name := a_object_test.name
			l_type := a_object_test.type
			l_source_type_set := dynamic_type_set (a_object_test.expression)
			l_source_type := l_source_type_set.static_type
			l_source_primary_type := l_source_type.primary_type
			if l_name /= Void then
				l_target_type_set := dynamic_type_set (l_name)
				l_target_type := l_target_type_set.static_type
			elseif l_type /= Void then
				l_target_type := current_dynamic_system.dynamic_type (l_type, current_type.base_type)
			else
				l_target_type := l_source_type
			end
			l_target_primary_type := l_target_type.primary_type
				-- Print signature to `header_file' and `current_file'.
			old_file := current_file
			current_file := current_function_header_buffer
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			l_result_type := dynamic_type_set (a_object_test).static_type.primary_type
			print_type_declaration (l_result_type, header_file)
			print_type_declaration (l_result_type, current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_object_test_function_name (i, current_feature, current_type, header_file)
			print_object_test_function_name (i, current_feature, current_type, current_file)
			header_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			current_file.put_character ('(')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			l_source_argument := formal_argument (1)
			print_type_declaration (l_source_primary_type, header_file)
			header_file.put_character (' ')
			print_argument_name (l_source_argument, header_file)
			print_type_declaration (l_source_primary_type, current_file)
			current_file.put_character (' ')
			print_argument_name (l_source_argument, current_file)
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print body to `current_file'.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			current_file := current_function_body_buffer
			nb := l_source_type_set.count
			l_conforming_types := conforming_types
			l_conforming_types.resize (nb)
			l_non_conforming_types := non_conforming_types
			l_non_conforming_types.resize (nb)
			from j := 1 until j > nb loop
				l_dynamic_type := l_source_type_set.dynamic_type (j)
				if l_dynamic_type.conforms_to_primary_type (l_target_type.primary_type) then
					l_conforming_types.put_last (l_dynamic_type)
				else
					l_non_conforming_types.put_last (l_dynamic_type)
				end
				j := j + 1
			end
			if l_conforming_types.is_empty then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				current_file.put_string (c_eif_false)
				current_file.put_character (';')
				current_file.put_new_line
			else
				if l_non_conforming_types.is_empty then
						-- Direct assignment.
					print_indentation
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (';')
					current_file.put_new_line
				else
					check
							-- None of `l_non_conforming_types' and `l_conforming_types' are empty.
							-- Therefore the source is polymorphic (at least two dynamic types, one
							-- conforming and one not conforming). As a consequence `l_source_type'
							-- is not expanded because for expanded types there is no type other
							-- than itself that conforms to it.
							-- So we know that a type-id is always available.
						not_expanded: not l_source_type.is_expanded
					end
					print_indentation
					current_file.put_string (c_switch)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_attribute_type_id_access (l_source_argument, l_source_primary_type, False)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					l_conforming_type_set := new_standalone_type_set (l_source_type)
					l_old_conforming_types := l_conforming_type_set.dynamic_types
					l_conforming_type_set.reset_with_types (l_source_type, l_conforming_types)
					l_conforming_type_set.set_never_void
					if l_non_conforming_types.count < l_conforming_types.count then
						nb := l_non_conforming_types.count
						from j := 1 until j > nb loop
							l_dynamic_type := l_non_conforming_types.dynamic_type (j)
							print_indentation
							current_file.put_string (c_case)
							current_file.put_character (' ')
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character (':')
							current_file.put_new_line
							j := j + 1
						end
						indent
						print_indentation
						current_file.put_string (c_return)
						current_file.put_character (' ')
						current_file.put_string (c_eif_false)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
						print_indentation
						current_file.put_string (c_default)
						current_file.put_character (':')
						current_file.put_new_line
						indent
						print_indentation
						current_file.put_string (c_return)
						current_file.put_character (' ')
						current_file.put_string (c_eif_true)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
					else
						nb := l_conforming_types.count
						from j := 1 until j > nb loop
							l_dynamic_type := l_conforming_types.dynamic_type (j)
							print_indentation
							current_file.put_string (c_case)
							current_file.put_character (' ')
							current_file.put_integer (l_dynamic_type.id)
							current_file.put_character (':')
							current_file.put_new_line
							j := j + 1
						end
						indent
						print_indentation
						current_file.put_string (c_return)
						current_file.put_character (' ')
						current_file.put_string (c_eif_true)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
						print_indentation
						current_file.put_string (c_default)
						current_file.put_character (':')
						current_file.put_new_line
						indent
						print_indentation
						current_file.put_string (c_return)
						current_file.put_character (' ')
						current_file.put_string (c_eif_false)
						current_file.put_character (';')
						current_file.put_new_line
						dedent
					end
					l_conforming_type_set.reset_with_types (current_dynamic_system.unknown_type, l_old_conforming_types)
					free_standalone_type_set (l_conforming_type_set)
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				end
			end
			dedent
			print_indentation
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			l_conforming_types.wipe_out
			l_non_conforming_types.wipe_out
			current_file := old_file
		end

feature {NONE} -- Query call generation

	print_adapted_named_query_call (a_name: ET_CALL_NAME; a_target_type, a_result_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_name' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- `a_result_type' is the static type of the result expected by the caller,
			-- used to adapt the result of the query if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_target_type_not_void: a_target_type /= Void
			a_result_type_not_void: a_result_type /= Void
			a_name_not_void: a_name /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_seed: INTEGER
			l_tuple_item_type_sets: ET_DYNAMIC_TYPE_SET_LIST
			l_query_type_set: ET_DYNAMIC_TYPE_SET
		do
			l_seed := a_name.seed
			if a_name.is_tuple_label then
				if not attached {ET_DYNAMIC_TUPLE_TYPE} a_target_type as l_tuple_type then
						-- Internal error: if `a_name' is a Tuple label then the
						-- target type should be a Tuple_type.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_adapted_named_query_call", 1, "target type of tuple label is not 'TUPLE'.")
				else
					l_tuple_item_type_sets := l_tuple_type.item_type_sets
					if l_seed < 1 or l_seed > l_tuple_item_type_sets.count then
							-- Internal error: invalid Tuple label.
						set_fatal_error
						error_handler.report_giaac_error (generator, "print_adapted_named_query_call", 2, "invalid tuple label.")
					else
						l_query_type_set := l_tuple_item_type_sets.item (l_seed)
						print_adapted_expression_with_agent (agent print_attribute_tuple_item_access (l_seed, call_operands.first, a_target_type, a_check_void_target), l_query_type_set, a_result_type, in_call_target)
					end
				end
			elseif attached a_target_type.seeded_dynamic_query (l_seed, current_dynamic_system) as l_dynamic_feature then
				print_adapted_query_call (l_dynamic_feature, a_target_type, a_result_type, a_check_void_target)
			else
					-- Internal error: there should be a query with `l_seed'.
					-- It has been computed in ET_FEATURE_CHECKER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_adapted_named_query_call", 3, "query not found.")
			end
		end

	print_adapted_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type, a_result_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- `a_result_type' is the static type of the result expected by the caller,
			-- used to adapt the result of the query if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_target_type_not_void: a_target_type /= Void
			a_result_type_not_void: a_result_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_static_query: ET_STANDALONE_CLOSURE
			l_use_boxed_pointer: BOOLEAN
		do
			l_static_query := a_feature.static_feature
			if attached {ET_EXTENDED_ATTRIBUTE} l_static_query as l_extended_attribute then
				print_adapted_extended_attribute_call (a_feature, l_extended_attribute, a_target_type, a_result_type, a_check_void_target)
			elseif a_feature.is_once then
				print_adapted_once_function_call (a_feature, a_target_type, a_result_type, a_check_void_target)
			elseif not attached a_feature.result_type_set as l_query_type_set then
					-- Internal error: `a_feature' is a query.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_adapted_query_call", 1, "feature is not a query.")
			else
				l_use_boxed_pointer := in_call_target and then (attached {ET_ATTRIBUTE} l_static_query or a_feature.is_builtin_special_item)
				print_adapted_expression_with_agent (agent print_query_call (a_feature, a_target_type, a_check_void_target), l_query_type_set, a_result_type, l_use_boxed_pointer)
			end
		end

	print_adapted_extended_attribute_call (a_feature: ET_DYNAMIC_FEATURE; a_static_feature: ET_EXTENDED_ATTRIBUTE; a_target_type, a_result_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to extended attribute `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- `a_result_type' is the static type of the result expected by the caller,
			-- used to adapt the result of the query if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_static_feature: a_static_feature = a_feature.static_feature
			a_target_type_not_void: a_target_type /= Void
			a_result_type_not_void: a_result_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_target: ET_EXPRESSION
		do
			if not attached a_feature.result_type_set as l_query_type_set then
					-- Internal error: `a_feature' is a query.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_adapted_extended_attribute_call", 1, "feature is not a query.")
			elseif
				a_static_feature.has_self_initializing_code and then
				not l_query_type_set.static_type.is_self_initializing
			then
					-- Note: If there is no attribute initialization code, then
					-- this is a regular attribute, possibly with assertions
					-- or a 'note' clause.
					-- Note: The semantics rule MEVS, in ECMA-367 3-36, section 8.19.20,
					-- says that the attribute initialization code is not executed
					-- if the type of the attribute is self-initializing.
				l_target := call_operands.first
				current_file.put_character ('(')
					-- Initialize the attribute if not already done.
				print_query_call (a_feature, a_target_type, a_check_void_target)
				print_comma
					-- Then return the attribute (or its address if it is the target of another call).
				print_adapted_expression_with_agent (agent print_adapted_attribute_access (a_feature, l_target, a_target_type, a_check_void_target), l_query_type_set, a_result_type, in_call_target)
				current_file.put_character (')')
			else
				print_adapted_expression_with_agent (agent print_query_call (a_feature, a_target_type, a_check_void_target), l_query_type_set, a_result_type, in_call_target)
			end
		end

	print_adapted_once_function_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type, a_result_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to once function `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- `a_result_type' is the static type of the result expected by the caller,
			-- used to adapt the result of the once function if needed (see header comment of
			-- `print_adapted_expression_with_agent' for details).
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_once: a_feature.is_once
			a_target_type_not_void: a_target_type /= Void
			a_result_type_not_void: a_result_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_once_index: INTEGER
			l_once_kind: INTEGER
		do
			if not attached a_feature.result_type_set as l_query_type_set then
					-- Internal error: `a_feature' is a query.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_adapted_once_function_call", 1, "feature is not a query.")
			elseif l_query_type_set.is_expanded and then not a_result_type.is_expanded then
				current_file.put_character ('(')
					-- Call the once function in order to be sure that it have been executed at least once.
				print_query_call (a_feature, a_target_type, a_check_void_target)
				print_comma
					-- Then return the boxed value.
				register_once_feature (a_feature)
				once_features.search (a_feature.static_feature.implementation_feature)
				if once_features.found then
					l_once_index := once_features.found_item
					l_once_kind := once_kind (a_feature)
				end
				if a_feature.is_once_per_object or l_query_type_set.static_type.is_basic then
					print_adapted_expression_with_agent (agent print_once_value (a_feature, l_once_kind, l_once_index), l_query_type_set, a_result_type, in_call_target)
				else
						-- Note that for once-per-process and once-per-thread functions with
						-- non-basic expanded types, the value is already boxed.
					print_once_value (a_feature, l_once_kind, l_once_index)
				end
				current_file.put_character (')')
			else
				print_adapted_expression_with_agent (agent print_query_call (a_feature, a_target_type, a_check_void_target), l_query_type_set, a_result_type, False)
			end
		end

	print_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_static_query: ET_STANDALONE_CLOSURE
			l_once_feature: ET_FEATURE
			l_target: ET_EXPRESSION
			old_index: INTEGER
			l_constant: ET_CONSTANT
			old_type: like current_type
			old_feature: like current_feature
		do
			l_static_query := a_feature.static_feature
			if not attached a_feature.result_type_set as l_result_type_set then
					-- Internal error: `a_feature' is a query.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_query_call", 1, "feature is not a query.")
			elseif attached {ET_CONSTANT_ATTRIBUTE} l_static_query as l_constant_attribute then
-- TODO: Need to check whether the target is Void or not, even though it does not matter here.
				l_constant := l_constant_attribute.constant
				if attached {ET_MANIFEST_STRING} l_constant as l_string_constant then
						-- Try to inject the commit SHA-1 to the Gobo version if not already done yet.
						-- Useful when compiling Gobo tools from a local Git clone.
					if
						l_string_constant.value.same_string ("xx.xx.xx+xxxxxxxxx") and
						l_constant_attribute.implementation_class.upper_name.same_string ("UT_GOBO_VERSION")
					then
						l_constant := gobo_version_string_constant
					end
					l_once_feature := l_constant_attribute.implementation_feature
					constant_features.force_last (l_constant, l_once_feature)
					print_once_value_name (l_once_feature, current_file)
				else
					old_type := current_type
					current_type := a_feature.target_type
					old_feature := current_feature
					current_feature := a_feature
					old_index := l_constant.index
					extra_dynamic_type_sets.force_last (l_result_type_set)
					l_constant.set_index (current_dynamic_type_sets.count + extra_dynamic_type_sets.count)
					print_expression (l_constant)
					l_constant.set_index (old_index)
					extra_dynamic_type_sets.remove_last
					current_feature := old_feature
					current_type := old_type
				end
			elseif attached {ET_UNIQUE_ATTRIBUTE} l_static_query as l_unique_attribute then
-- TODO: Need to check whether the target is Void or not, even though it does not matter here.
				print_type_cast (l_result_type_set.static_type.primary_type, current_file)
				current_file.put_character ('(')
				current_file.put_integer (l_unique_attribute.constant.to_integer_32)
				current_file.put_character (')')
			elseif a_feature.is_attribute_with_self_initializing_code then
					-- Note: If there is no attribute initialization code, then
					-- this is a regular attribute, possibly with assertions
					-- or a 'note' clause.
					-- Note: The semantics rule MEVS, in ECMA-367 3-36, section 8.19.20,
					-- says that the attribute initialization code is not executed
					-- if the type of the attribute is self-initializing.
					--
					-- Self-initialization code for this attribute.
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			elseif attached {ET_ATTRIBUTE} l_static_query then
				l_target := call_operands.first
				print_adapted_attribute_access (a_feature, l_target, a_target_type, a_check_void_target)
			elseif a_feature.is_builtin then
				print_builtin_query_call (a_feature, a_target_type, a_check_void_target)
			elseif is_inlinable_function (a_feature,  a_target_type) then
				print_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_non_inlined_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a non-inlined version of a call to query `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			i, nb: INTEGER
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type_set: ET_DYNAMIC_TYPE_SET
		do
			register_called_feature (a_feature)
			print_before_call (a_target_type, in_qualified_call)
			print_routine_name (a_feature, a_target_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			if not a_feature.is_static then
				print_comma
				print_target_expression (call_operands.first, a_target_type, a_check_void_target)
			end
			nb := call_operands.count
			from i := 2 until i > nb loop
				print_comma
				l_actual_type_set := dynamic_type_set (call_operands.item (i))
				l_formal_type_set := argument_type_set_in_feature (i - 1, a_feature)
				print_attachment_expression (call_operands.item (i), l_actual_type_set, l_formal_type_set.static_type)
				i := i + 1
			end
			current_file.put_character (')')
			print_after_call (a_target_type)
		end

	print_inlined_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' an inlined version of a call to function `a_feature' (static binding).
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
			a_feature_is_inlinable: is_inlinable_function (a_feature, a_target_type)
		local
			l_old_call_target_type: like call_target_type
			l_operands_count: INTEGER
			l_expression: detachable ET_EXPRESSION
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_old_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_caller_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_inlining_context: ET_DYNAMIC_CALL_CONTEXT
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_default_value_type: detachable ET_DYNAMIC_TYPE
			l_result_type: ET_DYNAMIC_TYPE
			l_old_current_index: INTEGER
			l_old_index_offset: INTEGER
		do
			l_operands_count := call_operands.count
			if l_operands_count /= a_feature.static_feature.arguments_count + 1 then
					-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 1, "wrong number of arguments.")
			elseif not attached a_feature.result_type_set as l_result_type_set then
					-- Internal error: it has to have a result, otherwise it's
					-- not a query call.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 2, "feature is not a query.")
			elseif not attached {ET_INTERNAL_FUNCTION} a_feature.static_feature as l_internal_function then
					-- Internal error: only internal routines are inlinable.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 3, "feature is not an internal function.")
			elseif not attached l_internal_function.compound as l_compound then
					-- Internal error: an inlinable function should have an single
					-- instruction which is an assignment to 'Result'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 4, "inlinable function with no single assignment to 'Result'.")
			elseif l_compound.count = 0 then
				l_result_type := l_result_type_set.static_type
				if not l_result_type.is_attached then
					l_expression := inlined_void
					l_default_value_type := current_dynamic_system.none_type
				elseif l_result_type.is_integer_n then
					l_expression := inlined_integer
					l_default_value_type := l_result_type
				elseif l_result_type.is_real_n then
					l_expression := inlined_real
					l_default_value_type := l_result_type
				elseif l_result_type.is_character_n then
					l_expression := inlined_character
					l_default_value_type := l_result_type
				elseif l_result_type = current_dynamic_system.boolean_type then
					l_expression := inlined_boolean
					l_default_value_type := l_result_type
				end
			elseif l_compound.count /= 1 then
					-- Internal error: an inlinable function should have an single
					-- instruction which is an assignment to 'Result'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 5, "inlinable function with no single assignment to 'Result'.")
			elseif not attached {ET_ASSIGNMENT} l_compound.first as l_assignment then
					-- Internal error: an inlinable function should have an single
					-- instruction which is an assignment to 'Result'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_inlined_query_call", 6, "inlinable function with no single assignment to 'Result'.")
			else
				l_expression := l_assignment.source
			end
			if l_expression /= Void then
				register_inlined_feature (a_feature)
				nested_inlining_count := nested_inlining_count + 1
				l_old_inlining_context := inlining_context
				l_caller_inlining_context := l_old_inlining_context
				if l_caller_inlining_context = Void then
					l_caller_inlining_context := new_call_context (current_feature)
				end
				l_target_type_set := dynamic_type_set (call_operands.first)
				l_inlining_context := new_call_context (a_feature)
				fill_inlining_operands (a_target_type, l_inlining_context, l_caller_inlining_context)
				inlining_context := l_inlining_context
				l_old_call_target_type := call_target_type
				current_feature := a_feature
				current_type := a_feature.target_type
				current_dynamic_type_sets := a_feature.dynamic_type_sets
				l_old_current_index := current_index
				current_index := a_feature.current_index
				l_old_index_offset := index_offset
				index_offset := 0
				if l_default_value_type /= Void then
					extra_dynamic_type_sets.force_last (l_default_value_type)
					l_expression.set_index (current_dynamic_type_sets.count + extra_dynamic_type_sets.count)
				end
				l_dynamic_type_set := dynamic_type_set (l_expression)
				print_indentation
				current_file.put_string ("/* -> ")
				print_routine_name (current_feature, current_type, current_file)
				current_file.put_string (" */")
				current_file.put_new_line
				print_attachment_operand (l_expression, l_dynamic_type_set)
				print_indentation
				current_file.put_string ("/* <- */")
				current_file.put_new_line
				if not operand_stack.is_empty and then not attached {ET_DYNAMIC_INLINED_EXPRESSION} operand_stack.item then
					operand_stack.replace (new_inlined_operand (operand_stack.item, l_inlining_context))
				end
				free_inlining_operands (l_target_type_set, l_inlining_context, l_caller_inlining_context)
				current_feature := l_caller_inlining_context.current_feature
				current_type := current_feature.target_type
				current_dynamic_type_sets := current_feature.dynamic_type_sets
				current_index := l_old_current_index
				index_offset := l_old_index_offset
				if l_default_value_type /= Void then
					extra_dynamic_type_sets.remove_last
				end
				call_target_type := l_old_call_target_type
				inlining_context := l_old_inlining_context
				nested_inlining_count := nested_inlining_count - 1
			end
		end

	print_builtin_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_class_code
			when {ET_TOKEN_CODES}.builtin_any_class then
				print_builtin_any_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_arguments_32_class then
				print_builtin_arguments_32_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_class then
				print_builtin_boolean_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_ref_class then
				print_builtin_boolean_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_8_class then
				print_builtin_character_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_8_ref_class then
				print_builtin_character_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_32_class then
				print_builtin_character_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_32_ref_class then
				print_builtin_character_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_class then
				print_builtin_com_failure_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_function_class then
				print_builtin_function_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_identified_routines_class then
				print_builtin_identified_routines_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_8_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_8_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_16_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_16_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_32_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_32_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_64_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_64_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_class then
				print_builtin_ise_runtime_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_memory_class then
				print_builtin_memory_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_8_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_8_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_16_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_16_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_32_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_32_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_64_class then
				print_builtin_integer_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_natural_64_ref_class then
				print_builtin_integer_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_class then
				print_builtin_platform_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_class then
				print_builtin_pointer_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_ref_class then
				print_builtin_pointer_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_32_class then
				print_builtin_real_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_32_ref_class then
				print_builtin_real_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_64_class then
				print_builtin_real_n_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_64_ref_class then
				print_builtin_real_n_ref_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_class then
				print_builtin_special_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_tuple_class then
				print_builtin_tuple_query_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_class then
				print_builtin_type_query_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_any_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ANY".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_any: a_feature.is_builtin_any_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_any_deep_twin then
				print_builtin_any_deep_twin_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_generating_type then
				print_builtin_any_generating_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_generator then
				print_builtin_any_generator_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_is_equal then
				print_builtin_any_is_equal_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_same_type then
				print_builtin_any_same_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_standard_is_equal then
				print_builtin_any_standard_is_equal_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_standard_twin then
				print_builtin_any_standard_twin_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_any_twin then
				print_builtin_any_twin_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_arguments_32_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ARGUMENTS_32".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_arguments_32: a_feature.is_builtin_arguments_32_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_arguments_32_argument_count then
				print_builtin_arguments_32_argument_count_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_arguments_32_i_th_argument_pointer then
				print_builtin_arguments_32_i_th_argument_pointer_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_arguments_32_i_th_argument_string then
				print_builtin_arguments_32_i_th_argument_string_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_boolean_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "BOOLEAN".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean: a_feature.is_builtin_boolean_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_and then
				print_builtin_boolean_and_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_and_then then
				print_builtin_boolean_and_then_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_implies then
				print_builtin_boolean_implies_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_not then
				print_builtin_boolean_not_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_or then
				print_builtin_boolean_or_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_or_else then
				print_builtin_boolean_or_else_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_boolean_xor then
				print_builtin_boolean_xor_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_boolean_ref_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "BOOLEAN_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_boolean_ref: a_feature.is_builtin_boolean_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_boolean_ref_item then
				print_builtin_boolean_ref_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_character_n_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "CHARACTER_N".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_target_type_not_void: a_target_type /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n: a_feature.is_builtin_character_n_class
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_code then
				print_builtin_character_n_code_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_n_natural_32_code then
				print_builtin_character_n_natural_32_code_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_n_to_character_8 then
				print_builtin_character_n_to_character_8_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_character_n_to_character_32 then
				print_builtin_character_n_to_character_32_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_character_n_ref_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "CHARACTER_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_target_type_not_void: a_target_type /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_character_n_ref: a_feature.is_builtin_character_n_ref_class
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_character_n_ref_item then
				print_builtin_character_n_ref_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_com_failure_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "COM_FAILURE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_com_failure: a_feature.is_builtin_com_failure_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_com_failure_c_strlen then
				print_builtin_com_failure_c_strlen_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult then
				print_builtin_com_failure_ccom_hresult_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult_code then
				print_builtin_com_failure_ccom_hresult_code_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_ccom_hresult_facility then
				print_builtin_com_failure_ccom_hresult_facility_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_character_size then
				print_builtin_com_failure_character_size_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_com_failure_cwin_error_text then
				print_builtin_com_failure_cwin_error_text_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_function_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "FUNCTION".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_function: a_feature.is_builtin_function_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_function_fast_item then
				print_builtin_function_fast_item_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_function_item then
				print_builtin_function_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_identified_routines_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "IDENTIFIED_ROUTINES".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_identified_routines: a_feature.is_builtin_identified_routines_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_current_object_id then
				print_builtin_identified_routines_eif_current_object_id_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_id_object then
				print_builtin_identified_routines_eif_id_object_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_is_object_id_of_current then
				print_builtin_identified_routines_eif_is_object_id_of_current_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_identified_routines_eif_object_id then
				print_builtin_identified_routines_eif_object_id_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_integer_n_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "INTEGER_N" and "NATURAL_N".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n: a_feature.is_builtin_integer_n_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_8 then
				print_builtin_integer_n_as_integer_8_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_16 then
				print_builtin_integer_n_as_integer_16_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_32 then
				print_builtin_integer_n_as_integer_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_integer_64 then
				print_builtin_integer_n_as_integer_64_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_8 then
				print_builtin_integer_n_as_natural_8_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_16 then
				print_builtin_integer_n_as_natural_16_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_32 then
				print_builtin_integer_n_as_natural_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_as_natural_64 then
				print_builtin_integer_n_as_natural_64_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_and then
				print_builtin_integer_n_bit_and_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_not then
				print_builtin_integer_n_bit_not_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_or then
				print_builtin_integer_n_bit_or_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_shift_left then
				print_builtin_integer_n_bit_shift_left_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_shift_right then
				print_builtin_integer_n_bit_shift_right_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_bit_xor then
				print_builtin_integer_n_bit_xor_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_identity then
				print_builtin_integer_n_identity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_integer_quotient then
				print_builtin_integer_n_integer_quotient_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_integer_remainder then
				print_builtin_integer_n_integer_remainder_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_is_less then
				print_builtin_integer_n_is_less_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_minus then
				print_builtin_integer_n_minus_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_opposite then
				print_builtin_integer_n_opposite_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_plus then
				print_builtin_integer_n_plus_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_power then
				print_builtin_integer_n_power_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_product then
				print_builtin_integer_n_product_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_quotient then
				print_builtin_integer_n_quotient_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_character_8 then
				print_builtin_integer_n_to_character_8_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_character_32 then
				print_builtin_integer_n_to_character_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_double then
				print_builtin_integer_n_to_double_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real then
				print_builtin_integer_n_to_real_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real_32 then
				print_builtin_integer_n_to_real_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_integer_n_to_real_64 then
				print_builtin_integer_n_to_real_64_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_integer_n_ref_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "INTEGER_N_REF" and "NATURAL_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_integer_n_ref: a_feature.is_builtin_integer_n_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_integer_n_ref_item then
				print_builtin_integer_n_ref_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_ise_runtime_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "ISE_RUNTIME".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_ise_runtime: a_feature.is_builtin_ise_runtime_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_ise_runtime_attached_type then
				print_builtin_ise_runtime_attached_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_boolean_field then
				print_builtin_ise_runtime_boolean_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_boolean_field_at then
				print_builtin_ise_runtime_boolean_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_8_field then
				print_builtin_ise_runtime_character_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_8_field_at then
				print_builtin_ise_runtime_character_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_32_field then
				print_builtin_ise_runtime_character_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_character_32_field_at then
				print_builtin_ise_runtime_character_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_check_assert then
				print_builtin_ise_runtime_check_assert_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_compiler_version then
				print_builtin_ise_runtime_compiler_version_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_detachable_type then
				print_builtin_ise_runtime_detachable_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_dynamic_type then
				print_builtin_ise_runtime_dynamic_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_dynamic_type_at_offset then
				print_builtin_ise_runtime_dynamic_type_at_offset_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_eif_gen_param_id then
				print_builtin_ise_runtime_eif_gen_param_id_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_count_of_type then
				print_builtin_ise_runtime_field_count_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_name_of_type then
				print_builtin_ise_runtime_field_name_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_offset_of_type then
				print_builtin_ise_runtime_field_offset_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_static_type_of_type then
				print_builtin_ise_runtime_field_static_type_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_field_type_of_type then
				print_builtin_ise_runtime_field_type_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generating_type_of_type then
				print_builtin_ise_runtime_generating_type_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generating_type_8_of_type then
				print_builtin_ise_runtime_generating_type_8_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generator_of_type then
				print_builtin_ise_runtime_generator_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generator_8_of_type then
				print_builtin_ise_runtime_generator_8_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_generic_parameter_count then
				print_builtin_ise_runtime_generic_parameter_count_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_8_field then
				print_builtin_ise_runtime_integer_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_8_field_at then
				print_builtin_ise_runtime_integer_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_16_field then
				print_builtin_ise_runtime_integer_16_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_16_field_at then
				print_builtin_ise_runtime_integer_16_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_32_field then
				print_builtin_ise_runtime_integer_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_32_field_at then
				print_builtin_ise_runtime_integer_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_64_field then
				print_builtin_ise_runtime_integer_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_integer_64_field_at then
				print_builtin_ise_runtime_integer_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_attached_type then
				print_builtin_ise_runtime_is_attached_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_copy_semantics_field then
				print_builtin_ise_runtime_is_copy_semantics_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_expanded then
				print_builtin_ise_runtime_is_expanded_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_field_expanded_of_type then
				print_builtin_ise_runtime_is_field_expanded_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_field_transient_of_type then
				print_builtin_ise_runtime_is_field_transient_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_object_marked then
				print_builtin_ise_runtime_is_object_marked_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special then
				print_builtin_ise_runtime_is_special_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_copy_semantics_item then
				print_builtin_ise_runtime_is_special_copy_semantics_item_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_expanded then
				print_builtin_ise_runtime_is_special_of_expanded_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference then
				print_builtin_ise_runtime_is_special_of_reference_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference_or_basic_type then
				print_builtin_ise_runtime_is_special_of_reference_or_basic_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_special_of_reference_type then
				print_builtin_ise_runtime_is_special_of_reference_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_tuple then
				print_builtin_ise_runtime_is_tuple_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_is_tuple_type then
				print_builtin_ise_runtime_is_tuple_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_8_field then
				print_builtin_ise_runtime_natural_8_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_8_field_at then
				print_builtin_ise_runtime_natural_8_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_16_field then
				print_builtin_ise_runtime_natural_16_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_16_field_at then
				print_builtin_ise_runtime_natural_16_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_32_field then
				print_builtin_ise_runtime_natural_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_32_field_at then
				print_builtin_ise_runtime_natural_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_64_field then
				print_builtin_ise_runtime_natural_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_natural_64_field_at then
				print_builtin_ise_runtime_natural_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_instance_of then
				print_builtin_ise_runtime_new_instance_of_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_special_of_reference_instance_of then
				print_builtin_ise_runtime_new_special_of_reference_instance_of_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_tuple_instance_of then
				print_builtin_ise_runtime_new_tuple_instance_of_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_new_type_instance_of then
				print_builtin_ise_runtime_new_type_instance_of_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_object_size then
				print_builtin_ise_runtime_object_size_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_once_objects then
				print_builtin_ise_runtime_once_objects_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_persistent_field_count_of_type then
				print_builtin_ise_runtime_persistent_field_count_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_pointer_field then
				print_builtin_ise_runtime_pointer_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_pointer_field_at then
				print_builtin_ise_runtime_pointer_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_raw_reference_field_at then
				print_builtin_ise_runtime_raw_reference_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_raw_reference_field_at_offset then
				print_builtin_ise_runtime_raw_reference_field_at_offset_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_32_field then
				print_builtin_ise_runtime_real_32_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_32_field_at then
				print_builtin_ise_runtime_real_32_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_64_field then
				print_builtin_ise_runtime_real_64_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_real_64_field_at then
				print_builtin_ise_runtime_real_64_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field then
				print_builtin_ise_runtime_reference_field_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field_at then
				print_builtin_ise_runtime_reference_field_at_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_reference_field_at_offset then
				print_builtin_ise_runtime_reference_field_at_offset_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_storable_version_of_type then
				print_builtin_ise_runtime_storable_version_of_type_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_type_conforms_to then
				print_builtin_ise_runtime_type_conforms_to_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_ise_runtime_type_id_from_name then
				print_builtin_ise_runtime_type_id_from_name_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_memory_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "MEMORY".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_memory: a_feature.is_builtin_memory_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_memory_find_referers then
				print_builtin_memory_find_referers_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_platform_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "PLATFORM".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_platform: a_feature.is_builtin_platform_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_platform_boolean_bytes then
				print_builtin_platform_boolean_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_character_bytes then
				print_builtin_platform_character_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_double_bytes then
				print_builtin_platform_double_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_integer_bytes then
				print_builtin_platform_integer_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_64_bits then
				print_builtin_platform_is_64_bits_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_dotnet then
				print_builtin_platform_is_dotnet_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_mac then
				print_builtin_platform_is_mac_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_scoop_capable then
				print_builtin_platform_is_scoop_capable_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_thread_capable then
				print_builtin_platform_is_thread_capable_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_unix then
				print_builtin_platform_is_unix_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_vms then
				print_builtin_platform_is_vms_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_vxworks then
				print_builtin_platform_is_vxworks_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_is_windows then
				print_builtin_platform_is_windows_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_pointer_bytes then
				print_builtin_platform_pointer_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_real_bytes then
				print_builtin_platform_real_bytes_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_platform_wide_character_bytes then
				print_builtin_platform_wide_character_bytes_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_pointer_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "POINTER".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer: a_feature.is_builtin_pointer_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_hash_code then
				print_builtin_pointer_hash_code_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_is_default_pointer then
				print_builtin_pointer_is_default_pointer_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_plus then
				print_builtin_pointer_plus_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_pointer_to_integer_32 then
				print_builtin_pointer_to_integer_32_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_pointer_ref_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "POINTER_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_pointer_ref: a_feature.is_builtin_pointer_ref_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_pointer_ref_item then
				print_builtin_pointer_ref_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_real_n_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "REAL_N".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_target_type_not_void: a_target_type /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n: a_feature.is_builtin_real_n_class
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ceiling_real_32 then
				print_builtin_real_n_ceiling_real_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ceiling_real_64 then
				print_builtin_real_n_ceiling_real_64_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_floor_real_32 then
				print_builtin_real_n_floor_real_32_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_floor_real_64 then
				print_builtin_real_n_floor_real_64_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_identity then
				print_builtin_real_n_identity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_equal then
				print_builtin_real_n_ieee_is_equal_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_greater then
				print_builtin_real_n_ieee_is_greater_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_greater_equal then
				print_builtin_real_n_ieee_is_greater_equal_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_less then
				print_builtin_real_n_ieee_is_less_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_is_less_equal then
				print_builtin_real_n_ieee_is_less_equal_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_maximum_number then
				print_builtin_real_n_ieee_maximum_number_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ieee_minimum_number then
				print_builtin_real_n_ieee_minimum_number_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_is_less then
				print_builtin_real_n_is_less_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_is_nan then
				print_builtin_real_n_is_nan_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_is_negative_infinity then
				print_builtin_real_n_is_negative_infinity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_is_positive_infinity then
				print_builtin_real_n_is_positive_infinity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_minus then
				print_builtin_real_n_minus_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_opposite then
				print_builtin_real_n_opposite_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_plus then
				print_builtin_real_n_plus_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_power then
				print_builtin_real_n_power_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_product then
				print_builtin_real_n_product_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_quotient then
				print_builtin_real_n_quotient_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_to_double then
				print_builtin_real_n_to_double_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_integer then
				print_builtin_real_n_truncated_to_integer_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_integer_64 then
				print_builtin_real_n_truncated_to_integer_64_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_truncated_to_real then
				print_builtin_real_n_truncated_to_real_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_real_n_ref_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in classes "REAL_N_REF".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_target_type_not_void: a_target_type /= Void
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_real_n_ref: a_feature.is_builtin_real_n_ref_class
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_real_n_ref_item then
				print_builtin_real_n_ref_item_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ref_nan then
				print_builtin_real_n_ref_nan_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ref_negative_infinity then
				print_builtin_real_n_ref_negative_infinity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_real_n_ref_positive_infinity then
				print_builtin_real_n_ref_positive_infinity_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_special_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "SPECIAL".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_special: a_feature.is_builtin_special_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_special_base_address then
				print_builtin_special_base_address_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_capacity then
				print_builtin_special_capacity_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_count then
				print_builtin_special_count_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_element_size then
				print_builtin_special_element_size_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_special_item then
				print_builtin_special_item_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_tuple_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "TUPLE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_tuple: a_feature.is_builtin_tuple_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_tuple_count then
				print_builtin_tuple_count_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_tuple_object_comparison then
				print_builtin_tuple_object_comparison_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

	print_builtin_type_query_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call to query `a_feature' (static binding).
			-- `a_feature' is a built-in feature introduced in class "TYPE".
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
			-- Note that the result of the query is not adapted to match the kind
			-- of result type expected by the caller. It is recommended to use
			-- `print_adapted_query_call' whenever possible.
		require
			a_feature_not_void: a_feature /= Void
			a_feature_is_query: a_feature.is_query
			a_feature_is_builtin: a_feature.is_builtin
			a_feature_is_builtin_type: a_feature.is_builtin_type_class
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			inspect a_feature.builtin_feature_code
			when {ET_TOKEN_CODES}.builtin_type_default then
				print_builtin_type_default_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_generic_parameter_count then
				print_builtin_type_generic_parameter_count_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_has_default then
				print_builtin_type_has_default_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_is_attached then
				print_builtin_type_is_attached_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_is_deferred then
				print_builtin_type_is_deferred_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_is_expanded then
				print_builtin_type_is_expanded_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_type_id then
				print_builtin_type_type_id_call (a_feature, a_target_type, a_check_void_target)
			when {ET_TOKEN_CODES}.builtin_type_runtime_name then
				print_builtin_type_runtime_name_call (a_feature, a_target_type, a_check_void_target)
			else
				print_non_inlined_query_call (a_feature, a_target_type, a_check_void_target)
			end
		end

feature {NONE} -- Agent generation

	print_agent (an_agent: ET_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		local
			l_temp: detachable ET_IDENTIFIER
			l_temp_index: INTEGER
			i, nb: INTEGER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_assignment_target: like assignment_target
			l_agent_type: ET_DYNAMIC_PRIMARY_TYPE
			nb_operands: INTEGER
			l_operand: ET_EXPRESSION
			l_operand_type_set: ET_DYNAMIC_TYPE_SET
		do
			l_assignment_target := assignment_target
			assignment_target := Void
			l_dynamic_type_set := dynamic_type_set (an_agent)
			l_agent_type := l_dynamic_type_set.static_type.primary_type
			if not an_agent.is_qualified_call then
				print_attachment_operand (tokens.current_keyword, current_type)
				nb_operands := 1
			elseif attached {ET_EXPRESSION} an_agent.target as l_closed_operand then
				print_attachment_operand (l_closed_operand, dynamic_type_set (l_closed_operand))
				nb_operands := 1
			end
			if attached an_agent.arguments as l_arguments then
				nb := l_arguments.count
				from i := 1 until i > nb loop
					if attached {ET_EXPRESSION} l_arguments.actual_argument (i) as l_closed_operand then
						print_attachment_operand (l_closed_operand, dynamic_type_set (l_closed_operand))
						nb_operands := nb_operands + 1
					end
					i := i + 1
				end
			end
			fill_call_operands (nb_operands)
			current_agents.force_last (an_agent, index_offset)
			if in_operand then
				print_indentation
				if l_assignment_target /= Void then
					operand_stack.force (l_assignment_target)
					print_writable (l_assignment_target)
				else
					l_temp := new_temp_variable (l_agent_type)
					mark_temp_variable_frozen (l_temp)
						-- We will set the index of `l_temp' later because
						-- it could still be used in `call_operands'.
					l_temp_index := an_agent.index
					operand_stack.force (l_temp)
					print_temp_name (l_temp, current_file)
				end
				print_assign_to
			end
			print_agent_creation_name (current_agents.count, current_feature, current_type, current_file)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			from i := 1 until i > nb_operands loop
				print_comma
				l_operand := call_operands.item (i)
				l_operand_type_set := dynamic_type_set (l_operand)
					-- Note that when some closed operands have the copy semantics, they will be cloned
					-- twice: once when creating the agent (here) because they will be used to create
					-- the tuple `ROUTINE.closed_operands`, and once when calling the agent because they
					-- will be used as actual arguments of the underlying feature.
				print_attachment_expression (l_operand, l_operand_type_set, l_operand_type_set.static_type)
				i := i + 1
			end
			current_file.put_character (')')
			if in_operand then
				print_semicolon_newline
			end
			call_operands.wipe_out
			if l_temp /= Void then
				mark_temp_variable_unfrozen (l_temp)
				if l_temp_index /= 0 then
						-- We had to wait until this stage to set the index of `l_temp'
						-- because it could have still been used in `call_operands'.
					l_temp.set_index (l_temp_index)
				end
			end
		end

	print_call_agent (an_agent: ET_CALL_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_agent (an_agent)
		end

	print_do_function_inline_agent (an_agent: ET_DO_FUNCTION_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_agent (an_agent)
		end

	print_do_procedure_inline_agent (an_agent: ET_DO_PROCEDURE_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_agent (an_agent)
		end

	print_external_function_inline_agent (an_agent: ET_EXTERNAL_FUNCTION_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_agent (an_agent)
		end

	print_external_procedure_inline_agent (an_agent: ET_EXTERNAL_PROCEDURE_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_agent (an_agent)
		end

	print_once_function_inline_agent (an_agent: ET_ONCE_FUNCTION_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			if an_agent.is_once_per_object then
error_handler.report_warning_message ("ET_C_GENERATOR.print_once_function_inline_agent: once key %"OBJECT%" not supported.")
			end
			print_agent (an_agent)
		end

	print_once_procedure_inline_agent (an_agent: ET_ONCE_PROCEDURE_INLINE_AGENT)
			-- Print `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			if an_agent.is_once_per_object then
error_handler.report_warning_message ("ET_C_GENERATOR.print_once_procedure_inline_agent: once key %"OBJECT%" not supported.")
			end
			print_agent (an_agent)
		end

	print_agent_open_operand (a_name: ET_IDENTIFIER)
			-- Print agent open operand `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_open_operand: a_name.is_agent_open_operand
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
			l_agent: like current_agent
		do
			if in_operand then
				if in_call_target and then a_name = agent_target then
					print_agent_open_target_operand (a_name)
				else
					operand_stack.force (a_name)
				end
			else
					-- Make sure that formal arguments are interpreted in the context
					-- of `current_feature' and not in the context of `current_agent'.
					-- For that we temporarily set `current_agent' to Void.
					-- It's reset to its original value when existing this routine.
				l_agent := current_agent
				current_agent := Void
				if attached call_target_type as l_call_target_type then
					check in_call_target: in_call_target end
					l_dynamic_type_set := dynamic_type_set (a_name)
					l_static_type := l_dynamic_type_set.static_type
					if l_static_type.is_expanded then
							-- Pass the address of the expanded object.
						current_file.put_character ('&')
						print_agent_open_operand_access (a_name)
					elseif l_call_target_type.is_expanded then
							-- We need to unbox the object and then pass its address.
						print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
					else
						print_agent_open_operand_access (a_name)
					end
				else
					print_agent_open_operand_access (a_name)
				end
				current_agent := l_agent
			end
		end

	print_agent_closed_operand (a_name: ET_IDENTIFIER)
			-- Print agent closed operand `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_closed_operand: a_name.is_agent_closed_operand
		local
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_static_type: ET_DYNAMIC_TYPE
			l_agent: like current_agent
		do
				-- Make sure that formal arguments are interpreted in the context
				-- of `current_feature' and not in the context of `current_agent'.
				-- For that we temporarily set `current_agent' to Void.
				-- It's reset to its original value when existing this routine.
			l_agent := current_agent
			current_agent := Void
			if in_operand then
				operand_stack.force (a_name)
			elseif attached call_target_type as l_call_target_type then
				check in_call_target: in_call_target end
				l_dynamic_type_set := dynamic_type_set (a_name)
				l_static_type := l_dynamic_type_set.static_type
				if l_static_type.is_expanded then
						-- Pass the address of the expanded object.
					current_file.put_character ('&')
					print_agent_closed_operand_access (a_name, formal_argument (1))
				elseif l_call_target_type.is_expanded then
						-- We need to unbox the object and then pass its address.
					print_boxed_attribute_pointer_access (a_name, l_call_target_type, call_target_check_void)
				else
					print_agent_closed_operand_access (a_name, formal_argument (1))
				end
			else
				print_agent_closed_operand_access (a_name, formal_argument (1))
			end
			current_agent := l_agent
		end

	print_agent_open_target_operand (a_name: ET_IDENTIFIER)
			-- Print agent open operand `a_name' when it's the target
			-- of the call to the feature of the agent.
		require
			a_name_not_void: a_name /= Void
			a_name_open_operand: a_name.is_agent_open_operand
			in_operand: in_operand
			in_call_target: in_call_target
			is_agent_target: a_name = agent_target
		local
			l_tuple: ET_IDENTIFIER
			l_old_index: INTEGER
			l_agent: like current_agent
			l_tuple_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
			l_tuple_label: ET_IDENTIFIER
			l_tuple_item_expression: ET_QUALIFIED_CALL_EXPRESSION
			l_query_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_tuple_item_type_set: ET_DYNAMIC_TYPE_SET
			l_tuple_item_static_type: ET_DYNAMIC_PRIMARY_TYPE
		do
				-- Make sure that formal arguments are interpreted in the context
				-- of `current_feature' and not in the context of `current_agent'.
				-- For that we temporarily set `current_agent' to Void.
				-- It's reset to its original value when existing this routine.
			l_agent := current_agent
			current_agent := Void
				-- We need to use the  first item of the tuple containing the open operands
				-- as target of the call, not a clone of this object even if it has copy semantics.
				-- Extract it from the tuple containing the open operands, if any.
			l_tuple_item_type_set := dynamic_type_set (a_name)
			if
				l_agent /= Void and then
				attached {ET_DYNAMIC_ROUTINE_TYPE} dynamic_type_set (l_agent).static_type.primary_type as l_agent_type and then
				attached l_agent_type.open_operand_tuple_type_set as l_open_operand_tuple_type_set and then
				attached call_target_type as l_call_target_type and then
				l_tuple_item_type_set.has_expanded
			then
				l_tuple := formal_argument (2)
				l_tuple_type_set := new_standalone_type_set (l_open_operand_tuple_type_set.static_type)
				l_tuple_type_set.put_types (l_open_operand_tuple_type_set)
				l_tuple_type_set.set_never_void
				extra_dynamic_type_sets.force_last (l_tuple_type_set)
				l_old_index := l_tuple.index
				l_tuple.set_index (current_dynamic_type_sets.count + extra_dynamic_type_sets.count)
				print_indentation_if
				print_argument_name (l_tuple, current_file)
				print_then_newline
				indent
				l_tuple_label := new_tuple_label (1)
				l_tuple_item_expression := new_qualified_call_expression (l_tuple, l_tuple_label, Void)
				l_tuple_item_expression.set_index (a_name.index)
					-- Register the call to extract the tuple item so that
					-- it is handled correctly in case of polymorphism.
					-- The special treatment for polymorphism only occurs
					-- when the target has more than 2 possible dynamic types.
				if l_tuple_type_set.count > 2 then
					create l_query_call.make (l_tuple_item_expression, l_tuple_type_set, l_tuple_item_type_set, current_feature, current_feature.static_feature, current_type, current_index, index_offset)
					l_tuple_type_set.static_type.primary_type.put_query_call (l_query_call)
				end
				if l_agent_type.result_type_set = Void then
					print_procedure_target_operand (l_tuple_item_expression, l_call_target_type)
				else
					print_target_operand (l_tuple_item_expression, l_call_target_type)
				end
					-- Clean up.
				l_tuple.set_index (l_old_index)
				extra_dynamic_type_sets.remove_last
				if l_query_call = Void then
					free_qualified_call_expression (l_tuple_item_expression)
					free_tuple_label (l_tuple_label)
					free_standalone_type_set (l_tuple_type_set)
				end
				dedent
				print_indentation_else_newline
				indent
				if
					attached {ET_IDENTIFIER} operand_stack.item as l_temp and then
					l_temp.is_temporary and then
					is_temp_variable_known (l_temp) and then
					attached temp_variable_type (l_temp) as l_temp_type
				then
					print_indentation
					print_temp_name (l_temp, current_file)
					print_assign_to
					l_tuple_item_static_type := l_tuple_item_type_set.static_type.primary_type
					if l_temp_type.is_expanded then
						if l_tuple_item_static_type.is_expanded then
							print_agent_open_operand_access (a_name)
						else
								-- We need to unbox the object.
							print_boxed_attribute_item_access (a_name, l_tuple_item_static_type, call_target_check_void)
						end
					else
						if l_tuple_item_static_type.is_expanded then
								-- We need to box the object.
							print_boxed_expression (a_name, l_tuple_item_static_type)
						else
								-- Already boxed.
							print_agent_open_operand_access (a_name)
						end
					end
					print_semicolon_newline
				else
						-- Internal error: a temporary variable should have properly been declared
						-- when printing `l_tuple_item_expression'.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_agent_open_target_operand", 1, "invalid temporary variable.")
				end
				dedent
				print_indentation_end_newline
			else
				operand_stack.force (a_name)
			end
			current_agent := l_agent
		end

	print_agent_open_operand_access (a_name: ET_IDENTIFIER)
			-- Print to `current_file' an access to agent open operand `a_name'.
		require
			a_name_not_void: a_name /= Void
			a_name_open_operand: a_name.is_agent_open_operand
		do
				-- The first two formal arguments are the tuple containing the closed
				-- operands and the tuple containing the open operands (if the latter
				-- is not a manifest tupe and is not empty, null otherwise).
			print_argument_name (formal_argument (a_name.seed + 2), current_file)
		end

	print_agent_closed_operand_access (a_name: ET_IDENTIFIER; a_target: ET_EXPRESSION)
			-- Print to `current_file' an access to agent closed operand `a_name' applied to `a_target'.
			-- `a_target' is meant to be the Tuple corresponding to the attribute 'closed_operands'
			-- in the Agent type.
		require
			a_name_not_void: a_name /= Void
			a_name_closed_operand: a_name.is_agent_closed_operand
			a_target_not_void: a_target /= Void
		do
			print_attribute_tuple_item_access (a_name.seed, a_target, agent_closed_operands_type, False)
		end

	print_agent_declaration (i: INTEGER; a_agent: ET_AGENT)
			-- Print declaration of `i'-th agent `a_agent'.
		require
			a_agent_not_void: a_agent /= Void
		local
			l_open_operand: ET_IDENTIFIER
			l_closed_operand: ET_IDENTIFIER
			nb_open_operands: INTEGER
			nb_closed_operands: INTEGER
			l_target: ET_AGENT_TARGET
			l_arguments: detachable ET_AGENT_ARGUMENT_OPERANDS
			l_argument: ET_AGENT_ARGUMENT_OPERAND
			j, nb: INTEGER
			old_closed_operands_type: ET_DYNAMIC_PRIMARY_TYPE
			l_parameters: ET_ACTUAL_PARAMETER_LIST
			l_tuple_type: ET_TUPLE_TYPE
		do
				--
				-- Determine agent type.
				--
			if not attached {ET_DYNAMIC_ROUTINE_TYPE} dynamic_type_set (a_agent).static_type.primary_type as l_agent_type then
					-- Internal error: the type of `a_agent' should be an Agent type.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_agent_declaration", 1, "type of agent is not an agent type.")
			else
					--
					-- Determine open and closed operands.
					--
				l_target := a_agent.target
				if l_target.is_open_operand then
					nb_open_operands := 1
					l_open_operand := agent_open_operand (nb_open_operands)
					l_open_operand.set_index (l_target.index)
				else
					nb_closed_operands := 1
					l_closed_operand := agent_closed_operand (nb_closed_operands)
					l_closed_operand.set_index (l_target.index)
				end
				l_arguments := a_agent.arguments
				if l_arguments /= Void then
					nb := l_arguments.count
					from j := 1 until j > nb loop
						l_argument := l_arguments.actual_argument (j)
						if l_argument.is_open_operand then
							nb_open_operands := nb_open_operands + 1
							l_open_operand := agent_open_operand (nb_open_operands)
							l_open_operand.set_index (l_argument.index)
						else
							nb_closed_operands := nb_closed_operands + 1
							l_closed_operand := agent_closed_operand (nb_closed_operands)
							l_closed_operand.set_index (l_argument.index)
						end
						j := j + 1
					end
				end
				old_closed_operands_type := agent_closed_operands_type
				create l_parameters.make_with_capacity (nb_closed_operands)
				from j := nb_closed_operands until j < 1 loop
					l_parameters.put_first (dynamic_type_set (agent_closed_operands.item (j)).static_type.base_type)
					j := j - 1
				end
				create l_tuple_type.make (tokens.implicit_attached_type_mark, l_parameters, l_agent_type.base_class.universe.tuple_type.named_base_class)
				agent_closed_operands_type := current_dynamic_system.dynamic_primary_type (l_tuple_type, current_system.any_type)
				print_agent_function_declaration (i, a_agent, l_agent_type, nb_open_operands, nb_closed_operands)
				print_agent_object_declaration (i, a_agent, l_agent_type, nb_open_operands, nb_closed_operands)
				agent_closed_operands_type := old_closed_operands_type
			end
		end

	print_agent_function_declaration (i: INTEGER; a_agent: ET_AGENT; a_agent_type: ET_DYNAMIC_ROUTINE_TYPE; nb_open_operands, nb_closed_operands: INTEGER)
			-- Print function to execute the `i'-th agent `a_agent' appearing in `current_feature`,
			-- to `current_file'.
		require
			a_agent_not_void: a_agent /= Void
			a_agent_type_not_void: a_agent_type /= Void
		local
			j, nb: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target: ET_AGENT_TARGET
			l_result: detachable ET_RESULT
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_arguments: detachable ET_AGENT_ARGUMENT_OPERANDS
			l_argument: ET_AGENT_ARGUMENT_OPERAND
			l_open_operand: ET_IDENTIFIER
			l_open_index: INTEGER
			l_closed_index: INTEGER
			old_file: KI_TEXT_OUTPUT_STREAM
		do
			old_file := current_file
			current_file := current_function_header_buffer
			current_file.put_character ('/')
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string ("Function for agent #")
			current_file.put_integer (i)
			current_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, current_file)
			current_file.put_character (' ')
			current_file.put_character ('*')
			current_file.put_character ('/')
			current_file.put_new_line
			l_result := a_agent.implicit_result
			if l_result = Void then
					-- Procedure.
				current_file.put_string (c_void)
			else
					-- Query or Tuple label.
				l_result_type := dynamic_type_set (l_result).static_type.primary_type
				print_type_declaration (l_result_type, current_file)
			end
			current_file.put_character (' ')
			print_agent_function_name (i, current_feature, current_type, current_file)
			current_file.put_character ('(')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
				-- Tuple with closed operands.
			print_type_declaration (agent_closed_operands_type, current_file)
			current_file.put_character (' ')
			print_argument_name (formal_argument (1), current_file)
			current_file.put_character (',')
			current_file.put_character (' ')
				-- Tuple with open operands, if any.
				-- Null if manifest tuple or empty tuple.
			if attached a_agent_type.open_operand_tuple_type_set as l_open_operand_tuple_type_set then
				print_type_declaration (l_open_operand_tuple_type_set.static_type.primary_type, current_file)
			else
				print_eif_any_type_name (current_file)
				current_file.put_character ('*')
			end
			current_file.put_character (' ')
			print_argument_name (formal_argument (2), current_file)
			from j := 1 until j > nb_open_operands loop
				l_open_operand := agent_open_operands.item (j)
				l_type := dynamic_type_set (l_open_operand).static_type.primary_type
				current_file.put_character (',')
				current_file.put_character (' ')
				print_type_declaration (l_type, current_file)
				current_file.put_character (' ')
				print_agent_open_operand_access (l_open_operand)
				j := j + 1
			end
			current_file.put_character (')')
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
				--
				-- Declaration of variables.
				--
			if a_agent.is_inline_agent and current_type.base_class.invariants_enabled then
				print_indentation
				current_file.put_string (c_uint32_t)
				current_file.put_character (' ')
				current_file.put_string (c_in_qualified_call)
				print_assign_to
				current_file.put_character ('1')
				print_semicolon_newline
			end
				-- Variable for exception trace.
			if exception_trace_mode then
				print_indentation
				current_file.put_string (c_ge_call)
				current_file.put_character (' ')
				current_file.put_string (c_tc)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('{')
				if current_in_exception_trace then
					print_current_name (current_file)
					current_file.put_character (',')
				end
				if l_result /= Void then
					print_escaped_string (tokens.capitalized_function_name)
					current_file.put_character (',')
					print_escaped_string (tokens.item_name)
				else
					print_escaped_string (tokens.capitalized_procedure_name)
					current_file.put_character (',')
					print_escaped_string (tokens.call_name)
				end
				current_file.put_character (',')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character ('}')
				current_file.put_character (';')
				current_file.put_new_line
			end
				-- Make sure that `agent_target' is set correctly
				-- before calling `print_agent_body_declaration'.
			l_target := a_agent.target
			if l_target.is_open_operand then
				agent_target := agent_open_operands.item (1)
			else
				agent_target := agent_closed_operands.item (1)
			end
				-- Make sure that `agent_arguments' is set correctly
				-- before calling `print_agent_body_declaration'.
			agent_arguments.wipe_out
			l_arguments := a_agent.arguments
			if l_arguments /= Void then
				nb := l_arguments.count
				if agent_arguments.capacity < nb then
					agent_arguments.resize (nb)
				end
				l_open_index := nb_open_operands
				l_closed_index := nb_closed_operands
				from j := nb until j < 1 loop
					l_argument := l_arguments.actual_argument (j)
					if l_argument.is_open_operand then
						agent_arguments.put_first (agent_open_operands.item (l_open_index))
						l_open_index := l_open_index - 1
					else
						agent_arguments.put_first (agent_closed_operands.item (l_closed_index))
						l_closed_index := l_closed_index - 1
					end
					j := j - 1
				end
			end
				--
				-- Instructions.
				--
			current_file := current_function_body_buffer
				-- Call stack.
			if exception_trace_mode then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_tc_address)
				current_file.put_character (';')
				current_file.put_new_line
			end
			print_agent_body_declaration (a_agent)
				-- Call stack.
			if exception_trace_mode then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_call)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_tc)
				current_file.put_character ('.')
				current_file.put_string (c_caller)
				current_file.put_character (';')
				current_file.put_new_line
			end
			if l_result_type /= Void then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Do not flush code to C file here (no call to `flush_to_c_file')
				-- because we want the function associated with this agent and the
				-- creation of the agent (printed just below) to be in the same C file.
				-- Indeed, the associated function is not declared in the C header file,
				-- but it is used in the creation of the agent.
			current_file := old_file
		end

	print_agent_object_declaration (i: INTEGER; a_agent: ET_AGENT; a_agent_type: ET_DYNAMIC_ROUTINE_TYPE; nb_open_operands, nb_closed_operands: INTEGER)
			-- Print function to create the agent object of `i'-th agent `a_agent' appearing
			-- in `current_feature`, to `current_file' and its signature to `header_file'.
		require
			a_agent_not_void: a_agent /= Void
			a_agent_type_not_void: a_agent_type /= Void
		local
			j: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target: ET_AGENT_TARGET
			old_file: KI_TEXT_OUTPUT_STREAM
		do
			old_file := current_file
				-- Print signature to `header_file' and `current_file'.
			current_file := current_function_body_buffer
			header_file.put_character ('/')
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string ("Creation of agent #")
			header_file.put_integer (i)
			header_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, header_file)
			header_file.put_character (' ')
			header_file.put_character ('*')
			header_file.put_character ('/')
			header_file.put_new_line
			current_file.put_character ('/')
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string ("Creation of agent #")
			current_file.put_integer (i)
			current_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, current_file)
			current_file.put_character (' ')
			current_file.put_character ('*')
			current_file.put_character ('/')
			current_file.put_new_line
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			print_type_declaration (a_agent_type, header_file)
			print_type_declaration (a_agent_type, current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_agent_creation_name (i, current_feature, current_type, header_file)
			print_agent_creation_name (i, current_feature, current_type, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			print_context_type_declaration (current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			header_file.put_string (c_ac)
			current_file.put_string (c_ac)
			from j := 1 until j > nb_closed_operands loop
				header_file.put_character (',')
				header_file.put_character (' ')
				current_file.put_character (',')
				current_file.put_character (' ')
				l_type := dynamic_type_set (agent_closed_operands.item (j)).static_type.primary_type
				print_type_declaration (l_type, header_file)
				print_type_declaration (l_type, current_file)
				header_file.put_character (' ')
				current_file.put_character (' ')
				header_file.put_character ('a')
				current_file.put_character ('a')
				header_file.put_integer (j)
				current_file.put_integer (j)
				j := j + 1
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print body to `current_file'.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			print_type_declaration (a_agent_type, current_file)
			current_file.put_character (' ')
			print_result_name (current_file)
			print_semicolon_newline
			print_indentation
			print_type_declaration (agent_closed_operands_type, current_file)
			current_file.put_character (' ')
			print_temp_name (temp_variable, current_file)
			print_semicolon_newline
			extra_dynamic_type_sets.force_last (agent_closed_operands_type)
			temp_variable.set_index (current_dynamic_type_sets.count + extra_dynamic_type_sets.count)
				-- Create agent object.
			print_indentation
			print_result_name (current_file)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_string (c_ge_new)
			current_file.put_integer (a_agent_type.id)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			print_comma
			current_file.put_string (c_eif_true)
			current_file.put_character (')')
			print_semicolon_newline
				-- Build 'closed_operands'.
			print_indentation
			print_temp_name (temp_variable, current_file)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_string (c_ge_new)
			current_file.put_integer (agent_closed_operands_type.id)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			print_comma
			current_file.put_string (c_eif_true)
			current_file.put_character (')')
			print_semicolon_newline
			from j := 1 until j > nb_closed_operands loop
				print_indentation
				print_agent_closed_operand_access (agent_closed_operands.item (j), temp_variable)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('a')
				current_file.put_integer (j)
				print_semicolon_newline
				j := j + 1
			end
				-- Call 'set_rout_disp_final'.
			if not attached a_agent_type.set_rout_disp_final_feature as l_set_rout_disp_final_feature then
					-- Internal error: the Agent type should have a procedure 'set_rout_disp_final'.
					-- See ET_DYNAMIC_SYSTEM.compile_kernel.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_agent_object_declaration", 1, "feature `set_rout_disp_final' not found in agent type.")
			elseif l_set_rout_disp_final_feature.is_static then
					-- Internal error: the procedure 'set_rout_disp_final' should not be instance-free.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_agent_object_declaration", 2, "feature `set_rout_disp_final' is an instance-free feature.")
			else
				register_called_feature (l_set_rout_disp_final_feature)
				print_indentation
				print_before_call (a_agent_type, False)
				print_routine_name (l_set_rout_disp_final_feature, a_agent_type, current_file)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				if a_agent_type.is_expanded then
					current_file.put_character ('&')
					print_result_name (current_file)
				else
					print_result_name (current_file)
				end
				print_comma
					-- 'a_rout_disp'.
				print_declaration_type_cast (current_dynamic_system.pointer_type, current_file)
				current_file.put_character ('&')
				print_agent_function_name (i, current_feature, current_type, current_file)
				print_comma
					-- 'a_encaps_rout_disp'.
				print_declaration_type_cast (current_dynamic_system.pointer_type, current_file)
				current_file.put_character ('&')
				print_agent_function_name (i, current_feature, current_type, current_file)
				print_comma
					-- 'a_calc_rout_addr'.
				print_declaration_type_cast (current_dynamic_system.pointer_type, current_file)
				current_file.put_character ('0')
				print_comma
					-- 'a_closed_operands'.
				print_temp_name (temp_variable, current_file)
				print_comma
					-- 'a_is_target_closed'.
				l_target := a_agent.target
				if l_target.is_open_operand then
					current_file.put_string (c_eif_false)
				else
					current_file.put_string (c_eif_true)
				end
				print_comma
					-- 'a_open_count'.
				current_file.put_integer (nb_open_operands)
				current_file.put_character (')')
				print_after_call (a_agent_type)
				print_semicolon_newline
			end
			extra_dynamic_type_sets.remove_last
				-- Return the agent object.
			print_indentation
			current_file.put_string (c_return)
			current_file.put_character (' ')
			print_result_name (current_file)
			print_semicolon_newline
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
		end

	print_agent_body_declaration (an_agent: ET_AGENT)
			-- Print body of declaration of agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		local
			l_old_agent: like current_agent
		do
			l_old_agent := current_agent
			current_agent := an_agent
			an_agent.process (Current)
			current_agent := l_old_agent
		end

	print_call_agent_body_declaration (an_agent: ET_CALL_AGENT)
			-- Print body of declaration of call agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		local
			l_result: detachable ET_RESULT
			l_arguments: detachable ET_AGENT_ARGUMENT_OPERANDS
			l_name: ET_FEATURE_NAME
			l_result_type_set: ET_DYNAMIC_TYPE_SET
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_type: ET_DYNAMIC_TYPE
			l_target_type_is_separate: BOOLEAN
			old_file: KI_TEXT_OUTPUT_STREAM
		do
			old_file := current_file
			l_name := an_agent.name
			l_arguments := an_agent.arguments
			l_result := an_agent.implicit_result
			l_target_type := dynamic_type_set (agent_target).static_type
			if use_scoop and l_target_type.is_separate then
				l_target_type_is_separate := True
				print_indentation
				current_file.put_string (c_ge_scoop_session)
				current_file.put_character ('*')
				current_file.put_character (' ')
				print_separate_argument_session_name (agent_inline_separate_argument.name, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_ge_scoop_region)
				current_file.put_character ('*')
				current_file.put_character (' ')
				current_file.put_string (c_sr)
				print_assign_to
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_region)
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_ge_rescue)
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ge_scoop_condition)
				current_file.put_character ('*')
				current_file.put_character (' ')
				current_file.put_string (c_scond)
				print_assign_to
				current_file.put_character ('0')
				print_semicolon_newline
				print_indentation
				print_type_declaration (l_target_type.primary_type, current_file)
				current_file.put_character (' ')
				print_inline_separate_argument_name (agent_inline_separate_argument.name, current_file)
				print_semicolon_newline
			end
			if l_result /= Void then
					-- Query or Tuple label.
				l_result_type_set := dynamic_type_set (l_result)
				current_file := current_function_header_buffer
				l_result_type := l_result_type_set.static_type.primary_type
				print_indentation
				print_type_declaration (l_result_type, current_file)
				current_file.put_character (' ')
				print_result_name (current_file)
				print_semicolon_newline
			end
			current_file := current_function_body_buffer
			if l_target_type_is_separate then
				print_indentation
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				current_file.put_string (c_ge_setjmp)
				current_file.put_character ('(')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_character ('j')
				current_file.put_character ('b')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('!')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('&')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_separate_argument_session_name (agent_inline_separate_argument.name, current_file)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_string (c_ge_scoop_session_close)
				current_file.put_character ('(')
				current_file.put_string (c_sr)
				print_comma
				print_separate_argument_session_name (agent_inline_separate_argument.name, current_file)
				current_file.put_character (')')
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_ge_jump_to_last_rescue)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			end
			if l_result /= Void then
					-- Note that we don't call `print_attachment_operand' or
					-- `print_assignment_operand' here because we don't want
					-- the result of the call to the feature `l_name' to be
					-- cloned (if it has the copy semantics) at this stage
					-- (e.g. `agent a.some_function_with_expanded_result`).
					-- This will be done later if the result is assigned to
					-- some entity (e.g. `x := my_agent.item ([])`).
					-- Hence setting 'no_source_twin'.
				agent_expression_assignment.set_no_source_twin (True)
				agent_expression_assignment.target.set_index (l_result.index)
				agent_expression.set_index (l_result.index)
				agent_expression.set_name (l_name)
				if l_arguments /= Void then
					agent_expression.set_arguments (agent_arguments)
				else
					agent_expression.set_arguments (Void)
				end
				if l_target_type_is_separate then
					agent_inline_separate_argument.set_expression (agent_target)
					agent_inline_separate_argument.name.set_index (agent_target.index)
					agent_expression.set_target (agent_inline_separate_argument.name)
					print_inline_separate_instruction (agent_controlled_target_expression)
				else
					agent_expression.set_target (agent_target)
					print_assignment (agent_expression_assignment)
				end
			else
					-- Procedure.
				agent_instruction.set_name (l_name)
				if l_arguments /= Void then
					agent_instruction.set_arguments (agent_arguments)
				else
					agent_instruction.set_arguments (Void)
				end
				if l_target_type_is_separate then
					agent_inline_separate_argument.set_expression (agent_target)
					agent_inline_separate_argument.name.set_index (agent_target.index)
					agent_instruction.set_target (agent_inline_separate_argument.name)
					print_inline_separate_instruction (agent_controlled_target_instruction)
				else
					agent_instruction.set_target (agent_target)
					print_qualified_call_instruction (agent_instruction)
				end
			end
			if l_target_type_is_separate then
				print_indentation
				current_file.put_string (c_ac)
				current_file.put_string (c_arrow)
				current_file.put_string (c_last_rescue)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character ('.')
				current_file.put_string (c_previous)
				print_semicolon_newline
			end
				-- Clean up.
			current_file := old_file
		end

	print_do_function_inline_agent_body_declaration (an_agent: ET_DO_FUNCTION_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_internal_routine_inline_agent_body_declaration (an_agent)
		end

	print_do_procedure_inline_agent_body_declaration (an_agent: ET_DO_PROCEDURE_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
			print_internal_routine_inline_agent_body_declaration (an_agent)
		end

	print_external_function_inline_agent_body_declaration (an_agent: ET_EXTERNAL_FUNCTION_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_external_function_inline_agent_body_declaration not implemented")
		end

	print_external_procedure_inline_agent_body_declaration (an_agent: ET_EXTERNAL_PROCEDURE_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_external_procedure_inline_agent_body_declaration not implemented")
		end

	print_once_function_inline_agent_body_declaration (an_agent: ET_ONCE_FUNCTION_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_once_function_inline_agent_body_declaration not implemented")
			print_internal_routine_inline_agent_body_declaration (an_agent)
		end

	print_once_procedure_inline_agent_body_declaration (an_agent: ET_ONCE_PROCEDURE_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		do
-- TODO
error_handler.report_warning_message ("ET_C_GENERATOR.print_once_procedure_inline_agent_body_declaration not implemented")
			print_internal_routine_inline_agent_body_declaration (an_agent)
		end

	print_internal_routine_inline_agent_body_declaration (an_agent: ET_INTERNAL_ROUTINE_INLINE_AGENT)
			-- Print body of declaration of inline agent `an_agent'.
		require
			an_agent_not_void: an_agent /= Void
		local
			l_result: detachable ET_RESULT
			l_result_type_set: ET_DYNAMIC_TYPE_SET
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
		do
-- TODO: handle case of once-routines
			print_agent_trace_message_call (an_agent, True)
			l_result := an_agent.implicit_result
			if l_result /= Void then
				l_result_type_set := dynamic_type_set (l_result)
				l_result_type := l_result_type_set.static_type.primary_type
			end
			print_internal_feature_body_declaration (an_agent, l_result_type, False)
			print_agent_trace_message_call (an_agent, False)
		end

	agent_instruction: ET_QUALIFIED_CALL_INSTRUCTION
			-- Agent instruction

	agent_expression: ET_QUALIFIED_CALL_EXPRESSION
			-- Agent expression

	agent_target: ET_EXPRESSION
			-- Agent target

	agent_arguments: ET_ACTUAL_ARGUMENT_LIST
			-- Agent arguments

	agent_expression_assignment: ET_ASSIGNMENT
			-- Assignment of agent expression to 'Result'

	agent_controlled_target_instruction: ET_INLINE_SEPARATE_INSTRUCTION
			-- Inline separate instruction in case of an agent target
			-- of separate type for a qualified procedure call (so that
			-- this target is controlled)

	agent_controlled_target_expression: ET_INLINE_SEPARATE_INSTRUCTION
			-- Inline separate instruction in case of an agent target
			-- of separate type for a qualified query call (so that
			-- this target is controlled)

	agent_inline_separate_argument: ET_INLINE_SEPARATE_ARGUMENT
			-- Inline separate argument for `agent_controlled_target_instruction'
			-- and `agent_controlled_target_expression'

	agent_open_operands: DS_ARRAYED_LIST [ET_IDENTIFIER]
			-- Agent open operands

	agent_open_operand (i: INTEGER): ET_IDENTIFIER
			-- `i'-th agent open operands
		require
			i_large_enough: i >= 1
		local
			j: INTEGER
		do
			if i <= agent_open_operands.count then
				Result := agent_open_operands.item (i)
			else
				from
					j := agent_open_operands.count + 1
					create Result.make ("a" + j.out)
					Result.set_agent_open_operand (True)
					Result.set_seed (j)
					agent_open_operands.force_last (Result)
					j := j + 1
				until
					j >= i
				loop
					create Result.make ("a" + j.out)
					Result.set_agent_open_operand (True)
					Result.set_seed (j)
					agent_open_operands.force_last (Result)
					j := j + 1
				end
			end
		ensure
			agent_open_operand_not_void: Result /= Void
			in_agent_open_operands: i <= agent_open_operands.count and then agent_open_operands.item (i) = Result
		end

	agent_closed_operands: DS_ARRAYED_LIST [ET_IDENTIFIER]
			-- Agent closed operands

	agent_closed_operand (i: INTEGER): ET_IDENTIFIER
			-- `i'-th agent closed operands
		require
			i_large_enough: i >= 1
		local
			j: INTEGER
		do
			if i <= agent_closed_operands.count then
				Result := agent_closed_operands.item (i)
			else
				from
					j := agent_closed_operands.count + 1
					create Result.make ("z" + j.out)
					Result.set_agent_closed_operand (True)
					Result.set_seed (j)
					agent_closed_operands.force_last (Result)
					j := j + 1
				until
					j >= i
				loop
					create Result.make ("z" + j.out)
					Result.set_agent_closed_operand (True)
					Result.set_seed (j)
					agent_closed_operands.force_last (Result)
					j := j + 1
				end
			end
		ensure
			agent_closed_operand_not_void: Result /= Void
			in_agent_closed_operands: i <= agent_closed_operands.count and then agent_closed_operands.item (i) = Result
		end

	agent_closed_operands_type: ET_DYNAMIC_PRIMARY_TYPE
			-- Type of attribute 'closed_operands' in Agent type

feature {NONE} -- Polymorphic call functions generation

	print_polymorphic_query_call_functions
			-- Print polymorphic query call functions.
		local
			l_dynamic_types: DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE]
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_first_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_last_call: ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_previous_call: ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_seed: INTEGER
			i, nb: INTEGER
			l_static_call: ET_CALL_COMPONENT
			l_force_result_boxing: BOOLEAN
			l_old_max_nested_inlining_count: INTEGER
		do
				-- No inlining.
			l_old_max_nested_inlining_count := max_nested_inlining_count
			max_nested_inlining_count := 0
			l_dynamic_types := dynamic_types
			nb := l_dynamic_types.count
			from i := 1 until i > nb loop
				l_type := l_dynamic_types.item (i)
				from
					l_first_call := l_type.query_calls
				until
					l_first_call = Void
				loop
					if not l_first_call.is_generated then
							-- It's not used in the generated C code.
							-- For example, polymorphic calls with one or two target
							-- dynamic types are handled differently.
							-- (See `print_qualified_call_expression'.)
						l_first_call := l_first_call.next
					elseif l_first_call.is_tuple_label then
							-- Polymorphic calls to Tuple labels are handled differently.
						l_first_call := l_first_call.next
					else
						l_static_call := l_first_call.static_call
						l_seed := l_static_call.name.seed
						from
							l_last_call := l_first_call
							l_previous_call := l_first_call
							l_call := l_first_call.next
							l_force_result_boxing := l_first_call.force_result_boxing
						until
							l_call = Void
						loop
							if
								l_call.is_generated and then
								(l_call.static_call.name.seed = l_seed and not l_call.is_tuple_label) and then
								l_force_result_boxing = l_call.force_result_boxing and then
								same_declared_signature (l_call, l_first_call)
							then
								if l_call /= l_last_call.next then
									l_previous_call.set_next (l_call.next)
									l_call.set_next (l_last_call.next)
									l_last_call.set_next (l_call)
									l_last_call := l_call
									l_call := l_previous_call
								else
									l_last_call := l_call
								end
							end
							l_previous_call := l_call
							l_call := l_call.next
						end
						print_polymorphic_call_function (l_first_call, l_last_call, l_type)
						l_first_call := l_last_call.next
					end
				end
				i := i + 1
			end
			max_nested_inlining_count := l_old_max_nested_inlining_count
		end

	print_polymorphic_tuple_label_call_functions
			-- Print polymorphic call to Tuple label functions.
		local
			l_dynamic_types: DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE]
			l_first_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_last_call: ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_call: detachable ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_previous_call: ET_DYNAMIC_QUALIFIED_QUERY_CALL
			l_seed: INTEGER
			i, nb: INTEGER
			l_static_call: ET_CALL_COMPONENT
			l_force_result_boxing: BOOLEAN
			l_old_max_nested_inlining_count: INTEGER
		do
				-- No inlining.
			l_old_max_nested_inlining_count := max_nested_inlining_count
			max_nested_inlining_count := 0
			l_dynamic_types := dynamic_types
			nb := l_dynamic_types.count
			from i := 1 until i > nb loop
				if attached {ET_DYNAMIC_TUPLE_TYPE} l_dynamic_types.item (i) as l_tuple_type then
					from
						l_first_call := l_tuple_type.query_calls
					until
						l_first_call = Void
					loop
						if not l_first_call.is_generated then
								-- It's not used in the generated C code.
								-- For example, polymorphic calls with one or two target
								-- dynamic types are handled differently.
								-- (See `print_qualified_call_expression'.)
							l_first_call := l_first_call.next
						elseif not l_first_call.is_tuple_label then
								-- Skip calls that are not to a Tuple label.
							l_first_call := l_first_call.next
						else
							l_static_call := l_first_call.static_call
							l_seed := l_static_call.name.seed
							from
								l_last_call := l_first_call
								l_previous_call := l_first_call
								l_call := l_first_call.next
								l_force_result_boxing := l_first_call.force_result_boxing
							until
								l_call = Void
							loop
								if
									l_call.is_generated and then
									(l_call.static_call.name.seed = l_seed and l_call.is_tuple_label) and then
									l_force_result_boxing = l_call.force_result_boxing and then
									same_declared_signature (l_call, l_first_call)
								then
									if l_call /= l_last_call.next then
										l_previous_call.set_next (l_call.next)
										l_call.set_next (l_last_call.next)
										l_last_call.set_next (l_call)
										l_last_call := l_call
										l_call := l_previous_call
									else
										l_last_call := l_call
									end
								end
								l_previous_call := l_call
								l_call := l_call.next
							end
							print_polymorphic_call_function (l_first_call, l_last_call, l_tuple_type)
							l_first_call := l_last_call.next
						end
					end
				end
				i := i + 1
			end
			max_nested_inlining_count := l_old_max_nested_inlining_count
		end

	print_polymorphic_procedure_call_functions
			-- Print polymorphic procedure call functions.
		local
			l_dynamic_types: DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE]
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_first_call: detachable ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL
			l_last_call: ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL
			l_call: detachable ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL
			l_previous_call: ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL
			l_seed: INTEGER
			i, nb: INTEGER
			l_static_call: ET_CALL_COMPONENT
			l_is_tuple_label: BOOLEAN
			l_old_max_nested_inlining_count: INTEGER
		do
				-- No inlining.
			l_old_max_nested_inlining_count := max_nested_inlining_count
			max_nested_inlining_count := 0
			l_dynamic_types := dynamic_types
			nb := l_dynamic_types.count
			from i := 1 until i > nb loop
				l_type := l_dynamic_types.item (i)
				from
					l_first_call := l_type.procedure_calls
				until
					l_first_call = Void
				loop
					if not l_first_call.is_generated then
							-- It's not used in the generated C code.
							-- For example, polymorphic calls with one or two target
							-- dynamic types are handled differently.
							-- (See `print_qualified_call_instruction'.)
						l_first_call := l_first_call.next
					else
						l_static_call := l_first_call.static_call
						l_seed := l_static_call.name.seed
						l_is_tuple_label := l_static_call.is_tuple_label
						from
							l_last_call := l_first_call
							l_previous_call := l_first_call
							l_call := l_first_call.next
						until
							l_call = Void
						loop
							if
								l_call.is_generated and then
								(l_call.static_call.name.seed = l_seed and l_call.is_tuple_label = l_is_tuple_label) and then
								same_declared_signature (l_call, l_first_call)
							then
								if l_call /= l_last_call.next then
									l_previous_call.set_next (l_call.next)
									l_call.set_next (l_last_call.next)
									l_last_call.set_next (l_call)
									l_last_call := l_call
									l_call := l_previous_call
								else
									l_last_call := l_call
								end
							end
							l_previous_call := l_call
							l_call := l_call.next
						end
						print_polymorphic_call_function (l_first_call, l_last_call, l_type)
						l_first_call := l_last_call.next
					end
				end
				i := i + 1
			end
			max_nested_inlining_count := l_old_max_nested_inlining_count
		end

	print_polymorphic_call_function (a_first_call, a_last_call: ET_DYNAMIC_QUALIFIED_CALL; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' dynamic binding code for the calls between `a_first_call'
			-- and `a_last_call' whose target static type is `a_target_type'.
			-- The generated code uses either a switch-statement or binary search to find out
			-- which feature to execute.
		require
			a_first_call_not_void: a_first_call /= Void
			a_last_call_not_void: a_last_call /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			l_static_call: ET_CALL_COMPONENT
			i, nb: INTEGER
			l_actual_arguments: detachable ET_ARGUMENT_OPERANDS
			l_formal_arguments_count: INTEGER
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_dynamic_type_ids: DS_ARRAYED_LIST [INTEGER]
			l_target_dynamic_types: DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER]
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_switch: BOOLEAN
			l_temp: ET_IDENTIFIER
			l_name: ET_IDENTIFIER
			l_call, l_last_call_next: detachable ET_DYNAMIC_QUALIFIED_CALL
			l_target_type_set: ET_DYNAMIC_TYPE_SET
			l_argument_type_sets: ET_DYNAMIC_STANDALONE_TYPE_SET_LIST
			l_formal_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_index: INTEGER
			l_feature: detachable ET_FEATURE
			l_seed: INTEGER
			l_manifest_tuple: detachable ET_MANIFEST_TUPLE
			l_manifest_tuple_operand: detachable ET_MANIFEST_TUPLE
			old_type: like current_type
			old_feature: like current_feature
			old_dynamic_type_sets: like current_dynamic_type_sets
			l_old_call_target_type: like call_target_type
			l_force_result_boxing: BOOLEAN
			l_old_current_index: INTEGER
			l_old_index_offset: INTEGER
			l_old_in_qualified_call: BOOLEAN
		do
			l_old_call_target_type := call_target_type
			call_target_type := Void
			l_old_in_qualified_call := in_qualified_call
			in_qualified_call := True
			old_type := current_type
			current_type := a_target_type
			old_feature := current_feature
			current_feature := dummy_feature
			current_feature.set_target_type (a_target_type)
			old_dynamic_type_sets := current_dynamic_type_sets
			l_argument_type_sets := standalone_type_sets
			current_dynamic_type_sets := l_argument_type_sets
			current_feature.set_dynamic_type_sets (current_dynamic_type_sets)
			l_old_current_index := current_index
			current_index := current_feature.current_index
			l_old_index_offset := index_offset
			index_offset := 0
			l_static_call := a_first_call.static_call
			l_actual_arguments := l_static_call.arguments
			if l_static_call.is_tuple_label then
				call_operands.wipe_out
				call_operands.force_last (tokens.current_keyword)
				l_index := l_static_call.name.seed
				if not attached {ET_DYNAMIC_TUPLE_TYPE} a_target_type as l_tuple_type then
						-- Internal error: a Tuple label can only be applied to a tuple object.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_polymorphic_call_function", 1, "type of tuple label is not 'TUPLE'.")
				elseif not l_tuple_type.item_type_sets.valid_index (l_index) then
						-- Internal error: invalid Tuple label.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_polymorphic_call_function", 2, "invalid tuple label.")
				elseif l_actual_arguments /= Void and then not l_actual_arguments.is_empty then
						-- This is a Tuple label setter.
					l_formal_arguments_count := 1
					call_operands.force_last (formal_argument (1))
				else
						-- This is a Tuple label getter.
					l_result_type := l_tuple_type.item_type_sets.item (l_index).static_type.primary_type
				end
			else
				l_feature := a_first_call.static_feature
				if l_feature = Void then
					call_operands.wipe_out
					call_operands.force_last (tokens.current_keyword)
						-- Internal error: there should be a feature with that seed.
						-- It has been computed in ET_FEATURE_FLATTENER.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_polymorphic_call_function", 3, "feature not found.")
				else
					if attached l_feature.type as l_feature_type then
						check is_query: l_feature.is_query end
						l_result_type := current_dynamic_system.dynamic_primary_type (l_feature_type, a_target_type.base_type)
					end
					l_formal_arguments := l_feature.arguments
					l_formal_arguments_count := l_feature.arguments_count
					fill_call_formal_arguments (l_feature)
						-- Check to see whether this is a call to ROUTINE.call or FUNCTION.item
						-- with a manifest tuple passed as argument. We have a special treatment
						-- in that case to avoid having to create the manifest tuple when possible.
					if l_formal_arguments_count = 1 then
						l_seed := l_static_call.name.seed
						if (l_seed = current_system.routine_call_seed or l_seed = current_system.function_item_seed) and then not l_static_call.is_call_agent then
							if l_actual_arguments /= Void and then l_actual_arguments.count = 1 then
								if attached {ET_MANIFEST_TUPLE} l_actual_arguments.actual_argument (1) as l_attached_manifest_tuple then
									l_manifest_tuple := l_attached_manifest_tuple
									nb := l_manifest_tuple.count
									l_manifest_tuple_operand := agent_manifest_tuple
									l_manifest_tuple_operand.resize (nb)
									from i := nb until i < 1 loop
										l_manifest_tuple_operand.put_first (formal_argument (i))
										i := i - 1
									end
									l_manifest_tuple_operand.set_index (nb + 1)
									call_operands.replace (l_manifest_tuple_operand, 2)
								end
							end
						end
					end
				end
			end
				-- Compute dynamic type sets.
			l_target_dynamic_type_ids := target_dynamic_type_ids
			l_target_dynamic_types := target_dynamic_types
			from
				l_call := a_first_call
				l_last_call_next := a_last_call.next
			until
				l_call = Void or else l_call = l_last_call_next
			loop
				l_target_type_set := l_call.target_type_set
				nb := l_target_type_set.count
				from i := 1 until i > nb loop
					l_dynamic_type := l_target_type_set.dynamic_type (i)
					if not l_target_dynamic_types.has (l_dynamic_type.id) then
						l_target_dynamic_type_ids.force_last (l_dynamic_type.id)
						l_target_dynamic_types.force_last_new (l_dynamic_type, l_dynamic_type.id)
					end
					i := i + 1
				end
				l_call := l_call.next
			end
			l_target_dynamic_type_ids.sort (dynamic_type_id_sorter)
				-- Print feature signature.
			print_call_name_comment (a_first_call, a_target_type, header_file)
			print_call_name_comment (a_first_call, a_target_type, current_file)
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			if l_result_type /= Void then
				if a_first_call.force_result_boxing then
						-- We want the result be boxed even though the result static type is expanded.
						-- This is needed when the call is an attribute, tuple label or "SPECIAL"
						-- item, the call result type is expanded, and it is the target of another call.
						-- In that case we force the result type to be of reference type in order
						-- to force the boxing of the result so that the other call is applied on
						-- the result object itself and not on a copy of this object.
					l_result_type := current_dynamic_system.any_type
					call_target_type := l_result_type
					l_force_result_boxing := True
				end
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
			else
				header_file.put_string (c_void)
				current_file.put_string (c_void)
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_call_name (a_first_call, a_target_type, l_force_result_boxing, header_file)
			print_call_name (a_first_call, a_target_type, l_force_result_boxing, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			print_type_declaration (a_target_type, header_file)
			print_type_declaration (a_target_type, current_file)
			if a_target_type.is_expanded then
				header_file.put_character ('*')
				current_file.put_character ('*')
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_current_name (header_file)
			print_current_name (current_file)
			if l_actual_arguments /= Void then
				nb := l_actual_arguments.count
				if nb /= l_formal_arguments_count then
						-- Internal error: we should have checked at this stage that
						-- the number of actual arguments should be the same as the
						-- number of formal arguments.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_polymorphic_call_function", 4, "wrong number of arguments.")
				elseif l_manifest_tuple /= Void then
					nb := l_manifest_tuple.count
					from i := 1 until i > nb loop
						header_file.put_character (',')
						current_file.put_character (',')
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_argument_type_set := dynamic_type_set_in_call (l_manifest_tuple.expression (i), a_first_call)
						l_argument_type := l_argument_type_set.static_type.primary_type
						print_type_declaration (l_argument_type, header_file)
						print_type_declaration (l_argument_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_name := formal_argument (i)
						print_argument_name (l_name, header_file)
						print_argument_name (l_name, current_file)
						i := i + 1
					end
				elseif nb > 0 then
					from i := 1 until i > nb loop
						header_file.put_character (',')
						current_file.put_character (',')
						header_file.put_character (' ')
						current_file.put_character (' ')
						l_argument_type_set := dynamic_type_set_in_call (l_actual_arguments.actual_argument (i), a_first_call)
						l_argument_type := l_argument_type_set.static_type.primary_type
						print_type_declaration (l_argument_type, header_file)
						print_type_declaration (l_argument_type, current_file)
						header_file.put_character (' ')
						current_file.put_character (' ')
						if l_formal_arguments /= Void then
							l_name := l_formal_arguments.formal_argument (i).name
						else
							l_name := formal_argument (i)
						end
						print_argument_name (l_name, header_file)
						print_argument_name (l_name, current_file)
						i := i + 1
					end
				end
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print feature body.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			if l_switch or l_target_dynamic_type_ids.count > 20 then
					-- Use switch statement.
				print_indentation
				current_file.put_string (c_switch)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_attribute_type_id_access (tokens.current_keyword, a_target_type, False)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				nb := l_target_dynamic_type_ids.count
				from i := 1 until i > nb loop
					l_dynamic_type := l_target_dynamic_types.item (l_target_dynamic_type_ids.item (i))
					print_indentation
					current_file.put_string (c_case)
					current_file.put_character (' ')
					current_file.put_integer (l_dynamic_type.id)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					if l_result_type /= Void then
						print_indentation
						current_file.put_string (c_return)
						current_file.put_character (' ')
						current_file.put_character ('(')
						set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
						print_adapted_named_query_call (l_static_call.name, l_dynamic_type, l_result_type, False)
						current_file.put_character (')')
						current_file.put_character (';')
						current_file.put_new_line
					else
						set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
						print_named_procedure_call (l_static_call.name, l_dynamic_type, False)
						print_indentation
						current_file.put_string (c_break)
						current_file.put_character (';')
						current_file.put_new_line
					end
					dedent
					i := i + 1
				end
				if l_result_type /= Void then
					print_indentation
					current_file.put_string (c_default)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_return)
					current_file.put_character (' ')
					print_default_entity_value (l_result_type, current_file)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
				end
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			else
				print_indentation
				current_file.put_string (c_eif_type_index)
				current_file.put_character (' ')
				l_temp := temp_variable
				print_temp_name (l_temp, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attribute_type_id_access (tokens.current_keyword, a_target_type, False)
				current_file.put_character (';')
				current_file.put_new_line
					-- Use binary search.
				print_binary_search_polymorphic_calls (a_first_call, a_last_call, l_result_type, 1, l_target_dynamic_type_ids.count, l_target_dynamic_type_ids, l_target_dynamic_types)
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
			call_operands.wipe_out
			if l_manifest_tuple_operand /= Void then
				l_manifest_tuple_operand.wipe_out
			end
			flush_to_c_file
			l_target_dynamic_type_ids.wipe_out
			l_target_dynamic_types.wipe_out
			current_dynamic_type_sets := old_dynamic_type_sets
			current_index := l_old_current_index
			index_offset := l_old_index_offset
			current_feature := old_feature
			current_type := old_type
			in_qualified_call := l_old_in_qualified_call
			call_target_type := l_old_call_target_type
		end

	print_binary_search_polymorphic_calls (a_first_call, a_last_call: ET_DYNAMIC_QUALIFIED_CALL; a_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE; l, u: INTEGER; a_target_dynamic_type_ids: DS_ARRAYED_LIST [INTEGER]; a_target_dynamic_types: DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER])
			-- Print to `current_file' dynamic binding code for the calls between `a_first_call'
			-- and `a_last_call' whose target dynamic types are those stored in `a_target_dynamic_types'
			-- whose type-id is itself stored between indexes `l' and `u' in `a_target_dynamic_type_ids'.
			-- The generated code uses binary search to find out which feature to execute.
		require
			a_first_call_not_void: a_first_call /= Void
			a_last_call_not_void: a_last_call /= Void
			a_target_dynamic_type_ids_not_void: a_target_dynamic_type_ids /= Void
			a_target_dynamic_types_not_void: a_target_dynamic_types /= Void
			no_void_target_dynamic_type: not a_target_dynamic_types.has_void_item
			consistent_count: a_target_dynamic_types.count = a_target_dynamic_type_ids.count
			l_large_enough: l >= 1
			l_small_enough: l <= u
			u_small_enough: u <= a_target_dynamic_type_ids.count
		local
			t: INTEGER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_type_id: INTEGER
			l_temp: ET_IDENTIFIER
			l_static_call: ET_CALL_COMPONENT
			l_argument_type_sets: ET_DYNAMIC_STANDALONE_TYPE_SET_LIST
		do
			l_argument_type_sets := standalone_type_sets
			l_static_call := a_first_call.static_call
			l_temp := temp_variable
			if l = u then
				l_type_id := a_target_dynamic_type_ids.item (l)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				print_indentation
				if a_result_type /= Void then
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_character ('(')
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_adapted_named_query_call (l_static_call.name, l_dynamic_type, a_result_type, False)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				else
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_named_procedure_call (l_static_call.name, l_dynamic_type, False)
				end
			elseif l + 1 = u then
				l_type_id := a_target_dynamic_type_ids.item (l)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp, current_file)
				current_file.put_character ('=')
				current_file.put_character ('=')
				current_file.put_integer (l_type_id)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_indentation
				if a_result_type /= Void then
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_character ('(')
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_adapted_named_query_call (l_static_call.name, l_dynamic_type, a_result_type, False)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				else
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_named_procedure_call (l_static_call.name, l_dynamic_type, False)
				end
				current_file.put_character ('}')
				current_file.put_character (' ')
				current_file.put_string (c_else)
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				l_type_id := a_target_dynamic_type_ids.item (u)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				print_indentation
				if a_result_type /= Void then
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_character ('(')
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_adapted_named_query_call (l_static_call.name, l_dynamic_type, a_result_type, False)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
				else
					set_polymorphic_call_argument_type_sets (l_argument_type_sets, a_first_call, a_last_call, l_dynamic_type)
					print_named_procedure_call (l_static_call.name, l_dynamic_type, False)
				end
				current_file.put_character ('}')
				current_file.put_new_line
			else
				t := l + (u - l) // 2
				l_type_id := a_target_dynamic_type_ids.item (t)
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp, current_file)
				current_file.put_character ('<')
				current_file.put_character ('=')
				current_file.put_integer (l_type_id)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_binary_search_polymorphic_calls (a_first_call, a_last_call, a_result_type, l, t, a_target_dynamic_type_ids, a_target_dynamic_types)
				current_file.put_character ('}')
				current_file.put_character (' ')
				current_file.put_string (c_else)
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_binary_search_polymorphic_calls (a_first_call, a_last_call, a_result_type, t + 1, u, a_target_dynamic_type_ids, a_target_dynamic_types)
				current_file.put_character ('}')
				current_file.put_new_line
			end
		end

	same_declared_signature (a_call1, a_call2: ET_DYNAMIC_QUALIFIED_CALL): BOOLEAN
			-- Do `a_call1' and `a_call2' have the same declared signature?
			-- Note1: Reference types are considered the same declared type.
			-- Note2: If the calls are idenfied to be to ROUTINE.call or FUNCTION.item
			-- and one of the calls has a manifest tuple as argument, then the other
			-- should also have a manifest tuple of the exact same type (note1 does not
			-- apply for the types of the manifest tuples here).
		require
			a_call1_not_void: a_call1 /= Void
			a_call2_not_void: a_call2 /= Void
			same_target_type: a_call1.target_type_set.static_type = a_call2.target_type_set.static_type
			same_seed: a_call1.static_call.name.seed = a_call2.static_call.name.seed
			same_kind: a_call1.is_tuple_label = a_call2.is_tuple_label
		local
			l_args1, l_args2: detachable ET_ARGUMENT_OPERANDS
			l_type_set1, l_type_set2: ET_DYNAMIC_TYPE_SET
			l_seed: INTEGER
			i, nb: INTEGER
		do
				-- The calls have the same seed with the same target type,
				-- so there result types if any are the same.
				-- Look at there actual argument types now.
			l_args1 := a_call1.static_call.arguments
			l_args2 := a_call2.static_call.arguments
			if l_args1 = Void then
				Result := (l_args2 = Void or else l_args2.is_empty)
			elseif l_args2 = Void then
				Result := l_args1.is_empty
			else
				nb := l_args1.count
				if l_args2.count = nb then
					Result := True
					from i := 1 until i > nb loop
						l_type_set1 := dynamic_type_set_in_call (l_args1.actual_argument (i), a_call1)
						l_type_set2 := dynamic_type_set_in_call (l_args2.actual_argument (i), a_call2)
						if nb = 1 then
							l_seed := a_call1.static_call.name.seed
							if not a_call1.is_tuple_label and then (l_seed = current_system.routine_call_seed or l_seed = current_system.function_item_seed) and then (not a_call1.is_call_agent and not a_call2.is_call_agent) then
								if attached {ET_MANIFEST_TUPLE} l_args1.actual_argument (1) then
									Result := attached {ET_MANIFEST_TUPLE} l_args2.actual_argument (1) and then (l_type_set1.static_type = l_type_set2.static_type)
								elseif attached {ET_MANIFEST_TUPLE} l_args2.actual_argument (1) then
									Result := False
								else
									Result := same_declaration_types (l_type_set1.static_type.primary_type, l_type_set2.static_type.primary_type)
								end
							else
								Result := same_declaration_types (l_type_set1.static_type.primary_type, l_type_set2.static_type.primary_type)
							end
							i := i + 1
						elseif same_declaration_types (l_type_set1.static_type.primary_type, l_type_set2.static_type.primary_type) then
							i := i + 1
						else
							Result := False
							i := nb + 1 -- Jump out of the loop.
						end
					end
				end
			end
		end

	set_polymorphic_call_argument_type_sets (an_argument_type_sets: ET_DYNAMIC_STANDALONE_TYPE_SET_LIST; a_first_call, a_last_call: ET_DYNAMIC_QUALIFIED_CALL; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Set `an_argument_type_sets' as the union of actual argument type
			-- sets for the calls between `a_first_call' and `a_last_call' with
			-- target type `a_target_type'.
		require
			an_argument_type_sets_not_void: an_argument_type_sets /= Void
			a_first_call_not_void: a_first_call /= Void
			a_last_call_not_void: a_last_call /= Void
			a_target_type_not_void: a_target_type /= Void
		local
			l_actual_arguments: detachable ET_ARGUMENT_OPERANDS
			i, nb, nb_args: INTEGER
			l_call: detachable ET_DYNAMIC_QUALIFIED_CALL
			l_is_first: BOOLEAN
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_standalone_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
			l_last_call_next: detachable ET_DYNAMIC_QUALIFIED_CALL
			l_seed: INTEGER
			l_manifest_tuple: detachable ET_MANIFEST_TUPLE
		do
			l_actual_arguments := a_first_call.static_call.arguments
			if l_actual_arguments /= Void and then l_actual_arguments.count > 0 then
				nb_args := l_actual_arguments.count
				nb := nb_args
				if nb_args = 1 then
					l_seed := a_first_call.static_call.name.seed
					if not a_first_call.is_tuple_label and then (l_seed = current_system.routine_call_seed or l_seed = current_system.function_item_seed) and then not a_first_call.is_call_agent then
						if attached {ET_MANIFEST_TUPLE} l_actual_arguments.actual_argument (1) as l_attached_manifest_tuple then
							l_manifest_tuple := l_attached_manifest_tuple
							nb := l_manifest_tuple.count + 1
						end
					end
				end
				if an_argument_type_sets.count < nb then
					an_argument_type_sets.resize (nb)
					from
						i := an_argument_type_sets.count + 1
					until
						i > nb
					loop
						create l_standalone_type_set.make (current_dynamic_system.any_type)
						an_argument_type_sets.put_last (l_standalone_type_set)
						i := i + 1
					end
				end
				if l_manifest_tuple /= Void then
					nb := l_manifest_tuple.count
					l_argument_type_set := dynamic_type_set_in_call (l_manifest_tuple, a_first_call)
					l_standalone_type_set := an_argument_type_sets.item (nb + 1)
					l_standalone_type_set.reset (l_argument_type_set.static_type)
					l_standalone_type_set.put_type (l_argument_type_set.static_type.primary_type)
					l_standalone_type_set.set_never_void
				end
				from
					l_call := a_first_call
					l_is_first := True
					l_last_call_next := a_last_call.next
				until
					l_call = Void or else l_call = l_last_call_next
				loop
					if l_call.target_type_set.has_type (a_target_type) then
						l_actual_arguments := l_call.static_call.arguments
						if l_actual_arguments = Void or else l_actual_arguments.count /= nb_args then
								-- Internal error: all calls should have the same signature.
								-- This is checked by the call to `same_declared_signature'.
							set_fatal_error
							error_handler.report_giaac_error (generator, "set_polymorphic_call_argument_type_sets", 1, "calls with different signatures.")
							l_call := a_last_call
						elseif l_manifest_tuple /= Void then
							if not attached {ET_MANIFEST_TUPLE} l_actual_arguments.actual_argument (1) as l_attached_manifest_tuple or else l_attached_manifest_tuple.count /= nb then
									-- Internal error: all calls should have the same signature.
									-- This is checked by the call to `same_declared_signature'.
								set_fatal_error
								error_handler.report_giaac_error (generator, "set_polymorphic_call_argument_type_sets", 2, "calls with different signatures.")
								l_manifest_tuple := Void
								l_call := a_last_call
							else
								l_manifest_tuple := l_attached_manifest_tuple
								from i := 1 until i > nb loop
									l_argument_type_set := dynamic_type_set_in_call (l_manifest_tuple.expression (i), l_call)
									l_standalone_type_set := an_argument_type_sets.item (i)
									if l_is_first then
										l_standalone_type_set.reset (l_argument_type_set.static_type)
										l_standalone_type_set.set_never_void
									elseif l_argument_type_set.static_type.conforms_to_type (l_standalone_type_set.static_type) then
										-- Nothing to be done.
									elseif l_standalone_type_set.static_type.conforms_to_type (l_argument_type_set.static_type) then
										l_standalone_type_set.set_static_type (l_argument_type_set.static_type)
									else
										l_standalone_type_set.set_static_type (current_dynamic_system.any_type)
									end
									l_standalone_type_set.put_types (l_argument_type_set)
									if not l_argument_type_set.is_never_void then
										l_standalone_type_set.propagate_can_be_void (l_argument_type_set)
									end
									i := i + 1
								end
							end
						else
							from i := 1 until i > nb_args loop
								l_argument_type_set := dynamic_type_set_in_call (l_actual_arguments.actual_argument (i), l_call)
								l_standalone_type_set := an_argument_type_sets.item (i)
								if l_is_first then
									l_standalone_type_set.reset (l_argument_type_set.static_type)
									l_standalone_type_set.set_never_void
								elseif l_argument_type_set.static_type.conforms_to_type (l_standalone_type_set.static_type) then
									-- Nothing to be done.
								elseif l_standalone_type_set.static_type.conforms_to_type (l_argument_type_set.static_type) then
									l_standalone_type_set.set_static_type (l_argument_type_set.static_type)
								else
									l_standalone_type_set.set_static_type (current_dynamic_system.any_type)
								end
								l_standalone_type_set.put_types (l_argument_type_set)
								if not l_argument_type_set.is_never_void then
									l_standalone_type_set.propagate_can_be_void (l_argument_type_set)
								end
								i := i + 1
							end
						end
						l_is_first := False
					end
					l_call := l_call.next
				end
			end
		end

feature {NONE} -- Deep features generation

	print_deep_twin_functions
			-- Print all functions necessary to implement 'deep_twin'.
		do
			if not deep_twin_types.is_empty then
				include_runtime_header_file ("ge_deep.h", True, header_file)
					-- Be aware that `print_deep_twin_function' can add
					-- new types at the end of `deep_twin_types'.
				from deep_twin_types.start until deep_twin_types.after loop
					print_deep_twin_function (deep_twin_types.item_for_iteration)
					deep_twin_types.forth
				end
				from deep_feature_target_type_sets.start until deep_feature_target_type_sets.after loop
					print_deep_twin_polymorphic_call_function (deep_feature_target_type_sets.item_for_iteration)
					deep_feature_target_type_sets.forth
				end
				deep_twin_types.wipe_out
				deep_feature_target_type_sets.wipe_out
			end
		end

	print_deep_twin_function (a_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print 'GE_deep_twin' function for type `a_type' to `current_file'
			-- and its signature to `header_file'.
		require
			a_type_not_void: a_type /= Void
		local
			l_has_nested_reference_attributes: BOOLEAN
			i, nb: INTEGER
			l_queries: ET_DYNAMIC_FEATURE_LIST
			l_attribute: ET_DYNAMIC_FEATURE
			l_attribute_type_set: detachable ET_DYNAMIC_TYPE_SET
			l_attribute_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_item_type_sets: ET_DYNAMIC_TYPE_SET_LIST
			old_type: like current_type
			old_file: KI_TEXT_OUTPUT_STREAM
			l_temp: ET_IDENTIFIER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_capacity_type: ET_DYNAMIC_PRIMARY_TYPE
			l_special_type: detachable ET_DYNAMIC_SPECIAL_TYPE
		do
			old_type := current_type
			current_type := a_type
				-- Print signature to `header_file' and `current_file'.
			old_file := current_file
			current_file := current_function_header_buffer
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			print_type_declaration (a_type, header_file)
			print_type_declaration (a_type, current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			header_file.put_string (c_ge_deep_twin)
			current_file.put_string (c_ge_deep_twin)
			header_file.put_integer (a_type.id)
			current_file.put_integer (a_type.id)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			print_type_declaration (current_type, header_file)
			print_type_declaration (current_type, current_file)
			if current_type.is_expanded then
				if current_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					header_file.put_character (' ')
					header_file.put_string (c_volatile)
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				header_file.put_character ('*')
				current_file.put_character ('*')
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_current_name (header_file)
			print_current_name (current_file)
			header_file.put_character (',')
			header_file.put_character (' ')
			current_file.put_character (',')
			current_file.put_character (' ')
			header_file.put_string (c_ge_deep)
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_character ('d')
			current_file.put_string (c_ge_deep)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_character ('d')
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print body to `current_file'.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
				-- Declare the Result entity.
			print_indentation
			print_type_declaration (a_type, current_file)
			current_file.put_character (' ')
			print_result_name (current_file)
			current_file.put_character (';')
			current_file.put_new_line
			current_file := current_function_body_buffer
			if a_type.base_class.is_type_class then
-- TODO: this built-in routine could be inlined.
					-- Cannot have two instances of class TYPE representing the same Eiffel type.
				print_indentation
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_current_name (current_file)
				current_file.put_character (';')
				current_file.put_new_line
			else
				l_has_nested_reference_attributes := a_type.has_nested_reference_attributes
				if l_has_nested_reference_attributes then
					print_indentation
					current_file.put_string (c_ge_deep)
					current_file.put_character ('*')
					current_file.put_character (' ')
					current_file.put_character ('t')
					current_file.put_character ('0')
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('d')
					current_file.put_character (';')
					current_file.put_new_line
				end
				if attached {ET_DYNAMIC_SPECIAL_TYPE} a_type as l_attached_special_type then
					l_special_type := l_attached_special_type
					l_attribute_type_set := l_special_type.item_type_set
					l_attribute_type := l_attribute_type_set.static_type.primary_type
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_ge_new)
					current_file.put_integer (l_special_type.id)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					print_comma
					print_attribute_special_capacity_access (tokens.current_keyword, l_special_type, False)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
					if l_special_type.attribute_count > 2 then
						print_indentation
						current_file.put_string (c_memcpy)
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_string (c_char)
						current_file.put_character ('*')
						current_file.put_character (')')
						if current_type.is_expanded then
							current_file.put_character ('&')
						end
						print_result_name (current_file)
						current_file.put_character (')')
						print_plus
						print_attribute_offset (l_special_type, current_file)
						print_comma
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_string (c_char)
						current_file.put_character ('*')
						current_file.put_character (')')
						if current_type.is_expanded then
							current_file.put_character ('&')
						end
						print_current_name (current_file)
						current_file.put_character (')')
						print_plus
						print_attribute_offset (l_special_type, current_file)
						print_comma
						current_file.put_string (c_offsetof)
						current_file.put_character ('(')
						print_type_name (current_type, current_file)
						print_comma
						print_attribute_special_item_name (current_type, current_file)
						current_file.put_character (')')
						if not l_has_nested_reference_attributes then
								-- Copy items if they are not reference objects or expanded
								-- objects containing (recursively) reference attributes.
							print_plus
							print_attribute_special_count_access (tokens.current_keyword, l_special_type, False)
							print_times
							current_file.put_string (c_sizeof)
							current_file.put_character ('(')
							print_type_declaration (l_attribute_type, current_file)
							current_file.put_character (')')
						end
						print_minus
						print_attribute_offset (l_special_type, current_file)
						current_file.put_character (')')
						print_semicolon_newline
					else
						print_indentation
						print_attribute_special_count_access (tokens.result_keyword, l_special_type, False)
						print_assign_to
						print_attribute_special_count_access (tokens.current_keyword, l_special_type, False)
						print_semicolon_newline
						if not l_has_nested_reference_attributes then
								-- Copy items if they are not reference objects or expanded
								-- objects containing (recursively) reference attributes.
							print_indentation
							current_file.put_string (c_memcpy)
							current_file.put_character ('(')
							current_file.put_character ('(')
							current_file.put_string (c_void)
							current_file.put_character ('*')
							current_file.put_character (')')
							current_file.put_character ('(')
							print_attribute_special_item_access (tokens.result_keyword, l_special_type, False)
							current_file.put_character (')')
							current_file.put_character (',')
							current_file.put_character ('(')
							current_file.put_string (c_void)
							current_file.put_character ('*')
							current_file.put_character (')')
							current_file.put_character ('(')
							print_attribute_special_item_access (tokens.current_keyword, l_special_type, False)
							current_file.put_character (')')
							current_file.put_character (',')
							print_attribute_special_count_access (tokens.current_keyword, l_special_type, False)
							current_file.put_character ('*')
							current_file.put_string (c_sizeof)
							current_file.put_character ('(')
							print_type_declaration (l_attribute_type, current_file)
							current_file.put_character (')')
							current_file.put_character (')')
							current_file.put_character (';')
							current_file.put_new_line
						end
					end
				elseif a_type.is_expanded and not a_type.has_once_per_object_routines then
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('*')
					print_current_name (current_file)
					current_file.put_character (';')
					current_file.put_new_line
				else
					if not a_type.is_expanded then
						print_indentation
						print_result_name (current_file)
						current_file.put_character (' ')
						current_file.put_character ('=')
						current_file.put_character (' ')
						current_file.put_string (c_ge_new)
						current_file.put_integer (a_type.id)
						current_file.put_character ('(')
						current_file.put_string (c_ac)
						print_comma
						current_file.put_string (c_eif_true)
						current_file.put_character (')')
						current_file.put_character (';')
						current_file.put_new_line
					end
						-- Note: do not copy the flag, SCOOP region, nor once-per-object.
					if
						a_type.attribute_count > 0 or
						attached {ET_DYNAMIC_TUPLE_TYPE} a_type as l_tuple_type and then not l_tuple_type.item_type_sets.is_empty
					then
						print_indentation
						current_file.put_string (c_memcpy)
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_string (c_char)
						current_file.put_character ('*')
						current_file.put_character (')')
						if a_type.is_expanded then
							current_file.put_character ('&')
						end
						print_result_name (current_file)
						current_file.put_character (')')
						print_plus
						print_attribute_offset (a_type, current_file)
						print_comma
						current_file.put_character ('(')
						current_file.put_character ('(')
						current_file.put_string (c_char)
						current_file.put_character ('*')
						current_file.put_character (')')
						if a_type.is_expanded then
							current_file.put_character ('&')
						end
						print_current_name (current_file)
						current_file.put_character (')')
						print_plus
						print_attribute_offset (a_type, current_file)
						print_comma
						current_file.put_string (c_sizeof)
						current_file.put_character ('(')
						print_type_name (a_type, current_file)
						current_file.put_character (')')
						print_minus
						print_attribute_offset (a_type, current_file)
						current_file.put_character (')')
						print_semicolon_newline
					end
				end
				if l_has_nested_reference_attributes then
						-- Allocate a 'GE_deep' struct to keep track of already twined
						-- reference objects, or use 'd' if not a null pointer (which
						-- means that the current object is not the root of the deep twin).
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('!')
					current_file.put_character ('t')
					current_file.put_character ('0')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string ("t0 = GE_deep_new();")
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					if not a_type.is_expanded then
							-- Keep track of reference objects already twined.
						print_indentation
						current_file.put_string ("GE_deep_put")
						current_file.put_character ('(')
						print_current_name (current_file)
						current_file.put_character (',')
						current_file.put_character (' ')
						print_result_name (current_file)
						current_file.put_character (',')
						current_file.put_character (' ')
						current_file.put_character ('t')
						current_file.put_character ('0')
						current_file.put_character (')')
						current_file.put_character (';')
						current_file.put_new_line
					end
						-- Twin reference attributes or expanded attributes that
						-- contain themselves (recursively) reference attributes.
					if l_special_type /= Void then
						check l_attribute_type_set_already_computed: l_attribute_type_set /= Void then end
						check l_attribute_type_already_computed: l_attribute_type /= Void then end
							-- Twin items.
						if l_special_type.attribute_count < 2 then
								-- Internal error: class "SPECIAL" should have at least the
								-- features 'capacity' and 'count' as first features.
								-- Already reported in ET_DYNAMIC_SYSTEM.compile_kernel.
							set_fatal_error
							error_handler.report_giaac_error (generator, "print_deep_twin_function", 1, "missing attributes in 'SPECIAL'.")
						else
							l_dynamic_type_set := result_type_set_in_feature (l_special_type.queries.first)
							l_capacity_type := l_dynamic_type_set.static_type.primary_type
							if l_attribute_type_set.is_empty then
									-- If the dynamic type set of the items is empty,
									-- then the items is always Void. No need to twin
									-- it in that case.
							elseif l_attribute_type.is_expanded then
									-- If the items are expanded.
									-- We need to deep twin them only if they themselves contain
									-- (recursively) reference attributes. Otherwise we can copy
									-- their contents without further ado.
								if l_attribute_type.has_nested_reference_attributes then
									l_temp := new_temp_variable (l_capacity_type)
									print_indentation
									current_file.put_string (c_for)
									current_file.put_character (' ')
									current_file.put_character ('(')
									print_temp_name (l_temp, current_file)
									current_file.put_character (' ')
									current_file.put_character ('=')
									current_file.put_character (' ')
									print_attribute_special_count_access (tokens.current_keyword, l_special_type, False)
									current_file.put_character (' ')
									current_file.put_character ('-')
									current_file.put_character (' ')
									current_file.put_character ('1')
									current_file.put_character (';')
									current_file.put_character (' ')
									print_temp_name (l_temp, current_file)
									current_file.put_character (' ')
									current_file.put_character ('>')
									current_file.put_character ('=')
									current_file.put_character (' ')
									current_file.put_character ('0')
									current_file.put_character (';')
									current_file.put_character (' ')
									print_temp_name (l_temp, current_file)
									current_file.put_character ('-')
									current_file.put_character ('-')
									current_file.put_character (')')
									current_file.put_character (' ')
									current_file.put_character ('{')
									current_file.put_new_line
									indent
									print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_special_indexed_item_access (l_temp,  tokens.current_keyword, a_type, False), agent print_attribute_special_indexed_item_access (l_temp,  tokens.result_keyword, a_type, False))
									dedent
									print_indentation
									current_file.put_character ('}')
									current_file.put_new_line
									mark_temp_variable_free (l_temp)
								end
							else
									-- We are in the case of reference items.
								l_temp := new_temp_variable (l_capacity_type)
								print_indentation
								current_file.put_string (c_for)
								current_file.put_character (' ')
								current_file.put_character ('(')
								print_temp_name (l_temp, current_file)
								current_file.put_character (' ')
								current_file.put_character ('=')
								current_file.put_character (' ')
								print_attribute_special_count_access (tokens.current_keyword, l_special_type, False)
								current_file.put_character (' ')
								current_file.put_character ('-')
								current_file.put_character (' ')
								current_file.put_character ('1')
								current_file.put_character (';')
								current_file.put_character (' ')
								print_temp_name (l_temp, current_file)
								current_file.put_character (' ')
								current_file.put_character ('>')
								current_file.put_character ('=')
								current_file.put_character (' ')
								current_file.put_character ('0')
								current_file.put_character (';')
								current_file.put_character (' ')
								print_temp_name (l_temp, current_file)
								current_file.put_character ('-')
								current_file.put_character ('-')
								current_file.put_character (')')
								current_file.put_character (' ')
								current_file.put_character ('{')
								current_file.put_new_line
								indent
								print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_special_indexed_item_access (l_temp,  tokens.current_keyword, a_type, False), agent print_attribute_special_indexed_item_access (l_temp,  tokens.result_keyword, a_type, False))
								dedent
								print_indentation
								current_file.put_character ('}')
								current_file.put_new_line
								mark_temp_variable_free (l_temp)
							end
						end
					else
						l_queries := a_type.queries
						nb := a_type.attribute_count
						from i := 1 until i > nb loop
							l_attribute := l_queries.item (i)
							l_attribute_type_set := l_attribute.result_type_set
							if l_attribute_type_set = Void then
									-- Internal error: an attribute should have a result type.
								set_fatal_error
								error_handler.report_giaac_error (generator, "print_deep_twin_function", 2, "attribue with no type.")
							elseif l_attribute_type_set.is_empty then
									-- If the dynamic type set of the attribute is empty,
									-- then this attribute is always Void. No need to twin
									-- it in that case.
							else
								l_attribute_type := l_attribute_type_set.static_type.primary_type
								if l_attribute_type.is_expanded then
										-- If the attribute is expanded, then its contents has
										-- already been copied. We need to deep twin it only if
										-- it itself contains (recursively) reference attributes.
									if l_attribute_type.has_nested_reference_attributes then
										print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_access (l_attribute, tokens.result_keyword, a_type, False), agent print_attribute_access (l_attribute, tokens.result_keyword, a_type, False))
									end
								else
									print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_access (l_attribute, tokens.result_keyword, a_type, False), agent print_attribute_access (l_attribute, tokens.result_keyword, a_type, False))
								end
							end
							i := i + 1
						end
						if attached {ET_DYNAMIC_TUPLE_TYPE} a_type as l_tuple_type then
							l_item_type_sets := l_tuple_type.item_type_sets
							nb := l_item_type_sets.count
							from i := 1 until i > nb loop
								l_attribute_type_set := l_item_type_sets.item (i)
								l_attribute_type := l_attribute_type_set.static_type.primary_type
								if l_attribute_type_set.is_empty then
										-- If the dynamic type set of the item is empty,
										-- then this item is always Void. No need to twin
										-- it in that case.
								elseif l_attribute_type.is_expanded then
										-- If the item is expanded, then its contents has
										-- already been copied. We need to deep twin it only if
										-- it itself contains (recursively) reference attributes.
									if l_attribute_type.has_nested_reference_attributes then
										print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_tuple_item_access (i, tokens.result_keyword, a_type, False), agent print_attribute_tuple_item_access (i, tokens.result_keyword, a_type, False))
									end
								else
									print_set_deep_twined_attribute (l_attribute_type_set, agent print_attribute_tuple_item_access (i, tokens.result_keyword, a_type, False), agent print_attribute_tuple_item_access (i, tokens.result_keyword, a_type, False))
								end
								i := i + 1
							end
						end
					end
						-- Free previously allocated 'GE_deep' struct, if any (i.e. if
						-- the current object was the root object of the deep twin).
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('t')
					current_file.put_character ('0')
					current_file.put_character (' ')
					current_file.put_character ('!')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('d')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string ("GE_deep_free(t0);")
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				elseif not a_type.is_expanded then
						-- Keep track of reference objects already twined.
						-- If 'd' is a null pointer, then there is no need
						-- to keep track of this object because it is the
						-- only object to be twined: it is the root object
						-- of the deep twin ('d' being a null pointer) and
						-- it has not reference attributes.
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_character ('d')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string ("GE_deep_put")
					current_file.put_character ('(')
					print_current_name (current_file)
					current_file.put_character (',')
					current_file.put_character (' ')
					print_result_name (current_file)
					current_file.put_character (',')
					current_file.put_character (' ')
					current_file.put_character ('d')
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				end
			end
				-- Return the deep twined object.
			print_indentation
			current_file.put_string (c_return)
			current_file.put_character (' ')
			print_result_name (current_file)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
			current_type := old_type
		end

	print_deep_twin_polymorphic_call_function (a_target_type_set: ET_DYNAMIC_TYPE_SET)
			-- Print 'GE_deep_twin<type-id>x' function to `current_file' and its signature to `header_file'.
			-- 'GE_deep_twin<type-id>x' corresponds to a polymorphic call to 'deep_twin'
			-- whose target has `a_target_type_set' as dynamic type set.
			-- 'type-id' is the type-id of the static type of the target.
		require
			a_target_type_set_not_void: a_target_type_set /= Void
		local
			l_target_dynamic_type_ids: DS_ARRAYED_LIST [INTEGER]
			l_target_dynamic_types: DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER]
			l_static_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_type_id: INTEGER
			l_temp: ET_IDENTIFIER
			i, nb: INTEGER
			l_switch: BOOLEAN
			old_type: like current_type
		do
			l_static_type := a_target_type_set.static_type.primary_type
			old_type := current_type
			current_type := l_static_type
				-- Print signature to `header_file' and `current_file'.
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			print_type_declaration (l_static_type, header_file)
			print_type_declaration (l_static_type, current_file)
			header_file.put_character (' ')
			current_file.put_character (' ')
			header_file.put_string (c_ge_deep_twin)
			current_file.put_string (c_ge_deep_twin)
			header_file.put_integer (l_static_type.id)
			current_file.put_integer (l_static_type.id)
			header_file.put_character ('x')
			current_file.put_character ('x')
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			print_type_declaration (current_type, header_file)
			print_type_declaration (current_type, current_file)
			if current_type.is_expanded then
				header_file.put_character ('*')
				current_file.put_character ('*')
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_current_name (header_file)
			print_current_name (current_file)
			header_file.put_character (',')
			header_file.put_character (' ')
			current_file.put_character (',')
			current_file.put_character (' ')
			header_file.put_string (c_ge_deep)
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_character ('t')
			header_file.put_character ('0')
			current_file.put_string (c_ge_deep)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_character ('t')
			current_file.put_character ('0')
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
				-- Print body to `current_file'.
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			l_target_dynamic_type_ids := target_dynamic_type_ids
			l_target_dynamic_types := target_dynamic_types
			nb := a_target_type_set.count
			from i := 1 until i > nb loop
				l_dynamic_type := a_target_type_set.dynamic_type (i)
				l_type_id := l_dynamic_type.id
				l_target_dynamic_type_ids.force_last (l_type_id)
				l_target_dynamic_types.force_last (l_dynamic_type, l_type_id)
				i := i + 1
			end
			l_target_dynamic_type_ids.sort (dynamic_type_id_sorter)
			if l_switch or l_target_dynamic_type_ids.count > 20 then
					-- Use switch statement.
				print_indentation
				current_file.put_string (c_switch)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_attribute_type_id_access (tokens.current_keyword, l_static_type, False)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				nb := l_target_dynamic_type_ids.count
				from i := 1 until i > nb loop
					l_type_id := l_target_dynamic_type_ids.item (i)
					l_dynamic_type := l_target_dynamic_types.item (l_type_id)
					print_indentation
					current_file.put_string (c_case)
					current_file.put_character (' ')
					current_file.put_integer (l_type_id)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_return)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_adapted_deep_twin_call (tokens.current_keyword, l_dynamic_type, l_static_type)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					i := i + 1
				end
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
			else
				print_indentation
				current_file.put_string (c_eif_type_index)
				current_file.put_character (' ')
				l_temp := temp_variable
				print_temp_name (l_temp, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_attribute_type_id_access (tokens.current_keyword, l_static_type, False)
				current_file.put_character (';')
				current_file.put_new_line
					-- Use binary search.
				print_deep_twin_binary_search_polymorphic_call (l_static_type, 1, l_target_dynamic_type_ids.count, l_target_dynamic_type_ids, l_target_dynamic_types)
			end
			l_target_dynamic_type_ids.wipe_out
			l_target_dynamic_types.wipe_out
			print_indentation
			current_file.put_string (c_return)
			current_file.put_character (' ')
			print_default_entity_value (l_static_type, current_file)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
			flush_to_c_file
			current_type := old_type
		end

	print_deep_twin_binary_search_polymorphic_call (a_target_static_type: ET_DYNAMIC_PRIMARY_TYPE; l, u: INTEGER; a_target_dynamic_type_ids: DS_ARRAYED_LIST [INTEGER]; a_target_dynamic_types: DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER])
			-- Print to `current_file' dynamic binding code for the call to 'GE_deep_twin'
			-- whose target's static type is `a_target_static_type' and whose target's
			-- dynamic types are those stored in `a_target_dynamic_types' whose type-id is
			-- itself stored between indexes `l' and `u' in `a_target_dynamic_type_ids'.
			-- The generated code uses binary search to find out which feature to execute.
		require
			a_target_static_type_not_void: a_target_static_type /= Void
			a_target_dynamic_type_ids_not_void: a_target_dynamic_type_ids /= Void
			a_target_dynamic_types_not_void: a_target_dynamic_types /= Void
			no_void_target_dynamic_type: not a_target_dynamic_types.has_void_item
			consistent_count: a_target_dynamic_types.count = a_target_dynamic_type_ids.count
			l_large_enough: l >= 1
			l_small_enough: l <= u
			u_small_enough: u <= a_target_dynamic_type_ids.count
		local
			t: INTEGER
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_type_id: INTEGER
			l_temp: ET_IDENTIFIER
		do
			l_temp := temp_variable
			if l = u then
				l_type_id := a_target_dynamic_type_ids.item (l)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_adapted_deep_twin_call (tokens.current_keyword, l_dynamic_type, a_target_static_type)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
			elseif l + 1 = u then
				l_type_id := a_target_dynamic_type_ids.item (l)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp, current_file)
				current_file.put_character ('=')
				current_file.put_character ('=')
				current_file.put_integer (l_type_id)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_adapted_deep_twin_call (tokens.current_keyword, l_dynamic_type, a_target_static_type)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				current_file.put_character ('}')
				current_file.put_character (' ')
				current_file.put_string (c_else)
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				l_type_id := a_target_dynamic_type_ids.item (u)
				l_dynamic_type := a_target_dynamic_types.item (l_type_id)
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_adapted_deep_twin_call (tokens.current_keyword, l_dynamic_type, a_target_static_type)
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				current_file.put_character ('}')
				current_file.put_new_line
			else
				t := l + (u - l) // 2
				l_type_id := a_target_dynamic_type_ids.item (t)
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp, current_file)
				current_file.put_character ('<')
				current_file.put_character ('=')
				current_file.put_integer (l_type_id)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_deep_twin_binary_search_polymorphic_call (a_target_static_type, l, t, a_target_dynamic_type_ids, a_target_dynamic_types)
				current_file.put_character ('}')
				current_file.put_character (' ')
				current_file.put_string (c_else)
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				print_deep_twin_binary_search_polymorphic_call (a_target_static_type, t + 1, u, a_target_dynamic_type_ids, a_target_dynamic_types)
				current_file.put_character ('}')
				current_file.put_new_line
			end
		end

	print_set_deep_twined_attribute (an_attribute_type_set: ET_DYNAMIC_TYPE_SET; a_print_attribute_access_read, a_print_attribute_access_write: PROCEDURE)
			-- Print to `current_file' the instructions needed to deep twin an attribute
			-- of `current_type' whose dynamic type set is `an_attribute_type_set'.
			-- `a_print_attribute_access_read' and `a_print_attribute_access_write' are
			-- used to print to `current_file' the code to access this attribute in
			-- read and write mode. Indeed, it can be a "regular" attribute, but it
			-- can also be items of a SPECIAL object, fields of a TUPLE object, closed
			-- operands of an Agent object, ...
		require
			an_attribute_type_set_not_void: an_attribute_type_set /= Void
			an_attribute_type_set_not_empty: not an_attribute_type_set.is_empty
			a_print_attribute_access_read_not_void: a_print_attribute_access_read /= Void
			a_print_attribute_access_write_not_void: a_print_attribute_access_write /= Void
		local
			l_attribute_type: ET_DYNAMIC_PRIMARY_TYPE
			l_temp1, l_temp2: ET_IDENTIFIER
		do
			l_attribute_type := an_attribute_type_set.static_type.primary_type
			l_temp1 := new_temp_variable (l_attribute_type)
			print_indentation
			print_temp_name (l_temp1, current_file)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			a_print_attribute_access_read.call ([])
			current_file.put_character (';')
			current_file.put_new_line
			if l_attribute_type.is_expanded then
					-- No need to test whether the attribute is Void or not:
					-- expanded attributes are never Void.
				print_indentation
				a_print_attribute_access_write.call ([])
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_deep_twined_attribute (l_temp1, an_attribute_type_set)
				current_file.put_character (';')
				current_file.put_new_line
			else
					-- If the attribute is Void, then there is no need to twin it.
				print_indentation
				current_file.put_string (c_if)
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp1, current_file)
				current_file.put_character (')')
				current_file.put_character (' ')
				current_file.put_character ('{')
				current_file.put_new_line
				indent
				l_temp2 := new_temp_variable (l_attribute_type)
				print_indentation
				print_temp_name (l_temp2, current_file)
				current_file.put_string (" = GE_deep_item(")
				print_temp_name (l_temp1, current_file)
				current_file.put_string (", t0);")
				current_file.put_new_line
				print_indentation
				a_print_attribute_access_write.call ([])
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('(')
				print_temp_name (l_temp2, current_file)
				current_file.put_character ('?')
					-- The object has not been twined yet.
				print_temp_name (l_temp2, current_file)
				current_file.put_character (':')
				current_file.put_character ('(')
				print_deep_twined_attribute (l_temp1, an_attribute_type_set)
				current_file.put_character (')')
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
				dedent
				print_indentation
				current_file.put_character ('}')
				current_file.put_new_line
				mark_temp_variable_free (l_temp2)
			end
			mark_temp_variable_free (l_temp1)
		end

	print_deep_twined_attribute (an_attribute: ET_EXPRESSION; an_attribute_type_set: ET_DYNAMIC_TYPE_SET)
			-- Print to `current_file' deep twined version of the attribute `an_attribute'
			-- belonging to `current_type', with dynamic type set `an_attribute_type_set'.
			-- The test for Void-ness of the attribute is assumed to have
			-- been generated elsewhere. And the attribute is assumed not to
			-- have been deep twined already.
		require
			an_attribute_not_void: an_attribute /= Void
			an_attribute_type_set_not_void: an_attribute_type_set /= Void
			an_attribute_type_set_not_empty: not an_attribute_type_set.is_empty
		local
			i, nb: INTEGER
			l_attribute_type: ET_DYNAMIC_PRIMARY_TYPE
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_standalone_type_set: ET_DYNAMIC_STANDALONE_TYPE_SET
		do
			l_attribute_type := an_attribute_type_set.static_type.primary_type
			nb := an_attribute_type_set.count
			if nb = 1 then
					-- Monomorphic call.
				l_dynamic_type := an_attribute_type_set.dynamic_type (1)
				deep_twin_types.force_last (l_dynamic_type)
				print_adapted_deep_twin_call (an_attribute, l_dynamic_type, l_attribute_type)
			elseif nb = 2 then
					-- Polymorphic with only two possible types at run-time.
				l_dynamic_type := an_attribute_type_set.dynamic_type (1)
				deep_twin_types.force_last (l_dynamic_type)
				current_file.put_character ('(')
				current_file.put_character ('(')
				print_attribute_type_id_access (an_attribute, l_attribute_type, False)
				current_file.put_character ('=')
				current_file.put_character ('=')
				current_file.put_integer (l_dynamic_type.id)
				current_file.put_character (')')
				current_file.put_character ('?')
				print_adapted_deep_twin_call (an_attribute, l_dynamic_type, l_attribute_type)
				current_file.put_character (':')
				l_dynamic_type := an_attribute_type_set.dynamic_type (2)
				deep_twin_types.force_last (l_dynamic_type)
				print_adapted_deep_twin_call (an_attribute, l_dynamic_type, l_attribute_type)
				current_file.put_character (')')
			else
					-- Polymorphic with more than two possible types at run-time.
					-- Wrap this polymorphic call into a funtion that will be
					-- shared by other polymorphic calls having the same target
					-- static type.
					--
					-- First, register all what is needed so that this shared
					-- function will be generated correctly.
				deep_feature_target_type_sets.search (l_attribute_type)
				if deep_feature_target_type_sets.found then
					l_standalone_type_set := deep_feature_target_type_sets.found_item
				else
					if standalone_type_sets.count > deep_feature_target_type_sets.count then
						l_standalone_type_set := standalone_type_sets.item (deep_feature_target_type_sets.count + 1)
						l_standalone_type_set.reset (l_attribute_type)
					else
						create l_standalone_type_set.make (l_attribute_type)
						standalone_type_sets.force_last (l_standalone_type_set)
					end
					deep_feature_target_type_sets.force_last_new (l_standalone_type_set, l_attribute_type)
				end
				l_standalone_type_set.put_types (an_attribute_type_set)
				from i := 1 until i > nb loop
					deep_twin_types.force_last (an_attribute_type_set.dynamic_type (i))
					i := i + 1
				end
					-- Now call the shared function that will trigger the
					-- polymorphic call.
				current_file.put_string (c_ge_deep_twin)
				current_file.put_integer (l_attribute_type.id)
				current_file.put_character ('x')
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				print_target_expression (an_attribute, l_attribute_type, False)
				current_file.put_character (',')
				current_file.put_character (' ')
				current_file.put_character ('t')
				current_file.put_character ('0')
				current_file.put_character (')')
			end
		end

	print_adapted_deep_twin_call (a_target: ET_EXPRESSION; a_target_type, a_result_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to the 'GE_deep_twin' function that
			-- will deep twin `a_target' of type `a_target_type'.
			-- `a_result_type' is the static type of the result expected by the caller,
			-- used to adapt the result of 'GE_deep_twin' if needed (see header comment
			-- of `print_adapted_expression_with_agent' for details).
			-- The test for Void-ness of the target is assumed to have
			-- been generated elsewhere.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
			a_result_type_not_void: a_result_type /= Void
		do
			print_adapted_expression_with_agent (agent print_deep_twin_call (a_target, a_target_type), a_target_type, a_result_type, False)
		end

	print_deep_twin_call (a_target: ET_EXPRESSION; a_target_type: ET_DYNAMIC_PRIMARY_TYPE)
			-- Print to `current_file' a call to the 'GE_deep_twin' function that
			-- will deep twin `a_target' of type `a_target_type'.
			-- The test for Void-ness of the target is assumed to have
			-- been generated elsewhere.
			-- Note that the result of 'GE_deep_twin' is not adapted to match the
			-- kind of result type expected by the caller. It is recommended to
			-- use `print_adapted_deep_twin_call' whenever possible.
		require
			a_target_not_void: a_target /= Void
			a_target_type_not_void: a_target_type /= Void
		do
			current_file.put_string (c_ge_deep_twin)
			current_file.put_integer (a_target_type.id)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			print_comma
			print_target_expression (a_target, a_target_type, False)
			current_file.put_character (',')
			current_file.put_character (' ')
			current_file.put_character ('t')
			current_file.put_character ('0')
			current_file.put_character (')')
		end

	deep_twin_types: DS_HASH_SET [ET_DYNAMIC_PRIMARY_TYPE]
			-- Types of object that need to be deep twined

	deep_equal_types: DS_HASH_SET [ET_DYNAMIC_PRIMARY_TYPE]
			-- Types of object that need deep equality

	deep_feature_target_type_sets: DS_HASH_TABLE [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE]
			-- Dynamic type sets of target of deep feature (deep twin or deep equal),
			-- indexed by target static type

feature {NONE} -- Separate calls

	print_separate_call_declaration (i: INTEGER; a_separate_call: ET_SEPARATE_CALL; a_closure: ET_CLOSURE)
			-- Print functions associated with the `i'-th separate call `a_separate_call'
			-- appearing in `a_closure' of `current_feature`, to `current_file' and its
			-- signature to `header_file'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			a_closure_not_void: a_closure /= Void
			use_scoop: use_scoop
		local
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_actual_argument: ET_EXPRESSION
			l_formal_argument: ET_IDENTIFIER
			j, nb, nb_args: INTEGER
			nb_operands: INTEGER
			nb_scoop_sessions: INTEGER
			l_separate_arguments: like separate_arguments
			l_separate_argument: ET_IDENTIFIER
			l_dynamic_type_set: ET_DYNAMIC_TYPE_SET
			l_type: ET_DYNAMIC_TYPE
			l_primary_type: ET_DYNAMIC_PRIMARY_TYPE
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_is_synchronous_call: BOOLEAN
			l_is_asynchronous_call: BOOLEAN
			l_separate_call_arguments: like separate_call_arguments
			l_has_reference_actual_argument: BOOLEAN
			l_has_non_separate_reference_actual_argument: BOOLEAN
			l_index: INTEGER
			l_is_creation_call: BOOLEAN
			l_is_passive_region: BOOLEAN
			l_and_then_needed: BOOLEAN
			l_old_current_type: like current_type
			l_old_max_nested_inlining_count: INTEGER
			old_file: KI_TEXT_OUTPUT_STREAM
			old_function_header_buffer: like current_function_header_buffer
			old_function_body_buffer: like current_function_body_buffer
			old_temp_variables: like temp_variables
			old_used_temp_variables: like used_temp_variables
			old_free_temp_variables: like free_temp_variables
			old_frozen_temp_variables: like frozen_temp_variables
			old_volatile_temp_variables: like volatile_temp_variables
			old_indentation: INTEGER
			old_assignment_target: like assignment_target
			old_in_operand: BOOLEAN
			old_call_target_type: like call_target_type
			old_current_agent: like current_agent
		do
			old_indentation := indentation
			indentation := 0
			old_in_operand := in_operand
			in_operand := False
			old_assignment_target := assignment_target
			assignment_target := Void
			old_call_target_type := call_target_type
			call_target_type := Void
			old_current_agent := current_agent
			current_agent := Void
			old_file := current_file
			old_function_header_buffer := current_function_header_buffer
			old_function_body_buffer := current_function_body_buffer
			current_function_header_buffer := current_separate_function_header_buffer
			current_function_body_buffer := current_separate_function_body_buffer
			current_file := current_function_header_buffer
			old_temp_variables := temp_variables
			old_used_temp_variables := used_temp_variables
			old_free_temp_variables := free_temp_variables
			old_frozen_temp_variables := frozen_temp_variables
			old_volatile_temp_variables := volatile_temp_variables
			temp_variables := separate_temp_variables
			used_temp_variables := used_separate_temp_variables
			free_temp_variables := free_separate_temp_variables
			frozen_temp_variables := frozen_separate_temp_variables
			volatile_temp_variables := volatile_separate_temp_variables
			if attached {ET_QUALIFIED_FEATURE_CALL_EXPRESSION} a_separate_call as l_call_expression then
				l_index := l_call_expression.index
				l_result_type := dynamic_type_set (l_call_expression).static_type.primary_type
			elseif attached {ET_CREATION_COMPONENT} a_separate_call as l_separate_creation_call then
				l_is_creation_call := True
				l_is_passive_region := l_separate_creation_call.is_scoop_region_passive
			end
			l_target_type := dynamic_type_set (a_separate_call.target).static_type.primary_type
				-- Print signature to `header_file' and `current_file'.
			header_file.put_character ('/')
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string ("Function for separate call #")
			header_file.put_integer (i)
			header_file.put_character (' ')
			header_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, header_file)
			header_file.put_character (')')
			header_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, header_file)
			header_file.put_character (' ')
			header_file.put_character ('*')
			header_file.put_character ('/')
			header_file.put_new_line
			current_file.put_character ('/')
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string ("Function for separate call #")
			current_file.put_integer (i)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, current_file)
			current_file.put_character (')')
			current_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, current_file)
			current_file.put_character (' ')
			current_file.put_character ('*')
			current_file.put_character ('/')
			current_file.put_new_line
			header_file.put_string (c_extern)
			header_file.put_character (' ')
			if l_result_type /= Void then
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
			elseif l_is_creation_call then
				print_type_declaration (l_target_type, header_file)
				print_type_declaration (l_target_type, current_file)
			else
				header_file.put_string (c_void)
				current_file.put_string (c_void)
			end
			header_file.put_character (' ')
			current_file.put_character (' ')
			print_separate_call_name (i, current_feature, current_type, header_file)
			print_separate_call_name (i, current_feature, current_type, current_file)
			header_file.put_character ('(')
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			nb_operands := 1
			if not l_is_creation_call then
				l_separate_arguments := separate_arguments
				l_separate_arguments.wipe_out
				a_separate_call.target.add_separate_arguments (l_separate_arguments, a_closure)
				nb := l_separate_arguments.count
				from j := 1 until j > nb loop
					l_separate_argument := l_separate_arguments.item (j)
					if
						l_separate_argument.is_inline_separate_argument or
						l_separate_argument.is_argument and then dynamic_type_set (l_separate_argument).static_type.is_separate
					then
						nb_scoop_sessions := nb_scoop_sessions + 1
						header_file.put_character (',')
						header_file.put_character (' ')
						header_file.put_string (c_ge_scoop_session)
						header_file.put_character ('*')
						header_file.put_character (' ')
						header_file.put_string (c_se)
						header_file.put_integer (nb_scoop_sessions)
						current_file.put_character (',')
						current_file.put_character (' ')
						current_file.put_string (c_ge_scoop_session)
						current_file.put_character ('*')
						current_file.put_character (' ')
						current_file.put_string (c_se)
						current_file.put_integer (nb_scoop_sessions)
					end
					j := j + 1
				end
				l_separate_arguments.wipe_out
					-- Separate call target.
				l_formal_argument := formal_argument (nb_operands)
				l_formal_argument.set_index (a_separate_call.target.index)
				header_file.put_character (',')
				header_file.put_character (' ')
				current_file.put_character (',')
				current_file.put_character (' ')
				print_type_declaration (l_target_type, header_file)
				print_type_declaration (l_target_type, current_file)
				header_file.put_character (' ')
				current_file.put_character (' ')
				print_argument_name (l_formal_argument, header_file)
				print_argument_name (l_formal_argument, current_file)
			end
				-- Separate call arguments.
			if attached a_separate_call.arguments as l_actual_arguments then
				nb_args := l_actual_arguments.count
				from j := 1 until j > nb_args loop
					nb_operands := nb_operands + 1
					l_actual_argument := l_actual_arguments.actual_argument (j)
					l_formal_argument := formal_argument (nb_operands)
					l_formal_argument.set_index (l_actual_argument.index)
					l_dynamic_type_set := dynamic_type_set (l_formal_argument)
					l_type := l_dynamic_type_set.static_type
					l_primary_type := l_type.primary_type
					if not l_type.is_expanded and (l_primary_type /= current_dynamic_system.none_type) and not l_dynamic_type_set.is_empty then
						l_has_reference_actual_argument := True
						if not l_type.is_separate then
							l_has_non_separate_reference_actual_argument := True
						end
					end
					header_file.put_character (',')
					header_file.put_character (' ')
					current_file.put_character (',')
					current_file.put_character (' ')
					print_type_declaration (l_primary_type, header_file)
					print_type_declaration (l_primary_type, current_file)
					header_file.put_character (' ')
					current_file.put_character (' ')
					print_argument_name (l_formal_argument, header_file)
					print_argument_name (l_formal_argument, current_file)
					j := j + 1
				end
			end
			header_file.put_character (')')
			current_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			current_file.put_string (c_ge_scoop_region)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_sr)
			print_assign_to
			current_file.put_string (c_ac)
			current_file.put_string (c_arrow)
			current_file.put_string (c_region)
			print_semicolon_newline
			if l_result_type /= Void then
				print_indentation
				print_type_declaration (l_result_type, current_file)
				if not l_result_type.is_expanded or else l_result_type.is_basic then
						-- Declare the 'Result' as 'volatile' because we will pass
						-- its address during the separate call so that it is set
						-- from the other SCOOP processor. This prevents the C optimizer
						-- to implement it with 'register'.
						-- https://barrgroup.com/embedded-systems/how-to/c-volatile-keyword
						--
						-- Note that non-basic expanded types are already declared as volatile.
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				current_file.put_character (' ')
				print_result_name (current_file)
				print_semicolon_newline
			elseif l_is_creation_call then
				print_indentation
				current_file.put_string (c_ge_scoop_region)
				current_file.put_character ('*')
				current_file.put_character (' ')
				current_file.put_string (c_sr)
				current_file.put_character ('2')
				print_semicolon_newline
				print_indentation
				print_type_declaration (l_target_type, current_file)
				current_file.put_character (' ')
				print_argument_name (formal_argument (1), current_file)
				print_semicolon_newline
			end
			print_indentation
			current_file.put_string (c_ge_scoop_session)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_se)
			if not l_is_creation_call then
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_character ('0')
			end
			print_semicolon_newline
			l_is_synchronous_call := l_result_type /= Void or l_has_non_separate_reference_actual_argument or l_is_passive_region
			l_is_asynchronous_call := l_is_creation_call and not l_has_reference_actual_argument and not l_is_passive_region
			if not l_is_creation_call or else l_is_passive_region then
				print_indentation
				current_file.put_string (c_ge_rescue)
				current_file.put_character (' ')
				current_file.put_character ('r')
				current_file.put_character (';')
				current_file.put_new_line
			end
				-- Print body to `current_file'.
			current_file := current_function_body_buffer
			if l_is_creation_call then
				print_indentation
				current_file.put_string (c_sr)
				current_file.put_character ('2')
				print_assign_to
				current_file.put_string (c_ge_new_scoop_region)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				if l_is_passive_region then
					current_file.put_character ('%'')
					current_file.put_character ('\')
					current_file.put_character ('1')
					current_file.put_character ('%'')
				else
					current_file.put_character ('0')
				end
				current_file.put_character (')')
				print_semicolon_newline
				if attached l_target_type.seeded_dynamic_procedure (a_separate_call.name.seed, current_dynamic_system) as l_procedure then
					l_old_current_type := current_type
					current_type := l_target_type
					print_malloc_current (l_procedure.static_feature)
					current_type := l_old_current_type
				else
						-- Internal error: there should be a procedure with the seed.
						-- It has been computed in ET_FEATURE_CHECKER or else an
						-- error should have already been reported.
					set_fatal_error
					error_handler.report_giaac_error (generator, "print_separate_call_declaration", 1, "procedure not found.")
				end
				print_indentation
				current_file.put_string (c_ge_scoop_region_set_context)
				current_file.put_character ('(')
				current_file.put_string (c_sr)
				print_comma
				current_file.put_string (c_ac)
				current_file.put_character (')')
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_se)
				print_assign_to
				current_file.put_string (c_ge_scoop_session_open)
				current_file.put_character ('(')
				current_file.put_string (c_sr)
				print_comma
				current_file.put_string (c_sr)
				current_file.put_character ('2')
				print_comma
				current_file.put_character ('0')
				current_file.put_character (')')
				print_semicolon_newline
				if l_is_passive_region then
					print_indentation
					current_file.put_string (c_ge_scoop_region_set_context)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					current_file.put_character ('2')
					print_comma
					current_file.put_character ('0')
					current_file.put_character (')')
					print_semicolon_newline
				else
					print_indentation
					current_file.put_string (c_ge_thread_create_with_attr)
					current_file.put_character ('(')
					print_current_name (current_file);
					print_comma
					current_file.put_character ('0')
					print_comma
					current_file.put_character ('0')
					print_comma
					current_file.put_character ('0')
					print_comma
					current_file.put_character ('1')
					current_file.put_character (')')
					print_semicolon_newline
				end
					-- Target.
				l_formal_argument := formal_argument (1)
				l_formal_argument.set_index (a_separate_call.target.index)
				print_indentation
				print_argument_name (l_formal_argument, current_file)
				print_assign_to
				print_current_name (current_file)
				print_semicolon_newline
			else
				print_indentation
				current_file.put_string (c_ge_scoop_region)
				current_file.put_character ('*')
				current_file.put_character (' ')
				current_file.put_string (c_sr)
				current_file.put_character ('1')
				print_assign_to
				print_attribute_region_access (formal_argument (1), l_target_type, True)
				print_semicolon_newline
				from j := 1 until j > nb_scoop_sessions loop
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_integer (j)
					print_and_then
					current_file.put_string (c_se)
					current_file.put_integer (j)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_sr)
					current_file.put_character ('1')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_string (c_se)
					print_assign_to
					current_file.put_string (c_se)
					current_file.put_integer (j)
					print_semicolon_newline
					j := j + 1
				end
			end
			if not l_is_synchronous_call then
					-- Potentially asynchronous call.
				if not l_is_asynchronous_call then
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					if not l_is_creation_call then
						current_file.put_string (c_se)
						l_and_then_needed := True
					end
					from j := 1 until j > nb_operands loop
							-- The first formal argument is in fact used for the target of the call.
						if j = 1 then
							if not l_is_creation_call then
									-- Synchronous call in case of callbacks.
								if l_and_then_needed then
									print_and_then
								end
								l_and_then_needed := True
								current_file.put_character ('!')
								current_file.put_string (c_ge_scoop_session_is_synchronized)
								current_file.put_character ('(')
								current_file.put_string (c_se)
								current_file.put_character (')')
								print_and_then
								current_file.put_character ('!')
								current_file.put_string (c_se)
								current_file.put_string (c_arrow)
								current_file.put_string (c_callee)
								current_file.put_string (c_arrow)
								current_file.put_string (c_is_passive)
							end
						else
							l_formal_argument := formal_argument (j)
							l_dynamic_type_set := dynamic_type_set (l_formal_argument)
							l_type := l_dynamic_type_set.static_type
							l_primary_type := l_type.primary_type
							if not l_type.is_expanded and (l_primary_type /= current_dynamic_system.none_type) and not l_dynamic_type_set.is_empty then
									-- Synchronous call when locked by current processor.
								if l_and_then_needed then
									print_and_then
								end
								l_and_then_needed := True
								current_file.put_character ('!')
								current_file.put_character ('(')
								print_argument_name (l_formal_argument, current_file)
								print_and_then
								current_file.put_string (c_ge_scoop_region_has_lock_on)
								current_file.put_character ('(')
								current_file.put_string (c_sr)
								print_comma
								print_attribute_region_access (l_formal_argument, l_primary_type, False)
								current_file.put_character (')')
								current_file.put_character (')')
							end
						end
						j := j + 1
					end
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				end
					-- Asynchronous call.
				print_indentation
				current_file.put_string (c_ge_scoop_session_add_call)
				current_file.put_character ('(')
				current_file.put_string (c_se)
				print_comma
				print_separate_call_object_name (i, current_feature, current_type, current_file)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				if l_result_type /= Void then
					current_file.put_character ('&')
					print_result_name (current_file)
					print_comma
				end
				print_argument_name (formal_argument (1), current_file)
				from j := 2 until j > nb_operands loop
					print_comma
					print_argument_name (formal_argument (j), current_file)
					j := j + 1
				end
				print_comma
				current_file.put_character ('0')
				current_file.put_character (')')
				current_file.put_character (')')
				print_semicolon_newline
				if not l_is_asynchronous_call then
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				end
			end
			if not l_is_asynchronous_call then
					-- Synchronous call with no impersonation.
					-- Always the case for creation procedures (to give them
					-- a chance to disable impersonation during creation)
					-- unless we create a passive region.
				if not l_is_creation_call or else l_is_passive_region then
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
						-- Note: we need to call 'GE_scoop_session_add_sync_call' before
						-- 'GE_scoop_session_is_impersonation_allowed' because the flag
						-- `is_impersonation_allowed` may be set by an asynchronous call
						-- (e.g. the creation procedure) which has not been executed yet.
					print_indentation
					current_file.put_string (c_ge_scoop_session_add_sync_call)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					print_comma
					current_file.put_string (c_se)
					current_file.put_character (')')
					print_semicolon_newline
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					print_and_then
					current_file.put_character ('!')
					current_file.put_string (c_ge_scoop_session_is_impersonation_allowed)
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_character (')')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
				end
				print_indentation
				current_file.put_string (c_ge_scoop_session_add_call)
				current_file.put_character ('(')
				current_file.put_string (c_se)
				print_comma
				print_separate_call_object_name (i, current_feature, current_type, current_file)
				current_file.put_character ('(')
				current_file.put_string (c_ac)
				print_comma
				if l_result_type /= Void then
					current_file.put_character ('&')
					print_result_name (current_file)
					print_comma
				end
				print_argument_name (formal_argument (1), current_file)
				from j := 2 until j > nb_operands loop
					print_comma
					print_argument_name (formal_argument (j), current_file)
					j := j + 1
				end
				print_comma
				current_file.put_character ('%'')
				current_file.put_character ('\')
				current_file.put_character ('1')
				current_file.put_character ('%'')
				current_file.put_character (')')
				current_file.put_character (')')
				print_semicolon_newline
				if not l_is_creation_call or else l_is_passive_region then
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_character (' ')
					current_file.put_string (c_else)
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
						-- Synchronous call with impersonation.
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_ge_scoop_region_pass_locks)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					print_comma
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					current_file.put_character (')')
					print_semicolon_newline
						-- Use impersonation (i.e. the call will be executed by
						-- the processor of the current region on behalf of the
						-- processor of the other SCOOP region, pretending
						-- that it is the processor associated with this other
						-- SCOOP region).
					print_indentation
					current_file.put_string (c_ge_scoop_region_impersonate)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					print_comma
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					current_file.put_character (')')
					print_semicolon_newline
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					print_indentation
					current_file.put_character ('r')
					current_file.put_character ('.')
					current_file.put_string (c_previous)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_last_rescue)
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_last_rescue)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('&')
					current_file.put_character ('r')
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_ge_setjmp)
					current_file.put_character ('(')
					current_file.put_character ('r')
					current_file.put_character ('.')
					current_file.put_character ('j')
					current_file.put_character ('b')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('!')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('0')
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_last_rescue)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('&')
					current_file.put_character ('r')
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_ge_scoop_region_impersonate)
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					print_comma
					current_file.put_string (c_sr)
					current_file.put_character (')')
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_scoop_region_release_locks)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					print_comma
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					current_file.put_character (')')
					print_semicolon_newline
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_last_rescue)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('r')
					current_file.put_character ('.')
					current_file.put_string (c_previous)
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_jump_to_last_rescue)
					current_file.put_character ('(')
					current_file.put_string (c_ac)
					current_file.put_character (')')
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
					separate_call_arguments.wipe_out
					if nb_args > 0 then
						l_separate_call_arguments := separate_call_arguments
						if l_separate_call_arguments.capacity < nb_args then
							l_separate_call_arguments.resize (nb_args)
						end
						from j := nb_operands until j < 2 loop
							l_separate_call_arguments.put_first (formal_argument (j))
							j := j - 1
						end
					end
						-- No inlining here.
					l_old_max_nested_inlining_count := max_nested_inlining_count
					max_nested_inlining_count := 0
					if l_result_type /= Void then
						print_indentation
						print_result_name (current_file)
						print_assign_to
						separate_call_expression.set_target (formal_argument (1))
						separate_call_expression.set_name (a_separate_call.name)
						separate_call_expression.set_arguments (l_separate_call_arguments)
						separate_call_expression.set_index (l_index)
						print_qualified_call_expression (separate_call_expression)
						print_semicolon_newline
					else
						separate_call_instruction.set_target (formal_argument (1))
						separate_call_instruction.set_name (a_separate_call.name)
						separate_call_instruction.set_arguments (l_separate_call_arguments)
						print_qualified_call_instruction (separate_call_instruction)
					end
					max_nested_inlining_count := l_old_max_nested_inlining_count
					print_indentation
					current_file.put_string (c_ac)
					current_file.put_string (c_arrow)
					current_file.put_string (c_last_rescue)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_character ('r')
					current_file.put_character ('.')
					current_file.put_string (c_previous)
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_if)
					current_file.put_character (' ')
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					indent
					print_indentation
					current_file.put_string (c_ge_scoop_region_impersonate)
					current_file.put_character ('(')
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					print_comma
					current_file.put_string (c_sr)
					current_file.put_character (')')
					print_semicolon_newline
					print_indentation
					current_file.put_string (c_ge_scoop_region_release_locks)
					current_file.put_character ('(')
					current_file.put_string (c_sr)
					print_comma
					current_file.put_string (c_se)
					current_file.put_string (c_arrow)
					current_file.put_string (c_callee)
					current_file.put_character (')')
					print_semicolon_newline
					dedent
					print_indentation
					current_file.put_character ('}')
					current_file.put_new_line
				end
				if not l_is_asynchronous_call then
					if not l_is_synchronous_call then
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_new_line
					end
					if not l_is_creation_call or else l_is_passive_region then
						dedent
						print_indentation
						current_file.put_character ('}')
						current_file.put_new_line
					end
				end
			end
			if l_result_type /= Void then
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_result_name (current_file)
				print_semicolon_newline
			elseif l_is_creation_call then
				print_indentation
				current_file.put_string (c_ge_scoop_session_close)
				current_file.put_character ('(')
				current_file.put_string (c_sr)
				print_comma
				current_file.put_string (c_se)
				current_file.put_character (')')
				print_semicolon_newline
				print_indentation
				current_file.put_string (c_return)
				current_file.put_character (' ')
				print_current_name (current_file)
				print_semicolon_newline
			end
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			from j := 1 until j > nb_operands loop
				formal_argument (j).set_index (j)
				j := j + 1
			end
			print_separate_call_object_type_declaration (i, a_separate_call)
			print_separate_call_function_declaration (i, a_separate_call)
			print_separate_call_object_declaration (i, a_separate_call)
				--
				-- Clean up.
				--
			current_function_header_buffer := old_function_header_buffer
			current_function_body_buffer := old_function_body_buffer
			current_file := old_file
			temp_variables := old_temp_variables
			used_temp_variables := old_used_temp_variables
			free_temp_variables := old_free_temp_variables
			frozen_temp_variables := old_frozen_temp_variables
			volatile_temp_variables := old_volatile_temp_variables
			indentation := old_indentation
			in_operand := old_in_operand
			assignment_target := old_assignment_target
			call_target_type := old_call_target_type
			current_agent := old_current_agent
		end

	print_separate_call_function_declaration (i: INTEGER; a_separate_call: ET_SEPARATE_CALL)
			-- Print function to execute the `i'-th separate call `a_separate_call'
			-- appearing in `current_feature`, to `current_file' and its signature to
			-- `header_file'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			use_scoop: use_scoop
		local
			l_index: INTEGER
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_argument: ET_EXPRESSION
			j, nb_args: INTEGER
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_operand: ET_IDENTIFIER
			nb_operands: INTEGER
			l_separate_call_arguments: like separate_call_arguments
			l_old_max_nested_inlining_count: INTEGER
			l_old_in_separate_creation_call: BOOLEAN
			old_file: KI_TEXT_OUTPUT_STREAM
		do
			old_file := current_file
			current_file := current_function_header_buffer
				-- Print signature to `header_file' and `current_file'.
			l_target_type := dynamic_type_set (a_separate_call.target).static_type.primary_type
			header_file.put_character ('/')
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string ("Function to execute separate call #")
			header_file.put_integer (i)
			header_file.put_character (' ')
			header_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, header_file)
			header_file.put_character (')')
			header_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, header_file)
			header_file.put_character (' ')
			header_file.put_character ('*')
			header_file.put_character ('/')
			header_file.put_new_line
			header_file.put_string (c_void)
			header_file.put_character (' ')
			print_separate_call_function_name (i, current_feature, current_type, header_file)
			header_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			header_file.put_string (c_ge_scoop_session)
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string (c_se)
			header_file.put_character (',')
			header_file.put_character (' ')
			header_file.put_string (c_ge_scoop_call)
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string (c_sc)
			header_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			current_file.put_character ('/')
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string ("Function to execute separate call #")
			current_file.put_integer (i)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, current_file)
			current_file.put_character (')')
			current_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, current_file)
			current_file.put_character (' ')
			current_file.put_character ('*')
			current_file.put_character ('/')
			current_file.put_new_line
			current_file.put_string (c_void)
			current_file.put_character (' ')
			print_separate_call_function_name (i, current_feature, current_type, current_file)
			current_file.put_character ('(')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			current_file.put_string (c_ge_scoop_session)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_se)
			current_file.put_character (',')
			current_file.put_character (' ')
			current_file.put_string (c_ge_scoop_call)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_sc)
			current_file.put_character ('1')
			current_file.put_character (')')
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
			print_indentation
			print_separate_call_object_type_name (i, current_feature, current_type, current_file)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_sc)
			print_assign_to
			current_file.put_character ('(')
			print_separate_call_object_type_name (i, current_feature, current_type, current_file)
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_string (c_sc)
			current_file.put_character ('1')
			print_semicolon_newline
				-- Print body to `current_file'.
			current_file := current_function_body_buffer
			if attached {ET_QUALIFIED_FEATURE_CALL_EXPRESSION} a_separate_call as l_call_expression then
				l_index := l_call_expression.index
				l_result_type := dynamic_type_set (l_call_expression).static_type.primary_type
				print_indentation
				print_type_declaration (l_result_type, current_file)
				current_file.put_character (' ')
				if not l_result_type.is_expanded or else l_result_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					current_file.put_string (c_volatile)
					current_file.put_character (' ')
				end
				current_file.put_character ('*')
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				current_file.put_string (c_sc)
				current_file.put_string (c_arrow)
				print_result_name (current_file)
				print_semicolon_newline
			end
				-- Separate call target.
			nb_operands := 1
			l_operand := formal_argument (nb_operands)
			l_operand.set_index (a_separate_call.target.index)
			print_indentation
			print_type_declaration (l_target_type, current_file)
			current_file.put_character (' ')
			print_argument_name (l_operand, current_file)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_string (c_sc)
			current_file.put_string (c_arrow)
			print_argument_name (l_operand, current_file)
			print_semicolon_newline
				-- Separate call arguments.
			if attached a_separate_call.arguments as l_arguments then
				nb_args := l_arguments.count
				from j := 1 until j > nb_args loop
					l_argument := l_arguments.actual_argument (j)
					nb_operands := nb_operands + 1
					l_operand := formal_argument (nb_operands)
					l_operand.set_index (l_argument.index)
					l_type := dynamic_type_set (l_operand).static_type.primary_type
					print_indentation
					print_type_declaration (l_type, current_file)
					current_file.put_character (' ')
					print_argument_name (l_operand, current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_sc)
					current_file.put_string (c_arrow)
					print_argument_name (l_operand, current_file)
					print_semicolon_newline
					j := j + 1
				end
			end
				--
				-- Instructions.
				--
			separate_call_arguments.wipe_out
			if nb_args > 0 then
				l_separate_call_arguments := separate_call_arguments
				if l_separate_call_arguments.capacity < nb_args then
					l_separate_call_arguments.resize (nb_args)
				end
				from j := nb_operands until j < 2 loop
					l_separate_call_arguments.put_first (formal_argument (j))
					j := j - 1
				end
			end
				-- No inlining here.
			l_old_max_nested_inlining_count := max_nested_inlining_count
			max_nested_inlining_count := 0
			if l_result_type /= Void then
				print_indentation
				current_file.put_character ('*')
				print_result_name (current_file)
				print_assign_to
				separate_call_expression.set_target (formal_argument (1))
				separate_call_expression.set_name (a_separate_call.name)
				separate_call_expression.set_arguments (l_separate_call_arguments)
				separate_call_expression.set_index (l_index)
				print_qualified_call_expression (separate_call_expression)
				print_semicolon_newline
			else
				separate_call_instruction.set_target (formal_argument (1))
				separate_call_instruction.set_name (a_separate_call.name)
				separate_call_instruction.set_arguments (l_separate_call_arguments)
				l_old_in_separate_creation_call := in_separate_creation_call
				in_separate_creation_call := attached {ET_CREATION_COMPONENT} a_separate_call
				print_qualified_call_instruction (separate_call_instruction)
				in_separate_creation_call := l_old_in_separate_creation_call
			end
			max_nested_inlining_count := l_old_max_nested_inlining_count
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
			from j := 1 until j > nb_operands loop
				formal_argument (j).set_index (j)
				j := j + 1
			end
		end

	print_separate_call_object_declaration (i: INTEGER; a_separate_call: ET_SEPARATE_CALL)
			-- Print function to create the separate call object of
			-- `i'-th separate call `a_separate_call' appearing in
			-- `current_feature`, to `current_file' and its signature to
			-- `header_file'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			use_scoop: use_scoop
		local
			l_result_type: detachable ET_DYNAMIC_PRIMARY_TYPE
			l_argument: ET_EXPRESSION
			j, nb: INTEGER
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_operand: ET_IDENTIFIER
			nb_operands: INTEGER
			old_file: KI_TEXT_OUTPUT_STREAM
		do
			old_file := current_file
			current_file := current_function_header_buffer
				-- Print signature to `header_file' and `current_file'.
			l_target_type := dynamic_type_set (a_separate_call.target).static_type.primary_type
			header_file.put_character ('/')
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string ("Creation of separate call object #")
			header_file.put_integer (i)
			header_file.put_character (' ')
			header_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, header_file)
			header_file.put_character (')')
			header_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, header_file)
			header_file.put_character (' ')
			header_file.put_character ('*')
			header_file.put_character ('/')
			header_file.put_new_line
			header_file.put_string (c_ge_scoop_call)
			header_file.put_character ('*')
			header_file.put_character (' ')
			print_separate_call_object_name (i, current_feature, current_type, header_file)
			header_file.put_character ('(')
			current_file.put_character ('/')
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string ("Creation of separate call object #")
			current_file.put_integer (i)
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, current_file)
			current_file.put_character (')')
			current_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, current_file)
			current_file.put_character (' ')
			current_file.put_character ('*')
			current_file.put_character ('/')
			current_file.put_new_line
			current_file.put_string (c_ge_scoop_call)
			current_file.put_character ('*')
			current_file.put_character (' ')
			print_separate_call_object_name (i, current_feature, current_type, current_file)
			current_file.put_character ('(')
			print_context_type_declaration (header_file)
			header_file.put_character (' ')
			header_file.put_string (c_ac)
			header_file.put_character (',')
			header_file.put_character (' ')
			print_context_type_declaration (current_file)
			current_file.put_character (' ')
			current_file.put_string (c_ac)
			current_file.put_character (',')
			current_file.put_character (' ')
			if attached {ET_QUALIFIED_FEATURE_CALL_EXPRESSION} a_separate_call as l_call_expression then
				l_result_type := dynamic_type_set (l_call_expression).static_type.primary_type
				print_type_declaration (l_result_type, header_file)
				print_type_declaration (l_result_type, current_file)
				if not l_result_type.is_expanded or else l_result_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					header_file.put_character (' ')
					header_file.put_string (c_volatile)
					current_file.put_character (' ')
					current_file.put_string (c_volatile)
				end
				header_file.put_character ('*')
				header_file.put_character (' ')
				print_result_name (header_file)
				header_file.put_character (',')
				header_file.put_character (' ')
				current_file.put_character ('*')
				current_file.put_character (' ')
				print_result_name (current_file)
				current_file.put_character (',')
				current_file.put_character (' ')
			end
				-- Separate call target.
			nb_operands := 1
			l_operand := formal_argument (nb_operands)
			l_operand.set_index (a_separate_call.target.index)
			print_type_declaration (l_target_type, header_file)
			header_file.put_character (' ')
			print_argument_name (l_operand, header_file)
			print_type_declaration (l_target_type, current_file)
			current_file.put_character (' ')
			print_argument_name (l_operand, current_file)
				-- Separate call arguments.
			if attached a_separate_call.arguments as l_arguments then
				nb := l_arguments.count
				from j := 1 until j > nb loop
					l_argument := l_arguments.actual_argument (j)
					nb_operands := nb_operands + 1
					l_operand := formal_argument (nb_operands)
					l_operand.set_index (l_argument.index)
					l_type := dynamic_type_set (l_operand).static_type.primary_type
					header_file.put_character (',')
					header_file.put_character (' ')
					print_type_declaration (l_type, header_file)
					header_file.put_character (' ')
					print_argument_name (l_operand, header_file)
					print_comma
					print_type_declaration (l_type, current_file)
					current_file.put_character (' ')
					print_argument_name (l_operand, current_file)
					j := j + 1
				end
			end
			header_file.put_character (',')
			header_file.put_character (' ')
			header_file.put_string (c_char)
			header_file.put_character (' ')
			header_file.put_string ("a_is_synchronous")
			header_file.put_character (')')
			header_file.put_character (';')
			header_file.put_new_line
			print_comma
			current_file.put_string (c_char)
			current_file.put_character (' ')
			current_file.put_string ("a_is_synchronous")
			current_file.put_character (')')
			current_file.put_new_line
			current_file.put_character ('{')
			current_file.put_new_line
			indent
				-- Print body to `current_file'.
			current_file := current_function_body_buffer
			print_indentation
			print_separate_call_object_type_name (i, current_feature, current_type, current_file)
			current_file.put_character ('*')
			current_file.put_character (' ')
			current_file.put_string (c_sc)
			current_file.put_character (';')
			current_file.put_new_line
				-- Create separate call object.
			print_indentation
			current_file.put_string (c_sc)
			current_file.put_character (' ')
			current_file.put_character ('=')
			current_file.put_character (' ')
			current_file.put_character ('(')
			print_separate_call_object_type_name (i, current_feature, current_type, current_file)
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_string (c_ge_new_scoop_call)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			current_file.put_string (c_arrow)
			current_file.put_string (c_region)
			print_comma
			current_file.put_string ("a_is_synchronous")
			print_comma
			current_file.put_character ('(')
			current_file.put_string (c_void)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_character ('(')
			print_context_type_declaration (current_file)
			print_comma
			current_file.put_string (c_ge_scoop_session)
			current_file.put_character ('*')
			print_comma
			current_file.put_string (c_ge_scoop_call)
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_character ('&')
			print_separate_call_function_name (i, current_feature, current_type, current_file)
			print_comma
			current_file.put_string (c_sizeof)
			current_file.put_character ('(')
			print_separate_call_object_type_name (i, current_feature, current_type, current_file)
			current_file.put_character (')')
			current_file.put_character (')')
			current_file.put_character (';')
			current_file.put_new_line
			if l_result_type /= Void then
				print_indentation
				current_file.put_string (c_sc)
				current_file.put_string (c_arrow)
				print_result_name (current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_result_name (current_file)
				print_semicolon_newline
			end
			from j := 1 until j > nb_operands loop
				l_operand := formal_argument (j)
				print_indentation
				current_file.put_string (c_sc)
				current_file.put_string (c_arrow)
				print_argument_name (l_operand, current_file)
				current_file.put_character (' ')
				current_file.put_character ('=')
				current_file.put_character (' ')
				print_argument_name (l_operand, current_file)
				print_semicolon_newline
				j := j + 1
			end
				-- Return the separate call object.
			print_indentation
			current_file.put_string (c_return)
			current_file.put_character (' ')
			current_file.put_character ('(')
			current_file.put_string (c_ge_scoop_call)
			current_file.put_character ('*')
			current_file.put_character (')')
			current_file.put_string (c_sc)
			current_file.put_character (';')
			current_file.put_new_line
			dedent
			current_file.put_character ('}')
			current_file.put_new_line
			current_file.put_new_line
				-- Declarations of temporary variables.
			current_file := current_function_header_buffer
			print_temporary_variable_declarations
				-- Flush to file.
			flush_to_c_file
				--
				-- Clean up.
				--
			current_file := old_file
			from j := 1 until j > nb_operands loop
				formal_argument (j).set_index (j)
				j := j + 1
			end
		end

	print_separate_call_object_type_declaration (i: INTEGER; a_separate_call: ET_SEPARATE_CALL)
			-- Print the declaration of the type of the separate call
			-- object of `i'-th separate call `a_separate_call' appearing
			-- in `current_feature`, to `header_file'.
		require
			a_separate_call_not_void: a_separate_call /= Void
			use_scoop: use_scoop
		local
			l_argument: ET_EXPRESSION
			j, nb: INTEGER
			l_type: ET_DYNAMIC_PRIMARY_TYPE
			l_target_type: ET_DYNAMIC_PRIMARY_TYPE
			l_operand: ET_IDENTIFIER
			nb_operands: INTEGER
			l_result_type: ET_DYNAMIC_PRIMARY_TYPE
		do
			l_target_type := dynamic_type_set (a_separate_call.target).static_type.primary_type
			header_file.put_character ('/')
			header_file.put_character ('*')
			header_file.put_character (' ')
			header_file.put_string ("Type of separate call object #")
			header_file.put_integer (i)
			header_file.put_character (' ')
			header_file.put_character ('(')
			print_call_name_in_comment (a_separate_call.name, l_target_type, header_file)
			header_file.put_character (')')
			header_file.put_string (" in feature ")
			print_feature_name_in_comment (current_feature, current_type, header_file)
			header_file.put_character (' ')
			header_file.put_character ('*')
			header_file.put_character ('/')
			header_file.put_new_line
			header_file.put_string (c_typedef)
			header_file.put_character (' ')
			header_file.put_string (c_volatile)
			header_file.put_character (' ')
			header_file.put_string (c_struct)
			header_file.put_character (' ')
			header_file.put_character ('{')
			header_file.put_new_line
			header_file.put_character ('%T')
			header_file.put_string ("GE_scoop_region* volatile caller;")
			header_file.put_new_line
			header_file.put_character ('%T')
			header_file.put_string ("char volatile is_synchronous;")
			header_file.put_new_line
			header_file.put_character ('%T')
			header_file.put_string ("char volatile is_condition;")
			header_file.put_new_line
			header_file.put_character ('%T')
			header_file.put_string ("void (*volatile execute)(GE_context*, GE_scoop_session*, GE_scoop_call*);")
			header_file.put_new_line
			header_file.put_character ('%T')
			header_file.put_string ("GE_scoop_call* volatile next;")
			header_file.put_new_line
			if attached {ET_QUALIFIED_FEATURE_CALL_EXPRESSION} a_separate_call as l_call_expression then
				l_result_type := dynamic_type_set (l_call_expression).static_type.primary_type
				header_file.put_character ('%T')
				print_type_declaration (l_result_type, header_file)
				if not l_result_type.is_expanded or else l_result_type.is_basic then
						-- Note that non-basic expanded types are already declared as volatile.
					header_file.put_character (' ')
					header_file.put_string (c_volatile)
				end
				header_file.put_character ('*')
				header_file.put_character (' ')
				header_file.put_string (c_volatile)
				header_file.put_character (' ')
				print_result_name (header_file)
				header_file.put_character (';')
				header_file.put_new_line
			end
				-- Separate call target.
			nb_operands := 1
			l_operand := formal_argument (nb_operands)
			l_operand.set_index (a_separate_call.target.index)
			header_file.put_character ('%T')
			print_type_declaration (l_target_type, header_file)
			if not l_target_type.is_expanded or else l_target_type.is_basic then
					-- Note that non-basic expanded types are already declared as volatile.
				header_file.put_character (' ')
				header_file.put_string (c_volatile)
			end
			header_file.put_character (' ')
			print_argument_name (l_operand, header_file)
			header_file.put_character (';')
			header_file.put_new_line
				-- Separate call arguments.
			if attached a_separate_call.arguments as l_arguments then
				nb := l_arguments.count
				from j := 1 until j > nb loop
					l_argument := l_arguments.actual_argument (j)
					nb_operands := nb_operands + 1
					l_operand := formal_argument (nb_operands)
					l_operand.set_index (l_argument.index)
					l_type := dynamic_type_set (l_operand).static_type.primary_type
					header_file.put_character ('%T')
					print_type_declaration (l_type, header_file)
					if not l_type.is_expanded or else l_type.is_basic then
							-- Note that non-basic expanded types are already declared as volatile.
						header_file.put_character (' ')
						header_file.put_string (c_volatile)
					end
					header_file.put_character (' ')
					print_argument_name (l_operand, header_file)
					header_file.put_character (';')
					header_file.put_new_line
					j := j + 1
				end
			end
			header_file.put_character ('}')
			header_file.put_character (' ')
			print_separate_call_object_type_name (i, current_feature, current_type, header_file)
			header_file.put_character (';')
			header_file.put_new_line
			header_file.put_new_line
				--
				-- Clean up.
				--
			from j := 1 until j > nb_operands loop
				formal_argument (j).set_index (j)
				j := j + 1
			end
		end

	separate_call_arguments: ET_ACTUAL_ARGUMENT_LIST
			-- Arguments to be used when generating separate call objects

	separate_call_instruction: ET_GENERAL_QUALIFIED_FEATURE_CALL_INSTRUCTION
			-- Instruction to be used when generating separate call objects

	separate_call_expression: ET_GENERAL_QUALIFIED_FEATURE_CALL_EXPRESSION
			-- Expression to be used when generating separate call objects

	separate_arguments: DS_ARRAYED_LIST [ET_IDENTIFIER]
			-- List of inline separate arguments or formal arguments which
			-- when controlled (i.e. when their type is separate) implies
			-- that the target of the current separate call is also controlled.
			-- (Used when determining the SCOOP sessions to be used when recording
			-- a separate call to another SCOOP region.)

	in_separate_creation_call: BOOLEAN
			-- Is `separate_call_instruction' a creation of a separate object?

feature {NONE} -- Built-in feature generation

	print_builtin_procedure_c_call (a_feature: ET_DYNAMIC_FEATURE; a_c_function_name: STRING; a_static: BOOLEAN; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding a built-in feature which is a call to a C function
			-- named `a_c_function_name'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_static' means that 'Current' is not passed to the C function.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			a_c_function_name_not_void: a_c_function_name /= Void
			a_c_function_name_not_empty: not a_c_function_name.is_empty
			call_operands_not_empty: not call_operands.is_empty
		local
			l_argument: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type: ET_DYNAMIC_TYPE
			i, nb: INTEGER
		do
			nb := a_feature.static_feature.arguments_count
			if call_operands.count /= nb + 1 then
					-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_builtin_procedure_c_call", 1, "wrong_number of arguments.")
			else
				print_indentation
				current_file.put_string (a_c_function_name)
				current_file.put_character ('(')
				from
					if a_static then
						i := 1
					end
				until
					i > nb
				loop
					current_file.put_character ('(')
					if i = 0 then
						l_argument := call_operands.first
						l_actual_type_set := dynamic_type_set (l_argument)
						l_formal_type := a_target_type
					else
						l_argument := call_operands.item (i + 1)
						l_actual_type_set := dynamic_type_set (l_argument)
						l_formal_type := argument_type_set_in_feature (i, a_feature).static_type
					end
					print_attachment_expression (l_argument, l_actual_type_set, l_formal_type)
					current_file.put_character (')')
					if i /= nb then
						current_file.put_character (',')
						current_file.put_character (' ')
					end
					i := i + 1
				end
				current_file.put_character (')')
				current_file.put_character (';')
				current_file.put_new_line
			end
		end

	print_builtin_query_c_call (a_feature: ET_DYNAMIC_FEATURE; a_c_function_name: STRING; a_static, a_context: BOOLEAN; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding a built-in feature which is a call to a C function
			-- named `a_c_function_name'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_static' means that 'Current' is not passed to the C function.
			-- `a_context' means that the call context is passed to the C function.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			a_c_function_name_not_void: a_c_function_name /= Void
			a_c_function_name_not_empty: not a_c_function_name.is_empty
			call_operands_not_empty: not call_operands.is_empty
		local
			l_argument: ET_EXPRESSION
			l_actual_type_set: ET_DYNAMIC_TYPE_SET
			l_formal_type: ET_DYNAMIC_TYPE
			i, nb: INTEGER
			l_comma_needed: BOOLEAN
		do
			nb := a_feature.static_feature.arguments_count
			if call_operands.count /= nb + 1 then
					-- Internal error: this should already have been reported in ET_FEATURE_FLATTENER.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_builtin_query_c_call", 1, "wrong_number of arguments.")
			elseif not attached a_feature.result_type_set as l_result_type_set then
					-- Internal error: `a_feature' is a query.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_builtin_query_c_call", 2, "feature is not a query.")
			else
				print_declaration_type_cast_from_c (l_result_type_set.static_type.primary_type, current_file)
				current_file.put_character ('(')
				current_file.put_string (a_c_function_name)
				current_file.put_character ('(')
				if a_context then
					current_file.put_string (c_ac)
					l_comma_needed := True
				end
				from
					if a_static then
						i := 1
					end
				until
					i > nb
				loop
					if l_comma_needed then
						print_comma
					else
						l_comma_needed := True
					end
					current_file.put_character ('(')
					if i = 0 then
						l_argument := call_operands.first
						l_actual_type_set := dynamic_type_set (l_argument)
						l_formal_type := a_target_type
					else
						l_argument := call_operands.item (i + 1)
						l_actual_type_set := dynamic_type_set (l_argument)
						l_formal_type := argument_type_set_in_feature (i, a_feature).static_type
					end
					print_attachment_expression (l_argument, l_actual_type_set, l_formal_type)
					current_file.put_character (')')
					i := i + 1
				end
				current_file.put_character (')')
				current_file.put_character (')')
			end
		end

	print_builtin_any_conforms_to_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of `a_feature' corresponding
			-- to built-in feature 'ANY.conforms_to'.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
		local
			l_arguments: detachable ET_FORMAL_ARGUMENT_LIST
			l_argument_type_set: ET_DYNAMIC_TYPE_SET
			l_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_non_conforming_types: ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			i, nb: INTEGER
		do
			l_arguments := a_feature.arguments
			if l_arguments = Void or else l_arguments.count /= 1 then
					-- Internal error: this error should have been reported by the parse.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_builtin_any_conforms_to_body", 1, "wrong number of arguments.")
			else
				l_argument_type_set := argument_type_set (1)
				nb := l_argument_type_set.count
				l_conforming_types := conforming_types
				l_conforming_types.resize (nb)
				l_non_conforming_types := non_conforming_types
				l_non_conforming_types.resize (nb)
				from i := 1 until i > nb loop
					l_dynamic_type := l_argument_type_set.dynamic_type (i)
					if current_type.conforms_to_primary_type (l_dynamic_type) then
						l_conforming_types.put_last (l_dynamic_type)
					else
						l_non_conforming_types.put_last (l_dynamic_type)
					end
					i := i + 1
				end
				if l_non_conforming_types.is_empty then
						-- `current_type' conforms to all types of `l_argument_type_set'.
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (';')
					current_file.put_new_line
				elseif l_conforming_types.is_empty then
						-- `current_type' conforms to none of the types of `l_argument_type_set'.
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_false)
					current_file.put_character (';')
					current_file.put_new_line
				elseif l_non_conforming_types.count < l_conforming_types.count then
					print_indentation
					current_file.put_string (c_switch)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_attribute_type_id_access (l_arguments.formal_argument (1).name, current_dynamic_system.any_type, True)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					nb := l_non_conforming_types.count
					from i := 1 until i > nb loop
						l_dynamic_type := l_non_conforming_types.dynamic_type (i)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						i := i + 1
					end
					indent
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_false)
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_break)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_string (c_default)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
				else
					print_indentation
					current_file.put_string (c_switch)
					current_file.put_character (' ')
					current_file.put_character ('(')
					print_attribute_type_id_access (l_arguments.formal_argument (1).name, current_dynamic_system.any_type, True)
					current_file.put_character (')')
					current_file.put_character (' ')
					current_file.put_character ('{')
					current_file.put_new_line
					nb := l_conforming_types.count
					from i := 1 until i > nb loop
						l_dynamic_type := l_conforming_types.dynamic_type (i)
						print_indentation
						current_file.put_string (c_case)
						current_file.put_character (' ')
						current_file.put_integer (l_dynamic_type.id)
						current_file.put_character (':')
						current_file.put_new_line
						i := i + 1
					end
					indent
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_true)
					current_file.put_character (';')
					current_file.put_new_line
					print_indentation
					current_file.put_string (c_break)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_string (c_default)
					current_file.put_character (':')
					current_file.put_new_line
					indent
					print_indentation
					print_result_name (current_file)
					current_file.put_character (' ')
					current_file.put_character ('=')
					current_file.put_character (' ')
					current_file.put_string (c_eif_false)
					current_file.put_character (';')
					current_file.put_new_line
					dedent
					print_indentation
					current_file.put_character ('}')
				end
				l_conforming_types.wipe_out
				l_non_conforming_types.wipe_out
			end
		end

	print_builtin_any_copy_body (a_feature: ET_EXTERNAL_ROUTINE)
			-- Print to `current_file' the body of `a_feature' corresponding
			-- to built-in feature 'ANY.copy'.
		require
			a_feature_not_void: a_feature /= Void
			valid_feature: current_feature.static_feature = a_feature
		do
			print_builtin_any_standard_copy_body (a_feature)
		end

	print_builtin_any_copy_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding to built-in feature 'ANY.copy'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			print_builtin_any_standard_copy_call (a_feature, a_target_type, a_check_void_target)
		end

	print_builtin_any_deep_twin_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding to built-in feature 'ANY.deep_twin'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		do
			deep_twin_types.force_last (a_target_type)
			current_file.put_string (c_ge_deep_twin)
			current_file.put_integer (a_target_type.id)
			current_file.put_character ('(')
			current_file.put_string (c_ac)
			print_comma
			print_target_expression (call_operands.first, a_target_type, a_check_void_target)
			current_file.put_character (',')
			current_file.put_character (' ')
			current_file.put_character ('0')
			current_file.put_character (')')
		end

	print_builtin_any_generating_type_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding to built-in feature 'ANY.generating_type'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_dynamic_type: ET_DYNAMIC_PRIMARY_TYPE
			l_attached_index: INTEGER
			l_meta_type: detachable ET_DYNAMIC_PRIMARY_TYPE
		do
			if a_target_type.base_class.is_type_class then
				l_dynamic_type := current_dynamic_system.none_type
				l_meta_type := current_dynamic_system.type_none_type
				l_attached_index := 1
			else
				l_dynamic_type := a_target_type
				l_meta_type := l_dynamic_type.meta_type
				l_attached_index := 0
			end
			if l_meta_type = Void then
					-- Internal error: the meta type of the target type should
					-- have been computed when analyzing the dynamic type sets of
					-- `a_feature'.
				set_fatal_error
				error_handler.report_giaac_error (generator, "print_builtin_any_generating_type_call", 1, "meta type not found.")
			else
				current_file.put_character ('(')
				print_type_declaration (l_meta_type, current_file)
				current_file.put_character (')')
				current_file.put_character ('&')
				current_file.put_character ('(')
				current_file.put_string (c_ge_types)
				current_file.put_character ('[')
				current_file.put_integer (l_dynamic_type.id)
				current_file.put_character (']')
				current_file.put_character ('[')
				current_file.put_integer (l_attached_index)
				current_file.put_character (']')
				current_file.put_character (')')
			end
		end

	print_builtin_any_generator_call (a_feature: ET_DYNAMIC_FEATURE; a_target_type: ET_DYNAMIC_PRIMARY_TYPE; a_check_void_target: BOOLEAN)
			-- Print to `current_file' a call (static binding) to `a_feature'
			-- corresponding to built-in feature 'ANY.generator'.
			-- `a_target_type' is the dynamic type of the target.
			-- `a_check_void_target' means that we need to check whether the target is Void or not.
			-- Operands can be found in `call_operands'.
		require
			a_feature_not_void: a_feature /= Void
			a_target_type_not_void: a_target_type /= Void
			call_operands_not_empty: not call_operands.is_empty
		local
			l_string: STRING
		do
			if attached a_feature.result_type_set as l_result_type_set and then l_result_type_set.static_type.base_type.same_named_type_with_type_marks (current_system.string_32_type, tokens.implicit_detachable_type_mark, current_system.any_type, tokens.implicit_detachable_type_mark, current_system.any_type) then
				current_file.put_string (c_ge_ms32)
			else
				current_file.put_string (c_ge_ms8)
			end
			current_file.put_character ('(')
			l_string := a_target_type.base_class.upper_name
			print_escaped_string (l_string)
			current_fil