#define GE_USE_THREADS

/*
	description:

		"C declarations for the Gobo Eiffel runtime."

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2005-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_EIFFEL_H
#define GE_EIFFEL_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

/* Class name mapping as defined in the FreeELKS library. */
#ifndef EIF_INTEGER
#define EIF_INTEGER EIF_INTEGER_32
#endif
#ifndef EIF_CHARACTER
#define EIF_CHARACTER EIF_CHARACTER_8
#endif
#ifndef EIF_REAL
#define EIF_REAL EIF_REAL_32
#endif
#ifndef EIF_DOUBLE
#define EIF_DOUBLE EIF_REAL_64
#endif
#ifndef GE_ms
#if EIF_CHARACTER == EIF_CHARACTER_8
#define GE_ms(s,c) GE_ms8((s),(c))
#else
#define GE_ms(s,c) GE_ms32((s),(c))
#endif
#endif

#if defined(__USE_POSIX) || defined(__unix__) || defined(_POSIX_C_SOURCE)
#include <unistd.h>
#endif
#if !defined(WIN32) && \
	(defined(WINVER) || defined(_WIN32_WINNT) || defined(_WIN32) || \
	defined(__WIN32__) || defined(__TOS_WIN__) || defined(_MSC_VER) || \
	defined(__MINGW32__))
#define WIN32 1
#endif
#ifdef WIN32
#define EIF_WINDOWS 1
#define GE_WINDOWS
#define WIN32_LEAN_AND_MEAN 1 /* Needed when using winsock2.h. */
#include <windows.h>
#endif

/* See https://sourceforge.net/p/predef/wiki/OperatingSystems/ */
#if (defined(macintosh) || defined(Macintosh))
#define EIF_MAC 1
#define EIF_MACOSX 1
#define GE_MACOS
#endif
#if (defined(__APPLE__) && defined(__MACH__))
/* Apparently ISE does not define EIF_MASOSX for Mac OS X >=10.4 (see EXECUTION_ENVIRONMENT.available_cpu_count) */
#define EIF_MAC 1
#define GE_MACOS
#endif

#if (defined(VMS) || defined(__VMS))
#define EIF_VMS 1
#endif

#if (defined(__VXWORKS__) || defined(__vxworks))
#define EIF_VXWORKS 1
#endif

#if defined(__OpenBSD__)
#define GE_OPENBSD
#endif

#define BYTEORDER 0x1234

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stddef.h>

#define EIF_OS_WINNT 	1
#define EIF_OS_LINUX 	2
#define EIF_OS_DARWIN 	4
#define EIF_OS_VXWORKS	11
#define EIF_OS_VMS	12

/* Platform definition */
/* Unix definition */
#define EIF_IS_UNIX EIF_TRUE
#define EIF_OS EIF_OS_LINUX
/* Windows definition */
#ifdef EIF_WINDOWS
#define EIF_IS_WINDOWS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_WINNT
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_WINDOWS EIF_FALSE
#endif
/* VMS definition */
#ifdef EIF_VMS
#define EIF_IS_VMS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_VMS
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VMS EIF_FALSE
#endif
/* MAC definition */
#ifdef EIF_MAC
#define EIF_IS_MAC EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_DARWIN
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_MAC EIF_FALSE
#endif
/* VxWorks definition */
#ifdef EIF_VXWORKS
#define EIF_IS_VXWORKS EIF_TRUE
#undef EIF_OS
#define EIF_OS EIF_OS_VXWORKS
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VXWORKS EIF_FALSE
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef _MSC_VER /* MSVC */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#if defined (__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed long int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#include <inttypes.h>
#endif
#endif

/* Portable integer pointers */
#ifdef EIF_WINDOWS
#ifndef _INTPTR_T_DEFINED
#define _INTPTR_T_DEFINED
#ifdef _WIN64
typedef __int64 intptr_t;
#else
typedef int intptr_t;
#endif
#endif
#ifndef _UINTPTR_T_DEFINED
#define _UINTPTR_T_DEFINED
#ifdef _WIN64
typedef unsigned __int64 uintptr_t;
#else
typedef unsigned int uintptr_t;
#endif
#endif
#endif

/* C type for underlying integer type identifying object's dynamic type. */
typedef uint16_t EIF_TYPE_INDEX;

/*
 * Abstraction representing an Eiffel type.
 * It is made of a compiler type-id,
 * and of some annotations (attached/detachable/separate/variant/frozen).
 */
typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	EIF_TYPE_INDEX volatile annotations;
} EIF_TYPE;

/*
 * Since EIF_TYPE and EIF_ENCODED_TYPE have the same size, the encoded version
 * is basically a memcpy version of the EIF_TYPE representation.
 * It is used to provide backward compatibility to most Eiffel and
 * C APIs manipulating types as an INTEGER.
 */
typedef int32_t EIF_ENCODED_TYPE;
typedef EIF_ENCODED_TYPE EIF_TYPE_ID;
#define EIF_NO_TYPE (EIF_TYPE_ID)(-1)

/* SCOOP */
#ifdef GE_USE_SCOOP
typedef struct GE_scoop_region_struct GE_scoop_region;
#endif

/* Basic Eiffel types */
typedef char EIF_BOOLEAN;
typedef unsigned char EIF_CHARACTER_8;
typedef uint32_t EIF_CHARACTER_32;
typedef int8_t EIF_INTEGER_8;
typedef int16_t EIF_INTEGER_16;
typedef int32_t EIF_INTEGER_32;
typedef int64_t EIF_INTEGER_64;
typedef uint8_t EIF_NATURAL_8;
typedef uint16_t EIF_NATURAL_16;
typedef uint32_t EIF_NATURAL_32;
typedef uint64_t EIF_NATURAL_64;
typedef volatile void* EIF_POINTER;
typedef float EIF_REAL_32;
typedef double EIF_REAL_64;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
} EIF_ANY;
typedef EIF_ANY* EIF_REFERENCE;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
#ifdef GE_HAS_STRING_8_ONCE_PER_OBJECT
	void* volatile onces;
#endif
	EIF_REFERENCE volatile area;
	EIF_INTEGER volatile count;
} EIF_STRING_8;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
#ifdef GE_HAS_STRING_32_ONCE_PER_OBJECT
	void* volatile onces;
#endif
	EIF_REFERENCE volatile area;
	EIF_INTEGER volatile count;
} EIF_STRING_32;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
#ifdef GE_HAS_IMMUTABLE_STRING_8_ONCE_PER_OBJECT
	void* volatile onces;
#endif
	EIF_REFERENCE volatile area;
	EIF_INTEGER volatile count;
} EIF_IMMUTABLE_STRING_8;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
#ifdef GE_HAS_IMMUTABLE_STRING_32_ONCE_PER_OBJECT
	void* volatile onces;
#endif
	EIF_REFERENCE volatile area;
	EIF_INTEGER volatile count;
} EIF_IMMUTABLE_STRING_32;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
#ifdef GE_HAS_SPECIAL_ONCE_PER_OBJECT
	void* volatile onces;
#endif
	EIF_INTEGER volatile capacity;
	EIF_INTEGER volatile count;
} EIF_SPECIAL;

/* SCOOP */
typedef uint16_t EIF_SCP_PID; /* Processor ID */
#ifdef GE_USE_SCOOP
#define RTS_PID(o) (EIF_SCP_PID)(uintptr_t)(((EIF_REFERENCE)(o))->region)
#else
#define RTS_PID(o) (EIF_SCP_PID)0
#endif

#ifdef EIF_WINDOWS
typedef wchar_t EIF_NATIVE_CHAR;
#else
typedef char EIF_NATIVE_CHAR;
#endif
typedef EIF_NATIVE_CHAR* EIF_FILENAME;

#define EIF_VOID ((EIF_REFERENCE)0)
#define EIF_FALSE ((EIF_BOOLEAN)'\0')
#define EIF_TRUE ((EIF_BOOLEAN)'\1')
#define EIF_TEST(x) ((x) ? EIF_TRUE : EIF_FALSE)

#define EIF_IS_WORKBENCH EIF_FALSE
#define EIF_POINTER_DISPLAY "lX"

/* For INTEGER and NATURAL manifest constants */
#define GE_int8(x) x
#define GE_nat8(x) x
#define GE_int16(x) x
#define GE_nat16(x) x
#define GE_int32(x) x##L
#define GE_nat32(x) x##U
#if defined (_MSC_VER) && (_MSC_VER < 1400) /* MSC older than v8 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#elif defined(__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#else /* ISO C 99 */
#define GE_int64(x) x##LL
#define GE_nat64(x) x##ULL
#endif
#ifdef __LCC__
/* lcc-win32 reports a constant overflow for -21474836478. */
#define GE_min_int32 (-GE_int32(2147483647)-GE_int32(1))
#else
#define GE_min_int32 GE_int32(-2147483648)
#endif
#define GE_max_int32 GE_int32(2147483647)
#if defined(__LCC__) || defined(__GNUC__) || defined(__MINGW32__)
/* lcc-win32 reports a constant overflow for -9223372036854775808. */
/* gcc and mingw-win64 warn that integer constant is so large that it is unsigned. */
#define GE_min_int64 (-GE_int64(9223372036854775807)-GE_int64(1))
#else
#define GE_min_int64 GE_int64(-9223372036854775808)
#endif
#if defined(__LCC__) && !defined(_WIN64)
/* lcc-win32 does not consider 64 bit constants as constants in case statement. */
#define GE_case_int64(x) ((int32_t)(x))
#define GE_case_nat64(x) ((uint32_t)(x))
#else
#define GE_case_int64(x) (x)
#define GE_case_nat64(x) (x)
#endif

#ifdef _WIN64
#define GE_IS_64_BITS EIF_TRUE
#else
#define GE_IS_64_BITS EIF_TEST(sizeof(void*)==8)
#endif

/* Posix threads */
#if !defined(EIF_WINDOWS)
#define GE_USE_POSIX_THREADS
#endif

#ifdef _MSC_VER /* MSVC */
/* MSVC does not support ISO C 99's 'snprintf' from stdio.h */
#define snprintf(a,b,c,d) sprintf(a,c,d)
#endif

/*
 * Gobo compiler version.
 * Starts with 6080 (looks like GOBO) followed by 5 digits.
 */
#define GE_compiler_version() 608000001

/*
	Interoperability with ISE.
*/
#define RTI64C(x) GE_int64(x)
#define RTU64C(x) GE_nat64(x)
#define EIF_OBJECT EIF_REFERENCE
#define EIF_OBJ EIF_OBJECT
#define OVERHEAD sizeof(EIF_ANY)
/* Function pointer call to make sure all arguments are correctly pushed onto stack. */
/* FUNCTION_CAST is for standard C calls. */
/* FUNCTION_CAST_TYPE is for non-standard C calls. */
#define FUNCTION_CAST(r_type,arg_types) (r_type (*) arg_types)
#define FUNCTION_CAST_TYPE(r_type,call_type,arg_types) (r_type (call_type *) arg_types)
#define SIGBLOCK
#define SIGRESUME
#define rt_public				/* default C scope */
#define rt_private static		/* static outside a block means private */
#define rt_shared				/* data shared between modules, but not public */
typedef intptr_t rt_int_ptr;
typedef uintptr_t rt_uint_ptr;
#define RTMS(s) GE_str8(s)
#define RTMS_EX(s,c) GE_ms8((s),(c))

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C types used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_THREAD_TYPES_H
#define GE_THREAD_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#define EIF_THREADS
#ifdef GE_USE_POSIX_THREADS
#define EIF_POSIX_THREADS
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef GE_USE_POSIX_THREADS
#include <pthread.h>
#include <semaphore.h>
#elif defined EIF_WINDOWS
#include <windows.h>
#include <process.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS

#define EIF_THR_TYPE            pthread_t
#define EIF_CS_TYPE				pthread_mutex_t
#define EIF_MUTEX_TYPE          pthread_mutex_t
#define EIF_COND_TYPE			pthread_cond_t
#define EIF_SEM_TYPE			sem_t
#define EIF_RWL_TYPE			pthread_rwlock_t

#elif defined EIF_WINDOWS

#define EIF_THR_TYPE            HANDLE
#define EIF_CS_TYPE				CRITICAL_SECTION
#define EIF_MUTEX_TYPE          CRITICAL_SECTION
#define EIF_SEM_TYPE            HANDLE

typedef volatile struct {
		/* Semaphore used to queue up threads waiting for the condition to become signaled. */
	EIF_SEM_TYPE* volatile semaphore;
		/* Serialize access to fields of Current. */
	EIF_CS_TYPE* volatile csection;
		/* Number of waiters. */
	unsigned long volatile num_waiting;
		/* Number of already awoken. */
	unsigned long volatile num_wake;
		/* Number of time we signaled/broadcasted for improving fairness.
		 * This ensures one thread won't steal wakeups from other threads in queue. */
	unsigned long volatile generation;
} EIF_COND_TYPE;

typedef volatile struct {
	EIF_MUTEX_TYPE* volatile m; /* Internal monitor lock. */
	int volatile rwlock; /* >0 = # readers, <0 = writer, 0 = none */
	EIF_COND_TYPE* volatile readers_ok; /* Start waiting readers. */
	unsigned int volatile waiting_writers; /* Number of waiting writers. */
	EIF_COND_TYPE* volatile writers_ok; /* Start a waiting writer. */
} EIF_RWL_TYPE;

#endif

typedef volatile struct {
	unsigned int volatile priority;
	unsigned int volatile stack_size;
} EIF_THR_ATTR_TYPE;

/* Struct for thread context. */
typedef volatile struct GE_thread_context_struct GE_thread_context;
struct GE_thread_context_struct {
	EIF_THR_TYPE volatile thread_id; /* Thread identifier for associated thread. */
	EIF_REFERENCE volatile current; /* Eiffel root object. */
	void (*volatile routine)(EIF_REFERENCE, EIF_INTEGER); /* Eiffel routine. */
	void (*volatile set_terminated)(EIF_REFERENCE, EIF_BOOLEAN); /* Eiffel routine to set {THREAD}.terminated. */
	unsigned int volatile initial_priority; /* Initial priority. */
	EIF_THR_TYPE volatile last_thread_id; /* Last thread created from current thread. */
	int volatile n_children; /* Number of direct thread children. */
	EIF_MUTEX_TYPE* volatile children_mutex; /* Mutex to wait for thread children. */
	EIF_COND_TYPE* volatile children_cond; /* Condition variable to wait for thread children. */
	GE_thread_context* volatile parent_context;	/* Context of parent thread, NULL if main thread. */
	int volatile thread_exiting; /* Has current thread already called GE_thread_exit? */
	int volatile is_alive; /* Is current thread still alive? */
	void* volatile wel_per_thread_data; /* WEL private data */
#ifdef GE_USE_SCOOP
	int volatile is_scoop_processor; /* Is current thread a SCOOP processor? */
#endif
};

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to implement once features"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_ONCE_H
#define GE_ONCE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Struct to keep track of the call status
 * and results of once features.
 */
typedef volatile struct {
	EIF_BOOLEAN* volatile boolean_value;
	EIF_REFERENCE* volatile boolean_exception;
	unsigned char* volatile boolean_status;
	uint32_t volatile boolean_count;
	EIF_CHARACTER_8* volatile character_8_value;
	EIF_REFERENCE* volatile character_8_exception;
	unsigned char* volatile character_8_status;
	uint32_t volatile character_8_count;
	EIF_CHARACTER_32* volatile character_32_value;
	EIF_REFERENCE* volatile character_32_exception;
	unsigned char* volatile character_32_status;
	uint32_t volatile character_32_count;
	EIF_INTEGER_8* volatile integer_8_value;
	EIF_REFERENCE* volatile integer_8_exception;
	unsigned char* volatile integer_8_status;
	uint32_t volatile integer_8_count;
	EIF_INTEGER_16* volatile integer_16_value;
	EIF_REFERENCE* volatile integer_16_exception;
	unsigned char* volatile integer_16_status;
	uint32_t volatile integer_16_count;
	EIF_INTEGER_32* volatile integer_32_value;
	EIF_REFERENCE* volatile integer_32_exception;
	unsigned char* volatile integer_32_status;
	uint32_t volatile integer_32_count;
	EIF_INTEGER_64* volatile integer_64_value;
	EIF_REFERENCE* volatile integer_64_exception;
	unsigned char* volatile integer_64_status;
	uint32_t volatile integer_64_count;
	EIF_NATURAL_8* volatile natural_8_value;
	EIF_REFERENCE* volatile natural_8_exception;
	unsigned char* volatile natural_8_status;
	uint32_t volatile natural_8_count;
	EIF_NATURAL_16* volatile natural_16_value;
	EIF_REFERENCE* volatile natural_16_exception;
	unsigned char* volatile natural_16_status;
	uint32_t volatile natural_16_count;
	EIF_NATURAL_32* volatile natural_32_value;
	EIF_REFERENCE* volatile natural_32_exception;
	unsigned char* volatile natural_32_status;
	uint32_t volatile natural_32_count;
	EIF_NATURAL_64* volatile natural_64_value;
	EIF_REFERENCE* volatile natural_64_exception;
	unsigned char* volatile natural_64_status;
	uint32_t volatile natural_64_count;
	EIF_POINTER* volatile pointer_value;
	EIF_REFERENCE* volatile pointer_exception;
	unsigned char* volatile pointer_status;
	uint32_t volatile pointer_count;
	EIF_REAL_32* volatile real_32_value;
	EIF_REFERENCE* volatile real_32_exception;
	unsigned char* volatile real_32_status;
	uint32_t volatile real_32_count;
	EIF_REAL_64* volatile real_64_value;
	EIF_REFERENCE* volatile real_64_exception;
	unsigned char* volatile real_64_status;
	uint32_t volatile real_64_count;
	EIF_REFERENCE* volatile reference_value;
	EIF_REFERENCE* volatile reference_exception;
	unsigned char* volatile reference_status;
	uint32_t volatile reference_count;
	EIF_REFERENCE* volatile procedure_exception;
	unsigned char* volatile procedure_status;
	uint32_t volatile procedure_count;
} GE_onces;

/*
 * Variable to keep track of the call status
 * and results of once-per-process features.
 */
extern GE_onces* GE_process_onces;

/*
 * Initialize `GE_process_onces'.
 */
extern void GE_init_onces(
	uint32_t a_boolean_count,
	uint32_t a_character_8_count,
	uint32_t a_character_32_count,
	uint32_t a_integer_8_count,
	uint32_t a_integer_16_count,
	uint32_t a_integer_32_count,
	uint32_t a_integer_64_count,
	uint32_t a_natural_8_count,
	uint32_t a_natural_16_count,
	uint32_t a_natural_32_count,
	uint32_t a_natural_64_count,
	uint32_t a_pointer_count,
	uint32_t a_real_32_count,
	uint32_t a_real_64_count,
	uint32_t a_reference_count,
	uint32_t a_procedure_count);

/*
 * Create a new 'GE_onces' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_onces* GE_new_onces(
	uint32_t a_boolean_count,
	uint32_t a_character_8_count,
	uint32_t a_character_32_count,
	uint32_t a_integer_8_count,
	uint32_t a_integer_16_count,
	uint32_t a_integer_32_count,
	uint32_t a_integer_64_count,
	uint32_t a_natural_8_count,
	uint32_t a_natural_16_count,
	uint32_t a_natural_32_count,
	uint32_t a_natural_64_count,
	uint32_t a_pointer_count,
	uint32_t a_real_32_count,
	uint32_t a_real_64_count,
	uint32_t a_reference_count,
	uint32_t a_procedure_count);

/*
 * Free memory allocated by `a_onces'.
 */
extern void GE_free_onces(GE_onces* a_onces);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class EXCEPTION"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_EXCEPTION_H
#define GE_EXCEPTION_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_ONCE_H
#include "ge_once.h"
#endif
#ifdef GE_USE_THREADS
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#endif

#include <setjmp.h>

/*
 * On Linux glibc systems, we need to use sig* versions of jmp_buf,
 * setjmp and longjmp to preserve the signal handling context.
 * One way to detect this is if _SIGSET_H_types has
 * been defined in /usr/include/setjmp.h.
 * NOTE: ANSI only recognizes the non-sig versions.
 */
#if (defined(_SIGSET_H_types) && !defined(__STRICT_ANSI__))
#define GE_jmp_buf sigjmp_buf
#define GE_setjmp(x) sigsetjmp(*(GE_jmp_buf*)&(x),1)
#define GE_longjmp(x,y) siglongjmp(*(GE_jmp_buf*)&(x),(y))
#else
#define GE_jmp_buf jmp_buf
#define GE_setjmp(x) setjmp(*(GE_jmp_buf*)&(x))
#define GE_longjmp(x,y) longjmp(*(GE_jmp_buf*)&(x),(y))
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Predefined exception codes.
 */
#define GE_EX_VOID		1			/* Feature applied to void reference */
#define GE_EX_MEM		2			/* No more memory */
#define GE_EX_PRE		3			/* Pre-condition violated */
#define GE_EX_POST		4			/* Post-condition violated */
#define GE_EX_FLOAT		5			/* Floating point exception (signal SIGFPE) */
#define GE_EX_CINV		6			/* Class invariant violated */
#define GE_EX_CHECK		7			/* Check instruction violated */
#define GE_EX_FAIL		8			/* Routine failure */
#define GE_EX_WHEN		9			/* Unmatched inspect value */
#define GE_EX_VAR		10			/* Non-decreasing loop variant */
#define GE_EX_LINV		11			/* Loop invariant violated */
#define GE_EX_SIG		12			/* Operating system signal */
#define GE_EX_BYE		13			/* Eiffel run-time panic */
#define GE_EX_RESC		14			/* Exception in rescue clause */
#define GE_EX_OMEM		15			/* Out of memory (cannot be ignored) */
#define GE_EX_RES		16			/* Resumption failed (retry did not succeed) */
#define GE_EX_CDEF		17			/* Create on deferred */
#define GE_EX_EXT		18			/* External event */
#define GE_EX_VEXP		19			/* Void assigned to expanded */
#define GE_EX_HDLR		20			/* Exception in signal handler */
#define GE_EX_IO		21			/* I/O error */
#define GE_EX_SYS		22			/* Operating system error */
#define GE_EX_RETR		23			/* Retrieval error */
#define GE_EX_PROG		24			/* Developer exception */
#define GE_EX_FATAL		25			/* Eiffel run-time fatal error */
#define GE_EX_DOL		26			/* $ applied to melted feature */
#define GE_EX_ISE_IO	27			/* I/O error raised by the ISE Eiffel runtime */
#define GE_EX_COM		28			/* COM error raised by EiffelCOM runtime */
#define GE_EX_RT_CHECK	29			/* Runtime check error such as out-of-bound array access */
#define GE_EX_OLD		30			/* Old violation */
#define GE_EX_SEL		31			/* Serialization failure */
#define GE_EX_DIRTY		32			/* SCOOP processor dirty exception. */
#define GE_EX_NEX		32			/* Number of internal exceptions */

/*
 * String buffer used to build the exception trace.
 */
typedef volatile struct {
	char* volatile area;
	uint32_t volatile count;
	uint32_t volatile capacity;
} GE_exception_trace_buffer;

/*
 * Information about the feature being executed.
 */
typedef volatile struct GE_call_struct GE_call;
struct GE_call_struct {
#ifdef GE_USE_CURRENT_IN_EXCEPTION_TRACE
	void* volatile object; /* Current object */
#endif
	const char* volatile class_name;
	const char* volatile feature_name;
	GE_call* volatile caller; /* previous feature in the call chain */
};

/*
 * Context of features containing a rescue clause.
 */
typedef volatile struct GE_rescue_struct GE_rescue;
struct GE_rescue_struct {
	GE_jmp_buf jb;
	GE_rescue* volatile previous; /* previous context in the call chain */
};

/*
 * Information about the execution context.
 * One such struct per thread.
 */
typedef volatile struct GE_context_struct GE_context;
struct GE_context_struct {
	GE_call* volatile call; /* Call stack */
	uint32_t volatile in_assertion; /* Is an assertion evaluated? */
	GE_rescue* volatile last_rescue; /* Context of last feature entered containing a rescue clause */
	uint32_t volatile in_rescue; /* Number of rescue clauses currently being executed */
	uint32_t volatile in_qualified_call; /* Is the current call a qualified call? 1 means that it is a regular call, 2 means that it is a creation call. */
	EIF_REFERENCE volatile exception_manager; /* Exception manager */
	char volatile raising_exception; /* Is an exception currently being raised? */
	char volatile exception_trace_enabled; /* Should exception trace be displayed? */
	long volatile exception_code; /* Code of the exception currently being raised, 0 otherwise */
	const char* volatile exception_tag; /* Tag of the exception currently being raised, NULL otherwise */
	GE_exception_trace_buffer exception_trace_buffer; /* String buffer used to build the exception trace */
	GE_exception_trace_buffer last_exception_trace; /* Last non-routine-failure exception trace */
	int volatile signal_number; /* Number of last signal received */
	int volatile pre_ecma_mapping_status; /* Do we map old names to new name? (i.e. STRING to STRING_8, INTEGER to INTEGER_32, ...). */
#ifdef GE_USE_THREADS
	GE_thread_context* volatile thread; /* Thread context */
	GE_onces* volatile process_onces; /* Cache for status and results of onces-per-process */
	GE_onces* volatile thread_onces; /* Status and results of onces-per-thread */
#endif
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region; /* SCOOP region whose processor is executing the current code */
	char volatile is_region_alive;
#endif
};
#define TC GE_context

/*
 * New execution context.
 */
extern GE_context* GE_new_context(int is_scoop_processor);

/*
 * Execution context of main thread.
 */
extern GE_context* GE_main_context;

/*
 * Execution context of current thread.
 */
extern GE_context* GE_current_context(void);

/*
 * Initialization of exception handling.
 */
extern void GE_init_exception(GE_context* context);

/*
 * Free memory allocated in `a_context' for exception handling.
 */
extern void GE_free_exception(GE_context* a_context);

/*
 * Pointer to function to create a new exception manager object
 * (of type ISE_EXCEPTION_MANAGER).
 */
extern EIF_REFERENCE (*GE_new_exception_manager)(GE_context*, EIF_BOOLEAN);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.init_exception_manager.
 */
extern void (*GE_init_exception_manager)(GE_context*);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.last_exception.
 */
extern EIF_REFERENCE (*GE_last_exception)(GE_context*);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.once_raise.
 */
extern void (*GE_once_raise)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.set_exception_data.
 */
extern void (*GE_set_exception_data)(GE_context*, EIF_INTEGER_32, EIF_BOOLEAN, EIF_INTEGER_32, EIF_INTEGER_32, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN);

/*
 * Exception tag associated with `a_code'.
 */
extern char* GE_exception_tag(long a_code);

/*
 * Append `a_string' to `a_trace'.
 * Resize area if needed.
 */
extern void GE_append_to_exception_trace_buffer(GE_exception_trace_buffer* a_trace, char* a_string);

/*
 * Wipe out `a_trace'.
 */
extern void GE_wipe_out_exception_trace_buffer(GE_exception_trace_buffer* a_trace);

/*
 * Raise an exception with code `a_code'.
 */
extern void GE_raise(long a_code);

/*
 * Raise an exception with code `a_code' and message `msg'.
 */
extern void GE_raise_with_message(long a_code, const char* msg);

/*
 * Raise an exception from EXCEPTION_MANAGER.
 */
extern void GE_developer_raise(long a_code, EIF_POINTER a_meaning, EIF_POINTER a_message);

/*
 * Raise exception which was raised the first time a once routine
 * was executed when executing it again.
 */
extern void GE_raise_once_exception(GE_context* a_context, EIF_REFERENCE a_exception);

/*
 * Raise exception which was raised when an old expression was evaluated.
 */
extern int GE_raise_old_exception(GE_context* a_context, EIF_REFERENCE a_exception);

/*
 * Exception, if any, which was last raised in `a_context'.
 */
extern EIF_REFERENCE GE_last_exception_raised(GE_context* a_context);

/*
 * Jump to execute the rescue of the last routine with a rescue
 * in the call stack.
 */
extern void GE_jump_to_last_rescue(GE_context* a_context);

/*
 * Set `in_assertion' to 'not b'.
 * Return the opposite of previous value.
 */
extern EIF_BOOLEAN GE_check_assert(EIF_BOOLEAN b);

/*
 * Check whether the type id of `obj' is not in `type_ids'.
 * If it is, then raise a CAT-call exception. Don't do anything if `obj' is Void.
 * `nb' is the number of ids in `type_ids' and is expected to be >0.
 * `type_ids' is sorted in increasing order.
 * Return `obj'.
 */
#define GE_catcall(obj,type_ids,nb) GE_check_catcall((obj),(type_ids),(nb))
extern EIF_REFERENCE GE_check_catcall(EIF_REFERENCE obj, EIF_TYPE_INDEX type_ids[], int nb);

/*
 * Check whether `obj' is Void.
 * If it is, then raise a call-on-void-target exception.
 * If `i' is provided, then include it in the message displayed
 * in the console to make debugging easier when `obj' is Void.
 * Return `obj'.
 */
#define GE_void(obj) ((obj)?(obj):GE_check_void(obj))
extern EIF_REFERENCE GE_check_void(EIF_REFERENCE obj);
#ifdef GE_DEBUG
#define GE_void2(obj,i) ((obj)?(obj):GE_check_void2((obj),(i)))
extern EIF_REFERENCE GE_check_void2(EIF_REFERENCE obj, EIF_INTEGER i);
#else
#define GE_void2(obj,i) ((obj)?(obj):GE_check_void(obj))
#endif

/*
 * Check whether `ptr' is a null pointer.
 * If it is, then raise a no-more-memory exception.
 * Return `ptr'.
 */
#define GE_null(ptr) GE_check_null(ptr)
extern void* GE_check_null(void* ptr);

/* Make a qualified call to `call'. */
#define GE_qualified(ac, call) (((ac)->in_qualified_call = 1), (call))
/* Make an unqualified call to `call'. */
#define GE_unqualified(ac, call) (((ac)->in_qualified_call = 0), (call))
/* Make a creation call to `call'. */
#define GE_creation(ac, call) (((ac)->in_qualified_call = 2), (call))

#ifdef EIF_WINDOWS
/*
 * Set default exception handler.
 */
extern void GE_set_windows_exception_filter(void);
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_THREAD_H
#define GE_THREAD_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS
#include <unistd.h>
#endif

#ifdef GE_USE_POSIX_THREADS
#	define EIF_TSD_VAL_TYPE        void*
#	define EIF_TSD_TYPE            pthread_key_t
#	define EIF_TSD_CREATE(key,msg) \
		if (pthread_key_create(&(key),NULL)) \
			GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (pthread_setspecific((key), (EIF_TSD_VAL_TYPE)(val))) \
			GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) (val = pthread_getspecific(key))
#	define EIF_TSD_GET(val_type,key,val,msg) \
		if (EIF_TSD_GET0(val_type,key,val) == (void*) 0) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) if (pthread_key_delete(key)) GE_raise_with_message(GE_EX_EXT, msg)
#elif defined EIF_WINDOWS
#	define EIF_TSD_VAL_TYPE        LPVOID
#	define EIF_TSD_TYPE            DWORD
#	define EIF_TSD_CREATE(key,msg) \
		if ((key=TlsAlloc())==TLS_OUT_OF_INDEXES) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (!TlsSetValue((key),(EIF_TSD_VAL_TYPE)(val))) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) \
		val=(val_type) TlsGetValue(key)
#	define EIF_TSD_GET(val_type,key,val,msg) \
		EIF_TSD_GET0(val_type,key,val); \
		if (GetLastError() != NO_ERROR) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) \
		if (!TlsFree(key)) GE_raise_with_message(GE_EX_EXT, msg)
#endif

/* Thread priority levels. */
#define EIF_MIN_THR_PRIORITY			0L
#define EIF_BELOW_NORMAL_THR_PRIORITY	100L
#define EIF_DEFAULT_THR_PRIORITY		127L
#define EIF_ABOVE_NORMAL_THR_PRIORITY	154L
#define EIF_MAX_THR_PRIORITY			255L

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
typedef volatile struct {
	EIF_POINTER* volatile boolean_mutex;
	EIF_POINTER* volatile character_8_mutex;
	EIF_POINTER* volatile character_32_mutex;
	EIF_POINTER* volatile integer_8_mutex;
	EIF_POINTER* volatile integer_16_mutex;
	EIF_POINTER* volatile integer_32_mutex;
	EIF_POINTER* volatile integer_64_mutex;
	EIF_POINTER* volatile natural_8_mutex;
	EIF_POINTER* volatile natural_16_mutex;
	EIF_POINTER* volatile natural_32_mutex;
	EIF_POINTER* volatile natural_64_mutex;
	EIF_POINTER* volatile pointer_mutex;
	EIF_POINTER* volatile real_32_mutex;
	EIF_POINTER* volatile real_64_mutex;
	EIF_POINTER* volatile reference_mutex;
	EIF_POINTER* volatile procedure_mutex;
} GE_once_mutexes;

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
extern GE_once_mutexes* GE_process_once_mutexes;

/*
 * Numbers of once-per-thread features.
 */
extern uint32_t GE_thread_onces_boolean_count;
extern uint32_t GE_thread_onces_character_8_count;
extern uint32_t GE_thread_onces_character_32_count;
extern uint32_t GE_thread_onces_integer_8_count;
extern uint32_t GE_thread_onces_integer_16_count;
extern uint32_t GE_thread_onces_integer_32_count;
extern uint32_t GE_thread_onces_integer_64_count;
extern uint32_t GE_thread_onces_natural_8_count;
extern uint32_t GE_thread_onces_natural_16_count;
extern uint32_t GE_thread_onces_natural_32_count;
extern uint32_t GE_thread_onces_natural_64_count;
extern uint32_t GE_thread_onces_pointer_count;
extern uint32_t GE_thread_onces_real_32_count;
extern uint32_t GE_thread_onces_real_64_count;
extern uint32_t GE_thread_onces_reference_count;
extern uint32_t GE_thread_onces_procedure_count;

/*
 * Create a new 'GE_once_mutexes' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_once_mutexes* GE_new_once_mutexes(
	uint32_t a_boolean_count,
	uint32_t a_character_8_count,
	uint32_t a_character_32_count,
	uint32_t a_integer_8_count,
	uint32_t a_integer_16_count,
	uint32_t a_integer_32_count,
	uint32_t a_integer_64_count,
	uint32_t a_natural_8_count,
	uint32_t a_natural_16_count,
	uint32_t a_natural_32_count,
	uint32_t a_natural_64_count,
	uint32_t a_pointer_count,
	uint32_t a_real_32_count,
	uint32_t a_real_64_count,
	uint32_t a_reference_count,
	uint32_t a_procedure_count);

/*
 * Keep track of the numbers of once-per-thread features.
 */
extern void GE_thread_onces_set_counts(
	uint32_t a_boolean_count,
	uint32_t a_character_8_count,
	uint32_t a_character_32_count,
	uint32_t a_integer_8_count,
	uint32_t a_integer_16_count,
	uint32_t a_integer_32_count,
	uint32_t a_integer_64_count,
	uint32_t a_natural_8_count,
	uint32_t a_natural_16_count,
	uint32_t a_natural_32_count,
	uint32_t a_natural_64_count,
	uint32_t a_pointer_count,
	uint32_t a_real_32_count,
	uint32_t a_real_64_count,
	uint32_t a_reference_count,
	uint32_t a_procedure_count);

/*
 * Initialize `process_onces' and `thread_onces' in `a_context'.
 */
extern void GE_thread_init_onces(GE_context* a_context);

/* Global mutex to protect creation of once-per-object data. */
extern EIF_POINTER GE_once_per_object_data_mutex;

/*
 * Initialize data to handle threads.
 * To be called at the beginning of the main function
 * on the main thread.
 */
extern void GE_init_thread(GE_context* a_context);

/*
 * Create a new thread with attributes `attr' and execute
 * Eiffel routine `routine' on object `current'.
 */
extern void GE_thread_create_with_attr(EIF_REFERENCE current, void (*routine)(EIF_REFERENCE, EIF_INTEGER), void (*set_terminated)(EIF_REFERENCE,EIF_BOOLEAN), EIF_THR_ATTR_TYPE* attr, int is_scoop_processor);

/*
 * Execution context of current thread.
 */
extern GE_context* GE_thread_current_context(void);

/*
 * Execution context of current thread.
 * Return Null in case of non-Eiffel threads.
 */
extern GE_context* GE_unprotected_thread_current_context(void);

/*
 * Thead ID of current thread.
 */
extern EIF_POINTER GE_thread_id(void);

/*
 * Thread ID of last thread created from current thread.
 */
extern EIF_POINTER GE_last_thread_created(void);

#ifdef EIF_WINDOWS
/*
 * Support for Windows GUI that requires that all GUI operations are performed in the same thread.
 * Allocate new structure of the given size `a_size', assign it to `wel_per_thread_data'.
 * Return newly allocated memory block. It will be freed automatically on thread termination.
 */
extern void* GE_thread_create_wel_per_thread_data(size_t a_size);
#endif

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated. The calling thread must be the
 * direct parent of the thread, or the function might loop indefinitely.
 */
extern void GE_thread_wait(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE));

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated, or reaching `a_timeout_ms'.
 * The calling thread must be the direct parent of the thread,
 * or the function might loop indefinitely.
 */
extern EIF_BOOLEAN GE_thread_wait_with_timeout(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE), EIF_NATURAL_64 a_timeout_ms);

/*
 * Yields execution to other threads.
 */
extern void GE_thread_yield(void);

/*
 * The calling thread waits for all other children threads to terminate.
 */
extern void GE_thread_join_all(void);

/*
 * Function called to terminate a thread launched by Eiffel with `GE_thread_create_with_attr'.
 * This function must be called from the thread itself (not the parent).
 */
extern void GE_thread_exit(void);

/*
 * Default thread priority level.
 */
extern EIF_INTEGER GE_thread_default_priority(void);

/*
 * Minimum thread priority level.
 */
extern EIF_INTEGER GE_thread_min_priority(void);

/*
 * Maximum thread priority level.
 */
extern EIF_INTEGER GE_thread_max_priority(void);

/*
 * Create a new mutex.
 */
extern EIF_POINTER GE_mutex_create(void);

/*
 * Lock mutex.
 */
extern void GE_mutex_lock(EIF_POINTER a_mutex);

/*
 * Try to lock mutex. Return True on success.
 */
extern EIF_BOOLEAN GE_mutex_try_lock(EIF_POINTER a_mutex);

/*
 * Unlock mutex.
 */
extern void GE_mutex_unlock(EIF_POINTER a_mutex);

/*
 * Destroy and free all resources used by mutex.
 */
extern void GE_mutex_destroy(EIF_POINTER a_mutex);

/*
 * Create a new semaphore allowing `a_count' threads
 * to go into a critical section.
 */
extern EIF_POINTER GE_semaphore_create(EIF_INTEGER a_count);

/*
 * Decrement semaphore count, waiting if necessary
 * until that becomes possible.
 */
extern void GE_semaphore_wait(EIF_POINTER a_semaphore);

/*
 * Has client been successful in decrementing semaphore
 * count without waiting?
 */
extern EIF_BOOLEAN GE_semaphore_try_wait(EIF_POINTER a_semaphore);

/*
 * Increment semaphore count.
 */
extern void GE_semaphore_post(EIF_POINTER a_semaphore);

/*
 * Destroy and free all resources used by semaphore.
 */
extern void GE_semaphore_destroy(EIF_POINTER a_semaphore);

/*
 * Create a new condition variable.
 */
extern EIF_POINTER GE_condition_variable_create(void);

/*
 * Unblock all threads blocked on condition variable.
 */
extern void GE_condition_variable_broadcast(EIF_POINTER a_condition_variable);

/*
 * Unblock one thread blocked on condition variable.
 */
extern void GE_condition_variable_signal(EIF_POINTER a_condition_variable);

/*
 * Block calling thread on condition variable.
 */
extern void GE_condition_variable_wait(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex);

/*
 * Block calling thread on condition variable for at most `a_timeout' milliseconds.
 * Return 1 is we got the condition variable on time, otherwise return 0.
 */
extern EIF_INTEGER GE_condition_variable_wait_with_timeout(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex, EIF_INTEGER a_timeout);

/*
 * Destroy and free all resources used by condition variable.
 */
extern void GE_condition_variable_destroy(EIF_POINTER a_condition_variable);

/*
 * Create a new read-write lock.
 */
extern EIF_POINTER GE_read_write_lock_create(void);

/*
 * Acquire a read lock. Multiple readers can go if there are no writer.
 */
extern void GE_read_write_lock_read_lock(EIF_POINTER a_read_write_lock);

/*
 * Acquire a write lock. Only a single write can proceed.
 */
extern void GE_read_write_lock_write_lock(EIF_POINTER a_read_write_lock);

/*
 * Unlock a read or write lock.
 */
extern void GE_read_write_lock_unlock(EIF_POINTER a_read_write_lock);

/*
 * Destroy and free all resources used by read-write lock.
 */
extern void GE_read_write_lock_destroy(EIF_POINTER a_read_write_lock);

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to manipulate native strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2013-2018, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_NATIVE_STRING_H
#define GE_NATIVE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#include <string.h>
#ifdef EIF_WINDOWS
#ifdef __LCC__
/* With lcc-win32, stat.h should be included before wchar.h. */
#include <sys/stat.h>
#endif
#include <wchar.h>
#else
#include <sys/types.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef EIF_WINDOWS

/* Macro used to manipulate native strings, i.e: (wchar_t*) */
#define GE_nstrlen wcslen /* size of string */
#define GE_nstrncpy wcsncpy /* Copy n characters of one string to another */
#define GE_nstrcpy wcscpy /* Copy one string to another */
#define GE_nstrncat wcsncat /* Append characters of a string */
#define GE_nstrcat wcscat /* Append a string */
#define GE_nstrstr wcsstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) L##quote /* Manifest Native string declaration */
#define GE_nstr_fopen _wfopen /* Open file using native string name */
#define GE_nstrcmp wcscmp /* Compare two strings. */
#define GE_nstrdup _wcsdup /* Duplicate string. */
#define GE_nstr_cat_ascii(dest, src) {							\
		int i;													\
		size_t dest_len, src_len;								\
		dest_len = rt_nstrlen (dest);							\
		src_len = strlen (src);									\
		for (i = 0; i < src_len; i++) {							\
			dest[dest_len + i] = (EIF_NATIVE_CHAR)src[i];		\
		}														\
		dest[dest_len + src_len] = (EIF_NATIVE_CHAR)0;			\
	}

#else /* not EIF_WINDOWS */

/* Macro used to manipulate native strings, i.e: (char*) */
#define GE_nstrlen strlen /* size of string */
#define GE_nstrncpy strncpy /* Copy n characters of one string to another */
#define GE_nstrcpy strcpy /* Copy one string to another */
#define GE_nstrncat strncat /* Append characters of a string */
#define GE_nstrcat strcat /* Append a string */
#define GE_nstrstr strstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) quote /* Manifest Native string declaration */
#define GE_nstr_fopen fopen /* Open file using native string name */
#define GE_nstrcmp strcmp /* Compare two strings. */
#define GE_nstrdup strdup /* Duplicate string. */
#define GE_nstr_cat_ascii strcat

#endif


#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class ARGUMENTS"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_ARGUMENTS_H
#define GE_ARGUMENTS_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_argc;
extern EIF_NATIVE_CHAR** GE_argv;

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement type information"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_TYPES_H
#define GE_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Type annotations.
 * When a type has no annotation, it means a detachable, non-separate, variant type.
 * In all other cases, there will be an annotation.
 */
#define ANNOTATION_MASK			0x007F	/* All possible annotations. */
#define ATTACHED_FLAG			0x0001
#define DETACHABLE_FLAG			0x0002	/* Only present when overriding an attached type. */
#define SEPARATE_FLAG			0x0004
#define VARIANT_FLAG			0x0008	/* Only present when overriding a frozen/poly type. */
#define UNUSABLE_FLAG			0x0010	/* Reserved for backward compatibility for storables. */
#define FROZEN_FLAG				0x0020
#define POLY_FLAG				0x0040

/*
 * Type flags.
 */
#define GE_TYPE_FLAG_SPECIAL		0x0010
#define GE_TYPE_FLAG_TUPLE			0x0020
#define GE_TYPE_FLAG_EXPANDED		0x0040
#define GE_TYPE_FLAG_DEFERRED		0x0080
#define GE_TYPE_FLAG_NONE			0x0100
#define GE_TYPE_FLAG_BASIC_MASK		0x000F /* One of "BOOLEAN", "CHARACTER_8", "CHARACTER_32", "INTEGER_8", "INTEGER_16", "INTEGER_32", "INTEGER_64", "NATURAL_8", "NATURAL_16", "NATURAL_32", "NATURAL_64", "POINTER", "REAL_32", "REAL_64" */
#define GE_TYPE_FLAG_BOOLEAN		0x0001
#define GE_TYPE_FLAG_CHARACTER_8	0x0002
#define GE_TYPE_FLAG_CHARACTER_32	0x0003
#define GE_TYPE_FLAG_INTEGER_8		0x0004
#define GE_TYPE_FLAG_INTEGER_16		0x0005
#define GE_TYPE_FLAG_INTEGER_32		0x0006
#define GE_TYPE_FLAG_INTEGER_64		0x0007
#define GE_TYPE_FLAG_NATURAL_8		0x0008
#define GE_TYPE_FLAG_NATURAL_16		0x0009
#define GE_TYPE_FLAG_NATURAL_32		0x000A
#define GE_TYPE_FLAG_NATURAL_64		0x000B
#define GE_TYPE_FLAG_POINTER		0x000C
#define GE_TYPE_FLAG_REAL_32		0x000D
#define GE_TYPE_FLAG_REAL_64		0x000E

/*
 * Convention for attribute types.
 * The values are in sync with REFLECTOR_CONSTANTS.
 */
#define GE_TYPE_KIND_INVALID		-1
#define GE_TYPE_KIND_POINTER		0
#define GE_TYPE_KIND_REFERENCE		1
#define GE_TYPE_KIND_CHARACTER_8	2
#define GE_TYPE_KIND_BOOLEAN		3
#define GE_TYPE_KIND_INTEGER_32		4
#define GE_TYPE_KIND_REAL_32		5
#define GE_TYPE_KIND_REAL_64		6
#define GE_TYPE_KIND_EXPANDED		7
#define GE_TYPE_KIND_INTEGER_8		9
#define GE_TYPE_KIND_INTEGER_16		10
#define GE_TYPE_KIND_INTEGER_64 	11
#define GE_TYPE_KIND_CHARACTER_32	12
#define GE_TYPE_KIND_NATURAL_8		13
#define GE_TYPE_KIND_NATURAL_16		14
#define GE_TYPE_KIND_NATURAL_32 	15
#define GE_TYPE_KIND_NATURAL_64 	16

/*
 * Object flags.
 */
#define GE_OBJECT_FLAG_MARKED		0x0001

/*
 * Ancestor relationship between two types X and Y.
 */
#ifdef GE_USE_ANCESTORS
typedef volatile struct {
	EIF_TYPE_INDEX volatile type_id; /* Type id of Y */
	EIF_BOOLEAN volatile conforms; /* Does X conform to Y? */
	void (**volatile qualified_calls)(); /* Function pointers, indexed by call id, when the static type of the target is Y and the dynamic type is X */
} GE_ancestor;
#endif

/*
 * Attribute.
 */
#ifdef GE_USE_ATTRIBUTES
typedef volatile struct {
#ifdef GE_USE_ATTRIBUTE_NAME
	const char* volatile name; /* Attribute name */
#endif
#ifdef GE_USE_ATTRIBUTE_TYPE_ID
	EIF_ENCODED_TYPE volatile type_id; /* Static type id */
#endif
#ifdef GE_USE_ATTRIBUTE_OFFSET
	uint32_t volatile offset; /* Address offset in object */
#endif
} GE_attribute;
#endif

/*
 * Type information.
 */
typedef volatile struct {
	EIF_TYPE_INDEX volatile type_id;
	uint16_t volatile flags;
#ifdef GE_USE_TYPE_GENERATOR
	const char* volatile generator; /* Generator class name */
#endif
#ifdef GE_USE_TYPE_NAME
	const char* volatile name; /* Full type name */
#endif
#ifdef GE_USE_TYPE_GENERIC_PARAMETERS
	EIF_ENCODED_TYPE* volatile generic_parameters;
	uint32_t volatile generic_parameter_count;
#endif
#ifdef GE_USE_ANCESTORS
	GE_ancestor** volatile ancestors;
	uint32_t volatile ancestor_count;
#endif
#ifdef GE_USE_ATTRIBUTES
	GE_attribute** volatile attributes;
	uint32_t volatile attribute_count;
#endif
#ifdef GE_USE_TYPE_OBJECT_SIZE
	uint64_t volatile object_size;
#endif
	EIF_REFERENCE (*new_instance)();
	void (*volatile dispose)(GE_context*, EIF_REFERENCE);
} GE_type_info;

typedef volatile struct {
	EIF_TYPE_INDEX volatile id; /* Type id of the "TYPE [X]" object */
	uint16_t volatile flags; 
#ifdef GE_USE_SCOOP
	GE_scoop_region* volatile region;
#endif
	EIF_INTEGER volatile type_id; /* Type id of the type "X" */
	EIF_BOOLEAN volatile is_special;
	void (*volatile dispose)(GE_context*, EIF_REFERENCE);
	EIF_REFERENCE volatile a1; /* internal_name */
	EIF_REFERENCE volatile a2; /* internal_name_32 */
} EIF_TYPE_OBJ;

/*
 * Types indexed by type id.
 * Generated by the compiler.
 */
extern EIF_TYPE_OBJ GE_types[][2];
extern GE_type_info GE_type_infos[];

/*
 * Number of type infos in `GE_type_infos'.
 * Do not take into account the fake item at index 0.
 */
extern int GE_type_info_count;

/*
 * Encode a EIF_TYPE into a EIF_ENCODED_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_ENCODED_TYPE GE_encoded_type(EIF_TYPE a_type);

/*
 * Decode a EIF_ENCODED_TYPE into a EIF_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_TYPE GE_decoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Type with `a_id' and `a_annotations'.
 */
extern EIF_TYPE GE_new_type(EIF_TYPE_INDEX a_id, EIF_TYPE_INDEX a_annotations);

/*
 * Type of object `obj'.
 */
#define GE_object_type(obj)	GE_new_type(((EIF_REFERENCE)(obj))->id, 0x0)
#define GE_object_encoded_type(obj) GE_encoded_type(GE_object_type(obj))

/*
 * Attachment status of `a_type'.
 */
#define GE_is_attached_type(a_type) EIF_TEST(((a_type).annotations & ATTACHED_FLAG) || GE_is_expanded_type_index((a_type).id))
#define GE_is_attached_encoded_type(a_type) GE_is_attached_type(GE_decoded_type(a_type))

/*
 * Associated detachable type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_non_attached_type(EIF_TYPE a_type);
#define GE_non_attached_encoded_type(a_type) GE_encoded_type(GE_non_attached_type(GE_decoded_type(a_type)))

/*
 * Associated attached type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_attached_type(EIF_TYPE a_type);
#define GE_attached_encoded_type(t) GE_encoded_type(GE_attached_type(GE_decoded_type(t)))

/*
 * Is `a_type' a SPECIAL type?
 */
#define GE_is_special_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_SPECIAL)
#define GE_is_special_encoded_type(a_type) GE_is_special_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_object(obj) GE_is_special_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of user-defined expanded type?
 */
extern EIF_BOOLEAN GE_is_special_of_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_expanded_encoded_type(a_type) GE_is_special_of_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_expanded_object(obj) GE_is_special_of_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type?
 */
extern EIF_BOOLEAN GE_is_special_of_reference_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_encoded_type(a_type) GE_is_special_of_reference_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_object(obj) GE_is_special_of_reference_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type or basic expanded type?
 * (Note that user-defined expanded types are excluded.)
 */
extern EIF_BOOLEAN GE_is_special_of_reference_or_basic_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_or_basic_expanded_encoded_type(a_type) GE_is_special_of_reference_or_basic_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_or_basic_expanded_object(obj) GE_is_special_of_reference_or_basic_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a TUPLE type?
 */
#define GE_is_tuple_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_TUPLE)
#define GE_is_tuple_encoded_type(a_type) GE_is_tuple_type_index(GE_decoded_type(a_type).id)
#define GE_is_tuple_object(obj) GE_is_tuple_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' an expanded type?
 */
#define GE_is_expanded_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_EXPANDED)
#define GE_is_expanded_encoded_type(a_type) GE_is_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_expanded_object(obj) GE_is_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a type whose base class is deferred?
 */
#define GE_is_deferred_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_DEFERRED)
#define GE_is_deferred_encoded_type(a_type) GE_is_deferred_type_index(GE_decoded_type(a_type).id)

/*
 * Does `i'-th field of `a_object + a_physical_offset' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_copy_semantics_field(EIF_INTEGER i, EIF_POINTER a_object, EIF_INTEGER a_physical_offset);

/*
 * Does `i'-th item of special `a_object' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_special_copy_semantics_item(EIF_INTEGER i, EIF_POINTER a_object);

/*
 * Generator class name of `a_type'.
 */
extern EIF_REFERENCE GE_generator_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_of_encoded_type(a_type) GE_generator_of_type_index(GE_decoded_type(a_type).id)
extern EIF_REFERENCE GE_generator_8_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_8_of_encoded_type(a_type) GE_generator_8_of_type_index(GE_decoded_type(a_type).id)

/*
 * Full name of `a_type'.
 */
extern EIF_REFERENCE GE_generating_type_of_encoded_type(EIF_ENCODED_TYPE a_type);
extern EIF_REFERENCE GE_generating_type_8_of_encoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Encoded type whose name is `a_name'.
 * -1 if no such type.
 */
extern EIF_ENCODED_TYPE GE_encoded_type_from_name(EIF_POINTER a_name);

/*
 * Does `a_type_1' conform to `a_type_2'?
 */
extern EIF_BOOLEAN GE_encoded_type_conforms_to(EIF_ENCODED_TYPE a_type_1, EIF_ENCODED_TYPE a_type_2);

/*
 * Number of generic parameters.
 */
extern EIF_INTEGER GE_generic_parameter_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generic_parameter_count_of_encoded_type(a_type) GE_generic_parameter_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Type of `i'-th generic parameter of `a_type'.
 */
extern EIF_INTEGER GE_generic_parameter_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER i);
#define GE_generic_parameter_of_encoded_type(a_type,i) GE_generic_parameter_of_type_index(GE_decoded_type(a_type).id, (i))

/*
 * Number of fields of an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_field_count_of_encoded_type(a_type) GE_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Physical offset of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_offset_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_offset_of_encoded_type(i, a_type) GE_field_offset_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Name of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_POINTER GE_field_name_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_name_of_encoded_type(i, a_type) GE_field_name_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Static type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_static_type_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_static_type_of_encoded_type(i, a_type) GE_field_static_type_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Kind of type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_type_kind_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_type_kind_of_encoded_type(i, a_type) GE_field_type_kind_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Physical size of `a_object'.
 */
extern EIF_NATURAL_64 GE_object_size(EIF_POINTER a_object);

/*
 * Is `i'-th field of objects of type `a_type' a user-defined expanded attribute?
 */
extern EIF_BOOLEAN GE_is_field_expanded_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_is_field_expanded_of_encoded_type(i, a_type) GE_is_field_expanded_of_type_index((i), GE_decoded_type(a_type).id)

#define GE_field_address_at(a_field_offset, a_object, a_physical_offset) ((char*)(a_object) + (a_physical_offset) + (a_field_offset))
#define GE_object_at_offset(a_enclosing, a_physical_offset) (EIF_REFERENCE)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_raw_object_at_offset(a_enclosing, a_physical_offset) (EIF_POINTER)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_object_encoded_type_at_offset(a_enclosing, a_physical_offset) GE_object_encoded_type(GE_raw_object_at_offset((a_enclosing), (a_physical_offset)))
#define GE_boolean_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_pointer_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_raw_reference_field_at(a_field_offset, a_object, a_physical_offset) (EIF_POINTER)*(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_reference_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_set_boolean_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_boolean_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_pointer_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_pointer_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_reference_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_reference_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value

#if defined(GE_USE_ATTRIBUTES) && defined(GE_USE_ATTRIBUTE_OFFSET)
#define GE_field_address(i, a_object, a_physical_offset) GE_field_address_at(GE_type_infos[((EIF_REFERENCE)(a_object))->id].attributes[(i) - 1]->offset, (a_object), (a_physical_offset))
#define GE_boolean_field(i, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_8_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_32_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_8_field(i, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_16_field(i, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_32_field(i, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_64_field(i, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_8_field(i, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_16_field(i, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_32_field(i, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_64_field(i, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_pointer_field(i, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_32_field(i, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_64_field(i, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_reference_field(i, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value) GE_boolean_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value) GE_character_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value) GE_character_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value) GE_integer_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value) GE_integer_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value) GE_integer_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value) GE_integer_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value) GE_natural_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value) GE_natural_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value) GE_natural_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value) GE_natural_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value) GE_pointer_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value) GE_real_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value) GE_real_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value) GE_reference_field((i), (a_object), (a_physical_offset)) = (a_value)
#else
#define GE_boolean_field(i, a_object, a_physical_offset) (EIF_BOOLEAN)0
#define GE_character_8_field(i, a_object, a_physical_offset) (EIF_CHARACTER_8)0
#define GE_character_32_field(i, a_object, a_physical_offset) (EIF_CHARACTER_32)0
#define GE_integer_8_field(i, a_object, a_physical_offset) (EIF_INTEGER_8)0
#define GE_integer_16_field(i, a_object, a_physical_offset) (EIF_INTEGER_16)0
#define GE_integer_32_field(i, a_object, a_physical_offset) (EIF_INTEGER_32)0
#define GE_integer_64_field(i, a_object, a_physical_offset) (EIF_INTEGER_64)0
#define GE_natural_8_field(i, a_object, a_physical_offset) (EIF_NATURAL_8)0
#define GE_natural_16_field(i, a_object, a_physical_offset) (EIF_NATURAL_16)0
#define GE_natural_32_field(i, a_object, a_physical_offset) (EIF_NATURAL_32)0
#define GE_natural_64_field(i, a_object, a_physical_offset) (EIF_NATURAL_64)0
#define GE_pointer_field(i, a_object, a_physical_offset) (EIF_POINTER)0
#define GE_real_32_field(i, a_object, a_physical_offset) (EIF_REAL_32)0
#define GE_real_64_field(i, a_object, a_physical_offset) (EIF_REAL_64)0
#define GE_reference_field(i, a_object, a_physical_offset) (EIF_REFERENCE)0
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value)
#endif

/*
 * Number of non-transient fields of an object of dynamic type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_persistent_field_count_of_type_index(a_type) GE_field_count_of_type_index(a_type)
#define GE_persistent_field_count_of_encoded_type(a_type) GE_persistent_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Is `i'-th field of objects of type `a_type' a transient field?
 * TODO: storable not implemented yet.
 */
#define GE_is_field_transient_of_type_index(i, a_type) EIF_FALSE
#define GE_is_field_transient_of_encoded_type(i, a_type) GE_is_field_transient_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Storable version of objects of type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_storable_version_of_type_index(a_type) EIF_VOID
#define GE_storable_version_of_encoded_type(a_type) GE_storable_version_of_type_index(GE_decoded_type(a_type).id)

/*
 * Get a lock on `GE_mark_object' and `GE_unmark_object' routines so that
 * 2 threads cannot `GE_mark_object' and `GE_unmark_object' at the same time.
 */
extern void GE_lock_marking(void);

/*
 * Release a lock on `GE_mark_object' and `GE_unmark_object', so that another
 * thread can use `GE_mark_object' and `GE_unmark_object'.
 */
extern void GE_unlock_marking(void);

/*
 * Is `obj' marked?
 */
extern EIF_BOOLEAN GE_is_object_marked(EIF_POINTER obj);

/*
 * Mark `obj'.
 */
extern void GE_mark_object(EIF_POINTER obj);

/*
 * Unmark `obj'.
 */
extern void GE_unmark_object(EIF_POINTER obj);

/*
 * New instance of dynamic `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 * `a_type' cannot represent a SPECIAL type, use
 * `GE_new_special_of_reference_instance_of_type_index' instead.
 */
extern EIF_REFERENCE GE_new_instance_of_type_index(GE_context* a_context, EIF_TYPE_INDEX a_type);
#define GE_new_instance_of_encoded_type(a_context, a_type) GE_new_instance_of_type_index((a_context), GE_decoded_type(a_type).id)

/*
 * New instance of dynamic `a_type' that represents
 * a SPECIAL with can contain `a_capacity' elements of reference type.
 * To create a SPECIAL of basic type, use class SPECIAL directly.
 */
extern EIF_REFERENCE GE_new_special_of_reference_instance_of_type_index(GE_context* a_context, EIF_TYPE_INDEX a_type, EIF_INTEGER a_capacity);
#define GE_new_special_of_reference_instance_of_encoded_type(a_context, a_type, a_capacity) GE_new_special_of_reference_instance_of_type_index((a_context), GE_decoded_type(a_type).id, (a_capacity))

/*
 * New instance of tuple of type `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 */
#define GE_new_tuple_instance_of_type_index(a_context, a_type) GE_new_instance_of_type_index((a_context), (a_type))
#define GE_new_tuple_instance_of_encoded_type(a_context, a_type) GE_new_tuple_instance_of_type_index((a_context), GE_decoded_type(a_type).id)

/*
 * New instance of TYPE for object of type `a_type'.
 */
extern EIF_REFERENCE GE_new_type_instance_of_encoded_type(GE_context* a_context, EIF_ENCODED_TYPE a_type);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to manipulate strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_STRING_H
#define GE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * New Eiffel empty string of type "STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str32(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr32(EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str8(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing
 * the first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_istr32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `n' native characters found in native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_ims32_from_nstr(EIF_NATIVE_CHAR* s, EIF_INTEGER n);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_istr32_from_nstr(EIF_NATIVE_CHAR* s);

/*
 * New Eiffel string of type "STRING" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'
 */
extern EIF_REFERENCE GE_str(const char* s);

/*
 * Base address of `o' of type "SPECIAL [CHARACTER_8].
 * The base address is the addresss of the first character in `o'.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_POINTER GE_sp8_base_address(EIF_REFERENCE o);

/*
 * Base address of `o' of type "SPECIAL [CHARACTER_832.
 * The base address is the addresss of the first character in `o'.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_POINTER GE_sp32_base_address(EIF_REFERENCE o);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class CONSOLE"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_CONSOLE_H
#define GE_CONSOLE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize mutex to determine whether a new
 * console needs to be created.
 */
#ifdef EIF_WINDOWS
extern void GE_init_console(void);
#else
#define GE_init_console()
#endif

/*
 * Create a new DOS console if needed (i.e. in case of a Windows application).
 */
#ifdef EIF_WINDOWS
extern void GE_show_console(void);
#else
#define GE_show_console()
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement the program initialization"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_MAIN_H
#define GE_MAIN_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_main(int argc, EIF_NATIVE_CHAR** argv);

/*
 * System name.
 */
extern char* GE_system_name;

/*
 * Root class name.
 */
extern char* GE_root_class_name;

#ifdef EIF_WINDOWS

#include <windows.h>

/*
 * Used in WEL.
 */
extern HINSTANCE eif_hInstance;
extern HINSTANCE eif_hPrevInstance;
extern LPWSTR eif_lpCmdLine;
extern int eif_nCmdShow;

/*
 * Main entry point when compiling a Windows application.
 * See:
 *    http://en.wikipedia.org/wiki/WinMain
 *    http://msdn2.microsoft.com/en-us/library/ms633559.aspx
 */
extern int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);

#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to access garbage collector facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_GC_H
#define GE_GC_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef GE_USE_BOEHM_GC
/*
 *	Use the Boehm garbage collector.
 *	See:
 *		http://en.wikipedia.org/wiki/Boehm_GC
 *		http://www.hpl.hp.com/personal/Hans_Boehm/gc/
 */

/* 
 * In the case of multithreaded code, gc.h should be included after the threads header file, 
 * and after defining the appropriate GC_XXXX_THREADS macro. (For 6.2alpha4 and later, 
 * simply defining GC_THREADS should suffice.) The header file gc.h must be included in files 
 * that use either GC or threads primitives, since threads primitives will be redefined to 
 * cooperate with the GC on many platforms. 
 * See: https://hboehm.info/gc/gcinterface.html
*/
#ifdef GE_USE_POSIX_THREADS
#include <pthread.h>
#include <semaphore.h>
#elif defined EIF_WINDOWS
#include <windows.h>
#include <process.h>
#endif

#define GC_IGNORE_WARN
#define GC_NOT_DLL
#define GC_THREADS
#define PARALLEL_MARK
#define THREAD_LOCAL_ALLOC
#define GC_ENABLE_SUSPEND_THREAD
#define LARGE_CONFIG
#define ALL_INTERIOR_POINTERS
#define ENABLE_DISCLAIM
#define GC_ATOMIC_UNCOLLECTABLE
#define GC_GCJ_SUPPORT
#define JAVA_FINALIZATION
#define NO_EXECUTE_PERMISSION
#define USE_MMAP
#define USE_MUNMAP

#if defined(GE_WINDOWS)
#	undef GC_NO_THREAD_DECLS
#	undef GC_NO_THREAD_REDIRECTS
#	define EMPTY_GETENV_RESULTS
#	define DONT_USE_USER32_DLL
#else
#	if !defined(GE_MACOS)
#		define GC_PTHREAD_START_STANDALONE
#	endif
#	ifndef _REENTRANT
#		define _REENTRANT
#	endif
#	define HANDLE_FORK
#endif

#if defined(__clang__) || defined(__GNUC__) || defined(__MINGW32__) || defined(__MINGW64__)
#	define GC_BUILTIN_ATOMIC
#endif

#if defined(__clang__)
#	define HAVE_DL_ITERATE_PHDR
#	define GC_REQUIRE_WCSDUP
#	define HAVE_DLADDR
#	define HAVE_SYS_TYPES_H
#	define HAVE_UNISTD_H
#	if defined(GE_MACOS)
#		define HAVE_PTHREAD_SETNAME_NP_WITHOUT_TID
#	elif !defined(GE_WINDOWS)
#		if defined(GE_OPENBSD)
#			define HAVE_PTHREAD_SET_NAME_NP
#		else
#			define HAVE_PTHREAD_SETNAME_NP_WITH_TID
#		endif
#		define HAVE_PTHREAD_SIGMASK
#		define NO_GETCONTEXT
#	endif
#endif

#include "gc.h"
#else
#include <stdlib.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * GC initialization.
 */

#if !defined(GE_USE_BOEHM_GC)
#define GE_init_gc() /* do nothing */
#elif defined(GE_WINDOWS) || (defined(GE_MACOS) && !defined(__aarch64__)) || !defined(__clang__)
#define GE_init_gc() \
	GC_INIT(); \
	GC_allow_register_threads(); \
	GC_enable_incremental()
#else
/*
 * No incremental GC under Macos arm64 and Linux when compiled wtih zig/clang,
 * because otherwise the program does not behave as expected.
 */
#define GE_init_gc() \
	GC_INIT(); \
	GC_allow_register_threads()
#endif

/*
 * Memory allocation.
 */

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc(size) GC_MALLOC(size)
#else /* No GC */
#define GE_unprotected_malloc(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_malloc(size) GE_null(GE_unprotected_malloc(size))

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic(size) GC_MALLOC_ATOMIC(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic(size) GE_unprotected_malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_malloc_atomic(size) GE_null(GE_unprotected_malloc_atomic(size))

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc(nelem, elsize) GC_MALLOC((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_calloc(nelem, elsize) GE_null(GE_unprotected_calloc((nelem), (elsize)))

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic(nelem, elsize) memset(GE_null(GC_MALLOC_ATOMIC((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc_atomic(nelem, elsize) GE_calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_uncollectable(size) GC_MALLOC_UNCOLLECTABLE(size)
#else /* No GC */
#define GE_unprotected_malloc_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_malloc_uncollectable(size) GE_null(GE_unprotected_malloc_uncollectable(size))

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic_uncollectable(size) GC_malloc_atomic_uncollectable(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_malloc_atomic_uncollectable(size) GE_null(GE_unprotected_malloc_atomic_uncollectable(size))
/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc_uncollectable(nelem, elsize) GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_calloc_uncollectable(nelem, elsize) GE_null(GE_unprotected_calloc_uncollectable((nelem), (elsize)))


/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
extern void* GE_unprotected_calloc_atomic_uncollectable(size_t nelem, size_t elsize);
#else /* No GC */
#define GE_unprotected_calloc_atomic_uncollectable(nelem, elsize) GE_unprotected_calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#define GE_calloc_atomic_uncollectable(nelem, elsize) GE_null(GE_unprotected_calloc_atomic_uncollectable((nelem), (elsize)))

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_realloc(p, size) GC_REALLOC((void*)(p), (size))
#else /* No GC */
#define GE_unprotected_realloc(p, size) realloc((void*)(p), (size))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Raise an exception when no-more-memory.
 */
#define GE_realloc(p, size) GE_null(GE_unprotected_realloc((p), (size)))

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Do not raise an exception when no-more-memory.
 */
extern void* GE_unprotected_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Raise an exception when no-more-memory.
 */
#define GE_recalloc(p, old_nelem, new_nelem, elsize) GE_null(GE_unprotected_recalloc((void*)(p), (old_nelem), (new_nelem), (elsize)))

/*
 * Explicitly deallocate an object.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_free(p) GC_FREE((void*)(p))
#else /* No GC */
#define GE_free(p) free((void*)(p))
#endif

/*
 * Dispose
 */

/*
 * Register dispose routine `disp' to be called on object `obj' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose(void* C, void* disp); /* Call dispose routine on object `C'. */
#define GE_register_dispose(obj, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(obj), (void (*) (void*, void*)) &GE_boehm_dispose, (void*)(disp), NULL, NULL)
#else /* No GC */
#define GE_register_dispose(obj, disp) /* do nothing */
#endif

/*
 * Register dispose routine `disp' to be called on once-per-object `data' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose_once_per_object_data(void* data, void* disp); /* Call dispose routine `disp' on once-per-object data `data'. */
#define GE_register_dispose_once_per_object_data(data, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(data), (void (*) (void*, void*)) &GE_boehm_dispose_once_per_object_data, (void*)(disp), NULL, NULL)
#else /* No GC */
#define GE_register_dispose_once_per_object_data(data, disp) /* do nothing */
#endif

/*
 * Access to objects, useful with GCs which move objects in memory.
 * This is not the case here, since the Boehm GC is not a moving GC.
 */

/* Access object through hector. */
#define eif_access(obj) (obj)
/* Freeze memory address. */
#define eif_freeze(obj) (obj)
/* The C side adopts an object. */
#define eif_adopt(obj) (obj)
/* The C side protects an object. */
#define eif_protect(obj) (obj)
/* The C side weans adopted object. */
extern EIF_REFERENCE eif_wean(EIF_OBJECT object);
/* Forget a frozen memory address. */
#define eif_unfreeze(obj)
/* Always frozen since they do not move. */
#define eif_frozen(obj) 1
/* Always frozen since they do not move. */
#define spfrozen(obj) 1

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to handle signals"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2024, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_SIGNAL_H
#define GE_SIGNAL_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Set signal handlers.
 * To be called at the beginning of the main thread.
 */
extern void GE_init_signal(void);

/* Initialize `GE_ignored_signals_mutex'. */
extern void GE_init_ignored_signals_mutex(void);

/* Description of sinal `a_sig'. */
extern char *GE_signal_name(EIF_INTEGER a_sig);

/* Is signal `a_sig' defined? */
extern char GE_is_signal_defined(EIF_INTEGER a_sig);

/* C signal code for signal of index `idx'. */
extern EIF_INTEGER GE_signal_map(EIF_INTEGER idx);

/*
 * Catch signal `a_sig'.
 * Check that the signal is defined.
 */
extern void GE_catch_signal(EIF_INTEGER a_sig);

/* 
 * Ignore signal `a_sig'.
 * Check that the signal is defined.
 */
extern void GE_ignore_signal(EIF_INTEGER a_sig);

/*
 * Is signal of number `a_sig' caught?
 * Check that the signal is defined.
 */
extern char GE_is_signal_caught(EIF_INTEGER a_sig);

/* Reset all the signals to their default handling. */
extern void GE_reset_all_signals(void);

/* Reset signal `a_sig' to its default handling. */
extern void GE_reset_signal_to_default(EIF_INTEGER a_sig);

/* Number of last signal. */
extern EIF_INTEGER GE_signal_number();

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class IDENTIFIED"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
*/

#ifndef GE_IDENTIFIED_H
#define GE_IDENTIFIED_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize data to keep track of object ids.
 */
extern void GE_init_identified(void);

/*
 * Get a new id for `object', assuming it is NOT in the stack.
 */
extern EIF_INTEGER_32 GE_object_id(EIF_OBJECT object);

/*
 * Return the object associated with `id'.
 */
extern EIF_REFERENCE GE_id_object(EIF_INTEGER_32 id);

/*
 * Remove the object associated with `id' from the stack.
 */
extern void GE_object_id_free(EIF_INTEGER_32 id);

#ifdef __cplusplus
}
#endif

#endif

#ifdef __cplusplus
extern "C" {
#endif

#define T0 EIF_ANY

/* CHARACTER */
#define EIF_CHARACTER EIF_CHARACTER_8

/* WIDE_CHARACTER */
#define EIF_WIDE_CHAR EIF_CHARACTER_32

/* INTEGER */
#define EIF_INTEGER EIF_INTEGER_32

/* NATURAL */
#define EIF_NATURAL EIF_NATURAL_32

/* REAL */
#define EIF_REAL EIF_REAL_32

/* DOUBLE */
#define EIF_DOUBLE EIF_REAL_64

/* BOOLEAN */
#define T1 EIF_BOOLEAN
extern T0* GE_boxed1(TC* ac, T1 a1);
extern T0* GE_boxed_pointer1(TC* ac, volatile T1* a1);
typedef volatile struct Sb1 Tb1;

/* CHARACTER_8 */
#define T2 EIF_CHARACTER_8
extern T0* GE_boxed2(TC* ac, T2 a1);
extern T0* GE_boxed_pointer2(TC* ac, volatile T2* a1);
typedef volatile struct Sb2 Tb2;

/* CHARACTER_32 */
#define T3 EIF_CHARACTER_32
extern T0* GE_boxed3(TC* ac, T3 a1);
extern T0* GE_boxed_pointer3(TC* ac, volatile T3* a1);
typedef volatile struct Sb3 Tb3;

/* INTEGER_8 */
#define T4 EIF_INTEGER_8
extern T0* GE_boxed4(TC* ac, T4 a1);
extern T0* GE_boxed_pointer4(TC* ac, volatile T4* a1);
typedef volatile struct Sb4 Tb4;

/* INTEGER_16 */
#define T5 EIF_INTEGER_16
extern T0* GE_boxed5(TC* ac, T5 a1);
extern T0* GE_boxed_pointer5(TC* ac, volatile T5* a1);
typedef volatile struct Sb5 Tb5;

/* INTEGER_32 */
#define T6 EIF_INTEGER_32
extern T0* GE_boxed6(TC* ac, T6 a1);
extern T0* GE_boxed_pointer6(TC* ac, volatile T6* a1);
typedef volatile struct Sb6 Tb6;

/* INTEGER_64 */
#define T7 EIF_INTEGER_64
extern T0* GE_boxed7(TC* ac, T7 a1);
extern T0* GE_boxed_pointer7(TC* ac, volatile T7* a1);
typedef volatile struct Sb7 Tb7;

/* NATURAL_8 */
#define T8 EIF_NATURAL_8
extern T0* GE_boxed8(TC* ac, T8 a1);
extern T0* GE_boxed_pointer8(TC* ac, volatile T8* a1);
typedef volatile struct Sb8 Tb8;

/* NATURAL_16 */
#define T9 EIF_NATURAL_16
extern T0* GE_boxed9(TC* ac, T9 a1);
extern T0* GE_boxed_pointer9(TC* ac, volatile T9* a1);
typedef volatile struct Sb9 Tb9;

/* NATURAL_32 */
#define T10 EIF_NATURAL_32
extern T0* GE_boxed10(TC* ac, T10 a1);
extern T0* GE_boxed_pointer10(TC* ac, volatile T10* a1);
typedef volatile struct Sb10 Tb10;

/* NATURAL_64 */
#define T11 EIF_NATURAL_64
extern T0* GE_boxed11(TC* ac, T11 a1);
extern T0* GE_boxed_pointer11(TC* ac, volatile T11* a1);
typedef volatile struct Sb11 Tb11;

/* REAL_32 */
#define T12 EIF_REAL_32
extern T0* GE_boxed12(TC* ac, T12 a1);
extern T0* GE_boxed_pointer12(TC* ac, volatile T12* a1);
typedef volatile struct Sb12 Tb12;

/* REAL_64 */
#define T13 EIF_REAL_64
extern T0* GE_boxed13(TC* ac, T13 a1);
extern T0* GE_boxed_pointer13(TC* ac, volatile T13* a1);
typedef volatile struct Sb13 Tb13;

/* POINTER */
#define T14 EIF_POINTER
extern T0* GE_boxed14(TC* ac, T14 a1);
extern T0* GE_boxed_pointer14(TC* ac, volatile T14* a1);
typedef volatile struct Sb14 Tb14;

/* SPECIAL [CHARACTER_8] */
typedef volatile struct S15 T15;

/* SPECIAL [CHARACTER_32] */
typedef volatile struct S16 T16;

/* STRING_8 */
typedef volatile struct S17 T17;

/* STRING_32 */
typedef volatile struct S18 T18;

/* IMMUTABLE_STRING_32 */
typedef volatile struct S20 T20;

/* ISE_EXCEPTION_MANAGER */
typedef volatile struct S21 T21;

/* GEC */
typedef volatile struct S26 T26;

/* CELL [detachable EXCEPTION] */
typedef volatile struct S27 T27;

/* HASH_TABLE [INTEGER_32, INTEGER_32] */
typedef volatile struct S28 T28;

/* CELL [detachable TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN]] */
typedef volatile struct S29 T29;

/* CELL [NO_MORE_MEMORY] */
typedef volatile struct S30 T30;

/* C_STRING */
typedef volatile struct S31 T31;

/* TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN] */
typedef volatile struct S32 T32;

/* KL_ARGUMENTS */
typedef volatile struct S33 T33;

/* ARRAY [STRING_8] */
typedef volatile struct S34 T34;

/* SPECIAL [STRING_8] */
typedef volatile struct S35 T35;

/* KL_EXCEPTIONS */
typedef volatile struct S36 T36;

/* EXCEPTIONS */
typedef volatile struct S37 T37;

/* KL_STANDARD_FILES */
typedef volatile struct S38 T38;

/* KL_STDERR_FILE */
typedef volatile struct S39 T39;

/* UTF_CONVERTER */
typedef volatile struct S40 T40;
extern T0* GE_boxed40(TC* ac, T40 a1);
extern T0* GE_boxed_pointer40(TC* ac, T40* a1);
typedef volatile struct Sb40 Tb40;

/* ET_ERROR_HANDLER */
typedef volatile struct S42 T42;

/* KL_TEXT_INPUT_FILE */
typedef volatile struct S43 T43;

/* UT_GOBO_VARIABLES */
typedef volatile struct S44 T44;

/* UT_ISE_VARIABLES */
typedef volatile struct S45 T45;

/* AP_FLAG */
typedef volatile struct S46 T46;

/* KL_EXECUTION_ENVIRONMENT */
typedef volatile struct S49 T49;

/* ET_SYSTEM */
typedef volatile struct S50 T50;

/* AP_PARSER */
typedef volatile struct S52 T52;

/* AP_ALTERNATIVE_OPTIONS_LIST */
typedef volatile struct S53 T53;

/* AP_STRING_OPTION */
typedef volatile struct S54 T54;

/* UT_VERSION */
typedef volatile struct S56 T56;

/* AP_ENUMERATION_OPTION */
typedef volatile struct S57 T57;

/* AP_BOOLEAN_OPTION */
typedef volatile struct S58 T58;

/* AP_INTEGER_OPTION */
typedef volatile struct S59 T59;

/* ET_NULL_ERROR_HANDLER */
typedef volatile struct S63 T63;

/* ET_AST_FACTORY */
typedef volatile struct S65 T65;

/* ET_SYSTEM_PROCESSOR */
typedef volatile struct S66 T66;

/* ET_CLUSTER */
typedef volatile struct S67 T67;

/* ET_EIFFEL_PARSER */
typedef volatile struct S68 T68;

/* DS_ARRAYED_LIST [ET_CLASS] */
typedef volatile struct S69 T69;

/* ET_CLASS */
typedef volatile struct S70 T70;

/* TUPLE [ET_CLASS] */
typedef volatile struct S74 T74;

/* PROCEDURE [TUPLE [ET_CLASS]] */
typedef volatile struct S75 T75;

/* TUPLE */
typedef volatile struct S76 T76;

/* TUPLE [DS_ARRAYED_LIST [ET_CLASS]] */
typedef volatile struct S77 T77;

/* PREDICATE [TUPLE [ET_CLASS]] */
typedef volatile struct S78 T78;

/* ET_CREATOR_LIST */
typedef volatile struct S80 T80;

/* ET_TOKEN_CONSTANTS */
typedef volatile struct S81 T81;

/* ET_CREATOR */
typedef volatile struct S82 T82;

/* UT_CANNOT_READ_FILE_ERROR */
typedef volatile struct S83 T83;

/* ET_ECF_SYSTEM_PARSER */
typedef volatile struct S85 T85;

/* ET_ECF_ERROR_HANDLER */
typedef volatile struct S86 T86;

/* ET_ECF_SYSTEM */
typedef volatile struct S87 T87;

/* ET_ECF_TARGET */
typedef volatile struct S88 T88;

/* ET_ECF_CLUSTER */
typedef volatile struct S89 T89;

/* ET_ECF_SETTINGS */
typedef volatile struct S90 T90;

/* ET_ECF_CAPABILITIES */
typedef volatile struct S91 T91;

/* ET_ECF_VARIABLES */
typedef volatile struct S92 T92;

/* ET_CLUSTERS */
typedef volatile struct S94 T94;

/* ET_DYNAMIC_SYSTEM */
typedef volatile struct S95 T95;

/* DT_DATE_TIME */
typedef volatile struct S97 T97;

/* DS_HASH_SET [STRING_8] */
typedef volatile struct S99 T99;

/* ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
typedef volatile struct S100 T100;

/* ET_CLASS_TYPE */
typedef volatile struct S102 T102;

/* UT_VERSION_NUMBER */
typedef volatile struct S103 T103;

/* EXECUTION_ENVIRONMENT */
typedef volatile struct S105 T105;

/* UT_MESSAGE */
typedef volatile struct S106 T106;

/* RX_PCRE_REGULAR_EXPRESSION */
typedef volatile struct S107 T107;

/* KL_STRING_ROUTINES */
typedef volatile struct S108 T108;

/* ST_SPLITTER */
typedef volatile struct S109 T109;

/* AP_ERROR */
typedef volatile struct S113 T113;

/* ET_C_GENERATOR */
typedef volatile struct S114 T114;

/* DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S116 T116;

/* KL_SHELL_COMMAND */
typedef volatile struct S117 T117;

/* GECC */
typedef volatile struct S118 T118;

/* KL_OPERATING_SYSTEM */
typedef volatile struct S119 T119;

/* VOID_TARGET */
typedef volatile struct S120 T120;

/* TYPE [VOID_TARGET] */
#define T121 EIF_TYPE_OBJ

/* ROUTINE_FAILURE */
typedef volatile struct S122 T122;

/* TYPE [ROUTINE_FAILURE] */
#define T123 EIF_TYPE_OBJ

/* OLD_VIOLATION */
typedef volatile struct S124 T124;

/* TYPE [OLD_VIOLATION] */
#define T125 EIF_TYPE_OBJ

/* NO_MORE_MEMORY */
typedef volatile struct S126 T126;

/* INVARIANT_VIOLATION */
typedef volatile struct S127 T127;

/* OPERATING_SYSTEM_SIGNAL_FAILURE */
typedef volatile struct S128 T128;

/* IO_FAILURE */
typedef volatile struct S129 T129;

/* OPERATING_SYSTEM_FAILURE */
typedef volatile struct S130 T130;

/* COM_FAILURE */
typedef volatile struct S131 T131;

/* EIFFEL_RUNTIME_PANIC */
typedef volatile struct S132 T132;

/* PRECONDITION_VIOLATION */
typedef volatile struct S134 T134;

/* POSTCONDITION_VIOLATION */
typedef volatile struct S135 T135;

/* FLOATING_POINT_FAILURE */
typedef volatile struct S136 T136;

/* CHECK_VIOLATION */
typedef volatile struct S137 T137;

/* BAD_INSPECT_VALUE */
typedef volatile struct S138 T138;

/* VARIANT_VIOLATION */
typedef volatile struct S139 T139;

/* LOOP_INVARIANT_VIOLATION */
typedef volatile struct S140 T140;

/* RESCUE_FAILURE */
typedef volatile struct S141 T141;

/* RESUMPTION_FAILURE */
typedef volatile struct S142 T142;

/* CREATE_ON_DEFERRED */
typedef volatile struct S143 T143;

/* EXTERNAL_FAILURE */
typedef volatile struct S144 T144;

/* VOID_ASSIGNED_TO_EXPANDED */
typedef volatile struct S145 T145;

/* EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
typedef volatile struct S146 T146;

/* MISMATCH_FAILURE */
typedef volatile struct S147 T147;

/* DEVELOPER_EXCEPTION */
typedef volatile struct S148 T148;

/* ADDRESS_APPLIED_TO_MELTED_FEATURE */
typedef volatile struct S149 T149;

/* SERIALIZATION_FAILURE */
typedef volatile struct S150 T150;

/* KL_WINDOWS_FILE_SYSTEM */
typedef volatile struct S151 T151;

/* KL_UNIX_FILE_SYSTEM */
typedef volatile struct S152 T152;

/* PRIMES */
typedef volatile struct S153 T153;

/* SPECIAL [INTEGER_32] */
typedef volatile struct S154 T154;

/* SPECIAL [BOOLEAN] */
typedef volatile struct S155 T155;

/* ARGUMENTS_32 */
typedef volatile struct S156 T156;

/* MUTEX */
typedef volatile struct S159 T159;

/* UT_ERROR_HANDLER */
typedef volatile struct S160 T160;

/* KL_STDOUT_FILE */
typedef volatile struct S161 T161;

/* MANAGED_POINTER */
typedef volatile struct S162 T162;

/* KL_LINKABLE [CHARACTER_8] */
typedef volatile struct S164 T164;

/* FILE_INFO */
typedef volatile struct S166 T166;

/* KL_DIRECTORY */
typedef volatile struct S167 T167;

/* UC_UTF8_ROUTINES */
typedef volatile struct S168 T168;

/* DS_ARRAYED_LIST [STRING_8] */
typedef volatile struct S170 T170;

/* TUPLE [ET_UNIVERSE] */
typedef volatile struct S171 T171;

/* PROCEDURE [TUPLE [ET_UNIVERSE]] */
typedef volatile struct S172 T172;

/* TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE [ET_CLASS], BOOLEAN]] */
typedef volatile struct S173 T173;

/* ET_ADAPTED_LIBRARIES */
typedef volatile struct S174 T174;

/* ET_ADAPTED_DOTNET_ASSEMBLIES */
typedef volatile struct S175 T175;

/* ET_IMPLICIT_TYPE_MARK */
typedef volatile struct S177 T177;

/* DS_HASH_TABLE [ET_MASTER_CLASS, ET_CLASS_NAME] */
typedef volatile struct S178 T178;

/* ET_CLASS_NAME_TESTER */
typedef volatile struct S179 T179;

/* ET_PARENT */
typedef volatile struct S181 T181;

/* ET_PARENT_LIST */
typedef volatile struct S182 T182;

/* ET_CLIENT_LIST */
typedef volatile struct S183 T183;

/* ET_TUPLE_TYPE */
typedef volatile struct S184 T184;

/* ET_ACTUAL_PARAMETER_LIST */
typedef volatile struct S186 T186;

/* ET_BUILTIN_CONVERT_FEATURE */
typedef volatile struct S187 T187;

/* DS_HASH_SET [ET_UNIVERSE] */
typedef volatile struct S188 T188;

/* ET_MASTER_CLASS */
typedef volatile struct S189 T189;

/* ET_CLIENT */
typedef volatile struct S190 T190;

/* ET_KEYWORD */
typedef volatile struct S192 T192;

/* ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
typedef volatile struct S193 T193;

/* ET_RENAME_LIST */
typedef volatile struct S194 T194;

/* ET_EXPORT_LIST */
typedef volatile struct S195 T195;

/* ET_KEYWORD_FEATURE_NAME_LIST */
typedef volatile struct S196 T196;

/* ET_NONE_GROUP */
typedef volatile struct S199 T199;

/* ET_LIKE_CURRENT */
typedef volatile struct S200 T200;

/* ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
typedef volatile struct S202 T202;

/* ET_LIBRARY */
typedef volatile struct S204 T204;

/* TUPLE [ET_LIBRARY] */
typedef volatile struct S205 T205;

/* PROCEDURE [TUPLE [ET_LIBRARY]] */
typedef volatile struct S206 T206;

/* TUPLE [DS_HASH_SET [ET_UNIVERSE]] */
typedef volatile struct S207 T207;

/* ET_DOTNET_ASSEMBLY */
typedef volatile struct S208 T208;

/* TUPLE [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S209 T209;

/* PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]] */
typedef volatile struct S210 T210;

/* ST_WORD_WRAPPER */
typedef volatile struct S211 T211;

/* AP_DISPLAY_HELP_FLAG */
typedef volatile struct S213 T213;

/* DS_ARRAYED_LIST [AP_OPTION] */
typedef volatile struct S214 T214;

/* DS_ARRAYED_LIST [AP_ALTERNATIVE_OPTIONS_LIST] */
typedef volatile struct S215 T215;

/* AP_ERROR_HANDLER */
typedef volatile struct S216 T216;

/* DS_LINKABLE [AP_OPTION] */
typedef volatile struct S217 T217;

/* DS_LINKED_LIST_CURSOR [AP_OPTION] */
typedef volatile struct S218 T218;

/* DS_ARRAYED_LIST [detachable STRING_8] */
typedef volatile struct S219 T219;

/* DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
typedef volatile struct S220 T220;

/* DS_LINKED_LIST [STRING_8] */
typedef volatile struct S221 T221;

/* KL_STRING_EQUALITY_TESTER */
typedef volatile struct S222 T222;

/* KL_EQUALITY_TESTER [STRING_8] */
typedef volatile struct S223 T223;

/* DS_LINKED_LIST [BOOLEAN] */
typedef volatile struct S226 T226;

/* DS_LINKED_LIST [INTEGER_32] */
typedef volatile struct S229 T229;

/* KL_NULL_TEXT_OUTPUT_STREAM */
typedef volatile struct S230 T230;

/* ET_EIFFEL_PREPARSER */
typedef volatile struct S232 T232;

/* ET_MASTER_CLASS_CHECKER */
typedef volatile struct S233 T233;

/* ET_PROVIDER_CHECKER */
typedef volatile struct S234 T234;

/* ET_ANCESTOR_BUILDER */
typedef volatile struct S235 T235;

/* ET_FEATURE_FLATTENER */
typedef volatile struct S236 T236;

/* ET_INTERFACE_CHECKER */
typedef volatile struct S237 T237;

/* ET_IMPLEMENTATION_CHECKER */
typedef volatile struct S238 T238;

/* ET_AST_NULL_PROCESSOR */
typedef volatile struct S239 T239;

/* DS_ARRAYED_LIST [INTEGER_32] */
typedef volatile struct S240 T240;

/* ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
typedef volatile struct S242 T242;

/* YY_UNICODE_FILE_BUFFER */
typedef volatile struct S244 T244;

/* KL_STDIN_FILE */
typedef volatile struct S245 T245;

/* DS_ARRAYED_STACK [INTEGER_32] */
typedef volatile struct S246 T246;

/* DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef volatile struct S247 T247;

/* DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
typedef volatile struct S248 T248;

/* DS_ARRAYED_STACK [detachable ET_KEYWORD] */
typedef volatile struct S249 T249;

/* DS_ARRAYED_STACK [detachable ET_SYMBOL] */
typedef volatile struct S250 T250;

/* DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
typedef volatile struct S251 T251;

/* DS_ARRAYED_STACK [ET_OBJECT_TEST_LIST] */
typedef volatile struct S252 T252;

/* DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S253 T253;

/* DS_ARRAYED_STACK [ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S254 T254;

/* DS_ARRAYED_STACK [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S255 T255;

/* DS_ARRAYED_STACK [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S256 T256;

/* DS_ARRAYED_LIST [ET_ASSERTION_ITEM] */
typedef volatile struct S257 T257;

/* DS_ARRAYED_LIST [ET_QUERY] */
typedef volatile struct S258 T258;

/* DS_ARRAYED_LIST [ET_PROCEDURE] */
typedef volatile struct S259 T259;

/* DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
typedef volatile struct S260 T260;

/* DS_HASH_SET [ET_NAMED_CLASS] */
typedef volatile struct S261 T261;

/* YY_BUFFER */
typedef volatile struct S262 T262;

/* ET_UNKNOWN_GROUP */
typedef volatile struct S263 T263;

/* KL_SPECIAL_ROUTINES [INTEGER_32] */
typedef volatile struct S264 T264;

/* KL_UNICODE_CHARACTER_BUFFER */
typedef volatile struct S266 T266;

/* ET_CLIENTS */
typedef volatile struct S268 T268;

/* ET_FEATURE_CLAUSE */
typedef volatile struct S269 T269;

/* KL_SPECIAL_ROUTINES [detachable ANY] */
typedef volatile struct S272 T272;

/* SPECIAL [detachable ANY] */
typedef volatile struct S273 T273;

/* KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
typedef volatile struct S274 T274;

/* SPECIAL [detachable ET_KEYWORD] */
typedef volatile struct S275 T275;

/* KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
typedef volatile struct S276 T276;

/* ET_AGENT_KEYWORD */
typedef volatile struct S277 T277;

/* SPECIAL [detachable ET_AGENT_KEYWORD] */
typedef volatile struct S278 T278;

/* KL_SPECIAL_ROUTINES [detachable ET_INVARIANT_KEYWORD] */
typedef volatile struct S279 T279;

/* ET_INVARIANT_KEYWORD */
typedef volatile struct S280 T280;

/* SPECIAL [detachable ET_INVARIANT_KEYWORD] */
typedef volatile struct S281 T281;

/* KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
typedef volatile struct S282 T282;

/* ET_PRECURSOR_KEYWORD */
typedef volatile struct S283 T283;

/* SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
typedef volatile struct S284 T284;

/* KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
typedef volatile struct S285 T285;

/* ET_SYMBOL */
typedef volatile struct S286 T286;

/* SPECIAL [detachable ET_SYMBOL] */
typedef volatile struct S287 T287;

/* KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
typedef volatile struct S288 T288;

/* SPECIAL [detachable ET_POSITION] */
typedef volatile struct S290 T290;

/* KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
typedef volatile struct S291 T291;

/* SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
typedef volatile struct S293 T293;

/* KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
typedef volatile struct S294 T294;

/* SPECIAL [detachable ET_BREAK] */
typedef volatile struct S296 T296;

/* KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
typedef volatile struct S297 T297;

/* SPECIAL [detachable ET_CHARACTER_CONSTANT] */
typedef volatile struct S299 T299;

/* KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
typedef volatile struct S300 T300;

/* ET_CURRENT */
typedef volatile struct S301 T301;

/* SPECIAL [detachable ET_CURRENT] */
typedef volatile struct S302 T302;

/* KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
typedef volatile struct S303 T303;

/* ET_FREE_OPERATOR */
typedef volatile struct S304 T304;

/* SPECIAL [detachable ET_FREE_OPERATOR] */
typedef volatile struct S305 T305;

/* KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
typedef volatile struct S306 T306;

/* ET_IDENTIFIER */
typedef volatile struct S307 T307;

/* SPECIAL [detachable ET_IDENTIFIER] */
typedef volatile struct S308 T308;

/* KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
typedef volatile struct S309 T309;

/* SPECIAL [detachable ET_INTEGER_CONSTANT] */
typedef volatile struct S311 T311;

/* KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
typedef volatile struct S312 T312;

/* ET_KEYWORD_OPERATOR */
typedef volatile struct S313 T313;

/* SPECIAL [detachable ET_KEYWORD_OPERATOR] */
typedef volatile struct S314 T314;

/* KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
typedef volatile struct S315 T315;

/* SPECIAL [detachable ET_MANIFEST_STRING] */
typedef volatile struct S317 T317;

/* KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
typedef volatile struct S318 T318;

/* SPECIAL [detachable ET_REAL_CONSTANT] */
typedef volatile struct S320 T320;

/* KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
typedef volatile struct S321 T321;

/* ET_RESULT */
typedef volatile struct S322 T322;

/* SPECIAL [detachable ET_RESULT] */
typedef volatile struct S323 T323;

/* KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
typedef volatile struct S324 T324;

/* ET_RETRY_INSTRUCTION */
typedef volatile struct S325 T325;

/* SPECIAL [detachable ET_RETRY_INSTRUCTION] */
typedef volatile struct S326 T326;

/* KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
typedef volatile struct S327 T327;

/* ET_SYMBOL_OPERATOR */
typedef volatile struct S328 T328;

/* SPECIAL [detachable ET_SYMBOL_OPERATOR] */
typedef volatile struct S329 T329;

/* KL_SPECIAL_ROUTINES [detachable ET_VOID] */
typedef volatile struct S330 T330;

/* ET_VOID */
typedef volatile struct S331 T331;

/* SPECIAL [detachable ET_VOID] */
typedef volatile struct S332 T332;

/* KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
typedef volatile struct S333 T333;

/* ET_SEMICOLON_SYMBOL */
typedef volatile struct S334 T334;

/* SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
typedef volatile struct S335 T335;

/* KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
typedef volatile struct S336 T336;

/* ET_BRACKET_SYMBOL */
typedef volatile struct S337 T337;

/* SPECIAL [detachable ET_BRACKET_SYMBOL] */
typedef volatile struct S338 T338;

/* KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
typedef volatile struct S339 T339;

/* ET_QUESTION_MARK_SYMBOL */
typedef volatile struct S340 T340;

/* SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
typedef volatile struct S341 T341;

/* KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
typedef volatile struct S342 T342;

/* ET_ACROSS_EXPRESSION */
typedef volatile struct S343 T343;

/* SPECIAL [detachable ET_ACROSS_EXPRESSION] */
typedef volatile struct S344 T344;

/* KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
typedef volatile struct S345 T345;

/* ET_ACROSS_INSTRUCTION */
typedef volatile struct S346 T346;

/* SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
typedef volatile struct S347 T347;

/* KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S348 T348;

/* ET_ACTUAL_ARGUMENT_LIST */
typedef volatile struct S349 T349;

/* SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S350 T350;

/* KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S351 T351;

/* SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S352 T352;

/* KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef volatile struct S353 T353;

/* SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef volatile struct S354 T354;

/* KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef volatile struct S355 T355;

/* SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef volatile struct S357 T357;

/* KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef volatile struct S358 T358;

/* SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef volatile struct S360 T360;

/* KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef volatile struct S361 T361;

/* ET_AGENT_ARGUMENT_OPERAND_LIST */
typedef volatile struct S362 T362;

/* SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef volatile struct S363 T363;

/* KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
typedef volatile struct S364 T364;

/* SPECIAL [detachable ET_AGENT_TARGET] */
typedef volatile struct S366 T366;

/* KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
typedef volatile struct S367 T367;

/* ET_ALIAS_NAME */
typedef volatile struct S368 T368;

/* SPECIAL [detachable ET_ALIAS_NAME] */
typedef volatile struct S369 T369;

/* KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
typedef volatile struct S370 T370;

/* ET_ALIAS_NAME_LIST */
typedef volatile struct S371 T371;

/* SPECIAL [detachable ET_ALIAS_NAME_LIST] */
typedef volatile struct S372 T372;

/* KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
typedef volatile struct S373 T373;

/* SPECIAL [detachable ET_ASSIGNER] */
typedef volatile struct S375 T375;

/* KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
typedef volatile struct S376 T376;

/* ET_BRACKET_EXPRESSION */
typedef volatile struct S377 T377;

/* SPECIAL [detachable ET_BRACKET_EXPRESSION] */
typedef volatile struct S378 T378;

/* KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
typedef volatile struct S379 T379;

/* ET_CALL_AGENT */
typedef volatile struct S380 T380;

/* SPECIAL [detachable ET_CALL_AGENT] */
typedef volatile struct S381 T381;

/* KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S382 T382;

/* ET_QUALIFIED_CALL_EXPRESSION */
typedef volatile struct S383 T383;

/* SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S384 T384;

/* KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
typedef volatile struct S385 T385;

/* ET_CHECK_INSTRUCTION */
typedef volatile struct S386 T386;

/* SPECIAL [detachable ET_CHECK_INSTRUCTION] */
typedef volatile struct S387 T387;

/* KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
typedef volatile struct S388 T388;

/* SPECIAL [detachable ET_CHOICE] */
typedef volatile struct S390 T390;

/* KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
typedef volatile struct S391 T391;

/* SPECIAL [detachable ET_CHOICE_CONSTANT] */
typedef volatile struct S393 T393;

/* KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
typedef volatile struct S394 T394;

/* SPECIAL [detachable ET_CHOICE_ITEM] */
typedef volatile struct S396 T396;

/* KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
typedef volatile struct S397 T397;

/* ET_CHOICE_LIST */
typedef volatile struct S398 T398;

/* SPECIAL [detachable ET_CHOICE_LIST] */
typedef volatile struct S399 T399;

/* KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
typedef volatile struct S400 T400;

/* SPECIAL [detachable ET_CLASS] */
typedef volatile struct S401 T401;

/* KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
typedef volatile struct S402 T402;

/* SPECIAL [detachable ET_CLIENT_ITEM] */
typedef volatile struct S403 T403;

/* KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
typedef volatile struct S404 T404;

/* SPECIAL [detachable ET_CLIENTS] */
typedef volatile struct S405 T405;

/* KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
typedef volatile struct S406 T406;

/* ET_COMPOUND */
typedef volatile struct S407 T407;

/* SPECIAL [detachable ET_COMPOUND] */
typedef volatile struct S408 T408;

/* KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
typedef volatile struct S409 T409;

/* SPECIAL [detachable ET_CONDITIONAL] */
typedef volatile struct S411 T411;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
typedef volatile struct S412 T412;

/* SPECIAL [detachable ET_CONSTANT] */
typedef volatile struct S414 T414;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S415 T415;

/* SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S417 T417;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef volatile struct S418 T418;

/* ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
typedef volatile struct S419 T419;

/* SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef volatile struct S420 T420;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
typedef volatile struct S421 T421;

/* ET_CONSTRAINT_CREATOR */
typedef volatile struct S422 T422;

/* SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
typedef volatile struct S423 T423;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
typedef volatile struct S424 T424;

/* ET_CONSTRAINT_RENAME_LIST */
typedef volatile struct S425 T425;

/* SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
typedef volatile struct S426 T426;

/* KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
typedef volatile struct S427 T427;

/* SPECIAL [detachable ET_CONSTRAINT_TYPE] */
typedef volatile struct S429 T429;

/* KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
typedef volatile struct S430 T430;

/* SPECIAL [detachable ET_CONVERT_FEATURE] */
typedef volatile struct S432 T432;

/* KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
typedef volatile struct S433 T433;

/* SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
typedef volatile struct S435 T435;

/* KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
typedef volatile struct S436 T436;

/* ET_CONVERT_FEATURE_LIST */
typedef volatile struct S437 T437;

/* SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
typedef volatile struct S438 T438;

/* KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
typedef volatile struct S439 T439;

/* ET_CREATE_EXPRESSION */
typedef volatile struct S440 T440;

/* SPECIAL [detachable ET_CREATE_EXPRESSION] */
typedef volatile struct S441 T441;

/* KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
typedef volatile struct S442 T442;

/* ET_CREATION_REGION */
typedef volatile struct S443 T443;

/* SPECIAL [detachable ET_CREATION_REGION] */
typedef volatile struct S444 T444;

/* KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
typedef volatile struct S445 T445;

/* SPECIAL [detachable ET_CREATOR] */
typedef volatile struct S446 T446;

/* KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
typedef volatile struct S447 T447;

/* SPECIAL [detachable ET_CREATOR_LIST] */
typedef volatile struct S448 T448;

/* KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
typedef volatile struct S449 T449;

/* ET_DEBUG_INSTRUCTION */
typedef volatile struct S450 T450;

/* SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
typedef volatile struct S451 T451;

/* KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
typedef volatile struct S452 T452;

/* ET_ELSEIF_EXPRESSION */
typedef volatile struct S453 T453;

/* SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
typedef volatile struct S454 T454;

/* KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef volatile struct S455 T455;

/* ET_ELSEIF_EXPRESSION_LIST */
typedef volatile struct S456 T456;

/* SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef volatile struct S457 T457;

/* KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
typedef volatile struct S458 T458;

/* ET_ELSEIF_PART */
typedef volatile struct S459 T459;

/* SPECIAL [detachable ET_ELSEIF_PART] */
typedef volatile struct S460 T460;

/* KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
typedef volatile struct S461 T461;

/* ET_ELSEIF_PART_LIST */
typedef volatile struct S462 T462;

/* SPECIAL [detachable ET_ELSEIF_PART_LIST] */
typedef volatile struct S463 T463;

/* KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
typedef volatile struct S464 T464;

/* SPECIAL [detachable ET_EXPORT] */
typedef volatile struct S466 T466;

/* KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
typedef volatile struct S467 T467;

/* SPECIAL [detachable ET_EXPORT_LIST] */
typedef volatile struct S468 T468;

/* KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
typedef volatile struct S469 T469;

/* SPECIAL [detachable ET_EXPRESSION] */
typedef volatile struct S471 T471;

/* KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
typedef volatile struct S472 T472;

/* SPECIAL [detachable ET_EXPRESSION_ITEM] */
typedef volatile struct S474 T474;

/* KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
typedef volatile struct S475 T475;

/* SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
typedef volatile struct S477 T477;

/* KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
typedef volatile struct S478 T478;

/* SPECIAL [detachable ET_EXTERNAL_ALIAS] */
typedef volatile struct S480 T480;

/* KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
typedef volatile struct S481 T481;

/* SPECIAL [detachable ET_FEATURE_CLAUSE] */
typedef volatile struct S482 T482;

/* KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
typedef volatile struct S483 T483;

/* ET_FEATURE_CLAUSE_LIST */
typedef volatile struct S484 T484;

/* SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
typedef volatile struct S485 T485;

/* KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
typedef volatile struct S486 T486;

/* ET_FEATURE_EXPORT */
typedef volatile struct S487 T487;

/* SPECIAL [detachable ET_FEATURE_EXPORT] */
typedef volatile struct S488 T488;

/* KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
typedef volatile struct S489 T489;

/* SPECIAL [detachable ET_FEATURE_NAME] */
typedef volatile struct S490 T490;

/* KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
typedef volatile struct S491 T491;

/* SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
typedef volatile struct S493 T493;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
typedef volatile struct S494 T494;

/* ET_FORMAL_ARGUMENT */
typedef volatile struct S495 T495;

/* SPECIAL [detachable ET_FORMAL_ARGUMENT] */
typedef volatile struct S496 T496;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef volatile struct S497 T497;

/* SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef volatile struct S499 T499;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef volatile struct S500 T500;

/* ET_FORMAL_ARGUMENT_LIST */
typedef volatile struct S501 T501;

/* SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef volatile struct S502 T502;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
typedef volatile struct S503 T503;

/* ET_FORMAL_PARAMETER */
typedef volatile struct S504 T504;

/* SPECIAL [detachable ET_FORMAL_PARAMETER] */
typedef volatile struct S505 T505;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef volatile struct S506 T506;

/* SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef volatile struct S508 T508;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
typedef volatile struct S509 T509;

/* ET_FORMAL_PARAMETER_LIST */
typedef volatile struct S510 T510;

/* SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
typedef volatile struct S511 T511;

/* KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
typedef volatile struct S512 T512;

/* ET_IF_EXPRESSION */
typedef volatile struct S513 T513;

/* SPECIAL [detachable ET_IF_EXPRESSION] */
typedef volatile struct S514 T514;

/* KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
typedef volatile struct S515 T515;

/* ET_IF_INSTRUCTION */
typedef volatile struct S516 T516;

/* SPECIAL [detachable ET_IF_INSTRUCTION] */
typedef volatile struct S517 T517;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
typedef volatile struct S518 T518;

/* SPECIAL [detachable ET_INLINE_AGENT] */
typedef volatile struct S520 T520;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S521 T521;

/* ET_INLINE_SEPARATE_ARGUMENT */
typedef volatile struct S522 T522;

/* SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S523 T523;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
typedef volatile struct S524 T524;

/* SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
typedef volatile struct S526 T526;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
typedef volatile struct S527 T527;

/* ET_INLINE_SEPARATE_ARGUMENTS */
typedef volatile struct S528 T528;

/* SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
typedef volatile struct S529 T529;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
typedef volatile struct S530 T530;

/* ET_INLINE_SEPARATE_INSTRUCTION */
typedef volatile struct S531 T531;

/* SPECIAL [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
typedef volatile struct S532 T532;

/* KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
typedef volatile struct S533 T533;

/* ET_INSPECT_EXPRESSION */
typedef volatile struct S534 T534;

/* SPECIAL [detachable ET_INSPECT_EXPRESSION] */
typedef volatile struct S535 T535;

/* KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
typedef volatile struct S536 T536;

/* ET_INSPECT_INSTRUCTION */
typedef volatile struct S537 T537;

/* SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
typedef volatile struct S538 T538;

/* KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
typedef volatile struct S539 T539;

/* SPECIAL [detachable ET_INSTRUCTION] */
typedef volatile struct S541 T541;

/* KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
typedef volatile struct S542 T542;

/* ET_INVARIANTS */
typedef volatile struct S543 T543;

/* SPECIAL [detachable ET_INVARIANTS] */
typedef volatile struct S544 T544;

/* KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef volatile struct S545 T545;

/* SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef volatile struct S546 T546;

/* KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
typedef volatile struct S547 T547;

/* SPECIAL [detachable ET_LIKE_TYPE] */
typedef volatile struct S549 T549;

/* KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
typedef volatile struct S550 T550;

/* ET_LOCAL_VARIABLE */
typedef volatile struct S551 T551;

/* SPECIAL [detachable ET_LOCAL_VARIABLE] */
typedef volatile struct S552 T552;

/* KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef volatile struct S553 T553;

/* SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef volatile struct S555 T555;

/* KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
typedef volatile struct S556 T556;

/* ET_LOCAL_VARIABLE_LIST */
typedef volatile struct S557 T557;

/* SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
typedef volatile struct S558 T558;

/* KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
typedef volatile struct S559 T559;

/* ET_LOOP_INVARIANTS */
typedef volatile struct S560 T560;

/* SPECIAL [detachable ET_LOOP_INVARIANTS] */
typedef volatile struct S561 T561;

/* KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
typedef volatile struct S562 T562;

/* ET_MANIFEST_ARRAY */
typedef volatile struct S563 T563;

/* SPECIAL [detachable ET_MANIFEST_ARRAY] */
typedef volatile struct S564 T564;

/* KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
typedef volatile struct S565 T565;

/* SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
typedef volatile struct S567 T567;

/* KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
typedef volatile struct S568 T568;

/* ET_MANIFEST_STRING_LIST */
typedef volatile struct S569 T569;

/* SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
typedef volatile struct S570 T570;

/* KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
typedef volatile struct S571 T571;

/* ET_MANIFEST_TUPLE */
typedef volatile struct S572 T572;

/* SPECIAL [detachable ET_MANIFEST_TUPLE] */
typedef volatile struct S573 T573;

/* KL_SPECIAL_ROUTINES [detachable ET_NOTE_LIST] */
typedef volatile struct S574 T574;

/* ET_NOTE_LIST */
typedef volatile struct S575 T575;

/* SPECIAL [detachable ET_NOTE_LIST] */
typedef volatile struct S576 T576;

/* KL_SPECIAL_ROUTINES [detachable ET_NOTE_ITEM] */
typedef volatile struct S577 T577;

/* SPECIAL [detachable ET_NOTE_ITEM] */
typedef volatile struct S579 T579;

/* KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM] */
typedef volatile struct S580 T580;

/* SPECIAL [detachable ET_NOTE_TERM] */
typedef volatile struct S582 T582;

/* KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_ITEM] */
typedef volatile struct S583 T583;

/* SPECIAL [detachable ET_NOTE_TERM_ITEM] */
typedef volatile struct S585 T585;

/* KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_LIST] */
typedef volatile struct S586 T586;

/* ET_NOTE_TERM_LIST */
typedef volatile struct S587 T587;

/* SPECIAL [detachable ET_NOTE_TERM_LIST] */
typedef volatile struct S588 T588;

/* KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
typedef volatile struct S589 T589;

/* SPECIAL [detachable ET_OBSOLETE] */
typedef volatile struct S591 T591;

/* KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef volatile struct S592 T592;

/* ET_PARENTHESIZED_EXPRESSION */
typedef volatile struct S593 T593;

/* SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef volatile struct S594 T594;

/* KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
typedef volatile struct S595 T595;

/* SPECIAL [detachable ET_PARENT] */
typedef volatile struct S596 T596;

/* KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
typedef volatile struct S597 T597;

/* ET_PARENT_CLAUSE_LIST */
typedef volatile struct S598 T598;

/* SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
typedef volatile struct S599 T599;

/* KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
typedef volatile struct S600 T600;

/* SPECIAL [detachable ET_PARENT_ITEM] */
typedef volatile struct S601 T601;

/* KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
typedef volatile struct S602 T602;

/* SPECIAL [detachable ET_PARENT_LIST] */
typedef volatile struct S603 T603;

/* KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
typedef volatile struct S604 T604;

/* ET_POSTCONDITIONS */
typedef volatile struct S605 T605;

/* SPECIAL [detachable ET_POSTCONDITIONS] */
typedef volatile struct S606 T606;

/* KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
typedef volatile struct S607 T607;

/* ET_PRECONDITIONS */
typedef volatile struct S608 T608;

/* SPECIAL [detachable ET_PRECONDITIONS] */
typedef volatile struct S609 T609;

/* KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
typedef volatile struct S610 T610;

/* SPECIAL [detachable ET_PROCEDURE] */
typedef volatile struct S612 T612;

/* KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef volatile struct S613 T613;

/* SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef volatile struct S615 T615;

/* KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
typedef volatile struct S616 T616;

/* ET_QUANTIFIER_EXPRESSION */
typedef volatile struct S617 T617;

/* SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
typedef volatile struct S618 T618;

/* KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
typedef volatile struct S619 T619;

/* SPECIAL [detachable ET_QUERY] */
typedef volatile struct S621 T621;

/* KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
typedef volatile struct S622 T622;

/* SPECIAL [detachable ET_RENAME_ITEM] */
typedef volatile struct S624 T624;

/* KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
typedef volatile struct S625 T625;

/* SPECIAL [detachable ET_RENAME_LIST] */
typedef volatile struct S626 T626;

/* KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
typedef volatile struct S627 T627;

/* ET_REPEAT_INSTRUCTION */
typedef volatile struct S628 T628;

/* SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
typedef volatile struct S629 T629;

/* KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
typedef volatile struct S630 T630;

/* ET_STATIC_CALL_EXPRESSION */
typedef volatile struct S631 T631;

/* SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
typedef volatile struct S632 T632;

/* KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
typedef volatile struct S633 T633;

/* ET_STRIP_EXPRESSION */
typedef volatile struct S634 T634;

/* SPECIAL [detachable ET_STRIP_EXPRESSION] */
typedef volatile struct S635 T635;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
typedef volatile struct S636 T636;

/* SPECIAL [detachable ET_TYPE] */
typedef volatile struct S637 T637;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
typedef volatile struct S638 T638;

/* SPECIAL [detachable ET_TYPE_CONSTRAINT] */
typedef volatile struct S640 T640;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
typedef volatile struct S641 T641;

/* SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
typedef volatile struct S643 T643;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
typedef volatile struct S644 T644;

/* ET_TYPE_CONSTRAINT_LIST */
typedef volatile struct S645 T645;

/* SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
typedef volatile struct S646 T646;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
typedef volatile struct S647 T647;

/* SPECIAL [detachable ET_TYPE_ITEM] */
typedef volatile struct S649 T649;

/* KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
typedef volatile struct S650 T650;

/* SPECIAL [detachable ET_TYPE_LIST] */
typedef volatile struct S652 T652;

/* KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
typedef volatile struct S653 T653;

/* ET_VARIANT */
typedef volatile struct S654 T654;

/* SPECIAL [detachable ET_VARIANT] */
typedef volatile struct S655 T655;

/* KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
typedef volatile struct S656 T656;

/* ET_WHEN_EXPRESSION */
typedef volatile struct S657 T657;

/* SPECIAL [detachable ET_WHEN_EXPRESSION] */
typedef volatile struct S658 T658;

/* KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
typedef volatile struct S659 T659;

/* ET_WHEN_EXPRESSION_LIST */
typedef volatile struct S660 T660;

/* SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
typedef volatile struct S661 T661;

/* KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
typedef volatile struct S662 T662;

/* ET_WHEN_PART */
typedef volatile struct S663 T663;

/* SPECIAL [detachable ET_WHEN_PART] */
typedef volatile struct S664 T664;

/* KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
typedef volatile struct S665 T665;

/* ET_WHEN_PART_LIST */
typedef volatile struct S666 T666;

/* SPECIAL [detachable ET_WHEN_PART_LIST] */
typedef volatile struct S667 T667;

/* KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
typedef volatile struct S668 T668;

/* SPECIAL [detachable ET_WRITABLE] */
typedef volatile struct S670 T670;

/* ET_OBJECT_TEST_LIST */
typedef volatile struct S672 T672;

/* ET_ITERATION_COMPONENT_LIST */
typedef volatile struct S673 T673;

/* ET_INLINE_SEPARATE_ARGUMENT_LIST */
typedef volatile struct S674 T674;

/* ET_C3_CHARACTER_CONSTANT */
typedef volatile struct S675 T675;

/* ET_REGULAR_MANIFEST_STRING */
typedef volatile struct S676 T676;

/* ET_SPECIAL_MANIFEST_STRING */
typedef volatile struct S677 T677;

/* ET_VERBATIM_STRING */
typedef volatile struct S678 T678;

/* ET_REGULAR_INTEGER_CONSTANT */
typedef volatile struct S679 T679;

/* ET_UNDERSCORED_INTEGER_CONSTANT */
typedef volatile struct S680 T680;

/* ET_HEXADECIMAL_INTEGER_CONSTANT */
typedef volatile struct S681 T681;

/* ET_OCTAL_INTEGER_CONSTANT */
typedef volatile struct S682 T682;

/* ET_BINARY_INTEGER_CONSTANT */
typedef volatile struct S683 T683;

/* ET_REGULAR_REAL_CONSTANT */
typedef volatile struct S684 T684;

/* ET_UNDERSCORED_REAL_CONSTANT */
typedef volatile struct S685 T685;

/* ET_COMMENT */
typedef volatile struct S686 T686;

/* ET_TRUE_CONSTANT */
typedef volatile struct S687 T687;

/* ET_FALSE_CONSTANT */
typedef volatile struct S688 T688;

/* ET_C1_CHARACTER_CONSTANT */
typedef volatile struct S689 T689;

/* ET_C2_CHARACTER_CONSTANT */
typedef volatile struct S690 T690;

/* ET_TAGGED_NOTE */
typedef volatile struct S692 T692;

/* ET_NOTE */
typedef volatile struct S693 T693;

/* ET_CUSTOM_ATTRIBUTE */
typedef volatile struct S694 T694;

/* ET_CONSTRAINED_FORMAL_PARAMETER */
typedef volatile struct S695 T695;

/* ET_TYPE_RENAME_CONSTRAINT */
typedef volatile struct S696 T696;

/* ET_CONSTRAINT_NAMED_TYPE */
typedef volatile struct S697 T697;

/* ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
typedef volatile struct S698 T698;

/* ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
typedef volatile struct S699 T699;

/* ET_RENAME */
typedef volatile struct S700 T700;

/* ET_ALL_EXPORT */
typedef volatile struct S701 T701;

/* ET_CONVERT_FUNCTION */
typedef volatile struct S703 T703;

/* ET_CONVERT_PROCEDURE */
typedef volatile struct S704 T704;

/* ET_ATTRIBUTE */
typedef volatile struct S706 T706;

/* ET_EXTENDED_ATTRIBUTE */
typedef volatile struct S707 T707;

/* ET_CONSTANT_ATTRIBUTE */
typedef volatile struct S708 T708;

/* ET_UNIQUE_ATTRIBUTE */
typedef volatile struct S709 T709;

/* ET_DO_FUNCTION */
typedef volatile struct S710 T710;

/* ET_ONCE_FUNCTION */
typedef volatile struct S711 T711;

/* ET_DEFERRED_FUNCTION */
typedef volatile struct S712 T712;

/* ET_EXTERNAL_FUNCTION */
typedef volatile struct S714 T714;

/* ET_DO_PROCEDURE */
typedef volatile struct S715 T715;

/* ET_ONCE_PROCEDURE */
typedef volatile struct S716 T716;

/* ET_DEFERRED_PROCEDURE */
typedef volatile struct S717 T717;

/* ET_EXTERNAL_PROCEDURE */
typedef volatile struct S718 T718;

/* ET_ALIASED_FEATURE_NAME */
typedef volatile struct S719 T719;

/* ET_ALIAS_FREE_NAME */
typedef volatile struct S720 T720;

/* ET_CLASS_ASSERTION */
typedef volatile struct S723 T723;

/* ET_LABELED_ACTUAL_PARAMETER */
typedef volatile struct S724 T724;

/* ET_LIKE_FEATURE */
typedef volatile struct S726 T726;

/* ET_QUALIFIED_LIKE_BRACED_TYPE */
typedef volatile struct S728 T728;

/* ET_QUALIFIED_LIKE_TYPE */
typedef volatile struct S729 T729;

/* ET_ASSIGNER_INSTRUCTION */
typedef volatile struct S730 T730;

/* ET_ASSIGNMENT */
typedef volatile struct S731 T731;

/* ET_ASSIGNMENT_ATTEMPT */
typedef volatile struct S732 T732;

/* ET_LOOP_INSTRUCTION */
typedef volatile struct S733 T733;

/* ET_BANG_INSTRUCTION */
typedef volatile struct S735 T735;

/* ET_QUALIFIED_CALL */
typedef volatile struct S736 T736;

/* ET_CREATE_INSTRUCTION */
typedef volatile struct S738 T738;

/* ET_CHOICE_RANGE */
typedef volatile struct S739 T739;

/* ET_QUALIFIED_CALL_INSTRUCTION */
typedef volatile struct S740 T740;

/* ET_PRECURSOR_INSTRUCTION */
typedef volatile struct S741 T741;

/* ET_STATIC_CALL_INSTRUCTION */
typedef volatile struct S743 T743;

/* ET_PRECURSOR_EXPRESSION */
typedef volatile struct S744 T744;

/* ET_FEATURE_ADDRESS */
typedef volatile struct S745 T745;

/* ET_CURRENT_ADDRESS */
typedef volatile struct S746 T746;

/* ET_RESULT_ADDRESS */
typedef volatile struct S747 T747;

/* ET_EXPRESSION_ADDRESS */
typedef volatile struct S748 T748;

/* ET_INFIX_EXPRESSION */
typedef volatile struct S749 T749;

/* ET_INFIX_AND_THEN_OPERATOR */
typedef volatile struct S750 T750;

/* ET_INFIX_OR_ELSE_OPERATOR */
typedef volatile struct S751 T751;

/* ET_EQUALITY_EXPRESSION */
typedef volatile struct S752 T752;

/* ET_OBJECT_EQUALITY_EXPRESSION */
typedef volatile struct S753 T753;

/* ET_MANIFEST_TYPE */
typedef volatile struct S754 T754;

/* ET_PREFIX_EXPRESSION */
typedef volatile struct S755 T755;

/* ET_OLD_EXPRESSION */
typedef volatile struct S756 T756;

/* ET_OLD_OBJECT_TEST */
typedef volatile struct S757 T757;

/* ET_OBJECT_TEST */
typedef volatile struct S758 T758;

/* ET_NAMED_OBJECT_TEST */
typedef volatile struct S759 T759;

/* ET_ONCE_MANIFEST_STRING */
typedef volatile struct S760 T760;

/* ET_ITERATION_CURSOR */
typedef volatile struct S761 T761;

/* ET_DO_FUNCTION_INLINE_AGENT */
typedef volatile struct S763 T763;

/* ET_ONCE_FUNCTION_INLINE_AGENT */
typedef volatile struct S764 T764;

/* ET_EXTERNAL_FUNCTION_INLINE_AGENT */
typedef volatile struct S765 T765;

/* ET_DO_PROCEDURE_INLINE_AGENT */
typedef volatile struct S766 T766;

/* ET_ONCE_PROCEDURE_INLINE_AGENT */
typedef volatile struct S767 T767;

/* ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
typedef volatile struct S768 T768;

/* ET_AGENT_OPEN_TARGET */
typedef volatile struct S769 T769;

/* ET_AGENT_TYPED_OPEN_ARGUMENT */
typedef volatile struct S770 T770;

/* ET_QUERY_LIST */
typedef volatile struct S773 T773;

/* ET_PROCEDURE_LIST */
typedef volatile struct S774 T774;

/* ET_TAGGED_ASSERTION */
typedef volatile struct S776 T776;

/* ET_FILE_POSITION */
typedef volatile struct S778 T778;

/* SPECIAL [ET_CLASS] */
typedef volatile struct S779 T779;

/* KL_SPECIAL_ROUTINES [ET_CLASS] */
typedef volatile struct S780 T780;

/* DS_ARRAYED_LIST_CURSOR [ET_CLASS] */
typedef volatile struct S781 T781;

/* ET_BASE_TYPE_LIST */
typedef volatile struct S782 T782;

/* ET_CLASS_CODES */
typedef volatile struct S783 T783;

/* ET_ECF_AST_FACTORY */
typedef volatile struct S784 T784;

/* DS_CELL [detachable ET_ECF_SYSTEM] */
typedef volatile struct S785 T785;

/* TUPLE [ET_ECF_TARGET] */
typedef volatile struct S786 T786;

/* PROCEDURE [TUPLE [ET_ECF_TARGET]] */
typedef volatile struct S787 T787;

/* TUPLE [ET_ECF_SYSTEM_PARSER] */
typedef volatile struct S788 T788;

/* XM_ELEMENT */
typedef volatile struct S789 T789;

/* XM_POSITION_TABLE */
typedef volatile struct S790 T790;

/* TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8] */
typedef volatile struct S791 T791;

/* PROCEDURE [TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8]] */
typedef volatile struct S792 T792;

/* TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, PROCEDURE [TUPLE [ET_ECF_TARGET]], DS_CELL [detachable ET_ECF_SYSTEM]] */
typedef volatile struct S793 T793;

/* DS_HASH_TABLE [ET_ECF_LIBRARY, STRING_8] */
typedef volatile struct S795 T795;

/* KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
typedef volatile struct S796 T796;

/* DS_HASH_TABLE [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
typedef volatile struct S797 T797;

/* XM_EIFFEL_PARSER */
typedef volatile struct S799 T799;

/* XM_TREE_CALLBACKS_PIPE */
typedef volatile struct S800 T800;

/* XM_CALLBACKS_TO_TREE_FILTER */
typedef volatile struct S803 T803;

/* ET_ECF_STATE */
typedef volatile struct S804 T804;

/* ET_ECF_SYSTEM_CONFIG */
typedef volatile struct S805 T805;

/* ET_ECF_LIBRARY */
typedef volatile struct S806 T806;

/* TUPLE [ET_ADAPTED_LIBRARY] */
typedef volatile struct S808 T808;

/* PROCEDURE [TUPLE [ET_ADAPTED_LIBRARY]] */
typedef volatile struct S809 T809;

/* XM_DOCUMENT */
typedef volatile struct S810 T810;

/* ET_COMPRESSED_POSITION */
typedef volatile struct S811 T811;

/* XM_STOP_ON_ERROR_FILTER */
typedef volatile struct S813 T813;

/* XM_ATTRIBUTE */
typedef volatile struct S814 T814;

/* DS_HASH_TABLE [ET_ECF_SYSTEM_CONFIG, STRING_8] */
typedef volatile struct S815 T815;

/* DS_HASH_TABLE [ET_ECF_TARGET, STRING_8] */
typedef volatile struct S816 T816;

/* ET_ECF_TARGET_PARENT */
typedef volatile struct S817 T817;

/* DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
typedef volatile struct S818 T818;

/* TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
typedef volatile struct S819 T819;

/* ET_ECF_TARGETS */
typedef volatile struct S820 T820;

/* DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, STRING_8] */
typedef volatile struct S821 T821;

/* DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, STRING_8] */
typedef volatile struct S822 T822;

/* DS_CELL [detachable ET_ECF_LIBRARY] */
typedef volatile struct S824 T824;

/* ET_ECF_ADAPTED_LIBRARY */
typedef volatile struct S825 T825;

/* TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, DS_CELL [detachable ET_ECF_LIBRARY]] */
typedef volatile struct S826 T826;

/* ET_ECF_DOTNET_ASSEMBLY */
typedef volatile struct S828 T828;

/* ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
typedef volatile struct S830 T830;

/* DS_HASH_TABLE [STRING_8, STRING_8] */
typedef volatile struct S831 T831;

/* ET_ECF_ERROR */
typedef volatile struct S832 T832;

/* ET_ECF_OPTIONS */
typedef volatile struct S833 T833;

/* ET_ECF_CLUSTERS */
typedef volatile struct S834 T834;

/* KL_AGENT_HASH_FUNCTION [STRING_8] */
typedef volatile struct S835 T835;

/* TUPLE [STRING_8] */
typedef volatile struct S836 T836;

/* FUNCTION [TUPLE [STRING_8], INTEGER_32] */
typedef volatile struct S837 T837;

/* TUPLE [KL_STRING_ROUTINES] */
typedef volatile struct S838 T838;

/* DS_HASH_TABLE [ET_IDENTIFIER, STRING_8] */
typedef volatile struct S840 T840;

/* DS_ARRAYED_LIST [ET_CLUSTER] */
typedef volatile struct S842 T842;

/* ET_DYNAMIC_PRIMARY_TYPE */
typedef volatile struct S843 T843;

/* ET_DYNAMIC_FEATURE_LIST */
typedef volatile struct S844 T844;

/* DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
typedef volatile struct S845 T845;

/* TUPLE [INTEGER_32] */
typedef volatile struct S846 T846;

/* ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
typedef volatile struct S847 T847;

/* TUPLE [ET_DYNAMIC_SYSTEM] */
typedef volatile struct S848 T848;

/* ET_DYNAMIC_FEATURE */
typedef volatile struct S850 T850;

/* ET_NESTED_TYPE_CONTEXT */
typedef volatile struct S852 T852;

/* ET_FORMAL_PARAMETER_TYPE */
typedef volatile struct S853 T853;

/* ARRAY [ET_TYPE] */
typedef volatile struct S854 T854;

/* SPECIAL [ET_TYPE] */
typedef volatile struct S855 T855;

/* ET_SYSTEM_MULTIPROCESSOR */
typedef volatile struct S858 T858;

/* DS_HASH_SET_CURSOR [STRING_8] */
typedef volatile struct S859 T859;

/* KL_SPECIAL_ROUTINES [STRING_8] */
typedef volatile struct S860 T860;

/* TYPE [STRING_8] */
#define T861 EIF_TYPE_OBJ

/* ET_FEATURE_CHECKER */
typedef volatile struct S862 T862;

/* ET_DYNAMIC_TYPE_SET_LIST */
typedef volatile struct S863 T863;

/* DS_HASH_SET [ET_FEATURE] */
typedef volatile struct S864 T864;

/* ET_DYNAMIC_PRECURSOR */
typedef volatile struct S865 T865;

/* ET_DYNAMIC_PRECURSOR_LIST */
typedef volatile struct S866 T866;

/* DS_HASH_TABLE [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
typedef volatile struct S867 T867;

/* ET_TYPE_CHECKER */
typedef volatile struct S869 T869;

/* DS_ARRAYED_LIST [ET_INLINE_AGENT] */
typedef volatile struct S872 T872;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_PROCEDURE]] */
typedef volatile struct S873 T873;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_QUERY]] */
typedef volatile struct S874 T874;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_FEATURE]] */
typedef volatile struct S875 T875;

/* DS_ARRAYED_LIST [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
typedef volatile struct S876 T876;

/* DS_ARRAYED_LIST [ET_NESTED_TYPE_CONTEXT] */
typedef volatile struct S877 T877;

/* DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
typedef volatile struct S878 T878;

/* ET_OBJECT_TEST_SCOPE */
typedef volatile struct S879 T879;

/* ET_OBJECT_TEST_SCOPE_BUILDER */
typedef volatile struct S880 T880;

/* DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
typedef volatile struct S881 T881;

/* ET_ITERATION_ITEM_SCOPE */
typedef volatile struct S882 T882;

/* DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S883 T883;

/* ET_INLINE_SEPARATE_ARGUMENT_SCOPE */
typedef volatile struct S884 T884;

/* ET_ATTACHMENT_SCOPE */
typedef volatile struct S885 T885;

/* ET_ATTACHMENT_SCOPE_BUILDER */
typedef volatile struct S886 T886;

/* DS_ARRAYED_LIST [ET_ATTACHMENT_SCOPE] */
typedef volatile struct S887 T887;

/* DS_HASH_TABLE [ET_ASSERTIONS, ET_FEATURE] */
typedef volatile struct S888 T888;

/* DS_ARRAYED_LIST [ET_NOTE_TERM] */
typedef volatile struct S889 T889;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
typedef volatile struct S890 T890;

/* ET_ADAPTED_BASE_CLASS_CHECKER */
typedef volatile struct S891 T891;

/* ET_DYNAMIC_QUALIFIED_QUERY_CALL */
typedef volatile struct S893 T893;

/* ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
typedef volatile struct S894 T894;

/* DS_ARRAYED_LIST_2 [ET_PRECONDITIONS, INTEGER_32] */
typedef volatile struct S896 T896;

/* DS_ARRAYED_LIST_2 [ET_POSTCONDITIONS, INTEGER_32] */
typedef volatile struct S897 T897;

/* DS_ARRAYED_LIST_2 [ET_INVARIANTS, INTEGER_32] */
typedef volatile struct S898 T898;

/* ET_DYNAMIC_TUPLE_TYPE */
typedef volatile struct S904 T904;

/* ET_FEATURE_LIST */
typedef volatile struct S907 T907;

/* NATIVE_STRING */
typedef volatile struct S909 T909;

/* HASH_TABLE [NATIVE_STRING, IMMUTABLE_STRING_32] */
typedef volatile struct S910 T910;

/* RX_BYTE_CODE */
typedef volatile struct S912 T912;

/* RX_CHARACTER_SET */
typedef volatile struct S913 T913;

/* RX_CASE_MAPPING */
typedef volatile struct S914 T914;

/* UC_UNICODE_ROUTINES */
typedef volatile struct S916 T916;

/* ARRAY [RX_CHARACTER_SET] */
typedef volatile struct S917 T917;

/* SPECIAL [RX_CHARACTER_SET] */
typedef volatile struct S918 T918;

/* UC_STRING */
typedef volatile struct S919 T919;

/* DS_LINKED_LIST_CURSOR [STRING_8] */
typedef volatile struct S920 T920;

/* DS_HASH_SET [NATURAL_32] */
typedef volatile struct S921 T921;

/* KL_STRING_OUTPUT_STREAM */
typedef volatile struct S922 T922;

/* DS_ARRAYED_LIST [ET_OLD_EXPRESSION] */
typedef volatile struct S923 T923;

/* DS_HASH_TABLE [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
typedef volatile struct S924 T924;

/* DS_ARRAYED_LIST [ET_IDENTIFIER] */
typedef volatile struct S925 T925;

/* DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S926 T926;

/* DS_ARRAYED_LIST [BOOLEAN] */
typedef volatile struct S927 T927;

/* DS_ARRAYED_LIST [ET_EQUALITY_EXPRESSION] */
typedef volatile struct S928 T928;

/* DS_ARRAYED_LIST [ET_DYNAMIC_EQUALITY_TYPES] */
typedef volatile struct S929 T929;

/* DS_ARRAYED_LIST [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
typedef volatile struct S930 T930;

/* DS_ARRAYED_LIST [ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S931 T931;

/* DS_ARRAYED_LIST [ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S932 T932;

/* DS_ARRAYED_LIST [ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S933 T933;

/* DS_ARRAYED_LIST [ET_RESULT] */
typedef volatile struct S934 T934;

/* ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
typedef volatile struct S935 T935;

/* ET_DYNAMIC_STANDALONE_TYPE_SET */
typedef volatile struct S936 T936;

/* DS_ARRAYED_LIST [ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef volatile struct S937 T937;

/* DS_ARRAYED_STACK [ET_EXPRESSION] */
typedef volatile struct S938 T938;

/* DS_ARRAYED_LIST [ET_EXPRESSION] */
typedef volatile struct S939 T939;

/* DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
typedef volatile struct S940 T940;

/* ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
typedef volatile struct S941 T941;

/* DS_HASH_SET [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S942 T942;

/* DS_HASH_TABLE [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S943 T943;

/* DS_ARRAYED_LIST_2 [ET_OBJECT_TEST, INTEGER_32] */
typedef volatile struct S944 T944;

/* DS_HASH_TABLE [INTEGER_32, INTEGER_32] */
typedef volatile struct S945 T945;

/* DS_HASH_SET [INTEGER_32] */
typedef volatile struct S946 T946;

/* DS_ARRAYED_LIST_2 [ET_AGENT, INTEGER_32] */
typedef volatile struct S947 T947;

/* ET_GENERAL_QUALIFIED_FEATURE_CALL_INSTRUCTION */
typedef volatile struct S948 T948;

/* ET_GENERAL_QUALIFIED_FEATURE_CALL_EXPRESSION */
typedef volatile struct S950 T950;

/* DS_HASH_SET [ET_DYNAMIC_TUPLE_TYPE] */
typedef volatile struct S951 T951;

/* DS_HASH_TABLE [INTEGER_32, ET_STANDALONE_CLOSURE] */
typedef volatile struct S952 T952;

/* ARRAY [INTEGER_32] */
typedef volatile struct S953 T953;

/* DS_HASH_TABLE [ET_CONSTANT, ET_FEATURE] */
typedef volatile struct S954 T954;

/* DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
typedef volatile struct S955 T955;

/* DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S956 T956;

/* DS_ARRAYED_LIST [ET_DYNAMIC_INLINED_EXPRESSION] */
typedef volatile struct S957 T957;

/* DS_ARRAYED_LIST [ET_DYNAMIC_CALL_CONTEXT] */
typedef volatile struct S958 T958;

/* DS_ARRAYED_LIST [ET_DYNAMIC_FEATURE] */
typedef volatile struct S959 T959;

/* DS_HASH_SET [ET_DYNAMIC_FEATURE] */
typedef volatile struct S960 T960;

/* DS_HASH_TABLE [BOOLEAN, STRING_8] */
typedef volatile struct S961 T961;

/* DS_STRING_HASH_TABLE */
typedef volatile struct S962 T962;

/* KL_TEXT_OUTPUT_FILE */
typedef volatile struct S963 T963;

/* DS_HASH_TABLE_CURSOR [STRING_8, STRING_8] */
typedef volatile struct S964 T964;

/* UT_UNDEFINED_ENVIRONMENT_VARIABLE_ERROR */
typedef volatile struct S965 T965;

/* UT_CANNOT_WRITE_TO_FILE_ERROR */
typedef volatile struct S966 T966;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [STRING_8]] */
typedef volatile struct S967 T967;

/* DS_QUICK_SORTER [STRING_8] */
typedef volatile struct S968 T968;

/* UC_STRING_COMPARATOR */
typedef volatile struct S969 T969;

/* DS_HASH_TABLE [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
typedef volatile struct S970 T970;

/* TUPLE [STRING_8, STRING_8] */
typedef volatile struct S973 T973;

/* DS_HASH_TOPOLOGICAL_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S974 T974;

/* ET_DYNAMIC_EQUALITY_TYPES */
typedef volatile struct S975 T975;

/* ET_DYNAMIC_OBJECT_EQUALITY_TYPES */
typedef volatile struct S977 T977;

/* ET_DYNAMIC_SPECIAL_TYPE */
typedef volatile struct S978 T978;

/* DS_QUICK_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S980 T980;

/* ET_DYNAMIC_SECONDARY_TYPE */
typedef volatile struct S981 T981;

/* ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
typedef volatile struct S982 T982;

/* DS_HASH_SET_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S988 T988;

/* ET_DYNAMIC_INLINED_EXPRESSION */
typedef volatile struct S992 T992;

/* ET_DYNAMIC_CALL_CONTEXT */
typedef volatile struct S993 T993;

/* DS_QUICK_SORTER [INTEGER_32] */
typedef volatile struct S994 T994;

/* PROCEDURE [TUPLE] */
typedef volatile struct S996 T996;

/* TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S997 T997;

/* TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S998 T998;

/* TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S999 T999;

/* TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S1000 T1000;

/* UT_INTEGER_FORMATTER */
typedef volatile struct S1001 T1001;

/* TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S1006 T1006;

/* TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1007 T1007;

/* TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S1011 T1011;

/* TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
typedef volatile struct S1016 T1016;

/* TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
typedef volatile struct S1017 T1017;

/* SPECIAL [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1018 T1018;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1019 T1019;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1020 T1020;

/* TUPLE [GECC, DS_ARRAYED_LIST [STRING_8], INTEGER_32, MUTEX] */
typedef volatile struct S1023 T1023;

/* WORKER_THREAD */
typedef volatile struct S1024 T1024;

/* PATH */
typedef volatile struct S1026 T1026;

/* STRING_TO_INTEGER_CONVERTOR */
typedef volatile struct S1027 T1027;

/* STRING_8_SEARCHER */
typedef volatile struct S1029 T1029;

/* STD_FILES */
typedef volatile struct S1031 T1031;

/* HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
typedef volatile struct S1032 T1032;

/* DT_SHARED_SYSTEM_CLOCK */
typedef volatile struct S1034 T1034;

/* DT_SYSTEM_CLOCK */
typedef volatile struct S1035 T1035;

/* DT_DATE_TIME_DURATION */
typedef volatile struct S1036 T1036;

/* DS_HASH_TABLE [INTEGER_32, STRING_8] */
typedef volatile struct S1041 T1041;

/* ET_CONSTRAINT_GENERIC_NAMED_TYPE */
typedef volatile struct S1044 T1044;

/* ET_UNQUALIFIED_CALL_EXPRESSION */
typedef volatile struct S1045 T1045;

/* ET_UNQUALIFIED_CALL_INSTRUCTION */
typedef volatile struct S1046 T1046;

/* KL_ARRAY_ROUTINES [INTEGER_32] */
typedef volatile struct S1047 T1047;

/* DS_CELL [detachable ET_ECF_CONFIG] */
typedef volatile struct S1050 T1050;

/* TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_SYSTEM, DS_CELL [detachable ET_ECF_CONFIG]] */
typedef volatile struct S1052 T1052;

/* DS_ARRAYED_LIST [ET_ECF_NOTE_ELEMENT] */
typedef volatile struct S1054 T1054;

/* XM_NAMESPACE */
typedef volatile struct S1055 T1055;

/* DS_LINKED_LIST_CURSOR [XM_ELEMENT_NODE] */
typedef volatile struct S1056 T1056;

/* ET_ECF_REDIRECTION_CONFIG */
typedef volatile struct S1058 T1058;

/* ET_DYNAMIC_PROCEDURE_TYPE */
typedef volatile struct S1059 T1059;

/* ET_DYNAMIC_FUNCTION_TYPE */
typedef volatile struct S1060 T1060;

/* ET_DYNAMIC_PUSH_TYPE_SET */
typedef volatile struct S1061 T1061;

/* KL_PLATFORM */
typedef volatile struct S1062 T1062;

/* UT_CONFIG_PARSER */
typedef volatile struct S1063 T1063;

/* KL_COMPARABLE_COMPARATOR [INTEGER_32] */
typedef volatile struct S1064 T1064;

/* CELL [INTEGER_32] */
typedef volatile struct S1067 T1067;

/* KL_PATHNAME */
typedef volatile struct S1069 T1069;

/* KL_ANY_ROUTINES */
typedef volatile struct S1070 T1070;

/* KL_WINDOWS_INPUT_FILE */
typedef volatile struct S1071 T1071;

/* KL_UNIX_INPUT_FILE */
typedef volatile struct S1072 T1072;

/* ARRAY [IMMUTABLE_STRING_32] */
typedef volatile struct S1073 T1073;

/* SPECIAL [IMMUTABLE_STRING_32] */
typedef volatile struct S1074 T1074;

/* SPECIAL [NATURAL_8] */
typedef volatile struct S1075 T1075;

/* TYPED_POINTER [SPECIAL [NATURAL_8]] */
typedef volatile struct S1076 T1076;
extern T0* GE_boxed1076(TC* ac, T1076 a1);
extern T0* GE_boxed_pointer1076(TC* ac, T1076* a1);
typedef volatile struct Sb1076 Tb1076;

/* UC_UTF8_STRING */
typedef volatile struct S1078 T1078;

/* DS_ARRAYED_LIST_CURSOR [STRING_8] */
typedef volatile struct S1079 T1079;

/* DS_ARRAYED_LIST [ET_ADAPTED_LIBRARY] */
typedef volatile struct S1080 T1080;

/* DS_ARRAYED_LIST [ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1081 T1081;

/* SPECIAL [ET_MASTER_CLASS] */
typedef volatile struct S1082 T1082;

/* KL_SPECIAL_ROUTINES [ET_MASTER_CLASS] */
typedef volatile struct S1085 T1085;

/* KL_SPECIAL_ROUTINES [ET_CLASS_NAME] */
typedef volatile struct S1086 T1086;

/* SPECIAL [ET_CLASS_NAME] */
typedef volatile struct S1087 T1087;

/* DS_HASH_TABLE_CURSOR [ET_MASTER_CLASS, ET_CLASS_NAME] */
typedef volatile struct S1088 T1088;

/* SPECIAL [ET_PARENT_ITEM] */
typedef volatile struct S1090 T1090;

/* KL_SPECIAL_ROUTINES [ET_PARENT_ITEM] */
typedef volatile struct S1091 T1091;

/* SPECIAL [ET_CLIENT_ITEM] */
typedef volatile struct S1092 T1092;

/* KL_SPECIAL_ROUTINES [ET_CLIENT_ITEM] */
typedef volatile struct S1093 T1093;

/* SPECIAL [ET_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S1094 T1094;

/* KL_SPECIAL_ROUTINES [ET_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S1096 T1096;

/* ET_BRACED_TYPE_LIST */
typedef volatile struct S1097 T1097;

/* DS_HASH_SET_CURSOR [ET_UNIVERSE] */
typedef volatile struct S1098 T1098;

/* KL_SPECIAL_ROUTINES [ET_UNIVERSE] */
typedef volatile struct S1100 T1100;

/* SPECIAL [ET_UNIVERSE] */
typedef volatile struct S1101 T1101;

/* DS_ARRAYED_LIST [ET_MASTER_CLASS] */
typedef volatile struct S1102 T1102;

/* TUPLE [ET_MASTER_CLASS] */
typedef volatile struct S1103 T1103;

/* PROCEDURE [TUPLE [ET_MASTER_CLASS]] */
typedef volatile struct S1104 T1104;

/* DS_ARRAYED_LIST_CURSOR [AP_OPTION] */
typedef volatile struct S1105 T1105;

/* SPECIAL [AP_OPTION] */
typedef volatile struct S1107 T1107;

/* KL_SPECIAL_ROUTINES [AP_OPTION] */
typedef volatile struct S1108 T1108;

/* DS_ARRAYED_LIST_CURSOR [AP_ALTERNATIVE_OPTIONS_LIST] */
typedef volatile struct S1109 T1109;

/* SPECIAL [AP_ALTERNATIVE_OPTIONS_LIST] */
typedef volatile struct S1110 T1110;

/* KL_SPECIAL_ROUTINES [AP_ALTERNATIVE_OPTIONS_LIST] */
typedef volatile struct S1111 T1111;

/* SPECIAL [detachable STRING_8] */
typedef volatile struct S1112 T1112;

/* KL_SPECIAL_ROUTINES [detachable STRING_8] */
typedef volatile struct S1113 T1113;

/* DS_LINKABLE [STRING_8] */
typedef volatile struct S1114 T1114;

/* DS_LINKABLE [BOOLEAN] */
typedef volatile struct S1115 T1115;

/* DS_LINKED_LIST_CURSOR [BOOLEAN] */
typedef volatile struct S1116 T1116;

/* DS_LINKABLE [INTEGER_32] */
typedef volatile struct S1117 T1117;

/* DS_LINKED_LIST_CURSOR [INTEGER_32] */
typedef volatile struct S1118 T1118;

/* YY_FILE_BUFFER */
typedef volatile struct S1120 T1120;

/* DS_HASH_TOPOLOGICAL_SORTER [ET_CLASS] */
typedef volatile struct S1121 T1121;

/* DS_HASH_TABLE [ET_BASE_TYPE, ET_CLASS] */
typedef volatile struct S1122 T1122;

/* ET_PARENT_CHECKER1 */
typedef volatile struct S1123 T1123;

/* ET_FORMAL_PARAMETER_CHECKER1 */
typedef volatile struct S1124 T1124;

/* DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
typedef volatile struct S1125 T1125;

/* ET_FEATURE_NAME_TESTER */
typedef volatile struct S1126 T1126;

/* DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
typedef volatile struct S1128 T1128;

/* ET_ALIAS_NAME_TESTER */
typedef volatile struct S1129 T1129;

/* DS_ARRAYED_LIST [ET_CLIENT_LIST] */
typedef volatile struct S1131 T1131;

/* DS_HASH_TABLE [ET_CLIENT, ET_CLASS] */
typedef volatile struct S1132 T1132;

/* ET_FEATURE_ADAPTATION_RESOLVER */
typedef volatile struct S1133 T1133;

/* ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
typedef volatile struct S1134 T1134;

/* ET_IDENTIFIER_TYPE_RESOLVER */
typedef volatile struct S1135 T1135;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
typedef volatile struct S1136 T1136;

/* ET_ANCHORED_TYPE_CHECKER */
typedef volatile struct S1137 T1137;

/* ET_SIGNATURE_CHECKER */
typedef volatile struct S1138 T1138;

/* ET_PARENT_CHECKER2 */
typedef volatile struct S1139 T1139;

/* ET_FORMAL_PARAMETER_CHECKER2 */
typedef volatile struct S1140 T1140;

/* ET_BUILTIN_FEATURE_CHECKER */
typedef volatile struct S1141 T1141;

/* ET_PRECURSOR_CHECKER */
typedef volatile struct S1142 T1142;

/* DS_HASH_TABLE [ET_FEATURE, INTEGER_32] */
typedef volatile struct S1143 T1143;

/* DS_HASH_SET [ET_CLASS] */
typedef volatile struct S1144 T1144;

/* ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
typedef volatile struct S1145 T1145;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
typedef volatile struct S1146 T1146;

/* DS_HASH_TABLE [ET_RENAME, ET_FEATURE_NAME] */
typedef volatile struct S1147 T1147;

/* DS_HASH_TABLE [ET_RENAME, ET_ALIAS_NAME] */
typedef volatile struct S1148 T1148;

/* ET_PARENT_CHECKER3 */
typedef volatile struct S1149 T1149;

/* DS_HASH_SET [ET_PROCEDURE] */
typedef volatile struct S1150 T1150;

/* DS_HASH_SET [ET_QUERY] */
typedef volatile struct S1151 T1151;

/* ET_SUPPLIER_BUILDER */
typedef volatile struct S1152 T1152;

/* DS_ARRAYED_LIST_CURSOR [INTEGER_32] */
typedef volatile struct S1153 T1153;

/* YY_UNICODE_BUFFER */
typedef volatile struct S1155 T1155;

/* SPECIAL [detachable ET_OBJECT_TEST_LIST] */
typedef volatile struct S1156 T1156;

/* KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
typedef volatile struct S1157 T1157;

/* SPECIAL [ET_OBJECT_TEST_LIST] */
typedef volatile struct S1158 T1158;

/* KL_SPECIAL_ROUTINES [ET_OBJECT_TEST_LIST] */
typedef volatile struct S1159 T1159;

/* SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S1160 T1160;

/* KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S1161 T1161;

/* SPECIAL [ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S1162 T1162;

/* KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT_LIST] */
typedef volatile struct S1163 T1163;

/* SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S1164 T1164;

/* KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S1165 T1165;

/* SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S1166 T1166;

/* KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
typedef volatile struct S1167 T1167;

/* SPECIAL [ET_ASSERTION_ITEM] */
typedef volatile struct S1168 T1168;

/* KL_SPECIAL_ROUTINES [ET_ASSERTION_ITEM] */
typedef volatile struct S1169 T1169;

/* DS_ARRAYED_LIST_CURSOR [ET_ASSERTION_ITEM] */
typedef volatile struct S1170 T1170;

/* SPECIAL [ET_QUERY] */
typedef volatile struct S1171 T1171;

/* DS_ARRAYED_LIST_CURSOR [ET_QUERY] */
typedef volatile struct S1172 T1172;

/* KL_SPECIAL_ROUTINES [ET_QUERY] */
typedef volatile struct S1173 T1173;

/* SPECIAL [ET_PROCEDURE] */
typedef volatile struct S1174 T1174;

/* DS_ARRAYED_LIST_CURSOR [ET_PROCEDURE] */
typedef volatile struct S1175 T1175;

/* KL_SPECIAL_ROUTINES [ET_PROCEDURE] */
typedef volatile struct S1176 T1176;

/* DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
typedef volatile struct S1177 T1177;

/* DS_HASH_SET_CURSOR [ET_NAMED_CLASS] */
typedef volatile struct S1178 T1178;

/* SPECIAL [ET_NAMED_CLASS] */
typedef volatile struct S1179 T1179;

/* KL_SPECIAL_ROUTINES [ET_NAMED_CLASS] */
typedef volatile struct S1180 T1180;

/* SPECIAL [ET_EXPRESSION_ITEM] */
typedef volatile struct S1182 T1182;

/* KL_SPECIAL_ROUTINES [ET_EXPRESSION_ITEM] */
typedef volatile struct S1184 T1184;

/* SPECIAL [ET_INSTRUCTION] */
typedef volatile struct S1186 T1186;

/* KL_SPECIAL_ROUTINES [ET_INSTRUCTION] */
typedef volatile struct S1187 T1187;

/* SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
typedef volatile struct S1189 T1189;

/* KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
typedef volatile struct S1190 T1190;

/* SPECIAL [ET_NAMED_OBJECT_TEST] */
typedef volatile struct S1191 T1191;

/* KL_SPECIAL_ROUTINES [ET_NAMED_OBJECT_TEST] */
typedef volatile struct S1192 T1192;

/* SPECIAL [ET_ITERATION_COMPONENT] */
typedef volatile struct S1193 T1193;

/* KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT] */
typedef volatile struct S1194 T1194;

/* SPECIAL [ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S1195 T1195;

/* KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S1196 T1196;

/* SPECIAL [ET_BASE_TYPE] */
typedef volatile struct S1197 T1197;

/* KL_SPECIAL_ROUTINES [ET_BASE_TYPE] */
typedef volatile struct S1198 T1198;

/* DS_HASH_TABLE [NATURAL_8, ET_CLASS_NAME] */
typedef volatile struct S1199 T1199;

/* DS_HASH_TABLE_CURSOR [ET_ECF_LIBRARY, STRING_8] */
typedef volatile struct S1200 T1200;

/* SPECIAL [ET_ECF_LIBRARY] */
typedef volatile struct S1201 T1201;

/* KL_SPECIAL_ROUTINES [ET_ECF_LIBRARY] */
typedef volatile struct S1204 T1204;

/* SPECIAL [ET_ECF_DOTNET_ASSEMBLY] */
typedef volatile struct S1205 T1205;

/* DS_HASH_TABLE_CURSOR [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
typedef volatile struct S1208 T1208;

/* KL_SPECIAL_ROUTINES [ET_ECF_DOTNET_ASSEMBLY] */
typedef volatile struct S1209 T1209;

/* XM_EIFFEL_SCANNER */
typedef volatile struct S1210 T1210;

/* XM_DEFAULT_POSITION */
typedef volatile struct S1211 T1211;

/* DS_BILINKED_LIST [XM_POSITION] */
typedef volatile struct S1213 T1213;

/* DS_LINKED_STACK [XM_EIFFEL_SCANNER] */
typedef volatile struct S1214 T1214;

/* XM_CALLBACKS_NULL */
typedef volatile struct S1215 T1215;

/* DS_HASH_TABLE [XM_EIFFEL_ENTITY_DEF, STRING_8] */
typedef volatile struct S1216 T1216;

/* XM_NULL_EXTERNAL_RESOLVER */
typedef volatile struct S1218 T1218;

/* XM_DTD_CALLBACKS_NULL */
typedef volatile struct S1220 T1220;

/* KL_SPECIAL_ROUTINES [XM_EIFFEL_PARSER_NAME] */
typedef volatile struct S1221 T1221;

/* XM_EIFFEL_PARSER_NAME */
typedef volatile struct S1222 T1222;

/* SPECIAL [XM_EIFFEL_PARSER_NAME] */
typedef volatile struct S1223 T1223;

/* KL_SPECIAL_ROUTINES [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
typedef volatile struct S1224 T1224;

/* DS_HASH_SET [XM_EIFFEL_PARSER_NAME] */
typedef volatile struct S1225 T1225;

/* SPECIAL [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
typedef volatile struct S1226 T1226;

/* KL_SPECIAL_ROUTINES [XM_DTD_EXTERNAL_ID] */
typedef volatile struct S1227 T1227;

/* XM_DTD_EXTERNAL_ID */
typedef volatile struct S1228 T1228;

/* SPECIAL [XM_DTD_EXTERNAL_ID] */
typedef volatile struct S1229 T1229;

/* KL_SPECIAL_ROUTINES [XM_DTD_ELEMENT_CONTENT] */
typedef volatile struct S1230 T1230;

/* XM_DTD_ELEMENT_CONTENT */
typedef volatile struct S1231 T1231;

/* SPECIAL [XM_DTD_ELEMENT_CONTENT] */
typedef volatile struct S1232 T1232;

/* KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
typedef volatile struct S1233 T1233;

/* DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT] */
typedef volatile struct S1234 T1234;

/* SPECIAL [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
typedef volatile struct S1235 T1235;

/* KL_SPECIAL_ROUTINES [XM_DTD_ATTRIBUTE_CONTENT] */
typedef volatile struct S1236 T1236;

/* XM_DTD_ATTRIBUTE_CONTENT */
typedef volatile struct S1237 T1237;

/* SPECIAL [XM_DTD_ATTRIBUTE_CONTENT] */
typedef volatile struct S1238 T1238;

/* KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [STRING_8]] */
typedef volatile struct S1239 T1239;

/* DS_BILINKED_LIST [STRING_8] */
typedef volatile struct S1240 T1240;

/* SPECIAL [DS_BILINKED_LIST [STRING_8]] */
typedef volatile struct S1241 T1241;

/* KL_SPECIAL_ROUTINES [BOOLEAN] */
typedef volatile struct S1242 T1242;

/* KL_SPECIAL_ROUTINES [XM_EIFFEL_DECLARATION] */
typedef volatile struct S1243 T1243;

/* XM_EIFFEL_DECLARATION */
typedef volatile struct S1244 T1244;

/* SPECIAL [XM_EIFFEL_DECLARATION] */
typedef volatile struct S1245 T1245;

/* XM_EIFFEL_ENTITY_DEF */
typedef volatile struct S1248 T1248;

/* XM_EIFFEL_SCANNER_DTD */
typedef volatile struct S1249 T1249;

/* XM_EIFFEL_PE_ENTITY_DEF */
typedef volatile struct S1251 T1251;

/* XM_NAMESPACE_RESOLVER */
typedef volatile struct S1252 T1252;

/* ARRAY [XM_CALLBACKS_FILTER] */
typedef volatile struct S1253 T1253;

/* SPECIAL [XM_CALLBACKS_FILTER] */
typedef volatile struct S1254 T1254;

/* DS_HASH_SET [XM_NAMESPACE] */
typedef volatile struct S1255 T1255;

/* XM_LINKED_LIST [XM_DOCUMENT_NODE] */
typedef volatile struct S1256 T1256;

/* SPECIAL [ET_ECF_SYSTEM_CONFIG] */
typedef volatile struct S1259 T1259;

/* KL_SPECIAL_ROUTINES [ET_ECF_SYSTEM_CONFIG] */
typedef volatile struct S1261 T1261;

/* DS_HASH_TABLE_CURSOR [ET_ECF_SYSTEM_CONFIG, STRING_8] */
typedef volatile struct S1262 T1262;

/* DS_HASH_TABLE_CURSOR [ET_ECF_TARGET, STRING_8] */
typedef volatile struct S1263 T1263;

/* SPECIAL [ET_ECF_TARGET] */
typedef volatile struct S1264 T1264;

/* KL_SPECIAL_ROUTINES [ET_ECF_TARGET] */
typedef volatile struct S1266 T1266;

/* DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, STRING_8] */
typedef volatile struct S1268 T1268;

/* KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
typedef volatile struct S1269 T1269;

/* SPECIAL [detachable RX_REGULAR_EXPRESSION] */
typedef volatile struct S1270 T1270;

/* ET_DOTNET_ASSEMBLIES */
typedef volatile struct S1271 T1271;

/* KL_SPECIAL_ROUTINES [ET_IDENTIFIER] */
typedef volatile struct S1275 T1275;

/* SPECIAL [ET_IDENTIFIER] */
typedef volatile struct S1276 T1276;

/* DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, STRING_8] */
typedef volatile struct S1277 T1277;

/* KL_SPECIAL_ROUTINES [ET_CLUSTER] */
typedef volatile struct S1278 T1278;

/* SPECIAL [ET_CLUSTER] */
typedef volatile struct S1279 T1279;

/* DS_ARRAYED_LIST_CURSOR [ET_CLUSTER] */
typedef volatile struct S1280 T1280;

/* DS_HASH_TABLE [ET_DYNAMIC_FEATURE, INTEGER_32] */
typedef volatile struct S1281 T1281;

/* ET_FEATURE_IDS */
typedef volatile struct S1282 T1282;

/* SPECIAL [ET_DYNAMIC_FEATURE] */
typedef volatile struct S1283 T1283;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_FEATURE] */
typedef volatile struct S1284 T1284;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
typedef volatile struct S1286 T1286;

/* DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
typedef volatile struct S1287 T1287;

/* DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
typedef volatile struct S1288 T1288;

/* ET_LIKE_N */
typedef volatile struct S1290 T1290;

/* KL_SPECIAL_ROUTINES [ET_TYPE] */
typedef volatile struct S1291 T1291;

/* DS_ARRAYED_LIST [ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1292 T1292;

/* TUPLE [ET_SYSTEM_PROCESSOR, DS_ARRAYED_LIST [ET_CLASS]] */
typedef volatile struct S1293 T1293;

/* SPECIAL [ET_DYNAMIC_TYPE_SET] */
typedef volatile struct S1294 T1294;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE_SET] */
typedef volatile struct S1295 T1295;

/* DS_HASH_SET_CURSOR [ET_FEATURE] */
typedef volatile struct S1296 T1296;

/* SPECIAL [ET_FEATURE] */
typedef volatile struct S1297 T1297;

/* KL_SPECIAL_ROUTINES [ET_FEATURE] */
typedef volatile struct S1298 T1298;

/* DS_ARRAYED_LIST [ET_ADAPTED_CLASS] */
typedef volatile struct S1300 T1300;

/* KL_SPECIAL_ROUTINES [ET_INLINE_AGENT] */
typedef volatile struct S1301 T1301;

/* SPECIAL [ET_INLINE_AGENT] */
typedef volatile struct S1302 T1302;

/* DS_ARRAYED_LIST_CURSOR [ET_INLINE_AGENT] */
typedef volatile struct S1303 T1303;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_PROCEDURE]] */
typedef volatile struct S1304 T1304;

/* SPECIAL [DS_ARRAYED_LIST [ET_PROCEDURE]] */
typedef volatile struct S1305 T1305;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_PROCEDURE]] */
typedef volatile struct S1306 T1306;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_QUERY]] */
typedef volatile struct S1307 T1307;

/* SPECIAL [DS_ARRAYED_LIST [ET_QUERY]] */
typedef volatile struct S1308 T1308;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_QUERY]] */
typedef volatile struct S1309 T1309;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_FEATURE]] */
typedef volatile struct S1310 T1310;

/* DS_ARRAYED_LIST [ET_FEATURE] */
typedef volatile struct S1311 T1311;

/* SPECIAL [DS_ARRAYED_LIST [ET_FEATURE]] */
typedef volatile struct S1312 T1312;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_FEATURE]] */
typedef volatile struct S1313 T1313;

/* KL_SPECIAL_ROUTINES [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
typedef volatile struct S1314 T1314;

/* TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT] */
typedef volatile struct S1315 T1315;

/* SPECIAL [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
typedef volatile struct S1316 T1316;

/* DS_ARRAYED_LIST_CURSOR [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
typedef volatile struct S1317 T1317;

/* SPECIAL [ET_NESTED_TYPE_CONTEXT] */
typedef volatile struct S1318 T1318;

/* KL_SPECIAL_ROUTINES [ET_NESTED_TYPE_CONTEXT] */
typedef volatile struct S1319 T1319;

/* DS_ARRAYED_LIST_CURSOR [ET_NESTED_TYPE_CONTEXT] */
typedef volatile struct S1320 T1320;

/* DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
typedef volatile struct S1321 T1321;

/* DS_ARRAYED_LIST [ET_NAMED_OBJECT_TEST] */
typedef volatile struct S1324 T1324;

/* DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
typedef volatile struct S1325 T1325;

/* DS_ARRAYED_LIST [ET_ITERATION_COMPONENT] */
typedef volatile struct S1327 T1327;

/* DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S1329 T1329;

/* DS_ARRAYED_LIST [ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S1330 T1330;

/* SPECIAL [ET_ATTACHMENT_SCOPE] */
typedef volatile struct S1332 T1332;

/* KL_SPECIAL_ROUTINES [ET_ATTACHMENT_SCOPE] */
typedef volatile struct S1333 T1333;

/* DS_ARRAYED_LIST_CURSOR [ET_ATTACHMENT_SCOPE] */
typedef volatile struct S1334 T1334;

/* DS_HASH_TABLE_CURSOR [ET_ASSERTIONS, ET_FEATURE] */
typedef volatile struct S1335 T1335;

/* SPECIAL [ET_ASSERTIONS] */
typedef volatile struct S1336 T1336;

/* KL_SPECIAL_ROUTINES [ET_ASSERTIONS] */
typedef volatile struct S1338 T1338;

/* KL_SPECIAL_ROUTINES [ET_NOTE_TERM] */
typedef volatile struct S1339 T1339;

/* SPECIAL [ET_NOTE_TERM] */
typedef volatile struct S1340 T1340;

/* DS_ARRAYED_LIST_CURSOR [ET_NOTE_TERM] */
typedef volatile struct S1341 T1341;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
typedef volatile struct S1342 T1342;

/* SPECIAL [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
typedef volatile struct S1343 T1343;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
typedef volatile struct S1344 T1344;

/* KL_SPECIAL_ROUTINES [ET_PRECONDITIONS] */
typedef volatile struct S1345 T1345;

/* SPECIAL [ET_PRECONDITIONS] */
typedef volatile struct S1346 T1346;

/* KL_SPECIAL_ROUTINES [ET_POSTCONDITIONS] */
typedef volatile struct S1347 T1347;

/* SPECIAL [ET_POSTCONDITIONS] */
typedef volatile struct S1348 T1348;

/* KL_SPECIAL_ROUTINES [ET_INVARIANTS] */
typedef volatile struct S1349 T1349;

/* SPECIAL [ET_INVARIANTS] */
typedef volatile struct S1350 T1350;

/* SPECIAL [NATIVE_STRING] */
typedef volatile struct S1351 T1351;

/* SPECIAL [NATURAL_32] */
typedef volatile struct S1352 T1352;

/* DS_ARRAYED_LIST [RX_CHARACTER_SET] */
typedef volatile struct S1353 T1353;

/* KL_SPECIAL_ROUTINES [NATURAL_32] */
typedef volatile struct S1354 T1354;

/* SPECIAL [NATURAL_64] */
typedef volatile struct S1355 T1355;

/* DS_HASH_TABLE [NATURAL_64, NATURAL_32] */
typedef volatile struct S1356 T1356;

/* DS_HASH_TABLE_CURSOR [NATURAL_64, NATURAL_32] */
typedef volatile struct S1357 T1357;

/* SPECIAL [ARRAY [INTEGER_32]] */
typedef volatile struct S1358 T1358;

/* SPECIAL [SPECIAL [ARRAY [INTEGER_32]]] */
typedef volatile struct S1359 T1359;

/* KL_INTEGER_ROUTINES */
typedef volatile struct S1360 T1360;

/* DS_HASH_SET_CURSOR [NATURAL_32] */
typedef volatile struct S1363 T1363;

/* KL_SPECIAL_ROUTINES [ET_OLD_EXPRESSION] */
typedef volatile struct S1365 T1365;

/* SPECIAL [ET_OLD_EXPRESSION] */
typedef volatile struct S1366 T1366;

/* DS_ARRAYED_LIST_CURSOR [ET_OLD_EXPRESSION] */
typedef volatile struct S1367 T1367;

/* DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
typedef volatile struct S1369 T1369;

/* DS_ARRAYED_LIST_CURSOR [ET_IDENTIFIER] */
typedef volatile struct S1370 T1370;

/* SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1371 T1371;

/* KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1372 T1372;

/* DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1373 T1373;

/* DS_ARRAYED_LIST_CURSOR [BOOLEAN] */
typedef volatile struct S1374 T1374;

/* KL_SPECIAL_ROUTINES [ET_EQUALITY_EXPRESSION] */
typedef volatile struct S1375 T1375;

/* SPECIAL [ET_EQUALITY_EXPRESSION] */
typedef volatile struct S1376 T1376;

/* DS_ARRAYED_LIST_CURSOR [ET_EQUALITY_EXPRESSION] */
typedef volatile struct S1377 T1377;

/* SPECIAL [ET_DYNAMIC_EQUALITY_TYPES] */
typedef volatile struct S1378 T1378;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_EQUALITY_TYPES] */
typedef volatile struct S1379 T1379;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_EQUALITY_TYPES] */
typedef volatile struct S1380 T1380;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
typedef volatile struct S1381 T1381;

/* SPECIAL [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
typedef volatile struct S1382 T1382;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
typedef volatile struct S1383 T1383;

/* KL_SPECIAL_ROUTINES [ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1384 T1384;

/* SPECIAL [ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1385 T1385;

/* DS_ARRAYED_LIST_CURSOR [ET_UNQUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1386 T1386;

/* SPECIAL [ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1387 T1387;

/* KL_SPECIAL_ROUTINES [ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1388 T1388;

/* DS_ARRAYED_LIST_CURSOR [ET_QUALIFIED_CALL_EXPRESSION] */
typedef volatile struct S1389 T1389;

/* KL_SPECIAL_ROUTINES [ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S1390 T1390;

/* SPECIAL [ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S1391 T1391;

/* DS_ARRAYED_LIST_CURSOR [ET_ACTUAL_ARGUMENT_LIST] */
typedef volatile struct S1392 T1392;

/* KL_SPECIAL_ROUTINES [ET_RESULT] */
typedef volatile struct S1393 T1393;

/* SPECIAL [ET_RESULT] */
typedef volatile struct S1394 T1394;

/* DS_ARRAYED_LIST_CURSOR [ET_RESULT] */
typedef volatile struct S1395 T1395;

/* SPECIAL [ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef volatile struct S1396 T1396;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef volatile struct S1397 T1397;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef volatile struct S1398 T1398;

/* SPECIAL [ET_EXPRESSION] */
typedef volatile struct S1399 T1399;

/* KL_SPECIAL_ROUTINES [ET_EXPRESSION] */
typedef volatile struct S1400 T1400;

/* DS_ARRAYED_LIST_CURSOR [ET_EXPRESSION] */
typedef volatile struct S1401 T1401;

/* KL_EQUALITY_TESTER [INTEGER_32] */
typedef volatile struct S1402 T1402;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
typedef volatile struct S1403 T1403;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1404 T1404;

/* SPECIAL [ET_OBJECT_TEST] */
typedef volatile struct S1406 T1406;

/* KL_SPECIAL_ROUTINES [ET_OBJECT_TEST] */
typedef volatile struct S1407 T1407;

/* DS_HASH_TABLE_CURSOR [INTEGER_32, INTEGER_32] */
typedef volatile struct S1409 T1409;

/* DS_HASH_SET_CURSOR [INTEGER_32] */
typedef volatile struct S1410 T1410;

/* TYPE [INTEGER_32] */
#define T1411 EIF_TYPE_OBJ

/* SPECIAL [ET_AGENT] */
typedef volatile struct S1413 T1413;

/* KL_SPECIAL_ROUTINES [ET_AGENT] */
typedef volatile struct S1414 T1414;

/* DS_HASH_SET_CURSOR [ET_DYNAMIC_TUPLE_TYPE] */
typedef volatile struct S1415 T1415;

/* SPECIAL [ET_DYNAMIC_TUPLE_TYPE] */
typedef volatile struct S1416 T1416;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_TUPLE_TYPE] */
typedef volatile struct S1417 T1417;

/* DS_HASH_TABLE_CURSOR [INTEGER_32, ET_STANDALONE_CLOSURE] */
typedef volatile struct S1419 T1419;

/* SPECIAL [ET_STANDALONE_CLOSURE] */
typedef volatile struct S1420 T1420;

/* KL_SPECIAL_ROUTINES [ET_STANDALONE_CLOSURE] */
typedef volatile struct S1421 T1421;

/* DS_HASH_TABLE_CURSOR [ET_CONSTANT, ET_FEATURE] */
typedef volatile struct S1422 T1422;

/* SPECIAL [ET_CONSTANT] */
typedef volatile struct S1423 T1423;

/* KL_SPECIAL_ROUTINES [ET_CONSTANT] */
typedef volatile struct S1425 T1425;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
typedef volatile struct S1426 T1426;

/* SPECIAL [ET_INLINE_CONSTANT] */
typedef volatile struct S1427 T1427;

/* KL_SPECIAL_ROUTINES [ET_INLINE_CONSTANT] */
typedef volatile struct S1429 T1429;

/* SPECIAL [detachable ET_DYNAMIC_FEATURE] */
typedef volatile struct S1430 T1430;

/* DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1432 T1432;

/* KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
typedef volatile struct S1433 T1433;

/* SPECIAL [ET_DYNAMIC_INLINED_EXPRESSION] */
typedef volatile struct S1434 T1434;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_INLINED_EXPRESSION] */
typedef volatile struct S1435 T1435;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_INLINED_EXPRESSION] */
typedef volatile struct S1436 T1436;

/* SPECIAL [ET_DYNAMIC_CALL_CONTEXT] */
typedef volatile struct S1437 T1437;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_CALL_CONTEXT] */
typedef volatile struct S1438 T1438;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_CALL_CONTEXT] */
typedef volatile struct S1439 T1439;

/* DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_FEATURE] */
typedef volatile struct S1440 T1440;

/* DS_HASH_SET_CURSOR [ET_DYNAMIC_FEATURE] */
typedef volatile struct S1441 T1441;

/* TYPE [ET_DYNAMIC_FEATURE] */
#define T1443 EIF_TYPE_OBJ

/* DS_HASH_TABLE_CURSOR [BOOLEAN, STRING_8] */
typedef volatile struct S1444 T1444;

/* TYPED_POINTER [ANY] */
typedef volatile struct S1447 T1447;
extern T0* GE_boxed1447(TC* ac, T1447 a1);
extern T0* GE_boxed_pointer1447(TC* ac, T1447* a1);
typedef volatile struct Sb1447 Tb1447;

/* SPECIAL [DS_ARRAYED_LIST [STRING_8]] */
typedef volatile struct S1448 T1448;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [STRING_8]] */
typedef volatile struct S1449 T1449;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [STRING_8]] */
typedef volatile struct S1450 T1450;

/* SPECIAL [TUPLE [STRING_8, detachable STRING_8]] */
typedef volatile struct S1451 T1451;

/* KL_SPECIAL_ROUTINES [TUPLE [STRING_8, detachable STRING_8]] */
typedef volatile struct S1453 T1453;

/* DS_HASH_TABLE_CURSOR [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
typedef volatile struct S1454 T1454;

/* DS_HASH_TABLE [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1455 T1455;

/* DS_ARRAYED_LIST [detachable DS_LINKABLE [INTEGER_32]] */
typedef volatile struct S1456 T1456;

/* ARRAY [BOOLEAN] */
typedef volatile struct S1457 T1457;

/* THREAD_ATTRIBUTES */
typedef volatile struct S1459 T1459;

/* CELL [BOOLEAN] */
typedef volatile struct S1460 T1460;

/* CHARACTER_PROPERTY */
typedef volatile struct S1462 T1462;

/* ET_SYSTEM_ERROR */
typedef volatile struct S1463 T1463;

/* ET_INTERNAL_ERROR */
typedef volatile struct S1464 T1464;

/* ET_SYNTAX_ERROR */
typedef volatile struct S1465 T1465;

/* ET_VALIDITY_ERROR */
typedef volatile struct S1466 T1466;

/* ET_FORMAL_COMMA_ARGUMENT */
typedef volatile struct S1467 T1467;

/* ET_LOCAL_COMMA_VARIABLE */
typedef volatile struct S1468 T1468;

/* ET_LABELED_COMMA_ACTUAL_PARAMETER */
typedef volatile struct S1469 T1469;

/* ET_KEYWORD_EXPRESSION */
typedef volatile struct S1471 T1471;

/* DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef volatile struct S1474 T1474;

/* SPECIAL [ET_CREATOR] */
typedef volatile struct S1475 T1475;

/* KL_SPECIAL_ROUTINES [ET_CREATOR] */
typedef volatile struct S1476 T1476;

/* KL_SPECIAL_ROUTINES [ET_FEATURE_NAME_ITEM] */
typedef volatile struct S1477 T1477;

/* SPECIAL [ET_FEATURE_NAME_ITEM] */
typedef volatile struct S1478 T1478;

/* ET_ECF_NOTE_ELEMENT */
typedef volatile struct S1479 T1479;

/* ET_ECF_ADAPTED_LIBRARIES */
typedef volatile struct S1480 T1480;

/* ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
typedef volatile struct S1481 T1481;

/* ET_ECF_FILE_RULES */
typedef volatile struct S1482 T1482;

/* ET_ECF_NAMESPACES */
typedef volatile struct S1483 T1483;

/* ET_ECF_EXTERNAL_CFLAGS */
typedef volatile struct S1484 T1484;

/* ET_ECF_EXTERNAL_INCLUDES */
typedef volatile struct S1485 T1485;

/* ET_ECF_EXTERNAL_LIBRARIES */
typedef volatile struct S1486 T1486;

/* ET_ECF_EXTERNAL_LINKER_FLAGS */
typedef volatile struct S1487 T1487;

/* ET_ECF_EXTERNAL_MAKES */
typedef volatile struct S1488 T1488;

/* ET_ECF_EXTERNAL_OBJECTS */
typedef volatile struct S1489 T1489;

/* ET_ECF_EXTERNAL_RESOURCES */
typedef volatile struct S1490 T1490;

/* DS_ARRAYED_LIST [ET_ECF_ACTION] */
typedef volatile struct S1491 T1491;

/* ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
typedef volatile struct S1492 T1492;

/* ET_ECF_VERSION */
typedef volatile struct S1494 T1494;

/* ET_ECF_NAMESPACE */
typedef volatile struct S1495 T1495;

/* ET_ECF_EXTERNAL_CFLAG */
typedef volatile struct S1496 T1496;

/* ET_ECF_EXTERNAL_INCLUDE */
typedef volatile struct S1497 T1497;

/* ET_ECF_EXTERNAL_LIBRARY */
typedef volatile struct S1498 T1498;

/* ET_ECF_EXTERNAL_LINKER_FLAG */
typedef volatile struct S1499 T1499;

/* ET_ECF_EXTERNAL_MAKE */
typedef volatile struct S1500 T1500;

/* ET_ECF_EXTERNAL_OBJECT */
typedef volatile struct S1501 T1501;

/* ET_ECF_EXTERNAL_RESOURCE */
typedef volatile struct S1502 T1502;

/* ET_ECF_FILE_RULE */
typedef volatile struct S1503 T1503;

/* ET_ECF_ACTION */
typedef volatile struct S1504 T1504;

/* DS_HASH_TABLE [ET_ECF_OPTIONS, STRING_8] */
typedef volatile struct S1505 T1505;

/* DS_ARRAYED_LIST [ET_ECF_VISIBLE_CLASS] */
typedef volatile struct S1506 T1506;

/* TUPLE [ET_ECF_OPTIONS] */
typedef volatile struct S1507 T1507;

/* PROCEDURE [TUPLE [ET_ECF_OPTIONS]] */
typedef volatile struct S1508 T1508;

/* ET_ECF_ROOT_CLASS */
typedef volatile struct S1509 T1509;

/* ET_ECF_ROOT_ALL_CLASSES */
typedef volatile struct S1510 T1510;

/* ET_ECF_ORED_CONDITIONS */
typedef volatile struct S1511 T1511;

/* ET_ECF_ANDED_CONDITIONS */
typedef volatile struct S1512 T1512;

/* ET_ECF_VISIBLE_CLASS */
typedef volatile struct S1513 T1513;

/* ET_ECF_BUILD_CONDITION */
typedef volatile struct S1515 T1515;

/* ET_ECF_CONCURRENCY_CONDITION */
typedef volatile struct S1516 T1516;

/* ET_ECF_VOID_SAFETY_CONDITION */
typedef volatile struct S1517 T1517;

/* ET_ECF_CUSTOM_CONDITION */
typedef volatile struct S1518 T1518;

/* ET_ECF_DOTNET_CONDITION */
typedef volatile struct S1519 T1519;

/* ET_ECF_DYNAMIC_RUNTIME_CONDITION */
typedef volatile struct S1520 T1520;

/* ET_ECF_PLATFORM_CONDITION */
typedef volatile struct S1521 T1521;

/* ET_ECF_COMPILER_VERSION_CONDITION */
typedef volatile struct S1523 T1523;

/* ET_ECF_MSIL_CLR_VERSION_CONDITION */
typedef volatile struct S1524 T1524;

/* UT_COUNTER */
typedef volatile struct S1526 T1526;

/* KL_AGENT_ROUTINES [ET_CLASS] */
typedef volatile struct S1527 T1527;

/* TUPLE [UT_COUNTER] */
typedef volatile struct S1528 T1528;

/* TUPLE [KL_AGENT_ROUTINES [ET_CLASS], PROCEDURE [TUPLE]] */
typedef volatile struct S1529 T1529;

/* TUPLE [PROCEDURE [TUPLE [ET_CLASS]]] */
typedef volatile struct S1530 T1530;

/* TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
typedef volatile struct S1531 T1531;

/* DS_HASH_SET [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S1533 T1533;

/* PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]] */
typedef volatile struct S1534 T1534;

/* TUPLE [DS_HASH_SET [ET_DOTNET_ASSEMBLY]] */
typedef volatile struct S1535 T1535;

/* TUPLE [ET_INTERNAL_UNIVERSE] */
typedef volatile struct S1536 T1536;

/* PROCEDURE [TUPLE [ET_INTERNAL_UNIVERSE]] */
typedef volatile struct S1537 T1537;

/* TUPLE [PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]], PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]]] */
typedef volatile struct S1538 T1538;

/* TUPLE [ET_DOTNET_ASSEMBLIES] */
typedef volatile struct S1539 T1539;

/* TUPLE [ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1540 T1540;

/* TUPLE [ET_CLUSTER] */
typedef volatile struct S1541 T1541;

/* PROCEDURE [TUPLE [ET_CLUSTER]] */
typedef volatile struct S1542 T1542;

/* TUPLE [ET_ECF_SYSTEM] */
typedef volatile struct S1543 T1543;

/* DS_HASH_SET [ET_INTERNAL_UNIVERSE] */
typedef volatile struct S1544 T1544;

/* TUPLE [ET_AST_PROCESSOR] */
typedef volatile struct S1545 T1545;

/* TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1546 T1546;

/* PROCEDURE [TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY]] */
typedef volatile struct S1547 T1547;

/* ET_ANCESTORS_STATUS_CHECKER */
typedef volatile struct S1548 T1548;

/* ET_FLATTENING_STATUS_CHECKER */
typedef volatile struct S1549 T1549;

/* ET_INTERFACE_STATUS_CHECKER */
typedef volatile struct S1550 T1550;

/* ET_IMPLEMENTATION_STATUS_CHECKER */
typedef volatile struct S1551 T1551;

/* TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
typedef volatile struct S1552 T1552;

/* TUPLE [ET_FLATTENING_STATUS_CHECKER] */
typedef volatile struct S1553 T1553;

/* TUPLE [ET_INTERFACE_STATUS_CHECKER] */
typedef volatile struct S1554 T1554;

/* TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
typedef volatile struct S1555 T1555;

/* TUPLE [BOOLEAN] */
typedef volatile struct S1556 T1556;

/* TUPLE [PROCEDURE [TUPLE [ET_CLUSTER]]] */
typedef volatile struct S1559 T1559;

/* TUPLE [DS_HASH_SET [ET_INTERNAL_UNIVERSE]] */
typedef volatile struct S1563 T1563;

/* TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]]] */
typedef volatile struct S1564 T1564;

/* ET_PARENTHESIS_EXPRESSION */
typedef volatile struct S1565 T1565;

/* TYPED_POINTER [NATURAL_8] */
typedef volatile struct S1566 T1566;
extern T0* GE_boxed1566(TC* ac, T1566 a1);
extern T0* GE_boxed_pointer1566(TC* ac, T1566* a1);
typedef volatile struct Sb1566 Tb1566;

/* TYPED_POINTER [NATURAL_16] */
typedef volatile struct S1567 T1567;
extern T0* GE_boxed1567(TC* ac, T1567 a1);
extern T0* GE_boxed_pointer1567(TC* ac, T1567* a1);
typedef volatile struct Sb1567 Tb1567;

/* DS_HASH_SET [ET_LIBRARY] */
typedef volatile struct S1568 T1568;

/* TUPLE [DS_HASH_SET [ET_LIBRARY]] */
typedef volatile struct S1569 T1569;

/* SPECIAL [ET_RENAME_ITEM] */
typedef volatile struct S1573 T1573;

/* KL_SPECIAL_ROUTINES [ET_RENAME_ITEM] */
typedef volatile struct S1574 T1574;

/* SPECIAL [ET_EXPORT] */
typedef volatile struct S1576 T1576;

/* KL_SPECIAL_ROUTINES [ET_EXPORT] */
typedef volatile struct S1577 T1577;

/* ET_CLUSTER_DEPENDENCE_CONSTRAINT */
typedef volatile struct S1580 T1580;

/* ET_PARENT_FEATURE */
typedef volatile struct S1582 T1582;

/* DS_QUICK_SORTER [ET_QUERY] */
typedef volatile struct S1583 T1583;

/* DS_QUICK_SORTER [ET_PROCEDURE] */
typedef volatile struct S1584 T1584;

/* ET_INHERITED_FEATURE */
typedef volatile struct S1585 T1585;

/* ET_REDECLARED_FEATURE */
typedef volatile struct S1587 T1587;

/* KL_CHARACTER_BUFFER */
typedef volatile struct S1595 T1595;

/* SPECIAL [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef volatile struct S1597 T1597;

/* KL_SPECIAL_ROUTINES [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef volatile struct S1598 T1598;

/* SPECIAL [ET_ALIAS_NAME] */
typedef volatile struct S1599 T1599;

/* KL_SPECIAL_ROUTINES [ET_ALIAS_NAME] */
typedef volatile struct S1600 T1600;

/* ET_AGENT_IMPLICIT_CURRENT_TARGET */
typedef volatile struct S1601 T1601;

/* SPECIAL [ET_CHOICE_ITEM] */
typedef volatile struct S1603 T1603;

/* KL_SPECIAL_ROUTINES [ET_CHOICE_ITEM] */
typedef volatile struct S1604 T1604;

/* SPECIAL [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S1605 T1605;

/* KL_SPECIAL_ROUTINES [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef volatile struct S1607 T1607;

/* SPECIAL [ET_CONVERT_FEATURE_ITEM] */
typedef volatile struct S1608 T1608;

/* KL_SPECIAL_ROUTINES [ET_CONVERT_FEATURE_ITEM] */
typedef volatile struct S1610 T1610;

/* SPECIAL [ET_ELSEIF_EXPRESSION] */
typedef volatile struct S1611 T1611;

/* KL_SPECIAL_ROUTINES [ET_ELSEIF_EXPRESSION] */
typedef volatile struct S1612 T1612;

/* SPECIAL [ET_ELSEIF_PART] */
typedef volatile struct S1613 T1613;

/* KL_SPECIAL_ROUTINES [ET_ELSEIF_PART] */
typedef volatile struct S1614 T1614;

/* SPECIAL [ET_FEATURE_CLAUSE] */
typedef volatile struct S1615 T1615;

/* KL_SPECIAL_ROUTINES [ET_FEATURE_CLAUSE] */
typedef volatile struct S1616 T1616;

/* SPECIAL [ET_FORMAL_ARGUMENT_ITEM] */
typedef volatile struct S1617 T1617;

/* KL_SPECIAL_ROUTINES [ET_FORMAL_ARGUMENT_ITEM] */
typedef volatile struct S1619 T1619;

/* SPECIAL [ET_FORMAL_PARAMETER_ITEM] */
typedef volatile struct S1620 T1620;

/* KL_SPECIAL_ROUTINES [ET_FORMAL_PARAMETER_ITEM] */
typedef volatile struct S1621 T1621;

/* SPECIAL [ET_LOCAL_VARIABLE_ITEM] */
typedef volatile struct S1622 T1622;

/* KL_SPECIAL_ROUTINES [ET_LOCAL_VARIABLE_ITEM] */
typedef volatile struct S1624 T1624;

/* SPECIAL [ET_MANIFEST_STRING_ITEM] */
typedef volatile struct S1626 T1626;

/* KL_SPECIAL_ROUTINES [ET_MANIFEST_STRING_ITEM] */
typedef volatile struct S1627 T1627;

/* SPECIAL [ET_NOTE_ITEM] */
typedef volatile struct S1629 T1629;

/* KL_SPECIAL_ROUTINES [ET_NOTE_ITEM] */
typedef volatile struct S1630 T1630;

/* SPECIAL [ET_NOTE_TERM_ITEM] */
typedef volatile struct S1631 T1631;

/* KL_SPECIAL_ROUTINES [ET_NOTE_TERM_ITEM] */
typedef volatile struct S1632 T1632;

/* SPECIAL [ET_PARENT_LIST] */
typedef volatile struct S1633 T1633;

/* KL_SPECIAL_ROUTINES [ET_PARENT_LIST] */
typedef volatile struct S1634 T1634;

/* SPECIAL [ET_TYPE_CONSTRAINT_ITEM] */
typedef volatile struct S1636 T1636;

/* KL_SPECIAL_ROUTINES [ET_TYPE_CONSTRAINT_ITEM] */
typedef volatile struct S1637 T1637;

/* SPECIAL [ET_WHEN_EXPRESSION] */
typedef volatile struct S1638 T1638;

/* KL_SPECIAL_ROUTINES [ET_WHEN_EXPRESSION] */
typedef volatile struct S1639 T1639;

/* SPECIAL [ET_WHEN_PART] */
typedef volatile struct S1640 T1640;

/* KL_SPECIAL_ROUTINES [ET_WHEN_PART] */
typedef volatile struct S1641 T1641;

/* XM_LINKED_LIST [XM_ELEMENT_NODE] */
typedef volatile struct S1643 T1643;

/* UC_STRING_EQUALITY_TESTER */
typedef volatile struct S1645 T1645;

/* DS_LINKED_LIST_CURSOR [DS_PAIR [XM_POSITION, XM_NODE]] */
typedef volatile struct S1647 T1647;

/* DS_LINKED_LIST [DS_PAIR [XM_POSITION, XM_NODE]] */
typedef volatile struct S1648 T1648;

/* DS_PAIR [XM_POSITION, XM_NODE] */
typedef volatile struct S1649 T1649;

/* XM_EIFFEL_INPUT_STREAM */
typedef volatile struct S1650 T1650;

/* DS_LINKED_LIST_CURSOR [XM_DOCUMENT_NODE] */
typedef volatile struct S1651 T1651;

/* DS_ARRAYED_LIST [ET_ECF_TARGET] */
typedef volatile struct S1652 T1652;

/* DS_HASH_TABLE_CURSOR [ET_ECF_OPTIONS, STRING_8] */
typedef volatile struct S1653 T1653;

/* DS_ARRAYED_LIST [ET_ADAPTED_UNIVERSE] */
typedef volatile struct S1654 T1654;

/* DS_ARRAYED_LIST [ET_ECF_TARGET_PARENT] */
typedef volatile struct S1656 T1656;

/* DS_ARRAYED_LIST [ET_ECF_CLUSTER] */
typedef volatile struct S1657 T1657;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
typedef volatile struct S1659 T1659;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE] */
typedef volatile struct S1662 T1662;

/* SPECIAL [ET_DYNAMIC_TYPE] */
typedef volatile struct S1663 T1663;

/* INTEGER_OVERFLOW_CHECKER */
typedef volatile struct S1668 T1668;

/* CONSOLE */
typedef volatile struct S1669 T1669;

/* C_DATE */
typedef volatile struct S1670 T1670;

/* DS_HASH_TABLE_CURSOR [INTEGER_32, STRING_8] */
typedef volatile struct S1672 T1672;

/* KL_SPECIAL_ROUTINES [ET_ECF_NOTE_ELEMENT] */
typedef volatile struct S1673 T1673;

/* SPECIAL [ET_ECF_NOTE_ELEMENT] */
typedef volatile struct S1674 T1674;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_NOTE_ELEMENT] */
typedef volatile struct S1675 T1675;

/* ET_DYNAMIC_TARGET_LIST */
typedef volatile struct S1677 T1677;

/* DS_ARRAYED_STACK [YY_BUFFER] */
typedef volatile struct S1678 T1678;

/* UT_SYNTAX_ERROR */
typedef volatile struct S1679 T1679;

/* UT_TOO_MANY_INCLUDES_ERROR */
typedef volatile struct S1680 T1680;

/* ARRAY [detachable STRING_8] */
typedef volatile struct S1681 T1681;

/* SPECIAL [ET_ADAPTED_LIBRARY] */
typedef volatile struct S1683 T1683;

/* KL_SPECIAL_ROUTINES [ET_ADAPTED_LIBRARY] */
typedef volatile struct S1684 T1684;

/* DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_LIBRARY] */
typedef volatile struct S1685 T1685;

/* SPECIAL [ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1686 T1686;

/* KL_SPECIAL_ROUTINES [ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1687 T1687;

/* DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1688 T1688;

/* SPECIAL [ET_TYPE_ITEM] */
typedef volatile struct S1689 T1689;

/* KL_SPECIAL_ROUTINES [ET_TYPE_ITEM] */
typedef volatile struct S1690 T1690;

/* DS_HASH_TABLE [INTEGER_32, ET_CLASS] */
typedef volatile struct S1691 T1691;

/* DS_HASH_TABLE_CURSOR [ET_BASE_TYPE, ET_CLASS] */
typedef volatile struct S1692 T1692;

/* DS_ARRAYED_LIST [ET_BASE_TYPE_CONSTRAINT] */
typedef volatile struct S1694 T1694;

/* DS_ARRAYED_LIST [NATURAL_32] */
typedef volatile struct S1695 T1695;

/* ET_BASE_TYPE_CONSTRAINT_LIST */
typedef volatile struct S1696 T1696;

/* DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
typedef volatile struct S1697 T1697;

/* SPECIAL [ET_FLATTENED_FEATURE] */
typedef volatile struct S1698 T1698;

/* SPECIAL [ET_FEATURE_NAME] */
typedef volatile struct S1701 T1701;

/* KL_SPECIAL_ROUTINES [ET_FLATTENED_FEATURE] */
typedef volatile struct S1702 T1702;

/* KL_SPECIAL_ROUTINES [ET_FEATURE_NAME] */
typedef volatile struct S1703 T1703;

/* DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
typedef volatile struct S1705 T1705;

/* KL_SPECIAL_ROUTINES [ET_CLIENT_LIST] */
typedef volatile struct S1706 T1706;

/* SPECIAL [ET_CLIENT_LIST] */
typedef volatile struct S1707 T1707;

/* DS_ARRAYED_LIST_CURSOR [ET_CLIENT_LIST] */
typedef volatile struct S1708 T1708;

/* DS_HASH_TABLE_CURSOR [ET_CLIENT, ET_CLASS] */
typedef volatile struct S1710 T1710;

/* KL_SPECIAL_ROUTINES [ET_CLIENT] */
typedef volatile struct S1711 T1711;

/* SPECIAL [ET_CLIENT] */
typedef volatile struct S1712 T1712;

/* DS_HASH_SET [ET_FEATURE_NAME] */
typedef volatile struct S1713 T1713;

/* DS_HASH_TABLE [BOOLEAN, ET_FEATURE_NAME] */
typedef volatile struct S1714 T1714;

/* DS_HASH_TABLE [ET_REPLICABLE_FEATURE, INTEGER_32] */
typedef volatile struct S1715 T1715;

/* ET_REPLICATED_FEATURE */
typedef volatile struct S1717 T1717;

/* DS_LINKED_LIST [ET_ADAPTED_FEATURE] */
typedef volatile struct S1718 T1718;

/* DS_ARRAYED_LIST [ET_PARENT_FEATURE] */
typedef volatile struct S1719 T1719;

/* ET_DOTNET_SIGNATURE_TESTER */
typedef volatile struct S1720 T1720;

/* DS_HASH_SET [ET_DOTNET_FEATURE] */
typedef volatile struct S1721 T1721;

/* DS_HASH_TABLE [DS_LINKED_LIST [ET_DOTNET_FEATURE], ET_DOTNET_FEATURE] */
typedef volatile struct S1723 T1723;

/* DS_LINKED_LIST [ET_DOTNET_FEATURE] */
typedef volatile struct S1725 T1725;

/* DS_HASH_TOPOLOGICAL_SORTER [ET_LIKE_FEATURE] */
typedef volatile struct S1726 T1726;

/* DS_ARRAYED_LIST [ET_LIKE_FEATURE] */
typedef volatile struct S1727 T1727;

/* DS_LINKED_LIST_CURSOR [ET_PARENT_FEATURE] */
typedef volatile struct S1728 T1728;

/* DS_LINKED_LIST [ET_PARENT_FEATURE] */
typedef volatile struct S1729 T1729;

/* DS_HASH_TABLE [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME], NATURAL_8] */
typedef volatile struct S1730 T1730;

/* DS_HASH_TABLE_CURSOR [ET_FEATURE, INTEGER_32] */
typedef volatile struct S1731 T1731;

/* TYPE [ET_FEATURE] */
#define T1732 EIF_TYPE_OBJ

/* DS_HASH_SET_CURSOR [ET_CLASS] */
typedef volatile struct S1734 T1734;

/* TYPE [ET_CLASS] */
#define T1735 EIF_TYPE_OBJ

/* DS_HASH_TABLE_CURSOR [ET_RENAME, ET_FEATURE_NAME] */
typedef volatile struct S1736 T1736;

/* SPECIAL [ET_RENAME] */
typedef volatile struct S1737 T1737;

/* KL_SPECIAL_ROUTINES [ET_RENAME] */
typedef volatile struct S1740 T1740;

/* TYPE [ET_RENAME] */
#define T1741 EIF_TYPE_OBJ

/* TYPE [ET_FEATURE_NAME] */
#define T1742 EIF_TYPE_OBJ

/* DS_HASH_TABLE_CURSOR [ET_RENAME, ET_ALIAS_NAME] */
typedef volatile struct S1744 T1744;

/* DS_HASH_SET_CURSOR [ET_PROCEDURE] */
typedef volatile struct S1745 T1745;

/* DS_HASH_SET_CURSOR [ET_QUERY] */
typedef volatile struct S1746 T1746;

/* KL_SPECIAL_ROUTINES [NATURAL_8] */
typedef volatile struct S1749 T1749;

/* DS_HASH_TABLE_CURSOR [NATURAL_8, ET_CLASS_NAME] */
typedef volatile struct S1750 T1750;

/* XM_EIFFEL_CHARACTER_ENTITY */
typedef volatile struct S1751 T1751;

/* DS_BILINKABLE [XM_POSITION] */
typedef volatile struct S1752 T1752;

/* DS_BILINKED_LIST_CURSOR [XM_POSITION] */
typedef volatile struct S1753 T1753;

/* DS_LINKABLE [XM_EIFFEL_SCANNER] */
typedef volatile struct S1754 T1754;

/* SPECIAL [XM_EIFFEL_ENTITY_DEF] */
typedef volatile struct S1755 T1755;

/* DS_HASH_TABLE_CURSOR [XM_EIFFEL_ENTITY_DEF, STRING_8] */
typedef volatile struct S1758 T1758;

/* KL_SPECIAL_ROUTINES [XM_EIFFEL_ENTITY_DEF] */
typedef volatile struct S1759 T1759;

/* KL_EQUALITY_TESTER [XM_EIFFEL_PARSER_NAME] */
typedef volatile struct S1762 T1762;

/* DS_HASH_SET_CURSOR [XM_EIFFEL_PARSER_NAME] */
typedef volatile struct S1763 T1763;

/* DS_BILINKED_LIST [XM_DTD_ELEMENT_CONTENT] */
typedef volatile struct S1764 T1764;

/* DS_BILINKED_LIST_CURSOR [XM_DTD_ATTRIBUTE_CONTENT] */
typedef volatile struct S1765 T1765;

/* DS_BILINKABLE [XM_DTD_ATTRIBUTE_CONTENT] */
typedef volatile struct S1766 T1766;

/* DS_BILINKED_LIST_CURSOR [STRING_8] */
typedef volatile struct S1767 T1767;

/* DS_BILINKABLE [STRING_8] */
typedef volatile struct S1768 T1768;

/* XM_NAMESPACE_RESOLVER_CONTEXT */
typedef volatile struct S1769 T1769;

/* DS_LINKED_QUEUE [detachable STRING_8] */
typedef volatile struct S1772 T1772;

/* DS_LINKED_QUEUE [STRING_8] */
typedef volatile struct S1773 T1773;

/* KL_EQUALITY_TESTER [XM_NAMESPACE] */
typedef volatile struct S1774 T1774;

/* DS_HASH_SET_CURSOR [XM_NAMESPACE] */
typedef volatile struct S1775 T1775;

/* KL_SPECIAL_ROUTINES [XM_NAMESPACE] */
typedef volatile struct S1776 T1776;

/* SPECIAL [XM_NAMESPACE] */
typedef volatile struct S1777 T1777;

/* DS_LINKABLE [XM_DOCUMENT_NODE] */
typedef volatile struct S1778 T1778;

/* DS_ARRAYED_LIST [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S1780 T1780;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_FEATURE, INTEGER_32] */
typedef volatile struct S1781 T1781;

/* SPECIAL [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
typedef volatile struct S1783 T1783;

/* SPECIAL [ET_CALL_NAME] */
typedef volatile struct S1785 T1785;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
typedef volatile struct S1787 T1787;

/* KL_SPECIAL_ROUTINES [ET_CALL_NAME] */
typedef volatile struct S1788 T1788;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
typedef volatile struct S1789 T1789;

/* SPECIAL [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
typedef volatile struct S1790 T1790;

/* DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
typedef volatile struct S1792 T1792;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
typedef volatile struct S1793 T1793;

/* SPECIAL [ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1794 T1794;

/* KL_SPECIAL_ROUTINES [ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1795 T1795;

/* DS_ARRAYED_LIST_CURSOR [ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1796 T1796;

/* KL_SPECIAL_ROUTINES [ET_ADAPTED_CLASS] */
typedef volatile struct S1797 T1797;

/* SPECIAL [ET_ADAPTED_CLASS] */
typedef volatile struct S1799 T1799;

/* DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_CLASS] */
typedef volatile struct S1800 T1800;

/* DS_ARRAYED_LIST_CURSOR [ET_NAMED_OBJECT_TEST] */
typedef volatile struct S1801 T1801;

/* DS_ARRAYED_LIST_CURSOR [ET_ITERATION_COMPONENT] */
typedef volatile struct S1802 T1802;

/* DS_ARRAYED_LIST_CURSOR [ET_INLINE_SEPARATE_ARGUMENT] */
typedef volatile struct S1803 T1803;

/* KL_SPECIAL_ROUTINES [RX_CHARACTER_SET] */
typedef volatile struct S1804 T1804;

/* DS_ARRAYED_LIST_CURSOR [RX_CHARACTER_SET] */
typedef volatile struct S1805 T1805;

/* KL_SPECIAL_ROUTINES [NATURAL_64] */
typedef volatile struct S1810 T1810;

/* TYPE [NATURAL_64] */
#define T1811 EIF_TYPE_OBJ

/* TYPE [NATURAL_32] */
#define T1812 EIF_TYPE_OBJ

/* DS_HASH_TABLE_CURSOR [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S1814 T1814;

/* SPECIAL [detachable DS_LINKABLE [INTEGER_32]] */
typedef volatile struct S1815 T1815;

/* KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [INTEGER_32]] */
typedef volatile struct S1816 T1816;

/* DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [INTEGER_32]] */
typedef volatile struct S1817 T1817;

/* KL_STRING_INPUT_STREAM */
typedef volatile struct S1819 T1819;

/* UT_TRISTATE */
typedef volatile struct S1821 T1821;

/* KL_AGENT_ROUTINES [ANY] */
typedef volatile struct S1822 T1822;

/* TUPLE [UT_TRISTATE] */
typedef volatile struct S1823 T1823;

/* TUPLE [KL_AGENT_ROUTINES [ANY], PROCEDURE [TUPLE]] */
typedef volatile struct S1824 T1824;

/* PREDICATE [TUPLE [ET_MASTER_CLASS]] */
typedef volatile struct S1825 T1825;

/* PREDICATE [TUPLE] */
typedef volatile struct S1826 T1826;

/* DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE] */
typedef volatile struct S1828 T1828;

/* DS_HASH_TABLE_CURSOR [INTEGER_32, ET_UNIVERSE] */
typedef volatile struct S1829 T1829;

/* TUPLE [ET_ECF_SYSTEM, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S1830 T1830;

/* KL_AGENT_ROUTINES [ET_UNIVERSE] */
typedef volatile struct S1831 T1831;

/* PREDICATE [TUPLE [ET_UNIVERSE]] */
typedef volatile struct S1832 T1832;

/* TUPLE [DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE]] */
typedef volatile struct S1833 T1833;

/* TUPLE [KL_AGENT_ROUTINES [ET_UNIVERSE], PREDICATE [TUPLE [ET_UNIVERSE]]] */
typedef volatile struct S1834 T1834;

/* TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
typedef volatile struct S1836 T1836;

/* TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN]] */
typedef volatile struct S1837 T1837;

/* TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN], FUNCTION [TUPLE, BOOLEAN]] */
typedef volatile struct S1838 T1838;

/* ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
typedef volatile struct S1839 T1839;

/* TUPLE [ET_MASTER_CLASS, ET_SYSTEM_PROCESSOR] */
typedef volatile struct S1847 T1847;

/* DS_CELL [detachable ET_CLASS] */
typedef volatile struct S1848 T1848;

/* TUPLE [DS_CELL [detachable ET_CLASS]] */
typedef volatile struct S1849 T1849;

/* TUPLE [ET_EIFFEL_PREPARSER, DS_ARRAYED_LIST [STRING_8]] */
typedef volatile struct S1851 T1851;

/* ET_SEEDED_QUERY_COMPARATOR */
typedef volatile struct S1853 T1853;

/* ET_SEEDED_PROCEDURE_COMPARATOR */
typedef volatile struct S1856 T1856;

/* TUPLE [ET_CLIENT_LIST] */
typedef volatile struct S1860 T1860;

/* ET_STANDARD_ONCE_KEYS */
typedef volatile struct S1864 T1864;

/* XM_COMMENT */
typedef volatile struct S1867 T1867;

/* XM_PROCESSING_INSTRUCTION */
typedef volatile struct S1868 T1868;

/* XM_CHARACTER_DATA */
typedef volatile struct S1869 T1869;

/* ET_PARENTHESIS_SYMBOL */
typedef volatile struct S1885 T1885;

/* ET_PARENTHESIS_INSTRUCTION */
typedef volatile struct S1888 T1888;

/* ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
typedef volatile struct S1892 T1892;

/* ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
typedef volatile struct S1893 T1893;

/* ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
typedef volatile struct S1894 T1894;

/* ET_INFIX_CAST_EXPRESSION */
typedef volatile struct S1896 T1896;

/* SPECIAL [ET_DYNAMIC_PRECURSOR] */
typedef volatile struct S1899 T1899;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRECURSOR] */
typedef volatile struct S1900 T1900;

/* TYPE [ET_NESTED_TYPE_CONTEXT] */
#define T1902 EIF_TYPE_OBJ

/* TYPE [ET_ITERATION_COMPONENT] */
#define T1903 EIF_TYPE_OBJ

/* TYPE [ET_INLINE_SEPARATE_ARGUMENT] */
#define T1905 EIF_TYPE_OBJ

/* DS_ARRAYED_LIST_CURSOR [ET_MASTER_CLASS] */
typedef volatile struct S1908 T1908;

/* ET_BASE_TYPE_RENAME_CONSTRAINT */
typedef volatile struct S1910 T1910;

/* DS_SPARSE_TABLE_KEYS [ET_CLIENT, ET_CLASS] */
typedef volatile struct S1913 T1913;

/* SPECIAL [NATURAL_16] */
typedef volatile struct S1922 T1922;

/* ARRAY [NATURAL_16] */
typedef volatile struct S1923 T1923;

/* ARRAY [NATURAL_32] */
typedef volatile struct S1924 T1924;

/* SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef volatile struct S1925 T1925;

/* KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef volatile struct S1926 T1926;

/* DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef volatile struct S1927 T1927;

/* DS_ARRAYED_LIST [ET_ECF_ADAPTED_LIBRARY] */
typedef volatile struct S1928 T1928;

/* DS_ARRAYED_LIST [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S1929 T1929;

/* DS_ARRAYED_LIST [ET_ECF_FILE_RULE] */
typedef volatile struct S1930 T1930;

/* DS_ARRAYED_LIST [ET_ECF_NAMESPACE] */
typedef volatile struct S1931 T1931;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_CFLAG] */
typedef volatile struct S1932 T1932;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_INCLUDE] */
typedef volatile struct S1933 T1933;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LIBRARY] */
typedef volatile struct S1934 T1934;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef volatile struct S1935 T1935;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_MAKE] */
typedef volatile struct S1936 T1936;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_OBJECT] */
typedef volatile struct S1937 T1937;

/* DS_ARRAYED_LIST [ET_ECF_EXTERNAL_RESOURCE] */
typedef volatile struct S1938 T1938;

/* KL_SPECIAL_ROUTINES [ET_ECF_ACTION] */
typedef volatile struct S1939 T1939;

/* SPECIAL [ET_ECF_ACTION] */
typedef volatile struct S1940 T1940;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_ACTION] */
typedef volatile struct S1941 T1941;

/* DS_ARRAYED_LIST [RX_PCRE_REGULAR_EXPRESSION] */
typedef volatile struct S1942 T1942;

/* SPECIAL [ET_ECF_OPTIONS] */
typedef volatile struct S1943 T1943;

/* KL_SPECIAL_ROUTINES [ET_ECF_OPTIONS] */
typedef volatile struct S1946 T1946;

/* KL_SPECIAL_ROUTINES [ET_ECF_VISIBLE_CLASS] */
typedef volatile struct S1947 T1947;

/* SPECIAL [ET_ECF_VISIBLE_CLASS] */
typedef volatile struct S1948 T1948;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_VISIBLE_CLASS] */
typedef volatile struct S1949 T1949;

/* DS_ARRAYED_LIST [ET_ECF_ANDED_CONDITIONS] */
typedef volatile struct S1950 T1950;

/* DS_ARRAYED_LIST [ET_ECF_CONDITION_ITEM] */
typedef volatile struct S1951 T1951;

/* DS_HASH_SET_CURSOR [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S1952 T1952;

/* KL_SPECIAL_ROUTINES [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S1953 T1953;

/* SPECIAL [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S1954 T1954;

/* DS_HASH_SET_CURSOR [ET_INTERNAL_UNIVERSE] */
typedef volatile struct S1956 T1956;

/* KL_SPECIAL_ROUTINES [ET_INTERNAL_UNIVERSE] */
typedef volatile struct S1958 T1958;

/* SPECIAL [ET_INTERNAL_UNIVERSE] */
typedef volatile struct S1959 T1959;

/* ET_CLASS_TYPE_STATUS_CHECKER1 */
typedef volatile struct S1960 T1960;

/* ET_CLASS_TYPE_STATUS_CHECKER2 */
typedef volatile struct S1961 T1961;

/* ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
typedef volatile struct S1962 T1962;

/* ET_CLASS_TYPE_STATUS_CHECKER3 */
typedef volatile struct S1963 T1963;

/* DS_HASH_SET_CURSOR [ET_LIBRARY] */
typedef volatile struct S1964 T1964;

/* KL_SPECIAL_ROUTINES [ET_LIBRARY] */
typedef volatile struct S1965 T1965;

/* SPECIAL [ET_LIBRARY] */
typedef volatile struct S1966 T1966;

/* LX_DFA_WILDCARD */
typedef volatile struct S1967 T1967;

/* DS_HASH_TABLE [LX_WILDCARD, STRING_8] */
typedef volatile struct S1968 T1968;

/* DS_LINKED_QUEUE [CHARACTER_8] */
typedef volatile struct S1971 T1971;

/* UC_UTF16_ROUTINES */
typedef volatile struct S1972 T1972;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET] */
typedef volatile struct S1973 T1973;

/* SPECIAL [ET_ADAPTED_UNIVERSE] */
typedef volatile struct S1974 T1974;

/* KL_SPECIAL_ROUTINES [ET_ADAPTED_UNIVERSE] */
typedef volatile struct S1975 T1975;

/* DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_UNIVERSE] */
typedef volatile struct S1976 T1976;

/* SPECIAL [ET_ECF_TARGET_PARENT] */
typedef volatile struct S1977 T1977;

/* KL_SPECIAL_ROUTINES [ET_ECF_TARGET_PARENT] */
typedef volatile struct S1978 T1978;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET_PARENT] */
typedef volatile struct S1979 T1979;

/* SPECIAL [ET_ECF_CLUSTER] */
typedef volatile struct S1980 T1980;

/* KL_SPECIAL_ROUTINES [ET_ECF_CLUSTER] */
typedef volatile struct S1981 T1981;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_CLUSTER] */
typedef volatile struct S1982 T1982;

/* SPECIAL [ET_DYNAMIC_TARGET] */
typedef volatile struct S1983 T1983;

/* KL_SPECIAL_ROUTINES [ET_DYNAMIC_TARGET] */
typedef volatile struct S1984 T1984;

/* SPECIAL [YY_BUFFER] */
typedef volatile struct S1985 T1985;

/* KL_SPECIAL_ROUTINES [YY_BUFFER] */
typedef volatile struct S1986 T1986;

/* TYPE [detachable STRING_8] */
#define T1987 EIF_TYPE_OBJ

/* DS_HASH_TABLE_CURSOR [INTEGER_32, ET_CLASS] */
typedef volatile struct S1988 T1988;

/* SPECIAL [ET_BASE_TYPE_CONSTRAINT] */
typedef volatile struct S1989 T1989;

/* KL_SPECIAL_ROUTINES [ET_BASE_TYPE_CONSTRAINT] */
typedef volatile struct S1990 T1990;

/* DS_ARRAYED_LIST_CURSOR [ET_BASE_TYPE_CONSTRAINT] */
typedef volatile struct S1991 T1991;

/* DS_ARRAYED_LIST_CURSOR [NATURAL_32] */
typedef volatile struct S1992 T1992;

/* DS_HASH_SET_CURSOR [ET_FEATURE_NAME] */
typedef volatile struct S1993 T1993;

/* DS_HASH_TABLE_CURSOR [BOOLEAN, ET_FEATURE_NAME] */
typedef volatile struct S1994 T1994;

/* DS_HASH_TABLE_CURSOR [ET_REPLICABLE_FEATURE, INTEGER_32] */
typedef volatile struct S1996 T1996;

/* SPECIAL [ET_REPLICABLE_FEATURE] */
typedef volatile struct S1997 T1997;

/* KL_SPECIAL_ROUTINES [ET_REPLICABLE_FEATURE] */
typedef volatile struct S1999 T1999;

/* DS_LINKED_LIST_CURSOR [ET_ADAPTED_FEATURE] */
typedef volatile struct S2000 T2000;

/* DS_LINKABLE [ET_ADAPTED_FEATURE] */
typedef volatile struct S2001 T2001;

/* KL_SPECIAL_ROUTINES [ET_PARENT_FEATURE] */
typedef volatile struct S2002 T2002;

/* SPECIAL [ET_PARENT_FEATURE] */
typedef volatile struct S2003 T2003;

/* DS_ARRAYED_LIST_CURSOR [ET_PARENT_FEATURE] */
typedef volatile struct S2004 T2004;

/* SPECIAL [ET_DOTNET_FEATURE] */
typedef volatile struct S2005 T2005;

/* DS_HASH_SET_CURSOR [ET_DOTNET_FEATURE] */
typedef volatile struct S2006 T2006;

/* KL_SPECIAL_ROUTINES [ET_DOTNET_FEATURE] */
typedef volatile struct S2007 T2007;

/* SPECIAL [DS_LINKED_LIST [ET_DOTNET_FEATURE]] */
typedef volatile struct S2008 T2008;

/* DS_HASH_TABLE_CURSOR [DS_LINKED_LIST [ET_DOTNET_FEATURE], ET_DOTNET_FEATURE] */
typedef volatile struct S2011 T2011;

/* KL_SPECIAL_ROUTINES [DS_LINKED_LIST [ET_DOTNET_FEATURE]] */
typedef volatile struct S2012 T2012;

/* DS_LINKED_LIST_CURSOR [ET_DOTNET_FEATURE] */
typedef volatile struct S2013 T2013;

/* DS_LINKABLE [ET_DOTNET_FEATURE] */
typedef volatile struct S2014 T2014;

/* DS_HASH_TABLE [INTEGER_32, ET_LIKE_FEATURE] */
typedef volatile struct S2015 T2015;

/* SPECIAL [ET_LIKE_FEATURE] */
typedef volatile struct S2016 T2016;

/* KL_SPECIAL_ROUTINES [ET_LIKE_FEATURE] */
typedef volatile struct S2017 T2017;

/* DS_ARRAYED_LIST_CURSOR [ET_LIKE_FEATURE] */
typedef volatile struct S2018 T2018;

/* DS_HASH_TABLE_CURSOR [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME], NATURAL_8] */
typedef volatile struct S2020 T2020;

/* KL_SPECIAL_ROUTINES [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME]] */
typedef volatile struct S2021 T2021;

/* DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME] */
typedef volatile struct S2022 T2022;

/* SPECIAL [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME]] */
typedef volatile struct S2023 T2023;

/* DS_BILINKED_LIST_CURSOR [XM_DTD_ELEMENT_CONTENT] */
typedef volatile struct S2024 T2024;

/* DS_BILINKABLE [XM_DTD_ELEMENT_CONTENT] */
typedef volatile struct S2025 T2025;

/* DS_BILINKED_LIST [DS_HASH_TABLE [STRING_8, STRING_8]] */
typedef volatile struct S2027 T2027;

/* DS_BILINKED_LIST_CURSOR [DS_HASH_TABLE [STRING_8, STRING_8]] */
typedef volatile struct S2028 T2028;

/* DS_LINKABLE [detachable STRING_8] */
typedef volatile struct S2029 T2029;

/* DS_ARRAYED_LIST_CURSOR [ET_DOTNET_ASSEMBLY] */
typedef volatile struct S2030 T2030;

/* ET_CLUSTER_ERROR */
typedef volatile struct S2031 T2031;

/* TUPLE [ET_SYSTEM, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S2032 T2032;

/* TUPLE [ET_ECF_CLUSTER] */
typedef volatile struct S2033 T2033;

/* PROCEDURE [TUPLE [ET_ECF_CLUSTER]] */
typedef volatile struct S2034 T2034;

/* TUPLE [ET_ECF_TARGET, STRING_8] */
typedef volatile struct S2035 T2035;

/* TUPLE [ET_ECF_ADAPTED_LIBRARY] */
typedef volatile struct S2037 T2037;

/* PROCEDURE [TUPLE [ET_ECF_ADAPTED_LIBRARY]] */
typedef volatile struct S2038 T2038;

/* TUPLE [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S2039 T2039;

/* PROCEDURE [TUPLE [ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
typedef volatile struct S2040 T2040;

/* PROCEDURE [TUPLE [STRING_8]] */
typedef volatile struct S2043 T2043;

/* TUPLE [ET_ECF_OPTIONS, STRING_8] */
typedef volatile struct S2044 T2044;

/* DS_HASH_TABLE [DS_HASH_TABLE [INTEGER_32, STRING_8], STRING_8] */
typedef volatile struct S2045 T2045;

/* ET_DYNAMIC_EQUALITY_EXPRESSION */
typedef volatile struct S2046 T2046;

/* ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
typedef volatile struct S2047 T2047;

/* ET_AST_PRETTY_PRINTER */
typedef volatile struct S2052 T2052;

/* KL_CHARACTER_ROUTINES */
typedef volatile struct S2054 T2054;

/* TUPLE [ET_C_GENERATOR, ET_SEPARATE_CALL, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S2055 T2055;

/* TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
typedef volatile struct S2056 T2056;

/* TUPLE [ET_C_GENERATOR, INTEGER_32, ET_CURRENT, ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
typedef volatile struct S2057 T2057;

/* TYPE [detachable DEVELOPER_EXCEPTION] */
#define T2058 EIF_TYPE_OBJ

/* TUPLE [ET_GROUP] */
typedef volatile struct S2065 T2065;

/* TUPLE [PROCEDURE [TUPLE [ET_CLASS]], PREDICATE [TUPLE [ET_CLASS]]] */
typedef volatile struct S2066 T2066;

/* TUPLE [ET_ECF_LIBRARY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S2067 T2067;

/* ET_CONVERT_BUILTIN_EXPRESSION */
typedef volatile struct S2068 T2068;

/* ET_CONVERT_FROM_EXPRESSION */
typedef volatile struct S2069 T2069;

/* ET_CONVERT_TO_EXPRESSION */
typedef volatile struct S2070 T2070;

/* DS_LINKABLE [XM_ELEMENT_NODE] */
typedef volatile struct S2082 T2082;

/* TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8] */
typedef volatile struct S2083 T2083;

/* DS_ARRAYED_LIST_CURSOR [ET_FEATURE] */
typedef volatile struct S2086 T2086;

/* TUPLE [ET_ECF_EXTERNAL_CFLAG] */
typedef volatile struct S2092 T2092;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_CFLAG]] */
typedef volatile struct S2093 T2093;

/* TUPLE [ET_ECF_TARGET, ET_ECF_STATE] */
typedef volatile struct S2094 T2094;

/* TUPLE [ET_ECF_EXTERNAL_INCLUDE] */
typedef volatile struct S2095 T2095;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_INCLUDE]] */
typedef volatile struct S2096 T2096;

/* TUPLE [ET_ECF_EXTERNAL_LIBRARY] */
typedef volatile struct S2097 T2097;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_LIBRARY]] */
typedef volatile struct S2098 T2098;

/* TUPLE [ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef volatile struct S2099 T2099;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_LINKER_FLAG]] */
typedef volatile struct S2100 T2100;

/* TUPLE [ET_ECF_EXTERNAL_MAKE] */
typedef volatile struct S2101 T2101;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_MAKE]] */
typedef volatile struct S2102 T2102;

/* TUPLE [ET_ECF_EXTERNAL_OBJECT] */
typedef volatile struct S2103 T2103;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_OBJECT]] */
typedef volatile struct S2104 T2104;

/* TUPLE [ET_ECF_EXTERNAL_RESOURCE] */
typedef volatile struct S2105 T2105;

/* PROCEDURE [TUPLE [ET_ECF_EXTERNAL_RESOURCE]] */
typedef volatile struct S2106 T2106;

/* TUPLE [LX_WILDCARD] */
typedef volatile struct S2109 T2109;

/* PREDICATE [TUPLE [LX_WILDCARD]] */
typedef volatile struct S2110 T2110;

/* TUPLE [ET_CLUSTER_DEPENDENCE_CONSTRAINT, ET_GROUP, STRING_8] */
typedef volatile struct S2111 T2111;

/* PREDICATE [TUPLE [STRING_8]] */
typedef volatile struct S2113 T2113;

/* TUPLE [KL_STRING_ROUTINES, STRING_8] */
typedef volatile struct S2114 T2114;

/* DS_LINKABLE [DS_PAIR [XM_POSITION, XM_NODE]] */
typedef volatile struct S2116 T2116;

/* SPECIAL [ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef volatile struct S2118 T2118;

/* KL_SPECIAL_ROUTINES [ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef volatile struct S2119 T2119;

/* DS_SPARSE_TABLE_KEYS_CURSOR [ET_CLIENT, ET_CLASS] */
typedef volatile struct S2120 T2120;

/* SPECIAL [ET_ECF_ADAPTED_LIBRARY] */
typedef volatile struct S2122 T2122;

/* KL_SPECIAL_ROUTINES [ET_ECF_ADAPTED_LIBRARY] */
typedef volatile struct S2123 T2123;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_ADAPTED_LIBRARY] */
typedef volatile struct S2124 T2124;

/* SPECIAL [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S2125 T2125;

/* KL_SPECIAL_ROUTINES [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S2126 T2126;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef volatile struct S2127 T2127;

/* KL_SPECIAL_ROUTINES [ET_ECF_FILE_RULE] */
typedef volatile struct S2128 T2128;

/* SPECIAL [ET_ECF_FILE_RULE] */
typedef volatile struct S2129 T2129;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_FILE_RULE] */
typedef volatile struct S2130 T2130;

/* KL_SPECIAL_ROUTINES [ET_ECF_NAMESPACE] */
typedef volatile struct S2131 T2131;

/* SPECIAL [ET_ECF_NAMESPACE] */
typedef volatile struct S2132 T2132;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_NAMESPACE] */
typedef volatile struct S2133 T2133;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_CFLAG] */
typedef volatile struct S2134 T2134;

/* SPECIAL [ET_ECF_EXTERNAL_CFLAG] */
typedef volatile struct S2135 T2135;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_CFLAG] */
typedef volatile struct S2136 T2136;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_INCLUDE] */
typedef volatile struct S2137 T2137;

/* SPECIAL [ET_ECF_EXTERNAL_INCLUDE] */
typedef volatile struct S2138 T2138;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_INCLUDE] */
typedef volatile struct S2139 T2139;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_LIBRARY] */
typedef volatile struct S2140 T2140;

/* SPECIAL [ET_ECF_EXTERNAL_LIBRARY] */
typedef volatile struct S2141 T2141;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_LIBRARY] */
typedef volatile struct S2142 T2142;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef volatile struct S2143 T2143;

/* SPECIAL [ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef volatile struct S2144 T2144;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef volatile struct S2145 T2145;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_MAKE] */
typedef volatile struct S2146 T2146;

/* SPECIAL [ET_ECF_EXTERNAL_MAKE] */
typedef volatile struct S2147 T2147;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_MAKE] */
typedef volatile struct S2148 T2148;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_OBJECT] */
typedef volatile struct S2149 T2149;

/* SPECIAL [ET_ECF_EXTERNAL_OBJECT] */
typedef volatile struct S2150 T2150;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_OBJECT] */
typedef volatile struct S2151 T2151;

/* KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_RESOURCE] */
typedef volatile struct S2152 T2152;

/* SPECIAL [ET_ECF_EXTERNAL_RESOURCE] */
typedef volatile struct S2153 T2153;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_RESOURCE] */
typedef volatile struct S2154 T2154;

/* KL_SPECIAL_ROUTINES [RX_PCRE_REGULAR_EXPRESSION] */
typedef volatile struct S2155 T2155;

/* SPECIAL [RX_PCRE_REGULAR_EXPRESSION] */
typedef volatile struct S2156 T2156;

/* DS_ARRAYED_LIST_CURSOR [RX_PCRE_REGULAR_EXPRESSION] */
typedef volatile struct S2157 T2157;

/* KL_SPECIAL_ROUTINES [ET_ECF_ANDED_CONDITIONS] */
typedef volatile struct S2158 T2158;

/* SPECIAL [ET_ECF_ANDED_CONDITIONS] */
typedef volatile struct S2159 T2159;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_ANDED_CONDITIONS] */
typedef volatile struct S2160 T2160;

/* KL_SPECIAL_ROUTINES [ET_ECF_CONDITION_ITEM] */
typedef volatile struct S2161 T2161;

/* SPECIAL [ET_ECF_CONDITION_ITEM] */
typedef volatile struct S2162 T2162;

/* DS_ARRAYED_LIST_CURSOR [ET_ECF_CONDITION_ITEM] */
typedef volatile struct S2163 T2163;

/* LX_WILDCARD_PARSER */
typedef volatile struct S2165 T2165;

/* LX_DESCRIPTION */
typedef volatile struct S2166 T2166;

/* LX_FULL_DFA */
typedef volatile struct S2167 T2167;

/* SPECIAL [LX_WILDCARD] */
typedef volatile struct S2169 T2169;

/* KL_SPECIAL_ROUTINES [LX_WILDCARD] */
typedef volatile struct S2172 T2172;

/* DS_HASH_TABLE_CURSOR [LX_WILDCARD, STRING_8] */
typedef volatile struct S2173 T2173;

/* DS_LINKABLE [CHARACTER_8] */
typedef volatile struct S2174 T2174;

/* DS_HASH_TABLE_CURSOR [INTEGER_32, ET_LIKE_FEATURE] */
typedef volatile struct S2176 T2176;

/* SPECIAL [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8]] */
typedef volatile struct S2177 T2177;

/* KL_SPECIAL_ROUTINES [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8]] */
typedef volatile struct S2180 T2180;

/* DS_HASH_TABLE_CURSOR [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME] */
typedef volatile struct S2181 T2181;

/* DS_BILINKABLE [DS_HASH_TABLE [STRING_8, STRING_8]] */
typedef volatile struct S2182 T2182;

/* ARRAYED_LIST [INTEGER_32] */
typedef volatile struct S2183 T2183;

/* ET_UNIVERSE_ERROR */
typedef volatile struct S2184 T2184;

/* ET_DOTNET_ASSEMBLY_ERROR */
typedef volatile struct S2185 T2185;

/* AP_OPTION_COMPARATOR */
typedef volatile struct S2186 T2186;

/* DS_QUICK_SORTER [AP_OPTION] */
typedef volatile struct S2187 T2187;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
typedef volatile struct S2190 T2190;

/* ET_ACTUAL_PARAMETER_SUBLIST */
typedef volatile struct S2191 T2191;

/* TUPLE [ET_ECF_FILE_RULE] */
typedef volatile struct S2194 T2194;

/* PREDICATE [TUPLE [ET_ECF_FILE_RULE]] */
typedef volatile struct S2195 T2195;

/* PROCEDURE [TUPLE [ET_ECF_FILE_RULE]] */
typedef volatile struct S2197 T2197;

/* TUPLE [ET_ECF_FILE_RULES] */
typedef volatile struct S2198 T2198;

/* TUPLE [ET_ECF_STATE] */
typedef volatile struct S2199 T2199;

/* SPECIAL [DS_HASH_TABLE [INTEGER_32, STRING_8]] */
typedef volatile struct S2202 T2202;

/* KL_SPECIAL_ROUTINES [DS_HASH_TABLE [INTEGER_32, STRING_8]] */
typedef volatile struct S2205 T2205;

/* DS_HASH_TABLE_CURSOR [DS_HASH_TABLE [INTEGER_32, STRING_8], STRING_8] */
typedef volatile struct S2206 T2206;

/* ET_COMMENT_FINDER */
typedef volatile struct S2208 T2208;

/* DS_ARRAYED_LIST [ET_BREAK] */
typedef volatile struct S2209 T2209;

/* TUPLE [ET_BREAK] */
typedef volatile struct S2210 T2210;

/* PROCEDURE [TUPLE [ET_BREAK]] */
typedef volatile struct S2211 T2211;

/* TUPLE [ET_AST_PRETTY_PRINTER] */
typedef volatile struct S2212 T2212;

/* DS_ARRAYED_LIST [LX_RULE] */
typedef volatile struct S2213 T2213;

/* LX_START_CONDITIONS */
typedef volatile struct S2214 T2214;

/* LX_ACTION_FACTORY */
typedef volatile struct S2215 T2215;

/* DS_HASH_TABLE [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]], LX_SYMBOL_CLASS] */
typedef volatile struct S2216 T2216;

/* LX_SYMBOL_CLASS */
typedef volatile struct S2217 T2217;

/* DS_HASH_TABLE [STRING_32, STRING_8] */
typedef volatile struct S2218 T2218;

/* DS_HASH_SET [LX_SYMBOL_CLASS] */
typedef volatile struct S2219 T2219;

/* DS_HASH_TABLE [LX_SYMBOL_CLASS, STRING_32] */
typedef volatile struct S2220 T2220;

/* KL_EQUALITY_TESTER [STRING_32] */
typedef volatile struct S2221 T2221;

/* DS_ARRAYED_STACK [BOOLEAN] */
typedef volatile struct S2222 T2222;

/* KL_SPECIAL_ROUTINES [STRING_32] */
typedef volatile struct S2223 T2223;

/* SPECIAL [STRING_32] */
typedef volatile struct S2224 T2224;

/* KL_SPECIAL_ROUTINES [LX_SYMBOL_CLASS] */
typedef volatile struct S2225 T2225;

/* SPECIAL [LX_SYMBOL_CLASS] */
typedef volatile struct S2226 T2226;

/* KL_SPECIAL_ROUTINES [LX_NFA] */
typedef volatile struct S2227 T2227;

/* LX_NFA */
typedef volatile struct S2228 T2228;

/* SPECIAL [LX_NFA] */
typedef volatile struct S2229 T2229;

/* LX_RULE */
typedef volatile struct S2230 T2230;

/* DS_HASH_SET_CURSOR [LX_SYMBOL_CLASS] */
typedef volatile struct S2231 T2231;

/* LX_EQUIVALENCE_CLASSES */
typedef volatile struct S2232 T2232;

/* LX_UNRECOGNIZED_RULE_ERROR */
typedef volatile struct S2233 T2233;

/* LX_INVALID_UNICODE_CHARACTER_ERROR */
typedef volatile struct S2234 T2234;

/* LX_MISSING_QUOTE_ERROR */
typedef volatile struct S2235 T2235;

/* LX_BAD_CHARACTER_CLASS_ERROR */
typedef volatile struct S2236 T2236;

/* LX_BAD_CHARACTER_ERROR */
typedef volatile struct S2237 T2237;

/* LX_FULL_AND_META_ERROR */
typedef volatile struct S2238 T2238;

/* LX_FULL_AND_REJECT_ERROR */
typedef volatile struct S2239 T2239;

/* LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
typedef volatile struct S2240 T2240;

/* LX_CHARACTER_OUT_OF_RANGE_ERROR */
typedef volatile struct S2241 T2241;

/* ARRAY [LX_RULE] */
typedef volatile struct S2242 T2242;

/* SPECIAL [LX_RULE] */
typedef volatile struct S2243 T2243;

/* LX_DFA_STATE */
typedef volatile struct S2244 T2244;

/* DS_ARRAYED_LIST [LX_NFA_STATE] */
typedef volatile struct S2245 T2245;

/* DS_ARRAYED_LIST [LX_DFA_STATE] */
typedef volatile struct S2246 T2246;

/* LX_SYMBOL_PARTITIONS */
typedef volatile struct S2247 T2247;

/* KL_ARRAY_ROUTINES [LX_RULE] */
typedef volatile struct S2248 T2248;

/* ARRAY [detachable LX_RULE] */
typedef volatile struct S2249 T2249;

/* SPECIAL [detachable LX_RULE] */
typedef volatile struct S2250 T2250;

/* LX_START_CONDITION */
typedef volatile struct S2251 T2251;

/* LX_TRANSITION_TABLE [LX_DFA_STATE] */
typedef volatile struct S2252 T2252;

/* DS_ARRAYED_LIST [LX_NFA] */
typedef volatile struct S2253 T2253;

/* DS_HASH_TABLE [LX_NFA, INTEGER_32] */
typedef volatile struct S2254 T2254;

/* LX_NFA_STATE */
typedef volatile struct S2255 T2255;

/* TYPE [detachable RX_REGULAR_EXPRESSION] */
#define T2259 EIF_TYPE_OBJ

/* DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, STRING_8] */
typedef volatile struct S2261 T2261;

/* TUPLE [RX_PCRE_REGULAR_EXPRESSION] */
typedef volatile struct S2262 T2262;

/* PREDICATE [TUPLE [RX_PCRE_REGULAR_EXPRESSION]] */
typedef volatile struct S2263 T2263;

/* TUPLE [DS_HASH_SET [STRING_8]] */
typedef volatile struct S2266 T2266;

/* DS_LINKABLE [ET_PARENT_FEATURE] */
typedef volatile struct S2268 T2268;

/* ARRAY [detachable LX_SYMBOL_CLASS] */
typedef volatile struct S2269 T2269;

/* SPECIAL [detachable LX_SYMBOL_CLASS] */
typedef volatile struct S2270 T2270;

/* ARRAY [detachable DS_HASH_SET [INTEGER_32]] */
typedef volatile struct S2271 T2271;

/* SPECIAL [detachable DS_HASH_SET [INTEGER_32]] */
typedef volatile struct S2272 T2272;

/* DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
typedef volatile struct S2273 T2273;

/* DS_ARRAYED_LIST [LX_SYMBOL_CLASS] */
typedef volatile struct S2274 T2274;

/* LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
typedef volatile struct S2275 T2275;

/* ET_AST_NULL_LEAF */
typedef volatile struct S2276 T2276;

/* DS_ARRAYED_LIST [ET_AST_NODE] */
typedef volatile struct S2277 T2277;

/* SPECIAL [ET_BREAK] */
typedef volatile struct S2279 T2279;

/* KL_SPECIAL_ROUTINES [ET_BREAK] */
typedef volatile struct S2280 T2280;

/* DS_ARRAYED_LIST_CURSOR [ET_BREAK] */
typedef volatile struct S2281 T2281;

/* KL_SPECIAL_ROUTINES [LX_RULE] */
typedef volatile struct S2282 T2282;

/* DS_ARRAYED_LIST_CURSOR [LX_RULE] */
typedef volatile struct S2283 T2283;

/* KL_ARRAY_ROUTINES [STRING_8] */
typedef volatile struct S2284 T2284;

/* SPECIAL [LX_START_CONDITION] */
typedef volatile struct S2285 T2285;

/* KL_SPECIAL_ROUTINES [LX_START_CONDITION] */
typedef volatile struct S2286 T2286;

/* DS_ARRAYED_LIST_CURSOR [LX_START_CONDITION] */
typedef volatile struct S2287 T2287;

/* SPECIAL [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]]] */
typedef volatile struct S2288 T2288;

/* KL_EQUALITY_TESTER [LX_SYMBOL_CLASS] */
typedef volatile struct S2289 T2289;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]]] */
typedef volatile struct S2291 T2291;

/* DS_HASH_TABLE_CURSOR [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]], LX_SYMBOL_CLASS] */
typedef volatile struct S2292 T2292;

/* SPECIAL [detachable SPECIAL [NATURAL_64]] */
typedef volatile struct S2293 T2293;

/* DS_HASH_TABLE_CURSOR [STRING_32, STRING_8] */
typedef volatile struct S2295 T2295;

/* DS_HASH_TABLE_CURSOR [LX_SYMBOL_CLASS, STRING_32] */
typedef volatile struct S2297 T2297;

/* LX_EPSILON_TRANSITION [LX_NFA_STATE] */
typedef volatile struct S2298 T2298;

/* LX_SYMBOL_CLASS_TRANSITION [LX_NFA_STATE] */
typedef volatile struct S2301 T2301;

/* LX_SYMBOL_TRANSITION [LX_NFA_STATE] */
typedef volatile struct S2302 T2302;

/* LX_ACTION */
typedef volatile struct S2304 T2304;

/* ARRAY [DS_BILINKABLE [INTEGER_32]] */
typedef volatile struct S2305 T2305;

/* DS_BILINKABLE [INTEGER_32] */
typedef volatile struct S2306 T2306;

/* SPECIAL [DS_BILINKABLE [INTEGER_32]] */
typedef volatile struct S2307 T2307;

/* DS_BUBBLE_SORTER [LX_NFA_STATE] */
typedef volatile struct S2308 T2308;

/* DS_BUBBLE_SORTER [LX_RULE] */
typedef volatile struct S2310 T2310;

/* SPECIAL [LX_NFA_STATE] */
typedef volatile struct S2312 T2312;

/* KL_SPECIAL_ROUTINES [LX_NFA_STATE] */
typedef volatile struct S2314 T2314;

/* DS_ARRAYED_LIST_CURSOR [LX_NFA_STATE] */
typedef volatile struct S2315 T2315;

/* SPECIAL [LX_DFA_STATE] */
typedef volatile struct S2317 T2317;

/* KL_SPECIAL_ROUTINES [LX_DFA_STATE] */
typedef volatile struct S2318 T2318;

/* DS_ARRAYED_LIST_CURSOR [LX_DFA_STATE] */
typedef volatile struct S2319 T2319;

/* KL_ARRAY [LX_RULE] */
typedef volatile struct S2320 T2320;

/* DS_HASH_TABLE [LX_DFA_STATE, INTEGER_32] */
typedef volatile struct S2321 T2321;

/* DS_ARRAYED_LIST_CURSOR [LX_NFA] */
typedef volatile struct S2322 T2322;

/* DS_HASH_TABLE_CURSOR [LX_NFA, INTEGER_32] */
typedef volatile struct S2324 T2324;

/* TUPLE [ET_LIBRARY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S2325 T2325;

/* TUPLE [ET_DOTNET_ASSEMBLY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S2326 T2326;

/* TUPLE [ET_ECF_LIBRARY] */
typedef volatile struct S2327 T2327;

/* TUPLE [ET_ECF_DOTNET_ASSEMBLY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
typedef volatile struct S2329 T2329;

/* KL_COMPARABLE_COMPARATOR [LX_RULE] */
typedef volatile struct S2334 T2334;

/* KL_COMPARABLE_COMPARATOR [LX_NFA_STATE] */
typedef volatile struct S2337 T2337;

/* SPECIAL [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
typedef volatile struct S2340 T2340;

/* KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
typedef volatile struct S2341 T2341;

/* DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
typedef volatile struct S2342 T2342;

/* DS_ARRAYED_LIST_CURSOR [LX_SYMBOL_CLASS] */
typedef volatile struct S2343 T2343;

/* SPECIAL [ET_AST_NODE] */
typedef volatile struct S2345 T2345;

/* KL_SPECIAL_ROUTINES [ET_AST_NODE] */
typedef volatile struct S2346 T2346;

/* DS_ARRAYED_LIST_CURSOR [ET_AST_NODE] */
typedef volatile struct S2347 T2347;

/* KL_ARRAY [STRING_8] */
typedef volatile struct S2348 T2348;

/* DS_HASH_TABLE_CURSOR [LX_DFA_STATE, INTEGER_32] */
typedef volatile struct S2350 T2350;

/* Struct for boxed version of type BOOLEAN */
struct Sb1 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile* volatile p1; /* pointer */
	T1 volatile z1; /* item */
};

/* Struct for boxed version of type CHARACTER_8 */
struct Sb2 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile* volatile p1; /* pointer */
	T2 volatile z1; /* item */
};

/* Struct for boxed version of type CHARACTER_32 */
struct Sb3 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T3 volatile* volatile p1; /* pointer */
	T3 volatile z1; /* item */
};

/* Struct for boxed version of type INTEGER_8 */
struct Sb4 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T4 volatile* volatile p1; /* pointer */
	T4 volatile z1; /* item */
};

/* Struct for boxed version of type INTEGER_16 */
struct Sb5 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T5 volatile* volatile p1; /* pointer */
	T5 volatile z1; /* item */
};

/* Struct for boxed version of type INTEGER_32 */
struct Sb6 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile* volatile p1; /* pointer */
	T6 volatile z1; /* item */
};

/* Struct for boxed version of type INTEGER_64 */
struct Sb7 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T7 volatile* volatile p1; /* pointer */
	T7 volatile z1; /* item */
};

/* Struct for boxed version of type NATURAL_8 */
struct Sb8 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T8 volatile* volatile p1; /* pointer */
	T8 volatile z1; /* item */
};

/* Struct for boxed version of type NATURAL_16 */
struct Sb9 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T9 volatile* volatile p1; /* pointer */
	T9 volatile z1; /* item */
};

/* Struct for boxed version of type NATURAL_32 */
struct Sb10 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T10 volatile* volatile p1; /* pointer */
	T10 volatile z1; /* item */
};

/* Struct for boxed version of type NATURAL_64 */
struct Sb11 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T11 volatile* volatile p1; /* pointer */
	T11 volatile z1; /* item */
};

/* Struct for boxed version of type REAL_32 */
struct Sb12 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T12 volatile* volatile p1; /* pointer */
	T12 volatile z1; /* item */
};

/* Struct for boxed version of type REAL_64 */
struct Sb13 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T13 volatile* volatile p1; /* pointer */
	T13 volatile z1; /* item */
};

/* Struct for boxed version of type POINTER */
struct Sb14 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile* volatile p1; /* pointer */
	T14 volatile z1; /* item */
};

/* Struct for type UTF_CONVERTER */
struct S40 {
	char volatile dummy;
};

/* Struct for boxed version of type UTF_CONVERTER */
struct Sb40 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T40* volatile p1; /* pointer */
	T40 z1; /* item */
};

/* Struct for type TYPED_POINTER [NATURAL_16] */
struct S1567 {
	T14 volatile a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [NATURAL_16] */
struct Sb1567 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1567* volatile p1; /* pointer */
	T1567 z1; /* item */
};

/* Struct for type TYPED_POINTER [NATURAL_8] */
struct S1566 {
	T14 volatile a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [NATURAL_8] */
struct Sb1566 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1566* volatile p1; /* pointer */
	T1566 z1; /* item */
};

/* Struct for type TYPED_POINTER [ANY] */
struct S1447 {
	T14 volatile a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [ANY] */
struct Sb1447 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1447* volatile p1; /* pointer */
	T1447 z1; /* item */
};

/* Struct for type TYPED_POINTER [SPECIAL [NATURAL_8]] */
struct S1076 {
	T14 volatile a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [SPECIAL [NATURAL_8]] */
struct Sb1076 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1076* volatile p1; /* pointer */
	T1076 z1; /* item */
};

/* Struct for type SPECIAL [CHARACTER_8] */
struct S15 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T2 volatile z2[]; /* item */
};

/* Struct for type SPECIAL [CHARACTER_32] */
struct S16 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T3 volatile z2[]; /* item */
};

/* Struct for type STRING_8 */
struct S17 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* count */
	T6 volatile a3; /* internal_hash_code */
	T6 volatile a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type STRING_32 */
struct S18 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* count */
	T6 volatile a3; /* internal_hash_code */
	T6 volatile a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type IMMUTABLE_STRING_32 */
struct S20 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* count */
	T6 volatile a3; /* area_lower */
	T6 volatile a4; /* internal_hash_code */
};

/* Struct for type ISE_EXCEPTION_MANAGER */
struct S21 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type GEC */
struct S26 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* exit_code */
	T0* volatile a2; /* error_handler */
	T0* volatile a3; /* version_flag */
	T0* volatile a4; /* input_filename */
	T0* volatile a5; /* override_root_type */
	T0* volatile a6; /* last_system */
	T0* volatile a7; /* target_option */
	T0* volatile a8; /* setting_option */
	T0* volatile a9; /* capability_option */
	T0* volatile a10; /* variable_option */
	T0* volatile a11; /* finalize_flag */
	T0* volatile a12; /* gelint_flag */
	T0* volatile a13; /* ise_option */
	T0* volatile a14; /* catcall_option */
	T0* volatile a15; /* c_compile_option */
	T0* volatile a16; /* split_option */
	T0* volatile a17; /* split_size_option */
	T0* volatile a18; /* new_instance_types_option */
	T0* volatile a19; /* gc_option */
	T0* volatile a20; /* thread_option */
	T0* volatile a21; /* silent_flag */
	T0* volatile a22; /* no_benchmark_flag */
	T0* volatile a23; /* nested_benchmark_flag */
	T0* volatile a24; /* metrics_flag */
	T0* volatile a25; /* verbose_flag */
	T0* volatile a26; /* ise_version */
	T0* volatile a27; /* override_root_cluster_pathname */
	T0* volatile a28; /* override_root_creation */
	T0* volatile a29; /* override_settings */
	T0* volatile a30; /* override_capabilities */
	T0* volatile a31; /* override_variables */
	T1 volatile a32; /* ecf_gelint_option */
	T0* volatile a33; /* ecf_scoop_exceptions_option */
	T0* volatile a34; /* new_instance_types */
	T6 volatile a35; /* split_size */
};

/* Struct for type CELL [detachable EXCEPTION] */
struct S27 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type HASH_TABLE [INTEGER_32, INTEGER_32] */
struct S28 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T0* volatile a2; /* content */
	T0* volatile a3; /* keys */
	T0* volatile a4; /* deleted_marks */
	T0* volatile a5; /* indexes_map */
	T6 volatile a6; /* iteration_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* deleted_item_position */
	T6 volatile a9; /* control */
	T6 volatile a10; /* found_item */
	T1 volatile a11; /* has_default */
	T6 volatile a12; /* item_position */
	T6 volatile a13; /* ht_lowest_deleted_position */
	T6 volatile a14; /* ht_deleted_item */
	T6 volatile a15; /* ht_deleted_key */
	T1 volatile a16; /* object_comparison */
};

/* Struct for type CELL [detachable TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN]] */
struct S29 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type CELL [NO_MORE_MEMORY] */
struct S30 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type C_STRING */
struct S31 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* managed_data */
	T6 volatile a2; /* count */
};

/* Struct for type TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN] */
struct S32 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile z1;
	T6 volatile z2;
	T6 volatile z3;
	T0* volatile z4;
	T0* volatile z5;
	T0* volatile z6;
	T0* volatile z7;
	T0* volatile z8;
	T0* volatile z9;
	T6 volatile z10;
	T1 volatile z11;
};

/* Struct for type KL_ARGUMENTS */
struct S33 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* program_name */
};

/* Struct for type ARRAY [STRING_8] */
struct S34 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [STRING_8] */
struct S35 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_EXCEPTIONS */
struct S36 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type EXCEPTIONS */
struct S37 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_STANDARD_FILES */
struct S38 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_STDERR_FILE */
struct S39 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ERROR_HANDLER */
struct S42 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_error */
	T0* volatile a2; /* info_file */
	T0* volatile a3; /* mutex */
	T0* volatile a4; /* error_file */
	T0* volatile a5; /* warning_file */
	T1 volatile a6; /* is_ise */
	T1 volatile a7; /* is_verbose */
	T1 volatile a8; /* has_eiffel_error */
	T1 volatile a9; /* has_internal_error */
	T1 volatile a10; /* is_ge */
};

/* Struct for type KL_TEXT_INPUT_FILE */
struct S43 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T1 volatile a2; /* end_of_file */
	T0* volatile a3; /* last_string */
	T6 volatile a4; /* mode */
	T0* volatile a5; /* internal_name */
	T0* volatile a6; /* internal_detachable_name_pointer */
	T0* volatile a7; /* character_buffer */
	T2 volatile a8; /* last_character */
	T14 volatile a9; /* file_pointer */
	T1 volatile a10; /* descriptor_available */
	T0* volatile a11; /* last_string_32 */
};

/* Struct for type UT_GOBO_VARIABLES */
struct S44 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type UT_ISE_VARIABLES */
struct S45 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type AP_FLAG */
struct S46 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* occurrences */
	T0* volatile a2; /* description */
	T0* volatile a3; /* long_form */
	T2 volatile a4; /* short_form */
	T1 volatile a5; /* has_short_form */
	T1 volatile a6; /* is_hidden */
	T1 volatile a7; /* is_mandatory */
	T6 volatile a8; /* maximum_occurrences */
};

/* Struct for type KL_EXECUTION_ENVIRONMENT */
struct S49 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SYSTEM */
struct S50 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* register_class_mutex */
	T0* volatile a2; /* external_include_pathnames */
	T0* volatile a3; /* external_object_pathnames */
	T0* volatile a4; /* external_library_pathnames */
	T0* volatile a5; /* external_make_pathnames */
	T0* volatile a6; /* external_resource_pathnames */
	T0* volatile a7; /* external_cflags */
	T0* volatile a8; /* external_linker_flags */
	T1 volatile a9; /* console_application_mode */
	T1 volatile a10; /* total_order_on_reals_mode */
	T0* volatile a11; /* current_system */
	T0* volatile a12; /* universe */
	T0* volatile a13; /* name */
	T0* volatile a14; /* libraries */
	T0* volatile a15; /* clusters */
	T0* volatile a16; /* dotnet_assemblies */
	T0* volatile a17; /* implicit_attachment_type_mark */
	T0* volatile a18; /* master_classes */
	T0* volatile a19; /* master_class_mutex */
	T0* volatile a20; /* system_name */
	T0* volatile a21; /* any_type */
	T0* volatile a22; /* detachable_any_type */
	T0* volatile a23; /* detachable_separate_any_type */
	T0* volatile a24; /* any_parent */
	T0* volatile a25; /* any_parents */
	T0* volatile a26; /* any_clients */
	T0* volatile a27; /* tuple_type */
	T0* volatile a28; /* detachable_tuple_type */
	T0* volatile a29; /* tuple_identity_type */
	T0* volatile a30; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a31; /* array_any_type */
	T0* volatile a32; /* array_detachable_any_type */
	T0* volatile a33; /* array_none_type */
	T0* volatile a34; /* array_identity_type */
	T0* volatile a35; /* boolean_type */
	T0* volatile a36; /* character_type */
	T0* volatile a37; /* character_8_type */
	T0* volatile a38; /* character_8_convert_feature */
	T0* volatile a39; /* character_32_type */
	T0* volatile a40; /* character_32_convert_feature */
	T0* volatile a41; /* double_type */
	T0* volatile a42; /* exception_type */
	T0* volatile a43; /* detachable_exception_type */
	T0* volatile a44; /* exception_manager_type */
	T0* volatile a45; /* function_identity_any_type */
	T0* volatile a46; /* immutable_string_8_type */
	T0* volatile a47; /* immutable_string_32_type */
	T0* volatile a48; /* integer_type */
	T0* volatile a49; /* integer_8_type */
	T0* volatile a50; /* integer_8_convert_feature */
	T0* volatile a51; /* integer_16_type */
	T0* volatile a52; /* integer_16_convert_feature */
	T0* volatile a53; /* integer_32_type */
	T0* volatile a54; /* integer_32_convert_feature */
	T0* volatile a55; /* integer_64_type */
	T0* volatile a56; /* integer_64_convert_feature */
	T0* volatile a57; /* ise_exception_manager_type */
	T0* volatile a58; /* iterable_detachable_separate_any_type */
	T0* volatile a59; /* natural_type */
	T0* volatile a60; /* natural_8_type */
	T0* volatile a61; /* natural_8_convert_feature */
	T0* volatile a62; /* natural_16_type */
	T0* volatile a63; /* natural_16_convert_feature */
	T0* volatile a64; /* natural_32_type */
	T0* volatile a65; /* natural_32_convert_feature */
	T0* volatile a66; /* natural_64_type */
	T0* volatile a67; /* natural_64_convert_feature */
	T0* volatile a68; /* none_type */
	T0* volatile a69; /* detachable_none_type */
	T0* volatile a70; /* pointer_type */
	T0* volatile a71; /* predicate_identity_type */
	T0* volatile a72; /* procedure_identity_type */
	T0* volatile a73; /* real_type */
	T0* volatile a74; /* real_32_type */
	T0* volatile a75; /* real_32_convert_feature */
	T0* volatile a76; /* real_64_type */
	T0* volatile a77; /* real_64_convert_feature */
	T0* volatile a78; /* routine_identity_type */
	T0* volatile a79; /* special_any_type */
	T0* volatile a80; /* special_detachable_any_type */
	T0* volatile a81; /* special_identity_type */
	T0* volatile a82; /* string_type */
	T0* volatile a83; /* detachable_string_type */
	T0* volatile a84; /* string_8_type */
	T0* volatile a85; /* detachable_string_8_type */
	T0* volatile a86; /* string_8_convert_feature */
	T0* volatile a87; /* string_32_type */
	T0* volatile a88; /* string_32_convert_feature */
	T0* volatile a89; /* system_object_type */
	T0* volatile a90; /* system_object_parents */
	T0* volatile a91; /* system_string_type */
	T0* volatile a92; /* type_detachable_any_type */
	T0* volatile a93; /* detachable_type_exception_type */
	T0* volatile a94; /* type_detachable_exception_type */
	T0* volatile a95; /* type_detachable_like_current_type */
	T0* volatile a96; /* type_identity_type */
	T0* volatile a97; /* typed_pointer_identity_type */
	T0* volatile a98; /* wide_character_type */
	T1 volatile a99; /* is_read_only */
	T6 volatile a100; /* registered_class_count */
	T1 volatile a101; /* use_obsolete_syntax_mode */
	T1 volatile a102; /* obsolete_iteration_mode */
	T1 volatile a103; /* attachment_type_conformance_mode */
	T1 volatile a104; /* scoop_mode */
	T6 volatile a105; /* is_equal_seed */
	T1 volatile a106; /* target_type_attachment_mode */
	T6 volatile a107; /* default_create_seed */
	T6 volatile a108; /* iterable_new_cursor_seed */
	T6 volatile a109; /* iteration_cursor_after_seed */
	T6 volatile a110; /* iteration_cursor_forth_seed */
	T6 volatile a111; /* iteration_cursor_item_seed */
	T1 volatile a112; /* is_dotnet */
	T6 volatile a113; /* copy_seed */
	T6 volatile a114; /* twin_seed */
	T6 volatile a115; /* routine_call_seed */
	T6 volatile a116; /* function_item_seed */
	T6 volatile a117; /* dispose_seed */
	T6 volatile a118; /* special_item_seed */
	T1 volatile a119; /* obsolete_routine_type_mode */
};

/* Struct for type AP_PARSER */
struct S52 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* options */
	T0* volatile a2; /* alternative_options_lists */
	T0* volatile a3; /* parameters */
	T0* volatile a4; /* parameters_description */
	T0* volatile a5; /* help_option */
	T0* volatile a6; /* error_handler */
	T0* volatile a7; /* application_description */
	T0* volatile a8; /* current_options */
	T1 volatile a9; /* is_first_option */
	T0* volatile a10; /* last_option_parameter */
};

/* Struct for type AP_ALTERNATIVE_OPTIONS_LIST */
struct S53 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* introduction_option */
	T0* volatile a5; /* parameters_description */
	T0* volatile a6; /* internal_cursor */
};

/* Struct for type AP_STRING_OPTION */
struct S54 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
	T0* volatile a2; /* default_parameter */
	T0* volatile a3; /* description */
	T0* volatile a4; /* parameter_description */
	T1 volatile a5; /* has_default_parameter */
	T1 volatile a6; /* needs_parameter */
	T0* volatile a7; /* long_form */
	T1 volatile a8; /* is_hidden */
	T1 volatile a9; /* is_mandatory */
	T6 volatile a10; /* maximum_occurrences */
	T1 volatile a11; /* has_short_form */
	T2 volatile a12; /* short_form */
};

/* Struct for type UT_VERSION */
struct S56 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* internal_major */
	T6 volatile a2; /* internal_minor */
	T6 volatile a3; /* internal_revision */
	T6 volatile a4; /* internal_build */
};

/* Struct for type AP_ENUMERATION_OPTION */
struct S57 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
	T0* volatile a2; /* default_parameter */
	T0* volatile a3; /* description */
	T0* volatile a4; /* possible_values */
	T0* volatile a5; /* parameter_description */
	T1 volatile a6; /* needs_parameter */
	T0* volatile a7; /* long_form */
	T1 volatile a8; /* is_hidden */
	T1 volatile a9; /* is_mandatory */
	T6 volatile a10; /* maximum_occurrences */
	T1 volatile a11; /* has_short_form */
	T2 volatile a12; /* short_form */
};

/* Struct for type AP_BOOLEAN_OPTION */
struct S58 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
	T1 volatile a2; /* default_parameter */
	T0* volatile a3; /* description */
	T0* volatile a4; /* parameter_description */
	T1 volatile a5; /* needs_parameter */
	T0* volatile a6; /* long_form */
	T1 volatile a7; /* is_hidden */
	T1 volatile a8; /* is_mandatory */
	T6 volatile a9; /* maximum_occurrences */
	T1 volatile a10; /* has_short_form */
	T2 volatile a11; /* short_form */
};

/* Struct for type AP_INTEGER_OPTION */
struct S59 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
	T6 volatile a2; /* default_parameter */
	T0* volatile a3; /* description */
	T0* volatile a4; /* parameter_description */
	T1 volatile a5; /* needs_parameter */
	T0* volatile a6; /* long_form */
	T2 volatile a7; /* short_form */
	T1 volatile a8; /* is_hidden */
	T1 volatile a9; /* is_mandatory */
	T6 volatile a10; /* maximum_occurrences */
	T1 volatile a11; /* has_short_form */
};

/* Struct for type ET_NULL_ERROR_HANDLER */
struct S63 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_error */
	T0* volatile a2; /* info_file */
	T0* volatile a3; /* mutex */
	T0* volatile a4; /* error_file */
	T0* volatile a5; /* warning_file */
	T1 volatile a6; /* is_ise */
	T1 volatile a7; /* is_verbose */
	T1 volatile a8; /* has_eiffel_error */
	T1 volatile a9; /* has_internal_error */
	T1 volatile a10; /* is_ge */
};

/* Struct for type ET_AST_FACTORY */
struct S65 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SYSTEM_PROCESSOR */
struct S66 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* eiffel_preparser */
	T0* volatile a2; /* master_class_checker */
	T0* volatile a3; /* eiffel_parser */
	T0* volatile a4; /* provider_checker */
	T0* volatile a5; /* ancestor_builder */
	T0* volatile a6; /* feature_flattener */
	T0* volatile a7; /* interface_checker */
	T0* volatile a8; /* implementation_checker */
	T0* volatile a9; /* error_handler */
	T0* volatile a10; /* processed_class_count_stack */
	T0* volatile a11; /* ast_factory */
	T0* volatile a12; /* dotnet_assembly_consumer */
	T0* volatile a13; /* ise_version */
	T0* volatile a14; /* ecma_version */
	T1 volatile a15; /* preparse_shallow_mode */
	T1 volatile a16; /* preparse_single_mode */
	T1 volatile a17; /* preparse_multiple_mode */
	T1 volatile a18; /* unknown_builtin_reported */
	T1 volatile a19; /* qualified_anchored_types_cycle_detection_enabled */
	T1 volatile a20; /* use_attribute_keyword */
	T1 volatile a21; /* use_note_keyword */
	T1 volatile a22; /* use_reference_keyword */
	T1 volatile a23; /* use_attached_keyword */
	T1 volatile a24; /* use_detachable_keyword */
	T1 volatile a25; /* benchmark_shown */
	T1 volatile a26; /* nested_benchmark_shown */
	T0* volatile a27; /* stop_request */
	T1 volatile a28; /* metrics_shown */
	T1 volatile a29; /* flat_mode */
	T1 volatile a30; /* flat_dbc_mode */
	T1 volatile a31; /* cluster_dependence_enabled */
	T6 volatile a32; /* postponed_class_count */
	T6 volatile a33; /* processed_class_count */
	T1 volatile a34; /* providers_enabled */
	T1 volatile a35; /* preparse_override_mode */
	T1 volatile a36; /* preparse_readonly_mode */
	T1 volatile a37; /* use_cluster_dependence_pathnames */
	T1 volatile a38; /* suppliers_enabled */
};

/* Struct for type ET_CLUSTER */
struct S67 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* pathname */
	T1 volatile a3; /* is_relative */
	T0* volatile a4; /* universe */
	T1 volatile a5; /* overridden_constraint_enabled */
	T1 volatile a6; /* scm_mapping_constraint_enabled */
	T0* volatile a7; /* subclusters */
	T1 volatile a8; /* use_obsolete_syntax */
	T1 volatile a9; /* is_override */
	T0* volatile a10; /* provider_constraint */
	T0* volatile a11; /* dependant_constraint */
	T0* volatile a12; /* scm_write_mapping */
	T0* volatile a13; /* parent */
	T1 volatile a14; /* is_read_only */
	T1 volatile a15; /* is_abstract */
	T0* volatile a16; /* cached_absolute_pathname */
};

/* Struct for type ET_EIFFEL_PARSER */
struct S68 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* yy_parsing_status */
	T0* volatile a2; /* eiffel_buffer */
	T0* volatile a3; /* counters */
	T0* volatile a4; /* last_formal_arguments_stack */
	T0* volatile a5; /* last_local_variables_stack */
	T0* volatile a6; /* last_keywords */
	T0* volatile a7; /* last_symbols */
	T0* volatile a8; /* last_object_tests_stack */
	T0* volatile a9; /* last_object_tests_pool */
	T0* volatile a10; /* last_iteration_components_stack */
	T0* volatile a11; /* last_iteration_components_pool */
	T0* volatile a12; /* last_inline_separate_arguments_stack */
	T0* volatile a13; /* last_inline_separate_arguments_pool */
	T0* volatile a14; /* assertions */
	T0* volatile a15; /* assertion_counters */
	T0* volatile a16; /* assertion_kinds */
	T6 volatile a17; /* assertion_kind */
	T0* volatile a18; /* queries */
	T0* volatile a19; /* procedures */
	T0* volatile a20; /* constraints */
	T0* volatile a21; /* providers */
	T0* volatile a22; /* current_class */
	T0* volatile a23; /* system_processor */
	T0* volatile a24; /* group */
	T0* volatile a25; /* filename */
	T6 volatile a26; /* time_stamp */
	T0* volatile a27; /* input_buffer */
	T0* volatile a28; /* last_class */
	T6 volatile a29; /* last_text_count */
	T6 volatile a30; /* last_literal_start */
	T0* volatile a31; /* verbatim_marker */
	T6 volatile a32; /* verbatim_marker_count */
	T0* volatile a33; /* verbatim_open_white_characters */
	T0* volatile a34; /* verbatim_close_white_characters */
	T0* volatile a35; /* yyss */
	T6 volatile a36; /* yy_end */
	T6 volatile a37; /* yy_start */
	T6 volatile a38; /* yy_line */
	T6 volatile a39; /* yy_column */
	T6 volatile a40; /* yy_position */
	T6 volatile a41; /* yy_suspended_yystacksize */
	T6 volatile a42; /* yy_suspended_yystate */
	T6 volatile a43; /* yy_suspended_yyn */
	T6 volatile a44; /* yy_suspended_yychar1 */
	T6 volatile a45; /* yy_suspended_index */
	T6 volatile a46; /* yy_suspended_yyss_top */
	T6 volatile a47; /* yy_suspended_yy_goto */
	T0* volatile a48; /* yyr1 */
	T6 volatile a49; /* yyssp */
	T0* volatile a50; /* yypgoto */
	T0* volatile a51; /* yycheck */
	T0* volatile a52; /* yytable */
	T0* volatile a53; /* yydefgoto */
	T6 volatile a54; /* error_count */
	T1 volatile a55; /* yy_lookahead_needed */
	T6 volatile a56; /* yyerrstatus */
	T0* volatile a57; /* yypact */
	T6 volatile a58; /* last_token */
	T0* volatile a59; /* yytranslate */
	T0* volatile a60; /* yydefact */
	T0* volatile a61; /* last_clients */
	T0* volatile a62; /* last_export_clients */
	T0* volatile a63; /* last_feature_clause */
	T6 volatile a64; /* last_literal_end */
	T1 volatile a65; /* yy_rejected */
	T6 volatile a66; /* yy_state_count */
	T6 volatile a67; /* yy_full_match */
	T6 volatile a68; /* yy_lp */
	T6 volatile a69; /* yy_looking_for_trail_begin */
	T6 volatile a70; /* yy_full_lp */
	T6 volatile a71; /* yy_full_state */
	T6 volatile a72; /* pushed_start_condition_count */
	T6 volatile a73; /* yy_start_state */
	T6 volatile a74; /* line */
	T6 volatile a75; /* column */
	T6 volatile a76; /* position */
	T1 volatile a77; /* yy_more_flag */
	T6 volatile a78; /* yy_more_len */
	T6 volatile a79; /* yy_last_accepting_state */
	T6 volatile a80; /* yy_last_accepting_cpos */
	T0* volatile a81; /* yy_pushed_start_conditions */
	T0* volatile a82; /* yyspecial_routines1 */
	T6 volatile a83; /* yyvsc1 */
	T0* volatile a84; /* yyvs1 */
	T0* volatile a85; /* yyspecial_routines2 */
	T6 volatile a86; /* yyvsc2 */
	T0* volatile a87; /* yyvs2 */
	T0* volatile a88; /* yyspecial_routines3 */
	T6 volatile a89; /* yyvsc3 */
	T0* volatile a90; /* yyvs3 */
	T0* volatile a91; /* yyspecial_routines4 */
	T6 volatile a92; /* yyvsc4 */
	T0* volatile a93; /* yyvs4 */
	T0* volatile a94; /* yyspecial_routines5 */
	T6 volatile a95; /* yyvsc5 */
	T0* volatile a96; /* yyvs5 */
	T0* volatile a97; /* yyspecial_routines6 */
	T6 volatile a98; /* yyvsc6 */
	T0* volatile a99; /* yyvs6 */
	T0* volatile a100; /* yyspecial_routines7 */
	T6 volatile a101; /* yyvsc7 */
	T0* volatile a102; /* yyvs7 */
	T0* volatile a103; /* yyspecial_routines8 */
	T6 volatile a104; /* yyvsc8 */
	T0* volatile a105; /* yyvs8 */
	T0* volatile a106; /* yyspecial_routines9 */
	T6 volatile a107; /* yyvsc9 */
	T0* volatile a108; /* yyvs9 */
	T0* volatile a109; /* yyspecial_routines10 */
	T6 volatile a110; /* yyvsc10 */
	T0* volatile a111; /* yyvs10 */
	T0* volatile a112; /* yyspecial_routines11 */
	T6 volatile a113; /* yyvsc11 */
	T0* volatile a114; /* yyvs11 */
	T0* volatile a115; /* yyspecial_routines12 */
	T6 volatile a116; /* yyvsc12 */
	T0* volatile a117; /* yyvs12 */
	T0* volatile a118; /* yyspecial_routines13 */
	T6 volatile a119; /* yyvsc13 */
	T0* volatile a120; /* yyvs13 */
	T0* volatile a121; /* yyspecial_routines14 */
	T6 volatile a122; /* yyvsc14 */
	T0* volatile a123; /* yyvs14 */
	T0* volatile a124; /* yyspecial_routines15 */
	T6 volatile a125; /* yyvsc15 */
	T0* volatile a126; /* yyvs15 */
	T0* volatile a127; /* yyspecial_routines16 */
	T6 volatile a128; /* yyvsc16 */
	T0* volatile a129; /* yyvs16 */
	T0* volatile a130; /* yyspecial_routines17 */
	T6 volatile a131; /* yyvsc17 */
	T0* volatile a132; /* yyvs17 */
	T0* volatile a133; /* yyspecial_routines18 */
	T6 volatile a134; /* yyvsc18 */
	T0* volatile a135; /* yyvs18 */
	T0* volatile a136; /* yyspecial_routines19 */
	T6 volatile a137; /* yyvsc19 */
	T0* volatile a138; /* yyvs19 */
	T0* volatile a139; /* yyspecial_routines20 */
	T6 volatile a140; /* yyvsc20 */
	T0* volatile a141; /* yyvs20 */
	T0* volatile a142; /* yyspecial_routines21 */
	T6 volatile a143; /* yyvsc21 */
	T0* volatile a144; /* yyvs21 */
	T0* volatile a145; /* yyspecial_routines22 */
	T6 volatile a146; /* yyvsc22 */
	T0* volatile a147; /* yyvs22 */
	T0* volatile a148; /* yyspecial_routines23 */
	T6 volatile a149; /* yyvsc23 */
	T0* volatile a150; /* yyvs23 */
	T0* volatile a151; /* yyspecial_routines24 */
	T6 volatile a152; /* yyvsc24 */
	T0* volatile a153; /* yyvs24 */
	T0* volatile a154; /* yyspecial_routines25 */
	T6 volatile a155; /* yyvsc25 */
	T0* volatile a156; /* yyvs25 */
	T0* volatile a157; /* yyspecial_routines26 */
	T6 volatile a158; /* yyvsc26 */
	T0* volatile a159; /* yyvs26 */
	T0* volatile a160; /* yyspecial_routines27 */
	T6 volatile a161; /* yyvsc27 */
	T0* volatile a162; /* yyvs27 */
	T0* volatile a163; /* yyspecial_routines28 */
	T6 volatile a164; /* yyvsc28 */
	T0* volatile a165; /* yyvs28 */
	T0* volatile a166; /* yyspecial_routines29 */
	T6 volatile a167; /* yyvsc29 */
	T0* volatile a168; /* yyvs29 */
	T0* volatile a169; /* yyspecial_routines30 */
	T6 volatile a170; /* yyvsc30 */
	T0* volatile a171; /* yyvs30 */
	T0* volatile a172; /* yyspecial_routines31 */
	T6 volatile a173; /* yyvsc31 */
	T0* volatile a174; /* yyvs31 */
	T0* volatile a175; /* yyspecial_routines32 */
	T6 volatile a176; /* yyvsc32 */
	T0* volatile a177; /* yyvs32 */
	T0* volatile a178; /* yyspecial_routines33 */
	T6 volatile a179; /* yyvsc33 */
	T0* volatile a180; /* yyvs33 */
	T0* volatile a181; /* yyspecial_routines34 */
	T6 volatile a182; /* yyvsc34 */
	T0* volatile a183; /* yyvs34 */
	T0* volatile a184; /* yyspecial_routines35 */
	T6 volatile a185; /* yyvsc35 */
	T0* volatile a186; /* yyvs35 */
	T0* volatile a187; /* yyspecial_routines36 */
	T6 volatile a188; /* yyvsc36 */
	T0* volatile a189; /* yyvs36 */
	T0* volatile a190; /* yyspecial_routines37 */
	T6 volatile a191; /* yyvsc37 */
	T0* volatile a192; /* yyvs37 */
	T0* volatile a193; /* yyspecial_routines38 */
	T6 volatile a194; /* yyvsc38 */
	T0* volatile a195; /* yyvs38 */
	T0* volatile a196; /* yyspecial_routines39 */
	T6 volatile a197; /* yyvsc39 */
	T0* volatile a198; /* yyvs39 */
	T0* volatile a199; /* yyspecial_routines40 */
	T6 volatile a200; /* yyvsc40 */
	T0* volatile a201; /* yyvs40 */
	T0* volatile a202; /* yyspecial_routines41 */
	T6 volatile a203; /* yyvsc41 */
	T0* volatile a204; /* yyvs41 */
	T0* volatile a205; /* yyspecial_routines42 */
	T6 volatile a206; /* yyvsc42 */
	T0* volatile a207; /* yyvs42 */
	T0* volatile a208; /* yyspecial_routines43 */
	T6 volatile a209; /* yyvsc43 */
	T0* volatile a210; /* yyvs43 */
	T0* volatile a211; /* yyspecial_routines44 */
	T6 volatile a212; /* yyvsc44 */
	T0* volatile a213; /* yyvs44 */
	T0* volatile a214; /* yyspecial_routines45 */
	T6 volatile a215; /* yyvsc45 */
	T0* volatile a216; /* yyvs45 */
	T0* volatile a217; /* yyspecial_routines46 */
	T6 volatile a218; /* yyvsc46 */
	T0* volatile a219; /* yyvs46 */
	T0* volatile a220; /* yyspecial_routines47 */
	T6 volatile a221; /* yyvsc47 */
	T0* volatile a222; /* yyvs47 */
	T0* volatile a223; /* yyspecial_routines48 */
	T6 volatile a224; /* yyvsc48 */
	T0* volatile a225; /* yyvs48 */
	T0* volatile a226; /* yyspecial_routines49 */
	T6 volatile a227; /* yyvsc49 */
	T0* volatile a228; /* yyvs49 */
	T0* volatile a229; /* yyspecial_routines50 */
	T6 volatile a230; /* yyvsc50 */
	T0* volatile a231; /* yyvs50 */
	T0* volatile a232; /* yyspecial_routines51 */
	T6 volatile a233; /* yyvsc51 */
	T0* volatile a234; /* yyvs51 */
	T0* volatile a235; /* yyspecial_routines52 */
	T6 volatile a236; /* yyvsc52 */
	T0* volatile a237; /* yyvs52 */
	T0* volatile a238; /* yyspecial_routines53 */
	T6 volatile a239; /* yyvsc53 */
	T0* volatile a240; /* yyvs53 */
	T0* volatile a241; /* yyspecial_routines54 */
	T6 volatile a242; /* yyvsc54 */
	T0* volatile a243; /* yyvs54 */
	T0* volatile a244; /* yyspecial_routines55 */
	T6 volatile a245; /* yyvsc55 */
	T0* volatile a246; /* yyvs55 */
	T0* volatile a247; /* yyspecial_routines56 */
	T6 volatile a248; /* yyvsc56 */
	T0* volatile a249; /* yyvs56 */
	T0* volatile a250; /* yyspecial_routines57 */
	T6 volatile a251; /* yyvsc57 */
	T0* volatile a252; /* yyvs57 */
	T0* volatile a253; /* yyspecial_routines58 */
	T6 volatile a254; /* yyvsc58 */
	T0* volatile a255; /* yyvs58 */
	T0* volatile a256; /* yyspecial_routines59 */
	T6 volatile a257; /* yyvsc59 */
	T0* volatile a258; /* yyvs59 */
	T0* volatile a259; /* yyspecial_routines60 */
	T6 volatile a260; /* yyvsc60 */
	T0* volatile a261; /* yyvs60 */
	T0* volatile a262; /* yyspecial_routines61 */
	T6 volatile a263; /* yyvsc61 */
	T0* volatile a264; /* yyvs61 */
	T0* volatile a265; /* yyspecial_routines62 */
	T6 volatile a266; /* yyvsc62 */
	T0* volatile a267; /* yyvs62 */
	T0* volatile a268; /* yyspecial_routines63 */
	T6 volatile a269; /* yyvsc63 */
	T0* volatile a270; /* yyvs63 */
	T0* volatile a271; /* yyspecial_routines64 */
	T6 volatile a272; /* yyvsc64 */
	T0* volatile a273; /* yyvs64 */
	T0* volatile a274; /* yyspecial_routines65 */
	T6 volatile a275; /* yyvsc65 */
	T0* volatile a276; /* yyvs65 */
	T0* volatile a277; /* yyspecial_routines66 */
	T6 volatile a278; /* yyvsc66 */
	T0* volatile a279; /* yyvs66 */
	T0* volatile a280; /* yyspecial_routines67 */
	T6 volatile a281; /* yyvsc67 */
	T0* volatile a282; /* yyvs67 */
	T0* volatile a283; /* yyspecial_routines68 */
	T6 volatile a284; /* yyvsc68 */
	T0* volatile a285; /* yyvs68 */
	T0* volatile a286; /* yyspecial_routines69 */
	T6 volatile a287; /* yyvsc69 */
	T0* volatile a288; /* yyvs69 */
	T0* volatile a289; /* yyspecial_routines70 */
	T6 volatile a290; /* yyvsc70 */
	T0* volatile a291; /* yyvs70 */
	T0* volatile a292; /* yyspecial_routines71 */
	T6 volatile a293; /* yyvsc71 */
	T0* volatile a294; /* yyvs71 */
	T0* volatile a295; /* yyspecial_routines72 */
	T6 volatile a296; /* yyvsc72 */
	T0* volatile a297; /* yyvs72 */
	T0* volatile a298; /* yyspecial_routines73 */
	T6 volatile a299; /* yyvsc73 */
	T0* volatile a300; /* yyvs73 */
	T0* volatile a301; /* yyspecial_routines74 */
	T6 volatile a302; /* yyvsc74 */
	T0* volatile a303; /* yyvs74 */
	T0* volatile a304; /* yyspecial_routines75 */
	T6 volatile a305; /* yyvsc75 */
	T0* volatile a306; /* yyvs75 */
	T0* volatile a307; /* yyspecial_routines76 */
	T6 volatile a308; /* yyvsc76 */
	T0* volatile a309; /* yyvs76 */
	T0* volatile a310; /* yyspecial_routines77 */
	T6 volatile a311; /* yyvsc77 */
	T0* volatile a312; /* yyvs77 */
	T0* volatile a313; /* yyspecial_routines78 */
	T6 volatile a314; /* yyvsc78 */
	T0* volatile a315; /* yyvs78 */
	T0* volatile a316; /* yyspecial_routines79 */
	T6 volatile a317; /* yyvsc79 */
	T0* volatile a318; /* yyvs79 */
	T0* volatile a319; /* yyspecial_routines80 */
	T6 volatile a320; /* yyvsc80 */
	T0* volatile a321; /* yyvs80 */
	T0* volatile a322; /* yyspecial_routines81 */
	T6 volatile a323; /* yyvsc81 */
	T0* volatile a324; /* yyvs81 */
	T0* volatile a325; /* yyspecial_routines82 */
	T6 volatile a326; /* yyvsc82 */
	T0* volatile a327; /* yyvs82 */
	T0* volatile a328; /* yyspecial_routines83 */
	T6 volatile a329; /* yyvsc83 */
	T0* volatile a330; /* yyvs83 */
	T0* volatile a331; /* yyspecial_routines84 */
	T6 volatile a332; /* yyvsc84 */
	T0* volatile a333; /* yyvs84 */
	T0* volatile a334; /* yyspecial_routines85 */
	T6 volatile a335; /* yyvsc85 */
	T0* volatile a336; /* yyvs85 */
	T0* volatile a337; /* yyspecial_routines86 */
	T6 volatile a338; /* yyvsc86 */
	T0* volatile a339; /* yyvs86 */
	T0* volatile a340; /* yyspecial_routines87 */
	T6 volatile a341; /* yyvsc87 */
	T0* volatile a342; /* yyvs87 */
	T0* volatile a343; /* yyspecial_routines88 */
	T6 volatile a344; /* yyvsc88 */
	T0* volatile a345; /* yyvs88 */
	T0* volatile a346; /* yyspecial_routines89 */
	T6 volatile a347; /* yyvsc89 */
	T0* volatile a348; /* yyvs89 */
	T0* volatile a349; /* yyspecial_routines90 */
	T6 volatile a350; /* yyvsc90 */
	T0* volatile a351; /* yyvs90 */
	T0* volatile a352; /* yyspecial_routines91 */
	T6 volatile a353; /* yyvsc91 */
	T0* volatile a354; /* yyvs91 */
	T0* volatile a355; /* yyspecial_routines92 */
	T6 volatile a356; /* yyvsc92 */
	T0* volatile a357; /* yyvs92 */
	T0* volatile a358; /* yyspecial_routines93 */
	T6 volatile a359; /* yyvsc93 */
	T0* volatile a360; /* yyvs93 */
	T0* volatile a361; /* yyspecial_routines94 */
	T6 volatile a362; /* yyvsc94 */
	T0* volatile a363; /* yyvs94 */
	T0* volatile a364; /* yyspecial_routines95 */
	T6 volatile a365; /* yyvsc95 */
	T0* volatile a366; /* yyvs95 */
	T0* volatile a367; /* yyspecial_routines96 */
	T6 volatile a368; /* yyvsc96 */
	T0* volatile a369; /* yyvs96 */
	T0* volatile a370; /* yyspecial_routines97 */
	T6 volatile a371; /* yyvsc97 */
	T0* volatile a372; /* yyvs97 */
	T0* volatile a373; /* yyspecial_routines98 */
	T6 volatile a374; /* yyvsc98 */
	T0* volatile a375; /* yyvs98 */
	T0* volatile a376; /* yyspecial_routines99 */
	T6 volatile a377; /* yyvsc99 */
	T0* volatile a378; /* yyvs99 */
	T0* volatile a379; /* yyspecial_routines100 */
	T6 volatile a380; /* yyvsc100 */
	T0* volatile a381; /* yyvs100 */
	T0* volatile a382; /* yyspecial_routines101 */
	T6 volatile a383; /* yyvsc101 */
	T0* volatile a384; /* yyvs101 */
	T0* volatile a385; /* yyspecial_routines102 */
	T6 volatile a386; /* yyvsc102 */
	T0* volatile a387; /* yyvs102 */
	T0* volatile a388; /* yyspecial_routines103 */
	T6 volatile a389; /* yyvsc103 */
	T0* volatile a390; /* yyvs103 */
	T0* volatile a391; /* yyspecial_routines104 */
	T6 volatile a392; /* yyvsc104 */
	T0* volatile a393; /* yyvs104 */
	T0* volatile a394; /* yyspecial_routines105 */
	T6 volatile a395; /* yyvsc105 */
	T0* volatile a396; /* yyvs105 */
	T0* volatile a397; /* yyspecial_routines106 */
	T6 volatile a398; /* yyvsc106 */
	T0* volatile a399; /* yyvs106 */
	T0* volatile a400; /* yyspecial_routines107 */
	T6 volatile a401; /* yyvsc107 */
	T0* volatile a402; /* yyvs107 */
	T0* volatile a403; /* yyspecial_routines108 */
	T6 volatile a404; /* yyvsc108 */
	T0* volatile a405; /* yyvs108 */
	T0* volatile a406; /* yyspecial_routines109 */
	T6 volatile a407; /* yyvsc109 */
	T0* volatile a408; /* yyvs109 */
	T0* volatile a409; /* yyspecial_routines110 */
	T6 volatile a410; /* yyvsc110 */
	T0* volatile a411; /* yyvs110 */
	T0* volatile a412; /* yyspecial_routines111 */
	T6 volatile a413; /* yyvsc111 */
	T0* volatile a414; /* yyvs111 */
	T0* volatile a415; /* yyspecial_routines112 */
	T6 volatile a416; /* yyvsc112 */
	T0* volatile a417; /* yyvs112 */
	T0* volatile a418; /* yyspecial_routines113 */
	T6 volatile a419; /* yyvsc113 */
	T0* volatile a420; /* yyvs113 */
	T0* volatile a421; /* yyspecial_routines114 */
	T6 volatile a422; /* yyvsc114 */
	T0* volatile a423; /* yyvs114 */
	T0* volatile a424; /* yyspecial_routines115 */
	T6 volatile a425; /* yyvsc115 */
	T0* volatile a426; /* yyvs115 */
	T0* volatile a427; /* yyspecial_routines116 */
	T6 volatile a428; /* yyvsc116 */
	T0* volatile a429; /* yyvs116 */
	T0* volatile a430; /* yyspecial_routines117 */
	T6 volatile a431; /* yyvsc117 */
	T0* volatile a432; /* yyvs117 */
	T0* volatile a433; /* yyspecial_routines118 */
	T6 volatile a434; /* yyvsc118 */
	T0* volatile a435; /* yyvs118 */
	T0* volatile a436; /* yyspecial_routines119 */
	T6 volatile a437; /* yyvsc119 */
	T0* volatile a438; /* yyvs119 */
	T0* volatile a439; /* yyspecial_routines120 */
	T6 volatile a440; /* yyvsc120 */
	T0* volatile a441; /* yyvs120 */
	T0* volatile a442; /* yyspecial_routines121 */
	T6 volatile a443; /* yyvsc121 */
	T0* volatile a444; /* yyvs121 */
	T0* volatile a445; /* yyspecial_routines122 */
	T6 volatile a446; /* yyvsc122 */
	T0* volatile a447; /* yyvs122 */
	T0* volatile a448; /* yyspecial_routines123 */
	T6 volatile a449; /* yyvsc123 */
	T0* volatile a450; /* yyvs123 */
	T0* volatile a451; /* yyspecial_routines124 */
	T6 volatile a452; /* yyvsc124 */
	T0* volatile a453; /* yyvs124 */
	T0* volatile a454; /* yyspecial_routines125 */
	T6 volatile a455; /* yyvsc125 */
	T0* volatile a456; /* yyvs125 */
	T0* volatile a457; /* yyspecial_routines126 */
	T6 volatile a458; /* yyvsc126 */
	T0* volatile a459; /* yyvs126 */
	T0* volatile a460; /* yyspecial_routines127 */
	T6 volatile a461; /* yyvsc127 */
	T0* volatile a462; /* yyvs127 */
	T0* volatile a463; /* yyspecial_routines128 */
	T6 volatile a464; /* yyvsc128 */
	T0* volatile a465; /* yyvs128 */
	T0* volatile a466; /* yyspecial_routines129 */
	T6 volatile a467; /* yyvsc129 */
	T0* volatile a468; /* yyvs129 */
	T0* volatile a469; /* yyspecial_routines130 */
	T6 volatile a470; /* yyvsc130 */
	T0* volatile a471; /* yyvs130 */
	T0* volatile a472; /* yyspecial_routines131 */
	T6 volatile a473; /* yyvsc131 */
	T0* volatile a474; /* yyvs131 */
	T0* volatile a475; /* yyspecial_routines132 */
	T6 volatile a476; /* yyvsc132 */
	T0* volatile a477; /* yyvs132 */
	T0* volatile a478; /* yyspecial_routines133 */
	T6 volatile a479; /* yyvsc133 */
	T0* volatile a480; /* yyvs133 */
	T0* volatile a481; /* yyspecial_routines134 */
	T6 volatile a482; /* yyvsc134 */
	T0* volatile a483; /* yyvs134 */
	T0* volatile a484; /* yyspecial_routines135 */
	T6 volatile a485; /* yyvsc135 */
	T0* volatile a486; /* yyvs135 */
	T0* volatile a487; /* yyspecial_routines136 */
	T6 volatile a488; /* yyvsc136 */
	T0* volatile a489; /* yyvs136 */
	T0* volatile a490; /* yyspecial_routines137 */
	T6 volatile a491; /* yyvsc137 */
	T0* volatile a492; /* yyvs137 */
	T0* volatile a493; /* yyspecial_routines138 */
	T6 volatile a494; /* yyvsc138 */
	T0* volatile a495; /* yyvs138 */
	T0* volatile a496; /* yyspecial_routines139 */
	T6 volatile a497; /* yyvsc139 */
	T0* volatile a498; /* yyvs139 */
	T0* volatile a499; /* yytypes1 */
	T0* volatile a500; /* yytypes2 */
	T0* volatile a501; /* yy_content */
	T0* volatile a502; /* yy_content_area */
	T0* volatile a503; /* yy_unicode_content_area */
	T0* volatile a504; /* yy_state_stack */
	T6 volatile a505; /* yyvsp1 */
	T6 volatile a506; /* yyvsp2 */
	T6 volatile a507; /* yyvsp3 */
	T6 volatile a508; /* yyvsp4 */
	T6 volatile a509; /* yyvsp5 */
	T6 volatile a510; /* yyvsp6 */
	T6 volatile a511; /* yyvsp7 */
	T6 volatile a512; /* yyvsp8 */
	T6 volatile a513; /* yyvsp9 */
	T6 volatile a514; /* yyvsp10 */
	T6 volatile a515; /* yyvsp11 */
	T6 volatile a516; /* yyvsp12 */
	T6 volatile a517; /* yyvsp13 */
	T6 volatile a518; /* yyvsp14 */
	T6 volatile a519; /* yyvsp15 */
	T6 volatile a520; /* yyvsp16 */
	T6 volatile a521; /* yyvsp17 */
	T6 volatile a522; /* yyvsp18 */
	T6 volatile a523; /* yyvsp19 */
	T6 volatile a524; /* yyvsp20 */
	T6 volatile a525; /* yyvsp21 */
	T6 volatile a526; /* yyvsp22 */
	T6 volatile a527; /* yyvsp23 */
	T6 volatile a528; /* yyvsp24 */
	T6 volatile a529; /* yyvsp25 */
	T6 volatile a530; /* yyvsp26 */
	T6 volatile a531; /* yyvsp27 */
	T6 volatile a532; /* yyvsp28 */
	T6 volatile a533; /* yyvsp29 */
	T6 volatile a534; /* yyvsp30 */
	T6 volatile a535; /* yyvsp31 */
	T6 volatile a536; /* yyvsp32 */
	T6 volatile a537; /* yyvsp33 */
	T6 volatile a538; /* yyvsp34 */
	T6 volatile a539; /* yyvsp35 */
	T6 volatile a540; /* yyvsp36 */
	T6 volatile a541; /* yyvsp37 */
	T6 volatile a542; /* yyvsp38 */
	T6 volatile a543; /* yyvsp39 */
	T6 volatile a544; /* yyvsp40 */
	T6 volatile a545; /* yyvsp41 */
	T6 volatile a546; /* yyvsp42 */
	T6 volatile a547; /* yyvsp43 */
	T6 volatile a548; /* yyvsp44 */
	T6 volatile a549; /* yyvsp45 */
	T6 volatile a550; /* yyvsp46 */
	T6 volatile a551; /* yyvsp47 */
	T6 volatile a552; /* yyvsp48 */
	T6 volatile a553; /* yyvsp49 */
	T6 volatile a554; /* yyvsp50 */
	T6 volatile a555; /* yyvsp51 */
	T6 volatile a556; /* yyvsp52 */
	T6 volatile a557; /* yyvsp53 */
	T6 volatile a558; /* yyvsp54 */
	T6 volatile a559; /* yyvsp55 */
	T6 volatile a560; /* yyvsp56 */
	T6 volatile a561; /* yyvsp57 */
	T6 volatile a562; /* yyvsp58 */
	T6 volatile a563; /* yyvsp59 */
	T6 volatile a564; /* yyvsp60 */
	T6 volatile a565; /* yyvsp61 */
	T6 volatile a566; /* yyvsp62 */
	T6 volatile a567; /* yyvsp63 */
	T6 volatile a568; /* yyvsp64 */
	T6 volatile a569; /* yyvsp65 */
	T6 volatile a570; /* yyvsp66 */
	T6 volatile a571; /* yyvsp67 */
	T6 volatile a572; /* yyvsp68 */
	T6 volatile a573; /* yyvsp69 */
	T6 volatile a574; /* yyvsp70 */
	T6 volatile a575; /* yyvsp71 */
	T6 volatile a576; /* yyvsp72 */
	T6 volatile a577; /* yyvsp73 */
	T6 volatile a578; /* yyvsp74 */
	T6 volatile a579; /* yyvsp75 */
	T6 volatile a580; /* yyvsp76 */
	T6 volatile a581; /* yyvsp77 */
	T6 volatile a582; /* yyvsp78 */
	T6 volatile a583; /* yyvsp79 */
	T6 volatile a584; /* yyvsp80 */
	T6 volatile a585; /* yyvsp81 */
	T6 volatile a586; /* yyvsp82 */
	T6 volatile a587; /* yyvsp83 */
	T6 volatile a588; /* yyvsp84 */
	T6 volatile a589; /* yyvsp85 */
	T6 volatile a590; /* yyvsp86 */
	T6 volatile a591; /* yyvsp87 */
	T6 volatile a592; /* yyvsp88 */
	T6 volatile a593; /* yyvsp89 */
	T6 volatile a594; /* yyvsp90 */
	T6 volatile a595; /* yyvsp91 */
	T6 volatile a596; /* yyvsp92 */
	T6 volatile a597; /* yyvsp93 */
	T6 volatile a598; /* yyvsp94 */
	T6 volatile a599; /* yyvsp95 */
	T6 volatile a600; /* yyvsp96 */
	T6 volatile a601; /* yyvsp97 */
	T6 volatile a602; /* yyvsp98 */
	T6 volatile a603; /* yyvsp99 */
	T6 volatile a604; /* yyvsp100 */
	T6 volatile a605; /* yyvsp101 */
	T6 volatile a606; /* yyvsp102 */
	T6 volatile a607; /* yyvsp103 */
	T6 volatile a608; /* yyvsp104 */
	T6 volatile a609; /* yyvsp105 */
	T6 volatile a610; /* yyvsp106 */
	T6 volatile a611; /* yyvsp107 */
	T6 volatile a612; /* yyvsp108 */
	T6 volatile a613; /* yyvsp109 */
	T6 volatile a614; /* yyvsp110 */
	T6 volatile a615; /* yyvsp111 */
	T6 volatile a616; /* yyvsp112 */
	T6 volatile a617; /* yyvsp113 */
	T6 volatile a618; /* yyvsp114 */
	T6 volatile a619; /* yyvsp115 */
	T6 volatile a620; /* yyvsp116 */
	T6 volatile a621; /* yyvsp117 */
	T6 volatile a622; /* yyvsp118 */
	T6 volatile a623; /* yyvsp119 */
	T6 volatile a624; /* yyvsp120 */
	T6 volatile a625; /* yyvsp121 */
	T6 volatile a626; /* yyvsp122 */
	T6 volatile a627; /* yyvsp123 */
	T6 volatile a628; /* yyvsp124 */
	T6 volatile a629; /* yyvsp125 */
	T6 volatile a630; /* yyvsp126 */
	T6 volatile a631; /* yyvsp127 */
	T6 volatile a632; /* yyvsp128 */
	T6 volatile a633; /* yyvsp129 */
	T6 volatile a634; /* yyvsp130 */
	T6 volatile a635; /* yyvsp131 */
	T6 volatile a636; /* yyvsp132 */
	T6 volatile a637; /* yyvsp133 */
	T6 volatile a638; /* yyvsp134 */
	T6 volatile a639; /* yyvsp135 */
	T6 volatile a640; /* yyvsp136 */
	T6 volatile a641; /* yyvsp137 */
	T6 volatile a642; /* yyvsp138 */
	T6 volatile a643; /* yyvsp139 */
	T0* volatile a644; /* yy_accept */
	T0* volatile a645; /* yy_ec */
	T0* volatile a646; /* yy_chk */
	T0* volatile a647; /* yy_base */
	T0* volatile a648; /* yy_def */
	T0* volatile a649; /* yy_meta */
	T0* volatile a650; /* yy_nxt */
	T0* volatile a651; /* yy_acclist */
	T0* volatile a652; /* last_detachable_any_value */
	T0* volatile a653; /* last_detachable_et_keyword_value */
	T0* volatile a654; /* last_detachable_et_agent_keyword_value */
	T0* volatile a655; /* last_detachable_et_invariant_keyword_value */
	T0* volatile a656; /* last_detachable_et_precursor_keyword_value */
	T0* volatile a657; /* last_detachable_et_symbol_value */
	T0* volatile a658; /* last_detachable_et_position_value */
	T0* volatile a659; /* last_detachable_et_boolean_constant_value */
	T0* volatile a660; /* last_detachable_et_break_value */
	T0* volatile a661; /* last_detachable_et_character_constant_value */
	T0* volatile a662; /* last_detachable_et_current_value */
	T0* volatile a663; /* last_detachable_et_free_operator_value */
	T0* volatile a664; /* last_detachable_et_identifier_value */
	T0* volatile a665; /* last_detachable_et_integer_constant_value */
	T0* volatile a666; /* last_detachable_et_keyword_operator_value */
	T0* volatile a667; /* last_detachable_et_manifest_string_value */
	T0* volatile a668; /* last_detachable_et_real_constant_value */
	T0* volatile a669; /* last_detachable_et_result_value */
	T0* volatile a670; /* last_detachable_et_retry_instruction_value */
	T0* volatile a671; /* last_detachable_et_symbol_operator_value */
	T0* volatile a672; /* last_detachable_et_void_value */
	T0* volatile a673; /* last_detachable_et_semicolon_symbol_value */
	T0* volatile a674; /* last_detachable_et_bracket_symbol_value */
	T0* volatile a675; /* last_detachable_et_question_mark_symbol_value */
	T0* volatile a676; /* last_formal_arguments */
	T0* volatile a677; /* last_local_variables */
	T0* volatile a678; /* last_object_tests */
	T0* volatile a679; /* last_iteration_components */
	T0* volatile a680; /* last_inline_separate_arguments */
	T6 volatile a681; /* last_break_end */
	T6 volatile a682; /* last_comment_end */
	T6 volatile a683; /* ms_column */
	T6 volatile a684; /* ms_line */
	T6 volatile a685; /* break_kind */
	T3 volatile a686; /* verbatim_opening_character */
	T1 volatile a687; /* has_natural_64_overflow */
	T11 volatile a688; /* last_natural_64 */
};

/* Struct for type DS_ARRAYED_LIST [ET_CLASS] */
struct S69 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
	T0* volatile a6; /* equality_tester */
};

/* Struct for type ET_CLASS */
struct S70 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* id */
	T0* volatile a3; /* ancestors */
	T0* volatile a4; /* conforming_ancestors */
	T0* volatile a5; /* queries */
	T0* volatile a6; /* procedures */
	T0* volatile a7; /* class_keyword */
	T0* volatile a8; /* end_keyword */
	T0* volatile a9; /* group */
	T0* volatile a10; /* status_mutex */
	T0* volatile a11; /* processing_mutex */
	T0* volatile a12; /* formal_parameter_types_mutex */
	T0* volatile a13; /* named_base_class */
	T6 volatile a14; /* time_stamp */
	T8 volatile a15; /* class_code */
	T1 volatile a16; /* tuple_actual_parameters_unfolded_1 */
	T1 volatile a17; /* tuple_actual_parameters_unfolded_2 */
	T0* volatile a18; /* filename */
	T1 volatile a19; /* unprotected_is_parsed */
	T0* volatile a20; /* formal_parameters */
	T1 volatile a21; /* unprotected_has_syntax_error */
	T1 volatile a22; /* unprotected_ancestors_built */
	T1 volatile a23; /* unprotected_has_ancestors_error */
	T1 volatile a24; /* unprotected_features_flattened */
	T1 volatile a25; /* unprotected_has_flattening_error */
	T1 volatile a26; /* unprotected_interface_checked */
	T1 volatile a27; /* unprotected_has_interface_error */
	T1 volatile a28; /* unprotected_implementation_checked */
	T1 volatile a29; /* unprotected_has_implementation_error */
	T0* volatile a30; /* frozen_keyword */
	T0* volatile a31; /* external_keyword */
	T1 volatile a32; /* has_utf8_bom */
	T0* volatile a33; /* providers */
	T6 volatile a34; /* tuple_constraint_position */
	T6 volatile a35; /* registered_feature_count */
	T0* volatile a36; /* class_mark */
	T6 volatile a37; /* index */
	T1 volatile a38; /* unprotected_is_marked */
	T6 volatile a39; /* registered_inline_constant_count */
	T0* volatile a40; /* suppliers */
	T1 volatile a41; /* has_deferred_features */
	T1 volatile a42; /* redeclared_signatures_checked */
	T0* volatile a43; /* creators */
	T0* volatile a44; /* convert_features */
	T0* volatile a45; /* feature_clauses */
	T0* volatile a46; /* first_note_clause */
	T0* volatile a47; /* second_note_clause */
	T0* volatile a48; /* invariants */
	T0* volatile a49; /* obsolete_message */
	T0* volatile a50; /* parent_clauses */
	T0* volatile a51; /* leading_break */
	T1 volatile a52; /* is_interface */
	T1 volatile a53; /* is_checking_implementation */
	T1 volatile a54; /* is_ignored */
	T1 volatile a55; /* invariants_enabled */
	T1 volatile a56; /* preconditions_enabled */
	T1 volatile a57; /* postconditions_enabled */
	T0* volatile a58; /* formal_parameter_types */
	T1 volatile a59; /* loop_assertions_enabled */
	T1 volatile a60; /* supplier_preconditions_enabled */
	T1 volatile a61; /* check_assertions_enabled */
};

/* Struct for type TUPLE [ET_CLASS] */
struct S74 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_CLASS]] */
struct S75 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE */
struct S76 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [DS_ARRAYED_LIST [ET_CLASS]] */
struct S77 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PREDICATE [TUPLE [ET_CLASS]] */
struct S78 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type ET_CREATOR_LIST */
struct S80 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_TOKEN_CONSTANTS */
struct S81 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CREATOR */
struct S82 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* create_keyword */
	T0* volatile a2; /* clients */
	T0* volatile a3; /* storage */
	T6 volatile a4; /* count */
};

/* Struct for type UT_CANNOT_READ_FILE_ERROR */
struct S83 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type ET_ECF_SYSTEM_PARSER */
struct S85 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_system */
	T0* volatile a2; /* ast_factory */
	T0* volatile a3; /* override_settings */
	T0* volatile a4; /* override_capabilities */
	T0* volatile a5; /* override_variables */
	T0* volatile a6; /* parsed_libraries */
	T0* volatile a7; /* parsed_dotnet_assemblies */
	T0* volatile a8; /* redirected_locations */
	T0* volatile a9; /* xml_parser */
	T0* volatile a10; /* tree_pipe */
	T0* volatile a11; /* error_handler */
	T0* volatile a12; /* ise_version */
	T0* volatile a13; /* ise_eiffel_to_gobo_uuid_mappings */
};

/* Struct for type ET_ECF_ERROR_HANDLER */
struct S86 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* error_count */
	T0* volatile a2; /* error_file */
	T0* volatile a3; /* warning_file */
	T0* volatile a4; /* info_file */
};

/* Struct for type ET_ECF_SYSTEM */
struct S87 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* filename */
	T0* volatile a3; /* register_class_mutex */
	T0* volatile a4; /* external_include_pathnames */
	T0* volatile a5; /* external_object_pathnames */
	T0* volatile a6; /* external_library_pathnames */
	T0* volatile a7; /* external_make_pathnames */
	T0* volatile a8; /* external_resource_pathnames */
	T0* volatile a9; /* external_cflags */
	T0* volatile a10; /* external_linker_flags */
	T1 volatile a11; /* console_application_mode */
	T1 volatile a12; /* total_order_on_reals_mode */
	T0* volatile a13; /* current_system */
	T0* volatile a14; /* universe */
	T0* volatile a15; /* libraries */
	T0* volatile a16; /* clusters */
	T0* volatile a17; /* dotnet_assemblies */
	T0* volatile a18; /* implicit_attachment_type_mark */
	T0* volatile a19; /* master_classes */
	T0* volatile a20; /* master_class_mutex */
	T0* volatile a21; /* system_name */
	T0* volatile a22; /* any_type */
	T0* volatile a23; /* detachable_any_type */
	T0* volatile a24; /* detachable_separate_any_type */
	T0* volatile a25; /* any_parent */
	T0* volatile a26; /* any_parents */
	T0* volatile a27; /* any_clients */
	T0* volatile a28; /* tuple_type */
	T0* volatile a29; /* detachable_tuple_type */
	T0* volatile a30; /* tuple_identity_type */
	T0* volatile a31; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a32; /* array_any_type */
	T0* volatile a33; /* array_detachable_any_type */
	T0* volatile a34; /* array_none_type */
	T0* volatile a35; /* array_identity_type */
	T0* volatile a36; /* boolean_type */
	T0* volatile a37; /* character_type */
	T0* volatile a38; /* character_8_type */
	T0* volatile a39; /* character_8_convert_feature */
	T0* volatile a40; /* character_32_type */
	T0* volatile a41; /* character_32_convert_feature */
	T0* volatile a42; /* double_type */
	T0* volatile a43; /* exception_type */
	T0* volatile a44; /* detachable_exception_type */
	T0* volatile a45; /* exception_manager_type */
	T0* volatile a46; /* function_identity_any_type */
	T0* volatile a47; /* immutable_string_8_type */
	T0* volatile a48; /* immutable_string_32_type */
	T0* volatile a49; /* integer_type */
	T0* volatile a50; /* integer_8_type */
	T0* volatile a51; /* integer_8_convert_feature */
	T0* volatile a52; /* integer_16_type */
	T0* volatile a53; /* integer_16_convert_feature */
	T0* volatile a54; /* integer_32_type */
	T0* volatile a55; /* integer_32_convert_feature */
	T0* volatile a56; /* integer_64_type */
	T0* volatile a57; /* integer_64_convert_feature */
	T0* volatile a58; /* ise_exception_manager_type */
	T0* volatile a59; /* iterable_detachable_separate_any_type */
	T0* volatile a60; /* natural_type */
	T0* volatile a61; /* natural_8_type */
	T0* volatile a62; /* natural_8_convert_feature */
	T0* volatile a63; /* natural_16_type */
	T0* volatile a64; /* natural_16_convert_feature */
	T0* volatile a65; /* natural_32_type */
	T0* volatile a66; /* natural_32_convert_feature */
	T0* volatile a67; /* natural_64_type */
	T0* volatile a68; /* natural_64_convert_feature */
	T0* volatile a69; /* none_type */
	T0* volatile a70; /* detachable_none_type */
	T0* volatile a71; /* pointer_type */
	T0* volatile a72; /* predicate_identity_type */
	T0* volatile a73; /* procedure_identity_type */
	T0* volatile a74; /* real_type */
	T0* volatile a75; /* real_32_type */
	T0* volatile a76; /* real_32_convert_feature */
	T0* volatile a77; /* real_64_type */
	T0* volatile a78; /* real_64_convert_feature */
	T0* volatile a79; /* routine_identity_type */
	T0* volatile a80; /* special_any_type */
	T0* volatile a81; /* special_detachable_any_type */
	T0* volatile a82; /* special_identity_type */
	T0* volatile a83; /* string_type */
	T0* volatile a84; /* detachable_string_type */
	T0* volatile a85; /* string_8_type */
	T0* volatile a86; /* detachable_string_8_type */
	T0* volatile a87; /* string_8_convert_feature */
	T0* volatile a88; /* string_32_type */
	T0* volatile a89; /* string_32_convert_feature */
	T0* volatile a90; /* system_object_type */
	T0* volatile a91; /* system_object_parents */
	T0* volatile a92; /* system_string_type */
	T0* volatile a93; /* type_detachable_any_type */
	T0* volatile a94; /* detachable_type_exception_type */
	T0* volatile a95; /* type_detachable_exception_type */
	T0* volatile a96; /* type_detachable_like_current_type */
	T0* volatile a97; /* type_identity_type */
	T0* volatile a98; /* typed_pointer_identity_type */
	T0* volatile a99; /* wide_character_type */
	T1 volatile a100; /* is_read_only */
	T6 volatile a101; /* registered_class_count */
	T0* volatile a102; /* targets */
	T0* volatile a103; /* library_target */
	T0* volatile a104; /* selected_target */
	T0* volatile a105; /* ecf_version */
	T0* volatile a106; /* executable_name */
	T0* volatile a107; /* root_type */
	T0* volatile a108; /* root_creation */
	T6 volatile a109; /* default_create_seed */
	T1 volatile a110; /* inlining_mode */
	T6 volatile a111; /* inlining_size */
	T1 volatile a112; /* scoop_mode */
	T1 volatile a113; /* attachment_type_conformance_mode */
	T6 volatile a114; /* twin_seed */
	T6 volatile a115; /* is_equal_seed */
	T6 volatile a116; /* dispose_seed */
	T1 volatile a117; /* use_boehm_gc */
	T6 volatile a118; /* routine_call_seed */
	T6 volatile a119; /* function_item_seed */
	T1 volatile a120; /* trace_mode */
	T1 volatile a121; /* exception_trace_mode */
	T1 volatile a122; /* check_for_void_target_mode */
	T1 volatile a123; /* line_generation_mode */
	T1 volatile a124; /* multithreaded_mode */
	T6 volatile a125; /* copy_seed */
	T0* volatile a126; /* ecf_namespace */
	T0* volatile a127; /* description */
	T0* volatile a128; /* notes */
	T0* volatile a129; /* uuid */
	T1 volatile a130; /* is_preparsed */
	T0* volatile a131; /* scm_read_mapping_builder */
	T0* volatile a132; /* scm_write_mapping_builder */
	T1 volatile a133; /* use_obsolete_syntax_mode */
	T1 volatile a134; /* obsolete_iteration_mode */
	T1 volatile a135; /* target_type_attachment_mode */
	T6 volatile a136; /* iterable_new_cursor_seed */
	T6 volatile a137; /* iteration_cursor_after_seed */
	T6 volatile a138; /* iteration_cursor_forth_seed */
	T6 volatile a139; /* iteration_cursor_item_seed */
	T1 volatile a140; /* is_dotnet */
	T6 volatile a141; /* special_item_seed */
	T1 volatile a142; /* obsolete_routine_type_mode */
};

/* Struct for type ET_ECF_TARGET */
struct S88 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* settings */
	T0* volatile a2; /* capabilities */
	T0* volatile a3; /* options */
	T0* volatile a4; /* system_config */
	T0* volatile a5; /* root */
	T0* volatile a6; /* variables */
	T0* volatile a7; /* name */
	T0* volatile a8; /* parent */
	T0* volatile a9; /* description */
	T0* volatile a10; /* precompiled_library */
	T0* volatile a11; /* version */
	T1 volatile a12; /* is_abstract */
	T0* volatile a13; /* clusters */
	T0* volatile a14; /* libraries */
	T0* volatile a15; /* dotnet_assemblies */
	T0* volatile a16; /* file_rules */
	T0* volatile a17; /* class_mappings */
	T0* volatile a18; /* namespaces */
	T0* volatile a19; /* external_cflags */
	T0* volatile a20; /* external_includes */
	T0* volatile a21; /* external_libraries */
	T0* volatile a22; /* external_linker_flags */
	T0* volatile a23; /* external_makes */
	T0* volatile a24; /* external_objects */
	T0* volatile a25; /* external_resources */
	T0* volatile a26; /* pre_compile_actions */
	T0* volatile a27; /* post_compile_actions */
	T0* volatile a28; /* notes */
};

/* Struct for type ET_ECF_CLUSTER */
struct S89 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* pathname */
	T1 volatile a3; /* is_relative */
	T0* volatile a4; /* universe */
	T0* volatile a5; /* target */
	T0* volatile a6; /* options */
	T1 volatile a7; /* overridden_constraint_enabled */
	T1 volatile a8; /* scm_mapping_constraint_enabled */
	T0* volatile a9; /* subclusters */
	T1 volatile a10; /* use_obsolete_syntax */
	T0* volatile a11; /* scm_read_mapping */
	T0* volatile a12; /* scm_write_mapping */
	T0* volatile a13; /* description */
	T1 volatile a14; /* is_hidden */
	T0* volatile a15; /* classname_prefix */
	T1 volatile a16; /* is_read_only */
	T1 volatile a17; /* is_recursive */
	T0* volatile a18; /* conditioned_subclusters */
	T0* volatile a19; /* conditioned_file_rules */
	T0* volatile a20; /* class_options */
	T0* volatile a21; /* class_renamings */
	T0* volatile a22; /* class_mappings */
	T0* volatile a23; /* visible_classes */
	T0* volatile a24; /* provider_groups */
	T0* volatile a25; /* conditions */
	T0* volatile a26; /* notes */
	T1 volatile a27; /* is_override */
	T0* volatile a28; /* overridden_group */
	T1 volatile a29; /* is_preparsed */
	T1 volatile a30; /* is_abstract */
	T0* volatile a31; /* parent */
	T0* volatile a32; /* file_rules */
	T1 volatile a33; /* is_implicit */
	T0* volatile a34; /* provider_constraint */
	T0* volatile a35; /* dependant_constraint */
	T0* volatile a36; /* cached_absolute_pathname */
};

/* Struct for type ET_ECF_SETTINGS */
struct S90 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* primary_settings */
	T0* volatile a2; /* secondary_settings */
};

/* Struct for type ET_ECF_CAPABILITIES */
struct S91 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* primary_use_capabilities */
	T0* volatile a2; /* primary_support_capabilities */
	T0* volatile a3; /* secondary_capabilities */
};

/* Struct for type ET_ECF_VARIABLES */
struct S92 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* secondary_variables */
	T0* volatile a2; /* primary_variables */
};

/* Struct for type ET_CLUSTERS */
struct S94 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* clusters */
};

/* Struct for type ET_DYNAMIC_SYSTEM */
struct S95 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_system */
	T0* volatile a3; /* dynamic_types */
	T0* volatile a4; /* dynamic_type_set_builder */
	T1 volatile a5; /* catcall_error_mode */
	T0* volatile a6; /* system_processor */
	T0* volatile a7; /* dynamic_generic_types_by_name */
	T0* volatile a8; /* null_dynamic_type_set_builder */
	T1 volatile a9; /* full_class_checking */
	T1 volatile a10; /* catcall_warning_mode */
	T0* volatile a11; /* new_instance_types */
	T0* volatile a12; /* unknown_type */
	T0* volatile a13; /* boolean_type */
	T0* volatile a14; /* character_8_type */
	T0* volatile a15; /* character_32_type */
	T0* volatile a16; /* immutable_string_8_type */
	T0* volatile a17; /* immutable_string_32_type */
	T0* volatile a18; /* integer_8_type */
	T0* volatile a19; /* integer_16_type */
	T0* volatile a20; /* integer_32_type */
	T0* volatile a21; /* integer_64_type */
	T0* volatile a22; /* natural_8_type */
	T0* volatile a23; /* natural_16_type */
	T0* volatile a24; /* natural_32_type */
	T0* volatile a25; /* natural_64_type */
	T0* volatile a26; /* real_32_type */
	T0* volatile a27; /* real_64_type */
	T0* volatile a28; /* pointer_type */
	T0* volatile a29; /* string_8_type */
	T0* volatile a30; /* string_32_type */
	T0* volatile a31; /* special_character_8_type */
	T0* volatile a32; /* special_character_32_type */
	T0* volatile a33; /* type_none_type */
	T0* volatile a34; /* ise_exception_manager_type */
	T0* volatile a35; /* any_type */
	T0* volatile a36; /* none_type */
	T0* volatile a37; /* root_type */
	T0* volatile a38; /* root_creation_procedure */
	T0* volatile a39; /* special_count_feature */
	T0* volatile a40; /* special_capacity_feature */
	T0* volatile a41; /* special_item_feature */
	T0* volatile a42; /* array_area_feature */
	T0* volatile a43; /* array_lower_feature */
	T0* volatile a44; /* array_upper_feature */
	T0* volatile a45; /* typed_pointer_to_pointer_feature */
	T0* volatile a46; /* routine_closed_operands_feature */
	T0* volatile a47; /* routine_rout_disp_feature */
	T0* volatile a48; /* routine_set_rout_disp_final_feature */
	T0* volatile a49; /* ise_exception_manager_init_exception_manager_feature */
	T0* volatile a50; /* ise_exception_manager_last_exception_feature */
	T0* volatile a51; /* ise_exception_manager_once_raise_feature */
	T0* volatile a52; /* ise_exception_manager_set_exception_data_feature */
	T1 volatile a53; /* has_special_once_per_object_routines */
	T0* volatile a54; /* ise_runtime_type_conforms_to_feature */
	T1 volatile a55; /* in_dynamic_primary_type */
	T0* volatile a56; /* ise_runtime_new_type_instance_of_feature */
	T1 volatile a57; /* in_create_meta_type */
	T1 volatile a58; /* all_attributes_used */
	T1 volatile a59; /* has_separate_creation */
	T0* volatile a60; /* ise_runtime_reference_field_feature */
	T0* volatile a61; /* ise_runtime_set_reference_field_feature */
};

/* Struct for type DT_DATE_TIME */
struct S97 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* time_storage */
	T6 volatile a2; /* date_storage */
};

/* Struct for type DS_HASH_SET [STRING_8] */
struct S99 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* internal_cursor */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* equality_tester */
	T6 volatile a5; /* count */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* free_slot */
	T6 volatile a9; /* last_position */
	T6 volatile a10; /* modulus */
	T6 volatile a11; /* clashes_previous_position */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* hash_function */
};

/* Struct for type ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
struct S100 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* dynamic_type_set_count */
	T1 volatile a2; /* has_fatal_error */
	T0* volatile a3; /* current_dynamic_system */
	T0* volatile a4; /* feature_checker */
	T0* volatile a5; /* current_dynamic_type */
	T0* volatile a6; /* current_dynamic_feature */
	T0* volatile a7; /* dynamic_type_sets */
	T0* volatile a8; /* object_id_dynamic_type_set */
	T0* volatile a9; /* precursors */
	T0* volatile a10; /* dummy_operand */
	T1 volatile a11; /* catcall_error_mode */
	T1 volatile a12; /* catcall_warning_mode */
	T1 volatile a13; /* no_debug */
	T1 volatile a14; /* is_built */
	T0* volatile a15; /* alive_conforming_descendants_per_type */
	T0* volatile a16; /* type_checker */
	T0* volatile a17; /* current_type */
	T0* volatile a18; /* current_class */
	T0* volatile a19; /* current_feature */
	T0* volatile a20; /* current_feature_impl */
	T0* volatile a21; /* current_class_impl */
	T0* volatile a22; /* enclosing_inline_agents */
	T0* volatile a23; /* unused_overloaded_procedures_list */
	T0* volatile a24; /* unused_overloaded_queries_list */
	T0* volatile a25; /* unused_overloaded_features_list */
	T0* volatile a26; /* unused_call_infos */
	T0* volatile a27; /* unused_contexts */
	T0* volatile a28; /* current_context */
	T0* volatile a29; /* current_target_type */
	T0* volatile a30; /* current_object_test_types */
	T0* volatile a31; /* current_object_test_scope */
	T0* volatile a32; /* object_test_scope_builder */
	T0* volatile a33; /* current_iteration_cursor_types */
	T0* volatile a34; /* current_iteration_item_types */
	T0* volatile a35; /* current_iteration_item_scope */
	T0* volatile a36; /* current_inline_separate_argument_types */
	T0* volatile a37; /* current_inline_separate_argument_scope */
	T0* volatile a38; /* current_initialization_scope */
	T0* volatile a39; /* current_attachment_scope */
	T0* volatile a40; /* attachment_scope_builder */
	T0* volatile a41; /* unused_attachment_scopes */
	T0* volatile a42; /* assertions_by_feature */
	T0* volatile a43; /* common_ancestor_type_list */
	T0* volatile a44; /* note_term_list */
	T0* volatile a45; /* vape_non_descendant_clients */
	T0* volatile a46; /* vape_creation_clients */
	T0* volatile a47; /* vape_client */
	T0* volatile a48; /* unused_adapted_base_classes */
	T0* volatile a49; /* adapted_base_class_checker */
	T0* volatile a50; /* system_processor */
	T1 volatile a51; /* in_precursor */
	T1 volatile a52; /* in_static_feature */
	T6 volatile a53; /* index_offset */
	T6 volatile a54; /* index_count */
	T6 volatile a55; /* result_index */
	T6 volatile a56; /* attached_result_index */
	T6 volatile a57; /* current_index */
	T6 volatile a58; /* boolean_index */
	T6 volatile a59; /* character_8_index */
	T6 volatile a60; /* character_32_index */
	T6 volatile a61; /* integer_8_index */
	T6 volatile a62; /* integer_16_index */
	T6 volatile a63; /* integer_32_index */
	T6 volatile a64; /* integer_64_index */
	T6 volatile a65; /* natural_8_index */
	T6 volatile a66; /* natural_16_index */
	T6 volatile a67; /* natural_32_index */
	T6 volatile a68; /* natural_64_index */
	T6 volatile a69; /* pointer_index */
	T6 volatile a70; /* real_32_index */
	T6 volatile a71; /* real_64_index */
	T6 volatile a72; /* string_8_index */
	T6 volatile a73; /* string_32_index */
	T6 volatile a74; /* immutable_string_8_index */
	T6 volatile a75; /* immutable_string_32_index */
	T6 volatile a76; /* void_index */
	T1 volatile a77; /* in_precondition */
	T1 volatile a78; /* in_postcondition */
	T1 volatile a79; /* in_invariant */
	T0* volatile a80; /* supplier_handler */
	T0* volatile a81; /* current_inline_agent */
	T1 volatile a82; /* in_check_instruction */
	T1 volatile a83; /* in_rescue */
	T0* volatile a84; /* precursor_queries */
	T0* volatile a85; /* precursor_procedures */
};

/* Struct for type ET_CLASS_TYPE */
struct S102 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type_mark */
	T0* volatile a2; /* name */
	T0* volatile a3; /* named_base_class */
	T0* volatile a4; /* actual_parameters */
	T1 volatile a5; /* tuple_actual_parameters_unfolded_2 */
	T1 volatile a6; /* tuple_actual_parameters_unfolded_1 */
};

/* Struct for type UT_VERSION_NUMBER */
struct S103 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type EXECUTION_ENVIRONMENT */
struct S105 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* return_code */
};

/* Struct for type UT_MESSAGE */
struct S106 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type RX_PCRE_REGULAR_EXPRESSION */
struct S107 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* match_count */
	T0* volatile a2; /* subject */
	T0* volatile a3; /* offset_vector */
	T6 volatile a4; /* offset_vector_count */
	T6 volatile a5; /* brastart_capacity */
	T0* volatile a6; /* brastart_vector */
	T6 volatile a7; /* brastart_lower */
	T6 volatile a8; /* brastart_count */
	T6 volatile a9; /* eptr_capacity */
	T0* volatile a10; /* eptr_vector */
	T6 volatile a11; /* eptr_lower */
	T6 volatile a12; /* eptr_upper */
	T0* volatile a13; /* byte_code */
	T0* volatile a14; /* internal_start_bits */
	T0* volatile a15; /* pattern_buffer */
	T6 volatile a16; /* subexpression_count */
	T0* volatile a17; /* pattern */
	T6 volatile a18; /* pattern_count */
	T6 volatile a19; /* pattern_position */
	T6 volatile a20; /* code_index */
	T6 volatile a21; /* maxbackrefs */
	T10 volatile a22; /* optchanged */
	T7 volatile a23; /* first_character */
	T7 volatile a24; /* required_character */
	T6 volatile a25; /* regexp_countlits */
	T0* volatile a26; /* start_bits */
	T0* volatile a27; /* error_message */
	T1 volatile a28; /* is_anchored */
	T0* volatile a29; /* character_case_mapping */
	T0* volatile a30; /* word_set */
	T6 volatile a31; /* subject_next_start */
	T6 volatile a32; /* subject_start */
	T6 volatile a33; /* subject_end */
	T6 volatile a34; /* error_code */
	T6 volatile a35; /* error_position */
	T1 volatile a36; /* is_startline */
	T1 volatile a37; /* is_caseless */
	T1 volatile a38; /* is_multiline */
	T1 volatile a39; /* is_dotall */
	T1 volatile a40; /* is_extended */
	T1 volatile a41; /* is_empty_allowed */
	T1 volatile a42; /* is_dollar_endonly */
	T1 volatile a43; /* is_bol */
	T1 volatile a44; /* is_eol */
	T1 volatile a45; /* is_greedy */
	T1 volatile a46; /* is_strict */
	T1 volatile a47; /* is_ichanged */
	T6 volatile a48; /* first_matched_index */
	T6 volatile a49; /* eptr */
	T6 volatile a50; /* offset_top */
	T1 volatile a51; /* is_matching_caseless */
	T1 volatile a52; /* is_matching_multiline */
	T1 volatile a53; /* is_matching_dotall */
};

/* Struct for type KL_STRING_ROUTINES */
struct S108 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ST_SPLITTER */
struct S109 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_escape_character */
	T2 volatile a2; /* escape_character */
	T0* volatile a3; /* separator_codes */
	T0* volatile a4; /* separators */
};

/* Struct for type AP_ERROR */
struct S113 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
	T0* volatile a2; /* default_template */
	T0* volatile a3; /* code */
};

/* Struct for type ET_C_GENERATOR */
struct S114 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* never_void_target_count */
	T6 volatile a2; /* can_be_void_target_count */
	T1 volatile a3; /* has_fatal_error */
	T0* volatile a4; /* system_processor */
	T0* volatile a5; /* current_dynamic_system */
	T0* volatile a6; /* dynamic_types */
	T1 volatile a7; /* short_names */
	T1 volatile a8; /* split_mode */
	T6 volatile a9; /* split_threshold */
	T1 volatile a10; /* use_boehm_gc */
	T0* volatile a11; /* system_name */
	T0* volatile a12; /* current_file */
	T0* volatile a13; /* header_file */
	T0* volatile a14; /* current_type */
	T0* volatile a15; /* current_feature */
	T0* volatile a16; /* current_non_inlined_feature */
	T0* volatile a17; /* extra_dynamic_type_sets */
	T0* volatile a18; /* current_dynamic_type_sets */
	T0* volatile a19; /* current_function_header_buffer */
	T0* volatile a20; /* current_function_body_buffer */
	T0* volatile a21; /* current_separate_function_header_buffer */
	T0* volatile a22; /* current_separate_function_body_buffer */
	T0* volatile a23; /* old_expressions */
	T0* volatile a24; /* old_expression_temp_variables */
	T0* volatile a25; /* old_expression_exception_temp_variables */
	T0* volatile a26; /* temp_variables */
	T0* volatile a27; /* used_temp_variables */
	T0* volatile a28; /* free_temp_variables */
	T0* volatile a29; /* frozen_temp_variables */
	T0* volatile a30; /* volatile_temp_variables */
	T0* volatile a31; /* separate_temp_variables */
	T0* volatile a32; /* used_separate_temp_variables */
	T0* volatile a33; /* free_separate_temp_variables */
	T0* volatile a34; /* frozen_separate_temp_variables */
	T0* volatile a35; /* volatile_separate_temp_variables */
	T0* volatile a36; /* unused_equality_expressions */
	T0* volatile a37; /* unused_equality_types */
	T0* volatile a38; /* unused_object_equality_types */
	T0* volatile a39; /* unused_unqualified_call_expressions */
	T0* volatile a40; /* unused_qualified_call_expressions */
	T0* volatile a41; /* unused_actual_argument_lists */
	T0* volatile a42; /* unused_result_expressions */
	T0* volatile a43; /* unused_identifiers */
	T0* volatile a44; /* conforming_types */
	T0* volatile a45; /* non_conforming_types */
	T0* volatile a46; /* equality_type_set */
	T0* volatile a47; /* equality_common_types */
	T0* volatile a48; /* equality_common_reference_types */
	T0* volatile a49; /* unused_standalone_type_sets */
	T0* volatile a50; /* operand_stack */
	T0* volatile a51; /* call_operands */
	T0* volatile a52; /* attachment_dynamic_type_ids */
	T0* volatile a53; /* target_dynamic_type_ids */
	T0* volatile a54; /* target_dynamic_types */
	T0* volatile a55; /* standalone_type_sets */
	T0* volatile a56; /* deep_twin_types */
	T0* volatile a57; /* deep_equal_types */
	T0* volatile a58; /* deep_feature_target_type_sets */
	T0* volatile a59; /* current_object_tests */
	T0* volatile a60; /* current_object_test_local_types */
	T0* volatile a61; /* current_object_test_locals */
	T0* volatile a62; /* object_test_local_name */
	T0* volatile a63; /* current_iteration_cursor_types */
	T0* volatile a64; /* current_iteration_cursors */
	T0* volatile a65; /* iteration_cursor_name */
	T0* volatile a66; /* current_inline_separate_arguments */
	T0* volatile a67; /* current_equalities */
	T0* volatile a68; /* current_agents */
	T0* volatile a69; /* agent_open_operands */
	T0* volatile a70; /* agent_closed_operands */
	T0* volatile a71; /* agent_target */
	T0* volatile a72; /* agent_arguments */
	T0* volatile a73; /* agent_instruction */
	T0* volatile a74; /* agent_expression */
	T0* volatile a75; /* agent_inline_separate_argument */
	T0* volatile a76; /* agent_controlled_target_instruction */
	T0* volatile a77; /* agent_expression_assignment */
	T0* volatile a78; /* agent_controlled_target_expression */
	T0* volatile a79; /* agent_closed_operands_type */
	T0* volatile a80; /* agent_manifest_tuple */
	T0* volatile a81; /* separate_call_arguments */
	T0* volatile a82; /* separate_call_instruction */
	T0* volatile a83; /* separate_call_expression */
	T0* volatile a84; /* separate_arguments */
	T0* volatile a85; /* formal_arguments */
	T0* volatile a86; /* manifest_array_types */
	T0* volatile a87; /* big_manifest_array_types */
	T0* volatile a88; /* manifest_tuple_types */
	T0* volatile a89; /* once_features */
	T0* volatile a90; /* once_per_process_counts */
	T0* volatile a91; /* once_per_thread_counts */
	T0* volatile a92; /* constant_features */
	T0* volatile a93; /* inline_constants */
	T0* volatile a94; /* dispose_procedures */
	T0* volatile a95; /* dynamic_type_id_set_names */
	T0* volatile a96; /* used_inlined_operands */
	T0* volatile a97; /* unused_inlined_operands */
	T0* volatile a98; /* used_call_contexts */
	T0* volatile a99; /* unused_call_contexts */
	T6 volatile a100; /* max_nested_inlining_count */
	T0* volatile a101; /* inlined_void */
	T0* volatile a102; /* inlined_integer */
	T0* volatile a103; /* inlined_real */
	T0* volatile a104; /* inlined_character */
	T0* volatile a105; /* inlined_boolean */
	T0* volatile a106; /* called_features */
	T0* volatile a107; /* inlined_features */
	T0* volatile a108; /* included_header_filenames */
	T0* volatile a109; /* included_cpp_header_filenames */
	T0* volatile a110; /* included_runtime_header_files */
	T0* volatile a111; /* included_runtime_c_files */
	T0* volatile a112; /* c_filenames */
	T1 volatile a113; /* scoop_exceptions_stop_when_dirty */
	T1 volatile a114; /* finalize_mode */
	T0* volatile a115; /* locals_written_in_body */
	T0* volatile a116; /* locals_written_in_rescue */
	T0* volatile a117; /* locals_written */
	T0* volatile a118; /* locals_read_in_body */
	T0* volatile a119; /* locals_read_in_rescue */
	T0* volatile a120; /* locals_read */
	T0* volatile a121; /* volatile_locals */
	T0* volatile a122; /* volatile_object_test_locals */
	T0* volatile a123; /* volatile_iteration_cursors */
	T0* volatile a124; /* volatile_inline_separate_arguments */
	T0* volatile a125; /* external_c_regexp */
	T0* volatile a126; /* external_c_macro_regexp */
	T0* volatile a127; /* external_c_struct_regexp */
	T0* volatile a128; /* external_c_inline_regexp */
	T0* volatile a129; /* old_external_c_regexp */
	T0* volatile a130; /* old_external_c_macro_regexp */
	T0* volatile a131; /* old_external_c_struct_regexp */
	T0* volatile a132; /* external_cpp_regexp */
	T0* volatile a133; /* external_cpp_macro_regexp */
	T0* volatile a134; /* external_cpp_struct_regexp */
	T0* volatile a135; /* external_cpp_inline_regexp */
	T0* volatile a136; /* external_dllwin_regexp */
	T0* volatile a137; /* old_external_dllwin32_regexp */
	T1 volatile a138; /* type_info_ancestors_used */
	T1 volatile a139; /* type_info_attributes_used */
	T1 volatile a140; /* type_info_attribute_name_used */
	T1 volatile a141; /* type_info_attribute_type_id_used */
	T1 volatile a142; /* type_info_attribute_offset_used */
	T1 volatile a143; /* type_info_generator_used */
	T1 volatile a144; /* type_info_name_used */
	T1 volatile a145; /* type_info_generic_parameters_used */
	T1 volatile a146; /* type_info_object_size_used */
	T6 volatile a147; /* current_index */
	T6 volatile a148; /* index_offset */
	T0* volatile a149; /* c_file */
	T6 volatile a150; /* c_file_size */
	T6 volatile a151; /* current_separate_call_count */
	T0* volatile a152; /* cpp_file */
	T6 volatile a153; /* cpp_file_size */
	T6 volatile a154; /* indentation */
	T0* volatile a155; /* call_target_type */
	T1 volatile a156; /* in_qualified_call */
	T0* volatile a157; /* assignment_target */
	T1 volatile a158; /* in_operand */
	T1 volatile a159; /* in_procedure_call_target */
	T0* volatile a160; /* inlining_context */
	T1 volatile a161; /* has_rescue */
	T1 volatile a162; /* result_written */
	T1 volatile a163; /* result_read */
	T0* volatile a164; /* current_agent */
	T1 volatile a165; /* in_separate_creation_call */
	T1 volatile a166; /* call_target_check_void */
	T6 volatile a167; /* nested_inlining_count */
	T1 volatile a168; /* volatile_result */
	T1 volatile a169; /* has_retry */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE] */
struct S116 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type KL_SHELL_COMMAND */
struct S117 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* exit_code */
	T0* volatile a2; /* command */
	T6 volatile a3; /* return_code */
	T1 volatile a4; /* is_system_code */
};

/* Struct for type GECC */
struct S118 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* exit_code */
	T0* volatile a2; /* error_handler */
	T0* volatile a3; /* version_flag */
	T0* volatile a4; /* script_filename */
	T0* volatile a5; /* thread_option */
};

/* Struct for type KL_OPERATING_SYSTEM */
struct S119 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type VOID_TARGET */
struct S120 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type ROUTINE_FAILURE */
struct S122 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* routine_name */
	T0* volatile a4; /* class_name */
	T0* volatile a5; /* c_description */
	T0* volatile a6; /* recipient_name */
	T0* volatile a7; /* type_name */
};

/* Struct for type OLD_VIOLATION */
struct S124 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type NO_MORE_MEMORY */
struct S126 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* internal_code */
	T0* volatile a4; /* c_description */
	T0* volatile a5; /* recipient_name */
	T0* volatile a6; /* type_name */
};

/* Struct for type INVARIANT_VIOLATION */
struct S127 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T1 volatile a3; /* is_entry */
	T0* volatile a4; /* c_description */
	T0* volatile a5; /* recipient_name */
	T0* volatile a6; /* type_name */
};

/* Struct for type OPERATING_SYSTEM_SIGNAL_FAILURE */
struct S128 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* signal_code */
	T0* volatile a4; /* c_description */
	T0* volatile a5; /* recipient_name */
	T0* volatile a6; /* type_name */
};

/* Struct for type IO_FAILURE */
struct S129 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* internal_code */
	T6 volatile a4; /* error_code */
	T0* volatile a5; /* c_description */
	T0* volatile a6; /* recipient_name */
	T0* volatile a7; /* type_name */
};

/* Struct for type OPERATING_SYSTEM_FAILURE */
struct S130 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* error_code */
	T0* volatile a4; /* c_description */
	T0* volatile a5; /* recipient_name */
	T0* volatile a6; /* type_name */
};

/* Struct for type COM_FAILURE */
struct S131 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* hresult_code */
	T0* volatile a4; /* exception_information */
	T6 volatile a5; /* hresult */
	T0* volatile a6; /* c_description */
	T0* volatile a7; /* recipient_name */
	T0* volatile a8; /* type_name */
};

/* Struct for type EIFFEL_RUNTIME_PANIC */
struct S132 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T6 volatile a3; /* internal_code */
	T0* volatile a4; /* c_description */
	T0* volatile a5; /* recipient_name */
	T0* volatile a6; /* type_name */
};

/* Struct for type PRECONDITION_VIOLATION */
struct S134 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type POSTCONDITION_VIOLATION */
struct S135 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type FLOATING_POINT_FAILURE */
struct S136 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type CHECK_VIOLATION */
struct S137 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type BAD_INSPECT_VALUE */
struct S138 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type VARIANT_VIOLATION */
struct S139 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type LOOP_INVARIANT_VIOLATION */
struct S140 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type RESCUE_FAILURE */
struct S141 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type RESUMPTION_FAILURE */
struct S142 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type CREATE_ON_DEFERRED */
struct S143 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type EXTERNAL_FAILURE */
struct S144 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type VOID_ASSIGNED_TO_EXPANDED */
struct S145 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
struct S146 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type MISMATCH_FAILURE */
struct S147 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type DEVELOPER_EXCEPTION */
struct S148 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type ADDRESS_APPLIED_TO_MELTED_FEATURE */
struct S149 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type SERIALIZATION_FAILURE */
struct S150 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* throwing_exception */
	T0* volatile a2; /* internal_trace */
	T0* volatile a3; /* c_description */
	T0* volatile a4; /* recipient_name */
	T0* volatile a5; /* type_name */
};

/* Struct for type KL_WINDOWS_FILE_SYSTEM */
struct S151 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_UNIX_FILE_SYSTEM */
struct S152 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type PRIMES */
struct S153 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [INTEGER_32] */
struct S154 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T6 volatile z2[]; /* item */
};

/* Struct for type SPECIAL [BOOLEAN] */
struct S155 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T1 volatile z2[]; /* item */
};

/* Struct for type ARGUMENTS_32 */
struct S156 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type MUTEX */
struct S159 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* mutex_pointer */
	T14 volatile a2; /* owner_thread_id */
};

/* Struct for type UT_ERROR_HANDLER */
struct S160 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* error_file */
	T0* volatile a2; /* warning_file */
	T0* volatile a3; /* info_file */
};

/* Struct for type KL_STDOUT_FILE */
struct S161 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type MANAGED_POINTER */
struct S162 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* item */
	T1 volatile a2; /* is_shared */
	T6 volatile a3; /* count */
};

/* Struct for type KL_LINKABLE [CHARACTER_8] */
struct S164 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type FILE_INFO */
struct S166 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* buffered_file_info */
	T1 volatile a2; /* is_following_symlinks */
	T1 volatile a3; /* exists */
	T0* volatile a4; /* internal_file_name */
	T0* volatile a5; /* internal_name_pointer */
};

/* Struct for type KL_DIRECTORY */
struct S167 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* end_of_input */
	T0* volatile a2; /* last_entry */
	T0* volatile a3; /* name */
	T6 volatile a4; /* mode */
	T0* volatile a5; /* internal_detachable_name_pointer */
	T0* volatile a6; /* string_name */
	T0* volatile a7; /* entry_buffer */
	T1 volatile a8; /* old_end_of_input */
	T14 volatile a9; /* directory_pointer */
	T14 volatile a10; /* last_entry_pointer */
	T0* volatile a11; /* lastentry */
	T0* volatile a12; /* internal_name */
};

/* Struct for type UC_UTF8_ROUTINES */
struct S168 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [STRING_8] */
struct S170 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
	T0* volatile a6; /* equality_tester */
};

/* Struct for type TUPLE [ET_UNIVERSE] */
struct S171 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_UNIVERSE]] */
struct S172 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE [ET_CLASS], BOOLEAN]] */
struct S173 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type ET_ADAPTED_LIBRARIES */
struct S174 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* libraries */
};

/* Struct for type ET_ADAPTED_DOTNET_ASSEMBLIES */
struct S175 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* dotnet_assemblies */
};

/* Struct for type ET_IMPLICIT_TYPE_MARK */
struct S177 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* is_expanded_mark */
	T1 volatile a2; /* is_separate_mark */
	T1 volatile a3; /* is_attached_mark */
	T1 volatile a4; /* is_reference_mark */
	T1 volatile a5; /* is_detachable_mark */
	T1 volatile a6; /* is_non_separate_mark */
};

/* Struct for type DS_HASH_TABLE [ET_MASTER_CLASS, ET_CLASS_NAME] */
struct S178 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T0* volatile a4; /* internal_keys */
	T6 volatile a5; /* position */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* count */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* modulus */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_CLASS_NAME_TESTER */
struct S179 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_PARENT */
struct S181 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* renames */
	T0* volatile a3; /* exports */
	T0* volatile a4; /* undefines */
	T0* volatile a5; /* redefines */
	T0* volatile a6; /* selects */
	T0* volatile a7; /* end_keyword */
};

/* Struct for type ET_PARENT_LIST */
struct S182 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* inherit_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* clients_clause */
};

/* Struct for type ET_CLIENT_LIST */
struct S183 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_TUPLE_TYPE */
struct S184 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* actual_parameters */
	T0* volatile a2; /* type_mark */
	T0* volatile a3; /* named_base_class */
	T0* volatile a4; /* tuple_keyword */
};

/* Struct for type ET_ACTUAL_PARAMETER_LIST */
struct S186 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* left_bracket */
	T0* volatile a4; /* right_bracket */
};

/* Struct for type ET_BUILTIN_CONVERT_FEATURE */
struct S187 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* types */
	T0* volatile a3; /* type */
};

/* Struct for type DS_HASH_SET [ET_UNIVERSE] */
struct S188 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* capacity */
	T6 volatile a3; /* modulus */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* clashes_previous_position */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* item_storage */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T0* volatile a14; /* internal_cursor */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type ET_MASTER_CLASS */
struct S189 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* processing_mutex */
	T0* volatile a2; /* first_local_override_class */
	T0* volatile a3; /* first_local_non_override_class */
	T0* volatile a4; /* first_local_ignored_class */
	T0* volatile a5; /* other_local_override_classes */
	T0* volatile a6; /* other_local_non_override_classes */
	T0* volatile a7; /* other_local_ignored_classes */
	T0* volatile a8; /* first_local_hidden_class */
	T0* volatile a9; /* other_local_hidden_classes */
	T0* volatile a10; /* mapped_class */
	T0* volatile a11; /* first_overriding_class */
	T0* volatile a12; /* intrinsic_class */
	T0* volatile a13; /* name */
	T0* volatile a14; /* universe */
	T0* volatile a15; /* status_mutex */
	T1 volatile a16; /* unprotected_is_marked */
	T1 volatile a17; /* is_modified */
	T0* volatile a18; /* first_imported_class */
	T0* volatile a19; /* other_imported_classes */
	T0* volatile a20; /* other_overriding_classes */
};

/* Struct for type ET_CLIENT */
struct S190 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* named_base_class */
};

/* Struct for type ET_KEYWORD */
struct S192 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T0* volatile a2; /* text */
	T6 volatile a3; /* compressed_position */
	T0* volatile a4; /* break */
};

/* Struct for type ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
struct S193 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* separateness_keyword */
	T0* volatile a2; /* attachment_mark */
};

/* Struct for type ET_RENAME_LIST */
struct S194 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* rename_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type ET_EXPORT_LIST */
struct S195 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* export_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type ET_KEYWORD_FEATURE_NAME_LIST */
struct S196 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type ET_NONE_GROUP */
struct S199 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* universe */
	T1 volatile a3; /* use_obsolete_syntax */
	T0* volatile a4; /* cached_absolute_pathname */
};

/* Struct for type ET_LIKE_CURRENT */
struct S200 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type_mark */
	T0* volatile a2; /* like_keyword */
	T0* volatile a3; /* current_keyword */
	T1 volatile a4; /* is_controlled */
};

/* Struct for type ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
struct S202 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* tuple_type */
};

/* Struct for type ET_LIBRARY */
struct S204 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_system */
	T0* volatile a2; /* library */
	T0* volatile a3; /* name */
	T0* volatile a4; /* libraries */
	T0* volatile a5; /* clusters */
	T0* volatile a6; /* dotnet_assemblies */
	T0* volatile a7; /* implicit_attachment_type_mark */
	T0* volatile a8; /* master_classes */
	T0* volatile a9; /* master_class_mutex */
	T0* volatile a10; /* any_type */
	T0* volatile a11; /* detachable_any_type */
	T0* volatile a12; /* detachable_separate_any_type */
	T0* volatile a13; /* any_parent */
	T0* volatile a14; /* any_parents */
	T0* volatile a15; /* any_clients */
	T0* volatile a16; /* tuple_type */
	T0* volatile a17; /* detachable_tuple_type */
	T0* volatile a18; /* tuple_identity_type */
	T0* volatile a19; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a20; /* array_any_type */
	T0* volatile a21; /* array_detachable_any_type */
	T0* volatile a22; /* array_none_type */
	T0* volatile a23; /* array_identity_type */
	T0* volatile a24; /* boolean_type */
	T0* volatile a25; /* character_type */
	T0* volatile a26; /* character_8_type */
	T0* volatile a27; /* character_8_convert_feature */
	T0* volatile a28; /* character_32_type */
	T0* volatile a29; /* character_32_convert_feature */
	T0* volatile a30; /* double_type */
	T0* volatile a31; /* exception_type */
	T0* volatile a32; /* detachable_exception_type */
	T0* volatile a33; /* exception_manager_type */
	T0* volatile a34; /* function_identity_any_type */
	T0* volatile a35; /* immutable_string_8_type */
	T0* volatile a36; /* immutable_string_32_type */
	T0* volatile a37; /* integer_type */
	T0* volatile a38; /* integer_8_type */
	T0* volatile a39; /* integer_8_convert_feature */
	T0* volatile a40; /* integer_16_type */
	T0* volatile a41; /* integer_16_convert_feature */
	T0* volatile a42; /* integer_32_type */
	T0* volatile a43; /* integer_32_convert_feature */
	T0* volatile a44; /* integer_64_type */
	T0* volatile a45; /* integer_64_convert_feature */
	T0* volatile a46; /* ise_exception_manager_type */
	T0* volatile a47; /* iterable_detachable_separate_any_type */
	T0* volatile a48; /* natural_type */
	T0* volatile a49; /* natural_8_type */
	T0* volatile a50; /* natural_8_convert_feature */
	T0* volatile a51; /* natural_16_type */
	T0* volatile a52; /* natural_16_convert_feature */
	T0* volatile a53; /* natural_32_type */
	T0* volatile a54; /* natural_32_convert_feature */
	T0* volatile a55; /* natural_64_type */
	T0* volatile a56; /* natural_64_convert_feature */
	T0* volatile a57; /* none_type */
	T0* volatile a58; /* detachable_none_type */
	T0* volatile a59; /* pointer_type */
	T0* volatile a60; /* predicate_identity_type */
	T0* volatile a61; /* procedure_identity_type */
	T0* volatile a62; /* real_type */
	T0* volatile a63; /* real_32_type */
	T0* volatile a64; /* real_32_convert_feature */
	T0* volatile a65; /* real_64_type */
	T0* volatile a66; /* real_64_convert_feature */
	T0* volatile a67; /* routine_identity_type */
	T0* volatile a68; /* special_any_type */
	T0* volatile a69; /* special_detachable_any_type */
	T0* volatile a70; /* special_identity_type */
	T0* volatile a71; /* string_type */
	T0* volatile a72; /* detachable_string_type */
	T0* volatile a73; /* string_8_type */
	T0* volatile a74; /* detachable_string_8_type */
	T0* volatile a75; /* string_8_convert_feature */
	T0* volatile a76; /* string_32_type */
	T0* volatile a77; /* string_32_convert_feature */
	T0* volatile a78; /* system_object_type */
	T0* volatile a79; /* system_object_parents */
	T0* volatile a80; /* system_string_type */
	T0* volatile a81; /* type_detachable_any_type */
	T0* volatile a82; /* detachable_type_exception_type */
	T0* volatile a83; /* type_detachable_exception_type */
	T0* volatile a84; /* type_detachable_like_current_type */
	T0* volatile a85; /* type_identity_type */
	T0* volatile a86; /* typed_pointer_identity_type */
	T0* volatile a87; /* wide_character_type */
	T1 volatile a88; /* is_read_only */
	T1 volatile a89; /* is_preparsed */
};

/* Struct for type TUPLE [ET_LIBRARY] */
struct S205 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_LIBRARY]] */
struct S206 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [DS_HASH_SET [ET_UNIVERSE]] */
struct S207 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type ET_DOTNET_ASSEMBLY */
struct S208 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* referenced_assemblies */
	T0* volatile a3; /* current_system */
	T0* volatile a4; /* dotnet_assembly */
	T0* volatile a5; /* name */
	T0* volatile a6; /* implicit_attachment_type_mark */
	T0* volatile a7; /* master_classes */
	T0* volatile a8; /* master_class_mutex */
	T0* volatile a9; /* any_type */
	T0* volatile a10; /* detachable_any_type */
	T0* volatile a11; /* detachable_separate_any_type */
	T0* volatile a12; /* any_parent */
	T0* volatile a13; /* any_parents */
	T0* volatile a14; /* any_clients */
	T0* volatile a15; /* tuple_type */
	T0* volatile a16; /* detachable_tuple_type */
	T0* volatile a17; /* tuple_identity_type */
	T0* volatile a18; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a19; /* array_any_type */
	T0* volatile a20; /* array_detachable_any_type */
	T0* volatile a21; /* array_none_type */
	T0* volatile a22; /* array_identity_type */
	T0* volatile a23; /* boolean_type */
	T0* volatile a24; /* character_type */
	T0* volatile a25; /* character_8_type */
	T0* volatile a26; /* character_8_convert_feature */
	T0* volatile a27; /* character_32_type */
	T0* volatile a28; /* character_32_convert_feature */
	T0* volatile a29; /* double_type */
	T0* volatile a30; /* exception_type */
	T0* volatile a31; /* detachable_exception_type */
	T0* volatile a32; /* exception_manager_type */
	T0* volatile a33; /* function_identity_any_type */
	T0* volatile a34; /* immutable_string_8_type */
	T0* volatile a35; /* immutable_string_32_type */
	T0* volatile a36; /* integer_type */
	T0* volatile a37; /* integer_8_type */
	T0* volatile a38; /* integer_8_convert_feature */
	T0* volatile a39; /* integer_16_type */
	T0* volatile a40; /* integer_16_convert_feature */
	T0* volatile a41; /* integer_32_type */
	T0* volatile a42; /* integer_32_convert_feature */
	T0* volatile a43; /* integer_64_type */
	T0* volatile a44; /* integer_64_convert_feature */
	T0* volatile a45; /* ise_exception_manager_type */
	T0* volatile a46; /* iterable_detachable_separate_any_type */
	T0* volatile a47; /* natural_type */
	T0* volatile a48; /* natural_8_type */
	T0* volatile a49; /* natural_8_convert_feature */
	T0* volatile a50; /* natural_16_type */
	T0* volatile a51; /* natural_16_convert_feature */
	T0* volatile a52; /* natural_32_type */
	T0* volatile a53; /* natural_32_convert_feature */
	T0* volatile a54; /* natural_64_type */
	T0* volatile a55; /* natural_64_convert_feature */
	T0* volatile a56; /* none_type */
	T0* volatile a57; /* detachable_none_type */
	T0* volatile a58; /* pointer_type */
	T0* volatile a59; /* predicate_identity_type */
	T0* volatile a60; /* procedure_identity_type */
	T0* volatile a61; /* real_type */
	T0* volatile a62; /* real_32_type */
	T0* volatile a63; /* real_32_convert_feature */
	T0* volatile a64; /* real_64_type */
	T0* volatile a65; /* real_64_convert_feature */
	T0* volatile a66; /* routine_identity_type */
	T0* volatile a67; /* special_any_type */
	T0* volatile a68; /* special_detachable_any_type */
	T0* volatile a69; /* special_identity_type */
	T0* volatile a70; /* string_type */
	T0* volatile a71; /* detachable_string_type */
	T0* volatile a72; /* string_8_type */
	T0* volatile a73; /* detachable_string_8_type */
	T0* volatile a74; /* string_8_convert_feature */
	T0* volatile a75; /* string_32_type */
	T0* volatile a76; /* string_32_convert_feature */
	T0* volatile a77; /* system_object_type */
	T0* volatile a78; /* system_object_parents */
	T0* volatile a79; /* system_string_type */
	T0* volatile a80; /* type_detachable_any_type */
	T0* volatile a81; /* detachable_type_exception_type */
	T0* volatile a82; /* type_detachable_exception_type */
	T0* volatile a83; /* type_detachable_like_current_type */
	T0* volatile a84; /* type_identity_type */
	T0* volatile a85; /* typed_pointer_identity_type */
	T0* volatile a86; /* wide_character_type */
	T1 volatile a87; /* is_read_only */
	T1 volatile a88; /* is_preparsed */
};

/* Struct for type TUPLE [ET_DOTNET_ASSEMBLY] */
struct S209 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]] */
struct S210 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type ST_WORD_WRAPPER */
struct S211 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* new_line_indentation */
	T6 volatile a2; /* broken_words */
	T6 volatile a3; /* maximum_text_width */
};

/* Struct for type AP_DISPLAY_HELP_FLAG */
struct S213 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* description */
	T2 volatile a2; /* short_form */
	T1 volatile a3; /* has_short_form */
	T0* volatile a4; /* long_form */
	T6 volatile a5; /* occurrences */
	T1 volatile a6; /* is_hidden */
	T1 volatile a7; /* is_mandatory */
	T6 volatile a8; /* maximum_occurrences */
};

/* Struct for type DS_ARRAYED_LIST [AP_OPTION] */
struct S214 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* count */
	T0* volatile a3; /* equality_tester */
	T0* volatile a4; /* storage */
	T0* volatile a5; /* special_routines */
	T6 volatile a6; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST [AP_ALTERNATIVE_OPTIONS_LIST] */
struct S215 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* count */
};

/* Struct for type AP_ERROR_HANDLER */
struct S216 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_error */
	T0* volatile a2; /* error_file */
	T0* volatile a3; /* warning_file */
	T0* volatile a4; /* info_file */
};

/* Struct for type DS_LINKABLE [AP_OPTION] */
struct S217 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_LINKED_LIST_CURSOR [AP_OPTION] */
struct S218 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
	T1 volatile a3; /* after */
	T0* volatile a4; /* current_cell */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_ARRAYED_LIST [detachable STRING_8] */
struct S219 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
struct S220 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* container */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_LINKED_LIST [STRING_8] */
struct S221 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* last_cell */
	T6 volatile a3; /* count */
	T0* volatile a4; /* first_cell */
	T0* volatile a5; /* equality_tester */
};

/* Struct for type KL_STRING_EQUALITY_TESTER */
struct S222 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_EQUALITY_TESTER [STRING_8] */
struct S223 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_LINKED_LIST [BOOLEAN] */
struct S226 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* last_cell */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* first_cell */
};

/* Struct for type DS_LINKED_LIST [INTEGER_32] */
struct S229 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* last_cell */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* first_cell */
};

/* Struct for type KL_NULL_TEXT_OUTPUT_STREAM */
struct S230 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
};

/* Struct for type ET_EIFFEL_PREPARSER */
struct S232 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* eiffel_buffer */
	T0* volatile a2; /* system_processor */
	T6 volatile a3; /* last_text_count */
	T6 volatile a4; /* last_literal_start */
	T0* volatile a5; /* filename */
	T0* volatile a6; /* group */
	T0* volatile a7; /* verbatim_marker */
	T6 volatile a8; /* verbatim_marker_count */
	T0* volatile a9; /* verbatim_open_white_characters */
	T0* volatile a10; /* verbatim_close_white_characters */
	T0* volatile a11; /* yy_pushed_start_conditions */
	T0* volatile a12; /* input_buffer */
	T0* volatile a13; /* yy_state_stack */
	T6 volatile a14; /* yy_start_state */
	T6 volatile a15; /* yy_line */
	T6 volatile a16; /* yy_column */
	T6 volatile a17; /* yy_position */
	T6 volatile a18; /* yy_start */
	T6 volatile a19; /* yy_end */
	T6 volatile a20; /* line */
	T6 volatile a21; /* column */
	T6 volatile a22; /* position */
	T0* volatile a23; /* yy_nxt */
	T0* volatile a24; /* yy_chk */
	T0* volatile a25; /* yy_base */
	T0* volatile a26; /* yy_def */
	T0* volatile a27; /* yy_ec */
	T0* volatile a28; /* yy_meta */
	T0* volatile a29; /* yy_accept */
	T0* volatile a30; /* yy_acclist */
	T0* volatile a31; /* yy_content */
	T0* volatile a32; /* yy_content_area */
	T0* volatile a33; /* yy_unicode_content_area */
	T0* volatile a34; /* last_classname */
	T1 volatile a35; /* class_keyword_found */
	T6 volatile a36; /* last_token */
	T1 volatile a37; /* yy_more_flag */
	T6 volatile a38; /* yy_more_len */
	T6 volatile a39; /* yy_state_count */
	T6 volatile a40; /* yy_last_accepting_state */
	T6 volatile a41; /* yy_last_accepting_cpos */
	T6 volatile a42; /* yy_lp */
	T6 volatile a43; /* yy_looking_for_trail_begin */
	T6 volatile a44; /* yy_full_match */
	T6 volatile a45; /* yy_full_state */
	T6 volatile a46; /* yy_full_lp */
	T1 volatile a47; /* yy_rejected */
	T6 volatile a48; /* last_literal_end */
	T6 volatile a49; /* pushed_start_condition_count */
	T6 volatile a50; /* ms_column */
	T6 volatile a51; /* ms_line */
	T6 volatile a52; /* last_break_end */
	T6 volatile a53; /* last_comment_end */
	T3 volatile a54; /* verbatim_opening_character */
};

/* Struct for type ET_MASTER_CLASS_CHECKER */
struct S233 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_class */
	T0* volatile a2; /* system_processor */
	T1 volatile a3; /* has_fatal_error */
};

/* Struct for type ET_PROVIDER_CHECKER */
struct S234 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_class */
	T0* volatile a2; /* system_processor */
};

/* Struct for type ET_ANCESTOR_BUILDER */
struct S235 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_sorter */
	T0* volatile a2; /* ancestors */
	T0* volatile a3; /* conforming_ancestors */
	T0* volatile a4; /* parent_checker */
	T0* volatile a5; /* formal_parameter_checker */
	T0* volatile a6; /* parent_context */
	T0* volatile a7; /* current_class */
	T0* volatile a8; /* system_processor */
};

/* Struct for type ET_FEATURE_FLATTENER */
struct S236 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* named_features */
	T0* volatile a2; /* queries */
	T0* volatile a3; /* procedures */
	T0* volatile a4; /* aliased_features */
	T0* volatile a5; /* clients_list */
	T0* volatile a6; /* client_classes */
	T0* volatile a7; /* feature_adaptation_resolver */
	T0* volatile a8; /* dotnet_feature_adaptation_resolver */
	T0* volatile a9; /* identifier_type_resolver */
	T0* volatile a10; /* unfolded_tuple_actual_parameters_resolver */
	T0* volatile a11; /* anchored_type_checker */
	T0* volatile a12; /* signature_checker */
	T0* volatile a13; /* parent_checker */
	T0* volatile a14; /* formal_parameter_checker */
	T0* volatile a15; /* builtin_feature_checker */
	T0* volatile a16; /* precursor_checker */
	T0* volatile a17; /* precursors */
	T0* volatile a18; /* current_class */
	T0* volatile a19; /* system_processor */
	T1 volatile a20; /* has_signature_error */
};

/* Struct for type ET_INTERFACE_CHECKER */
struct S237 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* classes_to_be_processed */
	T0* volatile a2; /* qualified_anchored_type_checker */
	T0* volatile a3; /* unfolded_tuple_actual_parameters_resolver */
	T0* volatile a4; /* old_name_rename_table */
	T0* volatile a5; /* new_name_rename_table */
	T0* volatile a6; /* new_alias_name_rename_table */
	T0* volatile a7; /* current_class */
	T0* volatile a8; /* system_processor */
};

/* Struct for type ET_IMPLEMENTATION_CHECKER */
struct S238 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* feature_checker */
	T0* volatile a2; /* named_features */
	T0* volatile a3; /* feature_adaptation_resolver */
	T0* volatile a4; /* system_processor */
	T0* volatile a5; /* dotnet_feature_adaptation_resolver */
	T0* volatile a6; /* signature_checker */
	T0* volatile a7; /* parent_checker3 */
	T0* volatile a8; /* precursor_procedures */
	T0* volatile a9; /* precursor_queries */
	T0* volatile a10; /* supplier_builder */
	T0* volatile a11; /* current_class */
	T0* volatile a12; /* no_suppliers */
};

/* Struct for type ET_AST_NULL_PROCESSOR */
struct S239 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [INTEGER_32] */
struct S240 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
struct S242 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* system_processor */
};

/* Struct for type YY_UNICODE_FILE_BUFFER */
struct S244 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_utf8_bom */
	T0* volatile a2; /* content */
	T6 volatile a3; /* index */
	T6 volatile a4; /* line */
	T6 volatile a5; /* column */
	T6 volatile a6; /* position */
	T6 volatile a7; /* count */
	T1 volatile a8; /* filled */
	T1 volatile a9; /* beginning_of_line */
	T6 volatile a10; /* capacity */
	T1 volatile a11; /* end_of_file */
	T1 volatile a12; /* interactive */
	T6 volatile a13; /* encoding */
	T0* volatile a14; /* file */
	T6 volatile a15; /* default_encoding */
};

/* Struct for type KL_STDIN_FILE */
struct S245 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* end_of_file */
	T0* volatile a2; /* last_string */
	T0* volatile a3; /* character_buffer */
	T2 volatile a4; /* last_character */
};

/* Struct for type DS_ARRAYED_STACK [INTEGER_32] */
struct S246 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T6 volatile a2; /* count */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S247 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
struct S248 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_KEYWORD] */
struct S249 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T6 volatile a2; /* count */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_SYMBOL] */
struct S250 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T6 volatile a2; /* count */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
struct S251 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [ET_OBJECT_TEST_LIST] */
struct S252 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
struct S253 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [ET_ITERATION_COMPONENT_LIST] */
struct S254 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S255 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_STACK [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S256 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST [ET_ASSERTION_ITEM] */
struct S257 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_QUERY] */
struct S258 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST [ET_PROCEDURE] */
struct S259 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
struct S260 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_SET [ET_NAMED_CLASS] */
struct S261 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* modulus */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T6 volatile a9; /* slots_position */
	T0* volatile a10; /* special_item_routines */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type YY_BUFFER */
struct S262 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* content */
	T6 volatile a2; /* index */
	T6 volatile a3; /* line */
	T6 volatile a4; /* column */
	T6 volatile a5; /* position */
	T6 volatile a6; /* count */
	T1 volatile a7; /* filled */
	T1 volatile a8; /* beginning_of_line */
	T6 volatile a9; /* capacity */
};

/* Struct for type ET_UNKNOWN_GROUP */
struct S263 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* use_obsolete_syntax */
	T0* volatile a2; /* name */
	T0* volatile a3; /* cached_absolute_pathname */
};

/* Struct for type KL_SPECIAL_ROUTINES [INTEGER_32] */
struct S264 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_UNICODE_CHARACTER_BUFFER */
struct S266 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* as_unicode_special */
	T2 volatile a2; /* invalid_character */
	T0* volatile a3; /* unicode_area */
	T0* volatile a4; /* area */
	T3 volatile a5; /* invalid_unicode_character */
};

/* Struct for type ET_CLIENTS */
struct S268 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_brace */
	T0* volatile a2; /* right_brace */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type ET_FEATURE_CLAUSE */
struct S269 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* feature_keyword */
	T0* volatile a2; /* clients */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ANY] */
struct S272 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ANY] */
struct S273 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
struct S274 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_KEYWORD] */
struct S275 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
struct S276 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_AGENT_KEYWORD */
struct S277 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* compressed_position */
	T0* volatile a3; /* name */
	T0* volatile a4; /* break */
};

/* Struct for type SPECIAL [detachable ET_AGENT_KEYWORD] */
struct S278 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INVARIANT_KEYWORD] */
struct S279 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INVARIANT_KEYWORD */
struct S280 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* compressed_position */
	T2 volatile a2; /* code */
	T0* volatile a3; /* name */
	T6 volatile a4; /* seed */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_INVARIANT_KEYWORD] */
struct S281 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
struct S282 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_PRECURSOR_KEYWORD */
struct S283 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* compressed_position */
	T0* volatile a3; /* name */
	T6 volatile a4; /* seed */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
struct S284 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
struct S285 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SYMBOL */
struct S286 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* break */
	T2 volatile a2; /* code */
	T6 volatile a3; /* compressed_position */
};

/* Struct for type SPECIAL [detachable ET_SYMBOL] */
struct S287 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
struct S288 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_POSITION] */
struct S290 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
struct S291 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
struct S293 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
struct S294 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_BREAK] */
struct S296 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
struct S297 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CHARACTER_CONSTANT] */
struct S299 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
struct S300 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CURRENT */
struct S301 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* text */
	T6 volatile a3; /* compressed_position */
	T2 volatile a4; /* code */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_CURRENT] */
struct S302 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
struct S303 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FREE_OPERATOR */
struct S304 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* hash_code */
	T6 volatile a3; /* compressed_position */
	T0* volatile a4; /* break */
	T0* volatile a5; /* operator_name */
	T6 volatile a6; /* seed */
};

/* Struct for type SPECIAL [detachable ET_FREE_OPERATOR] */
struct S305 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
struct S306 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_IDENTIFIER */
struct S307 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* seed */
	T6 volatile a3; /* index */
	T6 volatile a4; /* hash_code */
	T2 volatile a5; /* status_code */
	T6 volatile a6; /* compressed_position */
	T0* volatile a7; /* break */
};

/* Struct for type SPECIAL [detachable ET_IDENTIFIER] */
struct S308 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
struct S309 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_INTEGER_CONSTANT] */
struct S311 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
struct S312 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_KEYWORD_OPERATOR */
struct S313 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* compressed_position */
	T2 volatile a2; /* code */
	T0* volatile a3; /* text */
	T6 volatile a4; /* seed */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_KEYWORD_OPERATOR] */
struct S314 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
struct S315 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_MANIFEST_STRING] */
struct S317 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
struct S318 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_REAL_CONSTANT] */
struct S320 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
struct S321 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_RESULT */
struct S322 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T0* volatile a2; /* text */
	T6 volatile a3; /* compressed_position */
	T6 volatile a4; /* index */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_RESULT] */
struct S323 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
struct S324 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_RETRY_INSTRUCTION */
struct S325 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* compressed_position */
	T0* volatile a3; /* text */
	T0* volatile a4; /* break */
};

/* Struct for type SPECIAL [detachable ET_RETRY_INSTRUCTION] */
struct S326 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
struct S327 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SYMBOL_OPERATOR */
struct S328 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* break */
	T2 volatile a2; /* code */
	T6 volatile a3; /* compressed_position */
	T6 volatile a4; /* seed */
};

/* Struct for type SPECIAL [detachable ET_SYMBOL_OPERATOR] */
struct S329 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_VOID] */
struct S330 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_VOID */
struct S331 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T2 volatile a2; /* code */
	T0* volatile a3; /* text */
	T6 volatile a4; /* compressed_position */
	T0* volatile a5; /* break */
};

/* Struct for type SPECIAL [detachable ET_VOID] */
struct S332 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
struct S333 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SEMICOLON_SYMBOL */
struct S334 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* compressed_position */
	T0* volatile a3; /* break */
};

/* Struct for type SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
struct S335 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
struct S336 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_BRACKET_SYMBOL */
struct S337 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* break */
	T2 volatile a2; /* code */
	T6 volatile a3; /* compressed_position */
	T6 volatile a4; /* seed */
};

/* Struct for type SPECIAL [detachable ET_BRACKET_SYMBOL] */
struct S338 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
struct S339 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_QUESTION_MARK_SYMBOL */
struct S340 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T6 volatile a2; /* compressed_position */
	T6 volatile a3; /* index */
	T0* volatile a4; /* break */
};

/* Struct for type SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
struct S341 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
struct S342 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ACROSS_EXPRESSION */
struct S343 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item_name */
	T0* volatile a2; /* as_keyword */
	T0* volatile a3; /* unfolded_cursor_name */
	T6 volatile a4; /* index */
	T0* volatile a5; /* across_keyword */
	T0* volatile a6; /* iterable_expression */
	T0* volatile a7; /* until_conditional */
	T0* volatile a8; /* iteration_conditional */
	T0* volatile a9; /* end_keyword */
	T1 volatile a10; /* is_all */
	T1 volatile a11; /* has_cursor_name */
	T0* volatile a12; /* invariant_part */
	T0* volatile a13; /* variant_part */
	T0* volatile a14; /* new_cursor_expression */
	T0* volatile a15; /* cursor_item_expression */
	T0* volatile a16; /* cursor_after_expression */
	T0* volatile a17; /* cursor_forth_instruction */
};

/* Struct for type SPECIAL [detachable ET_ACROSS_EXPRESSION] */
struct S344 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
struct S345 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ACROSS_INSTRUCTION */
struct S346 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item_name */
	T0* volatile a2; /* as_keyword */
	T0* volatile a3; /* unfolded_cursor_name */
	T0* volatile a4; /* across_keyword */
	T0* volatile a5; /* iterable_expression */
	T0* volatile a6; /* from_compound */
	T0* volatile a7; /* until_conditional */
	T0* volatile a8; /* loop_compound */
	T0* volatile a9; /* end_keyword */
	T1 volatile a10; /* has_cursor_name */
	T0* volatile a11; /* invariant_part */
	T0* volatile a12; /* variant_part */
	T0* volatile a13; /* new_cursor_expression */
	T0* volatile a14; /* cursor_item_expression */
	T0* volatile a15; /* cursor_after_expression */
	T0* volatile a16; /* cursor_forth_instruction */
};

/* Struct for type SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
struct S347 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S348 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ACTUAL_ARGUMENT_LIST */
struct S349 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* left_symbol */
	T0* volatile a4; /* right_symbol */
};

/* Struct for type SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S350 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S351 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S352 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S353 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S354 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S355 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S357 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S358 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S360 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S361 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_AGENT_ARGUMENT_OPERAND_LIST */
struct S362 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_parenthesis */
	T0* volatile a2; /* right_parenthesis */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S363 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
struct S364 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_AGENT_TARGET] */
struct S366 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
struct S367 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ALIAS_NAME */
struct S368 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* alias_keyword */
	T0* volatile a2; /* alias_string */
	T2 volatile a3; /* code */
	T0* volatile a4; /* convert_keyword */
	T6 volatile a5; /* seed */
};

/* Struct for type SPECIAL [detachable ET_ALIAS_NAME] */
struct S369 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
struct S370 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ALIAS_NAME_LIST */
struct S371 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_ALIAS_NAME_LIST] */
struct S372 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
struct S373 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_ASSIGNER] */
struct S375 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
struct S376 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_BRACKET_EXPRESSION */
struct S377 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* target */
	T0* volatile a3; /* name */
	T0* volatile a4; /* arguments */
};

/* Struct for type SPECIAL [detachable ET_BRACKET_EXPRESSION] */
struct S378 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
struct S379 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CALL_AGENT */
struct S380 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* agent_keyword */
	T0* volatile a3; /* qualified_name */
	T0* volatile a4; /* arguments */
	T0* volatile a5; /* target */
	T0* volatile a6; /* implicit_result */
};

/* Struct for type SPECIAL [detachable ET_CALL_AGENT] */
struct S381 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S382 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_QUALIFIED_CALL_EXPRESSION */
struct S383 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T6 volatile a2; /* index */
	T0* volatile a3; /* arguments */
	T0* volatile a4; /* qualified_name */
	T0* volatile a5; /* parenthesis_call */
};

/* Struct for type SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S384 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
struct S385 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CHECK_INSTRUCTION */
struct S386 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* check_keyword */
	T0* volatile a2; /* end_keyword */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
	T0* volatile a5; /* then_compound */
};

/* Struct for type SPECIAL [detachable ET_CHECK_INSTRUCTION] */
struct S387 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
struct S388 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CHOICE] */
struct S390 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
struct S391 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CHOICE_CONSTANT] */
struct S393 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
struct S394 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CHOICE_ITEM] */
struct S396 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
struct S397 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CHOICE_LIST */
struct S398 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* when_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type SPECIAL [detachable ET_CHOICE_LIST] */
struct S399 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
struct S400 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CLASS] */
struct S401 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
struct S402 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CLIENT_ITEM] */
struct S403 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
struct S404 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CLIENTS] */
struct S405 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
struct S406 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_COMPOUND */
struct S407 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type SPECIAL [detachable ET_COMPOUND] */
struct S408 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
struct S409 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONDITIONAL] */
struct S411 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
struct S412 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONSTANT] */
struct S414 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S415 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S417 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S418 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
struct S419 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* left_bracket */
	T0* volatile a4; /* right_bracket */
};

/* Struct for type SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S420 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
struct S421 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CONSTRAINT_CREATOR */
struct S422 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* create_keyword */
	T0* volatile a2; /* end_keyword */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
struct S423 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
struct S424 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CONSTRAINT_RENAME_LIST */
struct S425 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* end_keyword */
	T0* volatile a2; /* rename_keyword */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
struct S426 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
struct S427 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONSTRAINT_TYPE] */
struct S429 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
struct S430 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONVERT_FEATURE] */
struct S432 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
struct S433 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
struct S435 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
struct S436 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CONVERT_FEATURE_LIST */
struct S437 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* convert_keyword */
};

/* Struct for type SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
struct S438 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
struct S439 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CREATE_EXPRESSION */
struct S440 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* creation_call */
	T0* volatile a3; /* creation_region */
	T0* volatile a4; /* internal_separate_target */
	T0* volatile a5; /* default_create_name */
	T0* volatile a6; /* creation_type */
	T0* volatile a7; /* create_keyword */
};

/* Struct for type SPECIAL [detachable ET_CREATE_EXPRESSION] */
struct S441 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
struct S442 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CREATION_REGION */
struct S443 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_name */
	T0* volatile a2; /* left_symbol */
	T0* volatile a3; /* right_symbol */
};

/* Struct for type SPECIAL [detachable ET_CREATION_REGION] */
struct S444 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
struct S445 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CREATOR] */
struct S446 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
struct S447 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_CREATOR_LIST] */
struct S448 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
struct S449 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_DEBUG_INSTRUCTION */
struct S450 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* keys */
	T0* volatile a2; /* compound */
	T0* volatile a3; /* end_keyword */
};

/* Struct for type SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
struct S451 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
struct S452 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ELSEIF_EXPRESSION */
struct S453 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditional */
	T0* volatile a2; /* then_keyword */
	T0* volatile a3; /* then_expression */
};

/* Struct for type SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
struct S454 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S455 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ELSEIF_EXPRESSION_LIST */
struct S456 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S457 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
struct S458 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ELSEIF_PART */
struct S459 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditional */
	T0* volatile a2; /* then_compound */
};

/* Struct for type SPECIAL [detachable ET_ELSEIF_PART] */
struct S460 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
struct S461 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ELSEIF_PART_LIST */
struct S462 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_ELSEIF_PART_LIST] */
struct S463 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
struct S464 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXPORT] */
struct S466 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
struct S467 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXPORT_LIST] */
struct S468 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
struct S469 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXPRESSION] */
struct S471 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
struct S472 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXPRESSION_ITEM] */
struct S474 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
struct S475 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
struct S477 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
struct S478 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_EXTERNAL_ALIAS] */
struct S480 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
struct S481 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_FEATURE_CLAUSE] */
struct S482 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
struct S483 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FEATURE_CLAUSE_LIST */
struct S484 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
struct S485 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
struct S486 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FEATURE_EXPORT */
struct S487 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* clients_clause */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* count */
};

/* Struct for type SPECIAL [detachable ET_FEATURE_EXPORT] */
struct S488 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
struct S489 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_FEATURE_NAME] */
struct S490 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
struct S491 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
struct S493 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
struct S494 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FORMAL_ARGUMENT */
struct S495 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name_item */
	T0* volatile a2; /* declared_type */
	T6 volatile a3; /* index */
	T6 volatile a4; /* attached_index */
	T1 volatile a5; /* is_used */
};

/* Struct for type SPECIAL [detachable ET_FORMAL_ARGUMENT] */
struct S496 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S497 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S499 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S500 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FORMAL_ARGUMENT_LIST */
struct S501 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* left_parenthesis */
	T0* volatile a4; /* right_parenthesis */
};

/* Struct for type SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S502 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
struct S503 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FORMAL_PARAMETER */
struct S504 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* index */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* type_mark */
};

/* Struct for type SPECIAL [detachable ET_FORMAL_PARAMETER] */
struct S505 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S506 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S508 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
struct S509 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FORMAL_PARAMETER_LIST */
struct S510 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* right_bracket */
	T0* volatile a4; /* left_bracket */
};

/* Struct for type SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
struct S511 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
struct S512 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_IF_EXPRESSION */
struct S513 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* conditional */
	T0* volatile a3; /* then_keyword */
	T0* volatile a4; /* then_expression */
	T0* volatile a5; /* else_keyword */
	T0* volatile a6; /* else_expression */
	T0* volatile a7; /* end_keyword */
	T0* volatile a8; /* elseif_parts */
};

/* Struct for type SPECIAL [detachable ET_IF_EXPRESSION] */
struct S514 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
struct S515 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_IF_INSTRUCTION */
struct S516 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditional */
	T0* volatile a2; /* then_compound */
	T0* volatile a3; /* end_keyword */
	T0* volatile a4; /* elseif_parts */
	T0* volatile a5; /* else_compound */
};

/* Struct for type SPECIAL [detachable ET_IF_INSTRUCTION] */
struct S517 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
struct S518 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_INLINE_AGENT] */
struct S520 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT] */
struct S521 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INLINE_SEPARATE_ARGUMENT */
struct S522 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* expression */
	T0* volatile a2; /* as_keyword */
	T0* volatile a3; /* name */
};

/* Struct for type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT] */
struct S523 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
struct S524 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
struct S526 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
struct S527 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INLINE_SEPARATE_ARGUMENTS */
struct S528 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* separate_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
struct S529 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
struct S530 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INLINE_SEPARATE_INSTRUCTION */
struct S531 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* compound */
	T0* volatile a3; /* end_keyword */
};

/* Struct for type SPECIAL [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
struct S532 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
struct S533 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INSPECT_EXPRESSION */
struct S534 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* conditional */
	T0* volatile a3; /* when_parts */
	T0* volatile a4; /* end_keyword */
	T0* volatile a5; /* else_part */
};

/* Struct for type SPECIAL [detachable ET_INSPECT_EXPRESSION] */
struct S535 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
struct S536 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INSPECT_INSTRUCTION */
struct S537 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditional */
	T0* volatile a2; /* when_parts */
	T0* volatile a3; /* end_keyword */
	T0* volatile a4; /* else_compound */
};

/* Struct for type SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
struct S538 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
struct S539 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_INSTRUCTION] */
struct S541 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
struct S542 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_INVARIANTS */
struct S543 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* implementation_class */
	T1 volatile a2; /* validity_checked */
	T1 volatile a3; /* has_validity_error */
	T6 volatile a4; /* count */
	T0* volatile a5; /* storage */
	T0* volatile a6; /* invariant_keyword */
	T0* volatile a7; /* object_tests */
	T0* volatile a8; /* iteration_components */
	T0* volatile a9; /* inline_separate_arguments */
};

/* Struct for type SPECIAL [detachable ET_INVARIANTS] */
struct S544 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S545 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S546 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
struct S547 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_LIKE_TYPE] */
struct S549 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
struct S550 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_LOCAL_VARIABLE */
struct S551 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name_item */
	T0* volatile a2; /* declared_type */
	T6 volatile a3; /* index */
	T6 volatile a4; /* attached_index */
	T1 volatile a5; /* is_used */
};

/* Struct for type SPECIAL [detachable ET_LOCAL_VARIABLE] */
struct S552 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S553 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S555 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
struct S556 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_LOCAL_VARIABLE_LIST */
struct S557 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* local_keyword */
};

/* Struct for type SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
struct S558 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
struct S559 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_LOOP_INVARIANTS */
struct S560 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* invariant_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type SPECIAL [detachable ET_LOOP_INVARIANTS] */
struct S561 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
struct S562 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_MANIFEST_ARRAY */
struct S563 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* left_symbol */
	T0* volatile a3; /* right_symbol */
	T6 volatile a4; /* count */
	T0* volatile a5; /* storage */
	T0* volatile a6; /* cast_type */
};

/* Struct for type SPECIAL [detachable ET_MANIFEST_ARRAY] */
struct S564 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
struct S565 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
struct S567 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
struct S568 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_MANIFEST_STRING_LIST */
struct S569 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_parenthesis */
	T0* volatile a2; /* right_parenthesis */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
struct S570 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
struct S571 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_MANIFEST_TUPLE */
struct S572 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* left_symbol */
	T0* volatile a4; /* right_symbol */
	T6 volatile a5; /* index */
};

/* Struct for type SPECIAL [detachable ET_MANIFEST_TUPLE] */
struct S573 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_NOTE_LIST] */
struct S574 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_NOTE_LIST */
struct S575 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* note_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type SPECIAL [detachable ET_NOTE_LIST] */
struct S576 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_NOTE_ITEM] */
struct S577 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_NOTE_ITEM] */
struct S579 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM] */
struct S580 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_NOTE_TERM] */
struct S582 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_ITEM] */
struct S583 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_NOTE_TERM_ITEM] */
struct S585 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_LIST] */
struct S586 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_NOTE_TERM_LIST */
struct S587 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_NOTE_TERM_LIST] */
struct S588 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
struct S589 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_OBSOLETE] */
struct S591 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S592 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_PARENTHESIZED_EXPRESSION */
struct S593 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* expression */
	T0* volatile a3; /* left_parenthesis */
	T0* volatile a4; /* right_parenthesis */
};

/* Struct for type SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S594 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
struct S595 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_PARENT] */
struct S596 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
struct S597 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_PARENT_CLAUSE_LIST */
struct S598 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
struct S599 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
struct S600 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_PARENT_ITEM] */
struct S601 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
struct S602 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_PARENT_LIST] */
struct S603 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
struct S604 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_POSTCONDITIONS */
struct S605 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* ensure_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* then_keyword */
	T1 volatile a5; /* validity_checked */
	T1 volatile a6; /* has_validity_error */
};

/* Struct for type SPECIAL [detachable ET_POSTCONDITIONS] */
struct S606 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
struct S607 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_PRECONDITIONS */
struct S608 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* require_keyword */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* else_keyword */
	T1 volatile a5; /* validity_checked */
	T1 volatile a6; /* has_validity_error */
};

/* Struct for type SPECIAL [detachable ET_PRECONDITIONS] */
struct S609 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
struct S610 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_PROCEDURE] */
struct S612 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S613 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S615 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
struct S616 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_QUANTIFIER_EXPRESSION */
struct S617 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item_name */
	T0* volatile a2; /* unfolded_cursor_name */
	T6 volatile a3; /* index */
	T0* volatile a4; /* quantifier_symbol */
	T0* volatile a5; /* colon_symbol */
	T0* volatile a6; /* iterable_expression */
	T0* volatile a7; /* bar_symbol */
	T0* volatile a8; /* iteration_expression */
	T1 volatile a9; /* is_all */
	T0* volatile a10; /* new_cursor_expression */
	T0* volatile a11; /* cursor_item_expression */
	T0* volatile a12; /* cursor_after_expression */
	T0* volatile a13; /* cursor_forth_instruction */
};

/* Struct for type SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
struct S618 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
struct S619 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_QUERY] */
struct S621 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
struct S622 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_RENAME_ITEM] */
struct S624 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
struct S625 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_RENAME_LIST] */
struct S626 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
struct S627 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_REPEAT_INSTRUCTION */
struct S628 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item_name */
	T0* volatile a2; /* unfolded_cursor_name */
	T0* volatile a3; /* open_repeat_symbol */
	T0* volatile a4; /* colon_symbol */
	T0* volatile a5; /* iterable_expression */
	T0* volatile a6; /* bar_symbol */
	T0* volatile a7; /* loop_compound */
	T0* volatile a8; /* close_repeat_symbol */
	T0* volatile a9; /* new_cursor_expression */
	T0* volatile a10; /* cursor_item_expression */
	T0* volatile a11; /* cursor_after_expression */
	T0* volatile a12; /* cursor_forth_instruction */
};

/* Struct for type SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
struct S629 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
struct S630 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_STATIC_CALL_EXPRESSION */
struct S631 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* static_type */
	T0* volatile a3; /* feature_keyword */
	T0* volatile a4; /* qualified_name */
	T0* volatile a5; /* arguments */
	T0* volatile a6; /* parenthesis_call */
};

/* Struct for type SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
struct S632 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
struct S633 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_STRIP_EXPRESSION */
struct S634 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* strip_keyword */
	T0* volatile a3; /* left_parenthesis */
	T0* volatile a4; /* right_parenthesis */
	T6 volatile a5; /* count */
	T0* volatile a6; /* storage */
};

/* Struct for type SPECIAL [detachable ET_STRIP_EXPRESSION] */
struct S635 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
struct S636 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_TYPE] */
struct S637 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
struct S638 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_TYPE_CONSTRAINT] */
struct S640 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
struct S641 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
struct S643 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
struct S644 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_TYPE_CONSTRAINT_LIST */
struct S645 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_brace */
	T0* volatile a2; /* right_brace */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
struct S646 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
struct S647 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_TYPE_ITEM] */
struct S649 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
struct S650 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_TYPE_LIST] */
struct S652 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
struct S653 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_VARIANT */
struct S654 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* variant_keyword */
	T0* volatile a2; /* tag */
	T0* volatile a3; /* expression */
};

/* Struct for type SPECIAL [detachable ET_VARIANT] */
struct S655 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
struct S656 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_WHEN_EXPRESSION */
struct S657 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* choices */
	T0* volatile a2; /* then_keyword */
	T0* volatile a3; /* then_expression */
};

/* Struct for type SPECIAL [detachable ET_WHEN_EXPRESSION] */
struct S658 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
struct S659 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_WHEN_EXPRESSION_LIST */
struct S660 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
struct S661 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
struct S662 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_WHEN_PART */
struct S663 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* choices */
	T0* volatile a2; /* then_compound */
};

/* Struct for type SPECIAL [detachable ET_WHEN_PART] */
struct S664 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
struct S665 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_WHEN_PART_LIST */
struct S666 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type SPECIAL [detachable ET_WHEN_PART_LIST] */
struct S667 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
struct S668 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_WRITABLE] */
struct S670 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ET_OBJECT_TEST_LIST */
struct S672 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_ITERATION_COMPONENT_LIST */
struct S673 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_INLINE_SEPARATE_ARGUMENT_LIST */
struct S674 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_C3_CHARACTER_CONSTANT */
struct S675 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* literal */
	T3 volatile a3; /* value */
	T1 volatile a4; /* has_invalid_code */
	T0* volatile a5; /* cast_type */
	T6 volatile a6; /* compressed_position */
	T0* volatile a7; /* type */
	T0* volatile a8; /* break */
};

/* Struct for type ET_REGULAR_MANIFEST_STRING */
struct S676 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T6 volatile a2; /* index */
	T6 volatile a3; /* compressed_position */
	T0* volatile a4; /* cast_type */
	T0* volatile a5; /* type */
	T0* volatile a6; /* break */
};

/* Struct for type ET_SPECIAL_MANIFEST_STRING */
struct S677 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T6 volatile a2; /* index */
	T0* volatile a3; /* cast_type */
	T6 volatile a4; /* compressed_position */
	T0* volatile a5; /* literal */
	T1 volatile a6; /* has_invalid_code */
	T0* volatile a7; /* type */
	T0* volatile a8; /* break */
};

/* Struct for type ET_VERBATIM_STRING */
struct S678 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T6 volatile a2; /* index */
	T0* volatile a3; /* cast_type */
	T6 volatile a4; /* compressed_position */
	T0* volatile a5; /* literal */
	T0* volatile a6; /* marker */
	T0* volatile a7; /* open_white_characters */
	T0* volatile a8; /* close_white_characters */
	T1 volatile a9; /* is_left_aligned */
	T0* volatile a10; /* type */
	T0* volatile a11; /* break */
};

/* Struct for type ET_REGULAR_INTEGER_CONSTANT */
struct S679 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* literal */
	T11 volatile a3; /* value */
	T1 volatile a4; /* has_overflow */
	T6 volatile a5; /* compressed_position */
	T0* volatile a6; /* sign */
	T0* volatile a7; /* cast_type */
	T0* volatile a8; /* type */
	T0* volatile a9; /* break */
};

/* Struct for type ET_UNDERSCORED_INTEGER_CONSTANT */
struct S680 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sign */
	T6 volatile a2; /* index */
	T0* volatile a3; /* literal */
	T11 volatile a4; /* value */
	T1 volatile a5; /* has_overflow */
	T0* volatile a6; /* cast_type */
	T6 volatile a7; /* compressed_position */
	T0* volatile a8; /* type */
	T0* volatile a9; /* break */
};

/* Struct for type ET_HEXADECIMAL_INTEGER_CONSTANT */
struct S681 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sign */
	T6 volatile a2; /* index */
	T0* volatile a3; /* literal */
	T11 volatile a4; /* value */
	T1 volatile a5; /* has_overflow */
	T0* volatile a6; /* cast_type */
	T6 volatile a7; /* compressed_position */
	T0* volatile a8; /* type */
	T0* volatile a9; /* break */
};

/* Struct for type ET_OCTAL_INTEGER_CONSTANT */
struct S682 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sign */
	T6 volatile a2; /* index */
	T0* volatile a3; /* literal */
	T11 volatile a4; /* value */
	T1 volatile a5; /* has_overflow */
	T0* volatile a6; /* cast_type */
	T6 volatile a7; /* compressed_position */
	T0* volatile a8; /* type */
	T0* volatile a9; /* break */
};

/* Struct for type ET_BINARY_INTEGER_CONSTANT */
struct S683 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sign */
	T6 volatile a2; /* index */
	T0* volatile a3; /* literal */
	T11 volatile a4; /* value */
	T1 volatile a5; /* has_overflow */
	T0* volatile a6; /* cast_type */
	T6 volatile a7; /* compressed_position */
	T0* volatile a8; /* type */
	T0* volatile a9; /* break */
};

/* Struct for type ET_REGULAR_REAL_CONSTANT */
struct S684 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* literal */
	T6 volatile a3; /* compressed_position */
	T0* volatile a4; /* sign */
	T0* volatile a5; /* cast_type */
	T0* volatile a6; /* type */
	T0* volatile a7; /* break */
};

/* Struct for type ET_UNDERSCORED_REAL_CONSTANT */
struct S685 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sign */
	T6 volatile a2; /* index */
	T0* volatile a3; /* literal */
	T0* volatile a4; /* cast_type */
	T6 volatile a5; /* compressed_position */
	T0* volatile a6; /* type */
	T0* volatile a7; /* break */
};

/* Struct for type ET_COMMENT */
struct S686 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* text */
};

/* Struct for type ET_TRUE_CONSTANT */
struct S687 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* code */
	T0* volatile a2; /* text */
	T6 volatile a3; /* compressed_position */
	T6 volatile a4; /* index */
	T0* volatile a5; /* break */
};

/* Struct for type ET_FALSE_CONSTANT */
struct S688 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T2 volatile a2; /* code */
	T0* volatile a3; /* text */
	T6 volatile a4; /* compressed_position */
	T0* volatile a5; /* break */
};

/* Struct for type ET_C1_CHARACTER_CONSTANT */
struct S689 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T3 volatile a2; /* value */
	T0* volatile a3; /* cast_type */
	T6 volatile a4; /* compressed_position */
	T0* volatile a5; /* type */
	T0* volatile a6; /* break */
};

/* Struct for type ET_C2_CHARACTER_CONSTANT */
struct S690 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T3 volatile a2; /* value */
	T6 volatile a3; /* compressed_position */
	T0* volatile a4; /* cast_type */
	T0* volatile a5; /* type */
	T0* volatile a6; /* break */
};

/* Struct for type ET_TAGGED_NOTE */
struct S692 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* tag */
	T0* volatile a2; /* terms */
};

/* Struct for type ET_NOTE */
struct S693 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* terms */
};

/* Struct for type ET_CUSTOM_ATTRIBUTE */
struct S694 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* creation_expression */
	T0* volatile a2; /* settings */
	T0* volatile a3; /* end_keyword */
};

/* Struct for type ET_CONSTRAINED_FORMAL_PARAMETER */
struct S695 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arrow_symbol */
	T0* volatile a2; /* constraint */
	T0* volatile a3; /* creation_procedures */
	T0* volatile a4; /* constraint_base_types */
	T0* volatile a5; /* implementation_class */
	T0* volatile a6; /* type_mark */
	T6 volatile a7; /* index */
	T0* volatile a8; /* name */
	T0* volatile a9; /* recursive_formal_constraints */
};

/* Struct for type ET_TYPE_RENAME_CONSTRAINT */
struct S696 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* renames */
};

/* Struct for type ET_CONSTRAINT_NAMED_TYPE */
struct S697 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type_mark */
	T0* volatile a2; /* name */
};

/* Struct for type ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
struct S698 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* label */
	T0* volatile a2; /* colon */
	T0* volatile a3; /* type */
};

/* Struct for type ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
struct S699 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* label */
	T0* volatile a2; /* comma */
	T0* volatile a3; /* type */
};

/* Struct for type ET_RENAME */
struct S700 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* old_name */
	T0* volatile a2; /* as_keyword */
	T0* volatile a3; /* new_name */
};

/* Struct for type ET_ALL_EXPORT */
struct S701 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* clients_clause */
	T0* volatile a2; /* all_keyword */
};

/* Struct for type ET_CONVERT_FUNCTION */
struct S703 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* types */
	T0* volatile a3; /* colon */
};

/* Struct for type ET_CONVERT_PROCEDURE */
struct S704 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* types */
	T0* volatile a3; /* left_parenthesis */
	T0* volatile a4; /* right_parenthesis */
};

/* Struct for type ET_ATTRIBUTE */
struct S706 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* declared_type */
	T0* volatile a2; /* implementation_class */
	T0* volatile a3; /* assigner */
	T0* volatile a4; /* clients */
	T0* volatile a5; /* semicolon */
	T0* volatile a6; /* feature_clause */
	T0* volatile a7; /* first_note */
	T0* volatile a8; /* object_tests */
	T0* volatile a9; /* iteration_components */
	T0* volatile a10; /* inline_separate_arguments */
	T0* volatile a11; /* extended_name */
	T6 volatile a12; /* hash_code */
	T0* volatile a13; /* implementation_feature */
	T0* volatile a14; /* frozen_keyword */
	T0* volatile a15; /* synonym */
	T6 volatile a16; /* first_seed */
	T0* volatile a17; /* other_seeds */
	T1 volatile a18; /* validity_checked */
	T1 volatile a19; /* has_validity_error */
	T0* volatile a20; /* first_precursor */
	T0* volatile a21; /* other_precursors */
	T6 volatile a22; /* id */
	T6 volatile a23; /* version */
};

/* Struct for type ET_EXTENDED_ATTRIBUTE */
struct S707 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* declared_type */
	T0* volatile a2; /* implementation_class */
	T0* volatile a3; /* assigner */
	T0* volatile a4; /* obsolete_message */
	T0* volatile a5; /* preconditions */
	T0* volatile a6; /* locals */
	T0* volatile a7; /* compound */
	T0* volatile a8; /* postconditions */
	T0* volatile a9; /* rescue_clause */
	T0* volatile a10; /* clients */
	T0* volatile a11; /* end_keyword */
	T0* volatile a12; /* semicolon */
	T0* volatile a13; /* feature_clause */
	T0* volatile a14; /* first_note */
	T0* volatile a15; /* object_tests */
	T0* volatile a16; /* iteration_components */
	T0* volatile a17; /* inline_separate_arguments */
	T0* volatile a18; /* extended_name */
	T6 volatile a19; /* hash_code */
	T0* volatile a20; /* implementation_feature */
	T0* volatile a21; /* frozen_keyword */
	T0* volatile a22; /* synonym */
	T6 volatile a23; /* first_seed */
	T0* volatile a24; /* other_seeds */
	T1 volatile a25; /* validity_checked */
	T1 volatile a26; /* has_validity_error */
	T0* volatile a27; /* first_precursor */
	T0* volatile a28; /* other_precursors */
	T6 volatile a29; /* id */
	T6 volatile a30; /* version */
};

/* Struct for type ET_CONSTANT_ATTRIBUTE */
struct S708 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* declared_type */
	T0* volatile a2; /* constant */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* assigner */
	T0* volatile a5; /* clients */
	T0* volatile a6; /* is_keyword */
	T0* volatile a7; /* semicolon */
	T0* volatile a8; /* feature_clause */
	T0* volatile a9; /* first_note */
	T0* volatile a10; /* object_tests */
	T0* volatile a11; /* iteration_components */
	T0* volatile a12; /* inline_separate_arguments */
	T0* volatile a13; /* extended_name */
	T6 volatile a14; /* hash_code */
	T0* volatile a15; /* implementation_feature */
	T0* volatile a16; /* frozen_keyword */
	T0* volatile a17; /* synonym */
	T6 volatile a18; /* first_seed */
	T0* volatile a19; /* other_seeds */
	T1 volatile a20; /* validity_checked */
	T1 volatile a21; /* has_validity_error */
	T0* volatile a22; /* first_precursor */
	T0* volatile a23; /* other_precursors */
	T6 volatile a24; /* id */
	T6 volatile a25; /* version */
};

/* Struct for type ET_UNIQUE_ATTRIBUTE */
struct S709 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* declared_type */
	T0* volatile a2; /* implementation_class */
	T0* volatile a3; /* assigner */
	T0* volatile a4; /* clients */
	T0* volatile a5; /* is_keyword */
	T0* volatile a6; /* unique_keyword */
	T0* volatile a7; /* semicolon */
	T0* volatile a8; /* feature_clause */
	T0* volatile a9; /* first_note */
	T0* volatile a10; /* object_tests */
	T0* volatile a11; /* iteration_components */
	T0* volatile a12; /* inline_separate_arguments */
	T0* volatile a13; /* constant */
	T0* volatile a14; /* extended_name */
	T6 volatile a15; /* hash_code */
	T0* volatile a16; /* implementation_feature */
	T0* volatile a17; /* frozen_keyword */
	T0* volatile a18; /* synonym */
	T6 volatile a19; /* id */
	T6 volatile a20; /* first_seed */
	T0* volatile a21; /* other_seeds */
	T1 volatile a22; /* validity_checked */
	T1 volatile a23; /* has_validity_error */
	T0* volatile a24; /* first_precursor */
	T0* volatile a25; /* other_precursors */
	T6 volatile a26; /* version */
};

/* Struct for type ET_DO_FUNCTION */
struct S710 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* declared_type */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* assigner */
	T0* volatile a5; /* obsolete_message */
	T0* volatile a6; /* preconditions */
	T0* volatile a7; /* locals */
	T0* volatile a8; /* compound */
	T0* volatile a9; /* postconditions */
	T0* volatile a10; /* rescue_clause */
	T0* volatile a11; /* clients */
	T0* volatile a12; /* is_keyword */
	T0* volatile a13; /* end_keyword */
	T0* volatile a14; /* semicolon */
	T0* volatile a15; /* feature_clause */
	T0* volatile a16; /* first_note */
	T0* volatile a17; /* object_tests */
	T0* volatile a18; /* iteration_components */
	T0* volatile a19; /* inline_separate_arguments */
	T0* volatile a20; /* extended_name */
	T6 volatile a21; /* hash_code */
	T0* volatile a22; /* implementation_feature */
	T0* volatile a23; /* frozen_keyword */
	T0* volatile a24; /* synonym */
	T6 volatile a25; /* first_seed */
	T0* volatile a26; /* other_seeds */
	T1 volatile a27; /* validity_checked */
	T1 volatile a28; /* has_validity_error */
	T0* volatile a29; /* first_precursor */
	T0* volatile a30; /* other_precursors */
	T6 volatile a31; /* id */
	T6 volatile a32; /* version */
};

/* Struct for type ET_ONCE_FUNCTION */
struct S711 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* declared_type */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* assigner */
	T0* volatile a5; /* obsolete_message */
	T0* volatile a6; /* preconditions */
	T0* volatile a7; /* locals */
	T0* volatile a8; /* compound */
	T0* volatile a9; /* postconditions */
	T0* volatile a10; /* rescue_clause */
	T0* volatile a11; /* clients */
	T0* volatile a12; /* is_keyword */
	T0* volatile a13; /* end_keyword */
	T0* volatile a14; /* semicolon */
	T0* volatile a15; /* feature_clause */
	T0* volatile a16; /* first_note */
	T0* volatile a17; /* object_tests */
	T0* volatile a18; /* iteration_components */
	T0* volatile a19; /* inline_separate_arguments */
	T0* volatile a20; /* extended_name */
	T6 volatile a21; /* hash_code */
	T0* volatile a22; /* implementation_feature */
	T0* volatile a23; /* frozen_keyword */
	T0* volatile a24; /* keys */
	T0* volatile a25; /* synonym */
	T6 volatile a26; /* first_seed */
	T0* volatile a27; /* other_seeds */
	T1 volatile a28; /* validity_checked */
	T1 volatile a29; /* has_validity_error */
	T0* volatile a30; /* first_precursor */
	T0* volatile a31; /* other_precursors */
	T6 volatile a32; /* id */
	T6 volatile a33; /* version */
};

/* Struct for type ET_DEFERRED_FUNCTION */
struct S712 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* declared_type */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* assigner */
	T0* volatile a5; /* obsolete_message */
	T0* volatile a6; /* preconditions */
	T0* volatile a7; /* postconditions */
	T0* volatile a8; /* clients */
	T0* volatile a9; /* is_keyword */
	T0* volatile a10; /* deferred_keyword */
	T0* volatile a11; /* end_keyword */
	T0* volatile a12; /* semicolon */
	T0* volatile a13; /* feature_clause */
	T0* volatile a14; /* first_note */
	T0* volatile a15; /* object_tests */
	T0* volatile a16; /* iteration_components */
	T0* volatile a17; /* inline_separate_arguments */
	T0* volatile a18; /* extended_name */
	T6 volatile a19; /* hash_code */
	T0* volatile a20; /* implementation_feature */
	T0* volatile a21; /* frozen_keyword */
	T0* volatile a22; /* synonym */
	T6 volatile a23; /* first_seed */
	T0* volatile a24; /* other_seeds */
	T1 volatile a25; /* validity_checked */
	T1 volatile a26; /* has_validity_error */
	T0* volatile a27; /* first_precursor */
	T0* volatile a28; /* other_precursors */
	T6 volatile a29; /* id */
	T6 volatile a30; /* version */
};

/* Struct for type ET_EXTERNAL_FUNCTION */
struct S714 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* declared_type */
	T0* volatile a3; /* language */
	T0* volatile a4; /* implementation_class */
	T0* volatile a5; /* assigner */
	T0* volatile a6; /* obsolete_message */
	T0* volatile a7; /* preconditions */
	T0* volatile a8; /* postconditions */
	T0* volatile a9; /* alias_clause */
	T0* volatile a10; /* clients */
	T0* volatile a11; /* is_keyword */
	T0* volatile a12; /* end_keyword */
	T0* volatile a13; /* semicolon */
	T0* volatile a14; /* feature_clause */
	T0* volatile a15; /* first_note */
	T0* volatile a16; /* object_tests */
	T0* volatile a17; /* iteration_components */
	T0* volatile a18; /* inline_separate_arguments */
	T8 volatile a19; /* builtin_class_code */
	T0* volatile a20; /* extended_name */
	T6 volatile a21; /* hash_code */
	T0* volatile a22; /* implementation_feature */
	T8 volatile a23; /* builtin_feature_code */
	T0* volatile a24; /* frozen_keyword */
	T0* volatile a25; /* synonym */
	T6 volatile a26; /* first_seed */
	T0* volatile a27; /* other_seeds */
	T1 volatile a28; /* validity_checked */
	T1 volatile a29; /* has_validity_error */
	T0* volatile a30; /* first_precursor */
	T0* volatile a31; /* other_precursors */
	T6 volatile a32; /* id */
	T6 volatile a33; /* version */
};

/* Struct for type ET_DO_PROCEDURE */
struct S715 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* extended_name */
	T6 volatile a2; /* hash_code */
	T0* volatile a3; /* arguments */
	T0* volatile a4; /* end_keyword */
	T0* volatile a5; /* clients */
	T0* volatile a6; /* implementation_class */
	T0* volatile a7; /* implementation_feature */
	T0* volatile a8; /* iteration_components */
	T6 volatile a9; /* first_seed */
	T0* volatile a10; /* rescue_clause */
	T0* volatile a11; /* locals */
	T0* volatile a12; /* object_tests */
	T0* volatile a13; /* compound */
	T0* volatile a14; /* other_seeds */
	T0* volatile a15; /* obsolete_message */
	T0* volatile a16; /* preconditions */
	T0* volatile a17; /* postconditions */
	T0* volatile a18; /* is_keyword */
	T0* volatile a19; /* semicolon */
	T0* volatile a20; /* feature_clause */
	T0* volatile a21; /* first_note */
	T0* volatile a22; /* inline_separate_arguments */
	T0* volatile a23; /* first_precursor */
	T0* volatile a24; /* other_precursors */
	T0* volatile a25; /* frozen_keyword */
	T0* volatile a26; /* synonym */
	T1 volatile a27; /* validity_checked */
	T1 volatile a28; /* has_validity_error */
	T6 volatile a29; /* id */
	T6 volatile a30; /* version */
};

/* Struct for type ET_ONCE_PROCEDURE */
struct S716 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* implementation_class */
	T0* volatile a3; /* obsolete_message */
	T0* volatile a4; /* preconditions */
	T0* volatile a5; /* locals */
	T0* volatile a6; /* compound */
	T0* volatile a7; /* postconditions */
	T0* volatile a8; /* rescue_clause */
	T0* volatile a9; /* clients */
	T0* volatile a10; /* is_keyword */
	T0* volatile a11; /* end_keyword */
	T0* volatile a12; /* semicolon */
	T0* volatile a13; /* feature_clause */
	T0* volatile a14; /* first_note */
	T0* volatile a15; /* object_tests */
	T0* volatile a16; /* iteration_components */
	T0* volatile a17; /* inline_separate_arguments */
	T0* volatile a18; /* extended_name */
	T6 volatile a19; /* hash_code */
	T0* volatile a20; /* implementation_feature */
	T0* volatile a21; /* frozen_keyword */
	T0* volatile a22; /* keys */
	T0* volatile a23; /* synonym */
	T6 volatile a24; /* first_seed */
	T0* volatile a25; /* other_seeds */
	T1 volatile a26; /* validity_checked */
	T1 volatile a27; /* has_validity_error */
	T0* volatile a28; /* first_precursor */
	T0* volatile a29; /* other_precursors */
	T6 volatile a30; /* id */
	T6 volatile a31; /* version */
};

/* Struct for type ET_DEFERRED_PROCEDURE */
struct S717 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* implementation_feature */
	T0* volatile a2; /* deferred_keyword */
	T0* volatile a3; /* extended_name */
	T6 volatile a4; /* hash_code */
	T0* volatile a5; /* arguments */
	T0* volatile a6; /* end_keyword */
	T0* volatile a7; /* clients */
	T0* volatile a8; /* implementation_class */
	T0* volatile a9; /* iteration_components */
	T6 volatile a10; /* first_seed */
	T0* volatile a11; /* other_seeds */
	T0* volatile a12; /* object_tests */
	T0* volatile a13; /* obsolete_message */
	T0* volatile a14; /* preconditions */
	T0* volatile a15; /* postconditions */
	T0* volatile a16; /* is_keyword */
	T0* volatile a17; /* semicolon */
	T0* volatile a18; /* feature_clause */
	T0* volatile a19; /* first_note */
	T0* volatile a20; /* inline_separate_arguments */
	T0* volatile a21; /* first_precursor */
	T0* volatile a22; /* other_precursors */
	T0* volatile a23; /* frozen_keyword */
	T0* volatile a24; /* synonym */
	T1 volatile a25; /* validity_checked */
	T1 volatile a26; /* has_validity_error */
	T6 volatile a27; /* id */
	T6 volatile a28; /* version */
};

/* Struct for type ET_EXTERNAL_PROCEDURE */
struct S718 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* language */
	T0* volatile a3; /* implementation_class */
	T0* volatile a4; /* obsolete_message */
	T0* volatile a5; /* preconditions */
	T0* volatile a6; /* postconditions */
	T0* volatile a7; /* alias_clause */
	T0* volatile a8; /* clients */
	T0* volatile a9; /* is_keyword */
	T0* volatile a10; /* end_keyword */
	T0* volatile a11; /* semicolon */
	T0* volatile a12; /* feature_clause */
	T0* volatile a13; /* first_note */
	T0* volatile a14; /* object_tests */
	T0* volatile a15; /* iteration_components */
	T0* volatile a16; /* inline_separate_arguments */
	T8 volatile a17; /* builtin_class_code */
	T0* volatile a18; /* extended_name */
	T6 volatile a19; /* hash_code */
	T0* volatile a20; /* implementation_feature */
	T8 volatile a21; /* builtin_feature_code */
	T0* volatile a22; /* frozen_keyword */
	T0* volatile a23; /* synonym */
	T6 volatile a24; /* first_seed */
	T0* volatile a25; /* other_seeds */
	T1 volatile a26; /* validity_checked */
	T1 volatile a27; /* has_validity_error */
	T0* volatile a28; /* first_precursor */
	T0* volatile a29; /* other_precursors */
	T6 volatile a30; /* id */
	T6 volatile a31; /* version */
};

/* Struct for type ET_ALIASED_FEATURE_NAME */
struct S719 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* feature_name */
	T0* volatile a2; /* alias_names */
};

/* Struct for type ET_ALIAS_FREE_NAME */
struct S720 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* alias_keyword */
	T0* volatile a2; /* alias_string */
	T2 volatile a3; /* code */
	T6 volatile a4; /* hash_code */
	T0* volatile a5; /* convert_keyword */
	T6 volatile a6; /* seed */
};

/* Struct for type ET_CLASS_ASSERTION */
struct S723 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_keyword */
};

/* Struct for type ET_LABELED_ACTUAL_PARAMETER */
struct S724 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* label_item */
	T0* volatile a2; /* declared_type */
};

/* Struct for type ET_LIKE_FEATURE */
struct S726 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* type_mark */
	T0* volatile a3; /* like_keyword */
	T6 volatile a4; /* seed */
	T1 volatile a5; /* is_procedure */
};

/* Struct for type ET_QUALIFIED_LIKE_BRACED_TYPE */
struct S728 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* qualified_name */
	T0* volatile a2; /* type_mark */
	T0* volatile a3; /* like_keyword */
	T0* volatile a4; /* target_type */
	T0* volatile a5; /* implementation_class */
	T0* volatile a6; /* left_brace */
	T0* volatile a7; /* right_brace */
};

/* Struct for type ET_QUALIFIED_LIKE_TYPE */
struct S729 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* qualified_name */
	T0* volatile a2; /* type_mark */
	T0* volatile a3; /* target_type */
	T0* volatile a4; /* implementation_class */
};

/* Struct for type ET_ASSIGNER_INSTRUCTION */
struct S730 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* call */
	T0* volatile a2; /* source */
	T0* volatile a3; /* assign_symbol */
	T0* volatile a4; /* name */
};

/* Struct for type ET_ASSIGNMENT */
struct S731 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* source */
	T0* volatile a3; /* assign_symbol */
	T1 volatile a4; /* no_source_twin */
};

/* Struct for type ET_ASSIGNMENT_ATTEMPT */
struct S732 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* source */
	T0* volatile a3; /* assign_attempt_symbol */
};

/* Struct for type ET_LOOP_INSTRUCTION */
struct S733 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* from_compound */
	T0* volatile a2; /* until_conditional */
	T0* volatile a3; /* loop_compound */
	T0* volatile a4; /* end_keyword */
	T0* volatile a5; /* invariant_part */
	T0* volatile a6; /* variant_part */
	T1 volatile a7; /* has_old_variant_syntax */
};

/* Struct for type ET_BANG_INSTRUCTION */
struct S735 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* target */
	T0* volatile a3; /* creation_call */
	T0* volatile a4; /* left_bang */
	T0* volatile a5; /* right_bang */
	T0* volatile a6; /* default_create_name */
	T0* volatile a7; /* internal_separate_target */
};

/* Struct for type ET_QUALIFIED_CALL */
struct S736 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* arguments */
	T0* volatile a2; /* qualified_name */
};

/* Struct for type ET_CREATE_INSTRUCTION */
struct S738 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* creation_type */
	T0* volatile a2; /* target */
	T0* volatile a3; /* creation_call */
	T0* volatile a4; /* create_keyword */
	T0* volatile a5; /* creation_region */
	T0* volatile a6; /* default_create_name */
	T0* volatile a7; /* internal_separate_target */
};

/* Struct for type ET_CHOICE_RANGE */
struct S739 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* lower */
	T0* volatile a2; /* upper */
	T0* volatile a3; /* dotdot */
};

/* Struct for type ET_QUALIFIED_CALL_INSTRUCTION */
struct S740 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* qualified_name */
	T0* volatile a3; /* arguments */
	T0* volatile a4; /* parenthesis_call */
};

/* Struct for type ET_PRECURSOR_INSTRUCTION */
struct S741 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parent_name */
	T0* volatile a2; /* arguments */
	T0* volatile a3; /* precursor_keyword */
	T1 volatile a4; /* is_parent_prefixed */
	T0* volatile a5; /* parenthesis_call */
	T0* volatile a6; /* parent_type */
};

/* Struct for type ET_STATIC_CALL_INSTRUCTION */
struct S743 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* static_type */
	T0* volatile a2; /* feature_keyword */
	T0* volatile a3; /* qualified_name */
	T0* volatile a4; /* arguments */
	T0* volatile a5; /* parenthesis_call */
};

/* Struct for type ET_PRECURSOR_EXPRESSION */
struct S744 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* parent_name */
	T0* volatile a3; /* arguments */
	T0* volatile a4; /* precursor_keyword */
	T1 volatile a5; /* is_parent_prefixed */
	T0* volatile a6; /* parenthesis_call */
	T0* volatile a7; /* parent_type */
};

/* Struct for type ET_FEATURE_ADDRESS */
struct S745 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* dollar */
	T0* volatile a3; /* name */
};

/* Struct for type ET_CURRENT_ADDRESS */
struct S746 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* dollar */
	T0* volatile a3; /* current_keyword */
};

/* Struct for type ET_RESULT_ADDRESS */
struct S747 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* dollar */
	T0* volatile a3; /* result_keyword */
};

/* Struct for type ET_EXPRESSION_ADDRESS */
struct S748 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* dollar */
	T0* volatile a3; /* expression */
};

/* Struct for type ET_INFIX_EXPRESSION */
struct S749 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* left */
	T0* volatile a3; /* name */
	T0* volatile a4; /* right */
	T1 volatile a5; /* is_boolean_operator */
};

/* Struct for type ET_INFIX_AND_THEN_OPERATOR */
struct S750 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* and_keyword */
	T0* volatile a2; /* then_keyword */
	T6 volatile a3; /* seed */
};

/* Struct for type ET_INFIX_OR_ELSE_OPERATOR */
struct S751 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* or_keyword */
	T0* volatile a2; /* else_keyword */
	T6 volatile a3; /* seed */
};

/* Struct for type ET_EQUALITY_EXPRESSION */
struct S752 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* left */
	T0* volatile a3; /* operator */
	T0* volatile a4; /* right */
};

/* Struct for type ET_OBJECT_EQUALITY_EXPRESSION */
struct S753 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* left */
	T0* volatile a3; /* operator */
	T0* volatile a4; /* right */
};

/* Struct for type ET_MANIFEST_TYPE */
struct S754 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* type */
	T0* volatile a3; /* left_brace */
	T0* volatile a4; /* right_brace */
};

/* Struct for type ET_PREFIX_EXPRESSION */
struct S755 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* name */
	T0* volatile a3; /* expression */
	T1 volatile a4; /* is_boolean_operator */
};

/* Struct for type ET_OLD_EXPRESSION */
struct S756 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* old_keyword */
	T0* volatile a3; /* expression */
};

/* Struct for type ET_OLD_OBJECT_TEST */
struct S757 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* index */
	T0* volatile a3; /* attached_keyword */
	T0* volatile a4; /* as_keyword */
	T0* volatile a5; /* left_brace */
	T0* volatile a6; /* colon */
	T0* volatile a7; /* declared_type */
	T0* volatile a8; /* right_brace */
	T0* volatile a9; /* expression */
};

/* Struct for type ET_OBJECT_TEST */
struct S758 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* attached_keyword */
	T0* volatile a3; /* declared_type */
	T0* volatile a4; /* expression */
};

/* Struct for type ET_NAMED_OBJECT_TEST */
struct S759 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* index */
	T0* volatile a3; /* as_keyword */
	T0* volatile a4; /* expression */
	T0* volatile a5; /* attached_keyword */
	T0* volatile a6; /* declared_type */
};

/* Struct for type ET_ONCE_MANIFEST_STRING */
struct S760 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* once_keyword */
	T0* volatile a3; /* manifest_string */
	T6 volatile a4; /* id */
};

/* Struct for type ET_ITERATION_CURSOR */
struct S761 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* at_symbol */
	T0* volatile a3; /* item_name */
};

/* Struct for type ET_DO_FUNCTION_INLINE_AGENT */
struct S763 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* formal_arguments */
	T0* volatile a3; /* declared_type */
	T0* volatile a4; /* end_keyword */
	T0* volatile a5; /* implicit_result */
	T0* volatile a6; /* object_tests */
	T0* volatile a7; /* iteration_components */
	T0* volatile a8; /* inline_separate_arguments */
	T0* volatile a9; /* actual_arguments */
	T0* volatile a10; /* agent_keyword */
	T0* volatile a11; /* preconditions */
	T0* volatile a12; /* locals */
	T0* volatile a13; /* compound */
	T0* volatile a14; /* postconditions */
	T0* volatile a15; /* rescue_clause */
	T0* volatile a16; /* target */
	T6 volatile a17; /* result_index */
	T6 volatile a18; /* attached_result_index */
};

/* Struct for type ET_ONCE_FUNCTION_INLINE_AGENT */
struct S764 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* formal_arguments */
	T0* volatile a3; /* declared_type */
	T0* volatile a4; /* end_keyword */
	T0* volatile a5; /* implicit_result */
	T0* volatile a6; /* object_tests */
	T0* volatile a7; /* iteration_components */
	T0* volatile a8; /* inline_separate_arguments */
	T0* volatile a9; /* actual_arguments */
	T0* volatile a10; /* agent_keyword */
	T0* volatile a11; /* preconditions */
	T0* volatile a12; /* locals */
	T0* volatile a13; /* keys */
	T0* volatile a14; /* compound */
	T0* volatile a15; /* postconditions */
	T0* volatile a16; /* rescue_clause */
	T0* volatile a17; /* target */
	T6 volatile a18; /* result_index */
	T6 volatile a19; /* attached_result_index */
};

/* Struct for type ET_EXTERNAL_FUNCTION_INLINE_AGENT */
struct S765 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* language */
	T0* volatile a3; /* object_tests */
	T0* volatile a4; /* iteration_components */
	T0* volatile a5; /* inline_separate_arguments */
	T0* volatile a6; /* actual_arguments */
	T0* volatile a7; /* agent_keyword */
	T0* volatile a8; /* preconditions */
	T0* volatile a9; /* alias_clause */
	T0* volatile a10; /* postconditions */
	T0* volatile a11; /* formal_arguments */
	T0* volatile a12; /* declared_type */
	T0* volatile a13; /* end_keyword */
	T0* volatile a14; /* implicit_result */
	T0* volatile a15; /* target */
	T6 volatile a16; /* result_index */
	T6 volatile a17; /* attached_result_index */
};

/* Struct for type ET_DO_PROCEDURE_INLINE_AGENT */
struct S766 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* formal_arguments */
	T0* volatile a3; /* end_keyword */
	T0* volatile a4; /* object_tests */
	T0* volatile a5; /* iteration_components */
	T0* volatile a6; /* inline_separate_arguments */
	T0* volatile a7; /* actual_arguments */
	T0* volatile a8; /* agent_keyword */
	T0* volatile a9; /* preconditions */
	T0* volatile a10; /* locals */
	T0* volatile a11; /* compound */
	T0* volatile a12; /* postconditions */
	T0* volatile a13; /* rescue_clause */
	T0* volatile a14; /* target */
	T6 volatile a15; /* result_index */
	T6 volatile a16; /* attached_result_index */
};

/* Struct for type ET_ONCE_PROCEDURE_INLINE_AGENT */
struct S767 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* formal_arguments */
	T0* volatile a3; /* end_keyword */
	T0* volatile a4; /* object_tests */
	T0* volatile a5; /* iteration_components */
	T0* volatile a6; /* inline_separate_arguments */
	T0* volatile a7; /* actual_arguments */
	T0* volatile a8; /* agent_keyword */
	T0* volatile a9; /* preconditions */
	T0* volatile a10; /* locals */
	T0* volatile a11; /* keys */
	T0* volatile a12; /* compound */
	T0* volatile a13; /* postconditions */
	T0* volatile a14; /* rescue_clause */
	T0* volatile a15; /* target */
	T6 volatile a16; /* result_index */
	T6 volatile a17; /* attached_result_index */
};

/* Struct for type ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
struct S768 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* language */
	T0* volatile a3; /* object_tests */
	T0* volatile a4; /* iteration_components */
	T0* volatile a5; /* inline_separate_arguments */
	T0* volatile a6; /* actual_arguments */
	T0* volatile a7; /* agent_keyword */
	T0* volatile a8; /* preconditions */
	T0* volatile a9; /* alias_clause */
	T0* volatile a10; /* postconditions */
	T0* volatile a11; /* formal_arguments */
	T0* volatile a12; /* end_keyword */
	T0* volatile a13; /* target */
	T6 volatile a14; /* result_index */
	T6 volatile a15; /* attached_result_index */
};

/* Struct for type ET_AGENT_OPEN_TARGET */
struct S769 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* left_brace */
	T0* volatile a3; /* right_brace */
	T6 volatile a4; /* index */
};

/* Struct for type ET_AGENT_TYPED_OPEN_ARGUMENT */
struct S770 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* left_brace */
	T0* volatile a3; /* right_brace */
	T0* volatile a4; /* question_mark */
	T6 volatile a5; /* index */
};

/* Struct for type ET_QUERY_LIST */
struct S773 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* declared_count */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
	T6 volatile a4; /* attribute_count */
	T6 volatile a5; /* declared_attribute_count */
};

/* Struct for type ET_PROCEDURE_LIST */
struct S774 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* declared_count */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
};

/* Struct for type ET_TAGGED_ASSERTION */
struct S776 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* tag */
	T0* volatile a2; /* untagged_assertion */
};

/* Struct for type ET_FILE_POSITION */
struct S778 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* compressed_position */
	T0* volatile a2; /* filename */
};

/* Struct for type SPECIAL [ET_CLASS] */
struct S779 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLASS] */
struct S780 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_CLASS] */
struct S781 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type ET_BASE_TYPE_LIST */
struct S782 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_CLASS_CODES */
struct S783 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ECF_AST_FACTORY */
struct S784 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_CELL [detachable ET_ECF_SYSTEM] */
struct S785 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type TUPLE [ET_ECF_TARGET] */
struct S786 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_TARGET]] */
struct S787 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM_PARSER] */
struct S788 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type XM_ELEMENT */
struct S789 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* namespace */
	T0* volatile a3; /* parent */
	T0* volatile a4; /* children */
};

/* Struct for type XM_POSITION_TABLE */
struct S790 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* table */
};

/* Struct for type TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8] */
struct S791 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type PROCEDURE [TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8]] */
struct S792 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, PROCEDURE [TUPLE [ET_ECF_TARGET]], DS_CELL [detachable ET_ECF_SYSTEM]] */
struct S793 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
};

/* Struct for type DS_HASH_TABLE [ET_ECF_LIBRARY, STRING_8] */
struct S795 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* key_equality_tester */
	T0* volatile a5; /* internal_keys */
	T6 volatile a6; /* count */
	T6 volatile a7; /* last_position */
	T6 volatile a8; /* free_slot */
	T6 volatile a9; /* position */
	T6 volatile a10; /* capacity */
	T0* volatile a11; /* equality_tester */
	T0* volatile a12; /* key_storage */
	T0* volatile a13; /* clashes */
	T6 volatile a14; /* modulus */
	T0* volatile a15; /* slots */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* clashes_previous_position */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
struct S796 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
struct S797 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T0* volatile a4; /* internal_keys */
	T6 volatile a5; /* count */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T6 volatile a9; /* capacity */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* internal_cursor */
	T0* volatile a12; /* key_storage */
	T0* volatile a13; /* clashes */
	T6 volatile a14; /* modulus */
	T0* volatile a15; /* slots */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* clashes_previous_position */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type XM_EIFFEL_PARSER */
struct S799 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_last_error_description */
	T0* volatile a2; /* scanner */
	T0* volatile a3; /* error_positions */
	T0* volatile a4; /* scanners */
	T1 volatile a5; /* in_external_dtd */
	T0* volatile a6; /* callbacks */
	T0* volatile a7; /* entities */
	T0* volatile a8; /* pe_entities */
	T0* volatile a9; /* dtd_resolver */
	T0* volatile a10; /* entity_resolver */
	T1 volatile a11; /* use_namespaces */
	T6 volatile a12; /* string_mode */
	T0* volatile a13; /* dtd_callbacks */
	T0* volatile a14; /* last_string_value */
	T0* volatile a15; /* yyss */
	T0* volatile a16; /* yyspecial_routines1 */
	T6 volatile a17; /* yyvsc1 */
	T0* volatile a18; /* yyvs1 */
	T0* volatile a19; /* yyspecial_routines2 */
	T6 volatile a20; /* yyvsc2 */
	T0* volatile a21; /* yyvs2 */
	T0* volatile a22; /* yyspecial_routines3 */
	T6 volatile a23; /* yyvsc3 */
	T0* volatile a24; /* yyvs3 */
	T0* volatile a25; /* yyspecial_routines4 */
	T6 volatile a26; /* yyvsc4 */
	T0* volatile a27; /* yyvs4 */
	T0* volatile a28; /* yyspecial_routines5 */
	T6 volatile a29; /* yyvsc5 */
	T0* volatile a30; /* yyvs5 */
	T0* volatile a31; /* yyspecial_routines6 */
	T6 volatile a32; /* yyvsc6 */
	T0* volatile a33; /* yyvs6 */
	T0* volatile a34; /* yyspecial_routines7 */
	T6 volatile a35; /* yyvsc7 */
	T0* volatile a36; /* yyvs7 */
	T0* volatile a37; /* yyspecial_routines8 */
	T6 volatile a38; /* yyvsc8 */
	T0* volatile a39; /* yyvs8 */
	T0* volatile a40; /* yyspecial_routines9 */
	T6 volatile a41; /* yyvsc9 */
	T0* volatile a42; /* yyvs9 */
	T0* volatile a43; /* yyspecial_routines10 */
	T6 volatile a44; /* yyvsc10 */
	T0* volatile a45; /* yyvs10 */
	T0* volatile a46; /* yyspecial_routines11 */
	T6 volatile a47; /* yyvsc11 */
	T0* volatile a48; /* yyvs11 */
	T0* volatile a49; /* yytranslate */
	T0* volatile a50; /* yyr1 */
	T0* volatile a51; /* yytypes1 */
	T0* volatile a52; /* yytypes2 */
	T0* volatile a53; /* yydefact */
	T0* volatile a54; /* yydefgoto */
	T0* volatile a55; /* yypact */
	T0* volatile a56; /* yypgoto */
	T0* volatile a57; /* yytable */
	T0* volatile a58; /* yycheck */
	T6 volatile a59; /* yy_parsing_status */
	T6 volatile a60; /* yy_suspended_yystacksize */
	T6 volatile a61; /* yy_suspended_yystate */
	T6 volatile a62; /* yy_suspended_yyn */
	T6 volatile a63; /* yy_suspended_yychar1 */
	T6 volatile a64; /* yy_suspended_index */
	T6 volatile a65; /* yy_suspended_yyss_top */
	T6 volatile a66; /* yy_suspended_yy_goto */
	T6 volatile a67; /* yyssp */
	T6 volatile a68; /* error_count */
	T1 volatile a69; /* yy_lookahead_needed */
	T6 volatile a70; /* yyerrstatus */
	T6 volatile a71; /* last_token */
	T6 volatile a72; /* yyvsp1 */
	T6 volatile a73; /* yyvsp2 */
	T6 volatile a74; /* yyvsp3 */
	T6 volatile a75; /* yyvsp4 */
	T6 volatile a76; /* yyvsp5 */
	T6 volatile a77; /* yyvsp6 */
	T6 volatile a78; /* yyvsp7 */
	T6 volatile a79; /* yyvsp8 */
	T6 volatile a80; /* yyvsp9 */
	T6 volatile a81; /* yyvsp10 */
	T6 volatile a82; /* yyvsp11 */
	T0* volatile a83; /* last_detachable_any_value */
};

/* Struct for type XM_TREE_CALLBACKS_PIPE */
struct S800 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* start */
	T0* volatile a2; /* tree */
	T0* volatile a3; /* error */
	T0* volatile a4; /* last */
};

/* Struct for type XM_CALLBACKS_TO_TREE_FILTER */
struct S803 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_position_table */
	T0* volatile a2; /* document */
	T0* volatile a3; /* source_parser */
	T0* volatile a4; /* current_element */
	T0* volatile a5; /* namespace_cache */
	T0* volatile a6; /* next */
};

/* Struct for type ET_ECF_STATE */
struct S804 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* ise_version */
};

/* Struct for type ET_ECF_SYSTEM_CONFIG */
struct S805 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* filename */
	T0* volatile a3; /* universe */
	T0* volatile a4; /* ecf_namespace */
	T0* volatile a5; /* ecf_version */
	T0* volatile a6; /* description */
	T0* volatile a7; /* notes */
	T0* volatile a8; /* targets */
	T1 volatile a9; /* is_read_only */
	T0* volatile a10; /* uuid */
	T0* volatile a11; /* library_target */
};

/* Struct for type ET_ECF_LIBRARY */
struct S806 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* filename */
	T0* volatile a3; /* ecf_namespace */
	T0* volatile a4; /* ecf_version */
	T0* volatile a5; /* description */
	T0* volatile a6; /* notes */
	T0* volatile a7; /* targets */
	T1 volatile a8; /* is_read_only */
	T0* volatile a9; /* uuid */
	T0* volatile a10; /* library_target */
	T0* volatile a11; /* current_system */
	T0* volatile a12; /* library */
	T0* volatile a13; /* libraries */
	T0* volatile a14; /* clusters */
	T0* volatile a15; /* dotnet_assemblies */
	T0* volatile a16; /* implicit_attachment_type_mark */
	T0* volatile a17; /* master_classes */
	T0* volatile a18; /* master_class_mutex */
	T0* volatile a19; /* any_type */
	T0* volatile a20; /* detachable_any_type */
	T0* volatile a21; /* detachable_separate_any_type */
	T0* volatile a22; /* any_parent */
	T0* volatile a23; /* any_parents */
	T0* volatile a24; /* any_clients */
	T0* volatile a25; /* tuple_type */
	T0* volatile a26; /* detachable_tuple_type */
	T0* volatile a27; /* tuple_identity_type */
	T0* volatile a28; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a29; /* array_any_type */
	T0* volatile a30; /* array_detachable_any_type */
	T0* volatile a31; /* array_none_type */
	T0* volatile a32; /* array_identity_type */
	T0* volatile a33; /* boolean_type */
	T0* volatile a34; /* character_type */
	T0* volatile a35; /* character_8_type */
	T0* volatile a36; /* character_8_convert_feature */
	T0* volatile a37; /* character_32_type */
	T0* volatile a38; /* character_32_convert_feature */
	T0* volatile a39; /* double_type */
	T0* volatile a40; /* exception_type */
	T0* volatile a41; /* detachable_exception_type */
	T0* volatile a42; /* exception_manager_type */
	T0* volatile a43; /* function_identity_any_type */
	T0* volatile a44; /* immutable_string_8_type */
	T0* volatile a45; /* immutable_string_32_type */
	T0* volatile a46; /* integer_type */
	T0* volatile a47; /* integer_8_type */
	T0* volatile a48; /* integer_8_convert_feature */
	T0* volatile a49; /* integer_16_type */
	T0* volatile a50; /* integer_16_convert_feature */
	T0* volatile a51; /* integer_32_type */
	T0* volatile a52; /* integer_32_convert_feature */
	T0* volatile a53; /* integer_64_type */
	T0* volatile a54; /* integer_64_convert_feature */
	T0* volatile a55; /* ise_exception_manager_type */
	T0* volatile a56; /* iterable_detachable_separate_any_type */
	T0* volatile a57; /* natural_type */
	T0* volatile a58; /* natural_8_type */
	T0* volatile a59; /* natural_8_convert_feature */
	T0* volatile a60; /* natural_16_type */
	T0* volatile a61; /* natural_16_convert_feature */
	T0* volatile a62; /* natural_32_type */
	T0* volatile a63; /* natural_32_convert_feature */
	T0* volatile a64; /* natural_64_type */
	T0* volatile a65; /* natural_64_convert_feature */
	T0* volatile a66; /* none_type */
	T0* volatile a67; /* detachable_none_type */
	T0* volatile a68; /* pointer_type */
	T0* volatile a69; /* predicate_identity_type */
	T0* volatile a70; /* procedure_identity_type */
	T0* volatile a71; /* real_type */
	T0* volatile a72; /* real_32_type */
	T0* volatile a73; /* real_32_convert_feature */
	T0* volatile a74; /* real_64_type */
	T0* volatile a75; /* real_64_convert_feature */
	T0* volatile a76; /* routine_identity_type */
	T0* volatile a77; /* special_any_type */
	T0* volatile a78; /* special_detachable_any_type */
	T0* volatile a79; /* special_identity_type */
	T0* volatile a80; /* string_type */
	T0* volatile a81; /* detachable_string_type */
	T0* volatile a82; /* string_8_type */
	T0* volatile a83; /* detachable_string_8_type */
	T0* volatile a84; /* string_8_convert_feature */
	T0* volatile a85; /* string_32_type */
	T0* volatile a86; /* string_32_convert_feature */
	T0* volatile a87; /* system_object_type */
	T0* volatile a88; /* system_object_parents */
	T0* volatile a89; /* system_string_type */
	T0* volatile a90; /* type_detachable_any_type */
	T0* volatile a91; /* detachable_type_exception_type */
	T0* volatile a92; /* type_detachable_exception_type */
	T0* volatile a93; /* type_detachable_like_current_type */
	T0* volatile a94; /* type_identity_type */
	T0* volatile a95; /* typed_pointer_identity_type */
	T0* volatile a96; /* wide_character_type */
	T0* volatile a97; /* selected_target */
	T1 volatile a98; /* use_obsolete_syntax_mode */
	T1 volatile a99; /* obsolete_iteration_mode */
	T1 volatile a100; /* obsolete_routine_type_mode */
	T1 volatile a101; /* is_preparsed */
};

/* Struct for type TUPLE [ET_ADAPTED_LIBRARY] */
struct S808 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ADAPTED_LIBRARY]] */
struct S809 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type XM_DOCUMENT */
struct S810 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* root_element */
	T0* volatile a2; /* children */
};

/* Struct for type ET_COMPRESSED_POSITION */
struct S811 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* compressed_position */
};

/* Struct for type XM_STOP_ON_ERROR_FILTER */
struct S813 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_error */
	T0* volatile a2; /* last_error */
	T0* volatile a3; /* next */
};

/* Struct for type XM_ATTRIBUTE */
struct S814 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* namespace */
	T0* volatile a3; /* value */
	T0* volatile a4; /* parent */
};

/* Struct for type DS_HASH_TABLE [ET_ECF_SYSTEM_CONFIG, STRING_8] */
struct S815 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* position */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T0* volatile a8; /* key_equality_tester */
	T6 volatile a9; /* modulus */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T0* volatile a14; /* special_key_routines */
	T0* volatile a15; /* key_storage */
	T0* volatile a16; /* equality_tester */
	T6 volatile a17; /* free_slot */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [ET_ECF_TARGET, STRING_8] */
struct S816 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* position */
	T0* volatile a3; /* internal_cursor */
	T0* volatile a4; /* key_storage */
	T0* volatile a5; /* item_storage */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* modulus */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* key_equality_tester */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* equality_tester */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_ECF_TARGET_PARENT */
struct S817 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name_id */
	T0* volatile a2; /* location_id */
	T0* volatile a3; /* target */
};

/* Struct for type DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
struct S818 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
struct S819 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type ET_ECF_TARGETS */
struct S820 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* targets */
};

/* Struct for type DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, STRING_8] */
struct S821 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* key_equality_tester */
	T0* volatile a2; /* internal_keys */
	T0* volatile a3; /* hash_function */
	T6 volatile a4; /* position */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* found_position */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* item_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T6 volatile a19; /* free_slot */
	T0* volatile a20; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, STRING_8] */
struct S822 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* container */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_CELL [detachable ET_ECF_LIBRARY] */
struct S824 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type ET_ECF_ADAPTED_LIBRARY */
struct S825 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* options */
	T0* volatile a2; /* name_id */
	T0* volatile a3; /* filename_id */
	T0* volatile a4; /* universe */
	T0* volatile a5; /* target */
	T0* volatile a6; /* description */
	T0* volatile a7; /* classname_prefix */
	T1 volatile a8; /* is_read_only */
	T1 volatile a9; /* use_application_options */
	T0* volatile a10; /* class_options */
	T0* volatile a11; /* class_renamings */
	T0* volatile a12; /* visible_classes */
	T0* volatile a13; /* conditions */
	T0* volatile a14; /* notes */
	T0* volatile a15; /* name */
	T0* volatile a16; /* library */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, DS_CELL [detachable ET_ECF_LIBRARY]] */
struct S826 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type ET_ECF_DOTNET_ASSEMBLY */
struct S828 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* referenced_assemblies */
	T0* volatile a3; /* current_system */
	T0* volatile a4; /* dotnet_assembly */
	T0* volatile a5; /* name */
	T0* volatile a6; /* implicit_attachment_type_mark */
	T0* volatile a7; /* master_classes */
	T0* volatile a8; /* master_class_mutex */
	T0* volatile a9; /* any_type */
	T0* volatile a10; /* detachable_any_type */
	T0* volatile a11; /* detachable_separate_any_type */
	T0* volatile a12; /* any_parent */
	T0* volatile a13; /* any_parents */
	T0* volatile a14; /* any_clients */
	T0* volatile a15; /* tuple_type */
	T0* volatile a16; /* detachable_tuple_type */
	T0* volatile a17; /* tuple_identity_type */
	T0* volatile a18; /* unfolded_empty_tuple_actual_parameters */
	T0* volatile a19; /* array_any_type */
	T0* volatile a20; /* array_detachable_any_type */
	T0* volatile a21; /* array_none_type */
	T0* volatile a22; /* array_identity_type */
	T0* volatile a23; /* boolean_type */
	T0* volatile a24; /* character_type */
	T0* volatile a25; /* character_8_type */
	T0* volatile a26; /* character_8_convert_feature */
	T0* volatile a27; /* character_32_type */
	T0* volatile a28; /* character_32_convert_feature */
	T0* volatile a29; /* double_type */
	T0* volatile a30; /* exception_type */
	T0* volatile a31; /* detachable_exception_type */
	T0* volatile a32; /* exception_manager_type */
	T0* volatile a33; /* function_identity_any_type */
	T0* volatile a34; /* immutable_string_8_type */
	T0* volatile a35; /* immutable_string_32_type */
	T0* volatile a36; /* integer_type */
	T0* volatile a37; /* integer_8_type */
	T0* volatile a38; /* integer_8_convert_feature */
	T0* volatile a39; /* integer_16_type */
	T0* volatile a40; /* integer_16_convert_feature */
	T0* volatile a41; /* integer_32_type */
	T0* volatile a42; /* integer_32_convert_feature */
	T0* volatile a43; /* integer_64_type */
	T0* volatile a44; /* integer_64_convert_feature */
	T0* volatile a45; /* ise_exception_manager_type */
	T0* volatile a46; /* iterable_detachable_separate_any_type */
	T0* volatile a47; /* natural_type */
	T0* volatile a48; /* natural_8_type */
	T0* volatile a49; /* natural_8_convert_feature */
	T0* volatile a50; /* natural_16_type */
	T0* volatile a51; /* natural_16_convert_feature */
	T0* volatile a52; /* natural_32_type */
	T0* volatile a53; /* natural_32_convert_feature */
	T0* volatile a54; /* natural_64_type */
	T0* volatile a55; /* natural_64_convert_feature */
	T0* volatile a56; /* none_type */
	T0* volatile a57; /* detachable_none_type */
	T0* volatile a58; /* pointer_type */
	T0* volatile a59; /* predicate_identity_type */
	T0* volatile a60; /* procedure_identity_type */
	T0* volatile a61; /* real_type */
	T0* volatile a62; /* real_32_type */
	T0* volatile a63; /* real_32_convert_feature */
	T0* volatile a64; /* real_64_type */
	T0* volatile a65; /* real_64_convert_feature */
	T0* volatile a66; /* routine_identity_type */
	T0* volatile a67; /* special_any_type */
	T0* volatile a68; /* special_detachable_any_type */
	T0* volatile a69; /* special_identity_type */
	T0* volatile a70; /* string_type */
	T0* volatile a71; /* detachable_string_type */
	T0* volatile a72; /* string_8_type */
	T0* volatile a73; /* detachable_string_8_type */
	T0* volatile a74; /* string_8_convert_feature */
	T0* volatile a75; /* string_32_type */
	T0* volatile a76; /* string_32_convert_feature */
	T0* volatile a77; /* system_object_type */
	T0* volatile a78; /* system_object_parents */
	T0* volatile a79; /* system_string_type */
	T0* volatile a80; /* type_detachable_any_type */
	T0* volatile a81; /* detachable_type_exception_type */
	T0* volatile a82; /* type_detachable_exception_type */
	T0* volatile a83; /* type_detachable_like_current_type */
	T0* volatile a84; /* type_identity_type */
	T0* volatile a85; /* typed_pointer_identity_type */
	T0* volatile a86; /* wide_character_type */
	T1 volatile a87; /* is_read_only */
	T1 volatile a88; /* is_preparsed */
};

/* Struct for type ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
struct S830 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* options */
	T0* volatile a2; /* name_id */
	T0* volatile a3; /* filename_id */
	T0* volatile a4; /* universe */
	T0* volatile a5; /* target */
	T0* volatile a6; /* description */
	T0* volatile a7; /* assembly_culture */
	T0* volatile a8; /* assembly_key */
	T0* volatile a9; /* assembly_name */
	T0* volatile a10; /* assembly_version */
	T0* volatile a11; /* classname_prefix */
	T1 volatile a12; /* is_read_only */
	T0* volatile a13; /* class_options */
	T0* volatile a14; /* class_renamings */
	T0* volatile a15; /* conditions */
	T0* volatile a16; /* notes */
	T0* volatile a17; /* name */
	T0* volatile a18; /* dotnet_assembly */
};

/* Struct for type DS_HASH_TABLE [STRING_8, STRING_8] */
struct S831 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T6 volatile a3; /* found_position */
	T0* volatile a4; /* item_storage */
	T0* volatile a5; /* key_storage */
	T0* volatile a6; /* key_equality_tester */
	T0* volatile a7; /* internal_keys */
	T6 volatile a8; /* position */
	T6 volatile a9; /* free_slot */
	T6 volatile a10; /* last_position */
	T6 volatile a11; /* slots_position */
	T0* volatile a12; /* hash_function */
	T6 volatile a13; /* capacity */
	T0* volatile a14; /* equality_tester */
	T6 volatile a15; /* modulus */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* clashes */
	T0* volatile a19; /* slots */
	T0* volatile a20; /* special_key_routines */
};

/* Struct for type ET_ECF_ERROR */
struct S832 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* system_config */
	T0* volatile a4; /* position */
	T0* volatile a5; /* code */
};

/* Struct for type ET_ECF_OPTIONS */
struct S833 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* primary_options */
	T0* volatile a2; /* primary_debugs */
	T0* volatile a3; /* primary_assertions */
	T0* volatile a4; /* primary_warnings */
	T0* volatile a5; /* secondary_options */
	T0* volatile a6; /* description */
};

/* Struct for type ET_ECF_CLUSTERS */
struct S834 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* clusters */
};

/* Struct for type KL_AGENT_HASH_FUNCTION [STRING_8] */
struct S835 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* hash_agent */
};

/* Struct for type TUPLE [STRING_8] */
struct S836 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type FUNCTION [TUPLE [STRING_8], INTEGER_32] */
struct S837 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [KL_STRING_ROUTINES] */
struct S838 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type DS_HASH_TABLE [ET_IDENTIFIER, STRING_8] */
struct S840 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* key_equality_tester */
	T0* volatile a2; /* internal_keys */
	T0* volatile a3; /* hash_function */
	T6 volatile a4; /* position */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* found_position */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* item_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T6 volatile a19; /* free_slot */
	T0* volatile a20; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_CLUSTER] */
struct S842 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type ET_DYNAMIC_PRIMARY_TYPE */
struct S843 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* is_alive */
	T1 volatile a2; /* has_once_per_object_routines */
	T6 volatile a3; /* attribute_count */
	T0* volatile a4; /* queries */
	T6 volatile a5; /* id */
	T0* volatile a6; /* base_type */
	T0* volatile a7; /* base_class */
	T0* volatile a8; /* meta_type */
	T0* volatile a9; /* separate_type */
	T0* volatile a10; /* attached_separate_type */
	T0* volatile a11; /* attached_type */
	T0* volatile a12; /* conforming_ancestors */
	T0* volatile a13; /* procedures_by_seed */
	T0* volatile a14; /* procedures */
	T0* volatile a15; /* queries_by_seed */
	T1 volatile a16; /* has_redefined_copy_routine */
	T1 volatile a17; /* has_redefined_is_equal_routine */
	T6 volatile a18; /* hash_code */
	T0* volatile a19; /* next_type */
	T1 volatile a20; /* has_reference_attributes */
	T0* volatile a21; /* query_calls */
	T0* volatile a22; /* invariants */
	T0* volatile a23; /* procedure_calls */
	T1 volatile a24; /* has_static */
	T0* volatile a25; /* equality_expressions */
	T0* volatile a26; /* object_equality_expressions */
};

/* Struct for type ET_DYNAMIC_FEATURE_LIST */
struct S844 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
struct S845 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T6 volatile a4; /* position */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* modulus */
	T6 volatile a11; /* clashes_previous_position */
	T0* volatile a12; /* special_item_routines */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_key_routines */
	T0* volatile a16; /* key_storage */
	T6 volatile a17; /* free_slot */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type TUPLE [INTEGER_32] */
struct S846 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile z1;
};

/* Struct for type ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
struct S847 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* dynamic_type_set_count */
	T1 volatile a2; /* has_fatal_error */
	T0* volatile a3; /* current_dynamic_system */
	T1 volatile a4; /* catcall_error_mode */
	T1 volatile a5; /* catcall_warning_mode */
	T1 volatile a6; /* no_debug */
	T0* volatile a7; /* alive_conforming_descendants_per_type */
};

/* Struct for type TUPLE [ET_DYNAMIC_SYSTEM] */
struct S848 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type ET_DYNAMIC_FEATURE */
struct S850 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* preconditions */
	T0* volatile a2; /* postconditions */
	T0* volatile a3; /* invariants */
	T6 volatile a4; /* current_index */
	T0* volatile a5; /* static_feature */
	T0* volatile a6; /* dynamic_type_sets */
	T1 volatile a7; /* is_static */
	T6 volatile a8; /* id */
	T0* volatile a9; /* target_type */
	T0* volatile a10; /* result_type_set */
	T1 volatile a11; /* is_generated */
	T1 volatile a12; /* is_inlined */
	T8 volatile a13; /* builtin_class_code */
	T8 volatile a14; /* builtin_feature_code */
	T1 volatile a15; /* is_copy_routine */
	T1 volatile a16; /* is_is_equal_routine */
	T0* volatile a17; /* query_calls */
	T0* volatile a18; /* procedure_calls */
	T1 volatile a19; /* is_regular */
	T0* volatile a20; /* first_precursor */
	T1 volatile a21; /* is_creation */
	T0* volatile a22; /* other_precursors */
	T1 volatile a23; /* is_built */
	T1 volatile a24; /* is_separate_creation */
	T1 volatile a25; /* is_address */
};

/* Struct for type ET_NESTED_TYPE_CONTEXT */
struct S852 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* root_context */
	T0* volatile a3; /* storage */
};

/* Struct for type ET_FORMAL_PARAMETER_TYPE */
struct S853 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* type_mark */
	T6 volatile a3; /* index */
	T0* volatile a4; /* implementation_class */
};

/* Struct for type ARRAY [ET_TYPE] */
struct S854 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [ET_TYPE] */
struct S855 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ET_SYSTEM_MULTIPROCESSOR */
struct S858 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* benchmark_shown */
	T1 volatile a2; /* nested_benchmark_shown */
	T0* volatile a3; /* interface_checker */
	T0* volatile a4; /* stop_request */
	T0* volatile a5; /* eiffel_parser */
	T0* volatile a6; /* error_handler */
	T0* volatile a7; /* implementation_checker */
	T0* volatile a8; /* ise_version */
	T0* volatile a9; /* other_processors */
	T0* volatile a10; /* eiffel_preparser */
	T0* volatile a11; /* master_class_checker */
	T1 volatile a12; /* preparse_multiple_mode */
	T0* volatile a13; /* ast_factory */
	T1 volatile a14; /* use_note_keyword */
	T1 volatile a15; /* use_attached_keyword */
	T1 volatile a16; /* use_attribute_keyword */
	T1 volatile a17; /* use_reference_keyword */
	T1 volatile a18; /* use_detachable_keyword */
	T1 volatile a19; /* preparse_shallow_mode */
	T1 volatile a20; /* providers_enabled */
	T0* volatile a21; /* provider_checker */
	T0* volatile a22; /* ancestor_builder */
	T0* volatile a23; /* feature_flattener */
	T0* volatile a24; /* ecma_version */
	T1 volatile a25; /* metrics_shown */
	T1 volatile a26; /* flat_mode */
	T1 volatile a27; /* flat_dbc_mode */
	T0* volatile a28; /* processed_class_count_stack */
	T0* volatile a29; /* dotnet_assembly_consumer */
	T6 volatile a30; /* processed_class_count */
	T1 volatile a31; /* preparse_single_mode */
	T1 volatile a32; /* unknown_builtin_reported */
	T1 volatile a33; /* qualified_anchored_types_cycle_detection_enabled */
	T1 volatile a34; /* cluster_dependence_enabled */
	T6 volatile a35; /* postponed_class_count */
	T1 volatile a36; /* preparse_override_mode */
	T1 volatile a37; /* preparse_readonly_mode */
	T1 volatile a38; /* use_cluster_dependence_pathnames */
	T1 volatile a39; /* suppliers_enabled */
};

/* Struct for type DS_HASH_SET_CURSOR [STRING_8] */
struct S859 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [STRING_8] */
struct S860 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_FEATURE_CHECKER */
struct S862 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* system_processor */
	T0* volatile a2; /* type_checker */
	T0* volatile a3; /* current_type */
	T0* volatile a4; /* current_class */
	T0* volatile a5; /* current_feature */
	T0* volatile a6; /* current_feature_impl */
	T0* volatile a7; /* current_class_impl */
	T0* volatile a8; /* enclosing_inline_agents */
	T0* volatile a9; /* unused_overloaded_procedures_list */
	T0* volatile a10; /* unused_overloaded_queries_list */
	T0* volatile a11; /* unused_overloaded_features_list */
	T0* volatile a12; /* unused_call_infos */
	T0* volatile a13; /* unused_contexts */
	T0* volatile a14; /* current_context */
	T0* volatile a15; /* current_target_type */
	T0* volatile a16; /* current_object_test_types */
	T0* volatile a17; /* current_object_test_scope */
	T0* volatile a18; /* object_test_scope_builder */
	T0* volatile a19; /* current_iteration_cursor_types */
	T0* volatile a20; /* current_iteration_item_types */
	T0* volatile a21; /* current_iteration_item_scope */
	T0* volatile a22; /* current_inline_separate_argument_types */
	T0* volatile a23; /* current_inline_separate_argument_scope */
	T0* volatile a24; /* current_initialization_scope */
	T0* volatile a25; /* current_attachment_scope */
	T0* volatile a26; /* attachment_scope_builder */
	T0* volatile a27; /* unused_attachment_scopes */
	T0* volatile a28; /* assertions_by_feature */
	T0* volatile a29; /* common_ancestor_type_list */
	T0* volatile a30; /* note_term_list */
	T0* volatile a31; /* vape_non_descendant_clients */
	T0* volatile a32; /* vape_creation_clients */
	T0* volatile a33; /* vape_client */
	T0* volatile a34; /* unused_adapted_base_classes */
	T0* volatile a35; /* adapted_base_class_checker */
	T1 volatile a36; /* has_fatal_error */
	T1 volatile a37; /* in_static_feature */
	T1 volatile a38; /* in_precondition */
	T6 volatile a39; /* index_count */
	T6 volatile a40; /* result_index */
	T6 volatile a41; /* attached_result_index */
	T6 volatile a42; /* current_index */
	T6 volatile a43; /* boolean_index */
	T6 volatile a44; /* character_8_index */
	T6 volatile a45; /* character_32_index */
	T6 volatile a46; /* integer_8_index */
	T6 volatile a47; /* integer_16_index */
	T6 volatile a48; /* integer_32_index */
	T6 volatile a49; /* integer_64_index */
	T6 volatile a50; /* natural_8_index */
	T6 volatile a51; /* natural_16_index */
	T6 volatile a52; /* natural_32_index */
	T6 volatile a53; /* natural_64_index */
	T6 volatile a54; /* pointer_index */
	T6 volatile a55; /* real_32_index */
	T6 volatile a56; /* real_64_index */
	T6 volatile a57; /* string_8_index */
	T6 volatile a58; /* string_32_index */
	T6 volatile a59; /* immutable_string_8_index */
	T6 volatile a60; /* immutable_string_32_index */
	T6 volatile a61; /* void_index */
	T0* volatile a62; /* supplier_handler */
	T1 volatile a63; /* in_invariant */
	T0* volatile a64; /* precursor_queries */
	T0* volatile a65; /* precursor_procedures */
	T1 volatile a66; /* in_postcondition */
	T1 volatile a67; /* in_precursor */
	T0* volatile a68; /* current_inline_agent */
	T1 volatile a69; /* in_check_instruction */
	T1 volatile a70; /* in_rescue */
};

/* Struct for type ET_DYNAMIC_TYPE_SET_LIST */
struct S863 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type DS_HASH_SET [ET_FEATURE] */
struct S864 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* count */
	T6 volatile a9; /* slots_position */
	T0* volatile a10; /* special_item_routines */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type ET_DYNAMIC_PRECURSOR */
struct S865 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target_type */
	T0* volatile a2; /* static_feature */
	T0* volatile a3; /* result_type_set */
	T1 volatile a4; /* is_static */
	T0* volatile a5; /* current_feature */
	T6 volatile a6; /* id */
	T0* volatile a7; /* dynamic_type_sets */
	T6 volatile a8; /* current_index */
	T0* volatile a9; /* preconditions */
	T0* volatile a10; /* postconditions */
	T1 volatile a11; /* is_generated */
	T1 volatile a12; /* is_inlined */
	T8 volatile a13; /* builtin_class_code */
	T8 volatile a14; /* builtin_feature_code */
	T1 volatile a15; /* is_built */
	T0* volatile a16; /* invariants */
	T0* volatile a17; /* first_precursor */
	T0* volatile a18; /* other_precursors */
	T1 volatile a19; /* is_copy_routine */
	T1 volatile a20; /* is_is_equal_routine */
	T0* volatile a21; /* parent_type */
	T0* volatile a22; /* procedure_calls */
	T0* volatile a23; /* query_calls */
	T1 volatile a24; /* is_regular */
	T1 volatile a25; /* is_creation */
	T1 volatile a26; /* is_address */
	T1 volatile a27; /* is_separate_creation */
};

/* Struct for type ET_DYNAMIC_PRECURSOR_LIST */
struct S866 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
struct S867 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* position */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* count */
	T6 volatile a8; /* modulus */
	T6 volatile a9; /* slots_position */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* key_equality_tester */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T0* volatile a14; /* special_item_routines */
	T0* volatile a15; /* special_key_routines */
	T0* volatile a16; /* key_storage */
	T0* volatile a17; /* equality_tester */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_TYPE_CHECKER */
struct S869 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_context */
	T0* volatile a2; /* current_class_impl */
	T0* volatile a3; /* current_feature_impl */
	T0* volatile a4; /* constraint_context */
	T0* volatile a5; /* current_class */
	T0* volatile a6; /* target_context */
	T0* volatile a7; /* other_context */
	T0* volatile a8; /* adapted_base_class_checker */
	T0* volatile a9; /* adapted_base_classes */
	T0* volatile a10; /* system_processor */
	T0* volatile a11; /* supplier_handler */
	T1 volatile a12; /* has_fatal_error */
	T1 volatile a13; /* class_interface_error_only */
	T1 volatile a14; /* in_qualified_anchored_type */
};

/* Struct for type DS_ARRAYED_LIST [ET_INLINE_AGENT] */
struct S872 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_PROCEDURE]] */
struct S873 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_QUERY]] */
struct S874 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_FEATURE]] */
struct S875 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
struct S876 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [ET_NESTED_TYPE_CONTEXT] */
struct S877 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
struct S878 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* count */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* position */
	T0* volatile a7; /* equality_tester */
	T0* volatile a8; /* key_equality_tester */
	T6 volatile a9; /* found_position */
	T0* volatile a10; /* key_storage */
	T0* volatile a11; /* clashes */
	T6 volatile a12; /* modulus */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* capacity */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_OBJECT_TEST_SCOPE */
struct S879 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* object_tests */
	T6 volatile a2; /* hidden_count */
};

/* Struct for type ET_OBJECT_TEST_SCOPE_BUILDER */
struct S880 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
	T0* volatile a4; /* scope */
	T1 volatile a5; /* is_negated */
};

/* Struct for type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
struct S881 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_storage */
	T6 volatile a4; /* count */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* equality_tester */
	T0* volatile a9; /* key_equality_tester */
	T6 volatile a10; /* found_position */
	T0* volatile a11; /* clashes */
	T6 volatile a12; /* modulus */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* capacity */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* internal_keys */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_ITERATION_ITEM_SCOPE */
struct S882 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* hidden_count */
	T0* volatile a2; /* iteration_components */
};

/* Struct for type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
struct S883 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T0* volatile a2; /* key_equality_tester */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* special_key_routines */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* internal_cursor */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* count */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* internal_keys */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_INLINE_SEPARATE_ARGUMENT_SCOPE */
struct S884 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* hidden_count */
	T0* volatile a2; /* inline_separate_arguments */
};

/* Struct for type ET_ATTACHMENT_SCOPE */
struct S885 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* locals_attached */
	T0* volatile a2; /* arguments_attached */
	T0* volatile a3; /* attributes_attached */
	T1 volatile a4; /* result_attached */
	T1 volatile a5; /* is_code_unreachable */
};

/* Struct for type ET_ATTACHMENT_SCOPE_BUILDER */
struct S886 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* scope */
	T1 volatile a2; /* is_negated */
};

/* Struct for type DS_ARRAYED_LIST [ET_ATTACHMENT_SCOPE] */
struct S887 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE [ET_ASSERTIONS, ET_FEATURE] */
struct S888 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T0* volatile a3; /* clashes */
	T0* volatile a4; /* item_storage */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* position */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* free_slot */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* key_equality_tester */
	T6 volatile a12; /* found_position */
	T6 volatile a13; /* modulus */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST [ET_NOTE_TERM] */
struct S889 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
struct S890 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type ET_ADAPTED_BASE_CLASS_CHECKER */
struct S891 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_class_impl */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
	T1 volatile a4; /* has_fatal_error */
	T1 volatile a5; /* feature_flattening_error_only */
	T1 volatile a6; /* class_interface_error_only */
};

/* Struct for type ET_DYNAMIC_QUALIFIED_QUERY_CALL */
struct S893 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_feature */
	T0* volatile a2; /* static_call */
	T0* volatile a3; /* target_type_set */
	T0* volatile a4; /* result_type_set */
	T0* volatile a5; /* current_feature_impl */
	T0* volatile a6; /* current_type */
	T6 volatile a7; /* current_index */
	T6 volatile a8; /* index_offset */
	T0* volatile a9; /* next */
	T1 volatile a10; /* is_generated */
	T1 volatile a11; /* force_result_boxing */
};

/* Struct for type ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
struct S894 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target_type_set */
	T0* volatile a2; /* static_call */
	T0* volatile a3; /* current_type */
	T0* volatile a4; /* current_feature_impl */
	T0* volatile a5; /* next */
	T1 volatile a6; /* is_generated */
	T0* volatile a7; /* current_feature */
	T6 volatile a8; /* current_index */
	T6 volatile a9; /* index_offset */
};

/* Struct for type DS_ARRAYED_LIST_2 [ET_PRECONDITIONS, INTEGER_32] */
struct S896 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* special_routines_1 */
	T0* volatile a3; /* storage_1 */
	T0* volatile a4; /* special_routines_2 */
	T0* volatile a5; /* storage_2 */
	T6 volatile a6; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST_2 [ET_POSTCONDITIONS, INTEGER_32] */
struct S897 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* special_routines_1 */
	T0* volatile a3; /* storage_1 */
	T0* volatile a4; /* special_routines_2 */
	T0* volatile a5; /* storage_2 */
	T6 volatile a6; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST_2 [ET_INVARIANTS, INTEGER_32] */
struct S898 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines_1 */
	T0* volatile a2; /* storage_1 */
	T0* volatile a3; /* special_routines_2 */
	T0* volatile a4; /* storage_2 */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* count */
};

/* Struct for type ET_DYNAMIC_TUPLE_TYPE */
struct S904 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* id */
	T0* volatile a2; /* base_class */
	T0* volatile a3; /* base_type */
	T1 volatile a4; /* is_alive */
	T1 volatile a5; /* has_once_per_object_routines */
	T6 volatile a6; /* attribute_count */
	T0* volatile a7; /* queries */
	T0* volatile a8; /* meta_type */
	T0* volatile a9; /* separate_type */
	T0* volatile a10; /* attached_separate_type */
	T0* volatile a11; /* attached_type */
	T0* volatile a12; /* conforming_ancestors */
	T0* volatile a13; /* queries_by_seed */
	T0* volatile a14; /* procedures_by_seed */
	T0* volatile a15; /* procedures */
	T1 volatile a16; /* has_redefined_copy_routine */
	T0* volatile a17; /* item_type_sets */
	T0* volatile a18; /* invariants */
	T1 volatile a19; /* has_redefined_is_equal_routine */
	T1 volatile a20; /* has_reference_attributes */
	T6 volatile a21; /* hash_code */
	T0* volatile a22; /* next_type */
	T0* volatile a23; /* query_calls */
	T0* volatile a24; /* procedure_calls */
	T1 volatile a25; /* has_static */
	T0* volatile a26; /* equality_expressions */
	T0* volatile a27; /* object_equality_expressions */
};

/* Struct for type ET_FEATURE_LIST */
struct S907 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T6 volatile a2; /* count */
};

/* Struct for type NATIVE_STRING */
struct S909 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* managed_data */
	T6 volatile a2; /* unit_count */
};

/* Struct for type HASH_TABLE [NATIVE_STRING, IMMUTABLE_STRING_32] */
struct S910 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T0* volatile a2; /* content */
	T0* volatile a3; /* keys */
	T0* volatile a4; /* deleted_marks */
	T0* volatile a5; /* indexes_map */
	T6 volatile a6; /* iteration_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* deleted_item_position */
	T6 volatile a9; /* control */
	T0* volatile a10; /* found_item */
	T1 volatile a11; /* has_default */
	T6 volatile a12; /* item_position */
	T6 volatile a13; /* ht_lowest_deleted_position */
	T0* volatile a14; /* ht_deleted_item */
	T0* volatile a15; /* ht_deleted_key */
	T1 volatile a16; /* object_comparison */
};

/* Struct for type RX_BYTE_CODE */
struct S912 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* byte_code */
	T0* volatile a3; /* character_sets */
	T6 volatile a4; /* capacity */
};

/* Struct for type RX_CHARACTER_SET */
struct S913 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* is_empty */
	T1 volatile a2; /* is_negated */
	T11 volatile a3; /* first_set */
	T11 volatile a4; /* second_set */
	T11 volatile a5; /* third_set */
	T11 volatile a6; /* fourth_set */
	T0* volatile a7; /* other_sets */
	T1 volatile a8; /* is_reverted */
};

/* Struct for type RX_CASE_MAPPING */
struct S914 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* lower_table */
	T0* volatile a2; /* flip_table */
};

/* Struct for type UC_UNICODE_ROUTINES */
struct S916 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ARRAY [RX_CHARACTER_SET] */
struct S917 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [RX_CHARACTER_SET] */
struct S918 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type UC_STRING */
struct S919 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* area */
	T6 volatile a3; /* byte_count */
	T6 volatile a4; /* internal_hash_code */
	T6 volatile a5; /* last_byte_index_input */
	T6 volatile a6; /* last_byte_index_result */
	T6 volatile a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type DS_LINKED_LIST_CURSOR [STRING_8] */
struct S920 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* next_cursor */
	T0* volatile a4; /* container */
	T1 volatile a5; /* before */
};

/* Struct for type DS_HASH_SET [NATURAL_32] */
struct S921 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* capacity */
	T6 volatile a3; /* modulus */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* clashes_previous_position */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* item_storage */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T0* volatile a14; /* internal_cursor */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type KL_STRING_OUTPUT_STREAM */
struct S922 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* string */
};

/* Struct for type DS_ARRAYED_LIST [ET_OLD_EXPRESSION] */
struct S923 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_HASH_TABLE [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
struct S924 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T0* volatile a2; /* key_equality_tester */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* special_key_routines */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* internal_cursor */
	T6 volatile a16; /* count */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST [ET_IDENTIFIER] */
struct S925 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S926 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [BOOLEAN] */
struct S927 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T0* volatile a2; /* special_routines */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_EQUALITY_EXPRESSION] */
struct S928 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_EQUALITY_TYPES] */
struct S929 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
struct S930 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S931 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [ET_QUALIFIED_CALL_EXPRESSION] */
struct S932 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ACTUAL_ARGUMENT_LIST] */
struct S933 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_ARRAYED_LIST [ET_RESULT] */
struct S934 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
struct S935 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* clashes */
	T0* volatile a4; /* slots */
};

/* Struct for type ET_DYNAMIC_STANDALONE_TYPE_SET */
struct S936 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* static_type */
	T6 volatile a2; /* count */
	T0* volatile a3; /* dynamic_types */
	T1 volatile a4; /* is_never_void */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S937 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_STACK [ET_EXPRESSION] */
struct S938 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type DS_ARRAYED_LIST [ET_EXPRESSION] */
struct S939 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
struct S940 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* last_position */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* count */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* modulus */
	T6 volatile a9; /* clashes_previous_position */
	T0* volatile a10; /* key_equality_tester */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
struct S941 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type DS_HASH_SET [ET_DYNAMIC_PRIMARY_TYPE] */
struct S942 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* count */
	T6 volatile a3; /* position */
	T0* volatile a4; /* item_storage */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* modulus */
	T6 volatile a7; /* last_position */
	T6 volatile a8; /* free_slot */
	T6 volatile a9; /* slots_position */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
struct S943 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T6 volatile a3; /* found_position */
	T0* volatile a4; /* item_storage */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* capacity */
	T0* volatile a9; /* equality_tester */
	T0* volatile a10; /* key_equality_tester */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T6 volatile a13; /* modulus */
	T0* volatile a14; /* slots */
	T6 volatile a15; /* slots_position */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST_2 [ET_OBJECT_TEST, INTEGER_32] */
struct S944 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage_2 */
	T0* volatile a3; /* storage_1 */
	T0* volatile a4; /* special_routines_1 */
	T0* volatile a5; /* special_routines_2 */
	T6 volatile a6; /* capacity */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, INTEGER_32] */
struct S945 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* modulus */
	T0* volatile a2; /* key_equality_tester */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* slots */
	T0* volatile a5; /* hash_function */
	T0* volatile a6; /* key_storage */
	T0* volatile a7; /* clashes */
	T0* volatile a8; /* equality_tester */
	T6 volatile a9; /* capacity */
	T6 volatile a10; /* last_position */
	T6 volatile a11; /* free_slot */
	T6 volatile a12; /* position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* special_key_routines */
	T6 volatile a15; /* found_position */
	T0* volatile a16; /* internal_cursor */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* count */
	T6 volatile a19; /* clashes_previous_position */
};

/* Struct for type DS_HASH_SET [INTEGER_32] */
struct S946 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* internal_cursor */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* modulus */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* count */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST_2 [ET_AGENT, INTEGER_32] */
struct S947 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage_2 */
	T0* volatile a3; /* storage_1 */
	T0* volatile a4; /* special_routines_1 */
	T0* volatile a5; /* special_routines_2 */
	T6 volatile a6; /* capacity */
};

/* Struct for type ET_GENERAL_QUALIFIED_FEATURE_CALL_INSTRUCTION */
struct S948 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* name */
	T0* volatile a3; /* arguments */
};

/* Struct for type ET_GENERAL_QUALIFIED_FEATURE_CALL_EXPRESSION */
struct S950 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* name */
	T0* volatile a3; /* arguments */
	T6 volatile a4; /* index */
};

/* Struct for type DS_HASH_SET [ET_DYNAMIC_TUPLE_TYPE] */
struct S951 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* clashes */
	T0* volatile a11; /* slots */
	T6 volatile a12; /* found_position */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, ET_STANDALONE_CLOSURE] */
struct S952 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* count */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* modulus */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* key_equality_tester */
	T0* volatile a12; /* equality_tester */
	T0* volatile a13; /* internal_cursor */
	T0* volatile a14; /* key_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ARRAY [INTEGER_32] */
struct S953 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type DS_HASH_TABLE [ET_CONSTANT, ET_FEATURE] */
struct S954 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* key_storage */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* count */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* capacity */
	T6 volatile a9; /* slots_position */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* key_equality_tester */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* clashes */
	T6 volatile a14; /* modulus */
	T0* volatile a15; /* slots */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
struct S955 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* key_storage */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* count */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* equality_tester */
	T0* volatile a9; /* key_equality_tester */
	T6 volatile a10; /* found_position */
	T0* volatile a11; /* clashes */
	T6 volatile a12; /* modulus */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* capacity */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
struct S956 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* count */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* position */
	T6 volatile a7; /* capacity */
	T0* volatile a8; /* equality_tester */
	T0* volatile a9; /* key_equality_tester */
	T0* volatile a10; /* internal_cursor */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T6 volatile a13; /* modulus */
	T0* volatile a14; /* slots */
	T6 volatile a15; /* slots_position */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_INLINED_EXPRESSION] */
struct S957 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_CALL_CONTEXT] */
struct S958 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_DYNAMIC_FEATURE] */
struct S959 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_SET [ET_DYNAMIC_FEATURE] */
struct S960 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* count */
	T6 volatile a9; /* slots_position */
	T0* volatile a10; /* special_item_routines */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [BOOLEAN, STRING_8] */
struct S961 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* key_storage */
	T0* volatile a5; /* key_equality_tester */
	T0* volatile a6; /* internal_keys */
	T6 volatile a7; /* count */
	T6 volatile a8; /* last_position */
	T6 volatile a9; /* free_slot */
	T6 volatile a10; /* capacity */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* equality_tester */
	T6 volatile a15; /* found_position */
	T0* volatile a16; /* clashes */
	T0* volatile a17; /* slots */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_STRING_HASH_TABLE */
struct S962 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T6 volatile a2; /* position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* modulus */
	T0* volatile a5; /* key_equality_tester */
	T0* volatile a6; /* slots */
	T0* volatile a7; /* hash_function */
	T0* volatile a8; /* key_storage */
	T0* volatile a9; /* clashes */
	T6 volatile a10; /* count */
	T6 volatile a11; /* capacity */
	T6 volatile a12; /* slots_position */
	T6 volatile a13; /* free_slot */
	T6 volatile a14; /* last_position */
	T0* volatile a15; /* internal_keys */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* equality_tester */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* internal_cursor */
};

/* Struct for type KL_TEXT_OUTPUT_FILE */
struct S963 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* mode */
	T14 volatile a3; /* file_pointer */
	T1 volatile a4; /* descriptor_available */
	T0* volatile a5; /* last_string_32 */
	T0* volatile a6; /* last_string */
	T0* volatile a7; /* internal_name */
	T0* volatile a8; /* internal_detachable_name_pointer */
};

/* Struct for type DS_HASH_TABLE_CURSOR [STRING_8, STRING_8] */
struct S964 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type UT_UNDEFINED_ENVIRONMENT_VARIABLE_ERROR */
struct S965 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type UT_CANNOT_WRITE_TO_FILE_ERROR */
struct S966 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [STRING_8]] */
struct S967 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_QUICK_SORTER [STRING_8] */
struct S968 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type UC_STRING_COMPARATOR */
struct S969 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
struct S970 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* modulus */
	T6 volatile a9; /* clashes_previous_position */
	T0* volatile a10; /* equality_tester */
	T6 volatile a11; /* found_position */
	T0* volatile a12; /* special_item_routines */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_key_routines */
	T0* volatile a16; /* key_storage */
	T6 volatile a17; /* free_slot */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type TUPLE [STRING_8, STRING_8] */
struct S973 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type DS_HASH_TOPOLOGICAL_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
struct S974 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sorted_items */
	T0* volatile a2; /* cycle */
	T0* volatile a3; /* indexes */
	T0* volatile a4; /* items */
	T0* volatile a5; /* counts */
	T0* volatile a6; /* successors */
};

/* Struct for type ET_DYNAMIC_EQUALITY_TYPES */
struct S975 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left */
	T0* volatile a2; /* right */
};

/* Struct for type ET_DYNAMIC_OBJECT_EQUALITY_TYPES */
struct S977 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left */
	T0* volatile a2; /* right */
};

/* Struct for type ET_DYNAMIC_SPECIAL_TYPE */
struct S978 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* id */
	T0* volatile a2; /* base_class */
	T0* volatile a3; /* base_type */
	T1 volatile a4; /* is_alive */
	T1 volatile a5; /* has_once_per_object_routines */
	T6 volatile a6; /* attribute_count */
	T0* volatile a7; /* queries */
	T0* volatile a8; /* meta_type */
	T0* volatile a9; /* separate_type */
	T0* volatile a10; /* attached_separate_type */
	T0* volatile a11; /* attached_type */
	T0* volatile a12; /* conforming_ancestors */
	T0* volatile a13; /* queries_by_seed */
	T0* volatile a14; /* procedures_by_seed */
	T0* volatile a15; /* procedures */
	T0* volatile a16; /* item_type_set */
	T1 volatile a17; /* has_redefined_copy_routine */
	T0* volatile a18; /* invariants */
	T1 volatile a19; /* has_redefined_is_equal_routine */
	T1 volatile a20; /* has_reference_attributes */
	T6 volatile a21; /* hash_code */
	T0* volatile a22; /* next_type */
	T0* volatile a23; /* query_calls */
	T0* volatile a24; /* procedure_calls */
	T1 volatile a25; /* has_static */
	T0* volatile a26; /* equality_expressions */
	T0* volatile a27; /* object_equality_expressions */
};

/* Struct for type DS_QUICK_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
struct S980 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type ET_DYNAMIC_SECONDARY_TYPE */
struct S981 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* primary_type */
	T0* volatile a2; /* meta_type */
	T0* volatile a3; /* type_mark */
	T0* volatile a4; /* internal_base_type */
};

/* Struct for type ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
struct S982 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
struct S988 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type ET_DYNAMIC_INLINED_EXPRESSION */
struct S992 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* dynamic_type_set */
	T6 volatile a2; /* index */
	T0* volatile a3; /* expression */
	T0* volatile a4; /* context */
};

/* Struct for type ET_DYNAMIC_CALL_CONTEXT */
struct S993 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_feature */
	T0* volatile a2; /* arguments */
	T0* volatile a3; /* target */
};

/* Struct for type DS_QUICK_SORTER [INTEGER_32] */
struct S994 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type PROCEDURE [TUPLE] */
struct S996 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S997 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S998 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S999 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S1000 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T6 volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type UT_INTEGER_FORMATTER */
struct S1001 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S1006 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T6 volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE] */
struct S1007 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S1011 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T1 volatile z4;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
struct S1016 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
struct S1017 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
	T6 volatile z4;
};

/* Struct for type SPECIAL [ET_DYNAMIC_PRIMARY_TYPE] */
struct S1018 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRIMARY_TYPE] */
struct S1019 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
struct S1020 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type TUPLE [GECC, DS_ARRAYED_LIST [STRING_8], INTEGER_32, MUTEX] */
struct S1023 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
	T0* volatile z4;
};

/* Struct for type WORKER_THREAD */
struct S1024 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* thread_procedure */
	T0* volatile a2; /* launch_mutex */
	T14 volatile a3; /* thread_id */
	T1 volatile a4; /* terminated */
};

/* Struct for type PATH */
struct S1026 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_name */
	T0* volatile a2; /* storage */
	T1 volatile a3; /* is_normalized */
};

/* Struct for type STRING_TO_INTEGER_CONVERTOR */
struct S1027 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* sign */
	T11 volatile a2; /* part1 */
	T11 volatile a3; /* part2 */
	T6 volatile a4; /* last_state */
	T1 volatile a5; /* internal_overflowed */
	T0* volatile a6; /* leading_separators */
	T0* volatile a7; /* trailing_separators */
	T1 volatile a8; /* leading_separators_acceptable */
	T1 volatile a9; /* trailing_separators_acceptable */
	T6 volatile a10; /* conversion_type */
};

/* Struct for type STRING_8_SEARCHER */
struct S1029 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* deltas */
};

/* Struct for type STD_FILES */
struct S1031 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_output */
};

/* Struct for type HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
struct S1032 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* last_state */
	T1 volatile a2; /* internal_overflowed */
	T11 volatile a3; /* part1 */
	T11 volatile a4; /* part2 */
	T0* volatile a5; /* leading_separators */
	T0* volatile a6; /* trailing_separators */
	T6 volatile a7; /* sign */
	T6 volatile a8; /* conversion_type */
	T2 volatile a9; /* internal_lookahead */
	T1 volatile a10; /* leading_separators_acceptable */
	T1 volatile a11; /* trailing_separators_acceptable */
};

/* Struct for type DT_SHARED_SYSTEM_CLOCK */
struct S1034 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DT_SYSTEM_CLOCK */
struct S1035 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* second */
	T6 volatile a2; /* year */
	T6 volatile a3; /* month */
	T6 volatile a4; /* day */
	T6 volatile a5; /* hour */
	T6 volatile a6; /* minute */
	T6 volatile a7; /* millisecond */
	T0* volatile a8; /* local_clock */
};

/* Struct for type DT_DATE_TIME_DURATION */
struct S1036 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* year */
	T6 volatile a2; /* month */
	T6 volatile a3; /* day */
	T6 volatile a4; /* hour */
	T6 volatile a5; /* minute */
	T6 volatile a6; /* second */
	T6 volatile a7; /* millisecond */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, STRING_8] */
struct S1041 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* key_storage */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* position */
	T6 volatile a5; /* count */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* last_position */
	T0* volatile a9; /* key_equality_tester */
	T0* volatile a10; /* internal_keys */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_CONSTRAINT_GENERIC_NAMED_TYPE */
struct S1044 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type_mark */
	T0* volatile a2; /* name */
	T0* volatile a3; /* actual_parameters */
};

/* Struct for type ET_UNQUALIFIED_CALL_EXPRESSION */
struct S1045 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* name */
	T0* volatile a3; /* arguments */
	T0* volatile a4; /* parenthesis_call */
};

/* Struct for type ET_UNQUALIFIED_CALL_INSTRUCTION */
struct S1046 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* arguments */
	T0* volatile a3; /* parenthesis_call */
};

/* Struct for type KL_ARRAY_ROUTINES [INTEGER_32] */
struct S1047 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_CELL [detachable ET_ECF_CONFIG] */
struct S1050 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_SYSTEM, DS_CELL [detachable ET_ECF_CONFIG]] */
struct S1052 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_NOTE_ELEMENT] */
struct S1054 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type XM_NAMESPACE */
struct S1055 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* uri */
	T0* volatile a2; /* ns_prefix */
};

/* Struct for type DS_LINKED_LIST_CURSOR [XM_ELEMENT_NODE] */
struct S1056 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type ET_ECF_REDIRECTION_CONFIG */
struct S1058 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* uuid */
	T0* volatile a2; /* location */
	T0* volatile a3; /* filename */
	T0* volatile a4; /* universe */
	T0* volatile a5; /* message */
	T0* volatile a6; /* ecf_namespace */
	T0* volatile a7; /* ecf_version */
};

/* Struct for type ET_DYNAMIC_PROCEDURE_TYPE */
struct S1059 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* id */
	T0* volatile a2; /* base_class */
	T0* volatile a3; /* base_type */
	T1 volatile a4; /* is_alive */
	T1 volatile a5; /* has_once_per_object_routines */
	T6 volatile a6; /* attribute_count */
	T0* volatile a7; /* queries */
	T0* volatile a8; /* meta_type */
	T0* volatile a9; /* separate_type */
	T0* volatile a10; /* attached_separate_type */
	T0* volatile a11; /* attached_type */
	T0* volatile a12; /* conforming_ancestors */
	T1 volatile a13; /* has_redefined_is_equal_routine */
	T1 volatile a14; /* has_redefined_copy_routine */
	T0* volatile a15; /* open_operand_tuple_type_set */
	T0* volatile a16; /* set_rout_disp_final_feature */
	T0* volatile a17; /* invariants */
	T0* volatile a18; /* next_type */
	T0* volatile a19; /* procedures */
	T0* volatile a20; /* query_calls */
	T0* volatile a21; /* procedure_calls */
	T1 volatile a22; /* has_static */
	T0* volatile a23; /* open_operand_type_sets */
	T6 volatile a24; /* hash_code */
	T0* volatile a25; /* queries_by_seed */
	T0* volatile a26; /* procedures_by_seed */
	T1 volatile a27; /* has_reference_attributes */
	T0* volatile a28; /* equality_expressions */
	T0* volatile a29; /* object_equality_expressions */
};

/* Struct for type ET_DYNAMIC_FUNCTION_TYPE */
struct S1060 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* id */
	T0* volatile a2; /* base_class */
	T0* volatile a3; /* base_type */
	T1 volatile a4; /* is_alive */
	T1 volatile a5; /* has_once_per_object_routines */
	T6 volatile a6; /* attribute_count */
	T0* volatile a7; /* queries */
	T0* volatile a8; /* meta_type */
	T0* volatile a9; /* separate_type */
	T0* volatile a10; /* attached_separate_type */
	T0* volatile a11; /* attached_type */
	T0* volatile a12; /* conforming_ancestors */
	T1 volatile a13; /* has_redefined_is_equal_routine */
	T1 volatile a14; /* has_redefined_copy_routine */
	T0* volatile a15; /* open_operand_tuple_type_set */
	T0* volatile a16; /* set_rout_disp_final_feature */
	T0* volatile a17; /* invariants */
	T0* volatile a18; /* next_type */
	T0* volatile a19; /* procedures */
	T0* volatile a20; /* query_calls */
	T0* volatile a21; /* procedure_calls */
	T1 volatile a22; /* has_static */
	T0* volatile a23; /* open_operand_type_sets */
	T0* volatile a24; /* result_type_set */
	T6 volatile a25; /* hash_code */
	T0* volatile a26; /* queries_by_seed */
	T0* volatile a27; /* procedures_by_seed */
	T1 volatile a28; /* has_reference_attributes */
	T0* volatile a29; /* equality_expressions */
	T0* volatile a30; /* object_equality_expressions */
};

/* Struct for type ET_DYNAMIC_PUSH_TYPE_SET */
struct S1061 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* static_type */
	T6 volatile a2; /* count */
	T1 volatile a3; /* is_never_void */
	T0* volatile a4; /* dynamic_types */
	T0* volatile a5; /* targets */
	T1 volatile a6; /* is_dynamic_types_readonly */
};

/* Struct for type KL_PLATFORM */
struct S1062 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type UT_CONFIG_PARSER */
struct S1063 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_error */
	T0* volatile a2; /* config_values */
	T0* volatile a3; /* error_handler */
	T0* volatile a4; /* defined_values */
	T0* volatile a5; /* include_stack */
	T0* volatile a6; /* line_nb_stack */
	T0* volatile a7; /* last_string_value */
	T6 volatile a8; /* line_nb */
	T0* volatile a9; /* yyss */
	T0* volatile a10; /* input_buffer */
	T6 volatile a11; /* yy_end */
	T6 volatile a12; /* yy_position */
	T6 volatile a13; /* yy_line */
	T6 volatile a14; /* yy_column */
	T6 volatile a15; /* yy_parsing_status */
	T6 volatile a16; /* yy_suspended_yystacksize */
	T6 volatile a17; /* yy_suspended_yystate */
	T6 volatile a18; /* yy_suspended_yyn */
	T6 volatile a19; /* yy_suspended_yychar1 */
	T6 volatile a20; /* yy_suspended_index */
	T6 volatile a21; /* yy_suspended_yyss_top */
	T6 volatile a22; /* yy_suspended_yy_goto */
	T0* volatile a23; /* yyr1 */
	T6 volatile a24; /* yyssp */
	T0* volatile a25; /* yypgoto */
	T0* volatile a26; /* yycheck */
	T0* volatile a27; /* yytable */
	T0* volatile a28; /* yydefgoto */
	T6 volatile a29; /* error_count */
	T1 volatile a30; /* yy_lookahead_needed */
	T6 volatile a31; /* yyerrstatus */
	T0* volatile a32; /* yypact */
	T6 volatile a33; /* last_token */
	T0* volatile a34; /* yytranslate */
	T0* volatile a35; /* yydefact */
	T0* volatile a36; /* yy_pushed_start_conditions */
	T0* volatile a37; /* yyspecial_routines1 */
	T6 volatile a38; /* yyvsc1 */
	T0* volatile a39; /* yyvs1 */
	T0* volatile a40; /* yyspecial_routines2 */
	T6 volatile a41; /* yyvsc2 */
	T0* volatile a42; /* yyvs2 */
	T0* volatile a43; /* yyspecial_routines3 */
	T6 volatile a44; /* yyvsc3 */
	T0* volatile a45; /* yyvs3 */
	T0* volatile a46; /* yytypes1 */
	T0* volatile a47; /* yytypes2 */
	T6 volatile a48; /* yy_start */
	T6 volatile a49; /* yyvsp1 */
	T6 volatile a50; /* yyvsp2 */
	T6 volatile a51; /* yyvsp3 */
	T1 volatile a52; /* yy_more_flag */
	T6 volatile a53; /* yy_more_len */
	T6 volatile a54; /* line */
	T6 volatile a55; /* column */
	T6 volatile a56; /* position */
	T6 volatile a57; /* yy_start_state */
	T0* volatile a58; /* yy_state_stack */
	T6 volatile a59; /* yy_state_count */
	T0* volatile a60; /* yy_content_area */
	T0* volatile a61; /* yy_unicode_content_area */
	T0* volatile a62; /* yy_accept */
	T6 volatile a63; /* yy_last_accepting_state */
	T6 volatile a64; /* yy_last_accepting_cpos */
	T0* volatile a65; /* yy_content */
	T0* volatile a66; /* yy_ec */
	T0* volatile a67; /* yy_chk */
	T0* volatile a68; /* yy_base */
	T0* volatile a69; /* yy_def */
	T0* volatile a70; /* yy_meta */
	T0* volatile a71; /* yy_nxt */
	T6 volatile a72; /* yy_lp */
	T0* volatile a73; /* yy_acclist */
	T6 volatile a74; /* yy_looking_for_trail_begin */
	T6 volatile a75; /* yy_full_match */
	T6 volatile a76; /* yy_full_state */
	T6 volatile a77; /* yy_full_lp */
	T1 volatile a78; /* yy_rejected */
	T0* volatile a79; /* last_detachable_any_value */
	T6 volatile a80; /* if_level */
	T6 volatile a81; /* ignored_level */
};

/* Struct for type KL_COMPARABLE_COMPARATOR [INTEGER_32] */
struct S1064 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type CELL [INTEGER_32] */
struct S1067 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* item */
};

/* Struct for type KL_PATHNAME */
struct S1069 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* drive */
	T0* volatile a3; /* hostname */
	T0* volatile a4; /* sharename */
	T1 volatile a5; /* is_relative */
	T0* volatile a6; /* components */
};

/* Struct for type KL_ANY_ROUTINES */
struct S1070 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_WINDOWS_INPUT_FILE */
struct S1071 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_string */
	T6 volatile a2; /* mode */
	T0* volatile a3; /* name */
	T0* volatile a4; /* character_buffer */
	T1 volatile a5; /* end_of_file */
	T2 volatile a6; /* last_character */
	T14 volatile a7; /* file_pointer */
	T1 volatile a8; /* descriptor_available */
	T0* volatile a9; /* internal_name */
	T0* volatile a10; /* internal_detachable_name_pointer */
};

/* Struct for type KL_UNIX_INPUT_FILE */
struct S1072 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_string */
	T6 volatile a2; /* mode */
	T0* volatile a3; /* name */
	T0* volatile a4; /* character_buffer */
	T1 volatile a5; /* end_of_file */
	T2 volatile a6; /* last_character */
	T14 volatile a7; /* file_pointer */
	T1 volatile a8; /* descriptor_available */
	T0* volatile a9; /* internal_name */
	T0* volatile a10; /* internal_detachable_name_pointer */
};

/* Struct for type ARRAY [IMMUTABLE_STRING_32] */
struct S1073 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
	T1 volatile a4; /* object_comparison */
};

/* Struct for type SPECIAL [IMMUTABLE_STRING_32] */
struct S1074 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [NATURAL_8] */
struct S1075 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T8 volatile z2[]; /* item */
};

/* Struct for type UC_UTF8_STRING */
struct S1078 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* area */
	T6 volatile a3; /* byte_count */
	T6 volatile a4; /* internal_hash_code */
	T6 volatile a5; /* last_byte_index_input */
	T6 volatile a6; /* last_byte_index_result */
	T6 volatile a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [STRING_8] */
struct S1079 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST [ET_ADAPTED_LIBRARY] */
struct S1080 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1081 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_MASTER_CLASS] */
struct S1082 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_MASTER_CLASS] */
struct S1085 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLASS_NAME] */
struct S1086 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CLASS_NAME] */
struct S1087 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_MASTER_CLASS, ET_CLASS_NAME] */
struct S1088 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_PARENT_ITEM] */
struct S1090 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_PARENT_ITEM] */
struct S1091 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CLIENT_ITEM] */
struct S1092 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLIENT_ITEM] */
struct S1093 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ACTUAL_PARAMETER_ITEM] */
struct S1094 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ACTUAL_PARAMETER_ITEM] */
struct S1096 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_BRACED_TYPE_LIST */
struct S1097 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_brace */
	T0* volatile a2; /* right_brace */
	T6 volatile a3; /* count */
	T0* volatile a4; /* storage */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_UNIVERSE] */
struct S1098 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_UNIVERSE] */
struct S1100 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_UNIVERSE] */
struct S1101 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST [ET_MASTER_CLASS] */
struct S1102 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* equality_tester */
	T0* volatile a6; /* internal_cursor */
};

/* Struct for type TUPLE [ET_MASTER_CLASS] */
struct S1103 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_MASTER_CLASS]] */
struct S1104 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [AP_OPTION] */
struct S1105 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [AP_OPTION] */
struct S1107 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [AP_OPTION] */
struct S1108 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [AP_ALTERNATIVE_OPTIONS_LIST] */
struct S1109 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [AP_ALTERNATIVE_OPTIONS_LIST] */
struct S1110 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [AP_ALTERNATIVE_OPTIONS_LIST] */
struct S1111 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable STRING_8] */
struct S1112 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable STRING_8] */
struct S1113 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_LINKABLE [STRING_8] */
struct S1114 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_LINKABLE [BOOLEAN] */
struct S1115 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_LINKED_LIST_CURSOR [BOOLEAN] */
struct S1116 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
};

/* Struct for type DS_LINKABLE [INTEGER_32] */
struct S1117 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_LINKED_LIST_CURSOR [INTEGER_32] */
struct S1118 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
};

/* Struct for type YY_FILE_BUFFER */
struct S1120 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T0* volatile a2; /* content */
	T1 volatile a3; /* end_of_file */
	T0* volatile a4; /* file */
	T6 volatile a5; /* count */
	T6 volatile a6; /* index */
	T6 volatile a7; /* line */
	T6 volatile a8; /* column */
	T6 volatile a9; /* position */
	T1 volatile a10; /* beginning_of_line */
	T1 volatile a11; /* filled */
	T1 volatile a12; /* interactive */
};

/* Struct for type DS_HASH_TOPOLOGICAL_SORTER [ET_CLASS] */
struct S1121 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* sorted_items */
	T0* volatile a2; /* cycle */
	T0* volatile a3; /* indexes */
	T0* volatile a4; /* counts */
	T0* volatile a5; /* successors */
	T0* volatile a6; /* items */
};

/* Struct for type DS_HASH_TABLE [ET_BASE_TYPE, ET_CLASS] */
struct S1122 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T6 volatile a3; /* found_position */
	T0* volatile a4; /* item_storage */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* capacity */
	T0* volatile a9; /* equality_tester */
	T0* volatile a10; /* key_equality_tester */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T6 volatile a13; /* modulus */
	T0* volatile a14; /* slots */
	T6 volatile a15; /* slots_position */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_PARENT_CHECKER1 */
struct S1123 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
	T0* volatile a4; /* current_parent */
};

/* Struct for type ET_FORMAL_PARAMETER_CHECKER1 */
struct S1124 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* all_base_types */
	T0* volatile a3; /* base_types */
	T0* volatile a4; /* formal_dependencies */
	T0* volatile a5; /* base_type_dependencies */
	T0* volatile a6; /* recursive_formal_constraints */
	T0* volatile a7; /* recursive_formal_constraints_to_be_processed */
	T0* volatile a8; /* current_class */
	T0* volatile a9; /* system_processor */
	T0* volatile a10; /* current_formal */
	T0* volatile a11; /* current_type_constraint */
};

/* Struct for type DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
struct S1125 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T6 volatile a3; /* position */
	T6 volatile a4; /* found_position */
	T0* volatile a5; /* item_storage */
	T0* volatile a6; /* key_equality_tester */
	T0* volatile a7; /* internal_keys */
	T6 volatile a8; /* last_position */
	T6 volatile a9; /* free_slot */
	T6 volatile a10; /* modulus */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* equality_tester */
	T0* volatile a14; /* key_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T6 volatile a17; /* capacity */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_FEATURE_NAME_TESTER */
struct S1126 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
struct S1128 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_storage */
	T0* volatile a4; /* key_equality_tester */
	T0* volatile a5; /* internal_keys */
	T6 volatile a6; /* position */
	T6 volatile a7; /* last_position */
	T6 volatile a8; /* capacity */
	T6 volatile a9; /* count */
	T6 volatile a10; /* free_slot */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_ALIAS_NAME_TESTER */
struct S1129 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [ET_CLIENT_LIST] */
struct S1131 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_HASH_TABLE [ET_CLIENT, ET_CLASS] */
struct S1132 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T0* volatile a2; /* key_equality_tester */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* special_key_routines */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* internal_cursor */
	T6 volatile a16; /* count */
	T0* volatile a17; /* internal_keys */
	T6 volatile a18; /* slots_position */
	T6 volatile a19; /* clashes_previous_position */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_FEATURE_ADAPTATION_RESOLVER */
struct S1133 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* rename_table */
	T0* volatile a3; /* export_table */
	T0* volatile a4; /* undefine_table */
	T0* volatile a5; /* redefine_table */
	T0* volatile a6; /* select_table */
	T0* volatile a7; /* replicable_features */
	T0* volatile a8; /* current_class */
	T0* volatile a9; /* system_processor */
	T0* volatile a10; /* free_parent_feature */
	T0* volatile a11; /* parent_feature_list */
	T0* volatile a12; /* free_inherited_feature */
	T0* volatile a13; /* inherited_feature_list */
	T0* volatile a14; /* free_redeclared_feature */
	T0* volatile a15; /* redeclared_feature_list */
};

/* Struct for type ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
struct S1134 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* dotnet_features */
	T0* volatile a3; /* other_dotnet_features */
	T0* volatile a4; /* current_class */
	T0* volatile a5; /* system_processor */
	T0* volatile a6; /* free_parent_feature */
	T0* volatile a7; /* parent_feature_list */
	T0* volatile a8; /* free_inherited_feature */
	T0* volatile a9; /* inherited_feature_list */
	T0* volatile a10; /* free_redeclared_feature */
	T0* volatile a11; /* redeclared_feature_list */
};

/* Struct for type ET_IDENTIFIER_TYPE_RESOLVER */
struct S1135 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
	T0* volatile a4; /* current_feature */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
struct S1136 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* constraint_context */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type ET_ANCHORED_TYPE_CHECKER */
struct S1137 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* anchored_type_sorter */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
	T0* volatile a5; /* current_anchored_type */
};

/* Struct for type ET_SIGNATURE_CHECKER */
struct S1138 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* parent_context */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
	T6 volatile a5; /* processing_mode */
};

/* Struct for type ET_PARENT_CHECKER2 */
struct S1139 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* constraint_context */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
};

/* Struct for type ET_FORMAL_PARAMETER_CHECKER2 */
struct S1140 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* constraint_context */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
};

/* Struct for type ET_BUILTIN_FEATURE_CHECKER */
struct S1141 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* builtin_features */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
};

/* Struct for type ET_PRECURSOR_CHECKER */
struct S1142 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_feature */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
};

/* Struct for type DS_HASH_TABLE [ET_FEATURE, INTEGER_32] */
struct S1143 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* internal_cursor */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* position */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* slots_position */
	T6 volatile a8; /* free_slot */
	T0* volatile a9; /* equality_tester */
	T0* volatile a10; /* key_equality_tester */
	T6 volatile a11; /* found_position */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* special_item_routines */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T0* volatile a19; /* internal_keys */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_HASH_SET [ET_CLASS] */
struct S1144 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* last_position */
	T0* volatile a3; /* clashes */
	T0* volatile a4; /* item_storage */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* modulus */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* slots */
	T6 volatile a11; /* found_position */
	T0* volatile a12; /* internal_cursor */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
struct S1145 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class_impl */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* adapted_base_class_checker */
	T0* volatile a5; /* adapted_base_classes */
	T0* volatile a6; /* target_context */
	T0* volatile a7; /* other_context */
	T0* volatile a8; /* classes_to_be_processed */
	T0* volatile a9; /* system_processor */
	T1 volatile a10; /* in_qualified_anchored_type */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
struct S1146 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* constraint_context */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type DS_HASH_TABLE [ET_RENAME, ET_FEATURE_NAME] */
struct S1147 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* found_position */
	T6 volatile a3; /* position */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
	T0* volatile a6; /* item_storage */
	T0* volatile a7; /* key_equality_tester */
	T0* volatile a8; /* internal_keys */
	T6 volatile a9; /* last_position */
	T6 volatile a10; /* free_slot */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* slots_position */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* equality_tester */
	T0* volatile a15; /* key_storage */
	T0* volatile a16; /* clashes */
	T0* volatile a17; /* slots */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [ET_RENAME, ET_ALIAS_NAME] */
struct S1148 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* position */
	T6 volatile a3; /* found_position */
	T0* volatile a4; /* item_storage */
	T0* volatile a5; /* key_storage */
	T0* volatile a6; /* key_equality_tester */
	T0* volatile a7; /* internal_keys */
	T6 volatile a8; /* count */
	T6 volatile a9; /* last_position */
	T6 volatile a10; /* free_slot */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* slots_position */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* equality_tester */
	T0* volatile a15; /* internal_cursor */
	T0* volatile a16; /* clashes */
	T0* volatile a17; /* slots */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type ET_PARENT_CHECKER3 */
struct S1149 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type DS_HASH_SET [ET_PROCEDURE] */
struct S1150 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* clashes */
	T0* volatile a11; /* slots */
	T6 volatile a12; /* found_position */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type DS_HASH_SET [ET_QUERY] */
struct S1151 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T6 volatile a8; /* count */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* clashes */
	T0* volatile a11; /* slots */
	T6 volatile a12; /* found_position */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type ET_SUPPLIER_BUILDER */
struct S1152 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* supplier_classes */
	T0* volatile a2; /* current_class */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [INTEGER_32] */
struct S1153 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type YY_UNICODE_BUFFER */
struct S1155 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* content */
	T6 volatile a2; /* index */
	T6 volatile a3; /* line */
	T6 volatile a4; /* column */
	T6 volatile a5; /* position */
	T6 volatile a6; /* count */
	T1 volatile a7; /* filled */
	T1 volatile a8; /* beginning_of_line */
	T6 volatile a9; /* capacity */
};

/* Struct for type SPECIAL [detachable ET_OBJECT_TEST_LIST] */
struct S1156 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
struct S1157 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_OBJECT_TEST_LIST] */
struct S1158 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_OBJECT_TEST_LIST] */
struct S1159 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
struct S1160 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
struct S1161 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ITERATION_COMPONENT_LIST] */
struct S1162 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT_LIST] */
struct S1163 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S1164 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S1165 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S1166 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
struct S1167 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ASSERTION_ITEM] */
struct S1168 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ASSERTION_ITEM] */
struct S1169 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ASSERTION_ITEM] */
struct S1170 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_QUERY] */
struct S1171 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_QUERY] */
struct S1172 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_QUERY] */
struct S1173 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_PROCEDURE] */
struct S1174 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_PROCEDURE] */
struct S1175 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_PROCEDURE] */
struct S1176 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
struct S1177 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_NAMED_CLASS] */
struct S1178 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_NAMED_CLASS] */
struct S1179 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NAMED_CLASS] */
struct S1180 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_EXPRESSION_ITEM] */
struct S1182 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_EXPRESSION_ITEM] */
struct S1184 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INSTRUCTION] */
struct S1186 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INSTRUCTION] */
struct S1187 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
struct S1189 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
struct S1190 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_NAMED_OBJECT_TEST] */
struct S1191 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NAMED_OBJECT_TEST] */
struct S1192 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ITERATION_COMPONENT] */
struct S1193 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT] */
struct S1194 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT] */
struct S1195 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT] */
struct S1196 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_BASE_TYPE] */
struct S1197 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_BASE_TYPE] */
struct S1198 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [NATURAL_8, ET_CLASS_NAME] */
struct S1199 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* position */
	T0* volatile a4; /* key_equality_tester */
	T0* volatile a5; /* internal_keys */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* count */
	T0* volatile a10; /* equality_tester */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ECF_LIBRARY, STRING_8] */
struct S1200 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_ECF_LIBRARY] */
struct S1201 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_LIBRARY] */
struct S1204 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_DOTNET_ASSEMBLY] */
struct S1205 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
struct S1208 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_DOTNET_ASSEMBLY] */
struct S1209 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_EIFFEL_SCANNER */
struct S1210 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* last_token */
	T0* volatile a2; /* last_value */
	T0* volatile a3; /* input_name */
	T6 volatile a4; /* position */
	T6 volatile a5; /* column */
	T6 volatile a6; /* line */
	T0* volatile a7; /* input_filter */
	T6 volatile a8; /* yy_start_state */
	T0* volatile a9; /* character_entity */
	T0* volatile a10; /* input_stream */
	T0* volatile a11; /* input_resolver */
	T1 volatile a12; /* yy_more_flag */
	T6 volatile a13; /* yy_more_len */
	T6 volatile a14; /* yy_end */
	T6 volatile a15; /* yy_start */
	T6 volatile a16; /* yy_line */
	T6 volatile a17; /* yy_column */
	T6 volatile a18; /* yy_position */
	T0* volatile a19; /* yy_state_stack */
	T6 volatile a20; /* yy_state_count */
	T0* volatile a21; /* yy_content_area */
	T0* volatile a22; /* yy_unicode_content_area */
	T0* volatile a23; /* yy_accept */
	T6 volatile a24; /* yy_last_accepting_state */
	T6 volatile a25; /* yy_last_accepting_cpos */
	T0* volatile a26; /* yy_content */
	T0* volatile a27; /* yy_ec */
	T0* volatile a28; /* yy_chk */
	T0* volatile a29; /* yy_base */
	T0* volatile a30; /* yy_def */
	T0* volatile a31; /* yy_meta */
	T0* volatile a32; /* yy_nxt */
	T6 volatile a33; /* yy_lp */
	T0* volatile a34; /* yy_acclist */
	T6 volatile a35; /* yy_looking_for_trail_begin */
	T6 volatile a36; /* yy_full_match */
	T6 volatile a37; /* yy_full_state */
	T6 volatile a38; /* yy_full_lp */
	T0* volatile a39; /* input_buffer */
	T1 volatile a40; /* yy_rejected */
	T0* volatile a41; /* yy_pushed_start_conditions */
	T0* volatile a42; /* last_string_value */
	T0* volatile a43; /* last_error */
	T6 volatile a44; /* pushed_start_condition_count */
};

/* Struct for type XM_DEFAULT_POSITION */
struct S1211 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* row */
	T6 volatile a2; /* column */
	T0* volatile a3; /* source_name */
	T6 volatile a4; /* byte_index */
};

/* Struct for type DS_BILINKED_LIST [XM_POSITION] */
struct S1213 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* first_cell */
	T0* volatile a2; /* last_cell */
	T6 volatile a3; /* count */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_LINKED_STACK [XM_EIFFEL_SCANNER] */
struct S1214 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* first_cell */
};

/* Struct for type XM_CALLBACKS_NULL */
struct S1215 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [XM_EIFFEL_ENTITY_DEF, STRING_8] */
struct S1216 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* count */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* capacity */
	T0* volatile a7; /* key_equality_tester */
	T0* volatile a8; /* internal_keys */
	T6 volatile a9; /* modulus */
	T6 volatile a10; /* slots_position */
	T6 volatile a11; /* clashes_previous_position */
	T0* volatile a12; /* internal_cursor */
	T6 volatile a13; /* found_position */
	T0* volatile a14; /* key_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* equality_tester */
	T0* volatile a20; /* hash_function */
};

/* Struct for type XM_NULL_EXTERNAL_RESOLVER */
struct S1218 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_DTD_CALLBACKS_NULL */
struct S1220 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_EIFFEL_PARSER_NAME] */
struct S1221 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_EIFFEL_PARSER_NAME */
struct S1222 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* use_namespaces */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first */
	T0* volatile a4; /* second */
	T0* volatile a5; /* tail */
};

/* Struct for type SPECIAL [XM_EIFFEL_PARSER_NAME] */
struct S1223 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
struct S1224 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET [XM_EIFFEL_PARSER_NAME] */
struct S1225 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* equality_tester */
	T6 volatile a3; /* count */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* modulus */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* clashes_previous_position */
	T6 volatile a10; /* found_position */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* item_storage */
	T0* volatile a15; /* internal_cursor */
	T0* volatile a16; /* hash_function */
};

/* Struct for type SPECIAL [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
struct S1226 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_DTD_EXTERNAL_ID] */
struct S1227 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_DTD_EXTERNAL_ID */
struct S1228 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* system_id */
	T0* volatile a2; /* public_id */
};

/* Struct for type SPECIAL [XM_DTD_EXTERNAL_ID] */
struct S1229 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_DTD_ELEMENT_CONTENT] */
struct S1230 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_DTD_ELEMENT_CONTENT */
struct S1231 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* items */
	T2 volatile a2; /* repetition */
	T0* volatile a3; /* name */
	T2 volatile a4; /* type */
	T1 volatile a5; /* is_character_data_allowed */
};

/* Struct for type SPECIAL [XM_DTD_ELEMENT_CONTENT] */
struct S1232 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
struct S1233 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT] */
struct S1234 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type SPECIAL [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
struct S1235 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_DTD_ATTRIBUTE_CONTENT] */
struct S1236 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_DTD_ATTRIBUTE_CONTENT */
struct S1237 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* default_value */
	T2 volatile a3; /* type */
	T1 volatile a4; /* is_list_type */
	T0* volatile a5; /* enumeration_list */
	T2 volatile a6; /* value */
};

/* Struct for type SPECIAL [XM_DTD_ATTRIBUTE_CONTENT] */
struct S1238 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [STRING_8]] */
struct S1239 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_BILINKED_LIST [STRING_8] */
struct S1240 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* equality_tester */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type SPECIAL [DS_BILINKED_LIST [STRING_8]] */
struct S1241 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [BOOLEAN] */
struct S1242 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_EIFFEL_DECLARATION] */
struct S1243 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_EIFFEL_DECLARATION */
struct S1244 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* encoding */
	T0* volatile a2; /* version */
	T1 volatile a3; /* stand_alone */
};

/* Struct for type SPECIAL [XM_EIFFEL_DECLARATION] */
struct S1245 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type XM_EIFFEL_ENTITY_DEF */
struct S1248 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* literal_name */
	T0* volatile a2; /* value */
	T0* volatile a3; /* resolver */
	T0* volatile a4; /* external_id */
	T0* volatile a5; /* character_entity */
	T0* volatile a6; /* yy_pushed_start_conditions */
	T0* volatile a7; /* input_buffer */
	T1 volatile a8; /* in_use */
	T0* volatile a9; /* input_name */
	T0* volatile a10; /* last_value */
	T0* volatile a11; /* last_string_value */
	T0* volatile a12; /* last_error */
	T1 volatile a13; /* yy_rejected */
	T6 volatile a14; /* yy_state_count */
	T6 volatile a15; /* yy_full_match */
	T6 volatile a16; /* yy_lp */
	T6 volatile a17; /* yy_looking_for_trail_begin */
	T6 volatile a18; /* yy_full_lp */
	T6 volatile a19; /* yy_full_state */
	T6 volatile a20; /* pushed_start_condition_count */
	T6 volatile a21; /* yy_start_state */
	T6 volatile a22; /* yy_line */
	T6 volatile a23; /* yy_column */
	T6 volatile a24; /* yy_position */
	T6 volatile a25; /* yy_start */
	T6 volatile a26; /* yy_end */
	T6 volatile a27; /* line */
	T6 volatile a28; /* column */
	T6 volatile a29; /* position */
	T1 volatile a30; /* yy_more_flag */
	T6 volatile a31; /* yy_more_len */
	T6 volatile a32; /* yy_last_accepting_state */
	T6 volatile a33; /* yy_last_accepting_cpos */
	T0* volatile a34; /* yy_state_stack */
	T0* volatile a35; /* yy_nxt */
	T0* volatile a36; /* yy_chk */
	T0* volatile a37; /* yy_base */
	T0* volatile a38; /* yy_def */
	T0* volatile a39; /* yy_ec */
	T0* volatile a40; /* yy_meta */
	T0* volatile a41; /* yy_accept */
	T0* volatile a42; /* yy_content */
	T0* volatile a43; /* yy_content_area */
	T0* volatile a44; /* yy_unicode_content_area */
	T6 volatile a45; /* last_token */
	T0* volatile a46; /* input_filter */
	T0* volatile a47; /* input_stream */
	T0* volatile a48; /* input_resolver */
	T0* volatile a49; /* yy_acclist */
};

/* Struct for type XM_EIFFEL_SCANNER_DTD */
struct S1249 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* last_token */
	T0* volatile a2; /* last_value */
	T0* volatile a3; /* input_name */
	T6 volatile a4; /* position */
	T6 volatile a5; /* column */
	T6 volatile a6; /* line */
	T0* volatile a7; /* input_filter */
	T6 volatile a8; /* yy_start_state */
	T0* volatile a9; /* character_entity */
	T0* volatile a10; /* input_stream */
	T0* volatile a11; /* input_resolver */
	T1 volatile a12; /* decl_start_sent */
	T1 volatile a13; /* decl_end_sent */
	T1 volatile a14; /* yy_more_flag */
	T6 volatile a15; /* yy_more_len */
	T6 volatile a16; /* yy_end */
	T6 volatile a17; /* yy_start */
	T6 volatile a18; /* yy_line */
	T6 volatile a19; /* yy_column */
	T6 volatile a20; /* yy_position */
	T0* volatile a21; /* yy_state_stack */
	T6 volatile a22; /* yy_state_count */
	T0* volatile a23; /* yy_content_area */
	T0* volatile a24; /* yy_unicode_content_area */
	T0* volatile a25; /* yy_accept */
	T6 volatile a26; /* yy_last_accepting_state */
	T6 volatile a27; /* yy_last_accepting_cpos */
	T0* volatile a28; /* yy_content */
	T0* volatile a29; /* yy_ec */
	T0* volatile a30; /* yy_chk */
	T0* volatile a31; /* yy_base */
	T0* volatile a32; /* yy_def */
	T0* volatile a33; /* yy_meta */
	T0* volatile a34; /* yy_nxt */
	T6 volatile a35; /* yy_lp */
	T0* volatile a36; /* yy_acclist */
	T6 volatile a37; /* yy_looking_for_trail_begin */
	T6 volatile a38; /* yy_full_match */
	T6 volatile a39; /* yy_full_state */
	T6 volatile a40; /* yy_full_lp */
	T0* volatile a41; /* input_buffer */
	T1 volatile a42; /* yy_rejected */
	T0* volatile a43; /* yy_pushed_start_conditions */
	T0* volatile a44; /* last_string_value */
	T0* volatile a45; /* last_error */
	T6 volatile a46; /* pushed_start_condition_count */
};

/* Struct for type XM_EIFFEL_PE_ENTITY_DEF */
struct S1251 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* resolver */
	T0* volatile a2; /* external_id */
	T0* volatile a3; /* literal_name */
	T0* volatile a4; /* value */
	T0* volatile a5; /* character_entity */
	T0* volatile a6; /* yy_pushed_start_conditions */
	T0* volatile a7; /* input_buffer */
	T1 volatile a8; /* in_use */
	T0* volatile a9; /* input_name */
	T0* volatile a10; /* last_value */
	T0* volatile a11; /* last_string_value */
	T0* volatile a12; /* last_error */
	T1 volatile a13; /* yy_rejected */
	T6 volatile a14; /* yy_state_count */
	T6 volatile a15; /* yy_full_match */
	T6 volatile a16; /* yy_lp */
	T6 volatile a17; /* yy_looking_for_trail_begin */
	T6 volatile a18; /* yy_full_lp */
	T6 volatile a19; /* yy_full_state */
	T6 volatile a20; /* pushed_start_condition_count */
	T6 volatile a21; /* yy_start_state */
	T6 volatile a22; /* yy_line */
	T6 volatile a23; /* yy_column */
	T6 volatile a24; /* yy_position */
	T6 volatile a25; /* yy_start */
	T6 volatile a26; /* yy_end */
	T6 volatile a27; /* line */
	T6 volatile a28; /* column */
	T6 volatile a29; /* position */
	T1 volatile a30; /* yy_more_flag */
	T6 volatile a31; /* yy_more_len */
	T6 volatile a32; /* yy_last_accepting_state */
	T6 volatile a33; /* yy_last_accepting_cpos */
	T0* volatile a34; /* yy_state_stack */
	T1 volatile a35; /* pre_sent */
	T1 volatile a36; /* post_sent */
	T0* volatile a37; /* yy_nxt */
	T0* volatile a38; /* yy_chk */
	T0* volatile a39; /* yy_base */
	T0* volatile a40; /* yy_def */
	T0* volatile a41; /* yy_ec */
	T0* volatile a42; /* yy_meta */
	T0* volatile a43; /* yy_accept */
	T0* volatile a44; /* yy_content */
	T0* volatile a45; /* yy_content_area */
	T0* volatile a46; /* yy_unicode_content_area */
	T6 volatile a47; /* last_token */
	T0* volatile a48; /* input_filter */
	T0* volatile a49; /* input_stream */
	T0* volatile a50; /* input_resolver */
	T0* volatile a51; /* yy_acclist */
};

/* Struct for type XM_NAMESPACE_RESOLVER */
struct S1252 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next */
	T0* volatile a2; /* element_local_part */
	T0* volatile a3; /* element_prefix */
	T0* volatile a4; /* context */
	T1 volatile a5; /* forward_xmlns */
	T0* volatile a6; /* attributes_prefix */
	T0* volatile a7; /* attributes_local_part */
	T0* volatile a8; /* attributes_value */
};

/* Struct for type ARRAY [XM_CALLBACKS_FILTER] */
struct S1253 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [XM_CALLBACKS_FILTER] */
struct S1254 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_SET [XM_NAMESPACE] */
struct S1255 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T6 volatile a2; /* capacity */
	T6 volatile a3; /* modulus */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* position */
	T0* volatile a7; /* special_item_routines */
	T0* volatile a8; /* item_storage */
	T0* volatile a9; /* clashes */
	T0* volatile a10; /* slots */
	T6 volatile a11; /* found_position */
	T0* volatile a12; /* internal_cursor */
	T6 volatile a13; /* slots_position */
	T6 volatile a14; /* count */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* hash_function */
};

/* Struct for type XM_LINKED_LIST [XM_DOCUMENT_NODE] */
struct S1256 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* last_cell */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_ECF_SYSTEM_CONFIG] */
struct S1259 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_SYSTEM_CONFIG] */
struct S1261 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ECF_SYSTEM_CONFIG, STRING_8] */
struct S1262 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ECF_TARGET, STRING_8] */
struct S1263 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_ECF_TARGET] */
struct S1264 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_TARGET] */
struct S1266 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, STRING_8] */
struct S1268 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* table */
	T0* volatile a2; /* equality_tester */
	T0* volatile a3; /* internal_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
struct S1269 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable RX_REGULAR_EXPRESSION] */
struct S1270 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ET_DOTNET_ASSEMBLIES */
struct S1271 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* dotnet_assemblies */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_IDENTIFIER] */
struct S1275 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_IDENTIFIER] */
struct S1276 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, STRING_8] */
struct S1277 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLUSTER] */
struct S1278 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CLUSTER] */
struct S1279 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_CLUSTER] */
struct S1280 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_FEATURE, INTEGER_32] */
struct S1281 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* modulus */
	T0* volatile a2; /* key_equality_tester */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* slots */
	T0* volatile a5; /* hash_function */
	T0* volatile a6; /* key_storage */
	T0* volatile a7; /* clashes */
	T6 volatile a8; /* position */
	T6 volatile a9; /* last_position */
	T6 volatile a10; /* capacity */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* count */
	T0* volatile a13; /* equality_tester */
	T6 volatile a14; /* found_position */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* special_item_routines */
	T0* volatile a17; /* special_key_routines */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
};

/* Struct for type ET_FEATURE_IDS */
struct S1282 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* feature_ids */
};

/* Struct for type SPECIAL [ET_DYNAMIC_FEATURE] */
struct S1283 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_FEATURE] */
struct S1284 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
struct S1286 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
struct S1287 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* modulus */
	T0* volatile a2; /* key_equality_tester */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* slots */
	T0* volatile a5; /* hash_function */
	T0* volatile a6; /* key_storage */
	T0* volatile a7; /* clashes */
	T6 volatile a8; /* position */
	T6 volatile a9; /* count */
	T6 volatile a10; /* capacity */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* free_slot */
	T6 volatile a13; /* last_position */
	T0* volatile a14; /* equality_tester */
	T6 volatile a15; /* found_position */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
struct S1288 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* modulus */
	T0* volatile a2; /* key_equality_tester */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* slots */
	T0* volatile a5; /* hash_function */
	T0* volatile a6; /* key_storage */
	T0* volatile a7; /* clashes */
	T0* volatile a8; /* equality_tester */
	T6 volatile a9; /* capacity */
	T6 volatile a10; /* last_position */
	T6 volatile a11; /* free_slot */
	T6 volatile a12; /* position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* special_key_routines */
	T6 volatile a15; /* found_position */
	T0* volatile a16; /* internal_cursor */
	T6 volatile a17; /* count */
	T6 volatile a18; /* slots_position */
	T6 volatile a19; /* clashes_previous_position */
};

/* Struct for type ET_LIKE_N */
struct S1290 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type_mark */
	T0* volatile a2; /* like_keyword */
	T6 volatile a3; /* index */
	T0* volatile a4; /* previous */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_TYPE] */
struct S1291 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [ET_SYSTEM_PROCESSOR] */
struct S1292 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type TUPLE [ET_SYSTEM_PROCESSOR, DS_ARRAYED_LIST [ET_CLASS]] */
struct S1293 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type SPECIAL [ET_DYNAMIC_TYPE_SET] */
struct S1294 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE_SET] */
struct S1295 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [ET_FEATURE] */
struct S1296 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_FEATURE] */
struct S1297 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FEATURE] */
struct S1298 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [ET_ADAPTED_CLASS] */
struct S1300 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INLINE_AGENT] */
struct S1301 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INLINE_AGENT] */
struct S1302 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_INLINE_AGENT] */
struct S1303 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_PROCEDURE]] */
struct S1304 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [ET_PROCEDURE]] */
struct S1305 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_PROCEDURE]] */
struct S1306 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_QUERY]] */
struct S1307 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [ET_QUERY]] */
struct S1308 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_QUERY]] */
struct S1309 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_FEATURE]] */
struct S1310 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST [ET_FEATURE] */
struct S1311 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* equality_tester */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
	T0* volatile a6; /* internal_cursor */
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [ET_FEATURE]] */
struct S1312 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_FEATURE]] */
struct S1313 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
struct S1314 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT] */
struct S1315 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type SPECIAL [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
struct S1316 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
struct S1317 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_NESTED_TYPE_CONTEXT] */
struct S1318 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NESTED_TYPE_CONTEXT] */
struct S1319 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_NESTED_TYPE_CONTEXT] */
struct S1320 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
struct S1321 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST [ET_NAMED_OBJECT_TEST] */
struct S1324 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* special_routines */
	T0* volatile a3; /* storage */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
struct S1325 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST [ET_ITERATION_COMPONENT] */
struct S1327 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
struct S1329 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_INLINE_SEPARATE_ARGUMENT] */
struct S1330 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_ATTACHMENT_SCOPE] */
struct S1332 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ATTACHMENT_SCOPE] */
struct S1333 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ATTACHMENT_SCOPE] */
struct S1334 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ASSERTIONS, ET_FEATURE] */
struct S1335 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_ASSERTIONS] */
struct S1336 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ASSERTIONS] */
struct S1338 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NOTE_TERM] */
struct S1339 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_NOTE_TERM] */
struct S1340 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_NOTE_TERM] */
struct S1341 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
struct S1342 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
struct S1343 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
struct S1344 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_PRECONDITIONS] */
struct S1345 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_PRECONDITIONS] */
struct S1346 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_POSTCONDITIONS] */
struct S1347 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_POSTCONDITIONS] */
struct S1348 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INVARIANTS] */
struct S1349 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INVARIANTS] */
struct S1350 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [NATIVE_STRING] */
struct S1351 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [NATURAL_32] */
struct S1352 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T10 volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST [RX_CHARACTER_SET] */
struct S1353 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [NATURAL_32] */
struct S1354 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [NATURAL_64] */
struct S1355 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T11 volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE [NATURAL_64, NATURAL_32] */
struct S1356 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* position */
	T0* volatile a3; /* internal_cursor */
	T6 volatile a4; /* found_position */
	T0* volatile a5; /* item_storage */
	T6 volatile a6; /* modulus */
	T0* volatile a7; /* key_equality_tester */
	T0* volatile a8; /* key_storage */
	T0* volatile a9; /* slots */
	T0* volatile a10; /* hash_function */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* internal_keys */
	T6 volatile a13; /* capacity */
	T6 volatile a14; /* free_slot */
	T6 volatile a15; /* last_position */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* clashes_previous_position */
	T0* volatile a18; /* equality_tester */
	T0* volatile a19; /* special_item_routines */
	T0* volatile a20; /* special_key_routines */
};

/* Struct for type DS_HASH_TABLE_CURSOR [NATURAL_64, NATURAL_32] */
struct S1357 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ARRAY [INTEGER_32]] */
struct S1358 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [SPECIAL [ARRAY [INTEGER_32]]] */
struct S1359 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_INTEGER_ROUTINES */
struct S1360 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [NATURAL_32] */
struct S1363 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_OLD_EXPRESSION] */
struct S1365 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_OLD_EXPRESSION] */
struct S1366 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_OLD_EXPRESSION] */
struct S1367 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
struct S1369 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_IDENTIFIER] */
struct S1370 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S1371 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S1372 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
struct S1373 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [BOOLEAN] */
struct S1374 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_EQUALITY_EXPRESSION] */
struct S1375 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_EQUALITY_EXPRESSION] */
struct S1376 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_EQUALITY_EXPRESSION] */
struct S1377 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_DYNAMIC_EQUALITY_TYPES] */
struct S1378 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_EQUALITY_TYPES] */
struct S1379 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_EQUALITY_TYPES] */
struct S1380 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
struct S1381 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
struct S1382 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
struct S1383 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S1384 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S1385 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_UNQUALIFIED_CALL_EXPRESSION] */
struct S1386 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_QUALIFIED_CALL_EXPRESSION] */
struct S1387 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_QUALIFIED_CALL_EXPRESSION] */
struct S1388 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_QUALIFIED_CALL_EXPRESSION] */
struct S1389 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ACTUAL_ARGUMENT_LIST] */
struct S1390 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ACTUAL_ARGUMENT_LIST] */
struct S1391 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ACTUAL_ARGUMENT_LIST] */
struct S1392 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_RESULT] */
struct S1393 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_RESULT] */
struct S1394 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_RESULT] */
struct S1395 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S1396 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S1397 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S1398 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_EXPRESSION] */
struct S1399 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_EXPRESSION] */
struct S1400 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_EXPRESSION] */
struct S1401 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_EQUALITY_TESTER [INTEGER_32] */
struct S1402 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
struct S1403 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
struct S1404 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_OBJECT_TEST] */
struct S1406 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_OBJECT_TEST] */
struct S1407 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, INTEGER_32] */
struct S1409 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_HASH_SET_CURSOR [INTEGER_32] */
struct S1410 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_AGENT] */
struct S1413 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_AGENT] */
struct S1414 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [ET_DYNAMIC_TUPLE_TYPE] */
struct S1415 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_DYNAMIC_TUPLE_TYPE] */
struct S1416 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TUPLE_TYPE] */
struct S1417 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_STANDALONE_CLOSURE] */
struct S1419 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type SPECIAL [ET_STANDALONE_CLOSURE] */
struct S1420 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_STANDALONE_CLOSURE] */
struct S1421 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_CONSTANT, ET_FEATURE] */
struct S1422 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_CONSTANT] */
struct S1423 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CONSTANT] */
struct S1425 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
struct S1426 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_INLINE_CONSTANT] */
struct S1427 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INLINE_CONSTANT] */
struct S1429 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [detachable ET_DYNAMIC_FEATURE] */
struct S1430 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
struct S1432 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
struct S1433 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_DYNAMIC_INLINED_EXPRESSION] */
struct S1434 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_INLINED_EXPRESSION] */
struct S1435 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_INLINED_EXPRESSION] */
struct S1436 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_DYNAMIC_CALL_CONTEXT] */
struct S1437 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_CALL_CONTEXT] */
struct S1438 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_CALL_CONTEXT] */
struct S1439 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_FEATURE] */
struct S1440 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_DYNAMIC_FEATURE] */
struct S1441 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [BOOLEAN, STRING_8] */
struct S1444 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [STRING_8]] */
struct S1448 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [STRING_8]] */
struct S1449 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [STRING_8]] */
struct S1450 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [TUPLE [STRING_8, detachable STRING_8]] */
struct S1451 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [TUPLE [STRING_8, detachable STRING_8]] */
struct S1453 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
struct S1454 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
struct S1455 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* count */
	T6 volatile a9; /* slots_position */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* key_equality_tester */
	T0* volatile a12; /* equality_tester */
	T0* volatile a13; /* slots */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* key_storage */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST [detachable DS_LINKABLE [INTEGER_32]] */
struct S1456 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T0* volatile a2; /* special_routines */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type ARRAY [BOOLEAN] */
struct S1457 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type THREAD_ATTRIBUTES */
struct S1459 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* shared */
	T14 volatile a2; /* internal_item */
	T0* volatile a3; /* managed_pointer */
};

/* Struct for type CELL [BOOLEAN] */
struct S1460 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* item */
};

/* Struct for type CHARACTER_PROPERTY */
struct S1462 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SYSTEM_ERROR */
struct S1463 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
};

/* Struct for type ET_INTERNAL_ERROR */
struct S1464 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
};

/* Struct for type ET_SYNTAX_ERROR */
struct S1465 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
	T0* volatile a5; /* filename */
	T0* volatile a6; /* position */
	T1 volatile a7; /* ise_reported */
	T1 volatile a8; /* ise_fatal */
	T1 volatile a9; /* ge_reported */
	T1 volatile a10; /* ge_fatal */
};

/* Struct for type ET_VALIDITY_ERROR */
struct S1466 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* ise_reported */
	T1 volatile a2; /* ge_reported */
	T0* volatile a3; /* default_template */
	T0* volatile a4; /* parameters */
	T0* volatile a5; /* current_class */
	T0* volatile a6; /* class_impl */
	T0* volatile a7; /* position */
	T0* volatile a8; /* code */
	T0* volatile a9; /* etl_code */
	T1 volatile a10; /* ise_fatal */
	T1 volatile a11; /* ge_fatal */
};

/* Struct for type ET_FORMAL_COMMA_ARGUMENT */
struct S1467 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name_item */
	T0* volatile a2; /* declared_type */
	T6 volatile a3; /* index */
	T6 volatile a4; /* attached_index */
	T1 volatile a5; /* is_used */
};

/* Struct for type ET_LOCAL_COMMA_VARIABLE */
struct S1468 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name_item */
	T0* volatile a2; /* declared_type */
	T6 volatile a3; /* index */
	T6 volatile a4; /* attached_index */
	T1 volatile a5; /* is_used */
};

/* Struct for type ET_LABELED_COMMA_ACTUAL_PARAMETER */
struct S1469 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* declared_type */
	T0* volatile a2; /* label_item */
};

/* Struct for type ET_KEYWORD_EXPRESSION */
struct S1471 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* expression */
	T0* volatile a2; /* keyword */
};

/* Struct for type DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1474 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_CREATOR] */
struct S1475 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CREATOR] */
struct S1476 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FEATURE_NAME_ITEM] */
struct S1477 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_FEATURE_NAME_ITEM] */
struct S1478 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ET_ECF_NOTE_ELEMENT */
struct S1479 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* attributes */
	T0* volatile a3; /* elements */
	T0* volatile a4; /* content */
};

/* Struct for type ET_ECF_ADAPTED_LIBRARIES */
struct S1480 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* libraries */
};

/* Struct for type ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
struct S1481 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* dotnet_assemblies */
};

/* Struct for type ET_ECF_FILE_RULES */
struct S1482 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* file_rules */
};

/* Struct for type ET_ECF_NAMESPACES */
struct S1483 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* namespaces */
};

/* Struct for type ET_ECF_EXTERNAL_CFLAGS */
struct S1484 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_INCLUDES */
struct S1485 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_LIBRARIES */
struct S1486 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_LINKER_FLAGS */
struct S1487 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_MAKES */
struct S1488 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_OBJECTS */
struct S1489 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_RESOURCES */
struct S1490 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* external_values */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_ACTION] */
struct S1491 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
struct S1492 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* options */
	T0* volatile a2; /* name_id */
	T0* volatile a3; /* filename_id */
	T0* volatile a4; /* universe */
	T0* volatile a5; /* target */
	T0* volatile a6; /* description */
	T0* volatile a7; /* eifgens_location */
	T0* volatile a8; /* classname_prefix */
	T1 volatile a9; /* is_read_only */
	T1 volatile a10; /* use_application_options */
	T0* volatile a11; /* class_options */
	T0* volatile a12; /* class_renamings */
	T0* volatile a13; /* visible_classes */
	T0* volatile a14; /* conditions */
	T0* volatile a15; /* name */
	T0* volatile a16; /* library */
};

/* Struct for type ET_ECF_VERSION */
struct S1494 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* internal_major */
	T6 volatile a2; /* internal_minor */
	T6 volatile a3; /* internal_release */
	T6 volatile a4; /* internal_build */
	T0* volatile a5; /* product */
	T0* volatile a6; /* company */
	T0* volatile a7; /* copyright */
};

/* Struct for type ET_ECF_NAMESPACE */
struct S1495 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* description */
	T0* volatile a3; /* classname_prefix */
	T0* volatile a4; /* class_renamings */
	T0* volatile a5; /* notes */
};

/* Struct for type ET_ECF_EXTERNAL_CFLAG */
struct S1496 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* flag */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_INCLUDE */
struct S1497 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_LIBRARY */
struct S1498 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_LINKER_FLAG */
struct S1499 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* flag */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_MAKE */
struct S1500 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_OBJECT */
struct S1501 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_RESOURCE */
struct S1502 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pathname */
	T0* volatile a2; /* description */
	T0* volatile a3; /* conditions */
};

/* Struct for type ET_ECF_FILE_RULE */
struct S1503 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* exclude */
	T0* volatile a2; /* include */
	T0* volatile a3; /* exclude_regexp */
	T0* volatile a4; /* include_regexp */
	T0* volatile a5; /* conditions */
	T0* volatile a6; /* description */
};

/* Struct for type ET_ECF_ACTION */
struct S1504 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* command_name */
	T0* volatile a2; /* description */
	T1 volatile a3; /* must_succeed */
	T0* volatile a4; /* working_directory */
	T0* volatile a5; /* conditions */
};

/* Struct for type DS_HASH_TABLE [ET_ECF_OPTIONS, STRING_8] */
struct S1505 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* last_position */
	T6 volatile a4; /* position */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T0* volatile a8; /* key_equality_tester */
	T0* volatile a9; /* internal_keys */
	T0* volatile a10; /* equality_tester */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_VISIBLE_CLASS] */
struct S1506 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type TUPLE [ET_ECF_OPTIONS] */
struct S1507 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_OPTIONS]] */
struct S1508 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type ET_ECF_ROOT_CLASS */
struct S1509 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_name */
	T0* volatile a2; /* creation_procedure_name */
	T0* volatile a3; /* cluster_name */
};

/* Struct for type ET_ECF_ROOT_ALL_CLASSES */
struct S1510 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_ECF_ORED_CONDITIONS */
struct S1511 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditions */
};

/* Struct for type ET_ECF_ANDED_CONDITIONS */
struct S1512 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* conditions */
};

/* Struct for type ET_ECF_VISIBLE_CLASS */
struct S1513 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_name */
	T0* volatile a2; /* feature_name */
	T0* volatile a3; /* new_class_name */
	T0* volatile a4; /* new_feature_name */
};

/* Struct for type ET_ECF_BUILD_CONDITION */
struct S1515 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T1 volatile a2; /* is_excluded */
};

/* Struct for type ET_ECF_CONCURRENCY_CONDITION */
struct S1516 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T1 volatile a2; /* is_excluded */
};

/* Struct for type ET_ECF_VOID_SAFETY_CONDITION */
struct S1517 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T1 volatile a2; /* is_excluded */
};

/* Struct for type ET_ECF_CUSTOM_CONDITION */
struct S1518 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T0* volatile a2; /* value */
	T0* volatile a3; /* match */
	T1 volatile a4; /* is_excluded */
};

/* Struct for type ET_ECF_DOTNET_CONDITION */
struct S1519 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* value */
};

/* Struct for type ET_ECF_DYNAMIC_RUNTIME_CONDITION */
struct S1520 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* value */
};

/* Struct for type ET_ECF_PLATFORM_CONDITION */
struct S1521 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* value */
	T1 volatile a2; /* is_excluded */
};

/* Struct for type ET_ECF_COMPILER_VERSION_CONDITION */
struct S1523 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* min_value */
	T0* volatile a2; /* max_value */
};

/* Struct for type ET_ECF_MSIL_CLR_VERSION_CONDITION */
struct S1524 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* min_value */
	T0* volatile a2; /* max_value */
};

/* Struct for type UT_COUNTER */
struct S1526 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* item */
};

/* Struct for type KL_AGENT_ROUTINES [ET_CLASS] */
struct S1527 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [UT_COUNTER] */
struct S1528 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [KL_AGENT_ROUTINES [ET_CLASS], PROCEDURE [TUPLE]] */
struct S1529 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_CLASS]]] */
struct S1530 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
struct S1531 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type DS_HASH_SET [ET_DOTNET_ASSEMBLY] */
struct S1533 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* modulus */
	T6 volatile a3; /* last_position */
	T6 volatile a4; /* free_slot */
	T6 volatile a5; /* position */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* clashes */
	T0* volatile a11; /* slots */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* internal_cursor */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]] */
struct S1534 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [DS_HASH_SET [ET_DOTNET_ASSEMBLY]] */
struct S1535 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_INTERNAL_UNIVERSE] */
struct S1536 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_INTERNAL_UNIVERSE]] */
struct S1537 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]], PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]]] */
struct S1538 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [ET_DOTNET_ASSEMBLIES] */
struct S1539 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_SYSTEM_PROCESSOR] */
struct S1540 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_CLUSTER] */
struct S1541 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_CLUSTER]] */
struct S1542 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM] */
struct S1543 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type DS_HASH_SET [ET_INTERNAL_UNIVERSE] */
struct S1544 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* capacity */
	T6 volatile a3; /* modulus */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* count */
	T6 volatile a8; /* clashes_previous_position */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* item_storage */
	T0* volatile a11; /* clashes */
	T0* volatile a12; /* slots */
	T6 volatile a13; /* found_position */
	T0* volatile a14; /* internal_cursor */
	T0* volatile a15; /* equality_tester */
	T0* volatile a16; /* hash_function */
};

/* Struct for type TUPLE [ET_AST_PROCESSOR] */
struct S1545 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1546 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY]] */
struct S1547 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type ET_ANCESTORS_STATUS_CHECKER */
struct S1548 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_type_checker */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type ET_FLATTENING_STATUS_CHECKER */
struct S1549 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* class_type_checker */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type ET_INTERFACE_STATUS_CHECKER */
struct S1550 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* qualified_anchored_type_checker */
	T0* volatile a2; /* class_type_checker */
	T0* volatile a3; /* current_class */
	T0* volatile a4; /* system_processor */
};

/* Struct for type ET_IMPLEMENTATION_STATUS_CHECKER */
struct S1551 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_class */
	T0* volatile a2; /* system_processor */
};

/* Struct for type TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
struct S1552 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_FLATTENING_STATUS_CHECKER] */
struct S1553 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_INTERFACE_STATUS_CHECKER] */
struct S1554 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
struct S1555 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [BOOLEAN] */
struct S1556 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile z1;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_CLUSTER]]] */
struct S1559 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [DS_HASH_SET [ET_INTERNAL_UNIVERSE]] */
struct S1563 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]]] */
struct S1564 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type ET_PARENTHESIS_EXPRESSION */
struct S1565 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* name */
	T0* volatile a3; /* arguments */
	T6 volatile a4; /* index */
};

/* Struct for type DS_HASH_SET [ET_LIBRARY] */
struct S1568 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* modulus */
	T6 volatile a3; /* last_position */
	T6 volatile a4; /* free_slot */
	T6 volatile a5; /* position */
	T0* volatile a6; /* special_item_routines */
	T0* volatile a7; /* item_storage */
	T0* volatile a8; /* clashes */
	T0* volatile a9; /* slots */
	T6 volatile a10; /* found_position */
	T0* volatile a11; /* internal_cursor */
	T6 volatile a12; /* slots_position */
	T6 volatile a13; /* count */
	T6 volatile a14; /* clashes_previous_position */
	T0* volatile a15; /* hash_function */
	T0* volatile a16; /* equality_tester */
};

/* Struct for type TUPLE [DS_HASH_SET [ET_LIBRARY]] */
struct S1569 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type SPECIAL [ET_RENAME_ITEM] */
struct S1573 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_RENAME_ITEM] */
struct S1574 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_EXPORT] */
struct S1576 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_EXPORT] */
struct S1577 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_CLUSTER_DEPENDENCE_CONSTRAINT */
struct S1580 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* current_cluster */
	T0* volatile a2; /* pathname_buffer */
	T0* volatile a3; /* group_names */
	T0* volatile a4; /* group_pathnames */
};

/* Struct for type ET_PARENT_FEATURE */
struct S1582 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next */
	T0* volatile a2; /* parent */
	T0* volatile a3; /* precursor_feature */
	T0* volatile a4; /* new_name */
	T0* volatile a5; /* undefine_name */
	T0* volatile a6; /* redefine_name */
	T0* volatile a7; /* select_name */
	T0* volatile a8; /* merged_feature */
	T1 volatile a9; /* has_other_deferred */
	T1 volatile a10; /* has_other_effective */
};

/* Struct for type DS_QUICK_SORTER [ET_QUERY] */
struct S1583 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type DS_QUICK_SORTER [ET_PROCEDURE] */
struct S1584 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type ET_INHERITED_FEATURE */
struct S1585 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next */
	T0* volatile a2; /* parent_feature */
	T0* volatile a3; /* flattened_feature */
	T0* volatile a4; /* flattened_parent */
	T6 volatile a5; /* first_seed */
	T0* volatile a6; /* other_seeds */
	T1 volatile a7; /* is_selected */
	T0* volatile a8; /* replicated_features */
	T0* volatile a9; /* replicated_seeds */
};

/* Struct for type ET_REDECLARED_FEATURE */
struct S1587 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next */
	T0* volatile a2; /* flattened_feature */
	T0* volatile a3; /* parent_feature */
	T6 volatile a4; /* first_seed */
	T0* volatile a5; /* other_seeds */
	T1 volatile a6; /* is_selected */
	T0* volatile a7; /* replicated_features */
	T0* volatile a8; /* replicated_seeds */
};

/* Struct for type KL_CHARACTER_BUFFER */
struct S1595 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* as_special */
	T0* volatile a2; /* area */
};

/* Struct for type SPECIAL [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1597 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S1598 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ALIAS_NAME] */
struct S1599 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ALIAS_NAME] */
struct S1600 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_AGENT_IMPLICIT_CURRENT_TARGET */
struct S1601 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* agent_expression */
};

/* Struct for type SPECIAL [ET_CHOICE_ITEM] */
struct S1603 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CHOICE_ITEM] */
struct S1604 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1605 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1607 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CONVERT_FEATURE_ITEM] */
struct S1608 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CONVERT_FEATURE_ITEM] */
struct S1610 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ELSEIF_EXPRESSION] */
struct S1611 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ELSEIF_EXPRESSION] */
struct S1612 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ELSEIF_PART] */
struct S1613 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ELSEIF_PART] */
struct S1614 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_FEATURE_CLAUSE] */
struct S1615 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FEATURE_CLAUSE] */
struct S1616 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_FORMAL_ARGUMENT_ITEM] */
struct S1617 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FORMAL_ARGUMENT_ITEM] */
struct S1619 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_FORMAL_PARAMETER_ITEM] */
struct S1620 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FORMAL_PARAMETER_ITEM] */
struct S1621 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_LOCAL_VARIABLE_ITEM] */
struct S1622 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_LOCAL_VARIABLE_ITEM] */
struct S1624 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_MANIFEST_STRING_ITEM] */
struct S1626 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_MANIFEST_STRING_ITEM] */
struct S1627 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_NOTE_ITEM] */
struct S1629 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NOTE_ITEM] */
struct S1630 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_NOTE_TERM_ITEM] */
struct S1631 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_NOTE_TERM_ITEM] */
struct S1632 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_PARENT_LIST] */
struct S1633 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_PARENT_LIST] */
struct S1634 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_TYPE_CONSTRAINT_ITEM] */
struct S1636 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_TYPE_CONSTRAINT_ITEM] */
struct S1637 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_WHEN_EXPRESSION] */
struct S1638 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_WHEN_EXPRESSION] */
struct S1639 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_WHEN_PART] */
struct S1640 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_WHEN_PART] */
struct S1641 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_LINKED_LIST [XM_ELEMENT_NODE] */
struct S1643 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* last_cell */
	T6 volatile a3; /* count */
	T0* volatile a4; /* first_cell */
};

/* Struct for type UC_STRING_EQUALITY_TESTER */
struct S1645 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_LINKED_LIST_CURSOR [DS_PAIR [XM_POSITION, XM_NODE]] */
struct S1647 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_LINKED_LIST [DS_PAIR [XM_POSITION, XM_NODE]] */
struct S1648 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T0* volatile a2; /* last_cell */
	T6 volatile a3; /* count */
	T0* volatile a4; /* first_cell */
};

/* Struct for type DS_PAIR [XM_POSITION, XM_NODE] */
struct S1649 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* first */
	T0* volatile a2; /* second */
};

/* Struct for type XM_EIFFEL_INPUT_STREAM */
struct S1650 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_string */
	T6 volatile a2; /* encoding */
	T0* volatile a3; /* utf_queue */
	T0* volatile a4; /* impl */
};

/* Struct for type DS_LINKED_LIST_CURSOR [XM_DOCUMENT_NODE] */
struct S1651 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_TARGET] */
struct S1652 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_ECF_OPTIONS, STRING_8] */
struct S1653 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* container */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ADAPTED_UNIVERSE] */
struct S1654 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_TARGET_PARENT] */
struct S1656 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_CLUSTER] */
struct S1657 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
struct S1659 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE] */
struct S1662 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_DYNAMIC_TYPE] */
struct S1663 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type INTEGER_OVERFLOW_CHECKER */
struct S1668 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* integer_overflow_state1 */
	T0* volatile a2; /* integer_overflow_state2 */
	T0* volatile a3; /* natural_overflow_state1 */
	T0* volatile a4; /* natural_overflow_state2 */
};

/* Struct for type CONSOLE */
struct S1669 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_string */
	T14 volatile a2; /* file_pointer */
	T0* volatile a3; /* last_string_32 */
	T6 volatile a4; /* mode */
	T0* volatile a5; /* internal_name */
	T0* volatile a6; /* internal_detachable_name_pointer */
	T2 volatile a7; /* last_character */
};

/* Struct for type C_DATE */
struct S1670 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* millisecond_now */
	T0* volatile a2; /* internal_item */
	T1 volatile a3; /* is_utc */
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, STRING_8] */
struct S1672 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_NOTE_ELEMENT] */
struct S1673 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_NOTE_ELEMENT] */
struct S1674 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_NOTE_ELEMENT] */
struct S1675 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type ET_DYNAMIC_TARGET_LIST */
struct S1677 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type DS_ARRAYED_STACK [YY_BUFFER] */
struct S1678 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
};

/* Struct for type UT_SYNTAX_ERROR */
struct S1679 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type UT_TOO_MANY_INCLUDES_ERROR */
struct S1680 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type ARRAY [detachable STRING_8] */
struct S1681 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [ET_ADAPTED_LIBRARY] */
struct S1683 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ADAPTED_LIBRARY] */
struct S1684 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_LIBRARY] */
struct S1685 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1686 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1687 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1688 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_TYPE_ITEM] */
struct S1689 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_TYPE_ITEM] */
struct S1690 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, ET_CLASS] */
struct S1691 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* count */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* modulus */
	T6 volatile a8; /* capacity */
	T6 volatile a9; /* slots_position */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* key_equality_tester */
	T0* volatile a12; /* equality_tester */
	T0* volatile a13; /* internal_cursor */
	T0* volatile a14; /* key_storage */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_BASE_TYPE, ET_CLASS] */
struct S1692 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST [ET_BASE_TYPE_CONSTRAINT] */
struct S1694 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [NATURAL_32] */
struct S1695 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T0* volatile a2; /* special_routines */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type ET_BASE_TYPE_CONSTRAINT_LIST */
struct S1696 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* left_brace */
	T0* volatile a2; /* right_brace */
	T0* volatile a3; /* storage */
	T6 volatile a4; /* count */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
struct S1697 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_FLATTENED_FEATURE] */
struct S1698 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [ET_FEATURE_NAME] */
struct S1701 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FLATTENED_FEATURE] */
struct S1702 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_FEATURE_NAME] */
struct S1703 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
struct S1705 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLIENT_LIST] */
struct S1706 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CLIENT_LIST] */
struct S1707 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_CLIENT_LIST] */
struct S1708 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_CLIENT, ET_CLASS] */
struct S1710 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CLIENT] */
struct S1711 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_CLIENT] */
struct S1712 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_SET [ET_FEATURE_NAME] */
struct S1713 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* internal_cursor */
	T6 volatile a4; /* count */
	T0* volatile a5; /* item_storage */
	T6 volatile a6; /* modulus */
	T6 volatile a7; /* last_position */
	T6 volatile a8; /* free_slot */
	T6 volatile a9; /* position */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* slots_position */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [BOOLEAN, ET_FEATURE_NAME] */
struct S1714 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T6 volatile a3; /* found_position */
	T0* volatile a4; /* internal_cursor */
	T0* volatile a5; /* key_storage */
	T0* volatile a6; /* item_storage */
	T0* volatile a7; /* key_equality_tester */
	T0* volatile a8; /* internal_keys */
	T6 volatile a9; /* position */
	T6 volatile a10; /* last_position */
	T6 volatile a11; /* free_slot */
	T6 volatile a12; /* modulus */
	T0* volatile a13; /* equality_tester */
	T6 volatile a14; /* slots_position */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* special_item_routines */
	T0* volatile a17; /* clashes */
	T0* volatile a18; /* slots */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [ET_REPLICABLE_FEATURE, INTEGER_32] */
struct S1715 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* internal_cursor */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T0* volatile a4; /* key_storage */
	T6 volatile a5; /* count */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T6 volatile a9; /* capacity */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* key_equality_tester */
	T0* volatile a12; /* clashes */
	T6 volatile a13; /* modulus */
	T0* volatile a14; /* slots */
	T6 volatile a15; /* slots_position */
	T6 volatile a16; /* clashes_previous_position */
	T0* volatile a17; /* special_item_routines */
	T0* volatile a18; /* special_key_routines */
	T0* volatile a19; /* hash_function */
};

/* Struct for type ET_REPLICATED_FEATURE */
struct S1717 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* features */
	T6 volatile a2; /* selected_count */
};

/* Struct for type DS_LINKED_LIST [ET_ADAPTED_FEATURE] */
struct S1718 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_PARENT_FEATURE] */
struct S1719 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type ET_DOTNET_SIGNATURE_TESTER */
struct S1720 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET [ET_DOTNET_FEATURE] */
struct S1721 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* found_position */
	T0* volatile a3; /* item_storage */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* equality_tester */
	T6 volatile a9; /* count */
	T6 volatile a10; /* slots_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T0* volatile a14; /* internal_cursor */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [DS_LINKED_LIST [ET_DOTNET_FEATURE], ET_DOTNET_FEATURE] */
struct S1723 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T0* volatile a4; /* internal_keys */
	T6 volatile a5; /* count */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T6 volatile a9; /* capacity */
	T0* volatile a10; /* equality_tester */
	T0* volatile a11; /* internal_cursor */
	T0* volatile a12; /* key_storage */
	T0* volatile a13; /* clashes */
	T6 volatile a14; /* modulus */
	T0* volatile a15; /* slots */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* clashes_previous_position */
	T0* volatile a18; /* special_item_routines */
	T0* volatile a19; /* special_key_routines */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_LINKED_LIST [ET_DOTNET_FEATURE] */
struct S1725 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_HASH_TOPOLOGICAL_SORTER [ET_LIKE_FEATURE] */
struct S1726 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* cycle */
	T0* volatile a2; /* counts */
	T0* volatile a3; /* successors */
	T0* volatile a4; /* items */
	T0* volatile a5; /* sorted_items */
	T0* volatile a6; /* indexes */
};

/* Struct for type DS_ARRAYED_LIST [ET_LIKE_FEATURE] */
struct S1727 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_LINKED_LIST_CURSOR [ET_PARENT_FEATURE] */
struct S1728 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_LINKED_LIST [ET_PARENT_FEATURE] */
struct S1729 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* last_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_HASH_TABLE [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME], NATURAL_8] */
struct S1730 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T0* volatile a2; /* key_equality_tester */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* special_key_routines */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* internal_cursor */
	T6 volatile a16; /* slots_position */
	T6 volatile a17; /* count */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_FEATURE, INTEGER_32] */
struct S1731 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_CLASS] */
struct S1734 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_RENAME, ET_FEATURE_NAME] */
struct S1736 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_RENAME] */
struct S1737 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_RENAME] */
struct S1740 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_RENAME, ET_ALIAS_NAME] */
struct S1744 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_PROCEDURE] */
struct S1745 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_QUERY] */
struct S1746 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_SPECIAL_ROUTINES [NATURAL_8] */
struct S1749 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [NATURAL_8, ET_CLASS_NAME] */
struct S1750 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type XM_EIFFEL_CHARACTER_ENTITY */
struct S1751 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* code */
};

/* Struct for type DS_BILINKABLE [XM_POSITION] */
struct S1752 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
	T0* volatile a3; /* left */
};

/* Struct for type DS_BILINKED_LIST_CURSOR [XM_POSITION] */
struct S1753 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
};

/* Struct for type DS_LINKABLE [XM_EIFFEL_SCANNER] */
struct S1754 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type SPECIAL [XM_EIFFEL_ENTITY_DEF] */
struct S1755 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [XM_EIFFEL_ENTITY_DEF, STRING_8] */
struct S1758 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_EIFFEL_ENTITY_DEF] */
struct S1759 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_EQUALITY_TESTER [XM_EIFFEL_PARSER_NAME] */
struct S1762 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [XM_EIFFEL_PARSER_NAME] */
struct S1763 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_BILINKED_LIST [XM_DTD_ELEMENT_CONTENT] */
struct S1764 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* first_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* last_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_BILINKED_LIST_CURSOR [XM_DTD_ATTRIBUTE_CONTENT] */
struct S1765 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_BILINKABLE [XM_DTD_ATTRIBUTE_CONTENT] */
struct S1766 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
	T0* volatile a3; /* left */
};

/* Struct for type DS_BILINKED_LIST_CURSOR [STRING_8] */
struct S1767 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* after */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* container */
	T1 volatile a4; /* before */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_BILINKABLE [STRING_8] */
struct S1768 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
	T0* volatile a3; /* left */
};

/* Struct for type XM_NAMESPACE_RESOLVER_CONTEXT */
struct S1769 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* context */
};

/* Struct for type DS_LINKED_QUEUE [detachable STRING_8] */
struct S1772 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* first_cell */
	T6 volatile a2; /* count */
	T0* volatile a3; /* last_cell */
};

/* Struct for type DS_LINKED_QUEUE [STRING_8] */
struct S1773 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* first_cell */
	T0* volatile a2; /* last_cell */
	T6 volatile a3; /* count */
};

/* Struct for type KL_EQUALITY_TESTER [XM_NAMESPACE] */
struct S1774 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_SET_CURSOR [XM_NAMESPACE] */
struct S1775 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [XM_NAMESPACE] */
struct S1776 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [XM_NAMESPACE] */
struct S1777 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_LINKABLE [XM_DOCUMENT_NODE] */
struct S1778 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_ARRAYED_LIST [ET_DOTNET_ASSEMBLY] */
struct S1780 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
	T0* volatile a6; /* equality_tester */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_FEATURE, INTEGER_32] */
struct S1781 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
struct S1783 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type SPECIAL [ET_CALL_NAME] */
struct S1785 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
struct S1787 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_CALL_NAME] */
struct S1788 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
struct S1789 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
struct S1790 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
struct S1792 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
struct S1793 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_SYSTEM_PROCESSOR] */
struct S1794 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_SYSTEM_PROCESSOR] */
struct S1795 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_SYSTEM_PROCESSOR] */
struct S1796 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ADAPTED_CLASS] */
struct S1797 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ADAPTED_CLASS] */
struct S1799 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_CLASS] */
struct S1800 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_NAMED_OBJECT_TEST] */
struct S1801 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ITERATION_COMPONENT] */
struct S1802 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_INLINE_SEPARATE_ARGUMENT] */
struct S1803 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [RX_CHARACTER_SET] */
struct S1804 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [RX_CHARACTER_SET] */
struct S1805 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [NATURAL_64] */
struct S1810 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
struct S1814 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [detachable DS_LINKABLE [INTEGER_32]] */
struct S1815 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [INTEGER_32]] */
struct S1816 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [INTEGER_32]] */
struct S1817 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_STRING_INPUT_STREAM */
struct S1819 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* end_of_input */
	T2 volatile a2; /* last_character */
	T0* volatile a3; /* last_string */
	T0* volatile a4; /* string */
	T6 volatile a5; /* location */
};

/* Struct for type UT_TRISTATE */
struct S1821 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* value */
};

/* Struct for type KL_AGENT_ROUTINES [ANY] */
struct S1822 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [UT_TRISTATE] */
struct S1823 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [KL_AGENT_ROUTINES [ANY], PROCEDURE [TUPLE]] */
struct S1824 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type PREDICATE [TUPLE [ET_MASTER_CLASS]] */
struct S1825 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type PREDICATE [TUPLE] */
struct S1826 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE] */
struct S1828 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T6 volatile a2; /* last_position */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* slots_position */
	T6 volatile a5; /* count */
	T6 volatile a6; /* modulus */
	T6 volatile a7; /* clashes_previous_position */
	T0* volatile a8; /* key_equality_tester */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* found_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* item_storage */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_key_routines */
	T0* volatile a16; /* key_storage */
	T6 volatile a17; /* free_slot */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_UNIVERSE] */
struct S1829 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* container */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type TUPLE [ET_ECF_SYSTEM, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S1830 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type KL_AGENT_ROUTINES [ET_UNIVERSE] */
struct S1831 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type PREDICATE [TUPLE [ET_UNIVERSE]] */
struct S1832 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE]] */
struct S1833 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [KL_AGENT_ROUTINES [ET_UNIVERSE], PREDICATE [TUPLE [ET_UNIVERSE]]] */
struct S1834 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
struct S1836 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN]] */
struct S1837 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN], FUNCTION [TUPLE, BOOLEAN]] */
struct S1838 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
struct S1839 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* static_type */
	T6 volatile a2; /* count */
	T1 volatile a3; /* is_never_void */
	T0* volatile a4; /* dynamic_types */
	T0* volatile a5; /* agent_type */
	T1 volatile a6; /* is_dynamic_types_readonly */
};

/* Struct for type TUPLE [ET_MASTER_CLASS, ET_SYSTEM_PROCESSOR] */
struct S1847 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type DS_CELL [detachable ET_CLASS] */
struct S1848 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
};

/* Struct for type TUPLE [DS_CELL [detachable ET_CLASS]] */
struct S1849 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_EIFFEL_PREPARSER, DS_ARRAYED_LIST [STRING_8]] */
struct S1851 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type ET_SEEDED_QUERY_COMPARATOR */
struct S1853 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ET_SEEDED_PROCEDURE_COMPARATOR */
struct S1856 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [ET_CLIENT_LIST] */
struct S1860 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type ET_STANDARD_ONCE_KEYS */
struct S1864 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type XM_COMMENT */
struct S1867 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* data */
	T0* volatile a2; /* parent */
};

/* Struct for type XM_PROCESSING_INSTRUCTION */
struct S1868 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* data */
	T0* volatile a3; /* parent */
};

/* Struct for type XM_CHARACTER_DATA */
struct S1869 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* content */
	T0* volatile a2; /* parent */
};

/* Struct for type ET_PARENTHESIS_SYMBOL */
struct S1885 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* seed */
	T6 volatile a2; /* compressed_position */
	T2 volatile a3; /* code */
};

/* Struct for type ET_PARENTHESIS_INSTRUCTION */
struct S1888 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
	T0* volatile a2; /* name */
	T0* volatile a3; /* arguments */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
struct S1892 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* left_symbol */
	T0* volatile a3; /* actual_arguments */
	T0* volatile a4; /* storage */
	T0* volatile a5; /* right_symbol */
};

/* Struct for type ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
struct S1893 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
};

/* Struct for type ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
struct S1894 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* agent_expression */
	T6 volatile a2; /* argument_index */
	T6 volatile a3; /* index */
};

/* Struct for type ET_INFIX_CAST_EXPRESSION */
struct S1896 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* expression */
	T0* volatile a3; /* type */
};

/* Struct for type SPECIAL [ET_DYNAMIC_PRECURSOR] */
struct S1899 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRECURSOR] */
struct S1900 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_MASTER_CLASS] */
struct S1908 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type ET_BASE_TYPE_RENAME_CONSTRAINT */
struct S1910 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* type */
	T0* volatile a2; /* renames */
};

/* Struct for type DS_SPARSE_TABLE_KEYS [ET_CLIENT, ET_CLASS] */
struct S1913 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* table */
	T0* volatile a2; /* equality_tester */
	T0* volatile a3; /* internal_cursor */
};

/* Struct for type SPECIAL [NATURAL_16] */
struct S1922 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T9 volatile z2[]; /* item */
};

/* Struct for type ARRAY [NATURAL_16] */
struct S1923 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type ARRAY [NATURAL_32] */
struct S1924 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1925 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1926 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S1927 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_ADAPTED_LIBRARY] */
struct S1928 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S1929 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_FILE_RULE] */
struct S1930 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_NAMESPACE] */
struct S1931 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_CFLAG] */
struct S1932 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_INCLUDE] */
struct S1933 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LIBRARY] */
struct S1934 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S1935 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_MAKE] */
struct S1936 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_OBJECT] */
struct S1937 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_RESOURCE] */
struct S1938 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_ACTION] */
struct S1939 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_ACTION] */
struct S1940 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_ACTION] */
struct S1941 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_ARRAYED_LIST [RX_PCRE_REGULAR_EXPRESSION] */
struct S1942 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_ECF_OPTIONS] */
struct S1943 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_OPTIONS] */
struct S1946 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_VISIBLE_CLASS] */
struct S1947 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_VISIBLE_CLASS] */
struct S1948 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_VISIBLE_CLASS] */
struct S1949 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_ANDED_CONDITIONS] */
struct S1950 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [ET_ECF_CONDITION_ITEM] */
struct S1951 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_DOTNET_ASSEMBLY] */
struct S1952 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DOTNET_ASSEMBLY] */
struct S1953 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_DOTNET_ASSEMBLY] */
struct S1954 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_INTERNAL_UNIVERSE] */
struct S1956 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_INTERNAL_UNIVERSE] */
struct S1958 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_INTERNAL_UNIVERSE] */
struct S1959 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER1 */
struct S1960 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER2 */
struct S1961 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
};

/* Struct for type ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
struct S1962 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
	T0* volatile a2; /* current_class */
	T0* volatile a3; /* system_processor */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER3 */
struct S1963 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* has_fatal_error */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_LIBRARY] */
struct S1964 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_LIBRARY] */
struct S1965 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_LIBRARY] */
struct S1966 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type LX_DFA_WILDCARD */
struct S1967 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* pattern */
	T1 volatile a2; /* is_case_insensitive */
	T0* volatile a3; /* yy_nxt */
	T0* volatile a4; /* yy_accept */
	T0* volatile a5; /* yy_ec */
	T6 volatile a6; /* yyNb_rows */
	T6 volatile a7; /* yyNull_equiv_class */
	T6 volatile a8; /* yyMax_symbol_equiv_class */
	T6 volatile a9; /* match_count */
	T6 volatile a10; /* subject_start */
	T6 volatile a11; /* subject_end */
	T0* volatile a12; /* subject */
	T6 volatile a13; /* matched_start */
	T6 volatile a14; /* matched_end */
};

/* Struct for type DS_HASH_TABLE [LX_WILDCARD, STRING_8] */
struct S1968 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* last_position */
	T0* volatile a3; /* clashes */
	T0* volatile a4; /* item_storage */
	T0* volatile a5; /* key_equality_tester */
	T0* volatile a6; /* internal_keys */
	T6 volatile a7; /* position */
	T6 volatile a8; /* capacity */
	T6 volatile a9; /* slots_position */
	T0* volatile a10; /* equality_tester */
	T6 volatile a11; /* found_position */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* special_item_routines */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type DS_LINKED_QUEUE [CHARACTER_8] */
struct S1971 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* first_cell */
	T0* volatile a3; /* last_cell */
};

/* Struct for type UC_UTF16_ROUTINES */
struct S1972 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET] */
struct S1973 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_ADAPTED_UNIVERSE] */
struct S1974 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ADAPTED_UNIVERSE] */
struct S1975 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_UNIVERSE] */
struct S1976 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_ECF_TARGET_PARENT] */
struct S1977 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_TARGET_PARENT] */
struct S1978 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET_PARENT] */
struct S1979 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_ECF_CLUSTER] */
struct S1980 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_CLUSTER] */
struct S1981 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_CLUSTER] */
struct S1982 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_DYNAMIC_TARGET] */
struct S1983 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TARGET] */
struct S1984 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [YY_BUFFER] */
struct S1985 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [YY_BUFFER] */
struct S1986 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_CLASS] */
struct S1988 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type SPECIAL [ET_BASE_TYPE_CONSTRAINT] */
struct S1989 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_BASE_TYPE_CONSTRAINT] */
struct S1990 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_BASE_TYPE_CONSTRAINT] */
struct S1991 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [NATURAL_32] */
struct S1992 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_FEATURE_NAME] */
struct S1993 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [BOOLEAN, ET_FEATURE_NAME] */
struct S1994 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [ET_REPLICABLE_FEATURE, INTEGER_32] */
struct S1996 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [ET_REPLICABLE_FEATURE] */
struct S1997 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_REPLICABLE_FEATURE] */
struct S1999 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_LINKED_LIST_CURSOR [ET_ADAPTED_FEATURE] */
struct S2000 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
	T1 volatile a3; /* after */
	T0* volatile a4; /* current_cell */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_LINKABLE [ET_ADAPTED_FEATURE] */
struct S2001 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_PARENT_FEATURE] */
struct S2002 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_PARENT_FEATURE] */
struct S2003 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_PARENT_FEATURE] */
struct S2004 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type SPECIAL [ET_DOTNET_FEATURE] */
struct S2005 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_SET_CURSOR [ET_DOTNET_FEATURE] */
struct S2006 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_DOTNET_FEATURE] */
struct S2007 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [DS_LINKED_LIST [ET_DOTNET_FEATURE]] */
struct S2008 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [DS_LINKED_LIST [ET_DOTNET_FEATURE], ET_DOTNET_FEATURE] */
struct S2011 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_LINKED_LIST [ET_DOTNET_FEATURE]] */
struct S2012 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_LINKED_LIST_CURSOR [ET_DOTNET_FEATURE] */
struct S2013 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
	T1 volatile a3; /* after */
	T0* volatile a4; /* current_cell */
	T0* volatile a5; /* next_cursor */
};

/* Struct for type DS_LINKABLE [ET_DOTNET_FEATURE] */
struct S2014 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_HASH_TABLE [INTEGER_32, ET_LIKE_FEATURE] */
struct S2015 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* count */
	T6 volatile a4; /* last_position */
	T6 volatile a5; /* free_slot */
	T6 volatile a6; /* position */
	T6 volatile a7; /* modulus */
	T6 volatile a8; /* capacity */
	T0* volatile a9; /* equality_tester */
	T0* volatile a10; /* key_equality_tester */
	T0* volatile a11; /* internal_cursor */
	T0* volatile a12; /* key_storage */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T6 volatile a17; /* slots_position */
	T6 volatile a18; /* clashes_previous_position */
	T0* volatile a19; /* hash_function */
};

/* Struct for type SPECIAL [ET_LIKE_FEATURE] */
struct S2016 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_LIKE_FEATURE] */
struct S2017 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_LIKE_FEATURE] */
struct S2018 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type DS_HASH_TABLE_CURSOR [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME], NATURAL_8] */
struct S2020 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME]] */
struct S2021 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME] */
struct S2022 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T0* volatile a4; /* internal_keys */
	T6 volatile a5; /* position */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* count */
	T0* volatile a10; /* equality_tester */
	T6 volatile a11; /* modulus */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T6 volatile a18; /* free_slot */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type SPECIAL [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME]] */
struct S2023 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_BILINKED_LIST_CURSOR [XM_DTD_ELEMENT_CONTENT] */
struct S2024 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T1 volatile a2; /* before */
};

/* Struct for type DS_BILINKABLE [XM_DTD_ELEMENT_CONTENT] */
struct S2025 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
	T0* volatile a3; /* left */
};

/* Struct for type DS_BILINKED_LIST [DS_HASH_TABLE [STRING_8, STRING_8]] */
struct S2027 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* last_cell */
	T0* volatile a3; /* first_cell */
	T0* volatile a4; /* internal_cursor */
};

/* Struct for type DS_BILINKED_LIST_CURSOR [DS_HASH_TABLE [STRING_8, STRING_8]] */
struct S2028 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* before */
	T0* volatile a2; /* current_cell */
	T0* volatile a3; /* next_cursor */
	T0* volatile a4; /* container */
	T1 volatile a5; /* after */
};

/* Struct for type DS_LINKABLE [detachable STRING_8] */
struct S2029 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_DOTNET_ASSEMBLY] */
struct S2030 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type ET_CLUSTER_ERROR */
struct S2031 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
	T0* volatile a5; /* cluster */
};

/* Struct for type TUPLE [ET_SYSTEM, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S2032 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type TUPLE [ET_ECF_CLUSTER] */
struct S2033 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_CLUSTER]] */
struct S2034 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_TARGET, STRING_8] */
struct S2035 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [ET_ECF_ADAPTED_LIBRARY] */
struct S2037 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_ADAPTED_LIBRARY]] */
struct S2038 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2039 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
struct S2040 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type PROCEDURE [TUPLE [STRING_8]] */
struct S2043 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_OPTIONS, STRING_8] */
struct S2044 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type DS_HASH_TABLE [DS_HASH_TABLE [INTEGER_32, STRING_8], STRING_8] */
struct S2045 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T6 volatile a3; /* position */
	T0* volatile a4; /* key_equality_tester */
	T0* volatile a5; /* internal_keys */
	T0* volatile a6; /* hash_function */
	T6 volatile a7; /* last_position */
	T6 volatile a8; /* capacity */
	T6 volatile a9; /* slots_position */
	T6 volatile a10; /* count */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* special_item_routines */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T6 volatile a19; /* free_slot */
	T0* volatile a20; /* internal_cursor */
};

/* Struct for type ET_DYNAMIC_EQUALITY_EXPRESSION */
struct S2046 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target_type_set */
	T0* volatile a2; /* static_equality */
	T0* volatile a3; /* current_feature */
	T0* volatile a4; /* current_type */
	T0* volatile a5; /* next */
};

/* Struct for type ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
struct S2047 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target_type_set */
	T0* volatile a2; /* static_equality */
	T0* volatile a3; /* current_feature */
	T0* volatile a4; /* current_type */
	T0* volatile a5; /* next */
};

/* Struct for type ET_AST_PRETTY_PRINTER */
struct S2052 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comment_finder */
	T0* volatile a2; /* comment_list */
	T0* volatile a3; /* file */
	T1 volatile a4; /* bom_enabled */
	T0* volatile a5; /* buffer */
	T1 volatile a6; /* comment_printed */
	T1 volatile a7; /* use_as_type */
	T1 volatile a8; /* comments_ignored */
	T1 volatile a9; /* indentation_printed */
	T6 volatile a10; /* indentation */
	T1 volatile a11; /* use_is_keyword */
};

/* Struct for type KL_CHARACTER_ROUTINES */
struct S2054 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_SEPARATE_CALL, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
struct S2055 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
	T0* volatile z4;
};

/* Struct for type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
struct S2056 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type TUPLE [ET_C_GENERATOR, INTEGER_32, ET_CURRENT, ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
struct S2057 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T6 volatile z2;
	T0* volatile z3;
	T0* volatile z4;
	T1 volatile z5;
};

/* Struct for type TUPLE [ET_GROUP] */
struct S2065 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [PROCEDURE [TUPLE [ET_CLASS]], PREDICATE [TUPLE [ET_CLASS]]] */
struct S2066 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [ET_ECF_LIBRARY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S2067 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type ET_CONVERT_BUILTIN_EXPRESSION */
struct S2068 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* index */
	T0* volatile a2; /* expression */
	T0* volatile a3; /* convert_feature */
	T0* volatile a4; /* type */
};

/* Struct for type ET_CONVERT_FROM_EXPRESSION */
struct S2069 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* index */
	T0* volatile a3; /* type */
	T0* volatile a4; /* expression */
	T0* volatile a5; /* convert_feature */
	T0* volatile a6; /* internal_separate_target */
};

/* Struct for type ET_CONVERT_TO_EXPRESSION */
struct S2070 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* index */
	T0* volatile a3; /* expression */
	T0* volatile a4; /* convert_feature */
};

/* Struct for type DS_LINKABLE [XM_ELEMENT_NODE] */
struct S2082 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* right */
	T0* volatile a2; /* item */
};

/* Struct for type TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8] */
struct S2083 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T8 volatile z3;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_FEATURE] */
struct S2086 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T6 volatile a2; /* position */
	T0* volatile a3; /* container */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_CFLAG] */
struct S2092 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_CFLAG]] */
struct S2093 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_TARGET, ET_ECF_STATE] */
struct S2094 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_INCLUDE] */
struct S2095 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_INCLUDE]] */
struct S2096 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_LIBRARY] */
struct S2097 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_LIBRARY]] */
struct S2098 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2099 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_LINKER_FLAG]] */
struct S2100 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_MAKE] */
struct S2101 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_MAKE]] */
struct S2102 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_OBJECT] */
struct S2103 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_OBJECT]] */
struct S2104 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_EXTERNAL_RESOURCE] */
struct S2105 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_EXTERNAL_RESOURCE]] */
struct S2106 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [LX_WILDCARD] */
struct S2109 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PREDICATE [TUPLE [LX_WILDCARD]] */
struct S2110 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_CLUSTER_DEPENDENCE_CONSTRAINT, ET_GROUP, STRING_8] */
struct S2111 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T0* volatile z3;
};

/* Struct for type PREDICATE [TUPLE [STRING_8]] */
struct S2113 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [KL_STRING_ROUTINES, STRING_8] */
struct S2114 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
};

/* Struct for type DS_LINKABLE [DS_PAIR [XM_POSITION, XM_NODE]] */
struct S2116 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type SPECIAL [ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2118 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2119 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_SPARSE_TABLE_KEYS_CURSOR [ET_CLIENT, ET_CLASS] */
struct S2120 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* table_cursor */
	T0* volatile a2; /* container */
};

/* Struct for type SPECIAL [ET_ECF_ADAPTED_LIBRARY] */
struct S2122 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_ADAPTED_LIBRARY] */
struct S2123 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_ADAPTED_LIBRARY] */
struct S2124 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2125 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2126 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2127 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_FILE_RULE] */
struct S2128 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_FILE_RULE] */
struct S2129 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_FILE_RULE] */
struct S2130 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_NAMESPACE] */
struct S2131 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_NAMESPACE] */
struct S2132 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_NAMESPACE] */
struct S2133 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_CFLAG] */
struct S2134 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_CFLAG] */
struct S2135 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_CFLAG] */
struct S2136 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_INCLUDE] */
struct S2137 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_INCLUDE] */
struct S2138 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_INCLUDE] */
struct S2139 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_LIBRARY] */
struct S2140 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_LIBRARY] */
struct S2141 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_LIBRARY] */
struct S2142 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2143 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2144 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2145 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_MAKE] */
struct S2146 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_MAKE] */
struct S2147 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_MAKE] */
struct S2148 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_OBJECT] */
struct S2149 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_OBJECT] */
struct S2150 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_OBJECT] */
struct S2151 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_EXTERNAL_RESOURCE] */
struct S2152 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_EXTERNAL_RESOURCE] */
struct S2153 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_EXTERNAL_RESOURCE] */
struct S2154 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [RX_PCRE_REGULAR_EXPRESSION] */
struct S2155 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [RX_PCRE_REGULAR_EXPRESSION] */
struct S2156 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [RX_PCRE_REGULAR_EXPRESSION] */
struct S2157 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_ANDED_CONDITIONS] */
struct S2158 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_ANDED_CONDITIONS] */
struct S2159 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_ANDED_CONDITIONS] */
struct S2160 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_ECF_CONDITION_ITEM] */
struct S2161 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [ET_ECF_CONDITION_ITEM] */
struct S2162 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_ECF_CONDITION_ITEM] */
struct S2163 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type LX_WILDCARD_PARSER */
struct S2165 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* successful */
	T0* volatile a2; /* pending_rules */
	T0* volatile a3; /* start_condition_stack */
	T0* volatile a4; /* action_factory */
	T0* volatile a5; /* old_singleton_lines */
	T0* volatile a6; /* old_singleton_columns */
	T0* volatile a7; /* old_singleton_counts */
	T0* volatile a8; /* old_regexp_lines */
	T0* volatile a9; /* old_regexp_columns */
	T0* volatile a10; /* old_regexp_counts */
	T0* volatile a11; /* unions_of_concatenations_of_symbol_classes_by_utf8_character_class */
	T0* volatile a12; /* buffer */
	T6 volatile a13; /* maximum_used_symbol */
	T0* volatile a14; /* description */
	T0* volatile a15; /* last_string_value */
	T0* volatile a16; /* last_string_32_value */
	T0* volatile a17; /* last_lx_symbol_class_value */
	T0* volatile a18; /* error_handler */
	T0* volatile a19; /* name_definitions */
	T0* volatile a20; /* character_classes */
	T0* volatile a21; /* character_classes_by_name */
	T0* volatile a22; /* equiv_character_classes */
	T0* volatile a23; /* utf8_character_classes_by_name */
	T0* volatile a24; /* utf8_mode */
	T6 volatile a25; /* line_nb */
	T0* volatile a26; /* eiffel_verbatim_marker */
	T0* volatile a27; /* yyss */
	T0* volatile a28; /* input_buffer */
	T6 volatile a29; /* yy_end */
	T6 volatile a30; /* yy_position */
	T6 volatile a31; /* yy_line */
	T6 volatile a32; /* yy_column */
	T6 volatile a33; /* yy_parsing_status */
	T6 volatile a34; /* yy_suspended_yystacksize */
	T6 volatile a35; /* yy_suspended_yystate */
	T6 volatile a36; /* yy_suspended_yyn */
	T6 volatile a37; /* yy_suspended_yychar1 */
	T6 volatile a38; /* yy_suspended_index */
	T6 volatile a39; /* yy_suspended_yyss_top */
	T6 volatile a40; /* yy_suspended_yy_goto */
	T0* volatile a41; /* yyr1 */
	T6 volatile a42; /* yyssp */
	T0* volatile a43; /* yypgoto */
	T0* volatile a44; /* yycheck */
	T0* volatile a45; /* yytable */
	T0* volatile a46; /* yydefgoto */
	T6 volatile a47; /* error_count */
	T1 volatile a48; /* yy_lookahead_needed */
	T6 volatile a49; /* yyerrstatus */
	T0* volatile a50; /* yypact */
	T6 volatile a51; /* last_token */
	T0* volatile a52; /* yytranslate */
	T0* volatile a53; /* yydefact */
	T0* volatile a54; /* yy_pushed_start_conditions */
	T0* volatile a55; /* yyspecial_routines1 */
	T6 volatile a56; /* yyvsc1 */
	T0* volatile a57; /* yyvs1 */
	T0* volatile a58; /* yyspecial_routines2 */
	T6 volatile a59; /* yyvsc2 */
	T0* volatile a60; /* yyvs2 */
	T0* volatile a61; /* yyspecial_routines3 */
	T6 volatile a62; /* yyvsc3 */
	T0* volatile a63; /* yyvs3 */
	T0* volatile a64; /* yyspecial_routines4 */
	T6 volatile a65; /* yyvsc4 */
	T0* volatile a66; /* yyvs4 */
	T0* volatile a67; /* yyspecial_routines5 */
	T6 volatile a68; /* yyvsc5 */
	T0* volatile a69; /* yyvs5 */
	T0* volatile a70; /* yytypes1 */
	T0* volatile a71; /* yytypes2 */
	T6 volatile a72; /* yy_start */
	T6 volatile a73; /* yyvsp1 */
	T6 volatile a74; /* yyvsp2 */
	T6 volatile a75; /* yyvsp3 */
	T6 volatile a76; /* yyvsp4 */
	T6 volatile a77; /* yyvsp5 */
	T1 volatile a78; /* yy_more_flag */
	T6 volatile a79; /* yy_more_len */
	T6 volatile a80; /* line */
	T6 volatile a81; /* column */
	T6 volatile a82; /* position */
	T6 volatile a83; /* yy_start_state */
	T0* volatile a84; /* yy_state_stack */
	T6 volatile a85; /* yy_state_count */
	T0* volatile a86; /* yy_content_area */
	T0* volatile a87; /* yy_unicode_content_area */
	T0* volatile a88; /* yy_accept */
	T6 volatile a89; /* yy_last_accepting_state */
	T6 volatile a90; /* yy_last_accepting_cpos */
	T0* volatile a91; /* yy_content */
	T0* volatile a92; /* yy_ec */
	T0* volatile a93; /* yy_chk */
	T0* volatile a94; /* yy_base */
	T0* volatile a95; /* yy_def */
	T0* volatile a96; /* yy_meta */
	T0* volatile a97; /* yy_nxt */
	T6 volatile a98; /* yy_lp */
	T0* volatile a99; /* yy_acclist */
	T6 volatile a100; /* yy_looking_for_trail_begin */
	T6 volatile a101; /* yy_full_match */
	T6 volatile a102; /* yy_full_state */
	T6 volatile a103; /* yy_full_lp */
	T1 volatile a104; /* yy_rejected */
	T0* volatile a105; /* last_detachable_any_value */
	T6 volatile a106; /* last_integer_value */
	T1 volatile a107; /* in_trail_context */
	T0* volatile a108; /* rule */
	T1 volatile a109; /* has_trail_context */
	T6 volatile a110; /* head_count */
	T6 volatile a111; /* head_line */
	T6 volatile a112; /* head_column */
	T6 volatile a113; /* trail_count */
	T6 volatile a114; /* i_ */
	T6 volatile a115; /* rule_line_nb */
	T1 volatile a116; /* has_bol_context */
};

/* Struct for type LX_DESCRIPTION */
struct S2166 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* maximum_symbol */
	T1 volatile a2; /* utf8_mode */
	T1 volatile a3; /* equiv_classes_used */
	T1 volatile a4; /* full_table */
	T1 volatile a5; /* meta_equiv_classes_used */
	T1 volatile a6; /* reject_used */
	T1 volatile a7; /* variable_trail_context */
	T0* volatile a8; /* rules */
	T0* volatile a9; /* start_conditions */
	T6 volatile a10; /* array_size */
	T1 volatile a11; /* line_pragma */
	T1 volatile a12; /* inspect_used */
	T0* volatile a13; /* eof_rules */
	T0* volatile a14; /* eiffel_header */
	T1 volatile a15; /* case_insensitive */
	T0* volatile a16; /* equiv_classes */
	T1 volatile a17; /* bol_needed */
	T0* volatile a18; /* input_filename */
	T1 volatile a19; /* has_utf8_enconding */
	T1 volatile a20; /* actions_separated */
	T0* volatile a21; /* eiffel_code */
	T1 volatile a22; /* pre_action_used */
	T1 volatile a23; /* post_action_used */
	T1 volatile a24; /* pre_eof_action_used */
	T1 volatile a25; /* post_eof_action_used */
	T1 volatile a26; /* line_used */
	T1 volatile a27; /* position_used */
};

/* Struct for type LX_FULL_DFA */
struct S2167 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* yy_nxt */
	T0* volatile a2; /* yy_accept */
	T0* volatile a3; /* yy_ec */
	T6 volatile a4; /* yyNb_rows */
	T6 volatile a5; /* yyNull_equiv_class */
	T6 volatile a6; /* yyMax_symbol_equiv_class */
	T0* volatile a7; /* input_filename */
	T1 volatile a8; /* has_utf8_enconding */
	T6 volatile a9; /* array_size */
	T1 volatile a10; /* inspect_used */
	T1 volatile a11; /* actions_separated */
	T0* volatile a12; /* eiffel_code */
	T0* volatile a13; /* eiffel_header */
	T1 volatile a14; /* bol_needed */
	T1 volatile a15; /* pre_action_used */
	T1 volatile a16; /* post_action_used */
	T1 volatile a17; /* pre_eof_action_used */
	T1 volatile a18; /* post_eof_action_used */
	T1 volatile a19; /* line_pragma */
	T0* volatile a20; /* yy_start_conditions */
	T6 volatile a21; /* yyNb_rules */
	T0* volatile a22; /* yy_rules */
	T6 volatile a23; /* yyEnd_of_buffer */
	T1 volatile a24; /* yyLine_used */
	T1 volatile a25; /* yyPosition_used */
	T6 volatile a26; /* minimum_symbol */
	T6 volatile a27; /* maximum_symbol */
	T0* volatile a28; /* states */
	T6 volatile a29; /* backing_up_count */
	T0* volatile a30; /* partitions */
	T6 volatile a31; /* start_states_count */
	T1 volatile a32; /* yyBacking_up */
	T0* volatile a33; /* yy_eof_rules */
};

/* Struct for type SPECIAL [LX_WILDCARD] */
struct S2169 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_WILDCARD] */
struct S2172 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [LX_WILDCARD, STRING_8] */
struct S2173 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type DS_LINKABLE [CHARACTER_8] */
struct S2174 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T2 volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_LIKE_FEATURE] */
struct S2176 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type SPECIAL [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8]] */
struct S2177 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8]] */
struct S2180 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME] */
struct S2181 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_BILINKABLE [DS_HASH_TABLE [STRING_8, STRING_8]] */
struct S2182 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* left */
	T0* volatile a3; /* right */
};

/* Struct for type ARRAYED_LIST [INTEGER_32] */
struct S2183 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area_v2 */
	T6 volatile a2; /* index */
};

/* Struct for type ET_UNIVERSE_ERROR */
struct S2184 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
	T0* volatile a5; /* universe */
};

/* Struct for type ET_DOTNET_ASSEMBLY_ERROR */
struct S2185 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* default_template */
	T0* volatile a2; /* parameters */
	T0* volatile a3; /* code */
	T0* volatile a4; /* etl_code */
	T0* volatile a5; /* assembly */
};

/* Struct for type AP_OPTION_COMPARATOR */
struct S2186 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_QUICK_SORTER [AP_OPTION] */
struct S2187 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
struct S2190 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* actual_parameters */
	T6 volatile a3; /* tuple_position */
	T0* volatile a4; /* tuple_type */
};

/* Struct for type ET_ACTUAL_PARAMETER_SUBLIST */
struct S2191 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* actual_parameters */
	T6 volatile a2; /* upper */
	T6 volatile a3; /* lower */
};

/* Struct for type TUPLE [ET_ECF_FILE_RULE] */
struct S2194 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PREDICATE [TUPLE [ET_ECF_FILE_RULE]] */
struct S2195 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type PROCEDURE [TUPLE [ET_ECF_FILE_RULE]] */
struct S2197 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_ECF_FILE_RULES] */
struct S2198 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_ECF_STATE] */
struct S2199 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type SPECIAL [DS_HASH_TABLE [INTEGER_32, STRING_8]] */
struct S2202 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_HASH_TABLE [INTEGER_32, STRING_8]] */
struct S2205 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [DS_HASH_TABLE [INTEGER_32, STRING_8], STRING_8] */
struct S2206 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type ET_COMMENT_FINDER */
struct S2208 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comment_list */
	T0* volatile a2; /* excluded_nodes */
	T1 volatile a3; /* comments_ignored */
};

/* Struct for type DS_ARRAYED_LIST [ET_BREAK] */
struct S2209 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type TUPLE [ET_BREAK] */
struct S2210 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PROCEDURE [TUPLE [ET_BREAK]] */
struct S2211 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [ET_AST_PRETTY_PRINTER] */
struct S2212 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type DS_ARRAYED_LIST [LX_RULE] */
struct S2213 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type LX_START_CONDITIONS */
struct S2214 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type LX_ACTION_FACTORY */
struct S2215 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]], LX_SYMBOL_CLASS] */
struct S2216 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T6 volatile a4; /* position */
	T6 volatile a5; /* count */
	T6 volatile a6; /* capacity */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* last_position */
	T0* volatile a9; /* equality_tester */
	T6 volatile a10; /* modulus */
	T6 volatile a11; /* slots_position */
	T6 volatile a12; /* clashes_previous_position */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T0* volatile a15; /* special_item_routines */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type LX_SYMBOL_CLASS */
struct S2217 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* upper */
	T1 volatile a2; /* is_empty */
	T1 volatile a3; /* is_negated */
	T11 volatile a4; /* first_set */
	T11 volatile a5; /* second_set */
	T11 volatile a6; /* third_set */
	T11 volatile a7; /* fourth_set */
	T0* volatile a8; /* other_sets */
	T6 volatile a9; /* lower */
};

/* Struct for type DS_HASH_TABLE [STRING_32, STRING_8] */
struct S2218 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* key_equality_tester */
	T0* volatile a2; /* internal_keys */
	T0* volatile a3; /* equality_tester */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* modulus */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T0* volatile a9; /* special_item_routines */
	T0* volatile a10; /* item_storage */
	T0* volatile a11; /* special_key_routines */
	T0* volatile a12; /* key_storage */
	T0* volatile a13; /* clashes */
	T0* volatile a14; /* slots */
	T6 volatile a15; /* found_position */
	T0* volatile a16; /* internal_cursor */
};

/* Struct for type DS_HASH_SET [LX_SYMBOL_CLASS] */
struct S2219 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* equality_tester */
	T6 volatile a4; /* capacity */
	T6 volatile a5; /* modulus */
	T6 volatile a6; /* last_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* position */
	T6 volatile a9; /* count */
	T6 volatile a10; /* slots_position */
	T0* volatile a11; /* special_item_routines */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T0* volatile a14; /* internal_cursor */
	T6 volatile a15; /* clashes_previous_position */
	T0* volatile a16; /* hash_function */
};

/* Struct for type DS_HASH_TABLE [LX_SYMBOL_CLASS, STRING_32] */
struct S2220 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* found_position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T0* volatile a4; /* internal_keys */
	T6 volatile a5; /* position */
	T6 volatile a6; /* count */
	T6 volatile a7; /* capacity */
	T6 volatile a8; /* slots_position */
	T6 volatile a9; /* free_slot */
	T6 volatile a10; /* last_position */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* modulus */
	T6 volatile a13; /* clashes_previous_position */
	T0* volatile a14; /* special_item_routines */
	T0* volatile a15; /* clashes */
	T0* volatile a16; /* slots */
	T0* volatile a17; /* special_key_routines */
	T0* volatile a18; /* key_storage */
	T0* volatile a19; /* internal_cursor */
	T0* volatile a20; /* hash_function */
};

/* Struct for type KL_EQUALITY_TESTER [STRING_32] */
struct S2221 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_STACK [BOOLEAN] */
struct S2222 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* storage */
	T6 volatile a2; /* count */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
};

/* Struct for type KL_SPECIAL_ROUTINES [STRING_32] */
struct S2223 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [STRING_32] */
struct S2224 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_SYMBOL_CLASS] */
struct S2225 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [LX_SYMBOL_CLASS] */
struct S2226 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_NFA] */
struct S2227 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type LX_NFA */
struct S2228 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* in_trail_context */
	T0* volatile a2; /* states */
};

/* Struct for type SPECIAL [LX_NFA] */
struct S2229 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type LX_RULE */
struct S2230 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* id */
	T0* volatile a2; /* pattern */
	T0* volatile a3; /* action */
	T6 volatile a4; /* head_count */
	T6 volatile a5; /* trail_count */
	T6 volatile a6; /* line_count */
	T6 volatile a7; /* column_count */
	T6 volatile a8; /* line_nb */
	T1 volatile a9; /* has_trail_context */
	T1 volatile a10; /* is_useful */
};

/* Struct for type DS_HASH_SET_CURSOR [LX_SYMBOL_CLASS] */
struct S2231 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* container */
	T0* volatile a3; /* next_cursor */
};

/* Struct for type LX_EQUIVALENCE_CLASSES */
struct S2232 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* new_upper */
	T0* volatile a2; /* storage */
};

/* Struct for type LX_UNRECOGNIZED_RULE_ERROR */
struct S2233 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_INVALID_UNICODE_CHARACTER_ERROR */
struct S2234 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_MISSING_QUOTE_ERROR */
struct S2235 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_BAD_CHARACTER_CLASS_ERROR */
struct S2236 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_BAD_CHARACTER_ERROR */
struct S2237 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_FULL_AND_META_ERROR */
struct S2238 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_FULL_AND_REJECT_ERROR */
struct S2239 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
struct S2240 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type LX_CHARACTER_OUT_OF_RANGE_ERROR */
struct S2241 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type ARRAY [LX_RULE] */
struct S2242 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [LX_RULE] */
struct S2243 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type LX_DFA_STATE */
struct S2244 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* accepted_rules */
	T0* volatile a2; /* states */
	T0* volatile a3; /* transitions */
	T0* volatile a4; /* accepted_head_rules */
	T6 volatile a5; /* code */
	T6 volatile a6; /* id */
};

/* Struct for type DS_ARRAYED_LIST [LX_NFA_STATE] */
struct S2245 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* equality_tester */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
	T0* volatile a6; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [LX_DFA_STATE] */
struct S2246 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T6 volatile a2; /* capacity */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type LX_SYMBOL_PARTITIONS */
struct S2247 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* symbols */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* new_upper */
};

/* Struct for type KL_ARRAY_ROUTINES [LX_RULE] */
struct S2248 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type ARRAY [detachable LX_RULE] */
struct S2249 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [detachable LX_RULE] */
struct S2250 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type LX_START_CONDITION */
struct S2251 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* name */
	T6 volatile a2; /* id */
	T1 volatile a3; /* is_exclusive */
	T0* volatile a4; /* patterns */
	T0* volatile a5; /* bol_patterns */
};

/* Struct for type LX_TRANSITION_TABLE [LX_DFA_STATE] */
struct S2252 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* lower */
	T6 volatile a2; /* upper */
	T6 volatile a3; /* minimum_label */
	T6 volatile a4; /* second_minimum_label */
	T6 volatile a5; /* maximum_label */
	T0* volatile a6; /* transitions */
};

/* Struct for type DS_ARRAYED_LIST [LX_NFA] */
struct S2253 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T0* volatile a4; /* internal_cursor */
	T6 volatile a5; /* count */
};

/* Struct for type DS_HASH_TABLE [LX_NFA, INTEGER_32] */
struct S2254 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* item_storage */
	T0* volatile a3; /* key_equality_tester */
	T6 volatile a4; /* count */
	T6 volatile a5; /* capacity */
	T6 volatile a6; /* slots_position */
	T6 volatile a7; /* free_slot */
	T6 volatile a8; /* last_position */
	T6 volatile a9; /* modulus */
	T6 volatile a10; /* clashes_previous_position */
	T0* volatile a11; /* equality_tester */
	T6 volatile a12; /* found_position */
	T0* volatile a13; /* special_item_routines */
	T0* volatile a14; /* clashes */
	T0* volatile a15; /* slots */
	T0* volatile a16; /* special_key_routines */
	T0* volatile a17; /* key_storage */
	T0* volatile a18; /* internal_cursor */
	T0* volatile a19; /* hash_function */
};

/* Struct for type LX_NFA_STATE */
struct S2255 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T1 volatile a1; /* in_trail_context */
	T0* volatile a2; /* transition */
	T0* volatile a3; /* epsilon_transition */
	T6 volatile a4; /* id */
	T0* volatile a5; /* accepted_rule */
};

/* Struct for type DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, STRING_8] */
struct S2261 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* table_cursor */
	T0* volatile a2; /* container */
};

/* Struct for type TUPLE [RX_PCRE_REGULAR_EXPRESSION] */
struct S2262 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type PREDICATE [TUPLE [RX_PCRE_REGULAR_EXPRESSION]] */
struct S2263 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T14 volatile a1; /* rout_disp */
	T0* volatile a2; /* closed_operands */
	T14 volatile a3; /* encaps_rout_disp */
	T14 volatile a4; /* calc_rout_addr */
	T1 volatile a5; /* is_target_closed */
	T6 volatile a6; /* open_count */
};

/* Struct for type TUPLE [DS_HASH_SET [STRING_8]] */
struct S2266 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type DS_LINKABLE [ET_PARENT_FEATURE] */
struct S2268 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* item */
	T0* volatile a2; /* right */
};

/* Struct for type ARRAY [detachable LX_SYMBOL_CLASS] */
struct S2269 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [detachable LX_SYMBOL_CLASS] */
struct S2270 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type ARRAY [detachable DS_HASH_SET [INTEGER_32]] */
struct S2271 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type SPECIAL [detachable DS_HASH_SET [INTEGER_32]] */
struct S2272 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
struct S2273 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* storage */
	T0* volatile a3; /* special_routines */
	T6 volatile a4; /* capacity */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type DS_ARRAYED_LIST [LX_SYMBOL_CLASS] */
struct S2274 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* special_routines */
	T0* volatile a2; /* storage */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* count */
	T0* volatile a5; /* internal_cursor */
};

/* Struct for type LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
struct S2275 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* parameters */
};

/* Struct for type ET_AST_NULL_LEAF */
struct S2276 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* break */
	T6 volatile a2; /* compressed_position */
};

/* Struct for type DS_ARRAYED_LIST [ET_AST_NODE] */
struct S2277 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* count */
	T0* volatile a2; /* equality_tester */
	T0* volatile a3; /* storage */
	T0* volatile a4; /* special_routines */
	T6 volatile a5; /* capacity */
	T0* volatile a6; /* internal_cursor */
};

/* Struct for type SPECIAL [ET_BREAK] */
struct S2279 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_BREAK] */
struct S2280 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_BREAK] */
struct S2281 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_RULE] */
struct S2282 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_RULE] */
struct S2283 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type KL_ARRAY_ROUTINES [STRING_8] */
struct S2284 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [LX_START_CONDITION] */
struct S2285 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_START_CONDITION] */
struct S2286 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_START_CONDITION] */
struct S2287 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]]] */
struct S2288 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_EQUALITY_TESTER [LX_SYMBOL_CLASS] */
struct S2289 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]]] */
struct S2291 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_HASH_TABLE_CURSOR [DS_ARRAYED_LIST [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]], LX_SYMBOL_CLASS] */
struct S2292 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [detachable SPECIAL [NATURAL_64]] */
struct S2293 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_HASH_TABLE_CURSOR [STRING_32, STRING_8] */
struct S2295 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE_CURSOR [LX_SYMBOL_CLASS, STRING_32] */
struct S2297 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type LX_EPSILON_TRANSITION [LX_NFA_STATE] */
struct S2298 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* target */
};

/* Struct for type LX_SYMBOL_CLASS_TRANSITION [LX_NFA_STATE] */
struct S2301 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* label */
	T0* volatile a2; /* target */
};

/* Struct for type LX_SYMBOL_TRANSITION [LX_NFA_STATE] */
struct S2302 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* label */
	T0* volatile a2; /* target */
};

/* Struct for type LX_ACTION */
struct S2304 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* text */
};

/* Struct for type ARRAY [DS_BILINKABLE [INTEGER_32]] */
struct S2305 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type DS_BILINKABLE [INTEGER_32] */
struct S2306 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* item */
	T0* volatile a2; /* left */
	T0* volatile a3; /* right */
};

/* Struct for type SPECIAL [DS_BILINKABLE [INTEGER_32]] */
struct S2307 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type DS_BUBBLE_SORTER [LX_NFA_STATE] */
struct S2308 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type DS_BUBBLE_SORTER [LX_RULE] */
struct S2310 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* comparator */
};

/* Struct for type SPECIAL [LX_NFA_STATE] */
struct S2312 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_NFA_STATE] */
struct S2314 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_NFA_STATE] */
struct S2315 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* position */
	T0* volatile a2; /* next_cursor */
	T0* volatile a3; /* container */
};

/* Struct for type SPECIAL [LX_DFA_STATE] */
struct S2317 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [LX_DFA_STATE] */
struct S2318 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_DFA_STATE] */
struct S2319 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type KL_ARRAY [LX_RULE] */
struct S2320 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type DS_HASH_TABLE [LX_DFA_STATE, INTEGER_32] */
struct S2321 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* equality_tester */
	T0* volatile a2; /* key_equality_tester */
	T6 volatile a3; /* capacity */
	T6 volatile a4; /* modulus */
	T6 volatile a5; /* last_position */
	T6 volatile a6; /* free_slot */
	T6 volatile a7; /* position */
	T0* volatile a8; /* special_item_routines */
	T0* volatile a9; /* item_storage */
	T0* volatile a10; /* special_key_routines */
	T0* volatile a11; /* key_storage */
	T0* volatile a12; /* clashes */
	T0* volatile a13; /* slots */
	T6 volatile a14; /* found_position */
	T0* volatile a15; /* internal_cursor */
	T0* volatile a16; /* hash_function */
	T6 volatile a17; /* count */
	T6 volatile a18; /* slots_position */
	T6 volatile a19; /* clashes_previous_position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_NFA] */
struct S2322 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_HASH_TABLE_CURSOR [LX_NFA, INTEGER_32] */
struct S2324 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type TUPLE [ET_LIBRARY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S2325 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type TUPLE [ET_DOTNET_ASSEMBLY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S2326 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type TUPLE [ET_ECF_LIBRARY] */
struct S2327 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
};

/* Struct for type TUPLE [ET_ECF_DOTNET_ASSEMBLY, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
struct S2329 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile z1;
	T0* volatile z2;
	T6 volatile z3;
};

/* Struct for type KL_COMPARABLE_COMPARATOR [LX_RULE] */
struct S2334 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type KL_COMPARABLE_COMPARATOR [LX_NFA_STATE] */
struct S2337 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type SPECIAL [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
struct S2340 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
struct S2341 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [LX_SYMBOL_CLASS]] */
struct S2342 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [LX_SYMBOL_CLASS] */
struct S2343 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};

/* Struct for type SPECIAL [ET_AST_NODE] */
struct S2345 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T6 volatile a1; /* capacity */
	T6 volatile a2; /* count */
	T0* volatile z2[]; /* item */
};

/* Struct for type KL_SPECIAL_ROUTINES [ET_AST_NODE] */
struct S2346 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
};

/* Struct for type DS_ARRAYED_LIST_CURSOR [ET_AST_NODE] */
struct S2347 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* next_cursor */
	T0* volatile a2; /* container */
	T6 volatile a3; /* position */
};

/* Struct for type KL_ARRAY [STRING_8] */
struct S2348 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* area */
	T6 volatile a2; /* lower */
	T6 volatile a3; /* upper */
};

/* Struct for type DS_HASH_TABLE_CURSOR [LX_DFA_STATE, INTEGER_32] */
struct S2350 {
	EIF_TYPE_INDEX volatile id;
	uint16_t volatile flags;
	T0* volatile a1; /* container */
	T6 volatile a2; /* position */
};


extern T40 GE_default40;
extern T1076 GE_default1076;
extern T1447 GE_default1447;
extern T1566 GE_default1566;
extern T1567 GE_default1567;

extern T0* GE_new_str8(T6 c);
extern T0* GE_new_str32(T6 c);
extern T0* GE_new_istr8(T6 c);
extern T0* GE_new_istr32(T6 c);
extern T14 GE_sp8_base_address(T0* a1);
extern T14 GE_sp32_base_address(T0* a1);
/* New instance of type SPECIAL [CHARACTER_8] */
extern T0* GE_new15(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [CHARACTER_32] */
extern T0* GE_new16(TC* ac, T6 a1, T1 initialize);
/* New instance of type STRING_8 */
extern T0* GE_new17(TC* ac, T1 initialize);
/* New instance of type STRING_32 */
extern T0* GE_new18(TC* ac, T1 initialize);
/* New instance of type IMMUTABLE_STRING_32 */
extern T0* GE_new20(TC* ac, T1 initialize);
/* New instance of type ISE_EXCEPTION_MANAGER */
extern T0* GE_new21(TC* ac, T1 initialize);
/* New instance of type GEC */
extern T0* GE_new26(TC* ac, T1 initialize);
/* New instance of type CELL [detachable EXCEPTION] */
extern T0* GE_new27(TC* ac, T1 initialize);
/* New instance of type HASH_TABLE [INTEGER_32, INTEGER_32] */
extern T0* GE_new28(TC* ac, T1 initialize);
/* New instance of type CELL [detachable TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN]] */
extern T0* GE_new29(TC* ac, T1 initialize);
/* New instance of type CELL [NO_MORE_MEMORY] */
extern T0* GE_new30(TC* ac, T1 initialize);
/* New instance of type C_STRING */
extern T0* GE_new31(TC* ac, T1 initialize);
/* New instance of type TUPLE [INTEGER_32, INTEGER_32, INTEGER_32, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, STRING_8, INTEGER_32, BOOLEAN] */
extern T0* GE_new32(TC* ac, T1 initialize);
/* New instance of type KL_ARGUMENTS */
extern T0* GE_new33(TC* ac, T1 initialize);
/* New instance of type ARRAY [STRING_8] */
extern T0* GE_new34(TC* ac, T1 initialize);
/* New instance of type SPECIAL [STRING_8] */
extern T0* GE_new35(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_EXCEPTIONS */
extern T0* GE_new36(TC* ac, T1 initialize);
/* New instance of type EXCEPTIONS */
extern T0* GE_new37(TC* ac, T1 initialize);
/* New instance of type KL_STANDARD_FILES */
extern T0* GE_new38(TC* ac, T1 initialize);
/* New instance of type KL_STDERR_FILE */
extern T0* GE_new39(TC* ac, T1 initialize);
/* New instance of type ET_ERROR_HANDLER */
extern T0* GE_new42(TC* ac, T1 initialize);
/* New instance of type KL_TEXT_INPUT_FILE */
extern T0* GE_new43(TC* ac, T1 initialize);
/* New instance of type UT_GOBO_VARIABLES */
extern T0* GE_new44(TC* ac, T1 initialize);
/* New instance of type UT_ISE_VARIABLES */
extern T0* GE_new45(TC* ac, T1 initialize);
/* New instance of type AP_FLAG */
extern T0* GE_new46(TC* ac, T1 initialize);
/* New instance of type KL_EXECUTION_ENVIRONMENT */
extern T0* GE_new49(TC* ac, T1 initialize);
/* New instance of type ET_SYSTEM */
extern T0* GE_new50(TC* ac, T1 initialize);
/* New instance of type AP_PARSER */
extern T0* GE_new52(TC* ac, T1 initialize);
/* New instance of type AP_ALTERNATIVE_OPTIONS_LIST */
extern T0* GE_new53(TC* ac, T1 initialize);
/* New instance of type AP_STRING_OPTION */
extern T0* GE_new54(TC* ac, T1 initialize);
/* New instance of type UT_VERSION */
extern T0* GE_new56(TC* ac, T1 initialize);
/* New instance of type AP_ENUMERATION_OPTION */
extern T0* GE_new57(TC* ac, T1 initialize);
/* New instance of type AP_BOOLEAN_OPTION */
extern T0* GE_new58(TC* ac, T1 initialize);
/* New instance of type AP_INTEGER_OPTION */
extern T0* GE_new59(TC* ac, T1 initialize);
/* New instance of type ET_NULL_ERROR_HANDLER */
extern T0* GE_new63(TC* ac, T1 initialize);
/* New instance of type ET_AST_FACTORY */
extern T0* GE_new65(TC* ac, T1 initialize);
/* New instance of type ET_SYSTEM_PROCESSOR */
extern T0* GE_new66(TC* ac, T1 initialize);
/* New instance of type ET_CLUSTER */
extern T0* GE_new67(TC* ac, T1 initialize);
/* New instance of type ET_EIFFEL_PARSER */
extern T0* GE_new68(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_CLASS] */
extern T0* GE_new69(TC* ac, T1 initialize);
/* New instance of type ET_CLASS */
extern T0* GE_new70(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_CLASS] */
extern T0* GE_new74(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_CLASS]] */
extern T0* GE_new75(TC* ac, T1 initialize);
/* New instance of type TUPLE */
extern T0* GE_new76(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_ARRAYED_LIST [ET_CLASS]] */
extern T0* GE_new77(TC* ac, T1 initialize);
/* New instance of type PREDICATE [TUPLE [ET_CLASS]] */
extern T0* GE_new78(TC* ac, T1 initialize);
/* New instance of type ET_CREATOR_LIST */
extern T0* GE_new80(TC* ac, T1 initialize);
/* New instance of type ET_TOKEN_CONSTANTS */
extern T0* GE_new81(TC* ac, T1 initialize);
/* New instance of type ET_CREATOR */
extern T0* GE_new82(TC* ac, T1 initialize);
/* New instance of type UT_CANNOT_READ_FILE_ERROR */
extern T0* GE_new83(TC* ac, T1 initialize);
/* New instance of type ET_ECF_SYSTEM_PARSER */
extern T0* GE_new85(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ERROR_HANDLER */
extern T0* GE_new86(TC* ac, T1 initialize);
/* New instance of type ET_ECF_SYSTEM */
extern T0* GE_new87(TC* ac, T1 initialize);
/* New instance of type ET_ECF_TARGET */
extern T0* GE_new88(TC* ac, T1 initialize);
/* New instance of type ET_ECF_CLUSTER */
extern T0* GE_new89(TC* ac, T1 initialize);
/* New instance of type ET_ECF_SETTINGS */
extern T0* GE_new90(TC* ac, T1 initialize);
/* New instance of type ET_ECF_CAPABILITIES */
extern T0* GE_new91(TC* ac, T1 initialize);
/* New instance of type ET_ECF_VARIABLES */
extern T0* GE_new92(TC* ac, T1 initialize);
/* New instance of type ET_CLUSTERS */
extern T0* GE_new94(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_SYSTEM */
extern T0* GE_new95(TC* ac, T1 initialize);
/* New instance of type DT_DATE_TIME */
extern T0* GE_new97(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [STRING_8] */
extern T0* GE_new99(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
extern T0* GE_new100(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_TYPE */
extern T0* GE_new102(TC* ac, T1 initialize);
/* New instance of type UT_VERSION_NUMBER */
extern T0* GE_new103(TC* ac, T1 initialize);
/* New instance of type EXECUTION_ENVIRONMENT */
extern T0* GE_new105(TC* ac, T1 initialize);
/* New instance of type UT_MESSAGE */
extern T0* GE_new106(TC* ac, T1 initialize);
/* New instance of type RX_PCRE_REGULAR_EXPRESSION */
extern T0* GE_new107(TC* ac, T1 initialize);
/* New instance of type KL_STRING_ROUTINES */
extern T0* GE_new108(TC* ac, T1 initialize);
/* New instance of type ST_SPLITTER */
extern T0* GE_new109(TC* ac, T1 initialize);
/* New instance of type AP_ERROR */
extern T0* GE_new113(TC* ac, T1 initialize);
/* New instance of type ET_C_GENERATOR */
extern T0* GE_new114(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new116(TC* ac, T1 initialize);
/* New instance of type KL_SHELL_COMMAND */
extern T0* GE_new117(TC* ac, T1 initialize);
/* New instance of type GECC */
extern T0* GE_new118(TC* ac, T1 initialize);
/* New instance of type KL_OPERATING_SYSTEM */
extern T0* GE_new119(TC* ac, T1 initialize);
/* New instance of type VOID_TARGET */
extern T0* GE_new120(TC* ac, T1 initialize);
/* New instance of type ROUTINE_FAILURE */
extern T0* GE_new122(TC* ac, T1 initialize);
/* New instance of type OLD_VIOLATION */
extern T0* GE_new124(TC* ac, T1 initialize);
/* New instance of type NO_MORE_MEMORY */
extern T0* GE_new126(TC* ac, T1 initialize);
/* New instance of type INVARIANT_VIOLATION */
extern T0* GE_new127(TC* ac, T1 initialize);
/* New instance of type OPERATING_SYSTEM_SIGNAL_FAILURE */
extern T0* GE_new128(TC* ac, T1 initialize);
/* New instance of type IO_FAILURE */
extern T0* GE_new129(TC* ac, T1 initialize);
/* New instance of type OPERATING_SYSTEM_FAILURE */
extern T0* GE_new130(TC* ac, T1 initialize);
/* New instance of type COM_FAILURE */
extern T0* GE_new131(TC* ac, T1 initialize);
/* New instance of type EIFFEL_RUNTIME_PANIC */
extern T0* GE_new132(TC* ac, T1 initialize);
/* New instance of type PRECONDITION_VIOLATION */
extern T0* GE_new134(TC* ac, T1 initialize);
/* New instance of type POSTCONDITION_VIOLATION */
extern T0* GE_new135(TC* ac, T1 initialize);
/* New instance of type FLOATING_POINT_FAILURE */
extern T0* GE_new136(TC* ac, T1 initialize);
/* New instance of type CHECK_VIOLATION */
extern T0* GE_new137(TC* ac, T1 initialize);
/* New instance of type BAD_INSPECT_VALUE */
extern T0* GE_new138(TC* ac, T1 initialize);
/* New instance of type VARIANT_VIOLATION */
extern T0* GE_new139(TC* ac, T1 initialize);
/* New instance of type LOOP_INVARIANT_VIOLATION */
extern T0* GE_new140(TC* ac, T1 initialize);
/* New instance of type RESCUE_FAILURE */
extern T0* GE_new141(TC* ac, T1 initialize);
/* New instance of type RESUMPTION_FAILURE */
extern T0* GE_new142(TC* ac, T1 initialize);
/* New instance of type CREATE_ON_DEFERRED */
extern T0* GE_new143(TC* ac, T1 initialize);
/* New instance of type EXTERNAL_FAILURE */
extern T0* GE_new144(TC* ac, T1 initialize);
/* New instance of type VOID_ASSIGNED_TO_EXPANDED */
extern T0* GE_new145(TC* ac, T1 initialize);
/* New instance of type EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
extern T0* GE_new146(TC* ac, T1 initialize);
/* New instance of type MISMATCH_FAILURE */
extern T0* GE_new147(TC* ac, T1 initialize);
/* New instance of type DEVELOPER_EXCEPTION */
extern T0* GE_new148(TC* ac, T1 initialize);
/* New instance of type ADDRESS_APPLIED_TO_MELTED_FEATURE */
extern T0* GE_new149(TC* ac, T1 initialize);
/* New instance of type SERIALIZATION_FAILURE */
extern T0* GE_new150(TC* ac, T1 initialize);
/* New instance of type KL_WINDOWS_FILE_SYSTEM */
extern T0* GE_new151(TC* ac, T1 initialize);
/* New instance of type KL_UNIX_FILE_SYSTEM */
extern T0* GE_new152(TC* ac, T1 initialize);
/* New instance of type PRIMES */
extern T0* GE_new153(TC* ac, T1 initialize);
/* New instance of type SPECIAL [INTEGER_32] */
extern T0* GE_new154(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [BOOLEAN] */
extern T0* GE_new155(TC* ac, T6 a1, T1 initialize);
/* New instance of type ARGUMENTS_32 */
extern T0* GE_new156(TC* ac, T1 initialize);
/* New instance of type MUTEX */
extern T0* GE_new159(TC* ac, T1 initialize);
/* New instance of type UT_ERROR_HANDLER */
extern T0* GE_new160(TC* ac, T1 initialize);
/* New instance of type KL_STDOUT_FILE */
extern T0* GE_new161(TC* ac, T1 initialize);
/* New instance of type MANAGED_POINTER */
extern T0* GE_new162(TC* ac, T1 initialize);
/* New instance of type KL_LINKABLE [CHARACTER_8] */
extern T0* GE_new164(TC* ac, T1 initialize);
/* New instance of type FILE_INFO */
extern T0* GE_new166(TC* ac, T1 initialize);
/* New instance of type KL_DIRECTORY */
extern T0* GE_new167(TC* ac, T1 initialize);
/* New instance of type UC_UTF8_ROUTINES */
extern T0* GE_new168(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [STRING_8] */
extern T0* GE_new170(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_UNIVERSE] */
extern T0* GE_new171(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_UNIVERSE]] */
extern T0* GE_new172(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE [ET_CLASS], BOOLEAN]] */
extern T0* GE_new173(TC* ac, T1 initialize);
/* New instance of type ET_ADAPTED_LIBRARIES */
extern T0* GE_new174(TC* ac, T1 initialize);
/* New instance of type ET_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new175(TC* ac, T1 initialize);
/* New instance of type ET_IMPLICIT_TYPE_MARK */
extern T0* GE_new177(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_MASTER_CLASS, ET_CLASS_NAME] */
extern T0* GE_new178(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_NAME_TESTER */
extern T0* GE_new179(TC* ac, T1 initialize);
/* New instance of type ET_PARENT */
extern T0* GE_new181(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_LIST */
extern T0* GE_new182(TC* ac, T1 initialize);
/* New instance of type ET_CLIENT_LIST */
extern T0* GE_new183(TC* ac, T1 initialize);
/* New instance of type ET_TUPLE_TYPE */
extern T0* GE_new184(TC* ac, T1 initialize);
/* New instance of type ET_ACTUAL_PARAMETER_LIST */
extern T0* GE_new186(TC* ac, T1 initialize);
/* New instance of type ET_BUILTIN_CONVERT_FEATURE */
extern T0* GE_new187(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_UNIVERSE] */
extern T0* GE_new188(TC* ac, T1 initialize);
/* New instance of type ET_MASTER_CLASS */
extern T0* GE_new189(TC* ac, T1 initialize);
/* New instance of type ET_CLIENT */
extern T0* GE_new190(TC* ac, T1 initialize);
/* New instance of type ET_KEYWORD */
extern T0* GE_new192(TC* ac, T1 initialize);
/* New instance of type ET_ATTACHMENT_MARK_SEPARATE_KEYWORD */
extern T0* GE_new193(TC* ac, T1 initialize);
/* New instance of type ET_RENAME_LIST */
extern T0* GE_new194(TC* ac, T1 initialize);
/* New instance of type ET_EXPORT_LIST */
extern T0* GE_new195(TC* ac, T1 initialize);
/* New instance of type ET_KEYWORD_FEATURE_NAME_LIST */
extern T0* GE_new196(TC* ac, T1 initialize);
/* New instance of type ET_NONE_GROUP */
extern T0* GE_new199(TC* ac, T1 initialize);
/* New instance of type ET_LIKE_CURRENT */
extern T0* GE_new200(TC* ac, T1 initialize);
/* New instance of type ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
extern T0* GE_new202(TC* ac, T1 initialize);
/* New instance of type ET_LIBRARY */
extern T0* GE_new204(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_LIBRARY] */
extern T0* GE_new205(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_LIBRARY]] */
extern T0* GE_new206(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_HASH_SET [ET_UNIVERSE]] */
extern T0* GE_new207(TC* ac, T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLY */
extern T0* GE_new208(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new209(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new210(TC* ac, T1 initialize);
/* New instance of type ST_WORD_WRAPPER */
extern T0* GE_new211(TC* ac, T1 initialize);
/* New instance of type AP_DISPLAY_HELP_FLAG */
extern T0* GE_new213(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [AP_OPTION] */
extern T0* GE_new214(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new215(TC* ac, T1 initialize);
/* New instance of type AP_ERROR_HANDLER */
extern T0* GE_new216(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [AP_OPTION] */
extern T0* GE_new217(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [AP_OPTION] */
extern T0* GE_new218(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [detachable STRING_8] */
extern T0* GE_new219(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
extern T0* GE_new220(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [STRING_8] */
extern T0* GE_new221(TC* ac, T1 initialize);
/* New instance of type KL_STRING_EQUALITY_TESTER */
extern T0* GE_new222(TC* ac, T1 initialize);
/* New instance of type KL_EQUALITY_TESTER [STRING_8] */
extern T0* GE_new223(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [BOOLEAN] */
extern T0* GE_new226(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [INTEGER_32] */
extern T0* GE_new229(TC* ac, T1 initialize);
/* New instance of type KL_NULL_TEXT_OUTPUT_STREAM */
extern T0* GE_new230(TC* ac, T1 initialize);
/* New instance of type ET_EIFFEL_PREPARSER */
extern T0* GE_new232(TC* ac, T1 initialize);
/* New instance of type ET_MASTER_CLASS_CHECKER */
extern T0* GE_new233(TC* ac, T1 initialize);
/* New instance of type ET_PROVIDER_CHECKER */
extern T0* GE_new234(TC* ac, T1 initialize);
/* New instance of type ET_ANCESTOR_BUILDER */
extern T0* GE_new235(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_FLATTENER */
extern T0* GE_new236(TC* ac, T1 initialize);
/* New instance of type ET_INTERFACE_CHECKER */
extern T0* GE_new237(TC* ac, T1 initialize);
/* New instance of type ET_IMPLEMENTATION_CHECKER */
extern T0* GE_new238(TC* ac, T1 initialize);
/* New instance of type ET_AST_NULL_PROCESSOR */
extern T0* GE_new239(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [INTEGER_32] */
extern T0* GE_new240(TC* ac, T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
extern T0* GE_new242(TC* ac, T1 initialize);
/* New instance of type YY_UNICODE_FILE_BUFFER */
extern T0* GE_new244(TC* ac, T1 initialize);
/* New instance of type KL_STDIN_FILE */
extern T0* GE_new245(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [INTEGER_32] */
extern T0* GE_new246(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new247(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new248(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_KEYWORD] */
extern T0* GE_new249(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_SYMBOL] */
extern T0* GE_new250(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new251(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [ET_OBJECT_TEST_LIST] */
extern T0* GE_new252(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new253(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new254(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new255(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new256(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ASSERTION_ITEM] */
extern T0* GE_new257(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_QUERY] */
extern T0* GE_new258(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_PROCEDURE] */
extern T0* GE_new259(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new260(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_NAMED_CLASS] */
extern T0* GE_new261(TC* ac, T1 initialize);
/* New instance of type YY_BUFFER */
extern T0* GE_new262(TC* ac, T1 initialize);
/* New instance of type ET_UNKNOWN_GROUP */
extern T0* GE_new263(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [INTEGER_32] */
extern T0* GE_new264(TC* ac, T1 initialize);
/* New instance of type KL_UNICODE_CHARACTER_BUFFER */
extern T0* GE_new266(TC* ac, T1 initialize);
/* New instance of type ET_CLIENTS */
extern T0* GE_new268(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_CLAUSE */
extern T0* GE_new269(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ANY] */
extern T0* GE_new272(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ANY] */
extern T0* GE_new273(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
extern T0* GE_new274(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_KEYWORD] */
extern T0* GE_new275(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new276(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_KEYWORD */
extern T0* GE_new277(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new278(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INVARIANT_KEYWORD] */
extern T0* GE_new279(TC* ac, T1 initialize);
/* New instance of type ET_INVARIANT_KEYWORD */
extern T0* GE_new280(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INVARIANT_KEYWORD] */
extern T0* GE_new281(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new282(TC* ac, T1 initialize);
/* New instance of type ET_PRECURSOR_KEYWORD */
extern T0* GE_new283(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new284(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
extern T0* GE_new285(TC* ac, T1 initialize);
/* New instance of type ET_SYMBOL */
extern T0* GE_new286(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_SYMBOL] */
extern T0* GE_new287(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
extern T0* GE_new288(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_POSITION] */
extern T0* GE_new290(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new291(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new293(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
extern T0* GE_new294(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_BREAK] */
extern T0* GE_new296(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new297(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new299(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
extern T0* GE_new300(TC* ac, T1 initialize);
/* New instance of type ET_CURRENT */
extern T0* GE_new301(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CURRENT] */
extern T0* GE_new302(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
extern T0* GE_new303(TC* ac, T1 initialize);
/* New instance of type ET_FREE_OPERATOR */
extern T0* GE_new304(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FREE_OPERATOR] */
extern T0* GE_new305(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
extern T0* GE_new306(TC* ac, T1 initialize);
/* New instance of type ET_IDENTIFIER */
extern T0* GE_new307(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_IDENTIFIER] */
extern T0* GE_new308(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new309(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new311(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new312(TC* ac, T1 initialize);
/* New instance of type ET_KEYWORD_OPERATOR */
extern T0* GE_new313(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new314(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
extern T0* GE_new315(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_MANIFEST_STRING] */
extern T0* GE_new317(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
extern T0* GE_new318(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_REAL_CONSTANT] */
extern T0* GE_new320(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
extern T0* GE_new321(TC* ac, T1 initialize);
/* New instance of type ET_RESULT */
extern T0* GE_new322(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_RESULT] */
extern T0* GE_new323(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new324(TC* ac, T1 initialize);
/* New instance of type ET_RETRY_INSTRUCTION */
extern T0* GE_new325(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new326(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new327(TC* ac, T1 initialize);
/* New instance of type ET_SYMBOL_OPERATOR */
extern T0* GE_new328(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new329(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_VOID] */
extern T0* GE_new330(TC* ac, T1 initialize);
/* New instance of type ET_VOID */
extern T0* GE_new331(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_VOID] */
extern T0* GE_new332(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new333(TC* ac, T1 initialize);
/* New instance of type ET_SEMICOLON_SYMBOL */
extern T0* GE_new334(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new335(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new336(TC* ac, T1 initialize);
/* New instance of type ET_BRACKET_SYMBOL */
extern T0* GE_new337(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new338(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new339(TC* ac, T1 initialize);
/* New instance of type ET_QUESTION_MARK_SYMBOL */
extern T0* GE_new340(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new341(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new342(TC* ac, T1 initialize);
/* New instance of type ET_ACROSS_EXPRESSION */
extern T0* GE_new343(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new344(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new345(TC* ac, T1 initialize);
/* New instance of type ET_ACROSS_INSTRUCTION */
extern T0* GE_new346(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new347(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new348(TC* ac, T1 initialize);
/* New instance of type ET_ACTUAL_ARGUMENT_LIST */
extern T0* GE_new349(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new350(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new351(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new352(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new353(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new354(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new355(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new357(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new358(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new360(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new361(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_ARGUMENT_OPERAND_LIST */
extern T0* GE_new362(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new363(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
extern T0* GE_new364(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_AGENT_TARGET] */
extern T0* GE_new366(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
extern T0* GE_new367(TC* ac, T1 initialize);
/* New instance of type ET_ALIAS_NAME */
extern T0* GE_new368(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ALIAS_NAME] */
extern T0* GE_new369(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME_LIST] */
extern T0* GE_new370(TC* ac, T1 initialize);
/* New instance of type ET_ALIAS_NAME_LIST */
extern T0* GE_new371(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ALIAS_NAME_LIST] */
extern T0* GE_new372(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
extern T0* GE_new373(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ASSIGNER] */
extern T0* GE_new375(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new376(TC* ac, T1 initialize);
/* New instance of type ET_BRACKET_EXPRESSION */
extern T0* GE_new377(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new378(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
extern T0* GE_new379(TC* ac, T1 initialize);
/* New instance of type ET_CALL_AGENT */
extern T0* GE_new380(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CALL_AGENT] */
extern T0* GE_new381(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new382(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_CALL_EXPRESSION */
extern T0* GE_new383(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new384(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new385(TC* ac, T1 initialize);
/* New instance of type ET_CHECK_INSTRUCTION */
extern T0* GE_new386(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new387(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
extern T0* GE_new388(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHOICE] */
extern T0* GE_new390(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new391(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new393(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
extern T0* GE_new394(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHOICE_ITEM] */
extern T0* GE_new396(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
extern T0* GE_new397(TC* ac, T1 initialize);
/* New instance of type ET_CHOICE_LIST */
extern T0* GE_new398(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CHOICE_LIST] */
extern T0* GE_new399(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
extern T0* GE_new400(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CLASS] */
extern T0* GE_new401(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
extern T0* GE_new402(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CLIENT_ITEM] */
extern T0* GE_new403(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
extern T0* GE_new404(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CLIENTS] */
extern T0* GE_new405(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
extern T0* GE_new406(TC* ac, T1 initialize);
/* New instance of type ET_COMPOUND */
extern T0* GE_new407(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_COMPOUND] */
extern T0* GE_new408(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
extern T0* GE_new409(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONDITIONAL] */
extern T0* GE_new411(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
extern T0* GE_new412(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTANT] */
extern T0* GE_new414(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new415(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new417(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new418(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
extern T0* GE_new419(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new420(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new421(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_CREATOR */
extern T0* GE_new422(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new423(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_RENAME_LIST] */
extern T0* GE_new424(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_RENAME_LIST */
extern T0* GE_new425(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTRAINT_RENAME_LIST] */
extern T0* GE_new426(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new427(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new429(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new430(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new432(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new433(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new435(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new436(TC* ac, T1 initialize);
/* New instance of type ET_CONVERT_FEATURE_LIST */
extern T0* GE_new437(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new438(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new439(TC* ac, T1 initialize);
/* New instance of type ET_CREATE_EXPRESSION */
extern T0* GE_new440(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new441(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
extern T0* GE_new442(TC* ac, T1 initialize);
/* New instance of type ET_CREATION_REGION */
extern T0* GE_new443(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CREATION_REGION] */
extern T0* GE_new444(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
extern T0* GE_new445(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CREATOR] */
extern T0* GE_new446(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
extern T0* GE_new447(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_CREATOR_LIST] */
extern T0* GE_new448(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new449(TC* ac, T1 initialize);
/* New instance of type ET_DEBUG_INSTRUCTION */
extern T0* GE_new450(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new451(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new452(TC* ac, T1 initialize);
/* New instance of type ET_ELSEIF_EXPRESSION */
extern T0* GE_new453(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new454(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new455(TC* ac, T1 initialize);
/* New instance of type ET_ELSEIF_EXPRESSION_LIST */
extern T0* GE_new456(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new457(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
extern T0* GE_new458(TC* ac, T1 initialize);
/* New instance of type ET_ELSEIF_PART */
extern T0* GE_new459(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ELSEIF_PART] */
extern T0* GE_new460(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new461(TC* ac, T1 initialize);
/* New instance of type ET_ELSEIF_PART_LIST */
extern T0* GE_new462(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new463(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
extern T0* GE_new464(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXPORT] */
extern T0* GE_new466(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
extern T0* GE_new467(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXPORT_LIST] */
extern T0* GE_new468(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
extern T0* GE_new469(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXPRESSION] */
extern T0* GE_new471(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new472(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new474(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new475(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new477(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new478(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new480(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new481(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new482(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new483(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_CLAUSE_LIST */
extern T0* GE_new484(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new485(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new486(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_EXPORT */
extern T0* GE_new487(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new488(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
extern T0* GE_new489(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FEATURE_NAME] */
extern T0* GE_new490(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new491(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new493(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new494(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_ARGUMENT */
extern T0* GE_new495(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new496(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new497(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new499(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new500(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_ARGUMENT_LIST */
extern T0* GE_new501(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new502(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new503(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER */
extern T0* GE_new504(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new505(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new506(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new508(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new509(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_LIST */
extern T0* GE_new510(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new511(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
extern T0* GE_new512(TC* ac, T1 initialize);
/* New instance of type ET_IF_EXPRESSION */
extern T0* GE_new513(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_IF_EXPRESSION] */
extern T0* GE_new514(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new515(TC* ac, T1 initialize);
/* New instance of type ET_IF_INSTRUCTION */
extern T0* GE_new516(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new517(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
extern T0* GE_new518(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_AGENT] */
extern T0* GE_new520(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new521(TC* ac, T1 initialize);
/* New instance of type ET_INLINE_SEPARATE_ARGUMENT */
extern T0* GE_new522(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new523(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
extern T0* GE_new524(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
extern T0* GE_new526(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
extern T0* GE_new527(TC* ac, T1 initialize);
/* New instance of type ET_INLINE_SEPARATE_ARGUMENTS */
extern T0* GE_new528(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENTS] */
extern T0* GE_new529(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
extern T0* GE_new530(TC* ac, T1 initialize);
/* New instance of type ET_INLINE_SEPARATE_INSTRUCTION */
extern T0* GE_new531(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_SEPARATE_INSTRUCTION] */
extern T0* GE_new532(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INSPECT_EXPRESSION] */
extern T0* GE_new533(TC* ac, T1 initialize);
/* New instance of type ET_INSPECT_EXPRESSION */
extern T0* GE_new534(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INSPECT_EXPRESSION] */
extern T0* GE_new535(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new536(TC* ac, T1 initialize);
/* New instance of type ET_INSPECT_INSTRUCTION */
extern T0* GE_new537(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new538(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
extern T0* GE_new539(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INSTRUCTION] */
extern T0* GE_new541(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
extern T0* GE_new542(TC* ac, T1 initialize);
/* New instance of type ET_INVARIANTS */
extern T0* GE_new543(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INVARIANTS] */
extern T0* GE_new544(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new545(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new546(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
extern T0* GE_new547(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_LIKE_TYPE] */
extern T0* GE_new549(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new550(TC* ac, T1 initialize);
/* New instance of type ET_LOCAL_VARIABLE */
extern T0* GE_new551(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new552(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new553(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new555(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new556(TC* ac, T1 initialize);
/* New instance of type ET_LOCAL_VARIABLE_LIST */
extern T0* GE_new557(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new558(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new559(TC* ac, T1 initialize);
/* New instance of type ET_LOOP_INVARIANTS */
extern T0* GE_new560(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new561(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new562(TC* ac, T1 initialize);
/* New instance of type ET_MANIFEST_ARRAY */
extern T0* GE_new563(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new564(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new565(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new567(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new568(TC* ac, T1 initialize);
/* New instance of type ET_MANIFEST_STRING_LIST */
extern T0* GE_new569(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new570(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new571(TC* ac, T1 initialize);
/* New instance of type ET_MANIFEST_TUPLE */
extern T0* GE_new572(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new573(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_NOTE_LIST] */
extern T0* GE_new574(TC* ac, T1 initialize);
/* New instance of type ET_NOTE_LIST */
extern T0* GE_new575(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_NOTE_LIST] */
extern T0* GE_new576(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_NOTE_ITEM] */
extern T0* GE_new577(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_NOTE_ITEM] */
extern T0* GE_new579(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM] */
extern T0* GE_new580(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_NOTE_TERM] */
extern T0* GE_new582(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_ITEM] */
extern T0* GE_new583(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_NOTE_TERM_ITEM] */
extern T0* GE_new585(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_NOTE_TERM_LIST] */
extern T0* GE_new586(TC* ac, T1 initialize);
/* New instance of type ET_NOTE_TERM_LIST */
extern T0* GE_new587(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_NOTE_TERM_LIST] */
extern T0* GE_new588(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
extern T0* GE_new589(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_OBSOLETE] */
extern T0* GE_new591(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new592(TC* ac, T1 initialize);
/* New instance of type ET_PARENTHESIZED_EXPRESSION */
extern T0* GE_new593(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new594(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
extern T0* GE_new595(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PARENT] */
extern T0* GE_new596(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new597(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_CLAUSE_LIST */
extern T0* GE_new598(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new599(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
extern T0* GE_new600(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PARENT_ITEM] */
extern T0* GE_new601(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
extern T0* GE_new602(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PARENT_LIST] */
extern T0* GE_new603(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
extern T0* GE_new604(TC* ac, T1 initialize);
/* New instance of type ET_POSTCONDITIONS */
extern T0* GE_new605(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_POSTCONDITIONS] */
extern T0* GE_new606(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
extern T0* GE_new607(TC* ac, T1 initialize);
/* New instance of type ET_PRECONDITIONS */
extern T0* GE_new608(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PRECONDITIONS] */
extern T0* GE_new609(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
extern T0* GE_new610(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_PROCEDURE] */
extern T0* GE_new612(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new613(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new615(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_QUANTIFIER_EXPRESSION] */
extern T0* GE_new616(TC* ac, T1 initialize);
/* New instance of type ET_QUANTIFIER_EXPRESSION */
extern T0* GE_new617(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_QUANTIFIER_EXPRESSION] */
extern T0* GE_new618(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
extern T0* GE_new619(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_QUERY] */
extern T0* GE_new621(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
extern T0* GE_new622(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_RENAME_ITEM] */
extern T0* GE_new624(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
extern T0* GE_new625(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_RENAME_LIST] */
extern T0* GE_new626(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_REPEAT_INSTRUCTION] */
extern T0* GE_new627(TC* ac, T1 initialize);
/* New instance of type ET_REPEAT_INSTRUCTION */
extern T0* GE_new628(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_REPEAT_INSTRUCTION] */
extern T0* GE_new629(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new630(TC* ac, T1 initialize);
/* New instance of type ET_STATIC_CALL_EXPRESSION */
extern T0* GE_new631(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new632(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new633(TC* ac, T1 initialize);
/* New instance of type ET_STRIP_EXPRESSION */
extern T0* GE_new634(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new635(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
extern T0* GE_new636(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE] */
extern T0* GE_new637(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT] */
extern T0* GE_new638(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE_CONSTRAINT] */
extern T0* GE_new640(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new641(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new643(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE_CONSTRAINT_LIST] */
extern T0* GE_new644(TC* ac, T1 initialize);
/* New instance of type ET_TYPE_CONSTRAINT_LIST */
extern T0* GE_new645(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE_CONSTRAINT_LIST] */
extern T0* GE_new646(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
extern T0* GE_new647(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE_ITEM] */
extern T0* GE_new649(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
extern T0* GE_new650(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_TYPE_LIST] */
extern T0* GE_new652(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
extern T0* GE_new653(TC* ac, T1 initialize);
/* New instance of type ET_VARIANT */
extern T0* GE_new654(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_VARIANT] */
extern T0* GE_new655(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION] */
extern T0* GE_new656(TC* ac, T1 initialize);
/* New instance of type ET_WHEN_EXPRESSION */
extern T0* GE_new657(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_WHEN_EXPRESSION] */
extern T0* GE_new658(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_WHEN_EXPRESSION_LIST] */
extern T0* GE_new659(TC* ac, T1 initialize);
/* New instance of type ET_WHEN_EXPRESSION_LIST */
extern T0* GE_new660(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_WHEN_EXPRESSION_LIST] */
extern T0* GE_new661(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
extern T0* GE_new662(TC* ac, T1 initialize);
/* New instance of type ET_WHEN_PART */
extern T0* GE_new663(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_WHEN_PART] */
extern T0* GE_new664(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new665(TC* ac, T1 initialize);
/* New instance of type ET_WHEN_PART_LIST */
extern T0* GE_new666(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new667(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
extern T0* GE_new668(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_WRITABLE] */
extern T0* GE_new670(TC* ac, T6 a1, T1 initialize);
/* New instance of type ET_OBJECT_TEST_LIST */
extern T0* GE_new672(TC* ac, T1 initialize);
/* New instance of type ET_ITERATION_COMPONENT_LIST */
extern T0* GE_new673(TC* ac, T1 initialize);
/* New instance of type ET_INLINE_SEPARATE_ARGUMENT_LIST */
extern T0* GE_new674(TC* ac, T1 initialize);
/* New instance of type ET_C3_CHARACTER_CONSTANT */
extern T0* GE_new675(TC* ac, T1 initialize);
/* New instance of type ET_REGULAR_MANIFEST_STRING */
extern T0* GE_new676(TC* ac, T1 initialize);
/* New instance of type ET_SPECIAL_MANIFEST_STRING */
extern T0* GE_new677(TC* ac, T1 initialize);
/* New instance of type ET_VERBATIM_STRING */
extern T0* GE_new678(TC* ac, T1 initialize);
/* New instance of type ET_REGULAR_INTEGER_CONSTANT */
extern T0* GE_new679(TC* ac, T1 initialize);
/* New instance of type ET_UNDERSCORED_INTEGER_CONSTANT */
extern T0* GE_new680(TC* ac, T1 initialize);
/* New instance of type ET_HEXADECIMAL_INTEGER_CONSTANT */
extern T0* GE_new681(TC* ac, T1 initialize);
/* New instance of type ET_OCTAL_INTEGER_CONSTANT */
extern T0* GE_new682(TC* ac, T1 initialize);
/* New instance of type ET_BINARY_INTEGER_CONSTANT */
extern T0* GE_new683(TC* ac, T1 initialize);
/* New instance of type ET_REGULAR_REAL_CONSTANT */
extern T0* GE_new684(TC* ac, T1 initialize);
/* New instance of type ET_UNDERSCORED_REAL_CONSTANT */
extern T0* GE_new685(TC* ac, T1 initialize);
/* New instance of type ET_COMMENT */
extern T0* GE_new686(TC* ac, T1 initialize);
/* New instance of type ET_TRUE_CONSTANT */
extern T0* GE_new687(TC* ac, T1 initialize);
/* New instance of type ET_FALSE_CONSTANT */
extern T0* GE_new688(TC* ac, T1 initialize);
/* New instance of type ET_C1_CHARACTER_CONSTANT */
extern T0* GE_new689(TC* ac, T1 initialize);
/* New instance of type ET_C2_CHARACTER_CONSTANT */
extern T0* GE_new690(TC* ac, T1 initialize);
/* New instance of type ET_TAGGED_NOTE */
extern T0* GE_new692(TC* ac, T1 initialize);
/* New instance of type ET_NOTE */
extern T0* GE_new693(TC* ac, T1 initialize);
/* New instance of type ET_CUSTOM_ATTRIBUTE */
extern T0* GE_new694(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINED_FORMAL_PARAMETER */
extern T0* GE_new695(TC* ac, T1 initialize);
/* New instance of type ET_TYPE_RENAME_CONSTRAINT */
extern T0* GE_new696(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_NAMED_TYPE */
extern T0* GE_new697(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
extern T0* GE_new698(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
extern T0* GE_new699(TC* ac, T1 initialize);
/* New instance of type ET_RENAME */
extern T0* GE_new700(TC* ac, T1 initialize);
/* New instance of type ET_ALL_EXPORT */
extern T0* GE_new701(TC* ac, T1 initialize);
/* New instance of type ET_CONVERT_FUNCTION */
extern T0* GE_new703(TC* ac, T1 initialize);
/* New instance of type ET_CONVERT_PROCEDURE */
extern T0* GE_new704(TC* ac, T1 initialize);
/* New instance of type ET_ATTRIBUTE */
extern T0* GE_new706(TC* ac, T1 initialize);
/* New instance of type ET_EXTENDED_ATTRIBUTE */
extern T0* GE_new707(TC* ac, T1 initialize);
/* New instance of type ET_CONSTANT_ATTRIBUTE */
extern T0* GE_new708(TC* ac, T1 initialize);
/* New instance of type ET_UNIQUE_ATTRIBUTE */
extern T0* GE_new709(TC* ac, T1 initialize);
/* New instance of type ET_DO_FUNCTION */
extern T0* GE_new710(TC* ac, T1 initialize);
/* New instance of type ET_ONCE_FUNCTION */
extern T0* GE_new711(TC* ac, T1 initialize);
/* New instance of type ET_DEFERRED_FUNCTION */
extern T0* GE_new712(TC* ac, T1 initialize);
/* New instance of type ET_EXTERNAL_FUNCTION */
extern T0* GE_new714(TC* ac, T1 initialize);
/* New instance of type ET_DO_PROCEDURE */
extern T0* GE_new715(TC* ac, T1 initialize);
/* New instance of type ET_ONCE_PROCEDURE */
extern T0* GE_new716(TC* ac, T1 initialize);
/* New instance of type ET_DEFERRED_PROCEDURE */
extern T0* GE_new717(TC* ac, T1 initialize);
/* New instance of type ET_EXTERNAL_PROCEDURE */
extern T0* GE_new718(TC* ac, T1 initialize);
/* New instance of type ET_ALIASED_FEATURE_NAME */
extern T0* GE_new719(TC* ac, T1 initialize);
/* New instance of type ET_ALIAS_FREE_NAME */
extern T0* GE_new720(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_ASSERTION */
extern T0* GE_new723(TC* ac, T1 initialize);
/* New instance of type ET_LABELED_ACTUAL_PARAMETER */
extern T0* GE_new724(TC* ac, T1 initialize);
/* New instance of type ET_LIKE_FEATURE */
extern T0* GE_new726(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_LIKE_BRACED_TYPE */
extern T0* GE_new728(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_LIKE_TYPE */
extern T0* GE_new729(TC* ac, T1 initialize);
/* New instance of type ET_ASSIGNER_INSTRUCTION */
extern T0* GE_new730(TC* ac, T1 initialize);
/* New instance of type ET_ASSIGNMENT */
extern T0* GE_new731(TC* ac, T1 initialize);
/* New instance of type ET_ASSIGNMENT_ATTEMPT */
extern T0* GE_new732(TC* ac, T1 initialize);
/* New instance of type ET_LOOP_INSTRUCTION */
extern T0* GE_new733(TC* ac, T1 initialize);
/* New instance of type ET_BANG_INSTRUCTION */
extern T0* GE_new735(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_CALL */
extern T0* GE_new736(TC* ac, T1 initialize);
/* New instance of type ET_CREATE_INSTRUCTION */
extern T0* GE_new738(TC* ac, T1 initialize);
/* New instance of type ET_CHOICE_RANGE */
extern T0* GE_new739(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_CALL_INSTRUCTION */
extern T0* GE_new740(TC* ac, T1 initialize);
/* New instance of type ET_PRECURSOR_INSTRUCTION */
extern T0* GE_new741(TC* ac, T1 initialize);
/* New instance of type ET_STATIC_CALL_INSTRUCTION */
extern T0* GE_new743(TC* ac, T1 initialize);
/* New instance of type ET_PRECURSOR_EXPRESSION */
extern T0* GE_new744(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_ADDRESS */
extern T0* GE_new745(TC* ac, T1 initialize);
/* New instance of type ET_CURRENT_ADDRESS */
extern T0* GE_new746(TC* ac, T1 initialize);
/* New instance of type ET_RESULT_ADDRESS */
extern T0* GE_new747(TC* ac, T1 initialize);
/* New instance of type ET_EXPRESSION_ADDRESS */
extern T0* GE_new748(TC* ac, T1 initialize);
/* New instance of type ET_INFIX_EXPRESSION */
extern T0* GE_new749(TC* ac, T1 initialize);
/* New instance of type ET_INFIX_AND_THEN_OPERATOR */
extern T0* GE_new750(TC* ac, T1 initialize);
/* New instance of type ET_INFIX_OR_ELSE_OPERATOR */
extern T0* GE_new751(TC* ac, T1 initialize);
/* New instance of type ET_EQUALITY_EXPRESSION */
extern T0* GE_new752(TC* ac, T1 initialize);
/* New instance of type ET_OBJECT_EQUALITY_EXPRESSION */
extern T0* GE_new753(TC* ac, T1 initialize);
/* New instance of type ET_MANIFEST_TYPE */
extern T0* GE_new754(TC* ac, T1 initialize);
/* New instance of type ET_PREFIX_EXPRESSION */
extern T0* GE_new755(TC* ac, T1 initialize);
/* New instance of type ET_OLD_EXPRESSION */
extern T0* GE_new756(TC* ac, T1 initialize);
/* New instance of type ET_OLD_OBJECT_TEST */
extern T0* GE_new757(TC* ac, T1 initialize);
/* New instance of type ET_OBJECT_TEST */
extern T0* GE_new758(TC* ac, T1 initialize);
/* New instance of type ET_NAMED_OBJECT_TEST */
extern T0* GE_new759(TC* ac, T1 initialize);
/* New instance of type ET_ONCE_MANIFEST_STRING */
extern T0* GE_new760(TC* ac, T1 initialize);
/* New instance of type ET_ITERATION_CURSOR */
extern T0* GE_new761(TC* ac, T1 initialize);
/* New instance of type ET_DO_FUNCTION_INLINE_AGENT */
extern T0* GE_new763(TC* ac, T1 initialize);
/* New instance of type ET_ONCE_FUNCTION_INLINE_AGENT */
extern T0* GE_new764(TC* ac, T1 initialize);
/* New instance of type ET_EXTERNAL_FUNCTION_INLINE_AGENT */
extern T0* GE_new765(TC* ac, T1 initialize);
/* New instance of type ET_DO_PROCEDURE_INLINE_AGENT */
extern T0* GE_new766(TC* ac, T1 initialize);
/* New instance of type ET_ONCE_PROCEDURE_INLINE_AGENT */
extern T0* GE_new767(TC* ac, T1 initialize);
/* New instance of type ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
extern T0* GE_new768(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_OPEN_TARGET */
extern T0* GE_new769(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_TYPED_OPEN_ARGUMENT */
extern T0* GE_new770(TC* ac, T1 initialize);
/* New instance of type ET_QUERY_LIST */
extern T0* GE_new773(TC* ac, T1 initialize);
/* New instance of type ET_PROCEDURE_LIST */
extern T0* GE_new774(TC* ac, T1 initialize);
/* New instance of type ET_TAGGED_ASSERTION */
extern T0* GE_new776(TC* ac, T1 initialize);
/* New instance of type ET_FILE_POSITION */
extern T0* GE_new778(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLASS] */
extern T0* GE_new779(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLASS] */
extern T0* GE_new780(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_CLASS] */
extern T0* GE_new781(TC* ac, T1 initialize);
/* New instance of type ET_BASE_TYPE_LIST */
extern T0* GE_new782(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_CODES */
extern T0* GE_new783(TC* ac, T1 initialize);
/* New instance of type ET_ECF_AST_FACTORY */
extern T0* GE_new784(TC* ac, T1 initialize);
/* New instance of type DS_CELL [detachable ET_ECF_SYSTEM] */
extern T0* GE_new785(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_TARGET] */
extern T0* GE_new786(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_ECF_TARGET]] */
extern T0* GE_new787(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM_PARSER] */
extern T0* GE_new788(TC* ac, T1 initialize);
/* New instance of type XM_ELEMENT */
extern T0* GE_new789(TC* ac, T1 initialize);
/* New instance of type XM_POSITION_TABLE */
extern T0* GE_new790(TC* ac, T1 initialize);
/* New instance of type TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8] */
extern T0* GE_new791(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [XM_ELEMENT, detachable XM_POSITION_TABLE, STRING_8]] */
extern T0* GE_new792(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, PROCEDURE [TUPLE [ET_ECF_TARGET]], DS_CELL [detachable ET_ECF_SYSTEM]] */
extern T0* GE_new793(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ECF_LIBRARY, STRING_8] */
extern T0* GE_new795(TC* ac, T1 initialize);
/* New instance of type KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
extern T0* GE_new796(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
extern T0* GE_new797(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_PARSER */
extern T0* GE_new799(TC* ac, T1 initialize);
/* New instance of type XM_TREE_CALLBACKS_PIPE */
extern T0* GE_new800(TC* ac, T1 initialize);
/* New instance of type XM_CALLBACKS_TO_TREE_FILTER */
extern T0* GE_new803(TC* ac, T1 initialize);
/* New instance of type ET_ECF_STATE */
extern T0* GE_new804(TC* ac, T1 initialize);
/* New instance of type ET_ECF_SYSTEM_CONFIG */
extern T0* GE_new805(TC* ac, T1 initialize);
/* New instance of type ET_ECF_LIBRARY */
extern T0* GE_new806(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ADAPTED_LIBRARY] */
extern T0* GE_new808(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_ADAPTED_LIBRARY]] */
extern T0* GE_new809(TC* ac, T1 initialize);
/* New instance of type XM_DOCUMENT */
extern T0* GE_new810(TC* ac, T1 initialize);
/* New instance of type ET_COMPRESSED_POSITION */
extern T0* GE_new811(TC* ac, T1 initialize);
/* New instance of type XM_STOP_ON_ERROR_FILTER */
extern T0* GE_new813(TC* ac, T1 initialize);
/* New instance of type XM_ATTRIBUTE */
extern T0* GE_new814(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ECF_SYSTEM_CONFIG, STRING_8] */
extern T0* GE_new815(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ECF_TARGET, STRING_8] */
extern T0* GE_new816(TC* ac, T1 initialize);
/* New instance of type ET_ECF_TARGET_PARENT */
extern T0* GE_new817(TC* ac, T1 initialize);
/* New instance of type DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new818(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
extern T0* GE_new819(TC* ac, T1 initialize);
/* New instance of type ET_ECF_TARGETS */
extern T0* GE_new820(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, STRING_8] */
extern T0* GE_new821(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, STRING_8] */
extern T0* GE_new822(TC* ac, T1 initialize);
/* New instance of type DS_CELL [detachable ET_ECF_LIBRARY] */
extern T0* GE_new824(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ADAPTED_LIBRARY */
extern T0* GE_new825(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, DS_CELL [detachable ET_ECF_LIBRARY]] */
extern T0* GE_new826(TC* ac, T1 initialize);
/* New instance of type ET_ECF_DOTNET_ASSEMBLY */
extern T0* GE_new828(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
extern T0* GE_new830(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [STRING_8, STRING_8] */
extern T0* GE_new831(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ERROR */
extern T0* GE_new832(TC* ac, T1 initialize);
/* New instance of type ET_ECF_OPTIONS */
extern T0* GE_new833(TC* ac, T1 initialize);
/* New instance of type ET_ECF_CLUSTERS */
extern T0* GE_new834(TC* ac, T1 initialize);
/* New instance of type KL_AGENT_HASH_FUNCTION [STRING_8] */
extern T0* GE_new835(TC* ac, T1 initialize);
/* New instance of type TUPLE [STRING_8] */
extern T0* GE_new836(TC* ac, T1 initialize);
/* New instance of type FUNCTION [TUPLE [STRING_8], INTEGER_32] */
extern T0* GE_new837(TC* ac, T1 initialize);
/* New instance of type TUPLE [KL_STRING_ROUTINES] */
extern T0* GE_new838(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_IDENTIFIER, STRING_8] */
extern T0* GE_new840(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_CLUSTER] */
extern T0* GE_new842(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PRIMARY_TYPE */
extern T0* GE_new843(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_FEATURE_LIST */
extern T0* GE_new844(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
extern T0* GE_new845(TC* ac, T1 initialize);
/* New instance of type TUPLE [INTEGER_32] */
extern T0* GE_new846(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
extern T0* GE_new847(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_DYNAMIC_SYSTEM] */
extern T0* GE_new848(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_FEATURE */
extern T0* GE_new850(TC* ac, T1 initialize);
/* New instance of type ET_NESTED_TYPE_CONTEXT */
extern T0* GE_new852(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_TYPE */
extern T0* GE_new853(TC* ac, T1 initialize);
/* New instance of type ARRAY [ET_TYPE] */
extern T0* GE_new854(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_TYPE] */
extern T0* GE_new855(TC* ac, T6 a1, T1 initialize);
/* New instance of type ET_SYSTEM_MULTIPROCESSOR */
extern T0* GE_new858(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [STRING_8] */
extern T0* GE_new859(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [STRING_8] */
extern T0* GE_new860(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_CHECKER */
extern T0* GE_new862(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_TYPE_SET_LIST */
extern T0* GE_new863(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_FEATURE] */
extern T0* GE_new864(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PRECURSOR */
extern T0* GE_new865(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PRECURSOR_LIST */
extern T0* GE_new866(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
extern T0* GE_new867(TC* ac, T1 initialize);
/* New instance of type ET_TYPE_CHECKER */
extern T0* GE_new869(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_INLINE_AGENT] */
extern T0* GE_new872(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_PROCEDURE]] */
extern T0* GE_new873(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_QUERY]] */
extern T0* GE_new874(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_FEATURE]] */
extern T0* GE_new875(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new876(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new877(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
extern T0* GE_new878(TC* ac, T1 initialize);
/* New instance of type ET_OBJECT_TEST_SCOPE */
extern T0* GE_new879(TC* ac, T1 initialize);
/* New instance of type ET_OBJECT_TEST_SCOPE_BUILDER */
extern T0* GE_new880(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
extern T0* GE_new881(TC* ac, T1 initialize);
/* New instance of type ET_ITERATION_ITEM_SCOPE */
extern T0* GE_new882(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new883(TC* ac, T1 initialize);
/* New instance of type ET_INLINE_SEPARATE_ARGUMENT_SCOPE */
extern T0* GE_new884(TC* ac, T1 initialize);
/* New instance of type ET_ATTACHMENT_SCOPE */
extern T0* GE_new885(TC* ac, T1 initialize);
/* New instance of type ET_ATTACHMENT_SCOPE_BUILDER */
extern T0* GE_new886(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ATTACHMENT_SCOPE] */
extern T0* GE_new887(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ASSERTIONS, ET_FEATURE] */
extern T0* GE_new888(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_NOTE_TERM] */
extern T0* GE_new889(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
extern T0* GE_new890(TC* ac, T1 initialize);
/* New instance of type ET_ADAPTED_BASE_CLASS_CHECKER */
extern T0* GE_new891(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_QUALIFIED_QUERY_CALL */
extern T0* GE_new893(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
extern T0* GE_new894(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_2 [ET_PRECONDITIONS, INTEGER_32] */
extern T0* GE_new896(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_2 [ET_POSTCONDITIONS, INTEGER_32] */
extern T0* GE_new897(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_2 [ET_INVARIANTS, INTEGER_32] */
extern T0* GE_new898(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_TUPLE_TYPE */
extern T0* GE_new904(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_LIST */
extern T0* GE_new907(TC* ac, T1 initialize);
/* New instance of type NATIVE_STRING */
extern T0* GE_new909(TC* ac, T1 initialize);
/* New instance of type HASH_TABLE [NATIVE_STRING, IMMUTABLE_STRING_32] */
extern T0* GE_new910(TC* ac, T1 initialize);
/* New instance of type RX_BYTE_CODE */
extern T0* GE_new912(TC* ac, T1 initialize);
/* New instance of type RX_CHARACTER_SET */
extern T0* GE_new913(TC* ac, T1 initialize);
/* New instance of type RX_CASE_MAPPING */
extern T0* GE_new914(TC* ac, T1 initialize);
/* New instance of type UC_UNICODE_ROUTINES */
extern T0* GE_new916(TC* ac, T1 initialize);
/* New instance of type ARRAY [RX_CHARACTER_SET] */
extern T0* GE_new917(TC* ac, T1 initialize);
/* New instance of type SPECIAL [RX_CHARACTER_SET] */
extern T0* GE_new918(TC* ac, T6 a1, T1 initialize);
/* New instance of type UC_STRING */
extern T0* GE_new919(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [STRING_8] */
extern T0* GE_new920(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [NATURAL_32] */
extern T0* GE_new921(TC* ac, T1 initialize);
/* New instance of type KL_STRING_OUTPUT_STREAM */
extern T0* GE_new922(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_OLD_EXPRESSION] */
extern T0* GE_new923(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
extern T0* GE_new924(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_IDENTIFIER] */
extern T0* GE_new925(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new926(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [BOOLEAN] */
extern T0* GE_new927(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_EQUALITY_EXPRESSION] */
extern T0* GE_new928(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_EQUALITY_TYPES] */
extern T0* GE_new929(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
extern T0* GE_new930(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new931(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new932(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new933(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_RESULT] */
extern T0* GE_new934(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PRIMARY_TYPE_HASH_LIST */
extern T0* GE_new935(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_STANDALONE_TYPE_SET */
extern T0* GE_new936(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new937(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [ET_EXPRESSION] */
extern T0* GE_new938(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_EXPRESSION] */
extern T0* GE_new939(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
extern T0* GE_new940(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
extern T0* GE_new941(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new942(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new943(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_2 [ET_OBJECT_TEST, INTEGER_32] */
extern T0* GE_new944(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, INTEGER_32] */
extern T0* GE_new945(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [INTEGER_32] */
extern T0* GE_new946(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_2 [ET_AGENT, INTEGER_32] */
extern T0* GE_new947(TC* ac, T1 initialize);
/* New instance of type ET_GENERAL_QUALIFIED_FEATURE_CALL_INSTRUCTION */
extern T0* GE_new948(TC* ac, T1 initialize);
/* New instance of type ET_GENERAL_QUALIFIED_FEATURE_CALL_EXPRESSION */
extern T0* GE_new950(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new951(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, ET_STANDALONE_CLOSURE] */
extern T0* GE_new952(TC* ac, T1 initialize);
/* New instance of type ARRAY [INTEGER_32] */
extern T0* GE_new953(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_CONSTANT, ET_FEATURE] */
extern T0* GE_new954(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
extern T0* GE_new955(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new956(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_INLINED_EXPRESSION] */
extern T0* GE_new957(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_CALL_CONTEXT] */
extern T0* GE_new958(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DYNAMIC_FEATURE] */
extern T0* GE_new959(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_DYNAMIC_FEATURE] */
extern T0* GE_new960(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [BOOLEAN, STRING_8] */
extern T0* GE_new961(TC* ac, T1 initialize);
/* New instance of type DS_STRING_HASH_TABLE */
extern T0* GE_new962(TC* ac, T1 initialize);
/* New instance of type KL_TEXT_OUTPUT_FILE */
extern T0* GE_new963(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [STRING_8, STRING_8] */
extern T0* GE_new964(TC* ac, T1 initialize);
/* New instance of type UT_UNDEFINED_ENVIRONMENT_VARIABLE_ERROR */
extern T0* GE_new965(TC* ac, T1 initialize);
/* New instance of type UT_CANNOT_WRITE_TO_FILE_ERROR */
extern T0* GE_new966(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [DS_ARRAYED_LIST [STRING_8]] */
extern T0* GE_new967(TC* ac, T1 initialize);
/* New instance of type DS_QUICK_SORTER [STRING_8] */
extern T0* GE_new968(TC* ac, T1 initialize);
/* New instance of type UC_STRING_COMPARATOR */
extern T0* GE_new969(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
extern T0* GE_new970(TC* ac, T1 initialize);
/* New instance of type TUPLE [STRING_8, STRING_8] */
extern T0* GE_new973(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TOPOLOGICAL_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new974(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_EQUALITY_TYPES */
extern T0* GE_new975(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_OBJECT_EQUALITY_TYPES */
extern T0* GE_new977(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_SPECIAL_TYPE */
extern T0* GE_new978(TC* ac, T1 initialize);
/* New instance of type DS_QUICK_SORTER [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new980(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_SECONDARY_TYPE */
extern T0* GE_new981(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PRIMARY_TYPE_COMPARATOR_BY_ID */
extern T0* GE_new982(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new988(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_INLINED_EXPRESSION */
extern T0* GE_new992(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_CALL_CONTEXT */
extern T0* GE_new993(TC* ac, T1 initialize);
/* New instance of type DS_QUICK_SORTER [INTEGER_32] */
extern T0* GE_new994(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE] */
extern T0* GE_new996(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new997(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new998(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new999(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new1000(TC* ac, T1 initialize);
/* New instance of type UT_INTEGER_FORMATTER */
extern T0* GE_new1001(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new1006(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1007(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new1011(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_EXPRESSION, ET_DYNAMIC_PRIMARY_TYPE, BOOLEAN] */
extern T0* GE_new1016(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, INTEGER_32, INTEGER_32] */
extern T0* GE_new1017(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1018(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1019(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1020(TC* ac, T1 initialize);
/* New instance of type TUPLE [GECC, DS_ARRAYED_LIST [STRING_8], INTEGER_32, MUTEX] */
extern T0* GE_new1023(TC* ac, T1 initialize);
/* New instance of type WORKER_THREAD */
extern T0* GE_new1024(TC* ac, T1 initialize);
/* New instance of type PATH */
extern T0* GE_new1026(TC* ac, T1 initialize);
/* New instance of type STRING_TO_INTEGER_CONVERTOR */
extern T0* GE_new1027(TC* ac, T1 initialize);
/* New instance of type STRING_8_SEARCHER */
extern T0* GE_new1029(TC* ac, T1 initialize);
/* New instance of type STD_FILES */
extern T0* GE_new1031(TC* ac, T1 initialize);
/* New instance of type HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
extern T0* GE_new1032(TC* ac, T1 initialize);
/* New instance of type DT_SHARED_SYSTEM_CLOCK */
extern T0* GE_new1034(TC* ac, T1 initialize);
/* New instance of type DT_SYSTEM_CLOCK */
extern T0* GE_new1035(TC* ac, T1 initialize);
/* New instance of type DT_DATE_TIME_DURATION */
extern T0* GE_new1036(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, STRING_8] */
extern T0* GE_new1041(TC* ac, T1 initialize);
/* New instance of type ET_CONSTRAINT_GENERIC_NAMED_TYPE */
extern T0* GE_new1044(TC* ac, T1 initialize);
/* New instance of type ET_UNQUALIFIED_CALL_EXPRESSION */
extern T0* GE_new1045(TC* ac, T1 initialize);
/* New instance of type ET_UNQUALIFIED_CALL_INSTRUCTION */
extern T0* GE_new1046(TC* ac, T1 initialize);
/* New instance of type KL_ARRAY_ROUTINES [INTEGER_32] */
extern T0* GE_new1047(TC* ac, T1 initialize);
/* New instance of type DS_CELL [detachable ET_ECF_CONFIG] */
extern T0* GE_new1050(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_SYSTEM, DS_CELL [detachable ET_ECF_CONFIG]] */
extern T0* GE_new1052(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new1054(TC* ac, T1 initialize);
/* New instance of type XM_NAMESPACE */
extern T0* GE_new1055(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [XM_ELEMENT_NODE] */
extern T0* GE_new1056(TC* ac, T1 initialize);
/* New instance of type ET_ECF_REDIRECTION_CONFIG */
extern T0* GE_new1058(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PROCEDURE_TYPE */
extern T0* GE_new1059(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_FUNCTION_TYPE */
extern T0* GE_new1060(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_PUSH_TYPE_SET */
extern T0* GE_new1061(TC* ac, T1 initialize);
/* New instance of type KL_PLATFORM */
extern T0* GE_new1062(TC* ac, T1 initialize);
/* New instance of type UT_CONFIG_PARSER */
extern T0* GE_new1063(TC* ac, T1 initialize);
/* New instance of type KL_COMPARABLE_COMPARATOR [INTEGER_32] */
extern T0* GE_new1064(TC* ac, T1 initialize);
/* New instance of type CELL [INTEGER_32] */
extern T0* GE_new1067(TC* ac, T1 initialize);
/* New instance of type KL_PATHNAME */
extern T0* GE_new1069(TC* ac, T1 initialize);
/* New instance of type KL_ANY_ROUTINES */
extern T0* GE_new1070(TC* ac, T1 initialize);
/* New instance of type KL_WINDOWS_INPUT_FILE */
extern T0* GE_new1071(TC* ac, T1 initialize);
/* New instance of type KL_UNIX_INPUT_FILE */
extern T0* GE_new1072(TC* ac, T1 initialize);
/* New instance of type ARRAY [IMMUTABLE_STRING_32] */
extern T0* GE_new1073(TC* ac, T1 initialize);
/* New instance of type SPECIAL [IMMUTABLE_STRING_32] */
extern T0* GE_new1074(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [NATURAL_8] */
extern T0* GE_new1075(TC* ac, T6 a1, T1 initialize);
/* New instance of type UC_UTF8_STRING */
extern T0* GE_new1078(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [STRING_8] */
extern T0* GE_new1079(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ADAPTED_LIBRARY] */
extern T0* GE_new1080(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1081(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_MASTER_CLASS] */
extern T0* GE_new1082(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_MASTER_CLASS] */
extern T0* GE_new1085(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLASS_NAME] */
extern T0* GE_new1086(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLASS_NAME] */
extern T0* GE_new1087(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_MASTER_CLASS, ET_CLASS_NAME] */
extern T0* GE_new1088(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_PARENT_ITEM] */
extern T0* GE_new1090(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_PARENT_ITEM] */
extern T0* GE_new1091(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLIENT_ITEM] */
extern T0* GE_new1092(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLIENT_ITEM] */
extern T0* GE_new1093(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1094(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1096(TC* ac, T1 initialize);
/* New instance of type ET_BRACED_TYPE_LIST */
extern T0* GE_new1097(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_UNIVERSE] */
extern T0* GE_new1098(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_UNIVERSE] */
extern T0* GE_new1100(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_UNIVERSE] */
extern T0* GE_new1101(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_MASTER_CLASS] */
extern T0* GE_new1102(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_MASTER_CLASS] */
extern T0* GE_new1103(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_MASTER_CLASS]] */
extern T0* GE_new1104(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [AP_OPTION] */
extern T0* GE_new1105(TC* ac, T1 initialize);
/* New instance of type SPECIAL [AP_OPTION] */
extern T0* GE_new1107(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [AP_OPTION] */
extern T0* GE_new1108(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new1109(TC* ac, T1 initialize);
/* New instance of type SPECIAL [AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new1110(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new1111(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable STRING_8] */
extern T0* GE_new1112(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable STRING_8] */
extern T0* GE_new1113(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [STRING_8] */
extern T0* GE_new1114(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [BOOLEAN] */
extern T0* GE_new1115(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [BOOLEAN] */
extern T0* GE_new1116(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [INTEGER_32] */
extern T0* GE_new1117(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [INTEGER_32] */
extern T0* GE_new1118(TC* ac, T1 initialize);
/* New instance of type YY_FILE_BUFFER */
extern T0* GE_new1120(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TOPOLOGICAL_SORTER [ET_CLASS] */
extern T0* GE_new1121(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_BASE_TYPE, ET_CLASS] */
extern T0* GE_new1122(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_CHECKER1 */
extern T0* GE_new1123(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_CHECKER1 */
extern T0* GE_new1124(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
extern T0* GE_new1125(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_NAME_TESTER */
extern T0* GE_new1126(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
extern T0* GE_new1128(TC* ac, T1 initialize);
/* New instance of type ET_ALIAS_NAME_TESTER */
extern T0* GE_new1129(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_CLIENT_LIST] */
extern T0* GE_new1131(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_CLIENT, ET_CLASS] */
extern T0* GE_new1132(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1133(TC* ac, T1 initialize);
/* New instance of type ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1134(TC* ac, T1 initialize);
/* New instance of type ET_IDENTIFIER_TYPE_RESOLVER */
extern T0* GE_new1135(TC* ac, T1 initialize);
/* New instance of type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
extern T0* GE_new1136(TC* ac, T1 initialize);
/* New instance of type ET_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1137(TC* ac, T1 initialize);
/* New instance of type ET_SIGNATURE_CHECKER */
extern T0* GE_new1138(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_CHECKER2 */
extern T0* GE_new1139(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_CHECKER2 */
extern T0* GE_new1140(TC* ac, T1 initialize);
/* New instance of type ET_BUILTIN_FEATURE_CHECKER */
extern T0* GE_new1141(TC* ac, T1 initialize);
/* New instance of type ET_PRECURSOR_CHECKER */
extern T0* GE_new1142(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_FEATURE, INTEGER_32] */
extern T0* GE_new1143(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_CLASS] */
extern T0* GE_new1144(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1145(TC* ac, T1 initialize);
/* New instance of type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
extern T0* GE_new1146(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_RENAME, ET_FEATURE_NAME] */
extern T0* GE_new1147(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_RENAME, ET_ALIAS_NAME] */
extern T0* GE_new1148(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_CHECKER3 */
extern T0* GE_new1149(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_PROCEDURE] */
extern T0* GE_new1150(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_QUERY] */
extern T0* GE_new1151(TC* ac, T1 initialize);
/* New instance of type ET_SUPPLIER_BUILDER */
extern T0* GE_new1152(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [INTEGER_32] */
extern T0* GE_new1153(TC* ac, T1 initialize);
/* New instance of type YY_UNICODE_BUFFER */
extern T0* GE_new1155(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new1156(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new1157(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_OBJECT_TEST_LIST] */
extern T0* GE_new1158(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_OBJECT_TEST_LIST] */
extern T0* GE_new1159(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1160(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1161(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1162(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT_LIST] */
extern T0* GE_new1163(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new1164(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new1165(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new1166(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_LIST] */
extern T0* GE_new1167(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ASSERTION_ITEM] */
extern T0* GE_new1168(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ASSERTION_ITEM] */
extern T0* GE_new1169(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ASSERTION_ITEM] */
extern T0* GE_new1170(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_QUERY] */
extern T0* GE_new1171(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_QUERY] */
extern T0* GE_new1172(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_QUERY] */
extern T0* GE_new1173(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_PROCEDURE] */
extern T0* GE_new1174(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_PROCEDURE] */
extern T0* GE_new1175(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_PROCEDURE] */
extern T0* GE_new1176(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1177(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_NAMED_CLASS] */
extern T0* GE_new1178(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NAMED_CLASS] */
extern T0* GE_new1179(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NAMED_CLASS] */
extern T0* GE_new1180(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_EXPRESSION_ITEM] */
extern T0* GE_new1182(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_EXPRESSION_ITEM] */
extern T0* GE_new1184(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INSTRUCTION] */
extern T0* GE_new1186(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INSTRUCTION] */
extern T0* GE_new1187(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
extern T0* GE_new1189(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT_ITEM] */
extern T0* GE_new1190(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1191(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1192(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ITERATION_COMPONENT] */
extern T0* GE_new1193(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ITERATION_COMPONENT] */
extern T0* GE_new1194(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new1195(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new1196(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_BASE_TYPE] */
extern T0* GE_new1197(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_BASE_TYPE] */
extern T0* GE_new1198(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [NATURAL_8, ET_CLASS_NAME] */
extern T0* GE_new1199(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ECF_LIBRARY, STRING_8] */
extern T0* GE_new1200(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_LIBRARY] */
extern T0* GE_new1201(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_LIBRARY] */
extern T0* GE_new1204(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new1205(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ECF_DOTNET_ASSEMBLY, STRING_8] */
extern T0* GE_new1208(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new1209(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_SCANNER */
extern T0* GE_new1210(TC* ac, T1 initialize);
/* New instance of type XM_DEFAULT_POSITION */
extern T0* GE_new1211(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST [XM_POSITION] */
extern T0* GE_new1213(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_STACK [XM_EIFFEL_SCANNER] */
extern T0* GE_new1214(TC* ac, T1 initialize);
/* New instance of type XM_CALLBACKS_NULL */
extern T0* GE_new1215(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [XM_EIFFEL_ENTITY_DEF, STRING_8] */
extern T0* GE_new1216(TC* ac, T1 initialize);
/* New instance of type XM_NULL_EXTERNAL_RESOLVER */
extern T0* GE_new1218(TC* ac, T1 initialize);
/* New instance of type XM_DTD_CALLBACKS_NULL */
extern T0* GE_new1220(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1221(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_PARSER_NAME */
extern T0* GE_new1222(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1223(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new1224(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1225(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_HASH_SET [XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new1226(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_DTD_EXTERNAL_ID] */
extern T0* GE_new1227(TC* ac, T1 initialize);
/* New instance of type XM_DTD_EXTERNAL_ID */
extern T0* GE_new1228(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_DTD_EXTERNAL_ID] */
extern T0* GE_new1229(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1230(TC* ac, T1 initialize);
/* New instance of type XM_DTD_ELEMENT_CONTENT */
extern T0* GE_new1231(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1232(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new1233(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1234(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_BILINKED_LIST [XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new1235(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1236(TC* ac, T1 initialize);
/* New instance of type XM_DTD_ATTRIBUTE_CONTENT */
extern T0* GE_new1237(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1238(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_BILINKED_LIST [STRING_8]] */
extern T0* GE_new1239(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST [STRING_8] */
extern T0* GE_new1240(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_BILINKED_LIST [STRING_8]] */
extern T0* GE_new1241(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [BOOLEAN] */
extern T0* GE_new1242(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_EIFFEL_DECLARATION] */
extern T0* GE_new1243(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_DECLARATION */
extern T0* GE_new1244(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_EIFFEL_DECLARATION] */
extern T0* GE_new1245(TC* ac, T6 a1, T1 initialize);
/* New instance of type XM_EIFFEL_ENTITY_DEF */
extern T0* GE_new1248(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_SCANNER_DTD */
extern T0* GE_new1249(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_PE_ENTITY_DEF */
extern T0* GE_new1251(TC* ac, T1 initialize);
/* New instance of type XM_NAMESPACE_RESOLVER */
extern T0* GE_new1252(TC* ac, T1 initialize);
/* New instance of type ARRAY [XM_CALLBACKS_FILTER] */
extern T0* GE_new1253(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_CALLBACKS_FILTER] */
extern T0* GE_new1254(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_SET [XM_NAMESPACE] */
extern T0* GE_new1255(TC* ac, T1 initialize);
/* New instance of type XM_LINKED_LIST [XM_DOCUMENT_NODE] */
extern T0* GE_new1256(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new1259(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new1261(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ECF_SYSTEM_CONFIG, STRING_8] */
extern T0* GE_new1262(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ECF_TARGET, STRING_8] */
extern T0* GE_new1263(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_TARGET] */
extern T0* GE_new1264(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_TARGET] */
extern T0* GE_new1266(TC* ac, T1 initialize);
/* New instance of type DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, STRING_8] */
extern T0* GE_new1268(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new1269(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new1270(TC* ac, T6 a1, T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLIES */
extern T0* GE_new1271(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_IDENTIFIER] */
extern T0* GE_new1275(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_IDENTIFIER] */
extern T0* GE_new1276(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, STRING_8] */
extern T0* GE_new1277(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLUSTER] */
extern T0* GE_new1278(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLUSTER] */
extern T0* GE_new1279(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_CLUSTER] */
extern T0* GE_new1280(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_FEATURE, INTEGER_32] */
extern T0* GE_new1281(TC* ac, T1 initialize);
/* New instance of type ET_FEATURE_IDS */
extern T0* GE_new1282(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_FEATURE] */
extern T0* GE_new1283(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_FEATURE] */
extern T0* GE_new1284(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, STRING_8] */
extern T0* GE_new1286(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
extern T0* GE_new1287(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
extern T0* GE_new1288(TC* ac, T1 initialize);
/* New instance of type ET_LIKE_N */
extern T0* GE_new1290(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_TYPE] */
extern T0* GE_new1291(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1292(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_SYSTEM_PROCESSOR, DS_ARRAYED_LIST [ET_CLASS]] */
extern T0* GE_new1293(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new1294(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new1295(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_FEATURE] */
extern T0* GE_new1296(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FEATURE] */
extern T0* GE_new1297(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FEATURE] */
extern T0* GE_new1298(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ADAPTED_CLASS] */
extern T0* GE_new1300(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INLINE_AGENT] */
extern T0* GE_new1301(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INLINE_AGENT] */
extern T0* GE_new1302(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_INLINE_AGENT] */
extern T0* GE_new1303(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_PROCEDURE]] */
extern T0* GE_new1304(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_ARRAYED_LIST [ET_PROCEDURE]] */
extern T0* GE_new1305(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_PROCEDURE]] */
extern T0* GE_new1306(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_QUERY]] */
extern T0* GE_new1307(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_ARRAYED_LIST [ET_QUERY]] */
extern T0* GE_new1308(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_QUERY]] */
extern T0* GE_new1309(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_FEATURE]] */
extern T0* GE_new1310(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_FEATURE] */
extern T0* GE_new1311(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_ARRAYED_LIST [ET_FEATURE]] */
extern T0* GE_new1312(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_FEATURE]] */
extern T0* GE_new1313(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new1314(TC* ac, T1 initialize);
/* New instance of type TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1315(TC* ac, T1 initialize);
/* New instance of type SPECIAL [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new1316(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [TUPLE [detachable ET_QUERY, ET_CLASS, ET_NESTED_TYPE_CONTEXT]] */
extern T0* GE_new1317(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1318(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1319(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1320(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1321(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1324(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_ITERATION_COMPONENT] */
extern T0* GE_new1325(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ITERATION_COMPONENT] */
extern T0* GE_new1327(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_NESTED_TYPE_CONTEXT, ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new1329(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new1330(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1332(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1333(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1334(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ASSERTIONS, ET_FEATURE] */
extern T0* GE_new1335(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ASSERTIONS] */
extern T0* GE_new1336(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ASSERTIONS] */
extern T0* GE_new1338(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NOTE_TERM] */
extern T0* GE_new1339(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NOTE_TERM] */
extern T0* GE_new1340(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_NOTE_TERM] */
extern T0* GE_new1341(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
extern T0* GE_new1342(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
extern T0* GE_new1343(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [ET_ADAPTED_CLASS]] */
extern T0* GE_new1344(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_PRECONDITIONS] */
extern T0* GE_new1345(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_PRECONDITIONS] */
extern T0* GE_new1346(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_POSTCONDITIONS] */
extern T0* GE_new1347(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_POSTCONDITIONS] */
extern T0* GE_new1348(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INVARIANTS] */
extern T0* GE_new1349(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INVARIANTS] */
extern T0* GE_new1350(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [NATIVE_STRING] */
extern T0* GE_new1351(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [NATURAL_32] */
extern T0* GE_new1352(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [RX_CHARACTER_SET] */
extern T0* GE_new1353(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [NATURAL_32] */
extern T0* GE_new1354(TC* ac, T1 initialize);
/* New instance of type SPECIAL [NATURAL_64] */
extern T0* GE_new1355(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE [NATURAL_64, NATURAL_32] */
extern T0* GE_new1356(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [NATURAL_64, NATURAL_32] */
extern T0* GE_new1357(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ARRAY [INTEGER_32]] */
extern T0* GE_new1358(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [SPECIAL [ARRAY [INTEGER_32]]] */
extern T0* GE_new1359(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_INTEGER_ROUTINES */
extern T0* GE_new1360(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [NATURAL_32] */
extern T0* GE_new1363(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_OLD_EXPRESSION] */
extern T0* GE_new1365(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_OLD_EXPRESSION] */
extern T0* GE_new1366(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_OLD_EXPRESSION] */
extern T0* GE_new1367(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_IDENTIFIER, ET_OLD_EXPRESSION] */
extern T0* GE_new1369(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_IDENTIFIER] */
extern T0* GE_new1370(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1371(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1372(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1373(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [BOOLEAN] */
extern T0* GE_new1374(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1375(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1376(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1377(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_EQUALITY_TYPES] */
extern T0* GE_new1378(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_EQUALITY_TYPES] */
extern T0* GE_new1379(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_EQUALITY_TYPES] */
extern T0* GE_new1380(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
extern T0* GE_new1381(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
extern T0* GE_new1382(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_OBJECT_EQUALITY_TYPES] */
extern T0* GE_new1383(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1384(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1385(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_UNQUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1386(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1387(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1388(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1389(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new1390(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new1391(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new1392(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_RESULT] */
extern T0* GE_new1393(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_RESULT] */
extern T0* GE_new1394(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_RESULT] */
extern T0* GE_new1395(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new1396(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new1397(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new1398(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_EXPRESSION] */
extern T0* GE_new1399(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_EXPRESSION] */
extern T0* GE_new1400(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_EXPRESSION] */
extern T0* GE_new1401(TC* ac, T1 initialize);
/* New instance of type KL_EQUALITY_TESTER [INTEGER_32] */
extern T0* GE_new1402(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, INTEGER_32] */
extern T0* GE_new1403(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_STANDALONE_TYPE_SET, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1404(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_OBJECT_TEST] */
extern T0* GE_new1406(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_OBJECT_TEST] */
extern T0* GE_new1407(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, INTEGER_32] */
extern T0* GE_new1409(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [INTEGER_32] */
extern T0* GE_new1410(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_AGENT] */
extern T0* GE_new1413(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_AGENT] */
extern T0* GE_new1414(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1415(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1416(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1417(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_STANDALONE_CLOSURE] */
extern T0* GE_new1419(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_STANDALONE_CLOSURE] */
extern T0* GE_new1420(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_STANDALONE_CLOSURE] */
extern T0* GE_new1421(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_CONSTANT, ET_FEATURE] */
extern T0* GE_new1422(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CONSTANT] */
extern T0* GE_new1423(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CONSTANT] */
extern T0* GE_new1425(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_PRIMARY_TYPE, ET_INLINE_CONSTANT] */
extern T0* GE_new1426(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INLINE_CONSTANT] */
extern T0* GE_new1427(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INLINE_CONSTANT] */
extern T0* GE_new1429(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new1430(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1432(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new1433(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_INLINED_EXPRESSION] */
extern T0* GE_new1434(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_INLINED_EXPRESSION] */
extern T0* GE_new1435(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_INLINED_EXPRESSION] */
extern T0* GE_new1436(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_CALL_CONTEXT] */
extern T0* GE_new1437(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_CALL_CONTEXT] */
extern T0* GE_new1438(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_CALL_CONTEXT] */
extern T0* GE_new1439(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_DYNAMIC_FEATURE] */
extern T0* GE_new1440(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_DYNAMIC_FEATURE] */
extern T0* GE_new1441(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [BOOLEAN, STRING_8] */
extern T0* GE_new1444(TC* ac, T1 initialize);
/* New instance of type SPECIAL [DS_ARRAYED_LIST [STRING_8]] */
extern T0* GE_new1448(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [DS_ARRAYED_LIST [STRING_8]] */
extern T0* GE_new1449(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [DS_ARRAYED_LIST [STRING_8]] */
extern T0* GE_new1450(TC* ac, T1 initialize);
/* New instance of type SPECIAL [TUPLE [STRING_8, detachable STRING_8]] */
extern T0* GE_new1451(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [TUPLE [STRING_8, detachable STRING_8]] */
extern T0* GE_new1453(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [TUPLE [STRING_8, detachable STRING_8], STRING_8] */
extern T0* GE_new1454(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1455(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [detachable DS_LINKABLE [INTEGER_32]] */
extern T0* GE_new1456(TC* ac, T1 initialize);
/* New instance of type ARRAY [BOOLEAN] */
extern T0* GE_new1457(TC* ac, T1 initialize);
/* New instance of type THREAD_ATTRIBUTES */
extern T0* GE_new1459(TC* ac, T1 initialize);
/* New instance of type CELL [BOOLEAN] */
extern T0* GE_new1460(TC* ac, T1 initialize);
/* New instance of type CHARACTER_PROPERTY */
extern T0* GE_new1462(TC* ac, T1 initialize);
/* New instance of type ET_SYSTEM_ERROR */
extern T0* GE_new1463(TC* ac, T1 initialize);
/* New instance of type ET_INTERNAL_ERROR */
extern T0* GE_new1464(TC* ac, T1 initialize);
/* New instance of type ET_SYNTAX_ERROR */
extern T0* GE_new1465(TC* ac, T1 initialize);
/* New instance of type ET_VALIDITY_ERROR */
extern T0* GE_new1466(TC* ac, T1 initialize);
/* New instance of type ET_FORMAL_COMMA_ARGUMENT */
extern T0* GE_new1467(TC* ac, T1 initialize);
/* New instance of type ET_LOCAL_COMMA_VARIABLE */
extern T0* GE_new1468(TC* ac, T1 initialize);
/* New instance of type ET_LABELED_COMMA_ACTUAL_PARAMETER */
extern T0* GE_new1469(TC* ac, T1 initialize);
/* New instance of type ET_KEYWORD_EXPRESSION */
extern T0* GE_new1471(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1474(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CREATOR] */
extern T0* GE_new1475(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CREATOR] */
extern T0* GE_new1476(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1477(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1478(TC* ac, T6 a1, T1 initialize);
/* New instance of type ET_ECF_NOTE_ELEMENT */
extern T0* GE_new1479(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ADAPTED_LIBRARIES */
extern T0* GE_new1480(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new1481(TC* ac, T1 initialize);
/* New instance of type ET_ECF_FILE_RULES */
extern T0* GE_new1482(TC* ac, T1 initialize);
/* New instance of type ET_ECF_NAMESPACES */
extern T0* GE_new1483(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_CFLAGS */
extern T0* GE_new1484(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_INCLUDES */
extern T0* GE_new1485(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LIBRARIES */
extern T0* GE_new1486(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LINKER_FLAGS */
extern T0* GE_new1487(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_MAKES */
extern T0* GE_new1488(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_OBJECTS */
extern T0* GE_new1489(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_RESOURCES */
extern T0* GE_new1490(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_ACTION] */
extern T0* GE_new1491(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
extern T0* GE_new1492(TC* ac, T1 initialize);
/* New instance of type ET_ECF_VERSION */
extern T0* GE_new1494(TC* ac, T1 initialize);
/* New instance of type ET_ECF_NAMESPACE */
extern T0* GE_new1495(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_CFLAG */
extern T0* GE_new1496(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_INCLUDE */
extern T0* GE_new1497(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LIBRARY */
extern T0* GE_new1498(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LINKER_FLAG */
extern T0* GE_new1499(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_MAKE */
extern T0* GE_new1500(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_OBJECT */
extern T0* GE_new1501(TC* ac, T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_RESOURCE */
extern T0* GE_new1502(TC* ac, T1 initialize);
/* New instance of type ET_ECF_FILE_RULE */
extern T0* GE_new1503(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ACTION */
extern T0* GE_new1504(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_ECF_OPTIONS, STRING_8] */
extern T0* GE_new1505(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1506(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_OPTIONS] */
extern T0* GE_new1507(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_ECF_OPTIONS]] */
extern T0* GE_new1508(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ROOT_CLASS */
extern T0* GE_new1509(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ROOT_ALL_CLASSES */
extern T0* GE_new1510(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ORED_CONDITIONS */
extern T0* GE_new1511(TC* ac, T1 initialize);
/* New instance of type ET_ECF_ANDED_CONDITIONS */
extern T0* GE_new1512(TC* ac, T1 initialize);
/* New instance of type ET_ECF_VISIBLE_CLASS */
extern T0* GE_new1513(TC* ac, T1 initialize);
/* New instance of type ET_ECF_BUILD_CONDITION */
extern T0* GE_new1515(TC* ac, T1 initialize);
/* New instance of type ET_ECF_CONCURRENCY_CONDITION */
extern T0* GE_new1516(TC* ac, T1 initialize);
/* New instance of type ET_ECF_VOID_SAFETY_CONDITION */
extern T0* GE_new1517(TC* ac, T1 initialize);
/* New instance of type ET_ECF_CUSTOM_CONDITION */
extern T0* GE_new1518(TC* ac, T1 initialize);
/* New instance of type ET_ECF_DOTNET_CONDITION */
extern T0* GE_new1519(TC* ac, T1 initialize);
/* New instance of type ET_ECF_DYNAMIC_RUNTIME_CONDITION */
extern T0* GE_new1520(TC* ac, T1 initialize);
/* New instance of type ET_ECF_PLATFORM_CONDITION */
extern T0* GE_new1521(TC* ac, T1 initialize);
/* New instance of type ET_ECF_COMPILER_VERSION_CONDITION */
extern T0* GE_new1523(TC* ac, T1 initialize);
/* New instance of type ET_ECF_MSIL_CLR_VERSION_CONDITION */
extern T0* GE_new1524(TC* ac, T1 initialize);
/* New instance of type UT_COUNTER */
extern T0* GE_new1526(TC* ac, T1 initialize);
/* New instance of type KL_AGENT_ROUTINES [ET_CLASS] */
extern T0* GE_new1527(TC* ac, T1 initialize);
/* New instance of type TUPLE [UT_COUNTER] */
extern T0* GE_new1528(TC* ac, T1 initialize);
/* New instance of type TUPLE [KL_AGENT_ROUTINES [ET_CLASS], PROCEDURE [TUPLE]] */
extern T0* GE_new1529(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_CLASS]]] */
extern T0* GE_new1530(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
extern T0* GE_new1531(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1533(TC* ac, T1 initialize);
/* New instance of type PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new1534(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_HASH_SET [ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new1535(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1536(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new1537(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_DOTNET_ASSEMBLY]], PREDICATE [TUPLE [ET_DOTNET_ASSEMBLY]]] */
extern T0* GE_new1538(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_DOTNET_ASSEMBLIES] */
extern T0* GE_new1539(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1540(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_CLUSTER] */
extern T0* GE_new1541(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_CLUSTER]] */
extern T0* GE_new1542(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM] */
extern T0* GE_new1543(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1544(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_AST_PROCESSOR] */
extern T0* GE_new1545(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1546(TC* ac, T1 initialize);
/* New instance of type PROCEDURE [TUPLE [ET_ADAPTED_DOTNET_ASSEMBLY]] */
extern T0* GE_new1547(TC* ac, T1 initialize);
/* New instance of type ET_ANCESTORS_STATUS_CHECKER */
extern T0* GE_new1548(TC* ac, T1 initialize);
/* New instance of type ET_FLATTENING_STATUS_CHECKER */
extern T0* GE_new1549(TC* ac, T1 initialize);
/* New instance of type ET_INTERFACE_STATUS_CHECKER */
extern T0* GE_new1550(TC* ac, T1 initialize);
/* New instance of type ET_IMPLEMENTATION_STATUS_CHECKER */
extern T0* GE_new1551(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
extern T0* GE_new1552(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_FLATTENING_STATUS_CHECKER] */
extern T0* GE_new1553(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_INTERFACE_STATUS_CHECKER] */
extern T0* GE_new1554(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
extern T0* GE_new1555(TC* ac, T1 initialize);
/* New instance of type TUPLE [BOOLEAN] */
extern T0* GE_new1556(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_CLUSTER]]] */
extern T0* GE_new1559(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_HASH_SET [ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new1563(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]]] */
extern T0* GE_new1564(TC* ac, T1 initialize);
/* New instance of type ET_PARENTHESIS_EXPRESSION */
extern T0* GE_new1565(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_LIBRARY] */
extern T0* GE_new1568(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_HASH_SET [ET_LIBRARY]] */
extern T0* GE_new1569(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_RENAME_ITEM] */
extern T0* GE_new1573(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_RENAME_ITEM] */
extern T0* GE_new1574(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_EXPORT] */
extern T0* GE_new1576(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_EXPORT] */
extern T0* GE_new1577(TC* ac, T1 initialize);
/* New instance of type ET_CLUSTER_DEPENDENCE_CONSTRAINT */
extern T0* GE_new1580(TC* ac, T1 initialize);
/* New instance of type ET_PARENT_FEATURE */
extern T0* GE_new1582(TC* ac, T1 initialize);
/* New instance of type DS_QUICK_SORTER [ET_QUERY] */
extern T0* GE_new1583(TC* ac, T1 initialize);
/* New instance of type DS_QUICK_SORTER [ET_PROCEDURE] */
extern T0* GE_new1584(TC* ac, T1 initialize);
/* New instance of type ET_INHERITED_FEATURE */
extern T0* GE_new1585(TC* ac, T1 initialize);
/* New instance of type ET_REDECLARED_FEATURE */
extern T0* GE_new1587(TC* ac, T1 initialize);
/* New instance of type KL_CHARACTER_BUFFER */
extern T0* GE_new1595(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1597(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new1598(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ALIAS_NAME] */
extern T0* GE_new1599(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ALIAS_NAME] */
extern T0* GE_new1600(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_IMPLICIT_CURRENT_TARGET */
extern T0* GE_new1601(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CHOICE_ITEM] */
extern T0* GE_new1603(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CHOICE_ITEM] */
extern T0* GE_new1604(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1605(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1607(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1608(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1610(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1611(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1612(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ELSEIF_PART] */
extern T0* GE_new1613(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ELSEIF_PART] */
extern T0* GE_new1614(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FEATURE_CLAUSE] */
extern T0* GE_new1615(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FEATURE_CLAUSE] */
extern T0* GE_new1616(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1617(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1619(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1620(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1621(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1622(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1624(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1626(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1627(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NOTE_ITEM] */
extern T0* GE_new1629(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NOTE_ITEM] */
extern T0* GE_new1630(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_NOTE_TERM_ITEM] */
extern T0* GE_new1631(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_NOTE_TERM_ITEM] */
extern T0* GE_new1632(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_PARENT_LIST] */
extern T0* GE_new1633(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_PARENT_LIST] */
extern T0* GE_new1634(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1636(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_TYPE_CONSTRAINT_ITEM] */
extern T0* GE_new1637(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_WHEN_EXPRESSION] */
extern T0* GE_new1638(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_WHEN_EXPRESSION] */
extern T0* GE_new1639(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_WHEN_PART] */
extern T0* GE_new1640(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_WHEN_PART] */
extern T0* GE_new1641(TC* ac, T1 initialize);
/* New instance of type XM_LINKED_LIST [XM_ELEMENT_NODE] */
extern T0* GE_new1643(TC* ac, T1 initialize);
/* New instance of type UC_STRING_EQUALITY_TESTER */
extern T0* GE_new1645(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [DS_PAIR [XM_POSITION, XM_NODE]] */
extern T0* GE_new1647(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [DS_PAIR [XM_POSITION, XM_NODE]] */
extern T0* GE_new1648(TC* ac, T1 initialize);
/* New instance of type DS_PAIR [XM_POSITION, XM_NODE] */
extern T0* GE_new1649(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_INPUT_STREAM */
extern T0* GE_new1650(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [XM_DOCUMENT_NODE] */
extern T0* GE_new1651(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_TARGET] */
extern T0* GE_new1652(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_ECF_OPTIONS, STRING_8] */
extern T0* GE_new1653(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1654(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_TARGET_PARENT] */
extern T0* GE_new1656(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_CLUSTER] */
extern T0* GE_new1657(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_TYPE_SET, ET_DYNAMIC_TYPE] */
extern T0* GE_new1659(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TYPE] */
extern T0* GE_new1662(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_TYPE] */
extern T0* GE_new1663(TC* ac, T6 a1, T1 initialize);
/* New instance of type INTEGER_OVERFLOW_CHECKER */
extern T0* GE_new1668(TC* ac, T1 initialize);
/* New instance of type CONSOLE */
extern T0* GE_new1669(TC* ac, T1 initialize);
/* New instance of type C_DATE */
extern T0* GE_new1670(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, STRING_8] */
extern T0* GE_new1672(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new1673(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new1674(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new1675(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_TARGET_LIST */
extern T0* GE_new1677(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_STACK [YY_BUFFER] */
extern T0* GE_new1678(TC* ac, T1 initialize);
/* New instance of type UT_SYNTAX_ERROR */
extern T0* GE_new1679(TC* ac, T1 initialize);
/* New instance of type UT_TOO_MANY_INCLUDES_ERROR */
extern T0* GE_new1680(TC* ac, T1 initialize);
/* New instance of type ARRAY [detachable STRING_8] */
extern T0* GE_new1681(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ADAPTED_LIBRARY] */
extern T0* GE_new1683(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ADAPTED_LIBRARY] */
extern T0* GE_new1684(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_LIBRARY] */
extern T0* GE_new1685(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1686(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1687(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1688(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_TYPE_ITEM] */
extern T0* GE_new1689(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_TYPE_ITEM] */
extern T0* GE_new1690(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, ET_CLASS] */
extern T0* GE_new1691(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_BASE_TYPE, ET_CLASS] */
extern T0* GE_new1692(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1694(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [NATURAL_32] */
extern T0* GE_new1695(TC* ac, T1 initialize);
/* New instance of type ET_BASE_TYPE_CONSTRAINT_LIST */
extern T0* GE_new1696(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_FEATURE_NAME] */
extern T0* GE_new1697(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_FLATTENED_FEATURE] */
extern T0* GE_new1698(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [ET_FEATURE_NAME] */
extern T0* GE_new1701(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FLATTENED_FEATURE] */
extern T0* GE_new1702(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_FEATURE_NAME] */
extern T0* GE_new1703(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_FLATTENED_FEATURE, ET_ALIAS_NAME] */
extern T0* GE_new1705(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLIENT_LIST] */
extern T0* GE_new1706(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLIENT_LIST] */
extern T0* GE_new1707(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_CLIENT_LIST] */
extern T0* GE_new1708(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_CLIENT, ET_CLASS] */
extern T0* GE_new1710(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CLIENT] */
extern T0* GE_new1711(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_CLIENT] */
extern T0* GE_new1712(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_SET [ET_FEATURE_NAME] */
extern T0* GE_new1713(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [BOOLEAN, ET_FEATURE_NAME] */
extern T0* GE_new1714(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [ET_REPLICABLE_FEATURE, INTEGER_32] */
extern T0* GE_new1715(TC* ac, T1 initialize);
/* New instance of type ET_REPLICATED_FEATURE */
extern T0* GE_new1717(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [ET_ADAPTED_FEATURE] */
extern T0* GE_new1718(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_PARENT_FEATURE] */
extern T0* GE_new1719(TC* ac, T1 initialize);
/* New instance of type ET_DOTNET_SIGNATURE_TESTER */
extern T0* GE_new1720(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET [ET_DOTNET_FEATURE] */
extern T0* GE_new1721(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [DS_LINKED_LIST [ET_DOTNET_FEATURE], ET_DOTNET_FEATURE] */
extern T0* GE_new1723(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [ET_DOTNET_FEATURE] */
extern T0* GE_new1725(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TOPOLOGICAL_SORTER [ET_LIKE_FEATURE] */
extern T0* GE_new1726(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_LIKE_FEATURE] */
extern T0* GE_new1727(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [ET_PARENT_FEATURE] */
extern T0* GE_new1728(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST [ET_PARENT_FEATURE] */
extern T0* GE_new1729(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [DS_HASH_TABLE [TUPLE [detachable ARRAY [ET_TYPE], detachable ET_TYPE, NATURAL_8], ET_FEATURE_NAME], NATURAL_8] */
extern T0* GE_new1730(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_FEATURE, INTEGER_32] */
extern T0* GE_new1731(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_CLASS] */
extern T0* GE_new1734(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_RENAME, ET_FEATURE_NAME] */
extern T0* GE_new1736(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_RENAME] */
extern T0* GE_new1737(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_RENAME] */
extern T0* GE_new1740(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_RENAME, ET_ALIAS_NAME] */
extern T0* GE_new1744(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_PROCEDURE] */
extern T0* GE_new1745(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_QUERY] */
extern T0* GE_new1746(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [NATURAL_8] */
extern T0* GE_new1749(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [NATURAL_8, ET_CLASS_NAME] */
extern T0* GE_new1750(TC* ac, T1 initialize);
/* New instance of type XM_EIFFEL_CHARACTER_ENTITY */
extern T0* GE_new1751(TC* ac, T1 initialize);
/* New instance of type DS_BILINKABLE [XM_POSITION] */
extern T0* GE_new1752(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST_CURSOR [XM_POSITION] */
extern T0* GE_new1753(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [XM_EIFFEL_SCANNER] */
extern T0* GE_new1754(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_EIFFEL_ENTITY_DEF] */
extern T0* GE_new1755(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [XM_EIFFEL_ENTITY_DEF, STRING_8] */
extern T0* GE_new1758(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_EIFFEL_ENTITY_DEF] */
extern T0* GE_new1759(TC* ac, T1 initialize);
/* New instance of type KL_EQUALITY_TESTER [XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1762(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new1763(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST [XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new1764(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST_CURSOR [XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1765(TC* ac, T1 initialize);
/* New instance of type DS_BILINKABLE [XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new1766(TC* ac, T1 initialize);
/* New instance of type DS_BILINKED_LIST_CURSOR [STRING_8] */
extern T0* GE_new1767(TC* ac, T1 initialize);
/* New instance of type DS_BILINKABLE [STRING_8] */
extern T0* GE_new1768(TC* ac, T1 initialize);
/* New instance of type XM_NAMESPACE_RESOLVER_CONTEXT */
extern T0* GE_new1769(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_QUEUE [detachable STRING_8] */
extern T0* GE_new1772(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_QUEUE [STRING_8] */
extern T0* GE_new1773(TC* ac, T1 initialize);
/* New instance of type KL_EQUALITY_TESTER [XM_NAMESPACE] */
extern T0* GE_new1774(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [XM_NAMESPACE] */
extern T0* GE_new1775(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [XM_NAMESPACE] */
extern T0* GE_new1776(TC* ac, T1 initialize);
/* New instance of type SPECIAL [XM_NAMESPACE] */
extern T0* GE_new1777(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_LINKABLE [XM_DOCUMENT_NODE] */
extern T0* GE_new1778(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1780(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_FEATURE, INTEGER_32] */
extern T0* GE_new1781(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
extern T0* GE_new1783(TC* ac, T6 a1, T1 initialize);
/* New instance of type SPECIAL [ET_CALL_NAME] */
extern T0* GE_new1785(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_QUERY_CALL] */
extern T0* GE_new1787(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_CALL_NAME] */
extern T0* GE_new1788(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_QUERY_CALL, ET_CALL_NAME] */
extern T0* GE_new1789(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
extern T0* GE_new1790(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL, ET_CALL_NAME] */
extern T0* GE_new1792(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL] */
extern T0* GE_new1793(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1794(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1795(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1796(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ADAPTED_CLASS] */
extern T0* GE_new1797(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ADAPTED_CLASS] */
extern T0* GE_new1799(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_CLASS] */
extern T0* GE_new1800(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1801(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ITERATION_COMPONENT] */
extern T0* GE_new1802(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_INLINE_SEPARATE_ARGUMENT] */
extern T0* GE_new1803(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [RX_CHARACTER_SET] */
extern T0* GE_new1804(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [RX_CHARACTER_SET] */
extern T0* GE_new1805(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [NATURAL_64] */
extern T0* GE_new1810(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_DYNAMIC_PRIMARY_TYPE] */
extern T0* GE_new1814(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable DS_LINKABLE [INTEGER_32]] */
extern T0* GE_new1815(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [INTEGER_32]] */
extern T0* GE_new1816(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [INTEGER_32]] */
extern T0* GE_new1817(TC* ac, T1 initialize);
/* New instance of type KL_STRING_INPUT_STREAM */
extern T0* GE_new1819(TC* ac, T1 initialize);
/* New instance of type UT_TRISTATE */
extern T0* GE_new1821(TC* ac, T1 initialize);
/* New instance of type KL_AGENT_ROUTINES [ANY] */
extern T0* GE_new1822(TC* ac, T1 initialize);
/* New instance of type TUPLE [UT_TRISTATE] */
extern T0* GE_new1823(TC* ac, T1 initialize);
/* New instance of type TUPLE [KL_AGENT_ROUTINES [ANY], PROCEDURE [TUPLE]] */
extern T0* GE_new1824(TC* ac, T1 initialize);
/* New instance of type PREDICATE [TUPLE [ET_MASTER_CLASS]] */
extern T0* GE_new1825(TC* ac, T1 initialize);
/* New instance of type PREDICATE [TUPLE] */
extern T0* GE_new1826(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE] */
extern T0* GE_new1828(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_UNIVERSE] */
extern T0* GE_new1829(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_ECF_SYSTEM, DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE], INTEGER_32] */
extern T0* GE_new1830(TC* ac, T1 initialize);
/* New instance of type KL_AGENT_ROUTINES [ET_UNIVERSE] */
extern T0* GE_new1831(TC* ac, T1 initialize);
/* New instance of type PREDICATE [TUPLE [ET_UNIVERSE]] */
extern T0* GE_new1832(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_HASH_TABLE [INTEGER_32, ET_UNIVERSE]] */
extern T0* GE_new1833(TC* ac, T1 initialize);
/* New instance of type TUPLE [KL_AGENT_ROUTINES [ET_UNIVERSE], PREDICATE [TUPLE [ET_UNIVERSE]]] */
extern T0* GE_new1834(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE, BOOLEAN]] */
extern T0* GE_new1836(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN]] */
extern T0* GE_new1837(TC* ac, T1 initialize);
/* New instance of type TUPLE [PROCEDURE [TUPLE [ET_MASTER_CLASS]], FUNCTION [TUPLE [ET_MASTER_CLASS], BOOLEAN], FUNCTION [TUPLE, BOOLEAN]] */
extern T0* GE_new1838(TC* ac, T1 initialize);
/* New instance of type ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
extern T0* GE_new1839(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_MASTER_CLASS, ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1847(TC* ac, T1 initialize);
/* New instance of type DS_CELL [detachable ET_CLASS] */
extern T0* GE_new1848(TC* ac, T1 initialize);
/* New instance of type TUPLE [DS_CELL [detachable ET_CLASS]] */
extern T0* GE_new1849(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_EIFFEL_PREPARSER, DS_ARRAYED_LIST [STRING_8]] */
extern T0* GE_new1851(TC* ac, T1 initialize);
/* New instance of type ET_SEEDED_QUERY_COMPARATOR */
extern T0* GE_new1853(TC* ac, T1 initialize);
/* New instance of type ET_SEEDED_PROCEDURE_COMPARATOR */
extern T0* GE_new1856(TC* ac, T1 initialize);
/* New instance of type TUPLE [ET_CLIENT_LIST] */
extern T0* GE_new1860(TC* ac, T1 initialize);
/* New instance of type ET_STANDARD_ONCE_KEYS */
extern T0* GE_new1864(TC* ac, T1 initialize);
/* New instance of type XM_COMMENT */
extern T0* GE_new1867(TC* ac, T1 initialize);
/* New instance of type XM_PROCESSING_INSTRUCTION */
extern T0* GE_new1868(TC* ac, T1 initialize);
/* New instance of type XM_CHARACTER_DATA */
extern T0* GE_new1869(TC* ac, T1 initialize);
/* New instance of type ET_PARENTHESIS_SYMBOL */
extern T0* GE_new1885(TC* ac, T1 initialize);
/* New instance of type ET_PARENTHESIS_INSTRUCTION */
extern T0* GE_new1888(TC* ac, T1 initialize);
/* New instance of type ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
extern T0* GE_new1892(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
extern T0* GE_new1893(TC* ac, T1 initialize);
/* New instance of type ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
extern T0* GE_new1894(TC* ac, T1 initialize);
/* New instance of type ET_INFIX_CAST_EXPRESSION */
extern T0* GE_new1896(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_PRECURSOR] */
extern T0* GE_new1899(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_PRECURSOR] */
extern T0* GE_new1900(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_MASTER_CLASS] */
extern T0* GE_new1908(TC* ac, T1 initialize);
/* New instance of type ET_BASE_TYPE_RENAME_CONSTRAINT */
extern T0* GE_new1910(TC* ac, T1 initialize);
/* New instance of type DS_SPARSE_TABLE_KEYS [ET_CLIENT, ET_CLASS] */
extern T0* GE_new1913(TC* ac, T1 initialize);
/* New instance of type SPECIAL [NATURAL_16] */
extern T0* GE_new1922(TC* ac, T6 a1, T1 initialize);
/* New instance of type ARRAY [NATURAL_16] */
extern T0* GE_new1923(TC* ac, T1 initialize);
/* New instance of type ARRAY [NATURAL_32] */
extern T0* GE_new1924(TC* ac, T1 initialize);
/* New instance of type SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1925(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1926(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
extern T0* GE_new1927(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_ADAPTED_LIBRARY] */
extern T0* GE_new1928(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new1929(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_FILE_RULE] */
extern T0* GE_new1930(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_NAMESPACE] */
extern T0* GE_new1931(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_CFLAG] */
extern T0* GE_new1932(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_INCLUDE] */
extern T0* GE_new1933(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LIBRARY] */
extern T0* GE_new1934(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_LINKER_FLAG] */
extern T0* GE_new1935(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_MAKE] */
extern T0* GE_new1936(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_OBJECT] */
extern T0* GE_new1937(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_EXTERNAL_RESOURCE] */
extern T0* GE_new1938(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_ACTION] */
extern T0* GE_new1939(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_ACTION] */
extern T0* GE_new1940(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_ACTION] */
extern T0* GE_new1941(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [RX_PCRE_REGULAR_EXPRESSION] */
extern T0* GE_new1942(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_OPTIONS] */
extern T0* GE_new1943(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_OPTIONS] */
extern T0* GE_new1946(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1947(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1948(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1949(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_ANDED_CONDITIONS] */
extern T0* GE_new1950(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST [ET_ECF_CONDITION_ITEM] */
extern T0* GE_new1951(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1952(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1953(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1954(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1956(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1958(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1959(TC* ac, T6 a1, T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER1 */
extern T0* GE_new1960(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER2 */
extern T0* GE_new1961(TC* ac, T1 initialize);
/* New instance of type ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
extern T0* GE_new1962(TC* ac, T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER3 */
extern T0* GE_new1963(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_LIBRARY] */
extern T0* GE_new1964(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_LIBRARY] */
extern T0* GE_new1965(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_LIBRARY] */
extern T0* GE_new1966(TC* ac, T6 a1, T1 initialize);
/* New instance of type LX_DFA_WILDCARD */
extern T0* GE_new1967(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE [LX_WILDCARD, STRING_8] */
extern T0* GE_new1968(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_QUEUE [CHARACTER_8] */
extern T0* GE_new1971(TC* ac, T1 initialize);
/* New instance of type UC_UTF16_ROUTINES */
extern T0* GE_new1972(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET] */
extern T0* GE_new1973(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1974(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1975(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1976(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_TARGET_PARENT] */
extern T0* GE_new1977(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_TARGET_PARENT] */
extern T0* GE_new1978(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_TARGET_PARENT] */
extern T0* GE_new1979(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_ECF_CLUSTER] */
extern T0* GE_new1980(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_ECF_CLUSTER] */
extern T0* GE_new1981(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_ECF_CLUSTER] */
extern T0* GE_new1982(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_DYNAMIC_TARGET] */
extern T0* GE_new1983(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_DYNAMIC_TARGET] */
extern T0* GE_new1984(TC* ac, T1 initialize);
/* New instance of type SPECIAL [YY_BUFFER] */
extern T0* GE_new1985(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [YY_BUFFER] */
extern T0* GE_new1986(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [INTEGER_32, ET_CLASS] */
extern T0* GE_new1988(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1989(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1990(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_BASE_TYPE_CONSTRAINT] */
extern T0* GE_new1991(TC* ac, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [NATURAL_32] */
extern T0* GE_new1992(TC* ac, T1 initialize);
/* New instance of type DS_HASH_SET_CURSOR [ET_FEATURE_NAME] */
extern T0* GE_new1993(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [BOOLEAN, ET_FEATURE_NAME] */
extern T0* GE_new1994(TC* ac, T1 initialize);
/* New instance of type DS_HASH_TABLE_CURSOR [ET_REPLICABLE_FEATURE, INTEGER_32] */
extern T0* GE_new1996(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_REPLICABLE_FEATURE] */
extern T0* GE_new1997(TC* ac, T6 a1, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_REPLICABLE_FEATURE] */
extern T0* GE_new1999(TC* ac, T1 initialize);
/* New instance of type DS_LINKED_LIST_CURSOR [ET_ADAPTED_FEATURE] */
extern T0* GE_new2000(TC* ac, T1 initialize);
/* New instance of type DS_LINKABLE [ET_ADAPTED_FEATURE] */
extern T0* GE_new2001(TC* ac, T1 initialize);
/* New instance of type KL_SPECIAL_ROUTINES [ET_PARENT_FEATURE] */
extern T0* GE_new2002(TC* ac, T1 initialize);
/* New instance of type SPECIAL [ET_PARENT_FEATURE] */
extern T0* GE_new2003(TC* ac, T6 a1, T1 initialize);
/* New instance of type DS_ARRAYED_LIST_CURSOR [ET_PARENT_FEATU